bl_info = {
    "name": "Physics Gen",
    "author": "AceOf3D",
    "version": (1, 4, 1),
    "blender": (2, 93, 0),
    "location": "3D Viewport > 'N' Key Sidebar > Physics > Physics Gen",
    "description": "Physics Simulation Properties Randomizer and Simplifier",
    "warning": "",
    "wiki_url": "",
    "category": "Object",
}

import bpy
import random
import time
from bpy.types import(
    AddonPreferences,
    PropertyGroup,
)

randomFieldList = ['FORCE', 'WIND', 'VORTEX', 'TURBULENCE', 'HARMONIC', 'MAGNET']

#----------------------Properties Group----------------------------

class CustomPropertyGroup(bpy.types.PropertyGroup):
    
    cacheFrameStart: bpy.props.IntProperty(name= "Simulation Start", default= 1)
        
    cacheFrameEnd: bpy.props.IntProperty(name= "Simulation End", default= 250)
    
    useFramesForCache: bpy.props.BoolProperty(name= "Use current frames as cache", default= True)
        
#----------------------Smoke Operator--------------------------------------------------------------------------------

class OBJECT_OT_smokerandomize(bpy.types.Operator):
    bl_idname = "object.smoke_randomize"
    bl_label = "Smoke Domain Randomize"
    bl_description = "Randomization of smoke sim domain settings"
    bl_options = {'REGISTER', 'UNDO'}
    
    # 1. First, select any object, add quick smoke setup
    # 2. Make sure domain is selected object
    # 3. 3D Viewport > N Tab > Physics > Physics Gen

    def execute(self, context):

        # Set Domain Settings
        bpy.context.object.modifiers["Fluid"].domain_settings.use_noise = True

        # Domain randomizations
        bpy.context.object.modifiers["Fluid"].domain_settings.dissolve_speed = random.randint(0, 10)
        bpy.context.object.modifiers["Fluid"].domain_settings.vorticity = random.randint(0, 1)
        bpy.context.object.modifiers["Fluid"].domain_settings.noise_strength = random.randint(0, 3)
        bpy.context.object.modifiers["Fluid"].domain_settings.noise_pos_scale = random.randint(1, 8)
        bpy.context.object.modifiers["Fluid"].domain_settings.alpha = random.uniform(1,4)
        
        bpy.ops.screen.frame_jump(end=False)

        return {'FINISHED'}

#------------------------------------Fire Operator-----------------------------------------------------

class OBJECT_OT_smokefirerandomize(bpy.types.Operator):
    bl_idname = "object.smokefire_randomize"
    bl_label = "Fire Domain Randomize"
    bl_description = "Randomization of smoke sim domain settings"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        
        #Fire Randomization
        bpy.context.object.modifiers["Fluid"].domain_settings.burning_rate = random.uniform(1,4)
        bpy.context.object.modifiers["Fluid"].domain_settings.flame_smoke = random.uniform(.2,4)
        bpy.context.object.modifiers["Fluid"].domain_settings.flame_vorticity = random.uniform(.1,1)
        bpy.context.object.modifiers["Fluid"].domain_settings.flame_max_temp = random.uniform(1,4)
        bpy.context.object.modifiers["Fluid"].domain_settings.flame_ignition = random.uniform(.1,1)

        bpy.ops.screen.frame_jump(end=False)

        return {'FINISHED'}
    
#----------------------Quick Smoke Setup----------------------------------
    
class OBJECT_OT_quickSmokeSetup(bpy.types.Operator):
    bl_idname = "object.quicksmokesetup"
    bl_label = "Quick Smoke Setup"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        bpy.ops.object.quick_smoke()
        bpy.ops.screen.frame_jump(end=False)
        

        return {'FINISHED'}

#----------------------Quick Fire Setup----------------------------------
    
class OBJECT_OT_quickFireSetup(bpy.types.Operator):
    bl_idname = "object.quickfiresetup"
    bl_label = "Quick Fire Setup"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        bpy.ops.object.quick_smoke(style='BOTH')
        bpy.ops.screen.frame_jump(end=False)
        

        return {'FINISHED'}
    
#----------------------Fluid Randomize Operator-------------------------------------------

class OBJECT_OT_liquidrandomize(bpy.types.Operator):
    bl_idname = "object.liquid_randomize"
    bl_label = "Liquid Domain Randomize"
    bl_description = "Randomization of liquid sim domain settings"
    bl_options = {'REGISTER', 'UNDO'}
    
    
    def execute(self, context):
        sourceName = bpy.context.object.name
        sourceObject = bpy.data.objects[sourceName]

        ## Set Domain Settings
        bpy.context.object.modifiers["Fluid"].domain_settings.use_diffusion = True

        # Domain randomizations
        bpy.context.object.modifiers["Fluid"].domain_settings.flip_ratio = random.uniform(.5, 0.97)
        bpy.context.object.modifiers["Fluid"].domain_settings.particle_randomness = random.uniform(0.1, 0.4)
        bpy.context.object.modifiers["Fluid"].domain_settings.viscosity_base = random.randint(1, 8)
        bpy.context.object.modifiers["Fluid"].domain_settings.viscosity_exponent = random.randint(1, 10) 
        bpy.context.object.modifiers["Fluid"].domain_settings.surface_tension = random.randint(0, 10)

        #3 Finalize
        bpy.ops.screen.frame_jump(end=False)
        
        return {'FINISHED'}

#----------------------Quick Fluid Setup----------------------------------
    
class OBJECT_OT_quickFluidSetup(bpy.types.Operator):
    bl_idname = "object.quickfluidsetup"
    bl_label = "Quick Fluid Setup"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        bpy.ops.object.quick_liquid()
        bpy.context.object.modifiers["Fluid"].domain_settings.cache_frame_end = 250
        bpy.ops.screen.frame_jump(end=False)

        return {'FINISHED'}
    
#----------------------Fluid Particles On----------------------------------
    
class OBJECT_OT_fluidParticlesOn(bpy.types.Operator):
    bl_idname = "object.particleson"
    bl_label = "Particles On/Off"
    bl_description = "Toggle fluid particles (spray, foam, bubble) on or off"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        bpy.context.object.modifiers["Fluid"].domain_settings.use_spray_particles = True
        bpy.context.object.modifiers["Fluid"].domain_settings.use_foam_particles = True
        bpy.context.object.modifiers["Fluid"].domain_settings.use_bubble_particles = True

        return {'FINISHED'}
        
#----------------------Particle Operator----------------------------

class OBJECT_OT_particlerandomize(bpy.types.Operator):
    bl_idname = "object.particle_randomize"
    bl_label = "Particle Randomize"
    bl_description = "Randomization of particle sim settings"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):

        # random particle settings
        bpy.data.particles["ParticleSettings"].normal_factor = random.randint(1, 30)
        bpy.data.particles["ParticleSettings"].count = random.randint(5000, 10000)
        bpy.data.particles["ParticleSettings"].lifetime = random.randint(50, 300)
        bpy.data.particles["ParticleSettings"].lifetime_random = random.uniform(0.1, 0.5)
        bpy.data.particles["ParticleSettings"].object_factor = 0.2
        bpy.data.particles["ParticleSettings"].factor_random = 0.2
        bpy.data.particles["ParticleSettings"].brownian_factor = random.uniform(0.01, 10.0)
        bpy.data.particles["ParticleSettings"].damping = random.uniform(0, 0.5) 
        bpy.data.particles["ParticleSettings"].render_type = 'HALO'
        bpy.data.particles["ParticleSettings"].size_random = random.uniform(0.15, 0.9)
        bpy.data.particles["ParticleSettings"].effector_weights.gravity = random.uniform(.5, 1)
        
        bpy.ops.screen.frame_jump(end=False)
        
        return {'FINISHED'}
        
#----------------------Random Force Field Add Operator-----------------------------

class OBJECT_OT_addfieldrandomize(bpy.types.Operator):
    bl_idname = "object.addfield_randomize"
    bl_label = "Add Random Force Field"
    bl_description = "Add Random Force Field"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):

        randomFieldSelection = random.choice(randomFieldList)
        
        # import random force field
        bpy.ops.object.effector_add(type=randomFieldSelection, enter_editmode=False, location=(0, 0, -1))
            
        return {'FINISHED'}
    
#----------------------Randomize Force Field Operator-----------------------------

class OBJECT_OT_fieldrandomize(bpy.types.Operator):
    bl_idname = "object.field_randomize"
    bl_label = "Randomize Force Field"
    bl_description = "Randomize force field settings"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
         
        randomFieldCheck = bpy.context.object.field.type
        
        # random force field settings based on field selection
        if randomFieldCheck == 'FORCE':
            bpy.context.object.field.strength = random.randint(-10, 10)
            bpy.context.object.field.flow = random.randint(0, 3)
        if randomFieldCheck == 'WIND':
            bpy.context.object.field.strength = random.randint(1, 20)
            bpy.context.object.field.flow = random.randint(0, 3)
        if randomFieldCheck == 'VORTEX':
            bpy.context.object.field.strength = random.randint(-8, 8)
            bpy.context.object.field.flow = random.randint(0, 3)
        if randomFieldCheck == 'TURBULENCE':
            bpy.context.object.field.strength = random.randint(10, 100)
            bpy.context.object.field.size = random.randint(1, 6)
            bpy.context.object.field.flow = random.uniform(0, 2)
        if randomFieldCheck == 'HARMONIC':
            bpy.context.object.field.strength = random.uniform(-2, 2)
            bpy.context.object.field.flow = random.randint(0, 3)
            bpy.context.object.field.rest_length = random.randint(0, 6)
        if randomFieldCheck == 'MAGNET':
            bpy.context.object.field.strength = random.randint(-8, 8)
            bpy.context.object.field.flow = random.randint(0, 3)
            
        return {'FINISHED'}
    
#----------------------Set Frame to 1 Operator-----------------------------

class OBJECT_OT_resetframe(bpy.types.Operator):
    bl_idname = "scene.resetframe"
    bl_label = "Reset Frame"
    bl_description = "Resets Current Frame to 1"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
        
        # Reset Frame to Beginning
        bpy.ops.screen.frame_jump(end=False)
        
        return {'FINISHED'}
    
#----------------------Collisions Toggle Operator-----------------------------

class OBJECT_OT_CollisionToggleOn(bpy.types.Operator):
    bl_idname = "object.collisiontoggleon"
    bl_label = "Border Collision On"
    bl_description = "Collisions Toggle"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
        
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_front = True
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_back = True
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_right = True
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_left = True
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_top = True
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_bottom = True
            
        return {'FINISHED'}
    
#----------------------Collisions Toggle Operator-----------------------------

class OBJECT_OT_CollisionToggleOff(bpy.types.Operator):
    bl_idname = "object.collisiontoggleoff"
    bl_label = "Border Collision Off"
    bl_description = "Collisions Toggle"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
        
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_front = False
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_back = False
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_right = False
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_left = False
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_top = False
        bpy.context.object.modifiers["Fluid"].domain_settings.use_collision_border_bottom = False
            
        return {'FINISHED'}
    
#----------------------Use Frames as Cache Operator-----------------------------

class OBJECT_OT_UseFramesAsCache(bpy.types.Operator):
    bl_idname = "object.framestoscache"
    bl_label = "Use Timeline Frames to Bake"
    bl_description = "Use current timeline frames as cache start and end"
    bl_options = {'REGISTER', 'UNDO'}
    
    def execute(self, context):
        
        bpy.context.object.modifiers["Fluid"].domain_settings.cache_frame_start = bpy.context.scene.frame_start
        bpy.context.object.modifiers["Fluid"].domain_settings.cache_frame_end = bpy.context.scene.frame_end
        
        return {'FINISHED'}
        
#----------------------UI Creation Smoke/Fire-----------------------------

class Panel_PT_SmokeFirePhysicsGen(bpy.types.Panel):
    bl_label = "Physics Gen Smoke/Fire"
    bl_idname = "PANEL_PT_smokePhysicsGen"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "Physics"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        
        box = layout.box()
        box.label(text=" Select mesh first:")
        row = box.row()
        row.scale_y = 1.5
        row.operator("object.quicksmokesetup")
        row.operator("object.quickfiresetup")
        row = box.row()
        
        try:
            if bpy.context.object.modifiers["Fluid"].fluid_type == 'DOMAIN':
                box = layout.box()
                box.label(text=" Select smoke or fire domain first:")
                row = box.row()
                row.scale_y = 1.5

                row.operator("object.smoke_randomize")
                row.operator("object.smokefire_randomize")
                row = box.row()
                
                box = layout.box()
                box.label(text=" Bake or free smoke sims:")
                row = box.row()
                row.operator("fluid.bake_all")
                row.operator("fluid.free_all")
                row = box.row()
                row.operator("scene.resetframe")
                
                box = layout.box()
                box.label(text=" Reselect after resizing domain:")
                box.prop(context.object.modifiers["Fluid"].domain_settings, 'cache_type')
                row = box.row()
        except:
            pass
        
#----------------------UI Creation Fluid-----------------------------   
 
class Panel_PT_FluidPhysicsGen(bpy.types.Panel):
    bl_label = "Physics Gen Fluid"
    bl_idname = "PANEL_PT_fluidPhysicsGen"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "Physics"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        
        box = layout.box()
        box.label(text=" Select mesh first: ")
        row = box.row()
        row.scale_y = 1.5
        row.operator("object.quickfluidsetup")
        row = box.row()
        
        try:
            if context.object.modifiers["Fluid"].fluid_type == 'DOMAIN':
                box = layout.box()
                box.label(text=" Select domain first:")
                box.scale_y = 1.5
                box.operator("object.liquid_randomize")
                row = box.row()
                row.scale_y = .8
                row = box.row()
                row.scale_y = .8
                row.operator("object.particleson")
                row.prop(context.object.modifiers["Fluid"].domain_settings, 'use_mesh')
                
                box = layout.box()
                box.label(text=" Bake or free fluid sims:")
                row = box.row()
                row.operator("fluid.bake_all")       
                row.operator("fluid.free_all")
                row = box.row()
                row.operator("scene.resetframe")
                
                box = layout.box()
                box.label(text=" Reselect after resizing domain:")
                box.prop(context.object.modifiers["Fluid"].domain_settings, 'cache_type')
                row = box.row()
        except:
            pass
        
#----------------------UI Creation Particle-----------------------------   
 
class Panel_PT_ParticlePhysicsGen(bpy.types.Panel):
    bl_label = "Physics Gen Particles"
    bl_idname = "PANEL_PT_ParticlePhysicsGen"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "Physics"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        
        box = layout.box()
        box.label(text=" Add particles:")
        row = box.row()
        row.scale_y = 1.5
        row.operator("object.particle_system_add")
        row.operator("object.particle_randomize")
        row = box.row()
        
        box = layout.box()
        box.label(text=" Bake or free particle sims:")  
        row = box.row()
        row.operator("ptcache.bake_all")
        row.operator("ptcache.free_bake_all")
        row = box.row()
        row.operator("scene.resetframe")
        
#----------------------UI Creation Particle Extras-----------------------------   
 
class Panel_PT_ParticleExtrasPhysicsGen(bpy.types.Panel):
    bl_label = "Physics Gen Particle Rendering"
    bl_idname = "PANEL_PT_ParticleExtrasPhysicsGen"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "Physics"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        
        box = layout.box()
        box.label(text=" Render particle as:")
        box.prop(bpy.data.particles["ParticleSettings"], 'render_type')
        box.prop(bpy.data.particles["ParticleSettings"], 'instance_object')
        box.prop(bpy.data.particles["ParticleSettings"], 'instance_collection')
        row = box.row()
        
        box = layout.box()
        box.label(text=" Particle size:")
        box.prop(bpy.data.particles["ParticleSettings"], 'particle_size')
        box.prop(bpy.data.particles["ParticleSettings"], 'size_random')
        row = box.row()
        
#----------------------UI Creation Force Field-----------------------------

class Panel_PT_ForcesPhysicsGen(bpy.types.Panel):
    bl_label = "Physics Gen Force Field "
    bl_idname = "PANEL_PT_ForceFieldphysicsGen"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "Physics"
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout
        
        box = layout.box()
        box.label(text=" Add or randomize:")
        row = box.row()
        row.scale_y = 1.5
        row.operator("object.addfield_randomize")
        row.operator("object.field_randomize")
        row = box.row()
        
        
#----------------------UI Creation Extra Settings-----------------------------

class Panel_PT_ExtraSettingsPhysicsGen(bpy.types.Panel):
    bl_label = "Physics Gen Extra Domain Settings "
    bl_idname = "PANEL_PT_ExtraSettingsphysicsGen"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "Physics"
    bl_options = {'DEFAULT_CLOSED'}
    
    def draw(self, context):
        layout = self.layout
        
        try:
            if bpy.context.object.modifiers["Fluid"].fluid_type == 'DOMAIN':
                box = layout.box()
                box.label(text=" Domain Resolution:")
                box.prop(context.object.modifiers["Fluid"].domain_settings, 'resolution_max')
                row = box.row()
                
                box = layout.box()
                box.label(text=" Domain Collisions:")
                box.operator('object.collisiontoggleon')
                box.operator('object.collisiontoggleoff')
                row = box.row()
                
                box = layout.box()
                row = box.row()
                row.scale_y = 1.5
                box.operator("object.framestoscache")
                box.label(text=" Domain Frames to Bake:")
                box.prop(context.object.modifiers["Fluid"].domain_settings, 'cache_frame_start')
                box.prop(context.object.modifiers["Fluid"].domain_settings, 'cache_frame_end')
                row = box.row()
                
                box = layout.box()
                box.prop(bpy.context.object, 'display_type')
                
            else:
                box = layout.box()
                box.label(text=" Select the domain object")
                row = box.row()
        except:
            pass
                
#----------------------Register-----------------------------        
    
classes = (
    CustomPropertyGroup,
    OBJECT_OT_smokerandomize,
    OBJECT_OT_smokefirerandomize,
    OBJECT_OT_quickSmokeSetup,
    OBJECT_OT_quickFireSetup,
    OBJECT_OT_liquidrandomize,
    OBJECT_OT_quickFluidSetup,
    OBJECT_OT_fluidParticlesOn,
    OBJECT_OT_particlerandomize,
    OBJECT_OT_fieldrandomize,
    OBJECT_OT_addfieldrandomize,
    OBJECT_OT_resetframe,
    OBJECT_OT_CollisionToggleOn,
    OBJECT_OT_CollisionToggleOff,
    OBJECT_OT_UseFramesAsCache,
    Panel_PT_SmokeFirePhysicsGen,
    Panel_PT_FluidPhysicsGen,
    Panel_PT_ParticlePhysicsGen,
    Panel_PT_ParticleExtrasPhysicsGen,
    Panel_PT_ForcesPhysicsGen,  
    Panel_PT_ExtraSettingsPhysicsGen,
)

def register():
    from bpy.utils import register_class
    for cls in classes:
        register_class(cls)
        
    bpy.types.Scene.custom_props = bpy.props.PointerProperty(type=CustomPropertyGroup)

def unregister():    
    from bpy.utils import unregister_class
    for cls in reversed(classes):
        unregister_class(cls)
    
    del bpy.types.Scene.custom_props 


if __name__ == "__main__":
    register()