# pylint: disable=missing-module-docstring
# pylint: disable=missing-class-docstring
# pylint: disable=missing-function-docstring
# pylint: disable=too-many-lines
from typing import Callable, Dict, List, Iterator, Optional, Sequence, Set, Tuple, Union
from operator import attrgetter
from functools import partial
from itertools import groupby, repeat
from dataclasses import dataclass, field, replace
from bpy.types import AnimData, Armature, FCurve, ID, PoseBone
import numpy as np
from . import rbf, distance, expression as xp
from .proxies import AnimDataProxy, FCurveProxy, DriverVariableProxy
from .utils import Keygen

class ExpressionLengthError(Exception):
    # pylint: disable=missing-class-docstring
    # pylint: disable=unnecessary-pass
    pass

def create_data_type(pose_count: int) -> np.dtype:
    return np.dtype([
        ('type', 'U12'),
        ('mode', 'U34'),
        ('norm', np.float),
        ('data', np.float, (pose_count,)),
    ])

def format_influence_idpropkey(id_: int, name: str) -> str:
    return f'RBF Driver Pose {name.title()} - rbfn.{str(id_).zfill(3)}'

def clone_influence_idprop(datablock: Union[ID, PoseBone],
                           animdata: Optional[AnimData],
                           prevname: str,
                           prevpath: str,
                           nextname: str,
                           nextpath: str) -> None:
    idprop = datablock.get(prevname)
    if idprop is not None:
        datablock[nextname] = idprop

        if isinstance(datablock, PoseBone):
            del datablock[prevname]

        rna_ui = datablock.get('_RNA_UI')
        if rna_ui is not None and prevname in rna_ui:
            rna_ui[nextname] = rna_ui[prevname]
            if isinstance(datablock, PoseBone):
                del rna_ui[prevname]

        if animdata:
            for fcurve in [d for d in animdata.drivers if d.data_path == prevpath]:
                fcurve = animdata.drivers.from_existing(fcurve)
                fcurve.data_path = nextpath

def get_or_create_influence_metadata(target: Union[ID, PoseBone], propname: str) -> dict:
    metadata = target.get('_RNA_UI')

    if metadata is None:
        target['_RNA_UI'] = {}
        metadata = target['_RNA_UI']

    if propname not in metadata:
        metadata[propname] = {
            "min": -10000.0,
            "max": 10000.0,
            "soft_min": 0.0,
            "soft_max": 1.0,
            "default": 1.0
        }

    return metadata[propname].to_dict()

@dataclass
class InputTemplate:

    type: str
    mode: str
    default_axes: List[Tuple[int, str]]
    tag_property: str
    map_function: Callable[['Poses'], Iterator[Sequence[float]]]
    tag_override: Optional[Sequence[bool]] = None

    @classmethod
    def define(cls,
               type_: str,
               mode: str,
               axes: Union[str, List[Tuple[int, str]]],
               prop: str,
               attr: str) -> 'InputTemplate':
        # pylint: disable=too-many-arguments
        if isinstance(axes, str):
            axes = list(enumerate(axes))
        return cls(type_, mode, axes, prop, partial(map, attrgetter(attr)))

    def refine(self, tags: Sequence[bool]) -> 'InputTemplate':
        return replace(self, tag_override=tags)

    def axes(self, driver: 'RBFDriver') -> Sequence[str]:
        return [i for (_, i), x in zip(self.default_axes, self.tags(driver)) if x]

    def tags(self, driver: 'RBFDriver') -> Sequence[bool]:
        return self.tag_override or getattr(driver, self.tag_property)

    def indices(self, driver: 'RBFDriver') -> Sequence[int]:
        tags = self.tags(driver)
        return [i for i, (x, _) in enumerate(self.default_axes) if tags[x]]

    def toarray(self, driver: 'RBFDriver') -> np.ndarray:
        poses = driver.poses
        types = [f'{self.type}_{i}' for i in self.axes(driver)]
        nchan = len(types)
        array = np.array(list(self.map_function(poses)), dtype=np.float).T
        array = array.take(self.indices(driver), axis=0)
        modes = list(repeat(self.mode, nchan))
        norms = list(repeat(0.0, nchan))
        dtype = create_data_type(len(poses))
        return np.array(list(zip(types, modes, norms, array)), dtype=dtype)

# pylint: disable=line-too-long
XYZ = InputTemplate.define('ROT', 'XYZ', [(1, 'X'), (2, 'Y'), (3, 'Z')], "use_rotation", "rotation_euler")
XZY = InputTemplate.define('ROT', 'XYZ', [(1, 'X'), (3, 'Z'), (2, 'Y')], "use_rotation", "rotation_euler")
YXZ = InputTemplate.define('ROT', 'XYZ', [(2, 'Y'), (1, 'X'), (3, 'Z')], "use_rotation", "rotation_euler")
YZX = InputTemplate.define('ROT', 'XYZ', [(2, 'Y'), (3, 'Z'), (1, 'X')], "use_rotation", "rotation_euler")
ZXY = InputTemplate.define('ROT', 'XYZ', [(3, 'Z'), (1, 'X'), (2, 'Y')], "use_rotation", "rotation_euler")
ZYX = InputTemplate.define('ROT', 'XYZ', [(3, 'Z'), (2, 'Y'), (1, 'Z')], "use_rotation", "rotation_euler")
AUTO = XYZ
LOCATION = InputTemplate.define('LOC', 'AUTO', 'XYZ', "use_location", "location")
QUATERNION = InputTemplate.define('ROT', 'QUATERNION', 'WXYZ', "use_rotation", "rotation_quaternion")
SCALE = InputTemplate.define('SCALE', 'AUTO', 'XYZ', "use_scale", "scale")
SWING_TWIST_X = InputTemplate.define('ROT', 'SWING_TWIST_X', 'WXYZ', "use_rotation", "rotation_swing_twist_x")
SWING_TWIST_Y = InputTemplate.define('ROT', 'SWING_TWIST_Y', 'WXYZ', "use_rotation", "rotation_swing_twist_y")
SWING_TWIST_Z = InputTemplate.define('ROT', 'SWING_TWIST_Z', 'WXYZ', "use_rotation", "rotation_swing_twist_z")
# pylint: enable=line-too-long

@dataclass
class InputGroup:

    type: str
    data: np.ndarray
    norm: float
    radius: float
    arrays: List[np.ndarray]

    def __init__(self, type_: str, data: List[np.ndarray]) -> None:
        self.type = type_
        self.data = np.concatenate(data, axis=0)
        self.norm = 0.0
        self.radius = 0.0
        self.arrays = data
        self.idprops = {}
        if type_ == 'SCALAR':
            self.normalize()

    def normalize(self) -> None:
        for channel in self.data:
            data = channel['data']
            norm = channel['norm'] = round(np.linalg.norm(data), 7)
            np.divide(data, norm, where=norm != 0.0, out=data)

    def distance_matrix(self,
                        function: rbf.Function,
                        radius: Union[float, Callable[[np.ndarray], float]]) -> np.ndarray:
        # metric = distance.quaternion if self.type == 'QUATERNION' else distance.euclidean
        metric = distance.euclidean if self.type == 'SCALAR' else distance.quaternion
        matrix = distance.matrix(self.data['data'].T, metric)

        if self.type == 'SCALAR':
            norm = self.norm = round(np.linalg.norm(matrix), 7)
            np.divide(matrix, norm, where=norm != 0.0, out=matrix)

        if function is not None:
            radius = radius(matrix) if callable(radius) else radius
            self.radius = radius
            matrix = function(matrix, radius)

        return matrix

@dataclass
class IDProp:

    root: Union[ID, PoseBone] = None
    name: str = ""
    path: str = ""
    data: object = None
    install: bool = False
    metadata: Dict[str, object] = field(default_factory=dict)
    installed: bool = False

@dataclass
class Layer:
    driver: 'RBFDriver'
    idprops: Dict[str, IDProp] = field(default_factory=dict)
    animdata: List[AnimDataProxy] = field(default_factory=list)

    def prepare(self) -> None:
        try:
            self.prepare_animdata(False)
        except ExpressionLengthError:
            for idprop in self.idprops.values():
                idprop.install = True
            self.prepare_animdata(True)

    def prepare_animdata(self, use_variables: bool) -> None:
        pass

    def install(self) -> None:
        for idprop in self.idprops.values():
            if idprop.install and not idprop.installed:
                idprop.root[idprop.name] = idprop.data
                if idprop.metadata:
                    try:
                        rnaui = idprop.root['_RNA_UI']
                    except KeyError:
                        idprop.root['_RNA_UI'] = {}
                        rnaui = idprop.root['_RNA_UI']
                    rnaui[idprop.name] = idprop.metadata
                idprop.installed = True
        for item in self.animdata:
            item.apply()

@dataclass
class InputDriver(FCurveProxy):

    keygen: Keygen = None

    def __init__(self, path: str, index: int, mute: bool) -> None:
        super().__init__(path, index, mute)
        self.keygen = Keygen()

    def prepare(self, layer: 'InputLayer', poses: IDProp) -> None:
        offset = 0
        driver = self.driver
        expressions = []
        driver.variables.clear()
        self.keygen.reset()

        for index, group in enumerate(layer.groups):
            inputs = self.input_symbols(group, poses, offset)

            if poses.install:
                params = self.group_symbols(group, poses, offset)
            else:
                params = group.data['data'].T[self.array_index]

            if group.type == 'SCALAR':
                expr = xp.euclidean_distance(inputs, params)
            else:
                expr = xp.quaternion_distance(inputs, params)

            norm = group.norm
            if norm != 0.0:
                if poses.install:
                    norm = self.group_norm_symbol(poses, index)
                expr = xp.divide(expr, norm)

            interpolation = layer.driver.interpolation
            if interpolation != 'NONE':
                radius = group.radius
                if radius != 0.0:
                    if poses.install:
                        radius = self.group_radius_symbol(poses, index)

                    expr = getattr(xp, f'rbf_{interpolation.lower()}')(expr, radius)

            expressions.append(expr)
            offset += len(group.data)

        expr = expressions[0] if len(expressions) == 1 else xp.mean(expressions)
        if len(expr) > 256:
            raise ExpressionLengthError()

        driver.type = 'SCRIPTED'
        driver.expression = expr

    def group_norm_symbol(self, params: IDProp, index: int) -> str:
        variable = DriverVariableProxy()
        variable.type = 'SINGLE_PROP'
        variable.name = self.keygen()

        target = variable.targets[0]
        target.id = params.root.id_data
        target.data_path = (f'{params.root.path_from_id()}["_RNA_UI"]'
                            f'["{params.name}"]["norms"][{index}]')

        self.driver.variables.append(variable)
        return variable.name

    def group_radius_symbol(self, params: IDProp, index: int) -> str:
        variable = DriverVariableProxy()
        variable.type = 'SINGLE_PROP'
        variable.name = self.keygen()

        target = variable.targets[0]
        target.id = params.root.id_data
        target.data_path = (f'{params.root.path_from_id()}["_RNA_UI"]'
                            f'["{params.name}"]["radii"][{index}]')

        self.driver.variables.append(variable)
        return variable.name

    def input_symbols(self, group: InputGroup, params: IDProp, offset: int) -> Iterator[str]:
        keygen = self.keygen

        for index, channel in enumerate(group.data):

            variable = DriverVariableProxy()
            variable.type = 'TRANSFORMS'
            variable.name = keygen()

            target = variable.targets[0]
            target.id = params.root.id_data
            target.bone_target = params.root.name
            target.transform_type = channel['type']
            target.rotation_mode = channel['mode']
            target.transform_space = 'LOCAL_SPACE'

            self.driver.variables.append(variable)

            expression = variable.name

            if group.type.startswith('SWING_TWIST'):
                expression = f'{"cos" if channel["type"] == "ROT_W" else "sin"}({expression}*0.5)'

            norm = channel['norm']
            if norm != 0.0:
                if params.install:
                    norm = self.channel_norm_symbol(params, offset, index)
                expression = xp.divide(expression, norm)
            
            yield expression

    def channel_norm_symbol(self, params: IDProp, offset: int, index: int) -> str:
        variable = DriverVariableProxy()
        variable.type = 'SINGLE_PROP'
        variable.name = self.keygen()

        count = params.data["data"].shape[1]

        target = variable.targets[0]
        target.id = params.root.id_data
        target.data_path = (f'{params.path}[{(count + 1) * (offset + index) + count}]')

        self.driver.variables.append(variable)
        return variable.name

    def group_symbols(self, group: InputGroup, params: IDProp, offset: int) -> Iterator[str]:
        keygen = self.keygen
        root = params.root
        path = params.path
        pose = self.array_index
        variables = self.driver.variables

        for row in range(group.data.shape[0]):
            variable = DriverVariableProxy()
            variable.type = 'SINGLE_PROP'
            variable.name = keygen()

            target = variable.targets[0]
            target.id = root.id_data
            target.data_path = f'{path}[{(offset+row)*(params.data["data"].shape[1]+1)+pose}]'

            variables.append(variable)
            yield variable.name

@dataclass
class InputLayer(Layer):

    target: PoseBone = None
    groups: List[InputGroup] = field(default_factory=list)

    def __init__(self, target: PoseBone, driver: 'RBFDriver') -> None:
        super().__init__(driver)
        self.target = target
        self.groups = []
        self.init_groups()
        self.init_idprops()
        self.init_animdata()

    def init_groups(self) -> None:
        arrays = []
        driver = self.driver
        groups = self.groups

        if any(driver.use_location):
            arrays.append(LOCATION.toarray(driver))
        if any(driver.use_scale):
            arrays.append(SCALE.toarray(driver))

        rmode = driver.rotation_mode
        flags = driver.use_rotation

        if all(flags) or ((len(rmode) == 3 or rmode == 'AUTO') and all(flags[1:])):
            arrays.append(QUATERNION.refine(tuple(repeat(True, 4))).toarray(driver))
        elif any(driver.use_rotation):
            arrays.append(globals()[driver.rotation_mode].toarray(driver))

        def key(group: np.ndarray) -> str:
            for mode, types in [
                    ('QUATERNION', ('ROT_W', 'ROT_X', 'ROT_Y', 'ROT_Z')),
                    ('SWING_TWIST_X', ('ROT_W', 'ROT_Y', 'ROT_Z')),
                    ('SWING_TWIST_Y', ('ROT_W', 'ROT_X', 'ROT_Z')),
                    ('SWING_TWIST_Z', ('ROT_W', 'ROT_X', 'ROT_Y')),
                ]:
                if np.all(group['mode'] == mode) and np.array_equal(group['type'], types):
                    return mode
            return 'SCALAR'

        for type_, group in groupby(arrays, key=key):
            groups.append(InputGroup(type_, list(group)))

    def init_idprops(self) -> None:
        bone = self.target
        path = bone.path_from_id()

        name = f'RBF Driver Input Distance - rbfn.{str(self.driver.id__internal).zfill(3)}'
        data = np.array(list(repeat(0.0, len(self.driver.poses))), dtype=np.float)
        meta = {"rbfnid": self.driver.id__internal}
        self.idprops['OUTPUT'] = IDProp(bone, name, f'{path}["{name}"]', data, True, meta)

        name = f'RBF Driver Input Matrix - rbfn.{str(self.driver.id__internal).zfill(3)}'
        data = np.concatenate([g.data for g in self.groups], axis=0)
        meta = {"norms": [], "radii": [], "rbfnid": self.driver.id__internal}
        self.idprops['PARAMS'] = IDProp(bone, name, f'{path}["{name}"]', data, False, meta)

        name = f'RBF Driver Pose Weight - rbfn.{str(self.driver.id__internal).zfill(3)}'
        make = name not in bone
        data = [1.0 for _ in self.driver.poses] if make else bone[name].to_list()
        meta = get_or_create_influence_metadata(bone, name)
        self.idprops['INFLUENCES'] = IDProp(bone, name,
                                            f'{path}["{name}"]', data, make, meta, not make)

    def init_animdata(self) -> None:
        mute = self.driver.mute
        prop = self.idprops['OUTPUT']
        path = prop.path
        data = AnimDataProxy(prop.root.id_data)
        data.drivers.extend(InputDriver(path, i, mute) for i in range(prop.data.shape[0]))
        self.animdata.append(data)

    def distance_matrix(self) -> np.ndarray:
        driver = self.driver
        function = getattr(rbf, driver.interpolation.lower(), None)
        dispersion = driver.dispersion
        radius = driver.radius if dispersion == 'NONE' else getattr(np, dispersion.lower())
        matrices = []

        for group in self.groups:
            matrix = group.distance_matrix(function, radius)
            matrices.append(matrix)

        if len(matrices) == 1:
            matrix = matrices[0]
        else:
            matrix = np.add.reduce(matrices)
            np.divide(matrix, float(len(matrices)), out=matrix)

        return matrix

    def variable_matrix(self) -> np.ndarray:

        distance_matrix = self.distance_matrix()
        identity_matrix = np.identity(distance_matrix.shape[0], dtype=np.float)

        smoothing = self.driver.smoothing
        if smoothing > 0.0:
            identity_matrix *= (1.0 + smoothing)

        try:
            return np.linalg.solve(distance_matrix, identity_matrix)
        except np.linalg.LinAlgError:
            return np.linalg.lstsq(distance_matrix, identity_matrix, rcond=None)[0]

    def prepare_animdata(self, use_variables: bool) -> None:
        params = self.idprops['PARAMS']
        for driver in self.animdata[0].drivers:
            driver.prepare(self, params)

    def install(self) -> None:
        prop = self.idprops['PARAMS']
        if prop.install:
            meta = prop.metadata
            data = []
            meta["norms"] = [g.norm for g in self.groups]
            meta["radii"] = [g.radius for g in self.groups]
            for channel in prop.data:
                data.extend(channel['data'])
                data.append(channel['norm'])
            prop.data = data
            prop.installed = True
        super().install()

@dataclass
class SummedInfluenceDriver(FCurveProxy):

    keygen: Keygen = None

    def __init__(self, path: str, mute: bool) -> None:
        super().__init__(path, 0, mute)
        self.keygen = Keygen()

    def prepare(self, influences: IDProp) -> None:
        driver = self.driver
        keygen = self.keygen
        variables = driver.variables
        symbols = []

        variables.clear()
        keygen.reset()

        for index in range(len(influences.data)):
            variable = DriverVariableProxy()
            variable.type = 'SINGLE_PROP'
            variable.name = keygen()

            target = variable.targets[0]
            target.id = influences.root.id_data
            target.data_path = f'{influences.path}[{index}]'

            variables.append(variable)
            symbols.append(variable.name)

        driver.type = 'SUM'

@dataclass
class WeightDriver(FCurveProxy):

    keygen: Keygen = field(default_factory=Keygen)

    def __init__(self, path: str, index: int, mute: bool) -> None:
        super().__init__(path, index, mute)
        self.keygen = Keygen()

    def prepare(self, inputs: IDProp, variables: IDProp, scalar: IDProp, clamp: bool) -> None:
        driver = self.driver
        driver.variables.clear()

        self.keygen.reset()
        inputs = list(self.input_symbols(inputs))

        if variables.install:
            params = list(self.param_symbols(variables))
        else:
            params = variables.data.T[self.array_index]

        expression = xp.dot(inputs, params)
        if clamp:
            expression = xp.clamp(expression, 0.0, 1.0)
        else:
            expression = xp.wrap(expression)

        variable = DriverVariableProxy()
        variable.type = 'SINGLE_PROP'
        variable.name = self.keygen()

        target = variable.targets[0]
        target.id = scalar.root.id_data
        target.data_path = f'{scalar.path}[{self.array_index}]'

        self.driver.variables.append(variable)
        expression = xp.multiply(expression, variable.name)

        if len(expression) > 256:
            raise ExpressionLengthError()

        driver.type = 'SCRIPTED'
        driver.expression = expression

    def input_symbols(self, distances: IDProp) -> Iterator[str]:
        keygen = self.keygen
        driver = self.driver
        root = distances.root
        path = distances.path
        variables = driver.variables

        for index in range(len(distances.data)):
            variable = DriverVariableProxy()
            variable.type = 'SINGLE_PROP'
            variable.name = keygen()

            target = variable.targets[0]
            target.id = root.id_data
            target.data_path = f'{path}[{index}]'

            variables.append(variable)
            yield variable.name

    def param_symbols(self, params: IDProp) -> Iterator[str]:
        keygen = self.keygen
        driver = self.driver
        root = params.root
        path = params.path
        pose = self.array_index
        variables = driver.variables
        nrows, ncols = params.data.shape

        for index in range(nrows):
            variable = DriverVariableProxy()
            variable.type = 'SINGLE_PROP'
            variable.name = keygen()

            target = variable.targets[0]
            target.id_type = 'ARMATURE'
            target.id = root
            target.data_path = f'{path}[{index * ncols + pose}]'

            variables.append(variable)
            yield variable.name

@dataclass
class SummedWeightDriver(FCurveProxy):

    keygen: Keygen = None

    def __init__(self, path: str, mute: bool) -> None:
        super().__init__(path, 0, mute)
        self.keygen = Keygen()

    def prepare(self, weights: IDProp) -> None:
        driver = self.driver
        keygen = self.keygen
        variables = driver.variables
        symbols = []

        variables.clear()
        keygen.reset()

        for index in range(len(weights.data)):
            variable = DriverVariableProxy()
            variable.type = 'SINGLE_PROP'
            variable.name = keygen()

            target = variable.targets[0]
            target.id_type = 'ARMATURE'
            target.id = weights.root
            target.data_path = f'{weights.path}[{index}]'

            variables.append(variable)
            symbols.append(variable.name)

        driver.type = 'SUM'

@dataclass
class FinalWeightDriver(FCurveProxy):

    keygen: Keygen = field(default_factory=Keygen)

    def __init__(self, path: str, index: int, mute: bool) -> None:
        super().__init__(path, index, mute)
        self.keygen = Keygen()

    def prepare(self, weight: IDProp, wgtsum: IDProp) -> None:
        driver = self.driver
        keygen = self.keygen

        driver.variables.clear()
        keygen.reset()

        wgt = DriverVariableProxy()
        wgt.type = 'SINGLE_PROP'
        wgt.name = keygen()

        tgt = wgt.targets[0]
        tgt.id_type = 'ARMATURE'
        tgt.id = weight.root
        tgt.data_path = f'{weight.path}[{self.array_index}]'

        driver.variables.append(wgt)

        sum_ = DriverVariableProxy()
        sum_.type = 'SINGLE_PROP'
        sum_.name = keygen()

        tgt = sum_.targets[0]
        tgt.id_type = 'ARMATURE'
        tgt.id = wgtsum.root
        tgt.data_path = f'{wgtsum.path}'

        driver.variables.append(sum_)

        driver.type = 'SCRIPTED'
        driver.expression = xp.where(xp.eq(sum_.name, 0.0),
                                     float(self.array_index == 0),
                                     xp.divide(wgt.name, sum_.name))

@dataclass
class WeightLayer(Layer):

    target: Armature = None
    inputs: InputLayer = None

    def __init__(self, inputs: InputLayer) -> None:
        driver = inputs.driver
        target = inputs.target.id_data.data

        super().__init__(driver)

        self.inputs = inputs
        self.target = target

        self.idprops['INPUTS'] = inputs.idprops['OUTPUT']
        self.idprops['SCALAR'] = inputs.idprops['INFLUENCES']

        name = f'RBF Driver Variables Matrix - rbfn.{str(driver.id__internal).zfill(3)}'
        data = inputs.variable_matrix()
        meta = {"rbfnid": driver.id__internal}
        self.idprops['PARAMS'] = IDProp(target, name, f'["{name}"]', data, False, meta)

        name = f'RBF Driver Pose Weights - rbfn.{str(driver.id__internal).zfill(3)}'
        path = f'["{name}"]'
        data = np.array(list(repeat(0.0, len(driver.poses))), dtype=np.float)
        meta = {"rbfnid": driver.id__internal}
        self.idprops['WEIGHT'] = IDProp(target, name, path, data, True, meta)

        mute = self.driver.mute
        animdata = AnimDataProxy(target)
        animdata.drivers.extend(WeightDriver(path, i, mute) for i in range(len(data)))
        self.animdata.append(animdata)

        name = f'RBF Driver Summed Weight - rbfn.{str(driver.id__internal).zfill(3)}'
        data = 0.0
        meta = {"rbfnid": driver.id__internal}
        self.idprops['WGTSUM'] = IDProp(target, name, f'["{name}"]', data, True, meta)

        animdata = AnimDataProxy(target)
        animdata.drivers.append(SummedWeightDriver(self.idprops['WGTSUM'].path, self.driver.mute))
        self.animdata.append(animdata)

        name = f'RBF Driver Pose Outputs - rbfn.{str(driver.id__internal).zfill(3)}'
        path = f'["{name}"]'
        data = np.array(list(repeat(0.0, len(driver.poses))), dtype=np.float)
        meta = {"rbfnid": driver.id__internal}
        self.idprops['OUTPUT'] = IDProp(target, name, f'["{name}"]', data, True, meta)

        animdata = AnimDataProxy(target)
        animdata.drivers.extend(FinalWeightDriver(path, i, mute) for i in range(len(data)))
        self.animdata.append(animdata)

    def prepare_animdata(self, use_variables: bool) -> None:
        inputs = self.idprops['INPUTS']
        params = self.idprops['PARAMS']
        scalar = self.idprops['SCALAR']
        clamp = self.driver.interpolation == 'GAUSSIAN'
        for driver in self.animdata[0].drivers:
            driver.prepare(inputs, params, scalar, clamp)

        weight = self.idprops['WEIGHT']
        for driver in self.animdata[1].drivers:
            driver.prepare(weight)

        wgtsum = self.idprops['WGTSUM']
        for driver in self.animdata[2].drivers:
            driver.prepare(weight, wgtsum)

@dataclass
class DrivenProperty:
    # pylint: disable=invalid-name
    id: ID = None
    data_path: str = ''
    array_index: int = 0
    data: np.ndarray = None

@dataclass
class OutputDriver(FCurveProxy):

    keygen: Keygen = field(default_factory=Keygen)

    def __init__(self, prop: DrivenProperty, mute: bool) -> None:
        super().__init__(prop.data_path, prop.array_index, mute)
        self.keygen = Keygen()

    def prepare(self, inputs: IDProp, effect: IDProp, params: IDProp, index: int) -> None:
        driver = self.driver
        driver.variables.clear()

        self.keygen.reset()
        inputs = list(self.input_symbols(inputs))

        if params.install:
            params = list(self.param_symbols(params, index))
        else:
            params = params.data[index]

        params = [
            self.apply_effect(p, effect, i) for i, p in enumerate(params)
            ]

        expression = xp.dot(inputs, params)
        if len(expression) > 256:
            raise ExpressionLengthError()

        driver.type = 'SCRIPTED'
        driver.expression = expression

    def input_symbols(self, weights: IDProp) -> Iterator[str]:
        # pylint: disable=missing-function-docstring
        keygen = self.keygen
        driver = self.driver
        path = weights.path
        root = weights.root
        variables = driver.variables

        for pose_index in range(weights.data.shape[0]):
            variable = DriverVariableProxy()
            variable.type = 'SINGLE_PROP'
            variable.name = keygen()

            target = variable.targets[0]
            target.id_type = 'ARMATURE'
            target.id = root
            target.data_path = f'{path}[{pose_index}]'

            variables.append(variable)
            yield variable.name

    def param_symbols(self, params: IDProp, index: int) -> Iterator[str]:
        # pylint: disable=missing-function-docstring
        keygen = self.keygen
        driver = self.driver
        path = params.path
        root = params.root
        variables = driver.variables
        nrows, ncols = params.data.shape

        for pose_index in range(ncols):
            variable = DriverVariableProxy()
            variable.type = 'SINGLE_PROP'
            variable.name = keygen()

            target = variable.targets[0]
            target.id_type = 'ARMATURE'
            target.id = root
            target.data_path = f'{path}[{index * ncols + pose_index}]'

            variables.append(variable)
            yield variable.name

    def apply_effect(self, param: Union[str, float], effect: IDProp, index: int) -> str:
        variable = DriverVariableProxy()
        variable.type = 'SINGLE_PROP'
        variable.name = self.keygen()

        target = variable.targets[0]
        target.id_type = 'ARMATURE'
        target.id = effect.root
        target.data_path = f'{effect.path}[{index}]'

        self.driver.variables.append(variable)
        return xp.multiply(param, variable.name)

@dataclass
class OutputLayer(Layer):

    target: Armature = None
    weight: WeightLayer = None
    properties: List[DrivenProperty] = field(default_factory=list)

    def __init__(self, weight: WeightLayer) -> None:
        driver = weight.driver
        target = weight.target

        super().__init__(driver)

        self.target = target
        self.weight = weight
        self.properties = []

        data = []
        for prop in driver.driven_properties:
            if prop.is_valid:
                data.append([x.value for x in prop.samples])

                if prop.type == 'SHAPE_KEY':
                    id_ = prop.id.data.shape_keys
                    path = f'key_blocks["{prop.shape_key}"].value'
                    index = 0
                else:
                    id_ = prop.id
                    path = prop.state__internal.dpath
                    index = prop.array_index

                self.properties.append(DrivenProperty(id_, path, index))

        name = f'RBF Driver Pose Matrix - rbfn.{str(driver.id__internal).zfill(3)}'
        data = np.array(data, dtype=np.float)
        self.idprops['PARAMS'] = IDProp(target, name, f'["{name}"]', data)

        for prop, data in zip(self.properties, data):
            prop.data = data

        table = {}
        for prop in self.properties:
            table.setdefault(prop.id, []).append(prop)

        mute = driver.mute
        for id_, props in table.items():
            data = AnimDataProxy(id_)
            data.drivers.extend(OutputDriver(p, mute) for p in props)
            self.animdata.append(data)

        self.idprops['INPUTS'] = weight.idprops['OUTPUT']

        name = f'RBF Driver Pose Effect - rbfn.{str(self.driver.id__internal).zfill(3)}'
        path = f'["{name}"]'
        make = name not in target
        data = [1.0 for _ in self.driver.poses] if make else target[name].to_list()
        meta = get_or_create_influence_metadata(target, name)
        self.idprops['EFFECT'] = IDProp(target, name, path, data, make, meta, not make)

    def prepare_animdata(self, use_variables: bool) -> None:
        weight = self.idprops['INPUTS']
        params = self.idprops['PARAMS']
        effect = self.idprops['EFFECT']
        offset = 0
        for item in self.animdata:
            for driver in item.drivers:
                driver.prepare(weight, effect, params, offset)
                offset += 1

def search_for_clone(target: PoseBone,
                     driver: 'RBFDriver') -> Tuple[Optional[PoseBone], Optional['RBFDriver']]:
    # pylint: disable=missing-function-docstring
    id_ = driver.id__internal
    for owner in target.id_data.pose.bones:
        if owner != target and owner.is_property_set("rbf_drivers"):
            clone = next((d for d in owner.rbf_drivers if d.id__internal == id_), None)
            if clone is not None:
                return owner, clone
    return None, None

def set_new_id(target: PoseBone, driver: 'RBFDriver') -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=redefined-outer-name
    drivers = []
    for bone in target.id_data.pose.bones:
        if bone.is_property_set("rbf_drivers"):
            drivers.extend(bone.rbf_drivers)

    def key(driver) -> int:
        return driver.id__internal

    drivers.sort(key=key)

    old_id = driver.id__internal
    new_id = next((i for i, x in enumerate(drivers) if key(x) != i), len(drivers))

    for datablock, animdata, pathprefix, propname in [
            (target, target.id_data.animation_data, f'pose.bones["{target.name}"]', 'Weight'),
            (target.id_data.data, target.id_data.data.animation_data, "", 'Effect')
        ]:
        old_name = format_influence_idpropkey(old_id, propname)
        new_name = format_influence_idpropkey(new_id, propname)

        clone_influence_idprop(datablock, animdata,
                               old_name, f'{pathprefix}["{old_name}"]',
                               new_name, f'{pathprefix}["{new_name}"]')

    driver.id__internal = new_id

def ensure_unique(target: PoseBone, driver: 'RBFDriver') -> None:

    owner, clone = search_for_clone(target, driver)
    if clone:
        destroy(owner, clone, {'INPUT', 'OUTPUT', 'NO_REBUILD'})
        set_new_id(owner, clone)
        ensure_unique(target, driver)

def update(target: PoseBone, driver: 'RBFDriver') -> None:

    ensure_unique(target, driver)

    inputs = InputLayer(target, driver)
    weight = WeightLayer(inputs)
    output = OutputLayer(weight)
    layers = (inputs, weight, output)

    for layer in layers:
        layer.prepare()

    for layer in layers:
        layer.install()

def destroy(target: PoseBone,
            driver: 'RBFDriver',
            options: Optional[Set[str]] = None) -> None:

    for animdata, fcurve in drivers(target, driver, options):
        animdata.drivers.remove(fcurve)

    if options is None or 'INPUT' in options:
        for datablock, key in idprops(target, driver):
            del datablock[key]
            del datablock['_RNA_UI'][key]

    if options is None or 'NO_REBUILD' not in options:
        ensure_unique(target, driver)

def idprops(target: PoseBone, driver: 'RBFDriver') -> Iterator[Tuple[Union[ID, PoseBone], str]]:
    nid = driver.id__internal

    for data in (target, target.id_data.data):
        metadata = data.get('_RNA_UI')
        if metadata is not None:
            for key in data.keys():
                if key in metadata and metadata[key].get("rbfnid") == nid:
                    yield data, key

def drivers(target: PoseBone,
            driver: 'RBFDriver',
            options: Optional[Set[str]] = None) -> Iterator[Tuple[AnimData, FCurve]]:
    nid = driver.id__internal

    if options is None:
        options = {'INPUT', 'OUTPUT'}

    if 'INPUT' in options:
        for root, data, path in [
                (target.id_data, target, target.path_from_id()),
                (target.id_data.data, target.id_data.data, "")
            ]:
            rna_ui = data.get('_RNA_UI')
            animdata = root.animation_data

            if rna_ui is None or animdata is None:
                continue

            for key in data.keys():
                metadata = rna_ui.get(key)
                if metadata is not None and metadata.get("rbfnid") == nid:
                    datapath = f'{path}["{key}"]'
                    for fcurve in animdata.drivers:
                        if fcurve.data_path == datapath:
                            yield animdata, fcurve

    if 'OUTPUT' in options:
        for prop in driver.driven_properties:
            if prop.is_valid:

                if prop.type == 'SHAPE_KEY':
                    id_ = prop.id.data.shape_keys
                    path = f'key_blocks["{prop.shape_key}"].value'
                else:
                    id_ = prop.id
                    path = prop.state__internal.dpath

                animdata = id_.animation_data
                if animdata is not None:
                    fcurve = animdata.drivers.find(path, index=prop.array_index)
                    if fcurve:
                        yield animdata, fcurve
