# pylint: disable=missing-module-docstring
# pylint: disable=too-many-lines
from typing import Any, Iterator, List, Optional, Tuple, Union
from itertools import chain
from math import acos, asin
from bpy.types import (bpy_prop_collection,
                       Context,
                       DriverTarget,
                       Object,
                       PoseBone,
                       PropertyGroup)
from bpy.props import (BoolProperty,
                       BoolVectorProperty,
                       CollectionProperty,
                       EnumProperty,
                       FloatProperty,
                       FloatVectorProperty,
                       IntProperty,
                       PointerProperty,
                       StringProperty)
from mathutils import Euler, Quaternion, Vector
from idprop.types import IDPropertyArray
from .utils import get_mirror_pose_bone, get_mirror_suffix_for_name, set_mirror_suffix_for_name
from . import network

EnumItem = Tuple[str, str, str, str, int]

ARRAY = (Euler, Quaternion, Vector, IDPropertyArray)

# region Utilities
###################################################################################################

def extract_enum_items(collection: bpy_prop_collection) -> List[EnumItem]:
    # pylint: disable=missing-function-docstring
    return [(x.identifier, x.name, x.description, x.icon, i) for i, x in enumerate(collection)]

def get_mirror_pose_bone_and_rbf_driver(driver: 'RBFDriver') -> Tuple[Optional[PoseBone],
                                                                      Optional['RBFDriver']]:
    # pylint: disable=missing-function-docstring
    owner = driver.owner
    other = get_mirror_pose_bone(owner)
    if other is not None:
        mirror = other.rbf_drivers.get(driver.name)
        if mirror is not None:
            return other, mirror
    return (None, None)

def get_mirror_rbf_driver(driver: 'RBFDriver') -> Optional['RBFDriver']:
    # pylint: disable=missing-function-docstring
    return get_mirror_pose_bone_and_rbf_driver(driver)[1]

# endregion

# region Poses
###################################################################################################

def Pose__location__get(pose: 'Pose') -> Vector:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    return pose.matrix.to_translation()


def Pose__rotation_euler__get(pose: 'Pose') -> Quaternion:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    return pose.matrix.to_euler()


def Pose__rotation_quaternion__get(pose: 'Pose') -> Quaternion:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    return pose.matrix.to_quaternion()


def Pose__rotation_swing_twist_x__get(pose: 'Pose') -> Tuple[float, float, float, float]:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    swing, twist = pose.rotation_quaternion.to_swing_twist('X')
    swing[1] = twist
    return swing


def Pose__rotation_swing_twist_y__get(pose: 'Pose') -> Tuple[float, float, float, float]:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    swing, twist = pose.rotation_quaternion.to_swing_twist('Y')
    swing[2] = twist
    return swing


def Pose__rotation_swing_twist_z__get(pose: 'Pose') -> Tuple[float, float, float, float]:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    swing, twist = pose.rotation_quaternion.to_swing_twist('Z')
    swing[2] = twist
    return swing


def Pose__scale__get(pose: 'Pose') -> Vector:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    return pose.matrix.to_scale()


def Pose__name__update(pose: 'Pose', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    poses = pose.parent_collection
    names = [p.name for p in poses if p != pose]
    index = 0
    cache = pose.name
    value = cache

    while value in names:
        value = f'{cache}.{str(index).zfill(3)}'
        index += 1

    if cache != value:
        pose.name = value
        return

    cache = pose.name__internal
    pose.name__internal = value

    if cache != "":
        driver = pose.rbf_driver
        if driver.xm__internal:
            mirror = get_mirror_rbf_driver(driver)
            if mirror is not None:
                counterpart = next((p for p in mirror.poses if p.name == cache), None)
                if counterpart:
                    mirror.xm__internal = False
                    counterpart.name = value
                    mirror.xm__internal = True


class Pose(PropertyGroup):
    """
    Transform state snapshot of the driving pose bone
    """

    name__internal: StringProperty(options=set())

    location: FloatVectorProperty(
        name="Location",
        description="The location vector for the pose (read-only)",
        size=3,
        subtype='XYZ',
        get=Pose__location__get
        )

    matrix: FloatVectorProperty(
        name="Matrix",
        description="The local transform matrix for the pose",
        size=16,
        subtype='MATRIX',
        options=set()
        )

    name: StringProperty(
        name="Name",
        description="User-defined name for the pose (unique)",
        update=Pose__name__update,
        options=set()
        )

    @property
    def parent_collection(self) -> 'Poses':
        """The parent collection to which this pose belongs"""
        return self.rbf_driver.poses

    # pylint: disable=inconsistent-return-statements
    @property
    def rbf_driver(self) -> 'RBFDriver':
        """The RBF driver to which this pose belongs"""
        root = self.id_data
        for bone in root.pose.bones:
            if bone.is_property_set("rbf_drivers"):
                for driver in bone.rbf_drivers:
                    if any(item == self for item in driver.poses):
                        return driver

    rotation_euler: FloatVectorProperty(
        name="Rotation",
        description="The euler rotation for the pose (read-only)",
        size=3,
        subtype='EULER',
        get=Pose__rotation_euler__get
        )

    rotation_quaternion: FloatVectorProperty(
        name="Rotation",
        description="The quaternion rotation for the pose (read-only)",
        size=4,
        subtype='QUATERNION',
        get=Pose__rotation_quaternion__get
        )

    rotation_swing_twist_x: FloatVectorProperty(
        name="Rotation",
        description="The quaternion for the swing and twist (X) for the pose (read-only)",
        size=4,
        subtype='QUATERNION',
        get=Pose__rotation_swing_twist_x__get
        )

    rotation_swing_twist_y: FloatVectorProperty(
        name="Rotation",
        description="The quaternion for the swing and twist (Y) for the pose (read-only)",
        size=4,
        subtype='QUATERNION',
        get=Pose__rotation_swing_twist_y__get
        )

    rotation_swing_twist_z: FloatVectorProperty(
        name="Rotation",
        description="The quaternion for the swing and twist (Z) for the pose (read-only)",
        size=4,
        subtype='QUATERNION',
        get=Pose__rotation_swing_twist_z__get
        )

    scale: FloatVectorProperty(
        name="Rotation",
        description="The scale for the pose (read-only)",
        size=3,
        subtype='XYZ',
        get=Pose__scale__get
        )


class Poses(PropertyGroup):
    """
    Collection of poses
    """

    data__internal: CollectionProperty(type=Pose)

    @property
    def active(self) -> Optional[Pose]:
        """The currently active pose"""
        try:
            return self[self.active_index]
        except IndexError:
            return None

    active_index: IntProperty(
        name="Active Index",
        description="The index of the currently active pose",
        min=0,
        options=set()
        )

    def __contains__(self, name: str) -> bool:
        return name in self.data__internal

    def __getitem__(self, key: Union[int, slice]) -> Union[Pose, List[Pose]]:
        return self.data__internal[key]

    def __iter__(self) -> Iterator[Pose]:
        return iter(self.data__internal)

    def __len__(self) -> int:
        return len(self.data__internal)

    def find(self, name: str) -> int:
        """Returns the index of a key in a collection or -1 when not found."""
        return next((i for i, p in enumerate(self.data__internal) if p.name == name), -1)

# endregion

# region Samples
###################################################################################################

class Sample(PropertyGroup):
    # pylint: disable=too-few-public-methods
    """
    Value container for a snapshot of a driven property
    """

    value: FloatProperty(
        name="Value",
        description="The float value of the sample",
        default=0.0,
        options=set()
        )


class Samples(PropertyGroup):
    """
    Collection of samples
    """

    data__internal: CollectionProperty(type=Sample)

    def __getitem__(self, key: Union[int, slice]) -> Union[Sample, List[Sample]]:
        return self.data__internal[key]

    def __iter__(self) -> Iterator[Sample]:
        return iter(self.data__internal)

    def __len__(self) -> int:
        return len(self.data__internal)

# endregion

# region Driven
###################################################################################################

def DrivenProperty__mirror_property_value(prop: 'DrivenProperty',
                                          name: str,
                                          value: Any = None) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    driver = prop.rbf_driver
    if driver.xm__internal:
        mirror = get_mirror_rbf_driver(driver)
        if mirror is not None:
            counterpart = next((p for p in mirror.driven_properties if p.name == prop.name), None)
            if counterpart:
                mirror.xm__internal = False
                setattr(counterpart, name, getattr(prop, name) if value is None else value)
                mirror.xm__internal = True


class DrivenPropertyUpdate:
    # pylint: disable=missing-class-docstring

    def __init__(self, prop: 'DrivenProperty'):
        self.driver = prop.rbf_driver
        self.driver_was_valid = self.driver.is_valid
        self.prop = prop
        self.prop_was_valid = prop.is_valid

    def execute(self) -> None:
        # pylint: disable=missing-function-docstring
        self.prepare()
        self.process()
        self.cleanup()

    def prepare(self) -> None:
        # pylint: disable=missing-function-docstring
        prop = self.prop
        if prop.is_valid:
            state = prop.state__internal
            owner = state.owner or prop.id

            if owner:
                dpath = state.dpath

                if dpath.startswith('data.shape_keys.key_blocks["'):
                    owner = owner.data.shape_keys
                    dpath = f'key_blocks["{dpath[28:]}'

                if owner and owner.animation_data:
                    fcurve = owner.animation_data.drivers.find(dpath, index=state.index)
                    if fcurve:
                        owner.animation_data.drivers.remove(fcurve)

    def process(self) -> None:
        # pylint: disable=missing-function-docstring
        prop = self.prop
        prop.state__internal.update(prop)

    def cleanup(self) -> None:
        # pylint: disable=missing-function-docstring
        driver = self.driver
        target = driver.owner

        if self.driver_was_valid and not driver.is_valid:
            network.destroy(target, driver)
        elif driver.is_valid:
            network.update(target, driver)


class DrivenPropertyState(PropertyGroup):
    """
    The current state of a driven property (internal-use)
    """

    owner: PointerProperty(type=Object, options=set())
    array: BoolProperty(default=False, options=set())
    index: IntProperty(min=0, default=0, options=set())
    valid: BoolProperty(default=False, options=set())
    dpath: StringProperty(options=set())

    def update(self, prop: 'DrivenProperty') -> None:
        # pylint: disable=missing-function-docstring
        owner = self.owner or prop.id

        if owner is None:
            self.valid = False
            self.array = False
            self.index = 0
            self.dpath = ""
            return

        if prop.type == 'SHAPE_KEY':
            self.array = False
            self.index = 0

            if (owner.type != 'MESH'
                    or owner.data.shape_keys is None
                    or prop.shape_key not in owner.data.shape_keys.key_blocks):
                self.valid = False
                self.dpath = ""
                return

            self.valid = True
            self.dpath = prop.data_path = f'data.shape_keys.key_blocks["{prop.shape_key}"].value'
            return

        if prop.type == 'SINGLE_PROP':

            dpath = prop.data_path
            if dpath.endswith(']') and dpath[-2].isdigit():
                start = dpath.rfind('[')
                try:
                    self.index = int(dpath[start+1:-1])
                except ValueError:
                    self.array = False
                    self.valid = False
                    self.index = 0
                    self.dpath = ""
                    return
                self.array = True
                self.dpath = dpath[:start]
            else:
                self.index = 0
                self.array = False
                self.dpath = dpath

            try:
                value = owner.path_resolve(self.dpath)
            except ValueError:
                self.valid = False
            else:
                if not self.array:
                    self.valid = isinstance(value, (int, bool, float))
                else:
                    try:
                        value = value[self.index]
                    except (TypeError, IndexError, KeyError, ValueError):
                        self.valid = False
                    else:
                        self.valid = isinstance(value, (int, bool, float))
            return

        if owner.type == 'ARMATURE' and prop.bone_target != "":
            bone = prop.bone_target
            if bone not in owner.data.bones:
                self.valid = False
                self.array = False
                self.index = 0
                self.dpath = ""
                return
            path = f'pose.bones["{bone}"].'
        else:
            path = ""

        ttype = prop.transform_type

        if ttype.startswith('LOC'):
            path = f'{path}location'
            axes = 'XYZ'
        elif ttype.startswith('SCALE'):
            path = f'{path}scale'
            axes = 'XYZ'
        else:
            mode = prop.rotation_mode
            path = f'{path}rotation_{mode.lower()}'
            if mode == 'EULER':
                axes = 'XYZ'
                if ttype == 'ROT_W':
                    self.valid = False
                    self.array = False
                    self.index = 0
                    self.dpath = ""
                    return
            else:
                axes = 'WXYZ'

        self.valid = True
        self.array = True
        self.index = axes.index(ttype[-1])
        self.dpath = prop.data_path = path


def DrivenProperty__array_index__get(prop: 'DrivenProperty') -> bool:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    return prop.state__internal.index


def DrivenProperty__is_array__get(self) -> bool:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    return self.state__internal.array


def DrivenProperty__is_valid__get(self) -> bool:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    return self.state__internal.valid


def did_update_driven_property_rotation_mode(prop: 'DrivenProperty') -> bool:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    target = prop.id

    if target is not None and target.type == 'ARMATURE' and prop.bone_target:
        target = target.pose.bones.get(prop.bone_target)

    if target is not None:
        mode = target.rotation_mode

        if len(mode) == 3:
            mode = 'EULER'

        if mode != prop.rotation_mode:
            prop.rotation_mode = mode
            return prop.type == 'TRANSFORMS' and prop.transform_type.startswith('ROT')

    return False


def DrivenProperty__bone_target__update(prop: 'DrivenProperty', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if prop.type == 'TRANSFORMS' and not did_update_driven_property_rotation_mode(prop):
        # prop.state__internal.update(prop)
        DrivenPropertyUpdate(prop).execute()

    value = prop.bone_target
    owner = prop.id_data.pose.bones.get(value)

    if owner is not None:
        other = get_mirror_pose_bone(owner)
        if other is not None:
            DrivenProperty__mirror_property_value(prop, "bone_target", other.name)
            return

    DrivenProperty__mirror_property_value(prop, "bone_target")


def DrivenProperty__data_path__update(prop: 'DrivenProperty', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if prop.type == 'SINGLE_PROP':
        # prop.state__internal.update(prop)
        DrivenPropertyUpdate(prop).execute()


def DrivenProperty__id__poll(prop: 'DrivenProperty', value: Object) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    return prop.type != 'SHAPE_KEY' or value.type == 'MESH'


def DrivenProperty__id__update(prop: 'DrivenProperty', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if prop.type == 'SHAPE_KEY' or not did_update_driven_property_rotation_mode(prop):
        # prop.state__internal.update(prop)
        update = DrivenPropertyUpdate(prop)
        update.prepare()
        prop.state__internal.owner = prop.id
        update.execute()
        update.cleanup()

    DrivenProperty__mirror_property_value(prop, "id")


def DrivenProperty__name__update(prop: 'DrivenProperty', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    props = prop.parent_collection
    names = [p.name for p in props if p != prop]
    index = 0
    cache = prop.name
    value = cache

    while value in names:
        value = f'{cache}.{str(index).zfill(3)}'
        index += 1

    if cache != value:
        prop.name = value
        return

    state = prop.state__internal
    cache = state.name
    state.name = value

    if cache == "":
        return

    driver = prop.rbf_driver
    if driver.xm__internal:
        mirror = get_mirror_rbf_driver(driver)
        if mirror is not None:
            counterpart = next((p for p in mirror.driven_properties if p.name == cache), None)
            if counterpart:
                mirror.xm__internal = False
                counterpart.name = value
                mirror.xm__internal = True


def DrivenProperty__rotation_mode__update(prop: 'DrivenProperty', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if prop.type == 'TRANSFORMS' and prop.transform_type.startswith('ROT'):
        # prop.state__internal.update(prop)
        DrivenPropertyUpdate(prop).execute()

    DrivenProperty__mirror_property_value(prop, "rotation_mode")


def DrivenProperty__shape_key__update(prop: 'DrivenProperty', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if prop.type == 'SHAPE_KEY':
        # prop.state__internal.update(prop)
        DrivenPropertyUpdate(prop).execute()

    value = prop.shape_key
    if value:
        suffix = get_mirror_suffix_for_name(value)
        if suffix:
            value = set_mirror_suffix_for_name(value, suffix)

    DrivenProperty__mirror_property_value(prop, "shape_key", value)


def DrivenProperty__transform_type__update(prop: 'DrivenProperty', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if prop.type == 'TRANSFORMS':
        # prop.state__internal.update(prop)
        DrivenPropertyUpdate(prop).execute()

    DrivenProperty__mirror_property_value(prop, "transform_type")


def DrivenProperty__type__update(prop: 'DrivenProperty', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name

    # prop.state__internal.update(prop)
    DrivenPropertyUpdate(prop).execute()
    DrivenProperty__mirror_property_value(prop, "type")


class DrivenProperty(PropertyGroup):
    """
    Output target properties
    """

    state__internal: PointerProperty(
        name="Internal state (API Defined)",
        type=DrivenPropertyState,
        options={'HIDDEN'}
        )

    array_index: IntProperty(
        name="Index",
        description="The array index of the driven property (read-only)",
        get=DrivenProperty__array_index__get,
        options=set()
        )

    bone_target: StringProperty(
        name="Bone",
        description="The name of the pose bone to which the driven property belongs",
        default="",
        options=set(),
        update=DrivenProperty__bone_target__update,
        )

    data_path: StringProperty(
        name="Path",
        description="The data path to the driven property relative to the ID",
        options=set(),
        update=DrivenProperty__data_path__update
        )

    # pylint: disable=inconsistent-return-statements
    @property
    def rbf_driver(self) -> 'RBFDriver':
        """The RBF driver to which this driven property belongs"""
        root = self.id_data
        for bone in root.pose.bones:
            if bone.is_property_set("rbf_drivers"):
                for driver in bone.rbf_drivers:
                    if any(item == self for item in driver.driven_properties):
                        return driver

    id: PointerProperty(
        name="ID",
        description="The ID data-block for the driven property",
        type=Object,
        poll=DrivenProperty__id__poll,
        update=DrivenProperty__id__update,
        options=set(),
        )

    is_array: BoolProperty(
        name="Array",
        description="True if the driven property is array-like, otherwise False (read-only)",
        get=DrivenProperty__is_array__get,
        options=set()
        )

    is_valid: BoolProperty(
        name="Valid",
        description="True if the driven property is valid, otherwise False (read-only)",
        get=DrivenProperty__is_valid__get,
        options=set()
        )

    name: StringProperty(
        name="Name",
        description="User-defined name for the driven property (unique)",
        update=DrivenProperty__name__update,
        options=set()
        )

    @property
    def parent_collection(self) -> 'DrivenProperties':
        """The collection to which this driven property belongs"""
        return self.rbf_driver.driven_properties

    rotation_mode: EnumProperty(
        name="Mode",
        description="The rotation channels to be driven",
        default='EULER',
        options=set(),
        update=DrivenProperty__rotation_mode__update,
        items=[
            ('EULER', "Euler", ""),
            ('AXIS_ANGLE', "Axis/Angle", ""),
            ('QUATERNION', "Quaternion", ""),
            ]
        )

    shape_key: StringProperty(
        name="Key",
        description="The shape key whose value is to be driven",
        default="",
        options=set(),
        update=DrivenProperty__shape_key__update
        )

    show_expanded: BoolProperty(
        name="Expand",
        description="Expand the driven property UI",
        options=set(),
        default=True
        )

    transform_type: EnumProperty(
        name="Type",
        description="The transform channel to be driven",
        default='LOC_X',
        options=set(),
        update=DrivenProperty__transform_type__update,
        items=[
            ('LOC_X', "X Location", ""),
            ('LOC_Y', "Y Location", ""),
            ('LOC_Z', "Z Location", ""),
            None,
            ('ROT_W', "W Rotation", ""),
            ('ROT_X', "X Rotation", ""),
            ('ROT_Y', "Y Rotation", ""),
            ('ROT_Z', "Z Rotation", ""),
            None,
            ('SCALE_X', "X Scale", ""),
            ('SCALE_Y', "Y Scale", ""),
            ('SCALE_Z', "Z Scale", ""),
            ]
        )

    type: EnumProperty(
        name="Type",
        description="The type of property to be driven",
        default='SINGLE_PROP',
        options=set(),
        update=DrivenProperty__type__update,
        items=[
            ('TRANSFORMS',
             "Transform Channel",
             "Drive the value of a transform channel from an object or bone.",
             'DRIVER_TRANSFORM',
             0),
            ('SINGLE_PROP',
             "Single Property",
             "Drive the value of an RNA property, specified by a data-block reference and a path.",
             'RNA',
             1),
            ('SHAPE_KEY',
             "Shape Key",
             "Drive the value of a shape key.",
             'SHAPEKEY_DATA',
             2)
            ]
        )

    samples: PointerProperty(
        name="Poses",
        description="The sampled property data used by the RBF network",
        type=Samples
        )


class DrivenProperties(PropertyGroup):
    """
    Collection of driven properties
    """

    data__internal: CollectionProperty(type=DrivenProperty)

    def __getitem__(self, key: Union[int, slice]) -> Union[DrivenProperty, List[DrivenProperty]]:
        return self.data__internal[key]

    def __iter__(self) -> Iterator[DrivenProperty]:
        return iter(self.data__internal)

    def __len__(self) -> int:
        return len(self.data__internal)

    def find(self, name: str) -> int:
        """Returns the index of a key in a collection or -1 when not found."""
        return self.data__internal.find(name)

# endregion

# region Driver
###################################################################################################

def RBFDriver__mirror_property_value(driver: 'RBFDriver', name: str) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    if driver.xm__internal:
        mirror = get_mirror_rbf_driver(driver)
        if mirror is not None:
            mirror.xm__internal = False
            setattr(mirror, name, getattr(driver, name))
            mirror.xm__internal = True


def RBFDriver__is_valid__get(driver: 'RBFDriver') -> bool:
    # pylint: disable=missing-function-docstring
    # pylint: disable=invalid-name
    return (len(driver.poses) > 1
            and any(p.is_valid for p in driver.driven_properties)
            and any(chain(driver.use_location, driver.use_rotation, driver.use_scale)))


def RBFDriver__name__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    owner = driver.owner
    props = owner.rbf_drivers
    names = [p.name for p in props if p != driver]
    index = 0
    cache = driver.name
    value = cache

    while value in names:
        value = f'{cache}.{str(index).zfill(3)}'
        index += 1

    if cache != value:
        driver.name = value
        return

    cache = driver.name__internal
    driver.name__internal = value

    if cache != "" and driver.xm__internal:
        other = get_mirror_pose_bone(owner)
        if other is not None:
            mirror = other.rbf_drivers.get(cache)
            if mirror is not None:
                mirror.xm__internal = False
                mirror.name = value
                mirror.xm__internal = True


def RBFDriver__interpolation__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    driver.update()
    RBFDriver__mirror_property_value(driver, "interpolation")


def RBFDriver__dispersion__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if driver.interpolation != 'NONE':
        driver.update()
    RBFDriver__mirror_property_value(driver, "dispersion")


def RBFDriver__radius__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if driver.interpolation != 'NONE' and driver.dispersion == 'NONE':
        driver.update()
    RBFDriver__mirror_property_value(driver, "radius")


def RBFDriver__smoothing__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if driver.interpolation != 'NONE':
        driver.update()
    RBFDriver__mirror_property_value(driver, "smoothing")

def RBFDriver__mute__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=unused-variable
    # pylint: disable=invalid-name
    if driver.is_valid:
        bone = driver.owner
        mute = driver.mute
        for _, fcurve in network.drivers(bone, driver):
            fcurve.mute = mute
    RBFDriver__mirror_property_value(driver, "mute")


def RBFDriver__rotation_mode__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    if any(driver.use_rotation):
        owner = driver.owner
        network.destroy(owner, driver)
        if driver.is_valid:
            network.update(owner, driver)
    RBFDriver__mirror_property_value(driver, "rotation_mode")


def RBFDriver__use_location__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    owner = driver.owner
    network.destroy(owner, driver)
    if driver.is_valid:
        network.update(owner, driver)
    RBFDriver__mirror_property_value(driver, "use_location")


def RBFDriver__use_rotation__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    owner = driver.owner
    network.destroy(owner, driver)
    if driver.is_valid:
        network.update(owner, driver)
    RBFDriver__mirror_property_value(driver, "use_rotation")


def RBFDriver__use_scale__update(driver: 'RBFDriver', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    # pylint: disable=invalid-name
    owner = driver.owner
    network.destroy(owner, driver)
    if driver.is_valid:
        network.update(owner, driver)
    RBFDriver__mirror_property_value(driver, "use_scale")


class RBFDriver(PropertyGroup):
    """
    RBF Driver properties
    """

    id__internal: IntProperty(options={'HIDDEN'})
    xm__internal: BoolProperty(options={'HIDDEN'}, default=True)
    name__internal: StringProperty(options={'HIDDEN'})

    dispersion: EnumProperty(
        name="Dispersion",
        description="The measure of dispersion to use when computing the radius for the RBF",
        default='STD',
        options=set(),
        update=RBFDriver__dispersion__update,
        items=[
            ('NONE', "Custom", ""),
            ('MEAN', "Mean", ""),
            ('VAR', "Variance", ""),
            ('STD', "Standard Deviation", ""),
            ]
        )

    driven_properties: PointerProperty(
        name="Driven Properties",
        description="Target properties to be driven",
        type=DrivenProperties,
        options=set()
        )

    interpolation: EnumProperty(
        name="Interpolation",
        description="The radial basis function to use when interpolating poses",
        default='MULTI_QUADRATIC_BIHARMONIC',
        options=set(),
        update=RBFDriver__interpolation__update,
        items=[
            ('NONE', "Linear", ""),
            ('GAUSSIAN', "Gaussian", ""),
            ('MULTI_QUADRATIC_BIHARMONIC', "Multi-Quadratic Biharmonic", ""),
            ('INVERSE_MULTI_QUADRATIC_BIHARMONIC', "Inverse Multi-Quadratic Biharmonic", ""),
            ],
        )

    is_valid: BoolProperty(
        name="Valid",
        description="True if the RBF driver is valid, otherwise False",
        options=set(),
        get=RBFDriver__is_valid__get
        )

    mute: BoolProperty(
        name="Mute",
        description="Whether or not drivers are active (enable before adding poses)",
        default=True,
        options=set(),
        update=RBFDriver__mute__update,
        )

    name: StringProperty(
        name="Name",
        description="User-defined name for the driver (unique)",
        update=RBFDriver__name__update,
        options=set()
        )

    @property
    def owner(self) -> PoseBone:
        """The pose bone the rbf driver belongs to"""
        bones = self.id_data.pose.bones
        return next(b for b in bones if (b.is_property_set("rbf_drivers")
                                         and any(x == self for x in b.rbf_drivers)))

    poses: PointerProperty(
        name="Poses",
        description="Collection of poses",
        type=Poses,
        options=set()
        )

    radius: FloatProperty(
        name="Radius",
        description="Custom radius to use when interpolating input poses",
        default=1.0,
        options=set(),
        update=RBFDriver__radius__update,
        )

    rotation_mode: EnumProperty(
        name="Rotation Mode",
        description="Rotation channels to use for the driving pose bone",
        items=extract_enum_items(DriverTarget.bl_rna.properties['rotation_mode'].enum_items),
        options=set(),
        update=RBFDriver__rotation_mode__update,
        )

    smoothing: FloatProperty(
        name="Smoothing",
        description="Tighten or loosen the correspondance between input and output poses",
        min=0.0,
        default=0.0,
        options=set(),
        update=RBFDriver__smoothing__update
        )

    use_location: BoolVectorProperty(
        name="Location",
        description="Location channels to use from the driving pose bone",
        size=3,
        subtype='XYZ',
        default=(False, False, False),
        options=set(),
        update=RBFDriver__use_location__update,
        )

    use_rotation: BoolVectorProperty(
        name="Rotation",
        description="Rotation channels to use from the driving pose bone",
        size=4,
        subtype='QUATERNION',
        default=(False, False, False, False),
        options=set(),
        update=RBFDriver__use_rotation__update,
        )

    use_scale: BoolVectorProperty(
        name="Scale",
        description="Scale channels to use from the driving pose bone",
        size=3,
        subtype='XYZ',
        default=(False, False, False),
        options=set(),
        update=RBFDriver__use_scale__update,
        )

    def update(self) -> None:
        """Rebuild the RBF network for this driver"""
        if self.is_valid:
            network.update(self.owner, self)


class RBFDrivers(PropertyGroup):
    """
    Collection of RBF drivers
    """

    data__internal: CollectionProperty(type=RBFDriver)

    @property
    def active(self) -> Optional[RBFDriver]:
        """The currently active RBF driver"""
        try:
            return self[self.active_index]
        except IndexError:
            return None

    active_index: IntProperty(
        name="Active Index",
        description="The index of the currently active RBF driver",
        min=0,
        options=set()
        )

    def __contains__(self, name: str) -> bool:
        return name in self.data__internal

    def __getitem__(self, key: Union[int, slice]) -> Union[RBFDriver, List[RBFDriver]]:
        return self.data__internal[key]

    def __iter__(self) -> Iterator[RBFDriver]:
        return iter(self.data__internal)

    def __len__(self) -> int:
        return len(self.data__internal)

    def find(self, name: str) -> Optional[RBFDriver]:
        """Returns the index of a key in a collection or -1 when not found."""
        return self.data__internal.find(name)

    def get(self, name: str, default: Optional[object] = None) -> object:
        """Returns the value of the item assigned to key or default when not found"""
        return self.data__internal.get(name, default)

# endregion
