# pylint: disable=missing-module-docstring
from typing import List
from contextlib import suppress
from dataclasses import dataclass, field
from bpy.types import Driver, DriverTarget, DriverVariable, ID

@dataclass
class DriverTargetProxy:
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    id_type: str = 'OBJECT'
    id: ID = None
    data_path: str = ''
    bone_target: str = ''
    transform_type: str = 'LOC_X'
    rotation_mode: str = 'AUTO'
    transform_space: str = 'WORLD_SPACE'

    def apply(self, target: DriverTarget) -> None:
        # pylint: disable=missing-function-docstring
        with suppress(AttributeError):
            target.id_type = self.id_type
        target.id = self.id
        target.data_path = self.data_path
        target.bone_target = self.bone_target
        target.transform_type = self.transform_type
        target.rotation_mode = self.rotation_mode
        target.transform_space = self.transform_space

@dataclass
class DriverVariableProxy:
    # pylint: disable=missing-class-docstring
    type: str = 'SINGLE_PROP'
    name: str = ''
    targets: List[DriverTargetProxy] = field(default_factory=lambda: [DriverTargetProxy()])

    def apply(self, variable: DriverVariable) -> None:
        # pylint: disable=missing-function-docstring
        variable.type = self.type
        variable.name = self.name
        for target, proxy in zip(variable.targets, self.targets):
            proxy.apply(target)

@dataclass
class DriverProxy:
    # pylint: disable=missing-class-docstring
    type: str = 'AVERAGE'
    expression: str = ''
    variables: List[DriverVariableProxy] = field(default_factory=list)

    def apply(self, driver: Driver) -> None:
        # pylint: disable=missing-function-docstring
        driver.type = self.type
        driver.expression = self.expression
        variables = driver.variables
        while len(variables) > 0:
            variables.remove(variables[-1])
        for proxy in self.variables:
            proxy.apply(variables.new())

@dataclass
class FCurveProxy:
    # pylint: disable=missing-class-docstring
    data_path: str = ''
    array_index: int = 0
    mute: bool = False
    driver: DriverProxy = field(default_factory=DriverProxy)

    def apply(self, data: ID) -> None:
        # pylint: disable=missing-function-docstring
        # fcurve = data.driver_add(self.data_path, self.array_index)
        drivers = data.animation_data_create().drivers
        path = self.data_path
        index = self.array_index

        fcurve = drivers.find(path, index=index)
        if fcurve is None:
            fcurve = drivers.new(path, index=index)

        fcurve.mute = self.mute
        self.driver.apply(fcurve.driver)

@dataclass
class AnimDataProxy:
    # pylint: disable=missing-class-docstring
    id_data: ID = None
    drivers: List[FCurveProxy] = field(default_factory=list)

    def apply(self) -> None:
        # pylint: disable=missing-function-docstring
        for fcurve in self.drivers:
            fcurve.apply(self.id_data)
