# pylint: disable=missing-module-docstring
from typing import Any, Optional, Tuple, Union
from contextlib import suppress
from bpy.types import Bone, ID, PoseBone
from mathutils import Euler, Quaternion, Vector, Matrix

Owner = Union[Bone, ID, PoseBone]
Scalar = Union[int, bool, float]
SCALAR = (int, bool, float)
AxisAngle = Tuple[float, float, float, float]
Rotation = Union[Euler, Quaternion, AxisAngle]

def flatten_matrix(matrix: Matrix) -> Tuple[float, ...]:
    # pylint: disable=missing-function-docstring
    return sum((matrix.col[i].to_tuple() for i in range(4)), tuple())

def compose_matrix(location: Vector, rotation: Quaternion, scale: Vector) -> Matrix:
    # pylint: disable=missing-function-docstring
    matrix = Matrix.Identity(3)
    matrix[0][0] = scale[0]
    matrix[1][1] = scale[1]
    matrix[2][2] = scale[2]

    matrix = (rotation.to_matrix() @ matrix).to_4x4()
    matrix[0][3] = location[0]
    matrix[1][3] = location[1]
    matrix[2][3] = location[2]

    return matrix

def euler_to_quaternion(euler: Euler) -> Quaternion:
    # pylint: disable=missing-function-docstring
    return euler.to_quaternion()

def euler_to_axis_angle(euler: Euler) -> AxisAngle:
    # pylint: disable=missing-function-docstring
    quaternion_to_axis_angle(euler.to_quaternion())

def quaternion_to_euler(quaternion: Quaternion, order: Optional[str] = 'XYZ') -> Euler:
    # pylint: disable=missing-function-docstring
    return quaternion.to_euler(order)

def quaternion_to_axis_angle(quaternion: Quaternion) -> AxisAngle:
    # pylint: disable=missing-function-docstring
    axis, angle = quaternion.to_axis_angle()
    return (angle, axis[0], axis[1], axis[2])

def axis_angle_to_euler(axis_angle: AxisAngle, order: Optional[str] = 'XYZ') -> Euler:
    # pylint: disable=missing-function-docstring
    return axis_angle_to_quaternion(axis_angle).to_euler(order)

def axis_angle_to_quaternion(axis_angle: AxisAngle) -> Quaternion:
    # pylint: disable=missing-function-docstring
    return Quaternion(axis_angle[1:], axis_angle[0])

def quaternion_to_swing_twist(quaternion: Quaternion, axis: str) -> Quaternion:
    # pylint: disable=missing-function-docstring
    result, angle = quaternion.to_swing_twist(axis)
    result['XYZ'.index(axis)+1] = angle
    return result

def euler_to_swing_twist(euler: Euler, axis: str) -> Quaternion:
    # pylint: disable=missing-function-docstring
    return quaternion_to_swing_twist(euler_to_quaternion(euler), axis)

def axis_angle_to_swing_twist(axis_angle: AxisAngle, axis: str) -> Quaternion:
    # pylint: disable=missing-function-docstring
    return quaternion_to_swing_twist(axis_angle_to_quaternion(axis_angle), axis)

_EULER_CONVERSIONS = {
    'XYZ': lambda euler: Euler(euler, 'XYZ'),
    'XZY': lambda euler: Euler(euler, 'XZY'),
    'YXZ': lambda euler: Euler(euler, 'YXZ'),
    'YZX': lambda euler: Euler(euler, 'YZX'),
    'ZXY': lambda euler: Euler(euler, 'ZXY'),
    'ZYX': lambda euler: Euler(euler, 'ZYX'),
    'QUATERNION': euler_to_quaternion,
    'AXIS_ANGLE': euler_to_axis_angle,
    'SWING_TWIST_X': lambda euler: euler_to_swing_twist(euler, 'X'),
    'SWING_TWIST_Y': lambda euler: euler_to_swing_twist(euler, 'Y'),
    'SWING_TWIST_Z': lambda euler: euler_to_swing_twist(euler, 'Z'),
    }

ROTATION_CONVERSION_TABLE = {
    'AXIS_ANGLE': {
        'XYZ': lambda axis_angle: axis_angle_to_euler(axis_angle, 'XYZ'),
        'XZY': lambda axis_angle: axis_angle_to_euler(axis_angle, 'XZY'),
        'YXZ': lambda axis_angle: axis_angle_to_euler(axis_angle, 'YXZ'),
        'YZX': lambda axis_angle: axis_angle_to_euler(axis_angle, 'YZX'),
        'ZXY': lambda axis_angle: axis_angle_to_euler(axis_angle, 'ZXY'),
        'ZYX': lambda axis_angle: axis_angle_to_euler(axis_angle, 'ZYX'),
        'QUATERNION': axis_angle_to_euler,
        'AXIS_ANGLE': lambda axis_angle: axis_angle,
        'SWING_TWIST_X': lambda axis_angle: axis_angle_to_swing_twist(axis_angle, 'X'),
        'SWING_TWIST_Y': lambda axis_angle: axis_angle_to_swing_twist(axis_angle, 'Y'),
        'SWING_TWIST_Z': lambda axis_angle: axis_angle_to_swing_twist(axis_angle, 'Z'),
        },
    'QUATERNION': {
        'XYZ': lambda quaternion: quaternion_to_euler(quaternion, 'XYZ'),
        'XZY': lambda quaternion: quaternion_to_euler(quaternion, 'XZY'),
        'YXZ': lambda quaternion: quaternion_to_euler(quaternion, 'YXZ'),
        'YZX': lambda quaternion: quaternion_to_euler(quaternion, 'YZX'),
        'ZXY': lambda quaternion: quaternion_to_euler(quaternion, 'ZXY'),
        'ZYX': lambda quaternion: quaternion_to_euler(quaternion, 'ZYX'),
        'AXIS_ANGLE': quaternion_to_axis_angle,
        'QUATERNION': lambda quaternion: quaternion,
        'SWING_TWIST_X': lambda quaternion: quaternion_to_swing_twist(quaternion, 'X'),
        'SWING_TWIST_Y': lambda quaternion: quaternion_to_swing_twist(quaternion, 'Y'),
        'SWING_TWIST_Z': lambda quaternion: quaternion_to_swing_twist(quaternion, 'Z'),
        },
    'XYZ': _EULER_CONVERSIONS,
    'XZY': _EULER_CONVERSIONS,
    'YXZ': _EULER_CONVERSIONS,
    'YZX': _EULER_CONVERSIONS,
    'ZXY': _EULER_CONVERSIONS,
    'ZYX': _EULER_CONVERSIONS,
    }

def convert_rotation(rotation: Rotation, from_mode: str, to_mode: str) -> Rotation:
    # pylint: disable=missing-function-docstring
    return ROTATION_CONVERSION_TABLE[from_mode][to_mode](rotation)

class itemsetter:
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    # pylint: disable=too-few-public-methods
    __slots__ = ("key",)

    def __init__(self, key: Union[int, str]) -> None:
        self.key = key

    def __call__(self, data: object, value: Any) -> None:
        data[self.key] = value

class attrsetter:
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    # pylint: disable=too-few-public-methods
    __slots__ = ("key",)

    def __init__(self, key: Union[int, str]) -> None:
        self.key = key

    def __call__(self, data: object, value: Any) -> None:
        setattr(data, self.key, value)

def getpath(owner: ID, path: str, index: Optional[int] = 0) -> Optional[Scalar]:
    # pylint: disable=missing-function-docstring
    value = owner.path_resolve(path)

    if isinstance(value, SCALAR):
        return value

    with suppress(TypeError, KeyError, IndexError, ValueError):
        value = value[index]

    if not isinstance(value, SCALAR):
        raise ValueError()

    return value

def setpath(owner: Owner,
            path: str,
            index: Optional[Scalar] = -1,
            value: Optional[Scalar] = None) -> None:
    # pylint: disable=missing-function-docstring
    if value is None:
        value = index
        index = -1

    if index >= 0:
        setter = itemsetter(index)
    elif path.endswith('"]'):
        index = path.rfind('["')
        setter = itemsetter(path[index+2:-2])
        path = path[:index]
    elif path.endswith(']'):
        index = path.rfind('[')
        setter = itemsetter(int(path[index+1:-1]))
        path = path[:index]
    else:
        index = path.rfind('.')
        setter = attrsetter(path[index+1:])
        path = path[:index]

    data = owner.path_resolve(path) if path else owner
    setter(data, value)

class Keygen:
    # pylint: disable=missing-class-docstring
    # pylint: disable=too-few-public-methods

    def __init__(self) -> None:
        self.index = 0
        self.value = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'
        self.count = len(self.value)

    def __call__(self) -> str:
        index = self.index
        value = self.value
        count = self.count
        self.index = index + 1
        return value[index] if index < count else f'{value[index%count]}{int(index/count)}'

    def reset(self) -> None:
        # pylint: disable=missing-function-docstring
        self.index = 0

# region MIRRORING
###################################################################################################

MIRROR_SUFFIX_LUT = {
    ".L": ".R",
    ".R": ".L",
    "_L": "_R",
    "_R": "_L",
    ".l": ".r",
    ".r": ".l",
    "_l": "_r",
    "_r": "_l",
    ".left": ".right",
    ".right": ".left",
    "_left": "_right",
    "_right": "_left",
    ".LEFT": ".RIGHT",
    ".RIGHT": ".LEFT",
    "_LEFT": "_RIGHT",
    "_RIGHT": "_LEFT",
    }

MIRROR_SUFFIXES = MIRROR_SUFFIX_LUT.keys()

def get_mirror_suffix_for_name(name: str) -> str:
    # pylint: disable=missing-function-docstring
    for suffix in MIRROR_SUFFIXES:
        if name.endswith(suffix):
            return suffix
    return ""

def set_mirror_suffix_for_name(name: str, suffix: str) -> str:
    # pylint: disable=missing-function-docstring
    return name[:len(name)-len(suffix)] + MIRROR_SUFFIX_LUT[suffix]

def get_mirror_suffix(pose_bone: PoseBone) -> str:
    # pylint: disable=missing-function-docstring
    return get_mirror_suffix_for_name(pose_bone.name)

def remove_mirror_suffix_for_name(name: str, suffix: Optional[str] = "") -> str:
    # pylint: disable=missing-function-docstring
    if suffix == "":
        suffix = get_mirror_suffix_for_name(name)
    if suffix:
        name = name[:len(name)-len(suffix)]
    return name

def get_mirror_pose_bone(pose_bone: PoseBone) -> Optional[PoseBone]:
    # pylint: disable=missing-function-docstring
    pose = pose_bone.id_data.pose
    if pose.use_mirror_x:
        suffix = get_mirror_suffix(pose_bone)
        if suffix:
            mirror = set_mirror_suffix_for_name(pose_bone.name, suffix)
            return pose_bone.id_data.pose.bones.get(mirror)
    return None

#  endregion
