# pylint: disable=missing-module-docstring
# pylint: disable=too-many-lines
from typing import Optional, Sequence, Set, Tuple, Union
from contextlib import suppress
from operator import attrgetter
from bpy.types import (AnimData,
                       Context,
                       Event,
                       ID,
                       Object,
                       Operator,
                       PoseBone,
                       PropertyGroup,
                       UILayout,
                       UIList)
from bpy.props import (BoolProperty,
                       BoolVectorProperty,
                       CollectionProperty,
                       EnumProperty,
                       FloatVectorProperty,
                       IntProperty,
                       StringProperty)
from mathutils import Euler
from .properties import (Poses,
                         Sample,
                         RBFDriver,
                         get_mirror_pose_bone_and_rbf_driver,
                         get_mirror_rbf_driver)
from . import network
from .utils import (get_mirror_suffix_for_name,
                    set_mirror_suffix_for_name,
                    get_mirror_pose_bone,
                    remove_mirror_suffix_for_name,
                    getpath,
                    setpath,
                    flatten_matrix,
                    compose_matrix,
                    convert_rotation,
                    euler_to_quaternion)

BUFFER = None
EXPLEN_MSG = ("RBF Driver expression is too long. "
              "You will need to reduce the number "
              "of poses, inputs, or driven properties")

# region MIXINS
###################################################################################################

def format_influence_idpropkey(id_: int, name: str) -> str:
    # pylint: disable=missing-function-docstring
    return f'RBF Driver Pose {name.title()} - rbfn.{str(id_).zfill(3)}'

def influence_idpropkey_for(driver: RBFDriver, name: str) -> str:
    # pylint: disable=missing-function-docstring
    return format_influence_idpropkey(driver.id__internal, name)

def create_influence_idprop(datablock: Union[ID, PoseBone], propname: str, poses: Poses) -> None:
    # pylint: disable=missing-function-docstring
    rna_ui = datablock.get('_RNA_UI')

    if rna_ui is None:
        datablock['_RNA_UI'] = {}
        rna_ui = datablock['_RNA_UI']

    if propname not in rna_ui:
        rna_ui[propname] = {}

    mdata = rna_ui[propname]
    mdata["min"] = -10000.0
    mdata["max"] = 10000.0
    mdata["soft_min"] = 0.0
    mdata["soft_max"] = 1.0
    mdata["default"] = 1.0

    datablock[propname] = [1.0 for _ in poses]

def remove_influence_idprop(datablock: Union[ID, PoseBone], propname: str) -> None:
    # pylint: disable=missing-function-docstring
    rna_ui = datablock.get('_RNA_UI')

    if rna_ui is not None and propname in rna_ui:
        del rna_ui[propname]

    if propname in datablock:
        del datablock[propname]

def splice_influence_idprop(datablock: Union[ID, PoseBone],
                            animdata: Optional[AnimData],
                            propname: str,
                            proppath: str,
                            index: int,
                            swap: Optional[int] = None) -> None:
    # pylint: disable=missing-function-docstring
    idprop = datablock.get(propname)

    if idprop is not None and len(idprop) > index:

        value = idprop.to_list()
        if swap is None:
            del value[index]
        else:
            element = value[index]
            value[index] = value[swap]
            value[swap] = element
        datablock[propname] = value

        if animdata and proppath:
            for fcurve in [d for d in animdata.drivers if d.data_path == proppath]:
                idx = fcurve.array_index
                if swap is None:
                    if idx == index:
                        animdata.drivers.remove(fcurve)
                    elif idx > index:
                        fcurve.array_index = idx - 1
                else:
                    if idx == index:
                        fcurve.array_index = swap
                    elif idx == swap:
                        fcurve.array_index = index

# endregion

# region MIXINS
###################################################################################################

class ActivePoseBone:
    # pylint: disable=missing-class-docstring
    # pylint: disable=too-few-public-methods
    @classmethod
    def poll(cls, context: Context) -> bool:
        # pylint: disable=missing-function-docstring
        return context.active_pose_bone is not None


class ActiveDriver(ActivePoseBone):
    # pylint: disable=missing-class-docstring
    # pylint: disable=too-few-public-methods
    @classmethod
    def poll(cls, context: Context) -> bool:
        # pylint: disable=missing-function-docstring
        return (super().poll(context)
                and context.active_pose_bone.is_property_set("rbf_drivers")
                and context.active_pose_bone.rbf_drivers.active is not None)


class ActivePose(ActiveDriver):
    # pylint: disable=missing-class-docstring
    # pylint: disable=too-few-public-methods
    @classmethod
    def poll(cls, context: Context) -> bool:
        # pylint: disable=missing-function-docstring
        return (super().poll(context)
                and context.active_pose_bone.rbf_drivers.active.poses.active is not None)


class PasteDriver(ActivePoseBone):
    # pylint: disable=missing-class-docstring
    # pylint: disable=too-few-public-methods
    @classmethod
    def poll(cls, context: Context) -> bool:
        return super().poll(context) and BUFFER is not None

    @staticmethod
    def process(context: Context, buffer: dict) -> RBFDriver:
        # pylint: disable=missing-function-docstring
        pose_bone = context.active_pose_bone
        driver = RBFDRIVERS_OT_driver_new.process(pose_bone, buffer["name"])
        driver.xm__internal = False

        for key in ("mute",
                    "dispersion", "interpolation", "radius", "smoothing",
                    "use_location", "use_rotation", "use_scale", "rotation_mode",
                    ):
            setattr(driver, key, buffer[key])

        props = driver.driven_properties.data__internal
        for bprop in buffer["driven_properties"]:
            dprop = props.add()
            for key in ("name", "type", "show_expanded",
                        "id", "bone_target", "data_path",
                        "rotation_mode", "transform_type", "shape_key",
                        ):
                setattr(dprop, key, bprop[key])

            samples = dprop.samples.data__internal
            for bsample in bprop["samples"]:
                samples.add().value = bsample["value"]

        poses = driver.poses
        for bpose in buffer["poses"]:
            dpose = poses.data__internal.add()
            for key in ("matrix", "name"):
                setattr(dpose, key, bpose[key])

        if len(poses) > 0:
            for datablock, name in [
                    (pose_bone, 'Weight'),
                    (pose_bone.id_data.data, 'Effect')
                ]:
                propname = influence_idpropkey_for(driver, name)
                create_influence_idprop(datablock, propname, poses)

        driver.xm__internal = True
        return driver

# endregion

# region DRIVER OPERATORS
###################################################################################################

class RBFDRIVERS_OT_driver_add(ActivePoseBone, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_add"
    bl_label = "Add RBF Driver"
    bl_description = "Add an RBF driver"
    bl_options = {'REGISTER', 'UNDO'}

    @staticmethod
    def draw(popup, context: Context) -> None:
        # pylint: disable=unused-argument
        # pylint: disable=missing-function-docstring
        layout = popup.layout
        layout.operator("rbf_drivers.driver_new", text="New")
        layout.operator("rbf_drivers.driver_paste", text="Paste")
        layout.operator("rbf_drivers.driver_paste_symmetrical", text="Paste Symmetrical")

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        context.window_manager.popup_menu(self.draw, title="Add RBF Driver")
        return {'FINISHED'}


class RBFDRIVERS_OT_driver_new(ActivePoseBone, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_new"
    bl_label = "New"
    bl_description = "Add an RBF driver"
    bl_options = {'REGISTER', 'UNDO'}

    @staticmethod
    def process(pose_bone: PoseBone, name: Optional[str] = "RBF Driver") -> RBFDriver:
        # pylint: disable=missing-function-docstring
        drivers = []
        for bone in pose_bone.id_data.pose.bones:
            if bone.is_property_set("rbf_drivers"):
                drivers.extend(bone.rbf_drivers)

        def key(driver) -> int:
            return driver.id__internal

        drivers.sort(key=key)
        netid = next((i for i, x in enumerate(drivers) if key(x) != i), len(drivers))

        drivers = pose_bone.rbf_drivers
        index = len(drivers)

        driver = drivers.data__internal.add()
        driver.name = name
        driver.id__internal = netid

        if len(pose_bone.rotation_mode) == 3:
            driver.rotation_mode = 'AUTO'
        else:
            driver.rotation_mode = 'QUATERNION'

        drivers.active_index = index
        return driver

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        bone = context.active_pose_bone
        self.process(bone)

        bone = get_mirror_pose_bone(bone)
        if bone is not None:
            self.process(bone)

        return {'FINISHED'}


class RBFDRIVERS_OT_driver_copy(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_copy"
    bl_label = "Copy RBF Driver"
    bl_description = "Add the specified RBF driver to the copy/paste buffer"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=global-statement
        global BUFFER
        driver = context.active_pose_bone.rbf_drivers.active
        BUFFER = {
            "dispersion": driver.dispersion,
            "driven_properties": [
                {
                    "bone_target": prop.bone_target,
                    "data_path": prop.data_path,
                    "id": prop.id,
                    "name": prop.name,
                    "rotation_mode": prop.rotation_mode,
                    "shape_key": prop.shape_key,
                    "show_expanded": prop.show_expanded,
                    "transform_type": prop.transform_type,
                    "type": prop.type,
                    "samples": [
                        {
                            "value": sample.value
                        } for sample in prop.samples
                    ]
                } for prop in driver.driven_properties],
            "interpolation": driver.interpolation,
            "mute": driver.mute,
            "name": driver.name,
            "poses": [
                {
                    "matrix": flatten_matrix(pose.matrix),
                    "name": pose.name
                } for pose in driver.poses
            ],
            "radius": driver.radius,
            "rotation_mode": driver.rotation_mode,
            "smoothing": driver.smoothing,
            "use_location": tuple(driver.use_location),
            "use_rotation": tuple(driver.use_rotation),
            "use_scale": tuple(driver.use_scale),
        }
        return {'FINISHED'}


class RBFDRIVERS_OT_driver_paste(PasteDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_paste"
    bl_label = "Paste RBF Driver"
    bl_description = "Add the specified RBF driver from the copy/paste buffer"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        driver = self.process(context, BUFFER)
        if driver.is_valid:
            try:
                network.update(context.active_pose_bone, driver)
            except network.ExpressionLengthError:
                self.report({'ERROR'}, EXPLEN_MSG)
                return {'CANCELLED'}
        return {'FINISHED'}


class RBFDRIVERS_OT_driver_paste_symmetrical(PasteDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_paste_symmetrical"
    bl_label = "Paste Symmetrical RBF Driver"
    bl_description = "Add the specified RBF driver from the copy/paste buffer, using symmetry"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        buffer = BUFFER.copy()
        buffer["driven_properties"] = [p.copy() for p in BUFFER["driven_properties"]]

        for prop in buffer["driven_properties"]:

            if prop["type"] == 'SHAPE_KEY':
                target = prop["shape_key"]
                suffix = get_mirror_suffix_for_name(target)
                if suffix:
                    prop["shape_key"] = set_mirror_suffix_for_name(target, suffix)

            elif prop["type"] == 'TRANSFORMS' and prop["bone_target"] != "":
                target = prop["bone_target"]
                suffix = get_mirror_suffix_for_name(target)
                if suffix:
                    prop["bone_target"] = set_mirror_suffix_for_name(target, suffix)

        driver = self.process(context, buffer)
        if driver.is_valid:
            try:
                network.update(context.active_pose_bone, driver)
            except network.ExpressionLengthError:
                self.report({'ERROR'}, EXPLEN_MSG)
                return {'CANCELLED'}
        return {'FINISHED'}


class RBFDRIVERS_OT_driver_remove(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_remove"
    bl_label = "Remove RBF Driver"
    bl_description = "Remove the specified RBF driver"
    bl_options = {'REGISTER', 'UNDO'}

    @staticmethod
    def process(pose_bone: PoseBone, index: int) -> None:
        # pylint: disable=missing-function-docstring
        drivers = pose_bone.rbf_drivers

        driver = drivers[index]
        if driver.is_valid:
            network.destroy(pose_bone, driver)

        for datablock, name in [
                (pose_bone, 'Weight'),
                (pose_bone.id_data.data, 'Effect')
                ]:
            propname = influence_idpropkey_for(driver, name)
            remove_influence_idprop(datablock, propname)

        drivers.data__internal.remove(index)

        if len(drivers) == 0:
            pose_bone.property_unset("rbf_drivers")
        elif index == len(drivers):
            drivers.active_index = index - 1

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        pose_bone = context.active_pose_bone
        mirr_bone = get_mirror_pose_bone(pose_bone)

        if mirr_bone is not None:
            index = mirr_bone.rbf_drivers.find(pose_bone.rbf_drivers.active.name)
        else:
            index = -1

        self.process(pose_bone, pose_bone.rbf_drivers.active_index)

        if index >= 0:
            self.process(mirr_bone, index)

        return {'FINISHED'}


class RBFDRIVERS_OT_driver_move_up(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_move_up"
    bl_label = "Move up"
    bl_description = "Move the active item up within the list"
    bl_options = {'INTERNAL'}

    @classmethod
    def poll(cls, context: Context) -> bool:
        # pylint: disable=missing-function-docstring
        return (super().poll(context)
                and context.active_pose_bone.rbf_drivers.active_index > 0)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        drivers = context.active_pose_bone.rbf_drivers
        index = drivers.active_index
        drivers.data__internal.move(index, index - 1)
        drivers.active_index -= 1

        return {'FINISHED'}


class RBFDRIVERS_OT_driver_move_down(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_move_down"
    bl_label = "Move down"
    bl_description = "Move the active item down within the list"
    bl_options = {'INTERNAL'}

    @classmethod
    def poll(cls, context: Context) -> bool:
        # pylint: disable=missing-function-docstring
        if super().poll(context):
            drivers = context.active_pose_bone.rbf_drivers
            return len(drivers) - 1 > drivers.active_index
        return False

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        drivers = context.active_pose_bone.rbf_drivers
        index = drivers.active_index
        drivers.data__internal.move(index, index + 1)
        drivers.active_index += 1

        return {'FINISHED'}


class RBFDRIVERS_OT_driver_update(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driver_update"
    bl_label = "Update"
    bl_description = "Rebuild the network for the specified RBF driver"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context: Context) -> bool:
        # pylint: disable=missing-function-docstring
        return super().poll(context) and context.active_pose_bone.rbf_drivers.active.is_valid

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        target = context.active_pose_bone
        driver = target.rbf_drivers.active

        try:
            network.update(target, driver)
        except network.ExpressionLengthError:
            self.report({'ERROR'}, EXPLEN_MSG)
            return {'CANCELLED'}

        target, driver = get_mirror_pose_bone_and_rbf_driver(driver)
        if driver:
            try:
                network.update(target, driver)
            except network.ExpressionLengthError:
                self.report({'ERROR'}, EXPLEN_MSG)
                return {'CANCELLED'}

        return {'FINISHED'}

# endregion

# region DRIVEN PROPERTY OPERATORS
###################################################################################################

class RBFDRIVERS_OT_driven_properties_add(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driven_properties_add"
    bl_label = "Add Driven Properties"
    bl_description = "Add a driven properties to the rbf driver"
    bl_options = {'INTERNAL', 'UNDO'}

    @staticmethod
    def draw(popup, context: Context) -> None:
        # pylint: disable=unused-argument
        # pylint: disable=missing-function-docstring
        layout = popup.layout
        layout.operator_context = 'INVOKE_DEFAULT'
        layout.operator("rbf_drivers.driven_property_add", text="Single Property")
        layout.operator("rbf_drivers.driven_property_add_transforms", text="Transforms")
        layout.operator("rbf_drivers.driven_property_add_shapekeys", text="Shape Keys")

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        context.window_manager.popup_menu(self.draw, title="Add Driven Properties")
        return {'FINISHED'}


class RBFDRIVERS_OT_driven_property_add(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driven_property_add"
    bl_label = "Add Driven Property"
    bl_description = "Add a driven property to the rbf driver"
    bl_options = {'INTERNAL', 'UNDO'}

    @staticmethod
    def process(driver: RBFDriver, name: str = "Driven Property") -> None:
        # pylint: disable=missing-function-docstring
        props = driver.driven_properties
        poses = driver.poses

        prop = props.data__internal.add()
        prop.name = name

        for _ in poses:
            prop.samples.data__internal.add()

        return prop

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        # pylint: disable=unused-variable
        driver = context.active_pose_bone.rbf_drivers.active
        drprop = self.process(driver)

        driver = get_mirror_rbf_driver(driver)
        if driver is not None:
            self.process(driver, drprop.name)

        return {'FINISHED'}

def update_rotation_mode(operator: 'RBFDRIVERS_OT_driven_property_add_transforms',
                         context: Context) -> None:
    # pylint: disable=missing-function-docstring
    # pylint: disable=unused-argument
    target = operator.object
    if target:
        target = context.scene.objects.get(target)
        if target is not None and target.type == 'ARMATURE' and operator.bone_target:
            target = target.pose.bones.get(operator.bone_target)
        if target is not None:
            mode = target.rotation_mode
            operator.rotation_mode = 'EULER' if len(mode) == 3 or mode == 'AUTO' else mode


class ShapeKeyBlock(PropertyGroup):
    # pylint: disable=missing-class-docstring
    # pylint: disable=too-few-public-methods
    use: BoolProperty(
        name="Selected",
        description="",
        default=False,
        options=set()
        )


class ShapeKeyedMesh(PropertyGroup):
    # pylint: disable=missing-class-docstring
    # pylint: disable=too-few-public-methods
    # name
    pass


def update_keys(operator: 'RBFDRIVERS_OT_driven_property_add_shapekeys', context: Context) -> None:
    # pylint: disable=missing-function-docstring
    operator.active_index = 0
    mesh = operator.object
    keys = operator.keys
    keys.clear()
    if mesh:
        mesh = context.scene.objects[mesh]
        for key in mesh.data.shape_keys.key_blocks.keys():
            keys.add().name = key


class RBFDRIVERS_UL_available_keys(UIList):
    # pylint: disable=invalid-name
    # pylint: disable=missing-class-docstring
    # pylint: disable=too-few-public-methods
    bl_idname = 'RBFDRIVERS_UL_available_keys'

    def draw_item(self, context, layout, data, item, icon, active_data, active_prop) -> None:
        # pylint: disable=no-self-use
        # pylint: disable=too-many-arguments
        # pylint: disable=unused-argument
        # pylint: disable=missing-function-docstring
        row = layout.row(align=True)
        row.label(text=item.name)
        row.prop(item, "use",
                 text="",
                 icon=f'CHECKBOX_{"" if item.use else "DE"}HLT',
                 emboss=False,
                 icon_only=True)


class RBFDRIVERS_OT_driven_property_add_shapekeys(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driven_property_add_shapekeys"
    bl_label = "Add Shape Keys"
    bl_description = "Add driven shape key properties to the rbf driver"
    bl_options = {'INTERNAL', 'UNDO'}

    active_index: IntProperty(default=0, options=set())

    object: StringProperty(
        name="Object",
        description="The mesh object for the driven shape keys",
        update=update_keys,
        options=set(),
        )

    meshes: CollectionProperty(
        type=ShapeKeyedMesh,
        options=set()
        )

    keys: CollectionProperty(
        name="Keys",
        type=ShapeKeyBlock,
        description="",
        options=set()
        )

    def invoke(self, context: Context, layout: UILayout) -> None:
        # pylint: disable=missing-function-docstring
        # pylint: disable=unused-argument
        meshes = self.meshes
        meshes.clear()
        for object_ in context.scene.objects:
            if object_.type == 'MESH' and object_.data.shape_keys is not None:
                meshes.add().name = object_.name
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context: Context) -> None:
        # pylint: disable=missing-function-docstring
        # pylint: disable=unused-argument
        layout = self.layout
        layout.separator()

        row = layout.row()
        row.prop_search(self, "object", self, "meshes", icon='MESH_DATA')
        row.alert = self.object == "" or self.object not in self.meshes

        if not row.alert and len(self.keys) > 0:
            layout.separator()
            layout.template_list('RBFDRIVERS_UL_available_keys', "",
                                 self, "keys", self, "active_index", rows=6)

        layout.separator()

    def _build_spec(self, context: Context) -> Tuple[Optional[ID], list]:
        keys = []
        mesh = context.scene.objects.get(self.object)
        if mesh:
            keys.extend(k.name for k in self.keys if k.use)
        return mesh, keys

    def _apply_spec(self, target: PoseBone, driver: RBFDriver, mesh: ID, keys: list) -> None:
        driver.xm__internal = False

        props = driver.driven_properties
        poses = driver.poses

        for key in keys:
            prop = props.data__internal.add()

            for _ in poses:
                prop.samples.data__internal.add()

            prop.name = f'{remove_mirror_suffix_for_name(key)} ({mesh.name})'
            prop.type = 'SHAPE_KEY'
            prop.shape_key = key
            prop.id = mesh

        driver.xm__internal = True

        if driver.is_valid:
            network.update(target, driver)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        # pylint: disable=unused-argument
        mesh, keys = self._build_spec(context)
        if mesh and keys:
            target = context.active_pose_bone
            driver = target.rbf_drivers.active
            self._apply_spec(target, driver, mesh, keys)

            target, mirror = get_mirror_pose_bone_and_rbf_driver(driver)
            if mirror:
                keys_ = []
                for key in keys:
                    suffix = get_mirror_suffix_for_name(key)
                    if suffix:
                        key = set_mirror_suffix_for_name(key, suffix)
                    keys_.append(key)

                self._apply_spec(target, mirror, mesh, keys_)

        return {'FINISHED'}


class RBFDRIVERS_OT_driven_property_add_transforms(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driven_property_add_transforms"
    bl_label = "Add Transforms"
    bl_description = "Add driven transform properties to the rbf driver"
    bl_options = {'INTERNAL', 'UNDO'}

    bone_target: StringProperty(
        name="Bone",
        description="The name of the pose bone to which the driven properties belong",
        default="",
        options=set(),
        update=update_rotation_mode,
        )

    object: StringProperty(
        name="Object",
        description="The object for the driven properties",
        default="",
        update=update_rotation_mode,
        options=set(),
        )

    rotation_mode: EnumProperty(
        name="Mode",
        description="The rotation channels to be driven",
        default='EULER',
        options=set(),
        items=[
            ('EULER', "Euler", ""),
            ('AXIS_ANGLE', "Axis/Angle", ""),
            ('QUATERNION', "Quaternion", ""),
            ]
        )

    use_location: BoolVectorProperty(
        name="Location",
        description="Location channels to be driven",
        size=3,
        subtype='XYZ',
        default=(False, False, False),
        options=set(),
        )

    use_rotation: BoolVectorProperty(
        name="Rotation",
        description="Rotation channels to be driven",
        size=4,
        subtype='QUATERNION',
        default=(False, False, False, False),
        options=set(),
        )

    use_scale: BoolVectorProperty(
        name="Scale",
        description="Scale channels to be driven",
        size=3,
        subtype='XYZ',
        default=(False, False, False),
        options=set(),
        )

    def invoke(self, context: Context, layout: UILayout) -> None:
        # pylint: disable=missing-function-docstring
        # pylint: disable=unused-argument
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context: Context) -> None:
        # pylint: disable=missing-function-docstring
        scene = context.scene
        layout = self.layout
        object_ = self.object
        objects = scene.objects

        layout.separator()

        row = layout.row()
        row.alert = object_ == "" or object_ not in objects
        row.prop_search(self, "object", scene, "objects")

        if object_:
            object_ = objects.get(object_)
            if object_ is not None:

                if object_.type == 'ARMATURE':
                    bone = self.bone_target
                    data = object_.data
                    row = layout.row()
                    row.alert = bone != "" and bone not in data.bones
                    row.prop_search(self, "bone_target", data, "bones")

                layout.separator()

                row = layout.row(align=True)

                col = row.column()
                col.alignment = 'RIGHT'
                spl = col.row(align=True).split(factor=0.75)

                lft = spl.column()
                rgt = spl.column()
                lft.alignment = 'RIGHT'

                lft.label(text="Location X")
                rgt.prop(self, "use_location", text="", index=0)
                for index, axis in zip(range(1, 3), 'YZ'):
                    lft.label(text=axis)
                    rgt.prop(self, "use_location", text="", index=index)

                col = row.column()
                col.alignment = 'RIGHT'
                spl = col.row(align=True).split(factor=0.75)

                lft = spl.column()
                rgt = spl.column()
                lft.alignment = 'RIGHT'

                mode = self.rotation_mode
                if mode == 'EULER':
                    lft.label(text="Rotation X")
                    rgt.prop(self, "use_rotation", text="", index=1)
                    for index, axis in zip(range(2, 4), 'YZ'):
                        lft.label(text=axis)
                        rgt.prop(self, "use_rotation", text="", index=index)
                else:
                    lft.label(text="Rotation W")
                    rgt.prop(self, "use_rotation", text="", index=0)
                    for index, axis in zip(range(1, 4), 'XYZ'):
                        lft.label(text=axis)
                        rgt.prop(self, "use_rotation", text="", index=index)

                col.prop(self, "rotation_mode", text="")

                col = row.column()
                col.alignment = 'RIGHT'
                spl = col.row(align=True).split(factor=0.75)

                lft = spl.column()
                rgt = spl.column()
                lft.alignment = 'RIGHT'

                lft.label(text="Scale X")
                rgt.prop(self, "use_scale", text="", index=0)
                for index, axis in zip(range(1, 3), 'YZ'):
                    lft.label(text=axis)
                    rgt.prop(self, "use_scale", text="", index=index)

        layout.separator()

    def _build_spec(self) -> Tuple[str, list]:
        mode = self.rotation_mode
        spec = [
            (self.use_location, 'LOC', "Location", 'XYZ'),
            (self.use_scale, 'SCALE', "Scale", 'XYZ')
        ]
        if mode == 'EULER':
            spec.insert(1, (self.use_rotation[1:], 'ROT', "Rotation", 'XYZ'))
        else:
            spec.insert(1, (self.use_rotation, 'ROT', "Rotation", 'WXYZ'))
        return mode, spec

    @staticmethod
    def _apply_spec(target: PoseBone,
                    driver: RBFDriver,
                    mode: str,
                    spec: list,
                    id_: Optional[Object] = None,
                    bone_target: Optional[str] = "") -> None:
        # pylint: disable=too-many-arguments
        # pylint: disable=too-many-locals
        driver.xm__internal = False
        props = driver.driven_properties
        poses = driver.poses

        if bone_target:
            suffix = f' ({remove_mirror_suffix_for_name(bone_target)})'
        elif id_ is not None:
            suffix = f' ({id_.name})'
        else:
            suffix = ""

        for used, prefix, name, axes in spec:
            for flag, axis in zip(used, axes):
                if flag:
                    prop = props.data__internal.add()

                    samples = prop.samples.data__internal
                    for _ in poses:
                        samples.add().value = float(prefix == 'SCALE')

                    prop.name = f'{name} {axis}{suffix}'
                    prop.type = 'TRANSFORMS'
                    prop.transform_type = f'{prefix}_{axis}'
                    prop.rotation_mode = mode

                    if bone_target:
                        prop.bone_target = bone_target
                    if id_ is not None:
                        prop.id = id_
        driver.xm__internal = True

        if driver.is_valid:
            network.update(target, driver)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        # pylint: disable=unused-argument
        target = context.active_pose_bone
        driver = target.rbf_drivers.active

        id_ = context.scene.objects.get(self.object)
        if id_ is None:
            bone_target = ""
        else:
            bone_target = self.bone_target if id_.type == 'ARMATURE' else ""

        mode, spec = self._build_spec()
        self._apply_spec(target, driver, mode, spec, id_, bone_target)

        target, mirror = get_mirror_pose_bone_and_rbf_driver(driver)
        if mirror:
            if bone_target:
                suffix = get_mirror_suffix_for_name(bone_target)
                if suffix:
                    bone_target = set_mirror_suffix_for_name(bone_target, suffix)
            self._apply_spec(target, mirror, mode, spec, id_, bone_target)

        return {'FINISHED'}


class RBFDRIVERS_OT_driven_property_remove(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driven_property_remove"
    bl_label = "Remove"
    bl_description = "Remove the driven property from the rbf driver"
    bl_options = {'INTERNAL'}

    property_index: IntProperty()

    @staticmethod
    def process(pose_bone: PoseBone, driver: RBFDriver, index: int) -> None:
        # pylint: disable=missing-function-docstring
        network.destroy(pose_bone, driver)
        driver.driven_properties.data__internal.remove(index)

        if driver.is_valid:
            network.update(pose_bone, driver)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        bone = context.active_pose_bone
        driver = bone.rbf_drivers.active

        index = self.property_index
        props = driver.driven_properties
        if 0 > index > len(props):
            self.report({'ERROR'}, "Driven property index is out of range")
            return {'CANCELLED'}

        name = props[index].name
        self.process(bone, driver, index)

        bone, driver = get_mirror_pose_bone_and_rbf_driver(driver)
        if driver is not None:
            index = driver.driven_properties.find(name)
            if index != -1:
                self.process(bone, driver, index)

        return {'FINISHED'}


class RBFDRIVERS_OT_driven_property_move_up(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driven_property_move_up"
    bl_label = "Move Up"
    bl_description = "Move the driven property up within the list"
    bl_options = {'INTERNAL'}

    property_index: IntProperty()

    @staticmethod
    def process(driver: RBFDriver, index: int) -> None:
        # pylint: disable=missing-function-docstring
        props = driver.driven_properties
        props.data__internal.move(index, index - 1)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        driver = context.active_pose_bone.rbf_drivers.active
        index = self.property_index
        self.process(driver, index)

        mirror = get_mirror_rbf_driver(driver)
        if mirror is not None:
            index = mirror.driven_properties.find(driver.driven_properties[index].name)
            if index > 0:
                self.process(mirror, index)

        return {'FINISHED'}


class RBFDRIVERS_OT_driven_property_move_down(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.driven_property_move_down"
    bl_label = "Move Down"
    bl_description = "Move the driven property down within the list"
    bl_options = {'INTERNAL'}

    property_index: IntProperty()

    @staticmethod
    def process(driver: RBFDriver, index: int) -> None:
        # pylint: disable=missing-function-docstring
        props = driver.driven_properties
        props.data__internal.move(index, index + 1)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        driver = context.active_pose_bone.rbf_drivers.active
        index = self.property_index
        self.process(driver, index)

        mirror = get_mirror_rbf_driver(driver)
        if mirror is not None:
            index = mirror.driven_properties.find(driver.driven_properties[index].name)
            if len(mirror.driven_properties) - 1 > index:
                self.process(mirror, index)

        return {'FINISHED'}

# endregion

# region POSE OPERATORS
###################################################################################################

class RBFDRIVERS_OT_pose_move_up(ActivePose, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.pose_move_up"
    bl_label = "Move up"
    bl_description = "Move the active item up within the list"
    bl_options = {'INTERNAL'}

    @classmethod
    def poll(cls, context: Context) -> bool:
        # pylint: disable=missing-function-docstring
        return (super().poll(context)
                and context.active_pose_bone.rbf_drivers.active.poses.active_index > 1)

    @staticmethod
    def process(pose_bone: PoseBone, driver: RBFDriver, index: int) -> None:
        # pylint: disable=missing-function-docstring
        poses = driver.poses
        poses.data__internal.move(index, index - 1)
        poses.active_index = index - 1

        for prop in driver.driven_properties:
            prop.samples.data__internal.move(index, index - 1)

        name = influence_idpropkey_for(driver, 'Weight')
        path = f'{pose_bone.path_from_id()}["{name}"]'
        splice_influence_idprop(pose_bone,
                                pose_bone.id_data.animation_data,
                                name, path, index, index-1)

        name = influence_idpropkey_for(driver, 'Weight')
        path = f'["{name}"]'
        splice_influence_idprop(pose_bone.id_data.data,
                                pose_bone.id_data.data.animation_data,
                                name, path, index, index-1)

        if driver.is_valid:
            network.update(pose_bone, driver)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        target = context.active_pose_bone
        driver = target.rbf_drivers.active
        self.process(target, driver, driver.poses.active_index)

        target, mirror = get_mirror_pose_bone_and_rbf_driver(driver)
        if mirror is not None:
            index = mirror.poses.find(driver.poses.active.name)
            if index > 0:
                self.process(target, mirror, index)

        return {'FINISHED'}


class RBFDRIVERS_OT_pose_move_down(ActivePose, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.pose_move_down"
    bl_label = "Move down"
    bl_description = "Move the active item down within the list"
    bl_options = {'INTERNAL'}

    @classmethod
    def poll(cls, context: Context) -> bool:
        # pylint: disable=missing-function-docstring
        if super().poll(context):
            poses = context.active_pose_bone.rbf_drivers.active.poses
            return 0 < poses.active_index < len(poses) - 1
        return False

    @staticmethod
    def process(pose_bone: PoseBone, driver: RBFDriver, index: int) -> None:
        # pylint: disable=missing-function-docstring
        poses = driver.poses
        poses.data__internal.move(index, index + 1)
        poses.active_index = index + 1

        for prop in driver.driven_properties:
            prop.samples.data__internal.move(index, index + 1)

        name = influence_idpropkey_for(driver, 'Weight')
        path = f'{pose_bone.path_from_id()}["{name}"]'
        splice_influence_idprop(pose_bone,
                                pose_bone.id_data.animation_data,
                                name, path, index, index+1)

        name = influence_idpropkey_for(driver, 'Weight')
        path = f'["{name}"]'
        splice_influence_idprop(pose_bone.id_data.data,
                                pose_bone.id_data.data.animation_data,
                                name, path, index, index+1)

        if driver.is_valid:
            network.update(pose_bone, driver)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        target = context.active_pose_bone
        driver = target.rbf_drivers.active
        self.process(target, driver, driver.poses.active_index)

        target, mirror = get_mirror_pose_bone_and_rbf_driver(driver)
        if mirror is not None:
            poses = mirror.poses
            index = poses.find(driver.poses.active.name)
            if len(poses) - 1 > index:
                self.process(target, mirror, index)

        return {'FINISHED'}


class RBFDRIVERS_OT_pose_update(ActivePose, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.pose_update"
    bl_label = "Update"
    bl_description = "Update the active pose from the current scene state"
    bl_options = {'REGISTER', 'UNDO'}

    @staticmethod
    def process(bone: PoseBone, driver: RBFDriver, index: int) -> None:
        # pylint: disable=missing-function-docstring
        matrix = bone.id_data.convert_space(pose_bone=bone,
                                            matrix=bone.matrix,
                                            from_space='POSE',
                                            to_space='LOCAL')

        driver.poses[index].matrix = flatten_matrix(matrix)

        for prop in driver.driven_properties:
            value = getpath(prop.id, prop.data_path, prop.array_index) if prop.is_valid else 0.0
            prop.samples[index].value = value

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        target = context.active_pose_bone
        driver = target.rbf_drivers.active
        index = driver.poses.active_index
        self.process(target, driver, index)

        name = driver.poses.active.name
        target, mirror = get_mirror_pose_bone_and_rbf_driver(driver)
        if mirror is not None:
            index = mirror.poses.find(name)
            self.process(target, mirror, index)

        if driver.is_valid:
            # pylint: disable=import-outside-toplevel
            import bpy
            bpy.ops.rbf_drivers.driver_update()

        return {'FINISHED'}


class RBFDRIVERS_OT_pose_add(ActiveDriver, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.pose_add"
    bl_label = "Add"
    bl_description = "Add a pose to the active driver"
    bl_options = {'REGISTER', 'UNDO'}

    @staticmethod
    def process(pose_bone: PoseBone, driver: RBFDriver, name: Optional[str] = "Pose") -> None:
        # pylint: disable=missing-function-docstring
        poses = driver.poses
        index = len(poses)

        pose = poses.data__internal.add()
        pose.name = "Rest" if index == 0 else name
        poses.active_index = index

        for prop in driver.driven_properties:
            prop.samples.data__internal.add()

        for suffix, datablock in [
                ('Weight', pose_bone),
                ('Effect', pose_bone.id_data.data)
            ]:
            propname = influence_idpropkey_for(driver, suffix)
            idprop = datablock.get(propname)

            if idprop is None:
                create_influence_idprop(datablock, propname, poses)
            else:
                idprop = idprop.to_list()
                idprop.append(1.0)
                datablock[propname] = idprop

        return pose

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        # pylint: disable=import-outside-toplevel
        target = context.active_pose_bone
        driver = target.rbf_drivers.active
        pose = self.process(target, driver)

        target, driver = get_mirror_pose_bone_and_rbf_driver(driver)
        if driver is not None:
            self.process(target, driver, pose.name)

        import bpy
        bpy.ops.rbf_drivers.pose_update()

        return {'FINISHED'}


class RBFDRIVERS_OT_pose_remove(ActivePose, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.pose_remove"
    bl_label = "Remove pose"
    bl_description = "Remove the specified RBF driver pose"
    bl_options = {'REGISTER', 'UNDO'}

    @staticmethod
    def process(pose_bone: PoseBone, driver: RBFDriver, index: int) -> None:
        # pylint: disable=missing-function-docstring
        driver.poses.data__internal.remove(index)

        for prop in driver.driven_properties:
            prop.samples.data__internal.remove(index)

        name = influence_idpropkey_for(driver, 'Weight')
        path = f'{pose_bone.path_from_id()}["{name}"]'
        splice_influence_idprop(pose_bone,
                                pose_bone.id_data.animation_data,
                                name, path, index)

        name = influence_idpropkey_for(driver, 'Weight')
        path = f'["{name}"]'
        splice_influence_idprop(pose_bone.id_data.data,
                                pose_bone.id_data.data.animation_data,
                                name, path, index)

        network.destroy(pose_bone, driver)
        if driver.is_valid:
            network.update(pose_bone, driver)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        bone = context.active_pose_bone
        driver = bone.rbf_drivers.active
        name = driver.poses.active.name

        self.process(bone, driver, driver.poses.active_index)

        bone, driver = get_mirror_pose_bone_and_rbf_driver(driver)
        if driver is not None:
            index = driver.poses.find(name)
            if index != -1:
                self.process(bone, driver, index)

        return {'FINISHED'}


class RBFDRIVERS_OT_pose_apply(ActivePose, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.pose_apply"
    bl_label = "Apply RBF Driver pose"
    bl_description = "Apply specified RBF driver pose to the scene"
    bl_options = {'REGISTER', 'UNDO'}

    @staticmethod
    def process(bone: PoseBone, driver: RBFDriver, index: int) -> None:
        # pylint: disable=missing-function-docstring
        # pylint: disable=too-many-locals
        rmode = driver.rotation_mode
        poses = driver.poses

        if rmode == 'AUTO' or rmode.startswith('SWING_TWIST'):
            rmode = bone.rotation_mode

        if len(rmode) == 3:
            raxes = tuple('XYZ'.index(a) for a in rmode)
            rattr = 'rotation_euler'
            rused = driver.use_rotation[1:]
        else:
            raxes = range(4)
            rattr = f'rotation_{rmode.lower()}'
            rused = driver.use_rotation

        location, rotation, scale = poses[index].matrix.decompose()
        rotation = convert_rotation(rotation, 'QUATERNION', rmode)

        for prop, axes, use, vec in [
                (bone.location, range(3), driver.use_location, location),
                (getattr(bone, rattr), raxes, rused, rotation),
                (bone.scale, range(3), driver.use_scale, scale)
                ]:
            for axis in axes:
                if use[axis]:
                    prop[axis] = vec[axis]

        for prop in filter(attrgetter("is_valid"), driver.driven_properties):
            with suppress(ValueError):
                setpath(prop.id,
                        prop.state__internal.dpath,
                        prop.array_index if prop.is_array else -1,
                        prop.samples[index].value)

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=no-self-use
        # pylint: disable=too-many-locals
        bone = context.active_pose_bone
        driver = bone.rbf_drivers.active
        name = driver.poses.active.name

        self.process(bone, driver, driver.poses.active_index)

        bone, driver = get_mirror_pose_bone_and_rbf_driver(driver)
        if driver is not None:
            index = driver.poses.find(name)
            if index != -1:
                self.process(bone, driver, index)

        return {'FINISHED'}

def _get_rotation_euler(self) -> Euler:
    return self.rotation_quaternion.to_euler()

def _set_rotation_euler(self, value: Tuple[float, float, float]) -> None:
    self.rotation_quaternion = euler_to_quaternion(Euler(value))

class RBFDRIVERS_OT_pose_edit(ActivePose, Operator):
    # pylint: disable=missing-class-docstring
    # pylint: disable=invalid-name
    bl_idname = "rbf_drivers.pose_edit"
    bl_label = "Edit Pose"
    bl_description = "Edit specified RBF driver pose data"
    bl_options = {'REGISTER', 'UNDO'}

    translation: FloatVectorProperty(
        name="Location",
        description="Pose location",
        size=3,
        subtype='XYZ'
        )

    rotation_euler: FloatVectorProperty(
        name="Euler Rotation",
        description="Pose rotation (Euler)",
        size=3,
        subtype='EULER',
        get=_get_rotation_euler,
        set=_set_rotation_euler,
        )

    rotation_quaternion: FloatVectorProperty(
        name="Quaternion Rotation",
        description="Pose rotation (quaternion)",
        size=4,
        subtype='QUATERNION'
        )

    scale: FloatVectorProperty(
        name="Scale",
        description="Pose scale",
        size=3,
        subtype='XYZ'
        )

    driven: CollectionProperty(
        name="Driven Properties",
        description="Driven properties for pose",
        type=Sample,
        options=set()
        )

    def invoke(self, context: Context, event: Event) -> Set[str]:
        # pylint: disable=missing-function-docstring
        # pylint: disable=unused-argument
        driver = context.active_pose_bone.rbf_drivers.active

        poses = driver.poses
        index = poses.active_index

        pose = poses.active
        self.translation = pose.location
        self.rotation_quaternion = pose.rotation_quaternion
        self.scale = pose.scale

        cache = self.driven
        cache.clear()

        for src in driver.driven_properties:
            tgt = cache.add()
            tgt.name = src.name
            tgt.value = src.samples[index].value

        return context.window_manager.invoke_props_dialog(self, width=600)

    def draw(self, context: Context) -> None:
        # pylint: disable=missing-function-docstring
        row = self.layout.row()
        lft = row.column()
        rgt = row.column()

        lft.label(text="Inputs")
        rgt.label(text="Driven Properties")
        lft.separator()
        rgt.separator()

        driver = context.active_pose_bone.rbf_drivers.active

        split = lft.split(factor=0.6666)
        label = split.column()
        value = split.column()
        label.alignment = 'RIGHT'

        flags = driver.use_location
        if any(flags):
            self.draw_inputs(label, value, "Location", flags, 'XYZ', "translation")

        flags = list(driver.use_rotation)
        if any(flags):
            mode = driver.rotation_mode

            if len(mode) == 3 or mode == 'AUTO':
                axes = 'XYZ'
                prop = "rotation_euler"
                pfix = "Rotation (Euler)"
                flags.pop(0)
            else:
                axes = 'WXYZ'
                prop = "rotation_quaternion"
                pfix = "Rotation (Quaternion)"

            self.draw_inputs(label, value, pfix, flags, axes, prop)

        flags = driver.use_scale
        if any(flags):
            self.draw_inputs(label, value, "Scale", flags, 'XYZ', "scale")

        split = rgt.split(factor=0.6666)
        label = split.column()
        value = split.column()
        label.alignment = 'RIGHT'

        for sample in self.driven:
            label.label(text=sample.name)
            value.prop(sample, "value", text="")

    def draw_inputs(self,
                    labels: UILayout,
                    values: UILayout,
                    prefix: str,
                    flags: Sequence[bool],
                    axes: str,
                    propname: str) -> None:
        # pylint: disable=missing-function-docstring
        # pylint: disable=too-many-arguments
        first = True
        for index, (used, axis) in enumerate(zip(flags, axes)):
            if used:
                labels.label(text=f'{prefix} {axis}' if first else axis)
                values.prop(self, propname, index=index, text="")
                first = False
        labels.separator()
        values.separator()

    def execute(self, context: Context) -> Set[str]:
        # pylint: disable=missing-function-docstring
        bone = context.active_pose_bone
        driver = bone.rbf_drivers.active

        poses = driver.poses
        index = poses.active_index

        matrix = compose_matrix(self.translation, self.rotation_quaternion, self.scale)
        poses[index].matrix = flatten_matrix(matrix)

        for src, tgt in zip(self.driven, driver.driven_properties):
            tgt.samples[index].value = src.value

        if driver.is_valid:
            try:
                network.update(bone, driver)
            except network.ExpressionLengthError:
                self.report({'ERROR'}, EXPLEN_MSG)
                return {'CANCELLED'}

        return {'FINISHED'}

# endregion
