// Copyright (C) 2nilfmt.Errorf("a")4-2nilfmt.Errorf("a")8 Goodrain Co., Ltd.
// RAINBOND, Application Management Platform

// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version. For any non-GPL usage of Rainbond,
// one or multiple Commercial Licenses authorized by Goodrain Co., Ltd.
// must be obtained first.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

package gc

import (
	"fmt"
	"os"
	"path"

	eventutil "github.com/goodrain/rainbond/eventlog/util"
	"github.com/goodrain/rainbond/worker/discover/model"
	"github.com/sirupsen/logrus"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/labels"
	"k8s.io/client-go/kubernetes"
)

// GarbageCollector -
type GarbageCollector struct {
	clientset kubernetes.Interface
}

// NewGarbageCollector -
func NewGarbageCollector(clientset kubernetes.Interface) *GarbageCollector {
	gcr := &GarbageCollector{
		clientset: clientset,
	}
	return gcr
}

// DelLogFile deletes persistent data related to the service based on serviceID.
func (g *GarbageCollector) DelLogFile(serviceGCReq model.ServiceGCTaskBody) {
	logrus.Infof("service id: %s; delete log file.", serviceGCReq.ServiceID)
	// log generated during service running
	logPath := "/grdata/logs"
	dockerLogPath := eventutil.DockerLogFilePath(logPath, serviceGCReq.ServiceID)
	if err := os.RemoveAll(dockerLogPath); err != nil {
		logrus.Warningf("remove docker log files: %v", err)
	}
	// log generated by the service event
	eventLogPath := eventutil.EventLogFilePath(logPath)
	for _, eventID := range serviceGCReq.EventIDs {
		eventLogFileName := eventutil.EventLogFileName(eventLogPath, eventID)
		logrus.Debugf("remove event log file: %s", eventLogFileName)
		if err := os.RemoveAll(eventLogFileName); err != nil {
			logrus.Warningf("file: %s; remove event log file: %v", eventLogFileName, err)
		}
	}
}

// DelVolumeData -
func (g *GarbageCollector) DelVolumeData(serviceGCReq model.ServiceGCTaskBody) {
	f := func(prefix string) {
		dir := path.Join(prefix, fmt.Sprintf("tenant/%s/service/%s", serviceGCReq.TenantID, serviceGCReq.ServiceID))
		logrus.Infof("volume data. delete %s", dir)
		if err := os.RemoveAll(dir); err != nil {
			logrus.Warningf("dir: %s; remove volume data: %v", dir, err)
		}
	}
	f("/grdata")
}

// DelPvPvcByServiceID -
func (g *GarbageCollector) DelPvPvcByServiceID(serviceGCReq model.ServiceGCTaskBody) {
	logrus.Infof("service_id: %s", serviceGCReq.ServiceID)
	deleteOpts := &metav1.DeleteOptions{}
	listOpts := g.listOptionsServiceID(serviceGCReq.ServiceID)
	if err := g.clientset.CoreV1().PersistentVolumes().DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("service id: %s; delete a collection fo PV: %v", serviceGCReq.ServiceID, err)
	}

	if err := g.clientset.CoreV1().PersistentVolumeClaims(serviceGCReq.TenantID).DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("service id: %s; delete a collection fo PVC: %v", serviceGCReq.ServiceID, err)
	}
}

// DelKubernetesObjects deletes all kubernetes objects.
func (g *GarbageCollector) DelKubernetesObjects(serviceGCReq model.ServiceGCTaskBody) {
	deleteOpts := &metav1.DeleteOptions{}
	listOpts := g.listOptionsServiceID(serviceGCReq.ServiceID)
	if err := g.clientset.AppsV1().Deployments(serviceGCReq.TenantID).DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("[DelKubernetesObjects] delete deployments(%s): %v", serviceGCReq.ServiceID, err)
	}
	if err := g.clientset.AppsV1().StatefulSets(serviceGCReq.TenantID).DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("[DelKubernetesObjects] delete statefulsets(%s): %v", serviceGCReq.ServiceID, err)
	}
	if err := g.clientset.ExtensionsV1beta1().Ingresses(serviceGCReq.TenantID).DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("[DelKubernetesObjects] delete ingresses(%s): %v", serviceGCReq.ServiceID, err)
	}
	if err := g.clientset.CoreV1().Secrets(serviceGCReq.TenantID).DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("[DelKubernetesObjects] delete secrets(%s): %v", serviceGCReq.ServiceID, err)
	}
	if err := g.clientset.CoreV1().ConfigMaps(serviceGCReq.TenantID).DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("[DelKubernetesObjects] delete configmaps(%s): %v", serviceGCReq.ServiceID, err)
	}
	if err := g.clientset.AutoscalingV2beta2().HorizontalPodAutoscalers(serviceGCReq.TenantID).DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("[DelKubernetesObjects] delete hpas(%s): %v", serviceGCReq.ServiceID, err)
	}
	// kubernetes does not support api for deleting collection of service
	// read: https://github.com/kubernetes/kubernetes/issues/68468#issuecomment-419981870
	serviceList, err := g.clientset.CoreV1().Services(serviceGCReq.TenantID).List(listOpts)
	if err != nil {
		logrus.Warningf("[DelKubernetesObjects] list services(%s): %v", serviceGCReq.ServiceID, err)
	} else {
		for _, svc := range serviceList.Items {
			if err := g.clientset.CoreV1().Services(serviceGCReq.TenantID).Delete(svc.Name, deleteOpts); err != nil {
				logrus.Warningf("[DelKubernetesObjects] delete service(%s): %v", svc.GetName(), err)
			}
		}
	}
	// delete endpoints after deleting services
	if err := g.clientset.CoreV1().Endpoints(serviceGCReq.TenantID).DeleteCollection(deleteOpts, listOpts); err != nil {
		logrus.Warningf("[DelKubernetesObjects] delete endpoints(%s): %v", serviceGCReq.ServiceID, err)
	}
}

func (g *GarbageCollector) listOptionsServiceID(serviceID string) metav1.ListOptions {
	labelSelector := metav1.LabelSelector{MatchLabels: map[string]string{
		"creator":    "Rainbond",
		"service_id": serviceID,
	}}
	return metav1.ListOptions{
		LabelSelector: labels.Set(labelSelector.MatchLabels).String(),
	}
}
