### (c) TIG 2011
# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.
# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
### RevCloud.rb
### Type 'revcloud' in the Ruby Console
### or choose 'RevCloud' off the Draw Menu [or shortcut to it].
### It draws a Revision Cloud.
### Pick a start-point, then hold down the left mouse-button to continue 
### drawing the revcloud - you release the mouse-button to end it.
### It is represented by an orange arced-polyline until it is completed. 
### The cloud is made 'flat', Z ia determined from the first picked point.
### You can (pre)set the 'segment-length' in the VCB; the default is 300mm.
### Each segments arc-radius should be approx. half of this.
### If you move the mouse very quickly the segment length may be larger.
### Move to within the 'segment-length' of the 1st point & release the 
### button to make a closed loop, the VCB tells you if a 'closed loop' is 
### possible.
### The last segment is then sized to close the remaining 'gap'
### Releasing the button before a 'closed loop' is possible makes a broken
### loop of the revcloud.
### You can then start a new revcloud after releasing the mouse-button.
### To exit  the Tool select any new tool or Esc+Esc or press <Enter>.
### Esc undoes the current revcloud (if any).
### Normal Inference Tips may apply.
### Draw counterclockwise for outwards 'clouds' arcs & clockwise to make 
### 'spikes'...
### The revcloud is made in a group called 'RevCloud#1' - with an 
### incrementing suffix.
###
### Donations:
### info @ revitrev.org via PayPal.com
###
### Version:
### 1.0 20110622 First Issue.
### 1.1 20110727 Draw Menu 'RevCloud' added.
###
#-----------------------------------------------------------------------------
require 'sketchup.rb'
#-----------------------------------------------------------------------------
class RevCloud

def initialize
    @ip1 = nil
    @ip = nil
    @iplast= nil
end

def activate
    @segment_length=300.mm if not @segment_length
    @ip1 = Sketchup::InputPoint.new
    @ip = Sketchup::InputPoint.new
    @iplast= Sketchup::InputPoint.new
    @points=[]
    self.reset(Sketchup.active_model.active_view)
end

def deactivate(view)
    view.invalidate if @drawn
    Sketchup::set_status_text("",SB_PROMPT)
    Sketchup::set_status_text("",SB_VCB_LABEL)
    Sketchup::set_status_text("",SB_VCB_VALUE)
end

def resume(view)
  view.invalidate
  Sketchup::set_status_text("RevCloud Seg'", SB_VCB_LABEL)
  Sketchup::set_status_text(@segment_length.to_s, SB_VCB_VALUE)
  if not @points[0]
    Sketchup::set_status_text("RevCloud: Pick Start Point. Hold down mouse-button to draw. To Exit press <Enter> or <Esc>", SB_PROMPT)
  else
    Sketchup::set_status_text("RevCloud: Release mouse-button to end. Move near to start to auto-close loop. To Cancel press <Esc>", SB_PROMPT)
  end
end

def getExtents
    bbox=Sketchup.active_model.bounds
    bbox.add(@points) if @points[0]
    return bbox
end

def onMouseMove(flags, x, y, view)
    if not @points[0]
        Sketchup::set_status_text("RevCloud: Pick Start Point. Hold down mouse-button to draw. To Exit press <Enter> or <Esc>", SB_PROMPT)
    else
        Sketchup::set_status_text("RevCloud: Release mouse-button to end. Move near to start to auto-close loop. To Cancel press <Esc>", SB_PROMPT)
    end
        @ip.pick(view,x,y)
        if @ip.valid?
            @ip1.copy!(@ip)
            view.tooltip = @ip1.tooltip
            view.invalidate if @iplast.display? or @ip1.display?
            p1=@ip1.position
            @z=@points[0].z if @points[0]
            p1.z=@z if @z
            if p1 and @points.last and p1.distance(@points.last)>=@segment_length
              @points<<p1
            end
            if @points[2] and @points.last.distance(@points.first)<=@segment_length
              Sketchup::set_status_text("Closed Loop", SB_VCB_VALUE)
            else
              Sketchup::set_status_text(@segment_length.to_s, SB_VCB_VALUE)
            end
            begin
               view.drawing_color = "black"
               view.draw_polyline(@points)
            rescue
              ###
            end
            @iplast=@ip1
            @ip1.clear
        end#if
        view.invalidate
end

def onLButtonDown(flags, x, y, view)
    @ip.pick view, x, y
    if @ip.valid?
      view.tooltip=@ip.tooltip
      p1=@ip.position; p1.z=@z if @z
      @points<<p1
      @iplast=@ip
      view.invalidate
    end
end

def onLButtonUp(flags, x, y, view)
    self.create_geometry(@points,view)### end this loop
    self.reset(view) ### start another
end

def onKeyDown(key, repeat, flags, view)
    ###
end

def onKeyUp(key, repeat, flags, view)
    ###
end

def enableVCB?
   return true
end

def onUserText(text, view)
    begin
        value = text.to_l
    rescue # Error parsing the text
        UI.beep
        puts ("Cannot convert "+text+" to a Length")
        value = nil
    end
    return if !value
    @segment_length=value
    Sketchup::set_status_text(@segment_length.to_s,SB_VCB_VALUE)
    view.invalidate
end

def draw(view)
    if @ip.valid?
       points=self.temp_arc_points(view)
       view.drawing_color = "orangered"
       begin
        view.draw_polyline(points) if points[1]
        view.draw_polyline([points[-1], @ip.position]) if points[1]
       rescue
        ###
       end
    end
    view.invalidate
end

def temp_arc_points(view)
    points=@points
    eds=[]
    begin
      (points.length-1).times{|i|
        begin
          p=points[i]
          p2=points[i+1]
          r=p.distance(p2)/2
          ct=p.offset(p.vector_to(p2),r)
          ang=0
          8.times{|i|
            tr=Geom::Transformation.rotation(ct, Z_AXIS, ang)
            pp=p.transform(tr)
            eds << pp
            ang=ang+(180.degrees/8)
          }
        rescue
          ###
        end
      }
    rescue
      ###
    end
    #eds.uniq!
    return eds
end

def onCancel(flag, view)
    if @points[1]
      view.invalidate
      self.reset(view)
    else
      view.invalidate
      Sketchup::set_status_text("",SB_PROMPT)
      Sketchup::set_status_text("",SB_VCB_LABEL)
      Sketchup::set_status_text("",SB_VCB_VALUE)
      Sketchup.send_action("selectSelectionTool:")
      return nil
    end#if
end

def onReturn(view)
    view.invalidate
    Sketchup::set_status_text("",SB_PROMPT)
    Sketchup::set_status_text("",SB_VCB_LABEL)
    Sketchup::set_status_text("",SB_VCB_VALUE)
    Sketchup.send_action("selectSelectionTool:")
    return nil
end

def reset(view)
    @ip1.clear
    @ip.clear
    @iplast.clear
    @points=[]
    @z=nil
    if view
        view.tooltip = nil
        view.invalidate
    end
    if not @points[0]
        Sketchup::set_status_text("RevCloud: Pick Start Point. Hold down mouse-button to draw. To Exit press <Enter> or <Esc>", SB_PROMPT)
    else
        Sketchup::set_status_text("RevCloud: Release mouse-button to end. Move near to start to auto-close loop. To Cancel press <Esc>", SB_PROMPT)
    end
end

def create_geometry(points, view)
    model=view.model
    points<<@points.first if points[2] and points.last.distance(points.first)<=@segment_length
    begin
      model.start_operation("RevCloud")
       group=model.active_entities.add_group()
       name="RevCloud#1"
       gps=[]
       model.definitions.each{|d|gps << d.instances[0] if d.group?}
       gps.compact!
       names=[]
       gps.each{|g|names << g.name}
       while names.include?(name)
         name.next!
       end#if
       group.name=name
       gents=group.entities
       points.each_with_index{|p,i|
         begin
           p2=points[i+1]
           d=p.distance(p2)
           r=d/2
           ct=p.offset(p.vector_to(p2),r)
           xa=ct.vector_to(p)
           gents.add_arc(ct, xa, Z_AXIS, r, 0, 180.degrees, 8)
         rescue
           ###
         end
       }
       view.invalidate
      model.commit_operation
    rescue
      ###
    end#if
end

end # class

#-----------------------------------------------------------------------------
# shortcut
def revcloud
    Sketchup.active_model.select_tool(RevCloud.new())
end
######-----------------------------------------------
# menu
unless file_loaded?(File.basename(__FILE__)) 
  # UI.menu("Draw").add_item("RevCloud"){Sketchup.active_model.select_tool(RevCloud.new())}
  path = Sketchup.find_support_file("Plugins/RevCloud")
  RevCloud_tb = UI::Toolbar.new("绘制云线")
  cmd1 = UI::Command.new("绘制云线") {Sketchup.active_model.select_tool(RevCloud.new())}
  cmd1.small_icon = File.join(path,"icons/RevCloud24.png")
  cmd1.large_icon = File.join(path,"icons/RevCloud36.png")
  cmd1.tooltip = "绘制云线"
  cmd1.status_bar_text = "绘制云线"
  RevCloud_tb.add_item(cmd1)
  RevCloud_tb.show

end
file_loaded(File.basename(__FILE__)) 
###

