REM  Copyright (c) 1999 by Oracle Corporation.
REM
REM  NAME
REM    utlinad.sql - PL/SQL Package for Internet address
REM                  (Package Specification of UTL_INADDR)
REM
REM  DESCRIPTION
REM    Routines to perform Internet address resolution.
REM
REM  NOTES
REM    The procedural option is needed to use this package.
REM    This package must be created under SYS.
REM
REM  MODIFIED (MM/DD/YY)
REM  rpang     08/11/99 - Added constants for error codes
REM  rpang     05/14/99 - Created

CREATE OR REPLACE PACKAGE utl_inaddr AS

 /*******************************************************************
  * OVERVIEW
  *
  * This package allows a PL/SQL program to retrieve host name and
  * IP address information of local or remote host.  The functionality
  * provided by this package does not cover the full functionality of
  * Domain Name Resolution (also known as DNS).
  *
  * USES
  *
  * Call get_host_name() to retrieve the name of the local host.  Call
  * get_host_address() to retrieve the IP address of local or remote
  * host.
  *
  * EXAMPLES
  *   Retrieve local host name and IP address.
  *
  * BEGIN
  *   dbms_output.put_line(utl_inaddr.get_host_name);  -- get local host name
  *   dbms_output.put_line(utl_inaddr.get_host_address);  -- get local IP addr
  * END;
  */

  /*
   * Exceptions
   */
  unknown_host    EXCEPTION;  -- Unknown host
  unknown_host_errcode  CONSTANT PLS_INTEGER := -20000;
  PRAGMA EXCEPTION_INIT(unknown_host, -20000);

  /**
   * Retrieves the name of the local host.  The name of the host
   * does not contain the domain to which the host belong.
   *
   * PARAMETERS
   *   None
   * RETURN
   *   The name of the local host
   * EXCEPTIONS
   *   None
   */
  FUNCTION get_host_name RETURN VARCHAR2;

  /**
   * Retrieves the IP address of the specified host.
   *
   * PARAMETERS
   *   p_hostname  the host to determine the IP address.  If this is null,
   *               the IP address of the local host will be returned.
   * RETURN
   *   The IP address of the specified host, or that of the local host
   * if p_hostname is NULL
   * EXCEPTIONS
   *   unknown_host  - the specified is not known.
   */
  FUNCTION get_host_address (host IN VARCHAR2 DEFAULT NULL) RETURN VARCHAR2;

END;
/

GRANT EXECUTE ON sys.utl_inaddr TO PUBLIC;
DROP PUBLIC SYNONYM utl_inaddr;
CREATE PUBLIC SYNONYM utl_inaddr FOR sys.utl_inaddr;
