#!/bin/sh
#
# $Header: rman1.sh 04-oct-99.17:07:55 dbeusee Exp $
#
# Copyright (c) 1998, 1999, Oracle Corporation.  All rights reserved.
#
# rman1.sh
#
#    NAME
#      rman1.sh - delete obsolete backups and copies
#
#    DESCRIPTION
#       This shell script automates deletion of obsolete datafilecopies and
#       backup pieces.  It uses sed and grep to process the output of the RMAN
#       "report obsolete" command and constructs an RMAN command file
#       containing the necessary "change...delete" commands.
#
#    NOTES
#      Some customization is necessary.
#
#    MODIFIED   (MM/DD/YY)
#    dbeusee     10/01/99 - demo_rman1_2: removed "from recovery catalog" from
#                           comments, since nocatalog can be used (so RC N/A).
#    mjaeger     07/26/99 - bug 808870: OCCS: convert tabs, no long lines
#    dbeusee     07/14/99 - fix_rman1sh_usage: fix usage message for cat/nocat
#    dbeusee     07/07/99 - fix_demo: add msgno and target/catalog args
#    gpongrac    08/11/98 - Creation
#

usage() {
   case "$1" in
      "") ;;
      *)  echo "Unknown argument: $1" ;;
   esac

   echo "
Usage: `basename $0` [option] ...

Option                       Description
---------------------------  --------------------------------------------------
target CNCTSTR               Connect to the target db using CNCTSTR.
catalog CNCTSTR              Connect to catalog db using CNCTSTR.
nocatalog                    Don't use a recovery catalog.
parms 'PARMS'                Use PARMS string for SBT_TAPE maintenance channel.
                             You can use single or double quotes depending on
                             your needs.  In the rman script, single-quotes
                             will be used.

The catalog or nocatalog option must be specified.  All others are optional
(target is defaulted to internal/manager which you may need to override if
using a orapwd file for your target db).

NOTE: This is only a sample script.  If your needs exceed the capabilities of
this script, please feel free to copy it and modify your copy as needed.
"
   exit 1
}

# Initialize default connect string variables:

target='target internal/manager'
catalog='' # force user to supply catalog option
parms=''

# process target and catalog arguments
while [ $# -gt 0 ]; do
   case "$1" in
      target)         target="$1 $2";  shift ;;
      catalog|rcvcat) catalog="$1 $2"; shift ;;
      nocatalog)      catalog=$1             ;;
      parms)          parms="$1 '$2'"; shift ;;
      *)              usage "$1" ;;
   esac
   shift
done

case "$catalog" in
   "") usage ;;
esac

# Initialize filenames:

obsolete_disk=/tmp/ob_disk.log   # list of obsolete disk backups
obsolete_tape=/tmp/ob_tape.log   # list of obsolete tape backups
cmdfile=/tmp/delete.cmd          # the generated RMAN commmand script
delete_log=/tmp/deleted.log      # execution log from generated script

# Get a list of obsolete disk files:

rman msgno $target $catalog msglog $obsolete_disk << EOF > /dev/null
report obsolete device type disk;
exit;
EOF

# Get a list of obsolete tape files:

rman msgno $target $catalog msglog $obsolete_tape << EOF > /dev/null
report obsolete device type 'sbt_tape';
exit;
EOF

# sed actions:

del_copy="-e /RMAN-06283/ s/\(.* \)\(.*\)$/change datafilecopy '\2' delete;/"

del_bpiece="-e /RMAN-06285/ s/\(.* \)\(.*\)$/change backuppiece '\2' delete;/"

#
# Create the RMAN command file to delete the obsolete files:
#
echo 'allocate channel for delete type disk;' > $cmdfile

grep RMAN-06283 $obsolete_disk | sed "$del_copy" >> $cmdfile
grep RMAN-06285 $obsolete_disk | sed "$del_bpiece" >> $cmdfile

echo "release channel;" >> $cmdfile
echo "allocate channel for delete type 'sbt_tape' $parms ;" >> $cmdfile

grep RMAN-06285 $obsolete_tape | sed "$del_bpiece" >> $cmdfile

echo "release channel;" >> $cmdfile

#
# Execute the RMAN command file to delete the obsolete files:
#
rman msgno $target $catalog msglog $delete_log cmdfile $cmdfile

exit # rman1.sh

