# $Header: libcache.tcl 30-mar-99.03:16:03 dholail Exp $
#
# copyright (c) 1999 by the Oracle Corporation
#
# NAME:
#
#  libcache.tcl : Event tcl file that checks if the library cache miss
#                 ratio is too high
#
# ARGUMENTS:
#           argv(0) == no of occurrance
#           argv(1) == threshold value for alert
#           argv(2) == threshold value for warning
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           The current library cache miss ratio
#
# $Log:  $
# Revision 1.5  1995/10/19  00:03:26  yliu
# moved the init code into EvalEvent
#
# Revision 1.4  1995/10/10  23:35:10  yliu
# added no of occurrence
#
# Revision 1.3  1995/09/28  23:21:31  yliu
# use message file
#
# Revision 1.2  1995/09/15  00:02:14  yliu
# Modified default values
#
# Revision 1.1  1995/09/14  23:25:03  yliu
# Initial revision
#

# Event definition
oradefine event /oracle/rdbms/perf/libcache description=VOC-01151 \
report=VOC-01152
oraarguments no_occurrence alert_threshold warning_threshold
oravardesc no_occurrence unsigned default=3 message=VOC-01153
oravardesc alert_threshold number minimum=0 maximum=100 default=1 message=VOC-01154
oravardesc warning_threshold number minimum=0 maximum=100 default=0.5 message=VOC-01155
oraresults ratio
oravardesc ratio number
oradefine end


# Initialize global variables
#comment out to fix bug#606739
#set last_report $CLEAR_EVENT
set output ""
set no_occurrence 0
set initial 1
set prev_values(lib_pins) 0
set prev_values(lib_reloads) 0


# The main event checking functions
proc EvalEvent {} {


    # Declare globals we will use
    global argv last_report output no_occurrence initial
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg prev_values


    # initialize the return code and output
    set ret_code $CLEAR_EVENT
    set output ""
    set err ""

    # get current value
    if {[catch {set curr_values(lib_pins) [lindex [oradbsnmp get \
        oraDbLibraryCacheSumPins.$oramsg(oraindex)] 1]} err]} {
	lappend output $err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    if {[catch {set curr_values(lib_reloads) [lindex [oradbsnmp get \
        oraDbLibraryCacheSumReloads.$oramsg(oraindex)] 1]} err]} {
	lappend output $err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }


    # During initialization, copy current values to previous values
    if {$initial == 1} {
        foreach att_name [array names curr_values] {
            set prev_values($att_name) $curr_values($att_name)
        }
        set initial 0
        return $NO_EVENT
    }


    # calculate delta 
    foreach att_name [array names curr_values] {
        set delta($att_name) [expr $curr_values($att_name) - $prev_values($att_name)]
        set prev_values($att_name) $curr_values($att_name)
    }

    set v1 [expr double ($delta(lib_reloads))]
    set v2 [expr double ($delta(lib_pins))]


    # deal with rollovers
    if { ($v1 < 0) || ($v2 < 0) } {
        ORATCL_DEBUG "libcache : $oramsg(oraobject) : [oratime] : rollover - $v1,$v2"
        return $NO_EVENT
    }


    # watch out for divide by 0
    if { $v2 == 0 } {
        ORATCL_DEBUG "libcache : $oramsg(oraobject) : [oratime] : divide by zero - $v2"
        return $NO_EVENT
    }


    # calculate ratio 
    set ratio [divide [expr 100 * $v1] $v2]
    set ratio [format "%3.2f" $ratio]


    # ignore negative ratios
    if { $ratio < 0 } {
        ORATCL_DEBUG "libcache : $oramsg(oraobject) : [oratime] : negative ratio - $v1,$v2,$ratio"
        return $NO_EVENT
    }


    # print out event trace info 
    ORATCL_DEBUG "libcache : $oramsg(oraobject) : [oratime] : $v1,$v2,$ratio"


    # Check for alert and warning threshold
    if {$ratio > [lindex $argv 1]} {
        if {$no_occurrence < [expr [lindex $argv 0]-1]} {
            incr no_occurrence
        } else {
            set ret_code $ALERT_EVENT
            set output $ratio
        }
    } elseif {$ratio > [lindex $argv 2]} {
        if {$no_occurrence < [expr [lindex $argv 0]-1]} {
            incr no_occurrence
        } else {
            set ret_code $WARNING_EVENT
            set output $ratio
        }
    } else {
        set no_occurrence 0
    } 


    # return
    if {$last_report == $ret_code} {
        return $NO_EVENT
    } else {
        ORATCL_DEBUG "libcache : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, $output"
        set last_report $ret_code
        return $ret_code 
    }


}

