# $Header: bufcache.tcl 30-mar-99 dholail Exp $
#
# copyright (c) 1999 by the Oracle Corporation
#
# NAME:
#
#  memsort.tcl : Event tcl file that checks if the in memory sort
#                ratio is too low
#
# ARGUMENTS:
#           argv(0) == connect string
#           argv(1) == no of occurrance
#           argv(2) == threshold value for alert
#           argv(3) == threshold value for warning
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           The current in memory sort ratio.
#
# $Log:  $
#
# Revision 1.1  1998/02/13  groyal
# Initial revision
#

# Event definition
oradefine event /oracle/rdbms/perf/memsort description=VOC-01303 \
report=VOC-01304
oraarguments connect_str no_occurrence alert_threshold warning_threshold
oravardesc connect_str oracle_signon
oravardesc no_occurrence unsigned default=3 message=VOC-01305
oravardesc alert_threshold number default=90 message=VOC-01306
oravardesc warning_threshold number default=100 message=VOC-01307
oraresults ratio sort_area_size sga_size
oravardesc ratio number
oravardesc sort_area_size unsigned
oravardesc sga_size unsigned
oradefine end


# Initialize global variables

#comment out to fix bug#606739
#set last_report $CLEAR_EVENT
set output ""
set no_occurrence 0
set initial 1
set prev_values(memory_sorts) 0
set prev_values(disk_sorts) 0


# The main event checking functions
proc EvalEvent {} {


    # Declare globals we will use
    global argv last_report output no_occurrence initial
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg prev_values


    # initialize the return code and output
    set ret_code $CLEAR_EVENT
    set output ""
    set err ""

    # get current value
    if {[catch {set curr_values(memory_sorts) [lindex [oradbsnmp get \
        oraDbSysSortsMemory.$oramsg(oraindex)] 1]} err]} {
	lappend output $err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    if {[catch {set curr_values(disk_sorts) [lindex [oradbsnmp get \
        oraDbSysSortsDisk.$oramsg(oraindex)] 1]} err]} {
	lappend output $err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }



    # During initialization, copy current values to previous values
    if {$initial == 1} {
        foreach att_name [array names curr_values] {
            set prev_values($att_name) $curr_values($att_name)
        }
        set initial 0
        return $NO_EVENT
    }


    # calculate delta 
    foreach att_name [array names curr_values] {
        set delta($att_name) [expr $curr_values($att_name) - $prev_values($att_name)]
        set prev_values($att_name) $curr_values($att_name)
    }


    # calculate ratio 
    set v1 [expr double ($delta(memory_sorts))]
    set v2 [expr double ([expr $delta(memory_sorts) + $delta(disk_sorts)])]


    # deal with rollovers
    if { ($v1 < 0) || ($v2 < 0) } {
        ORATCL_DEBUG "memsort : $oramsg(oraobject) : [oratime] : rollover : $v1,$v2"
        return $NO_EVENT
    }


    # watch for divide by 0
    if { $v2 == 0 } {
        ORATCL_DEBUG "memsort: $oramsg(oraobject) : [oratime] : divide by zero : $v2"
        return $NO_EVENT
    }


    # calculate ratio 
    set ratio [expr [divide $v1 $v2] * 100]
    set ratio [format "%3.2f" $ratio]


    # ignore negative ratios
    if { $ratio < 0 } {
        ORATCL_DEBUG "memsort : $oramsg(oraobject) : [oratime] : negative ratio : $v1,$v2,$ratio"
        return $NO_EVENT
    }


    # print out event trace info 
    ORATCL_DEBUG "memsort : $oramsg(oraobject) : [oratime] : $v1,$v2,$ratio"


    # Check for alert and warning threshold
    if {$ratio < [lindex $argv 2]} {
        if {$no_occurrence < [expr [lindex $argv 1]-1]} {
            incr no_occurrence
        } else {
            set ret_code $ALERT_EVENT
            set output $ratio
        }
    } elseif {$ratio < [lindex $argv 3]} {
        if {$no_occurrence < [expr [lindex $argv 1]-1]} {
            incr no_occurrence
        } else {
            set ret_code $WARNING_EVENT
            set output $ratio
        }
    } else {
        set no_occurrence 0
    }


    # return
    if {$last_report == $ret_code} {
        catch {oraclose $cur} err
        catch {oralogoff $lda} err
        return $NO_EVENT
    } else {
        if {($ret_code == $WARNING_EVENT) || ($ret_code == $ALERT_EVENT) } {
            # retrieve current value for the sort_area_size instance parameter
            catch {set temp_list [oradbsnmp get rdbmsSrvParamCurrValue.$oramsg(oraindex).14.115.111.114.116.95.97.114.101.97.95.115.105.122.101.1]} err
            set sort_area_size [divide [lindex $temp_list 1] 1024]

            # connect to the database and open a cursor
            set connect [format "%s@%s" [lindex $argv 0] $oramsg(oraobject)]
            if {[catch {oralogon $connect} lda]} {
                lappend output [msgtxt [RDBMS] ora $oramsg(rc)] ""
                if {$last_report == $SCRIPT_FAIL} {
                    return $NO_EVENT
                } else {
                    set last_report $SCRIPT_FAIL
                    return $SCRIPT_FAIL
                }
            }
	    if {[catch {set cur [oraopen $lda]} err]} {
		lappend output $err
	        catch {oralogoff $lda} err
	        if {$last_report == $SCRIPT_FAIL} {
	            return $NO_EVENT
	        } else {
	            set last_report $SCRIPT_FAIL
	            return $SCRIPT_FAIL
	        }
	    }

            # get total sga size
            set sql {select floor(sum(bytes)/1024) from v$sgastat}
            if { [catch { orasql $cur $sql }] } {
                lappend output [convertout $oramsg(db_characterset) \
                    $oramsg(errortxt)] $sql
                catch {oraclose $cur} err
                catch {oralogoff $lda} err
                if {$last_report == $SCRIPT_FAIL} {
                    return $NO_EVENT
                } else {
                    set last_report $SCRIPT_FAIL
                    return $SCRIPT_FAIL
                }
            }

            catch {set sga_size [orafetch $cur]} err

            catch {oraclose $cur} err
            catch {oralogoff $lda} err

            lappend output $sort_area_size
            lappend output $sga_size
        }


        ORATCL_DEBUG "memsort : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, $output"
        set last_report $ret_code
        return $ret_code 
    }
}

