# $Header: netio.tcl 31-mar-00.10:52:22 sgrover Exp $
#
# copyright (c) 1999 by the Oracle Corporation
#
# NAME:
#
#  netio.tcl : Event tcl file that checks if the network i/o rate (bytes/sec)
#              has exceeded its threshold.
#
# ARGUMENTS:
#           argv(0) == no of occurrance
#           argv(1) == threshold value for alert
#           argv(2) == threshold value for warning
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           the current network i/o rate
#
# $Log:  $
#
# Revision 1.10 1998/02/12  groyal
# improved error messages issued for script evaluation failures
#
# Revision 1.9  1996/06/24  23:54:45  yliu
# fixed bug 370148
#
# Revision 1.8  1996/02/07  01:48:54  yliu
# use character set conversion
#
# Revision 1.7  1996/01/10  01:35:11  yliu
# use message file
#
# Revision 1.6  1995/10/25 01:33:24  yliu
# add connect string to the parameter list
#
# Revision 1.5  1995/10/18  22:38:04  yliu
# moved the init code into EvalEvent
#
# Revision 1.4  1995/10/10  23:36:12  yliu
# added no of occurrence
#

# Event definition

oradefine event /oracle/rdbms/perf/netio description=VOC-01161 \
report=VOC-01162
oraarguments connect_str no_occurrence alert_threshold warning_threshold
oravardesc connect_str oracle_signon 
oravardesc no_occurrence unsigned default=3 message=VOC-01163
oravardesc alert_threshold unsigned message=VOC-01164
oravardesc warning_threshold unsigned message=VOC-01165
oraresults current
oravardesc current unsigned
oradefine end


# Initialize global variables
#comment out to fix bug#606739
#set last_report $CLEAR_EVENT
set output ""
set no_occurrence 0
set initial 1
set prev_second 0
set prev_netio 0


# The main event checking functions
proc EvalEvent {} {


    # Declare globals we will use
    global argv last_report output no_occurrence initial
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg prev_second prev_netio


    # initialize the return code and output
    set ret_code $CLEAR_EVENT
    set output ""
    set err ""

    # get the current values and time
    catch {set curr_second [oraclock]} err
    set connect [format "%s@%s" [lindex $argv 0] $oramsg(oraobject)]
    if {[catch {oralogon $connect} lda]} {
        lappend output [msgtxt [RDBMS] ora $oramsg(rc)] ""
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {[catch {set cur [oraopen $lda]} err]} {
	lappend output $err
        catch {oralogoff $lda} err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    set sql {select sum(value) from v$sysstat where name like 'bytes%SQL*Net%'}
    if {[catch {orasql $cur $sql}]} {
        lappend output [convertout $oramsg(db_characterset) \
        $oramsg(errortxt)] $sql
        catch {oraclose $cur} err
        catch {oralogoff $lda}
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {[catch {set row [orafetch $cur]} err]} {
	lappend output $err
        catch {oraclose $cur} err
        catch {oralogoff $lda}
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    if {$oramsg(rows) == 0} {
        lappend output [msgtxt [NETWORK] nms 1008] $sql
        catch {oraclose $cur} err
        catch {oralogoff $lda}
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    set curr_netio [lindex $row 0]
    catch {oraclose $cur} err
    catch {oralogoff $lda}


    # During initialization, copy current values to previous values
    if {$initial == 1} {
        set prev_second $curr_second
        set prev_netio $curr_netio
        set initial 0
        return $NO_EVENT
    }


    # Calculate network i/o rate
    set duration [expr $curr_second - $prev_second]
    set delta_netio [expr $curr_netio - $prev_netio]


    if { $duration < 0 || $duration == 0 } {
        lappend output [format [msgtxt [NETWORK] nms 1005] "netio.tcl"]  
        lappend output [msgtxt [NETWORK] nms 1081]

        ORATCL_DEBUG "netio : $oramsg(oraobject) : [oratime] : invalid duration : $duration" 

        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }


    # deal with rollovers
    if { $delta_netio < 0 } {
        ORATCL_DEBUG "netio : $oramsg(oraobject) : [oratime] : rollover : $delta_netio"
        return $NO_EVENT
    }

    set rate [divide $delta_netio $duration]

    set prev_second $curr_second
    set prev_netio $curr_netio


    # print out event trace info 
    ORATCL_DEBUG "netio : $oramsg(oraobject) : [oratime] : $curr_netio, $rate"


    # Check for alert and warning threshold
    if {$rate >= [lindex $argv 2]} {
        if {$no_occurrence < [expr [lindex $argv 1]-1]} {
            incr no_occurrence
        } else {
            set ret_code $ALERT_EVENT
            set output $rate
        }
    } elseif {$rate >= [lindex $argv 3]} {
        if {$no_occurrence < [expr [lindex $argv 1]-1]} {
            incr no_occurrence
        } else {
            set ret_code $WARNING_EVENT
            set output $rate
        }
    } else {
        set no_occurrence 0
    } 


    # retunr
    if {$last_report == $ret_code} {
        return $NO_EVENT
    } else {
        ORATCL_DEBUG "netio : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, $output"
        set last_report $ret_code
        return $ret_code 
    }


}

