# $Header: redologalloc 30-mar-99.15:57:18 dholail Exp $
#
# copyright (c) 1999 by the Oracle Corporation
#
# NAME:
#
#  redolog.tcl : Event tcl file that checks if the redo log 
#                allocation hit ratio is too low
#
# ARGUMENTS:
#           argv(0) == connect string
#           argv(1) == no of occurrences
#           argv(2) == threshold value for alert
#           argv(3) == threshold value for warning
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           The current redo log allocation hit ratio.
#
# $Log:  $
#
# Revision 1.1  1998/02/16  groyal
# Initial revision
#

# Event definition
oradefine event /oracle/rdbms/perf/redolog description=VOC-01335 \
report=VOC-01336
oraarguments connect_str no_occurrence alert_threshold warning_threshold
oravardesc connect_str oracle_signon
oravardesc no_occurrence unsigned default=3 message=VOC-01337
oravardesc alert_threshold number minimum=0 maximum=100 default=98 message=VOC-01338
oravardesc warning_threshold number minimum=0 maximum=100 default=100 message=VOC-01339
oraresults ratio log_buffer
oravardesc ratio number
oravardesc log_buffer unsigned
oradefine end


# Initialize global variables

#comment out to fix bug#606739
#set last_report $CLEAR_EVENT
set output ""
set no_occurrence 0
set initial 1
set prev_values(redo_entries) 0
set prev_values(redo_retries) 0


# The main event checking functions
proc EvalEvent {} {


    # Declare globals we will use
    global argv last_report output no_occurrence initial
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg prev_values


    # initialize the return code and output
    set ret_code $CLEAR_EVENT
    set output ""
    set err ""

    # get current values

    # connect to the database and open a cursor
    set connect [format "%s@%s" [lindex $argv 0] $oramsg(oraobject)]
    if {[catch {oralogon $connect} lda]} {
        lappend output [msgtxt [RDBMS] ora $oramsg(rc)] ""
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {[catch {set cur [oraopen $lda]} err]} {
	lappend output $err
        catch {oralogoff $lda}
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    # get free buffer waits 
    set sql {select value from v$sysstat 
             where name = 'redo buffer allocation retries'}
    if { [catch { orasql $cur $sql }] } {
        lappend output [convertout $oramsg(db_characterset) \
            $oramsg(errortxt)] $sql
        catch {oraclose $cur} err
        catch {oralogoff $lda} err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {[catch {set curr_values(redo_retries) [orafetch $cur]} err]} {
	lappend output $err
        catch {oraclose $cur} err
        catch {oralogoff $lda}
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    
    # log off
    catch {oraclose $cur} err
    catch {oralogoff $lda} err


    if {[catch {set curr_values(redo_entries) [lindex [oradbsnmp get \
        oraDbSysRedoEntries.$oramsg(oraindex)] 1]} err]} {
	lappend output $err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    # During initialization, copy current values to previous values
    if {$initial == 1} {
        foreach att_name [array names curr_values] {
            set prev_values($att_name) $curr_values($att_name)
        }
        set initial 0
        return $NO_EVENT
    }


    # calculate delta 
    foreach att_name [array names curr_values] {
        set delta($att_name) [expr $curr_values($att_name) - $prev_values($att_name)]
        set prev_values($att_name) $curr_values($att_name)
    }


    set v1 [expr double ($delta(redo_retries))]
    set v2 [expr double ($delta(redo_entries))]


    # deal with rollovers
    if { ($v1 < 0) || ($v2 < 0) } {
        ORATCL_DEBUG "redolog : $oramsg(oraobject) : [oratime] : rollover : $v1,$v2"
        return $NO_EVENT
    }


    # watch out for divide by 0
    if { $v2 == 0 } {
        ORATCL_DEBUG "redolog : $oramsg(oraobject) : [oratime] : divide by zero : $v2"
        return $NO_EVENT

    }


    # calculate ratio 
    set ratio [expr [expr 1 - [divide $v1 $v2]] * 100]
    set ratio [format "%3.2f" $ratio]

  
    # ignore negative ratios
    if { $ratio < 0 } {
        ORATCL_DEBUG "redolog : $oramsg(oraobject) : [oratime] : negative ratio : $v1,$v2,$ratio"
        return $NO_EVENT
    }


    # print out event trace info 
    ORATCL_DEBUG "redolog : $oramsg(oraobject) : [oratime] : $v1,$v2,$ratio"


    # Check for alert and warning threshold
    if {$ratio < [lindex $argv 2]} {
        if {$no_occurrence < [expr [lindex $argv 1]-1]} {
            incr no_occurrence
        } else {
            set ret_code $ALERT_EVENT

            # retrieve current value for the log_buffer instance parameter
            catch {set temp_list [oradbsnmp get rdbmsSrvParamCurrValue.$oramsg(oraindex).10.108.111.103.95.98.117.102.102.101.114.1]} err
            set log_buffer [divide [lindex $temp_list 1] 1024]

            lappend output $ratio $log_buffer
        }
    } elseif {$ratio < [lindex $argv 3]} {
        if {$no_occurrence < [expr [lindex $argv 1]-1]} {
            incr no_occurrence
        } else {
            set ret_code $WARNING_EVENT

            # retrieve current value for the log_buffer instance parameter
            catch {set temp_list [oradbsnmp get rdbmsSrvParamCurrValue.$oramsg(oraindex).10.108.111.103.95.98.117.102.102.101.114.1]} err
            set log_buffer [divide [lindex $temp_list 1] 1024]

            lappend output $ratio $log_buffer
        }
    } else {
        set no_occurrence 0
    }


    # return
    if {$last_report == $ret_code} {
        return $NO_EVENT
    } else {
        ORATCL_DEBUG "redolog : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, $output"
        set last_report $ret_code
        return $ret_code 
    }

}

