# $Header: ddcache.tcl 20-mar-00.11:30:59 sgrover Exp $
#
# copyright (c) 1999 by the Oracle Corporation
#
# NAME:
#
#  ddcache.tcl : Event tcl file that checks if the Data Dictionary cache miss
#                ratio is too high
#
# ARGUMENTS:
#           argv(0) == no of occurrance
#           argv(1) == threshold value for alert
#           argv(2) == threshold value for warning
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           The current data dictionary cache miss ratio
#
# $Log:  $
# Revision 1.9  1998/02/12  groyal
# improved error messages issued for script evaluation failures
#
# Revision 1.8  1996/06/24  23:54:18  yliu
# fixed bug 370148
#
# Revision 1.7  1996/02/07  01:47:51  yliu
# use character set conversion
#
# Revision 1.6  1996/01/10  01:31:11  yliu
# use message file
#
# Revision 1.5  1995/10/25 01:26:51  yliu
# add connect string to the parameter list
#
# Revision 1.4  1995/10/18  23:59:54  yliu
#  moved the init code into EvalEvent
#
# Revision 1.3  1995/10/10  23:29:39  yliu
# added no of occurrence
#

# Event definition
oradefine event /oracle/rdbms/perf/ddcache description=VOC-01131 \
report=VOC-01132
oraarguments connect_str no_occurrence alert_threshold warning_threshold
oravardesc connect_str oracle_signon
oravardesc no_occurrence unsigned default=3 message=VOC-01133
oravardesc alert_threshold number minimum=0 maximum=100 default=15 message=VOC-01134
oravardesc warning_threshold number minimum=0 maximum=100 default=10 message=VOC-01135
oraresults ratio
oravardesc ratio number
oradefine end


# Initialize global variables
#comment out to fix bug#606739
#set last_report $CLEAR_EVENT
set output ""
set no_occurrence 0
set initial 1
set prev_values(dc_gets) 0
set prev_values(dc_getmisses) 0


# The main event checking functions
proc EvalEvent {} {


    # Declare globals we will use
    global argv last_report output no_occurrence initial
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg prev_values


    # initialize the return code and output
    set ret_code $CLEAR_EVENT
    set output ""
    set err ""

    # get current value
    set connect [format "%s@%s" [lindex $argv 0] $oramsg(oraobject)]
    if {[catch {oralogon $connect} lda]} {
        lappend output [msgtxt [RDBMS] ora $oramsg(rc)] ""
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {[catch {set cur [oraopen $lda]} err]} {
	lappend output $err
        catch {oralogoff $lda} err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    set sql {select sum(gets), sum(getmisses) from v$rowcache}
    if {[catch {orasql $cur $sql}]} {
        lappend output [convertout $oramsg(db_characterset) \
            $oramsg(errortxt)] $sql
        catch {oraclose $cur} err
        catch {oralogoff $lda} err
	lappend output $err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {[catch {set row [orafetch $cur]} err]} {
	lappend output $err
        catch {oraclose $cur} err
        catch {oralogoff $lda} err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {$oramsg(rows) == 0} {
        lappend output [msgtxt [NETWORK] nms 1015] $sql
        catch {oraclose $cur} err
        catch {oralogoff $lda} err
        lappend output $err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    set curr_values(dc_gets) [lindex $row 0]
    set curr_values(dc_getmisses) [lindex $row 1]
    catch {oraclose $cur} err
    catch {oralogoff $lda} err


    # During initialization, copy current values to previous values
    if {$initial == 1} {
        foreach att_name [array names curr_values] {
            set prev_values($att_name) $curr_values($att_name)
        }
        set initial 0
        return $NO_EVENT
    }


    # calculate delta 
    foreach att_name [array names curr_values] {
        set delta($att_name) [expr $curr_values($att_name) - $prev_values($att_name)]
        set prev_values($att_name) $curr_values($att_name)
    }

    set v1 [expr double ($delta(dc_getmisses))]
    set v2 [expr double ($delta(dc_gets))]

    # deal with rollovers
    if { ($v1 < 0) || ($v2 < 0) } {
        ORATCL_DEBUG "ddcache : $oramsg(oraobject) : [oratime] : rollover - $v1,$v2"
        return $NO_EVENT
    }

    # watch for divide by 0
    if { $v2 == 0 } {
        return $NO_EVENT
    }


    # calculate ratio 
    set ratio [expr [divide $v1 $v2] * 100]
    set ratio [format "%3.2f" $ratio]


    # ignore negative ratios
    if { $ratio < 0 } {
        ORATCL_DEBUG "ddcache : $oramsg(oraobject) : [oratime] : negative ratio - $v1,$v2,$ratio"
        return $NO_EVENT
    }


    # print out event trace info 
    ORATCL_DEBUG "ddcache: $oramsg(oraobject) : [oratime] : $v1,$v2,$ratio"


    # Check for alert and warning threshold
    if {$ratio >= [lindex $argv 2]} {
        if {$no_occurrence < [expr [lindex $argv 1]-1]} {
            incr no_occurrence
        } else {
            set ret_code $ALERT_EVENT
            set output $ratio
        }
    } elseif {$ratio >= [lindex $argv 3]} {
        if {$no_occurrence < [expr [lindex $argv 1]-1]} {
            incr no_occurrence
        } else {
            set ret_code $WARNING_EVENT
            set output $ratio
        }
    } else {
        set no_occurrence 0
    } 


    # return
    if {$last_report == $ret_code} {
        return $NO_EVENT
    } else {
        ORATCL_DEBUG "ddcache : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, $output"
        set last_report $ret_code
        return $ret_code 
    }
}

