# $Header: userlmt.tcl 30-mar-99.15:57:18 dholail Exp $
#
# copyright (c) 1999 by the Oracle Corporation
#
# NAME:
#
#  userlmt.tcl : Event tcl file that checks if the number of users logged on is 
#                reaching its limit.
#
# ARGUMENTS:
#           argv(0) == threshold value for alert
#           argv(1) == threshold value for warning
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           no of users currently logged on, and its limit
#
# $Log:  $
# Revision 1.6  1996/06/24  23:56:13  yliu
# fixed bug 370148
#
# Revision 1.5  1996/02/07  01:55:34  yliu
# use character set conversion
#
# Revision 1.4  1996/01/10  02:02:58  yliu
# use message file
#
# Revision 1.3  1995/10/25 17:59:57  yliu
# add connect string to the parameter list
#
# Revision 1.2  1995/09/28  22:03:03  yliu
# use message file
#
# Revision 1.1  1995/09/15  20:25:36  yliu
# Initial revision
#


# Event definition
oradefine event /oracle/rdbms/resource/userlmt description=VOC-01101 \
report=VOC-01102
oraarguments connect_str alert_threshold warning_threshold
oravardesc connect_str oracle_signon
oravardesc alert_threshold unsigned minimum=0 maximum=100 default=95 message=VOC-01103
oravardesc warning_threshold unsigned minimum=0 maximum=100 default=90 message=VOC-01104
oraresults currents limits
oravardesc currents int
oravardesc limits int
oradefine end


# Initialize global variables
#comment out to fix bug#606739
#set last_report $CLEAR_EVENT
set output ""


# The main event checking functions
proc EvalEvent {} {


    # Declare globals we will use
    global argv last_report output
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg


    # initialize the return code and output
    set ret_code $CLEAR_EVENT
    set output ""
    set err ""

    # get no of users logged on, the limit and their ratio
    set connect [format "%s@%s" [lindex $argv 0] $oramsg(oraobject)]
    if {[catch {oralogon $connect} lda]} {
        lappend output [msgtxt [RDBMS] ora $oramsg(rc)] ""
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {[catch {set cur [oraopen $lda]} err]} {
	lappend output $err
        catch {oralogoff $lda} err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    set sql {select sessions_current, sessions_max from v$license}
    if {[catch {orasql $cur $sql}]} {
        lappend output [convertout $oramsg(db_characterset) \
            $oramsg(errortxt)] $sql
        catch {oraclose $cur} err
        catch {oralogoff $lda} err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }
    if {[catch {set row [orafetch $cur]} err]} {
	lappend output $err
        catch {oraclose $cur} err
        catch {oralogoff $lda} err
        if {$last_report == $SCRIPT_FAIL} {
            return $NO_EVENT
        } else {
            set last_report $SCRIPT_FAIL
            return $SCRIPT_FAIL
        }
    }

    set current [lindex $row 0]
    set limit [lindex $row 1]
    catch {oraclose $cur} err
    catch {oralogoff $lda} err

    if { $limit == 0 } {
        set ratio 0
    } else {
        set ratio [divide [expr 100 * $current] $limit] 
    }


    # print out event trace info 
    ORATCL_DEBUG "userlmt : $oramsg(oraobject) : [oratime] : $current, $limit, $ratio"


    # Check ratio against alert and warning thresholds
    if {$ratio > [lindex $argv 1]} {
        set ret_code $ALERT_EVENT
        lappend output $current $limit
    } elseif {$ratio > [lindex $argv 2]} {
        set ret_code $WARNING_EVENT
        lappend output $current $limit
    } 


    # return
    if {$last_report == $ret_code} {
        return $NO_EVENT
    } else {
        ORATCL_DEBUG "userlmt : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, $output"
        set last_report $ret_code
        return $ret_code 
    }
}

