#include "ReShade.fxh"
#include "ReShadeUI.fxh"

#define ADAPTIVE_TONEMAPPER_SMALL_TEX_SIZE 256
#define ADAPTIVE_TONEMAPPER_SMALL_TEX_MIPLEVELS 9

static const int2 AdaptResolution = ADAPTIVE_TONEMAPPER_SMALL_TEX_SIZE;
static const int AdaptMipLevels = ADAPTIVE_TONEMAPPER_SMALL_TEX_MIPLEVELS;

#ifndef USE_HDR_BLOOM
#define USE_HDR_BLOOM 1
#endif

#ifndef USE_ADAPTATION
#define USE_ADAPTATION 0
#endif

#ifndef BlurSamples
#define BlurSamples 4
#endif

#ifndef DOWNSAMPLE
#define DOWNSAMPLE 4
#endif

static const float sigma = float(BlurSamples) / 2.0;
static const float double_pi = 6.283185307179586476925286766559;
static const int2 DownsampleAmount = DOWNSAMPLE;
#define LumCoeff float3(0.299, 0.587, 0.114)
uniform float FrameTime < source = "frametime";>;
#define PixelSize rcp(ReShade::ScreenSize)
static const float EPSILON = 1e-3;
// ------------------------------------
// ------------------------------------
// ------------------------------------
uniform float ToeStrength
<
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_step    = 0.01;
	ui_label = "Toe Strength";
	ui_min = 0.0;
	ui_max = 1.0;
> = 0.0;

uniform float ToeLength
<
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_step    = 0.01;
	ui_label = "Toe Length";
	ui_min = 0.0;
	ui_max = 1.0;
> = 1.0;

uniform float ShoulderStrength
<
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_step    = 0.01;
	ui_label = "Shoulder Strength";
	ui_min = 0.0;
	ui_max = 3.0;
> = 0.0;

uniform float ShoulderLength
<
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_step    = 0.01;
	ui_label = "Shoulder Length";
	ui_min = 0.0;
	ui_max = 3.0;
> = 0.0;

uniform float ShoulderAngle
<
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_step    = 0.01;
	ui_label = "Shoulder Angle";
	ui_min = 0.0;
	ui_max = 3.0;
> = 0.0;


uniform float Gamma<
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_label = "Gamma";
	ui_min = 0.0;
	ui_max = 5.0;
	ui_step    = 0.01;
> = 1.0;

uniform float brightness <
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_label = "Brightness";
	ui_min = -1.0;
	ui_max = 1.0;
	ui_step    = 0.01;
> = 0.0;
/*
uniform float WhitePoint <
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_step = 0.01;
	ui_min = 0.0; ui_max = 1.0;
	ui_label = "White Point";
	ui_tooltip = "The black point is the new black - literally. Everything darker than this will become completely black.";
> = 0.0;
*/
uniform float BlackPoint <
	ui_category = "Filmic Tonemap";
      ui_type = "slider";
	ui_step = 0.01;
	ui_min = 0.0; ui_max = 1.0;
	ui_label = "Black Point";
	ui_tooltip = "The black point is the new black - literally. Everything darker than this will become completely black.";
> = 0.0;

uniform float  HDRPower <
	ui_category = "Filmic Tonemap";
	ui_label   = "Local ToneMapping";
      ui_type = "slider";
	ui_min     = 0;
	ui_max     = 1.0;
	ui_step    = 0.01;
> = 0.0;

uniform float  sldClarity <
	ui_category = "Filmic Tonemap";
	ui_label   = "Clarity";
	ui_type = "slider";
	ui_min     = 0;
	ui_max     = 1.0;
	ui_step    = 0.01;
> = 0.81;

uniform float Exposure<
	ui_category = "High-Dynamic Range";
      ui_type = "slider";
	ui_step    = 1.0;
	ui_label = "HDR Exposure";
	ui_min = 0.0;
	ui_max = 5.0;
> = 0.0;

uniform float HDR_WHITEPOINT <
	ui_category = "High-Dynamic Range";
      ui_type = "slider";
	ui_step = 1.0;
	ui_min = 1.0; ui_max = 100.0;
    ui_label = "HDR Whitepoint";
> = 50.0;

#if USE_HDR_BLOOM == 1
uniform float bloom_amount <
	ui_category = "High-Dynamic Range";
      ui_type = "slider";
	ui_min = 0; ui_max = 1.0;
    ui_step = 0.01;
    ui_label = "Bloom Intensity";
> = 0.0;

uniform int uDebugOptions <
	ui_category = "High-Dynamic Range";
	ui_label = "Debug Options";
	ui_category = "Debug";
	ui_type = "combo";
	ui_items = "None\0Display Texture\0";
> = 0;
#endif

#if USE_ADAPTATION == 1
uniform float AdaptBrightness<
	ui_category = "Adaptation";
	ui_category_closed = true;
	ui_type = "drag";
	ui_step    = 0.01;
	ui_label = "Adapt Brightness<";
	ui_min = 0.0;
	ui_max = 5.0;
> = 0.1;

uniform float2 AdaptRange
<
	ui_label = "Range";
	ui_category = "Adaptation";
	ui_type = "drag";
	ui_min = 0.001;
	ui_max = 1.0;
	ui_step = 0.001;
> = float2(0.05, 0.2);

uniform float AdaptTime
<
	ui_label = "Time";
	ui_category = "Adaptation";
	ui_type = "drag";
	ui_min = 0.0;
	ui_max = 3.0;
	ui_step = 0.01;
> = 1.0;

uniform float AdaptSensitivity
<
	ui_label = "Sensititvity";
	ui_category = "Adaptation";
	ui_type = "drag";
	ui_min = 0.0;
	ui_max = 3.0;
	ui_step = 0.01;
> = 1.0;

uniform int AdaptPrecision
<
	ui_label = "Precision";
	ui_category = "Adaptation";
	ui_type = "drag";
	ui_min = 0;
	ui_max = 8.0;
	ui_max = AdaptMipLevels;
> = 0;

uniform float2 AdaptFocalPoint
<
	ui_label = "Focal Point";
	ui_type = "drag";
	ui_category = "Adaptation";
	ui_min = 0.0;
	ui_max = 1.0;
	ui_step = 0.001;
> = 0.48;
#endif
// ------------------------------------
// ------------------------------------
texture texBlur
{
	Width = BUFFER_WIDTH / DownsampleAmount.x;
	Height = BUFFER_HEIGHT / DownsampleAmount.y;
	Format = RGBA16F;
};

sampler SamplerBlur { Texture = texBlur; };

#if USE_ADAPTATION == 1
texture SmallTex
{
	Width = AdaptResolution.x;
	Height = AdaptResolution.y;
	Format = R16F;
	MipLevels = AdaptMipLevels;
};
sampler Small
{
	Texture = SmallTex;
};

texture LastAdaptTex
{
	Format = R16F;
};
sampler LastAdapt
{
	Texture = LastAdaptTex;
	MinFilter = POINT;
	MagFilter = POINT;
	MipFilter = POINT;
};
#endif
// ------------------------------------
// ------------------------------------

#if USE_HDR_BLOOM == 1
texture Bloom0Tex
{ 
	Width = BUFFER_WIDTH / DownsampleAmount.x;
	Height = BUFFER_HEIGHT / DownsampleAmount.y;
	Format = RGBA16F;
}; 

sampler sBloom0Tex
{ 
	Texture = Bloom0Tex; 
};

texture Bloom1Tex
{ 
	Width = BUFFER_WIDTH / DownsampleAmount.x / 2;
	Height = BUFFER_HEIGHT / DownsampleAmount.y / 2;
	Format = RGBA16F;
}; 

sampler sBloom1Tex
{ 
	Texture = Bloom1Tex; 
};

texture Bloom2Tex
{ 
	Width = BUFFER_WIDTH / DownsampleAmount.x / 4;
	Height = BUFFER_HEIGHT / DownsampleAmount.y / 4;
	Format = RGBA16F;
}; 

sampler sBloom2Tex
{ 
	Texture = Bloom2Tex; 
};

texture Bloom3Tex
{ 
	Width = BUFFER_WIDTH / DownsampleAmount.x / 8;
	Height = BUFFER_HEIGHT / DownsampleAmount.y / 8;
	Format = RGBA16F;
}; 

sampler sBloom3Tex
{ 
	Texture = Bloom3Tex; 
};

texture Bloom4Tex
{ 
	Width = BUFFER_WIDTH / DownsampleAmount.x / 16;
	Height = BUFFER_HEIGHT / DownsampleAmount.y / 16;
	Format = RGBA16F;
}; 

sampler sBloom4Tex
{ 
	Texture = Bloom4Tex; 
};
#endif
// ------------------------------------
// ------------------------------------
float3 srgb_to_linear(float3 srgb)
{
	return (srgb < 0.04045) ? srgb / 12.92 : pow(abs((srgb + 0.055) / 1.055), 2.4);
}

float3 linear_to_srgb(float3 lin)
{
	return (lin < 0.0031308) ? 12.92 * lin : 1.055 * pow(abs(lin), 0.41666666) - 0.055;
}


float3 srgb_to_acescg(float3 srgb)
{
    float3x3 m = float3x3(  0.613097, 0.339523, 0.047379,
                            0.070194, 0.916354, 0.013452,
                            0.020616, 0.109570, 0.869815);
    return mul(m, srgb);           
}

float3 acescg_to_srgb(float3 acescg)
{     
    float3x3 m = float3x3(  1.704859, -0.621715, -0.083299,
                            -0.130078,  1.140734, -0.010560,
                            -0.023964, -0.128975,  1.153013);                 
    return mul(m, acescg);            
}

float3 sdr_to_hdr(float3 color, float w)
{ 
                  float toe_length = pow(abs(ToeLength), 2.2);
                  color = (color - ToeStrength) / (1.0f - ToeStrength) * toe_length;

                  float a = 1.04f + exp2(-w); 

                  color = color * rcp(a - saturate(color));
                  color = color * (1.0 - 0.48) + 0.48;
                  return color;
}

float3 sdr_to_hdr_lum(float3 color, float w)
{ 
                  float lum = dot(color, LumCoeff);
                  float a = 1.04f + exp2(-w); 
                  color *= lum * rcp(a - saturate(lum));
                  return color;
}

float3 hdr_to_sdr(float3 color, float w)
{    
                  float a = 1.04f + exp2(-w);

                  color = pow(color, Gamma);
                  float extra_w = exp2(ShoulderStrength);
                  float toe_length = pow(abs(ToeLength), 2.2);
                  color = color * ((ShoulderStrength * color + 1.0 + ShoulderLength) + extra_w) / (ShoulderAngle * color + ToeStrength + 1.0);

	            color *= 1.0 + brightness;

                  color = a * color * rcp(color + 1.0);

                  return color;
}

// ------------------------------------

float gaussian_function(float2 i) {
    static const float first_part = 1.0 / (double_pi * pow(sigma, 2.0));
    static const float second_part_a = 1.0 / (2.0 * pow(sigma, 2.0));
    float second_part_b = (pow(i.x, 2.0) + pow(i.y, 2.0)) * second_part_a;
    return first_part * exp(-second_part_b);
}

float3 blur(sampler sp, float2 uv, float scale) {
    float2 ps = PixelSize * scale;

    float accum = 0.0;
    float gaussian_weight = 0.0;
    float3 col = 0.0;
    
    [loop]
    for (int x = -BlurSamples; x <= BlurSamples; ++x) {
        for (int y = -BlurSamples; y <= BlurSamples; ++y) {
            gaussian_weight = gaussian_function(float2(x, y));
            accum += gaussian_weight;
            col += tex2D(sp, uv + ps * float2(x, y)).rgb * gaussian_weight;
        }
    }

    return col / accum;
}

//-----------------------------
#if USE_ADAPTATION == 1
float get_adapt()
{
	return tex2Dlod(
		Small,
		float4(AdaptFocalPoint, 0, AdaptMipLevels - AdaptPrecision)).x;
}

//pixel shaders
float4 GetSmallPS(float4 p : SV_POSITION, float2 uv : TEXCOORD) : SV_TARGET
{
                  float4 color = tex2D(ReShade::BackBuffer, uv);
                  color = srgb_to_linear(color);
                  float adapt = dot(color, LumCoeff);
                  adapt *= AdaptSensitivity;

                  float last = tex2Dfetch(LastAdapt, 0).x;

                  if (AdaptTime > 0.0)
		adapt = lerp(last, adapt, saturate((FrameTime * 0.001) / AdaptTime));

                  return adapt;
}

float4 SaveAdaptPS(float4 p : SV_POSITION, float2 uv : TEXCOORD) : SV_TARGET
{
                  return get_adapt();
}
#endif

//-----------------------------
void PS_LargeBlur(in float4 position : SV_POSITION, in float2 uv : TEXCOORD, out float4 color : SV_Target)
{
                  color = blur(ReShade::BackBuffer, uv, 4);
}
//-----------------------------
#if USE_HDR_BLOOM == 1
void Bloom0Pass(in float4 position : SV_POSITION, in float2 uv : TEXCOORD, out float4 color : SV_Target)
{
                  color = blur(SamplerBlur, uv, 2);
                  color = srgb_to_linear(color);
                  color *= sdr_to_hdr(color, 1.0 / HDR_WHITEPOINT);
                  color *= exp2(Exposure);
}

void Bloom1Pass(in float4 position : SV_POSITION, in float2 uv : TEXCOORD, out float4 color : SV_Target)
{
                  color = blur(sBloom0Tex, uv, 4);
}

void Bloom2Pass(in float4 position : SV_POSITION, in float2 uv : TEXCOORD, out float4 color : SV_Target)
{
                  color = blur(sBloom1Tex, uv, 8);
}

void Bloom3Pass(in float4 position : SV_POSITION, in float2 uv : TEXCOORD, out float4 color : SV_Target)
{
                  color = blur(sBloom2Tex, uv, 16);
}

void Bloom4Pass(in float4 position : SV_POSITION, in float2 uv : TEXCOORD, out float4 color : SV_Target)
{
                  color = blur(sBloom3Tex, uv, 32);
}
#endif
//-----------------------------
//-----------------------------
// ------------------------------------
#if USE_ADAPTATION == 1
void MainVS(
	uint id : SV_VERTEXID,
	out float4 p : SV_POSITION,
	out float2 uv : TEXCOORD0,
	out float exposure : TEXCOORD1)
{
                  PostProcessVS(id, p, uv);
                  exposure = AdaptBrightness;

                  float adapt = get_adapt();
                  adapt = clamp(adapt, AdaptRange.x, AdaptRange.y);
                  exposure /= adapt;
}
#endif

float4 MainPS(float4 p : SV_POSITION, float2 uv : TEXCOORD) : SV_TARGET
{
                  float4 color = tex2D(ReShade::BackBuffer, uv);

                  float3 largeblur = blur(SamplerBlur, uv, 16);
                  float ar = dot(color, LumCoeff);
                  float br = dot(largeblur, LumCoeff);

                  float Clarity = (0.5 + ar - br);
                  Clarity = lerp(2 * Clarity + ar * (1 - 2 * Clarity), 2 * (1 - Clarity) + (2 * Clarity - 1) * rsqrt(ar), ar > br);
                  color *= lerp(1.0, Clarity, sldClarity);

                  color = srgb_to_linear(color);
                  color *= sdr_to_hdr(color, HDR_WHITEPOINT);
                  color *= exp2(Exposure);

#if USE_HDR_BLOOM == 1

                  float4 bloom = tex2D(sBloom0Tex, uv)
                               + tex2D(sBloom1Tex, uv)
                               + tex2D(sBloom2Tex, uv)
                               + tex2D(sBloom3Tex, uv)
                               + tex2D(sBloom4Tex, uv);

                  bloom *= bloom_amount / HDR_WHITEPOINT;
                  color = uDebugOptions == 1 ? bloom : color + bloom;
#endif

#if USE_ADAPTATION == 1
                  float adapt = get_adapt();
                  float exposure = AdaptBrightness;
                  adapt = clamp(adapt, AdaptRange.x, AdaptRange.y);
                  exposure /= adapt;
                  color *= exposure;
#endif

                  br = log(br);
                  float sqrta = sqrt(ar);
                  float HDRToning = sqrta * lerp(sqrta*(2*ar*br-ar-2*br+2.0), (2*sqrta*br-2*br+1), br > 0.5);
                  color = color / (ar+1e-6) * lerp(ar, HDRToning, HDRPower);

                  color = hdr_to_sdr(color, HDR_WHITEPOINT);
                  color = linear_to_srgb(color);
                  color = (color - BlackPoint) / (1.0f - BlackPoint);

                  return color;
}

// ------------------------------------
// ------------------------------------

technique TrueHDR
{
  pass  {
        VertexShader = PostProcessVS;
        PixelShader = PS_LargeBlur;
	RenderTarget = texBlur;
    }

#if USE_HDR_BLOOM == 1
	pass
	{
		VertexShader = PostProcessVS;
		PixelShader = Bloom0Pass;
                                RenderTarget = Bloom0Tex;
	}

	pass
	{
		VertexShader = PostProcessVS;
		PixelShader = Bloom1Pass;
                                RenderTarget = Bloom1Tex;
	}

	pass
	{
		VertexShader = PostProcessVS;
		PixelShader = Bloom2Pass;
                                RenderTarget = Bloom2Tex;
	}

	pass
	{
		VertexShader = PostProcessVS;
		PixelShader = Bloom3Pass;
                                RenderTarget = Bloom3Tex;
	}

	pass
	{
		VertexShader = PostProcessVS;
		PixelShader = Bloom4Pass;
                                RenderTarget = Bloom4Tex;
	}
#endif

#if USE_ADAPTATION == 1
	pass GetSmall
	{
		VertexShader = PostProcessVS;
		PixelShader = GetSmallPS;
		RenderTarget = SmallTex;
	}
	pass SaveAdapt
	{
		VertexShader = PostProcessVS;
		PixelShader = SaveAdaptPS;
		RenderTarget = LastAdaptTex;
	}
#endif

	pass
	{
		VertexShader = PostProcessVS;
		PixelShader = MainPS;
	}

}