# coding: utf-8
from __future__ import unicode_literals

import time
import hashlib
import re

from .common import InfoExtractor
from ..utils import (
    ExtractorError,
    unescapeHTML,
    unified_strdate,
    urljoin,
)
from ..utilsEX import downloadWebPage_BYHeadlessBrowser_localFile

class DouyuTVIE(InfoExtractor):
    IE_DESC = '斗鱼'
    _VALID_URL = r'https?://(?:www\.)?douyu(?:tv)?\.com/(?:[^/]+/)*(?P<id>[A-Za-z0-9]+)'
    _TESTS = [{
        'url': 'http://www.douyutv.com/iseven',
        'info_dict': {
            'id': '17732',
            'display_id': 'iseven',
            'ext': 'flv',
            'title': 're:^清晨醒脑！根本停不下来！ [0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}$',
            'description': r're:.*m7show@163\.com.*',
            'thumbnail': r're:^https?://.*\.jpg$',
            'uploader': '7师傅',
            'is_live': True,
        },
        'params': {
            'skip_download': True,
        },
    }, {
        'url': 'http://www.douyutv.com/85982',
        'info_dict': {
            'id': '85982',
            'display_id': '85982',
            'ext': 'flv',
            'title': 're:^小漠从零单排记！——CSOL2躲猫猫 [0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}$',
            'description': 'md5:746a2f7a253966a06755a912f0acc0d2',
            'thumbnail': r're:^https?://.*\.jpg$',
            'uploader': 'douyu小漠',
            'is_live': True,
        },
        'params': {
            'skip_download': True,
        },
        'skip': 'Room not found',
    }, {
        'url': 'http://www.douyutv.com/17732',
        'info_dict': {
            'id': '17732',
            'display_id': '17732',
            'ext': 'flv',
            'title': 're:^清晨醒脑！根本停不下来！ [0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}$',
            'description': r're:.*m7show@163\.com.*',
            'thumbnail': r're:^https?://.*\.jpg$',
            'uploader': '7师傅',
            'is_live': True,
        },
        'params': {
            'skip_download': True,
        },
    }, {
        'url': 'http://www.douyu.com/xiaocang',
        'only_matching': True,
    }, {
        # \"room_id\"
        'url': 'http://www.douyu.com/t/lpl',
        'only_matching': True,
    }]

    def _real_extract(self, url):
        video_id = self._match_id(url)

        if video_id.isdigit():
            room_id = video_id
        else:
            page = self._download_webpage(url, video_id)
            room_id = self._html_search_regex(
                r'"room_id\\?"\s*:\s*(\d+),', page, 'room id')

        # Grab metadata from mobile API
        room = self._download_json(
            'http://m.douyu.com/html5/live?roomId=%s' % room_id, video_id,
            note='Downloading room info')['data']

        # 1 = live, 2 = offline
        if room.get('show_status') == '2':
            raise ExtractorError('Live stream is offline', expected=True)

        # Grab the URL from PC client API
        # The m3u8 url from mobile API requires re-authentication every 5 minutes
        tt = int(time.time())
        signContent = 'lapi/live/thirdPart/getPlay/%s?aid=pcclient&rate=0&time=%d9TUk5fjjUjg9qIMH3sdnh' % (room_id, tt)
        sign = hashlib.md5(signContent.encode('ascii')).hexdigest()
        video_url = self._download_json(
            'http://coapi.douyucdn.cn/lapi/live/thirdPart/getPlay/' + room_id,
            video_id, note='Downloading video URL info',
            query={'rate': 0}, headers={
                'auth': sign,
                'time': str(tt),
                'aid': 'pcclient'
            })['data']['live_url']

        title = self._live_title(unescapeHTML(room['room_name']))
        description = room.get('show_details')
        thumbnail = room.get('room_src')
        uploader = room.get('nickname')

        return {
            'id': room_id,
            'display_id': video_id,
            'url': video_url,
            'title': title,
            'description': description,
            'thumbnail': thumbnail,
            'uploader': uploader,
            'is_live': True,
        }


class DouyuShowIE(InfoExtractor):
    _VALID_URL = r'https?://v(?:mobile)?\.douyu\.com/show/(?P<id>[0-9a-zA-Z]+)'

    _TESTS = [{
        'url': 'https://v.douyu.com/show/rjNBdvnVXNzvE2yw',
        'md5': '0c2cfd068ee2afe657801269b2d86214',
        'info_dict': {
            'id': 'rjNBdvnVXNzvE2yw',
            'ext': 'mp4',
            'title': '陈一发儿：砒霜 我有个室友系列！04-01 22点场',
            'duration': 7150.08,
            'thumbnail': r're:^https?://.*\.jpg$',
            'uploader': '陈一发儿',
            'uploader_id': 'XrZwYelr5wbK',
            'uploader_url': 'https://v.douyu.com/author/XrZwYelr5wbK',
            'upload_date': '20170402',
        },
    }, {
        'url': 'https://vmobile.douyu.com/show/rjNBdvnVXNzvE2yw',
        'only_matching': True,
    }]

    def getPostData(self, vid, did, jsScript):
        tt = '%s' % int(time.time())
        html = '''
        <html><head><meta name="robots" content="noindex"></head><body>
        <script src='https://cdnjs.cloudflare.com/ajax/libs/crypto-js/4.0.0/crypto-js.min.js'></script>
        <script>
        var n7e0bf9fa2da3668=[0xd9f139f6,0x3573600b,0x1c5b84e1,0x4bc3881b,0x267e0dd7,0x799784e6,0x5d3579e6,0xd92b781d,0xe6d9ae94,0x121754f5,0x6e0f975a,0x3b232d34,0x27648d0e,0xa2d97171,0x41d07916,0xd15431bb,0x302b3ecf,0xd71d4e7,0x9c553e63,0x22506b52,0xbf5c0bad,0x4ae03016,0xbf00e6a3,0x9547993a,0x9d1ea56e,0xd9e885c2,0xa715a9b4,0xa186a720,0xd7e5793,0xc47ac51c,0x69f152e8,0xd1688be5,0xfb873dca,0x268ebdb9,0xdf8b362a,0xbe3ebc83,0xe29d7a6b,0x2e4637ba,0xd57a23da,0x933f424e,0x59c12d4,0x9fc8ba55,0xf81c8733,0xc4854f9,0x5350255b,0x9965b11b,0xc9e393be,0x2b893c21,0x81941fee,0x1e0077d4,0xf665b31b,0xa7d9ce74,0x4e51a987,0xadc5792b,0x3ab6be39,0x1cb91b1c,0x9b32f4e6,0x134739e2,0xf8862a81,0x692dd85c,0xbc9cde6e,0x31e8b331,0x38c7290f,0x1ecc331b,0x94d141b8,0xf8d64954,0x3a2a24e9,0x6256895d,0x7ac044ca,0x75acad67,0x95c35b6,0xd2f3bf06,0x7121e29a,0xabbdb821,0xd5030f40,0xd3968e5c,0x1d16ff8b,0x178560e1,0xe9408a4c,0x7b1c62b7,0x3fd040ca,0x791bf445,0x52bca6ff,0x7c316589,0x69dfe455,0x542d2b59,0x5fe4af09,0x822307c2,0xe01b7334,0x65d1dee7,0xb214b1c3,0x6864b2ec,0xeacf7f21,0x833168f8,0x552b7939,0x93fe10e8,0x2d78b2be,0x52069f3d,0x3a81b3fa,0x76174957,0x18223777,0xa6b5f7ac,0x620603e6,0x104b983b,0x2cb9310c,0x8fd6f44b,0xfa07ec20,0x769ebee0,0xdc98d9be,0x18d10a4c,0x6ffa5ebc,0x24de4e11,0x9ad42a47,0x2b6db921,0x102f7f61,0x4310dbc1,0x6bb2e628,0xb352215,0xcc8d0bfc,0x1493a271,0x5a50f7cf,0xaac1f8d7,0x28c829c2,0x2e36d613,0x51ad32ad,0x8adad565,0x46d1e4e7,0xce40eb17,0xa350b655,0xf63ea064,0x491de5df,0xff4e5901,0x7f7034b4,0xc297f2a5,0x4c69347e,0x87a7771f,0x1b970034,0xd4753665,0xc153bbc,0xda6e2317,0x45238744,0xe002118d,0x6b3b1c9f,0x9c10864e,0xf525e2df,0x532a1f00,0xc3e4649d,0x3a0276e8,0xc95a7b1c,0x8f5f65f3,0x5419bbd0,0x1e64b651,0xd66c388b,0x553d737d,0xd4ebdd01,0x5f12cbe5,0x9c925e56,0xc696647a,0x8d192a6e,0x299d4e94,0x503ce247,0xd43e703d,0x21d52cc0,0x912300a0,0xd735f392,0x7f62f343,0x5b274137,0x813b9558,0xfdff45d1,0x50fff3c1,0xe2f6238d,0xd145883a,0x8d8baf43,0x19d825d,0x602dede3,0xbd6b2010,0x8405d014,0xf4623b4e,0xa9367834,0x179ec903,0xbc936c36,0x78580ecc,0xe6f5940c,0x9905fe8a,0xc5bffb08,0xe58e63e0,0x2a9736b0,0xe29cb92d,0xa5b4990d,0x13fcbfe4,0x2b45f05d,0x7a09cb0e,0xab547f8c,0x6ff2514c,0xe9a88d81,0xc2844831,0x5d74132d,0xb20c2085,0x782368ac,0x9c31340c,0x5753389,0x1ae3b447,0xb1af536f,0x6547ea0c,0x236fced1,0x5acd9ca,0x7e94f4ab,0xe4d6d0d3,0xdf8898cc,0xb846729c,0x223703a6,0x441ddc2b,0x1006f4fc,0x482454f0,0x915a1081,0xf08a6e84,0x555c66a0,0x3cf5b3fb,0xd595171a,0x980be423,0xe8f2326e,0xf4d80981,0x84d8412,0x10dbedec,0xd00c7d49,0x9adf1777,0x9a3ef07b,0x8d2aba34,0xd7365f56,0xe16744ca,0x9aae6878,0xf71da828,0x4989427,0x3e2a4a57,0xf22b0edb,0x665da63e,0xe0ee726d,0x5a7f648e,0xfad26324,0x34acee5,0x3d3c509d,0xa3582317,0x17a3168c,0x270a72b4,0xc1dde127,0xc8ae4cc9,0xfaa0fec5,0x5528a294,0x27ae6c82,0xfde1fe6f,0x1886e52d,0x34862dfb,0x7e972325,0x8aa9b609,0xada27028,0x191ce100,0xa001c6fd,0x46e395c,0x8e281c42,0xe4a710de,0x8e04cf06,0x4064ebc9,0x8bdeb7e7,0xb5541dc5,0x8ab79530,0x755b86a6,0x3e35e73b,0xacb9e95,0x24557433,0xffb27569,0x5c1a69b,0x6942aaad,0x48b050e1,0x747c731c,0x2ae87f60,0xe0f33309,0x4926e1cf,0x6e455cfc,0x7edaed8f,0x602eaada,0x9a8736cd,0xf981a0f9,0x5cf4257e,0x302319,0x24d33351,0xfc959a65,0xeae09c,0xc2698a1e,0xba4a7e85,0x91042e59,0x1c22efca,0xa2bd813f,0x93083f77,0x23cdea0b,0xbb48f24d,0xbcd8f90d,0xbf44c254,0xf3d36416,0x831ad864,0x10be2c33,0x42de62df,0x12ca4a,0x798aaaca,0x256d151b,0x7d1d3f3d,0xc9f86e0f,0x3f0d5d69,0x11f3f84e,0xd2451ac7,0x578088c3,0x56ec78e5,0xa61ad97e,0x34afade9,0xafaa8df7,0xbae7bc00,0x618bdecd,0x9ab26746,0xe24daf8f,0xcea148c0,0x998b6cf0,0x3d2358f8,0x816bc383,0xd94052b1,0xc07c9228,0x9d14d8d6,0xa9221bd,0x2e55d1d1,0x98517e2a,0xa9db5f5,0x5a27c202,0xcc043b55,0x6749a534,0xe1db4486,0x656502e5,0x41a2fb27,0x6f4df511,0x21a962f9,0xe8023582,0xa0dfdf4c,0xa84887be,0x6a7188ec,0x7752a995,0xcb07ea7d,0x2d962481,0xa9a4ba45,0x8013f7f1,0x2a0dd807,0xa00cf18d,0x31a2ca33,0x520d7292,0x5d3032a4,0x7554be97,0xce5bf0d7,0x1a8c4bd8,0xee7eb6dd,0x9c3b3588,0x698195d0,0x5afa081,0xc04ed116,0xee36b84,0x8e76055f,0xf0ee7504,0x97cd7135,0xb6e43b76,0xc0b902a2,0x52f6917e,0x2d38575d,0xf9c7c776,0xe54f991d,0xeb644beb,0x98e5b0ad,0x3517b3d6,0x7e020cf9,0x4941ad6f,0xa88a1ba0,0x6e1f9840,0x2739ebb9,0xb554e5f1,0x4e6d9117,0x4a6d6aac,0x7f662bde,0x5fe1ef54,0xaa139f88,0x6f5431b9,0x131d9d49,0x9f8147e8,0x78605ac5,0x32fa18e7];
        function ub98484234(n7e0bf9fa2da36680,n7e0bf9fa2da36681,n7e0bf9fa2da36682){var rk=[13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26,19,31,24,10,13,22,16,41,26];var k2=[0x76bb510f,0x367a57d7];var lk=[0x76bb510f,0x367a57d7];var v = n7e0bf9fa2da3668.slice(0);var k=[0xbd7375f8,0xdfcfb71b,0x3f7fed8e,0xf7bb9631];for(var O=0;O<388;O++){v[O]^=0x118867f9;}v[1]^=lk[1];v[0]+=lk[0];v[3]-=lk[1];v[2]-=lk[0];v[5]-=lk[1];v[4]+=lk[0];v[7]+=lk[1];v[6]+=lk[0];v[9]-=lk[1];v[8]-=lk[0];v[11]=(v[11]<<(lk[1]%16))|(v[11]>>>(32-(lk[1]%16)));v[10]+=lk[0];v[13]^=lk[1];v[12]=(v[12]>>>(lk[0]%16))|(v[12]<<(32-(lk[0]%16)));v[15]=(v[15]<<(lk[1]%16))|(v[15]>>>(32-(lk[1]%16)));v[14]=(v[14]<<(lk[0]%16))|(v[14]>>>(32-(lk[0]%16)));v[17]-=lk[1];v[16]-=lk[0];v[19]^=lk[1];v[18]+=lk[0];v[21]-=lk[1];v[20]-=lk[0];v[23]-=lk[1];v[22]+=lk[0];v[25]+=lk[1];v[24]+=lk[0];v[27]-=lk[1];v[26]-=lk[0];v[29]=(v[29]<<(lk[1]%16))|(v[29]>>>(32-(lk[1]%16)));v[28]+=lk[0];v[31]^=lk[1];v[30]=(v[30]>>>(lk[0]%16))|(v[30]<<(32-(lk[0]%16)));v[33]=(v[33]<<(lk[1]%16))|(v[33]>>>(32-(lk[1]%16)));v[32]=(v[32]<<(lk[0]%16))|(v[32]>>>(32-(lk[0]%16)));v[35]-=lk[1];v[34]-=lk[0];v[37]^=lk[1];v[36]+=lk[0];v[39]-=lk[1];v[38]-=lk[0];v[41]-=lk[1];v[40]+=lk[0];v[43]+=lk[1];v[42]+=lk[0];v[45]-=lk[1];v[44]-=lk[0];v[47]=(v[47]<<(lk[1]%16))|(v[47]>>>(32-(lk[1]%16)));v[46]+=lk[0];v[49]^=lk[1];v[48]=(v[48]>>>(lk[0]%16))|(v[48]<<(32-(lk[0]%16)));v[51]=(v[51]<<(lk[1]%16))|(v[51]>>>(32-(lk[1]%16)));v[50]=(v[50]<<(lk[0]%16))|(v[50]>>>(32-(lk[0]%16)));v[53]-=lk[1];v[52]-=lk[0];v[55]^=lk[1];v[54]+=lk[0];v[57]-=lk[1];v[56]-=lk[0];v[59]-=lk[1];v[58]+=lk[0];v[61]+=lk[1];v[60]+=lk[0];v[63]-=lk[1];v[62]-=lk[0];v[65]=(v[65]<<(lk[1]%16))|(v[65]>>>(32-(lk[1]%16)));v[64]+=lk[0];v[67]^=lk[1];v[66]=(v[66]>>>(lk[0]%16))|(v[66]<<(32-(lk[0]%16)));v[69]=(v[69]<<(lk[1]%16))|(v[69]>>>(32-(lk[1]%16)));v[68]=(v[68]<<(lk[0]%16))|(v[68]>>>(32-(lk[0]%16)));v[71]-=lk[1];v[70]-=lk[0];v[73]^=lk[1];v[72]+=lk[0];v[75]-=lk[1];v[74]-=lk[0];v[77]-=lk[1];v[76]+=lk[0];v[79]+=lk[1];v[78]+=lk[0];v[81]-=lk[1];v[80]-=lk[0];v[83]=(v[83]<<(lk[1]%16))|(v[83]>>>(32-(lk[1]%16)));v[82]+=lk[0];v[85]^=lk[1];v[84]=(v[84]>>>(lk[0]%16))|(v[84]<<(32-(lk[0]%16)));v[87]=(v[87]<<(lk[1]%16))|(v[87]>>>(32-(lk[1]%16)));v[86]=(v[86]<<(lk[0]%16))|(v[86]>>>(32-(lk[0]%16)));v[89]-=lk[1];v[88]-=lk[0];v[91]^=lk[1];v[90]+=lk[0];v[93]-=lk[1];v[92]-=lk[0];v[95]-=lk[1];v[94]+=lk[0];v[97]+=lk[1];v[96]+=lk[0];v[99]-=lk[1];v[98]-=lk[0];v[101]=(v[101]<<(lk[1]%16))|(v[101]>>>(32-(lk[1]%16)));v[100]+=lk[0];v[103]^=lk[1];v[102]=(v[102]>>>(lk[0]%16))|(v[102]<<(32-(lk[0]%16)));v[105]=(v[105]<<(lk[1]%16))|(v[105]>>>(32-(lk[1]%16)));v[104]=(v[104]<<(lk[0]%16))|(v[104]>>>(32-(lk[0]%16)));v[107]-=lk[1];v[106]-=lk[0];v[109]^=lk[1];v[108]+=lk[0];v[111]-=lk[1];v[110]-=lk[0];v[113]-=lk[1];v[112]+=lk[0];v[115]+=lk[1];v[114]+=lk[0];v[117]-=lk[1];v[116]-=lk[0];v[119]=(v[119]<<(lk[1]%16))|(v[119]>>>(32-(lk[1]%16)));v[118]+=lk[0];v[121]^=lk[1];v[120]=(v[120]>>>(lk[0]%16))|(v[120]<<(32-(lk[0]%16)));v[123]=(v[123]<<(lk[1]%16))|(v[123]>>>(32-(lk[1]%16)));v[122]=(v[122]<<(lk[0]%16))|(v[122]>>>(32-(lk[0]%16)));v[125]-=lk[1];v[124]-=lk[0];v[127]^=lk[1];v[126]+=lk[0];v[129]-=lk[1];v[128]-=lk[0];v[131]-=lk[1];v[130]+=lk[0];v[133]+=lk[1];v[132]+=lk[0];v[135]-=lk[1];v[134]-=lk[0];v[137]=(v[137]<<(lk[1]%16))|(v[137]>>>(32-(lk[1]%16)));v[136]+=lk[0];v[139]^=lk[1];v[138]=(v[138]>>>(lk[0]%16))|(v[138]<<(32-(lk[0]%16)));v[141]=(v[141]<<(lk[1]%16))|(v[141]>>>(32-(lk[1]%16)));v[140]=(v[140]<<(lk[0]%16))|(v[140]>>>(32-(lk[0]%16)));v[143]-=lk[1];v[142]-=lk[0];v[145]^=lk[1];v[144]+=lk[0];v[147]-=lk[1];v[146]-=lk[0];v[149]-=lk[1];v[148]+=lk[0];v[151]+=lk[1];v[150]+=lk[0];v[153]-=lk[1];v[152]-=lk[0];v[155]=(v[155]<<(lk[1]%16))|(v[155]>>>(32-(lk[1]%16)));v[154]+=lk[0];v[157]^=lk[1];v[156]=(v[156]>>>(lk[0]%16))|(v[156]<<(32-(lk[0]%16)));v[159]=(v[159]<<(lk[1]%16))|(v[159]>>>(32-(lk[1]%16)));v[158]=(v[158]<<(lk[0]%16))|(v[158]>>>(32-(lk[0]%16)));v[161]-=lk[1];v[160]-=lk[0];v[163]^=lk[1];v[162]+=lk[0];v[165]-=lk[1];v[164]-=lk[0];v[167]-=lk[1];v[166]+=lk[0];v[169]+=lk[1];v[168]+=lk[0];v[171]-=lk[1];v[170]-=lk[0];v[173]=(v[173]<<(lk[1]%16))|(v[173]>>>(32-(lk[1]%16)));v[172]+=lk[0];v[175]^=lk[1];v[174]=(v[174]>>>(lk[0]%16))|(v[174]<<(32-(lk[0]%16)));v[177]=(v[177]<<(lk[1]%16))|(v[177]>>>(32-(lk[1]%16)));v[176]=(v[176]<<(lk[0]%16))|(v[176]>>>(32-(lk[0]%16)));v[179]-=lk[1];v[178]-=lk[0];v[181]^=lk[1];v[180]+=lk[0];v[183]-=lk[1];v[182]-=lk[0];v[185]-=lk[1];v[184]+=lk[0];v[187]+=lk[1];v[186]+=lk[0];v[189]-=lk[1];v[188]-=lk[0];v[191]=(v[191]<<(lk[1]%16))|(v[191]>>>(32-(lk[1]%16)));v[190]+=lk[0];v[193]^=lk[1];v[192]=(v[192]>>>(lk[0]%16))|(v[192]<<(32-(lk[0]%16)));v[195]=(v[195]<<(lk[1]%16))|(v[195]>>>(32-(lk[1]%16)));v[194]=(v[194]<<(lk[0]%16))|(v[194]>>>(32-(lk[0]%16)));v[197]-=lk[1];v[196]-=lk[0];v[199]^=lk[1];v[198]+=lk[0];v[201]-=lk[1];v[200]-=lk[0];v[203]-=lk[1];v[202]+=lk[0];v[205]+=lk[1];v[204]+=lk[0];v[207]-=lk[1];v[206]-=lk[0];v[209]=(v[209]<<(lk[1]%16))|(v[209]>>>(32-(lk[1]%16)));v[208]+=lk[0];v[211]^=lk[1];v[210]=(v[210]>>>(lk[0]%16))|(v[210]<<(32-(lk[0]%16)));v[213]=(v[213]<<(lk[1]%16))|(v[213]>>>(32-(lk[1]%16)));v[212]=(v[212]<<(lk[0]%16))|(v[212]>>>(32-(lk[0]%16)));v[215]-=lk[1];v[214]-=lk[0];v[217]^=lk[1];v[216]+=lk[0];v[219]-=lk[1];v[218]-=lk[0];v[221]-=lk[1];v[220]+=lk[0];v[223]+=lk[1];v[222]+=lk[0];v[225]-=lk[1];v[224]-=lk[0];v[227]=(v[227]<<(lk[1]%16))|(v[227]>>>(32-(lk[1]%16)));v[226]+=lk[0];v[229]^=lk[1];v[228]=(v[228]>>>(lk[0]%16))|(v[228]<<(32-(lk[0]%16)));v[231]=(v[231]<<(lk[1]%16))|(v[231]>>>(32-(lk[1]%16)));v[230]=(v[230]<<(lk[0]%16))|(v[230]>>>(32-(lk[0]%16)));v[233]-=lk[1];v[232]-=lk[0];v[235]^=lk[1];v[234]+=lk[0];v[237]-=lk[1];v[236]-=lk[0];v[239]-=lk[1];v[238]+=lk[0];v[241]+=lk[1];v[240]+=lk[0];v[243]-=lk[1];v[242]-=lk[0];v[245]=(v[245]<<(lk[1]%16))|(v[245]>>>(32-(lk[1]%16)));v[244]+=lk[0];v[247]^=lk[1];v[246]=(v[246]>>>(lk[0]%16))|(v[246]<<(32-(lk[0]%16)));v[249]=(v[249]<<(lk[1]%16))|(v[249]>>>(32-(lk[1]%16)));v[248]=(v[248]<<(lk[0]%16))|(v[248]>>>(32-(lk[0]%16)));v[251]-=lk[1];v[250]-=lk[0];v[253]^=lk[1];v[252]+=lk[0];v[255]-=lk[1];v[254]-=lk[0];v[257]-=lk[1];v[256]+=lk[0];v[259]+=lk[1];v[258]+=lk[0];v[261]-=lk[1];v[260]-=lk[0];v[263]=(v[263]<<(lk[1]%16))|(v[263]>>>(32-(lk[1]%16)));v[262]+=lk[0];v[265]^=lk[1];v[264]=(v[264]>>>(lk[0]%16))|(v[264]<<(32-(lk[0]%16)));v[267]=(v[267]<<(lk[1]%16))|(v[267]>>>(32-(lk[1]%16)));v[266]=(v[266]<<(lk[0]%16))|(v[266]>>>(32-(lk[0]%16)));v[269]-=lk[1];v[268]-=lk[0];v[271]^=lk[1];v[270]+=lk[0];v[273]-=lk[1];v[272]-=lk[0];v[275]-=lk[1];v[274]+=lk[0];v[277]+=lk[1];v[276]+=lk[0];v[279]-=lk[1];v[278]-=lk[0];v[281]=(v[281]<<(lk[1]%16))|(v[281]>>>(32-(lk[1]%16)));v[280]+=lk[0];v[283]^=lk[1];v[282]=(v[282]>>>(lk[0]%16))|(v[282]<<(32-(lk[0]%16)));v[285]=(v[285]<<(lk[1]%16))|(v[285]>>>(32-(lk[1]%16)));v[284]=(v[284]<<(lk[0]%16))|(v[284]>>>(32-(lk[0]%16)));v[287]-=lk[1];v[286]-=lk[0];v[289]^=lk[1];v[288]+=lk[0];v[291]-=lk[1];v[290]-=lk[0];v[293]-=lk[1];v[292]+=lk[0];v[295]+=lk[1];v[294]+=lk[0];v[297]-=lk[1];v[296]-=lk[0];v[299]=(v[299]<<(lk[1]%16))|(v[299]>>>(32-(lk[1]%16)));v[298]+=lk[0];v[301]^=lk[1];v[300]=(v[300]>>>(lk[0]%16))|(v[300]<<(32-(lk[0]%16)));v[303]=(v[303]<<(lk[1]%16))|(v[303]>>>(32-(lk[1]%16)));v[302]=(v[302]<<(lk[0]%16))|(v[302]>>>(32-(lk[0]%16)));v[305]-=lk[1];v[304]-=lk[0];v[307]^=lk[1];v[306]+=lk[0];v[309]-=lk[1];v[308]-=lk[0];v[311]-=lk[1];v[310]+=lk[0];v[313]+=lk[1];v[312]+=lk[0];v[315]-=lk[1];v[314]-=lk[0];v[317]=(v[317]<<(lk[1]%16))|(v[317]>>>(32-(lk[1]%16)));v[316]+=lk[0];v[319]^=lk[1];v[318]=(v[318]>>>(lk[0]%16))|(v[318]<<(32-(lk[0]%16)));v[321]=(v[321]<<(lk[1]%16))|(v[321]>>>(32-(lk[1]%16)));v[320]=(v[320]<<(lk[0]%16))|(v[320]>>>(32-(lk[0]%16)));v[323]-=lk[1];v[322]-=lk[0];v[325]^=lk[1];v[324]+=lk[0];v[327]-=lk[1];v[326]-=lk[0];v[329]-=lk[1];v[328]+=lk[0];v[331]+=lk[1];v[330]+=lk[0];v[333]-=lk[1];v[332]-=lk[0];v[335]=(v[335]<<(lk[1]%16))|(v[335]>>>(32-(lk[1]%16)));v[334]+=lk[0];v[337]^=lk[1];v[336]=(v[336]>>>(lk[0]%16))|(v[336]<<(32-(lk[0]%16)));v[339]=(v[339]<<(lk[1]%16))|(v[339]>>>(32-(lk[1]%16)));v[338]=(v[338]<<(lk[0]%16))|(v[338]>>>(32-(lk[0]%16)));v[341]-=lk[1];v[340]-=lk[0];v[343]^=lk[1];v[342]+=lk[0];v[345]-=lk[1];v[344]-=lk[0];v[347]-=lk[1];v[346]+=lk[0];v[349]+=lk[1];v[348]+=lk[0];v[351]-=lk[1];v[350]-=lk[0];v[353]=(v[353]<<(lk[1]%16))|(v[353]>>>(32-(lk[1]%16)));v[352]+=lk[0];v[355]^=lk[1];v[354]=(v[354]>>>(lk[0]%16))|(v[354]<<(32-(lk[0]%16)));v[357]=(v[357]<<(lk[1]%16))|(v[357]>>>(32-(lk[1]%16)));v[356]=(v[356]<<(lk[0]%16))|(v[356]>>>(32-(lk[0]%16)));v[359]-=lk[1];v[358]-=lk[0];v[361]^=lk[1];v[360]+=lk[0];v[363]-=lk[1];v[362]-=lk[0];v[365]-=lk[1];v[364]+=lk[0];v[367]+=lk[1];v[366]+=lk[0];v[369]-=lk[1];v[368]-=lk[0];v[371]=(v[371]<<(lk[1]%16))|(v[371]>>>(32-(lk[1]%16)));v[370]+=lk[0];v[373]^=lk[1];v[372]=(v[372]>>>(lk[0]%16))|(v[372]<<(32-(lk[0]%16)));v[375]=(v[375]<<(lk[1]%16))|(v[375]>>>(32-(lk[1]%16)));v[374]=(v[374]<<(lk[0]%16))|(v[374]>>>(32-(lk[0]%16)));v[377]-=lk[1];v[376]-=lk[0];v[379]^=lk[1];v[378]+=lk[0];v[381]-=lk[1];v[380]-=lk[0];v[383]-=lk[1];v[382]+=lk[0];v[385]+=lk[1];v[384]+=lk[0];v[387]-=lk[1];v[386]-=lk[0];for(var I=0;I<388;I+=2){var i,v0=v[I]^k2[0],v1=v[I+1]^k2[1],d=0x9E3779B9,sum=d*rk[I/2];for(i=0;i<rk[I/2];i++){v1-=(((v0<<4)^(v0>>>5))+v0)^(sum+k[(sum>>>11)&3]);sum-=d;v0-=(((v1<<4)^(v1>>>5))+v1)^(sum+k[sum&3]);}v[I]=v0^k2[1];v[I+1]=v1^k2[0];}for(var O=387;O>0;O--){v[O]^=v[O-1];}v[0]^=0x118867f9;var strc="";for(var i=0;i<v.length;i++) {strc+=String.fromCharCode(v[i]&0xff,v[i]>>>8&0xff,v[i]>>>16&0xff,v[i]>>>24&0xff);}return eval(strc)(n7e0bf9fa2da36680,n7e0bf9fa2da36681,n7e0bf9fa2da36682);}             
        function output(n){
            if(window.external){
                window.external.output('<MyManifest>' + n + '</MyManifest>');
            }else{
                window.WebViewJavascriptBridge.callHandler('JSSendToOC','<MyManifest>' + n + '</MyManifest>');
            }
        } 
        output(ub98484234("''' + vid + '''", "''' + did + '''", '''+ tt + '''))            
        </script></body></html>'''     
        if (jsScript):
            html = '''
            <html><head><meta name="robots" content="noindex"></head><body>
            <script src='https://cdnjs.cloudflare.com/ajax/libs/crypto-js/4.0.0/crypto-js.min.js'></script>
            <script>
            ''' + jsScript + '''
            function output(n){
                if(window.external){
                    window.external.output('<MyManifest>' + n + '</MyManifest>');
                }else{
                    window.WebViewJavascriptBridge.callHandler('JSSendToOC','<MyManifest>' + n + '</MyManifest>');
                }
            } 
            output(ub98484234("''' + vid + '''", "''' + did + '''", '''+ tt + '''))          
            </script></body></html>'''  
        # html = template % ('18042548', 'AND-CHR|87-357128951609410381805')
        webpage = downloadWebPage_BYHeadlessBrowser_localFile(html)
        return self._search_regex(r'<MyManifest>([\s\S]+)<\/MyManifest>', webpage, 'json') 

    def _real_extract(self, url):
        url = url.replace('vmobile.', 'v.')
        video_id = self._match_id(url)
        
        webpage = self._download_webpage(url, video_id)
        jsScript = self._search_regex(r'(var vdwdae325w_64we\s*\=\s*.+}\s*;)</script><script>', webpage, 'jsScript')
        point_id = self._search_regex(r'"point_id"\:(\d+)', webpage, 'jsScript')
        
        room_info = self._parse_json(self._search_regex(
            r'ROOM\:\s*({.+}),DATA', webpage, 'room info'), video_id)

        video_info = None
        acf_did = self._get_cookies(url)['acf_did'].value
        postdata = self.getPostData(point_id, acf_did, jsScript=jsScript) + '&vid=' + video_id
        for trial in range(5):
            # Sometimes Douyu rejects our request. Let's try it more times
            try:
                video_info = self._download_json(
                    'https://vmobile.douyu.com/video/getInfo', video_id,
                    query={'vid': video_id},
                    headers={
                        'Referer': url,
                        'x-requested-with': 'XMLHttpRequest',
                    }, data=postdata.encode())
                break
            except ExtractorError:
                self._sleep(1, video_id)

        if not video_info:
            raise ExtractorError('Can\'t fetch video info')

        formats = self._extract_m3u8_formats(
            video_info['data']['video_url'], video_id,
            entry_protocol='m3u8_native', ext='mp4')

        upload_date = unified_strdate(self._html_search_regex(
            r'<em>上传时间：</em><span>([^<]+)</span>', webpage,
            'upload date', fatal=False))

        uploader = uploader_id = uploader_url = None
        mobj = re.search(
            r'(?m)<a[^>]+href="/author/([0-9a-zA-Z]+)".+?<strong[^>]+title="([^"]+)"',
            webpage)
        if mobj:
            uploader_id, uploader = mobj.groups()
            uploader_url = urljoin(url, '/author/' + uploader_id)

        return {
            'id': video_id,
            'title': room_info['name'],
            'formats': formats,
            'duration': room_info.get('duration'),
            'thumbnail': room_info.get('pic'),
            'upload_date': upload_date,
            'uploader': uploader,
            'uploader_id': uploader_id,
            'uploader_url': uploader_url,
        }
