'/*
' * Copyright (c) 2013 Martin Mitas
' *
' * This library is free software; you can redistribute it and/or modify it
' * under the terms of the GNU Lesser General Public License as published by
' * the Free Software Foundation; either version 2.1 of the License, or
' * (at your option) any later version.
' *
' * This library is distributed in the hope that it will be useful,
' * but WITHOUT ANY WARRANTY; without even the implied warranty of
' * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
' * GNU Lesser General Public License for more details.
' *
' * You should have received a copy of the GNU Lesser General Public License
' * along with this library; if not, write to the Free Software Foundation,
' * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
' */



#include Once "_common.bi"

'/**
' * @file
' * @brief Chart control (@c MC_WC_CHART).
' *
' * The chart control is intended to display large sets of numerical data in a
' * visually illustrative way.
' *
' * @attention The chart control requires @c GDIPLUS.DLL version 1.0 or newer to
' * work correctly. See @ref sec_dll_deps for more info.
' *
' * In general, the control is able to hold a multiple series of data (data
' * sets). Each series typically denotes some quantity evolving in time or its
' * dependency on other quantity. In the chart each series is represented with a
' * different color, and accompanied with a legend text.
' *
' * The chart control supports charts of many types, depending on the used
' * control style. The type determines how the control presents the data.
' *
' *
' * @section chart_dataset Data Sets
' *
' * To inset, delete or reset the data set values, use messages
' * @ref MC_CHM_INSERTDATASET, @ref MC_CHM_DELETEDATASET or
' * @ref MC_CHM_DELETEALLDATASETS and @ref MC_CHM_SETDATASET respectively.
' *
' * Data sets can have the data only virtually. That is application can provide
' * to the control only an information the data set exists and how large it is.
' * Whenever the control paints and needs really the data, it asks for them
' * with @ref MC_CHN_GETDISPINFO message. This is useful especially if the data
' * for the chart are really large as it avoids duplication of the data in
' * program memory.
' *
' * To set various other attributes of the data set, you may use
' * @ref MC_CHM_SETDATASETLEGEND to set its legend or @ref MC_CHM_SETDATASETCOLOR
' * to set the color representing the values form the data set.
' *
' *
' * @section chart_axes Axes
' *
' * The control has a concept of two axes, the primary and secondary one.
' * It depends on particular chart type though, whether and how the control uses
' * them.
' *
' * Usually (for chart types where it gives any sense), the primary axis
' * corresponds to the index of data set and in most cases is displayed as the
' * horizontal (X) axis, and the secondary one corresponds to values in a data
' * set and in most cases is displayed as the vertical (Y) axis. An important
' * exception to this rule of thumb are bar (@ref MC_CHS_BAR) and stacked bar
' * (@ref MC_CHS_STACKEDBAR) charts which are similar to the (stacked) column
' * chart, but with horizontal and vertical axes swapped.
' *
' * Each axis has some attributes associated with it that determine the value
' * tags along the axis.
' *
' * The factor exponent is an integer value in the range (@c -9 to @c +9),
' * and it it is used when painting values for the given axis. An integer value
' * is multiplied with <tt>(10 ^ exp)</tt>, where @c ^ marks power and
' * @c exp the exponent.
' *
' * This is especially useful for charts with non-integer values as negative
' * factor exponent actually turns the data set values into fixed point numbers.
' *
' * For example with factor exponent @c -2, data set with values <tt>{ 5, 100,
' * 101, 102 }</tt> would be displayed as <tt>{ 0.05, 1.00, 1.01, 1.02 }.</tt>
' *
' * To set or get the factor exponent, use messages @ref MC_CHM_SETFACTOREXPONENT
' * and @ref MC_CHM_GETFACTOREXPONENT respectively.
' *
' * Other import axis attribute is a value offset. The offset specifies a number
' * which is added to an index. This way, application can set the offset of the
' * axis for example to @c 2003, and hence, the value tags along the axis shall
' * be 2003, 2004, 2005, etc. This can be used, for example, in charts which
' * show some time development of some values across, e.g. several years.
' *
' * To set or get the axis offset, use messages @ref MC_CHM_SETAXISOFFSET and
' * @ref MC_CHM_GETAXISOFFSET respectively.
' *
' *
' * @section chart_types Chart Types
' *
' * @attention Different types of chart have different requirements about the
' * data and if the application does not follow these requirements, the chart
' * can just display gibberish, or change the output in future mCtrl versions.
' *
' * The pie chart (@ref MC_CHS_PIE) expects each data set consisting of a single
' * non-negative value. It does not display any axis, but factor exponent of
' * the primary one is used for displaying data set values.
' *
' * @note The pie chart visually differs quite a lot from majority of the other
' * chart types this control provides. Many control attributes settable by
' * control messages actually have no impact to the pie chart.
' *
' * The scatter chart (@ref MC_CHS_SCATTER) expects to have all data set sizes
' * of even count of values. Unlike the other chart types, the sizes of the data
' * sets can differ. The scatter chart interprets each data set as a set of
' * value pairs. Each value with odd index corresponds with the primary (X) axis,
' * and even index with secondary (Y) axis respectively.
' *
' * The line chart (@ref MC_CHS_LINE), the area chart (@ref MC_CHS_AREA), the
' * column chart (@ref MC_CHS_COLUMN) and the bar chart (@ref MC_CHS_BAR) only
' * expect that all the data sets are of the same size.
' *
' * Stacked variants of the charts above (@ref MC_CHS_STACKEDLINE, @ref
' * MC_CHS_STACKEDAREA, @ref MC_CHS_STACKEDCOLUMN and @ref MC_CHS_STACKEDBAR)
' * additionally expect all the data in all data sets are positive. Actually
' * the charts work for negative values, but the result can be confusing for
' * the user.
' *
' *
' * @section chart_std_msgs Standard Messages
' *
' * These standard messages are handled by the control:
' * - @c WM_GETTEXT
' * - @c WM_SETTEXT
' * - @c CCM_SETNOTIFYWINDOW
' *
' * These standard notifications are sent by the control:
' * - @c NM_OUTOFMEMORY
' * - @c NM_TOOLTIPSCREATED
' */
'
'
'/**
' * @name Initialization Functions
' */
'/*@{*/
'
'/**
' * Registers window class of the control.
' *
' * Note that the function fails, if @c GDIPLUS.DLL is not available.
' *
' * @return @c TRUE on success, @c FALSE on failure.
' */

Extern "C"     
    Declare Function mcChart_Initialize MCTRL_API  Lib "mCtrl" Alias "mcChart_Initialize"() As  BOOL
    Declare Sub mcChart_Terminate MCTRL_API  Lib "mCtrl" Alias "mcChart_Terminate"()
'/**
' * Unregisters window class of the control.
' */
End Extern
mcChart_Initialize

'/*@}*/
'
'
'/**
' * @name Window Class
' */
'/*@{*/
'
'/** Window class name (Unicode variant). */
'#define MC_WC_CHARTW           L"mCtrl.chart"
'/** Window class name (ANSI variant). */
'#define MC_WC_CHARTA            "mCtrl.chart"
'
'/*@}*/
'
'
'/**
' * @name Control Styles
' */
'/*@{*/

'/** @brief Pie chart. */
#define MC_CHS_PIE             &H0000
'/** @brief Scatter chart. */
#define MC_CHS_SCATTER         &H0001
'/** @brief Line chart. */
#define MC_CHS_LINE            &H0002
'/** @brief Stacked line chart. */
#define MC_CHS_STACKEDLINE     &H0003
'/** @brief Area chart. */
#define MC_CHS_AREA            &H0004
'/** @brief Stacked area chart. */
#define MC_CHS_STACKEDAREA     &H0005
'/** @brief Column chart. */
#define MC_CHS_COLUMN          &H0006
'/** @brief Stacked column chart. */
#define MC_CHS_STACKEDCOLUMN   &H0007
'/** @brief Bar chart. */
#define MC_CHS_BAR             &H0008
'/** @brief Stacked bar chart. */
#define MC_CHS_STACKEDBAR      &H0009
'/** @brief This is not actually a valid style, it's the bit-mask of chart type. */
#define MC_CHS_TYPEMASK        &H003f

'/**
' * @brief Disables a tooltip window.
' *
' * When having a tooltip window associated, the control uses to show additional
' * information about the pointed value in the chart. By default the control
' * creates its own control when this style is not set.
' *
' * @sa MC_CHM_SETTOOLTIPS
' */
#define MC_CHS_NOTOOLTIPS      &H0040
'
'/**
' * @brief Enable painting with double buffering.
' *
' * It prevents flickering when the control is being continuously resized.
' */
#define MC_CHS_DOUBLEBUFFER   &H0080
'
'/*@}*/
'
'
'/**
' * @anchor MC_CHDIF_xxxx
' * @name MC_NMCHDISPINFO::fMask Bits
' */
'/*@{*/
'
'/**
' * @brief The control asks for @ref MC_NMCHDISPINFO::piValues.
' *
' * The application is responsible to fill the buffer @c piValues with values
' * of data set determined by @c iDataSet, corresponding to the value indexes
' * in the interval from @c iValueFirst to @c iValueLast (including).
' *
' * The control guarantees the buffer @c piValues is large enough for
' * <tt>(iValueLast-iValueFirst+ 1)</tt> integers.
' */
#define MC_CHDIM_VALUES               1
'
'/*@}*/
'
'
'/**
' * @name Structures
' */
'/*@{*/
'
'/**
' * @brief Structure for manipulating with a data set.
' *
' * On input, set @c dwCount to a count of values in the data set (when inserting
' * or setting a data set), or to a number of values which can be written to the
' * buffer pointed by @c piValues (when getting a data set).
' *
' * If inserting or setting a data set and if @c piValues is set to @c NULL, then
' * the control will retrieve the data dynamically from its parent through
' * a notification @ref MC_CHN_GETDISPINFO.
' *
' * @sa MC_CHM_INSERTDATASET MC_CHM_GETDATASET MC_CHM_SETDATASET
' */
Type MC_CHDATASET
'    /** Count of values in the data set. */
     dwCount As DWORD
'    /** Pointer to an array of values in the data set. */
   piValues As Any Ptr 
End Type 

'/**
' * @brief Structure for notification @ref MC_CHN_GETDISPINFO.
' */
Type MC_NMCHDISPINFO
'    /** Standard notification structure header. */
     hdr As NMHDR
'    /** Mask of members the control asks for. See @ref MC_CHDIF_xxxx. */
     fMask As DWORD
'    /** Data set index. */
     iDataSet As Integer
'    /** Index of a first value the controls asks for. */
     iValueFirst As Integer
'    /** Index of last value the controls asks for. */
     iValueLast As Integer
'    /** Pointer to a buffer where application fills the values. */
     piValues As Any Ptr 
End Type 

'/*@}*/
'
'
'/**
' * @name Control Messages
' */
'/*@{*/
'
'/**
' * @brief Get count of data sets.
' * @param wParam Reserved, set to zero.
' * @param lParam Reserved, set to zero.
' * @return (@c int) Count of data sets.
' */
#define MC_CHM_GETDATASETCOUNT        (MC_CHM_FIRST + 0)
'
'/**
' * @brief Delete all data sets.
' * @param wParam Reserved, set to zero.
' * @param lParam Reserved, set to zero.
' * @return (@c BOOL) @c TRUE on success, @c FALSE otherwise.
' *
' * @sa MC_CHM_DELETEDATASET
' */
#define MC_CHM_DELETEALLDATASETS      (MC_CHM_FIRST + 1)
'
'/**
' * @brief Insert a data set.
' *
' * If the @ref MC_CHDATASET::piValues is set to @c NULL, the control will
' * retrieve the data dynamically from its parent through a notification
' * @ref MC_CHN_GETDISPINFO.
' *
' * @param[in] wParam (@c int) Index of the new data set.
' * @param[in] lParam (@ref MC_CHDATASET*) Pointer to a data of the new data set.
' * @return (@c int) Index of the new data set, or @c -1 on failure.
' */
#define MC_CHM_INSERTDATASET          (MC_CHM_FIRST + 2)
'
'/**
' * @brief Delete a data set.
' * @param[in] wParam (@c int) Index of the data set.
' * @param lParam Reserved, set to zero.
' * @return (@c BOOL) @c TRUE on success, @c FALSE otherwise.
' */
#define MC_CHM_DELETEDATASET          (MC_CHM_FIRST + 3)
'
'/**
' * @brief Get a data set.
' *
' * @param[in] wParam (@c int) Index of the data set.
' * @param[in,out] lParam (@ref MC_CHDATASET*) Pointer to a data of the new data
' * set. On input, its member @c dwCount must be set to indicate how many values
' * can be written to the buffer pointed by @c piValues. In output, the @c dwCount
' * is updated to a count of values written into the @c piValues. You may also set
' * @c lParam to @c NULL to just get number of values in the data set.
' * @return (@c int) Number of values in the data set, or @c -1 on failure.
' *
' * @note If the data set holds its value only virtually (i.e. if the
' * @c MC_CHDATASET::piValues was @c NULL when the dataset was set), then this
' * message just sets @c MC_CHDATASET::dwCount to zero.
' */
#define MC_CHM_GETDATASET             (MC_CHM_FIRST + 4)

'/**
' * @brief Set a data set.
' *
' * If the @c MC_CHDATASET::piValues is set to @c NULL, the control will
' * retrieve the data dynamically from its parent through a notification
' * @ref MC_CHN_GETDISPINFO.
' *
' * @param[in] wParam (@c int) Index of the data set to change.
' * @param[in] lParam (@ref MC_CHDATASET*) Pointer to a data of the new data set.
' * @return (@c BOOL) @c TRUE on success, @c FALSE otherwise.
' */
#define MC_CHM_SETDATASET             (MC_CHM_FIRST + 5)

'/**
' * @brief Get a color representing the data set in the chart.
' * @param[in] wParam (@c int) Index of the data set.
' * @param lParam Reserved, set to zero.
' * @return (@c COLORREF) @c The color, or @c -1 on failure.
' */
#define MC_CHM_GETDATASETCOLOR        (MC_CHM_FIRST + 6)

'/**
' * @brief Set a color representing the data set in the chart.
' * @param[in] wParam (@c int) Index of the data set.
' * @param[in] lParam (@c COLORREF) The color. It may be @ref MC_CLR_DEFAULT.
' * @return (@c BOOL) @c TRUE on success, @c FALSE otherwise.
' */
#define MC_CHM_SETDATASETCOLOR        (MC_CHM_FIRST + 7)

#define MC_CHM_GETDATASETLEGENDW      (MC_CHM_FIRST + 8)

#define MC_CHM_GETDATASETLEGENDA      (MC_CHM_FIRST + 9)

#define MC_CHM_SETDATASETLEGENDW      (MC_CHM_FIRST + 10)

#define MC_CHM_SETDATASETLEGENDA      (MC_CHM_FIRST + 11)

#define MC_CHM_GETFACTOREXPONENT      (MC_CHM_FIRST + 12)

#define MC_CHM_SETFACTOREXPONENT      (MC_CHM_FIRST + 13)

#define MC_CHM_GETAXISOFFSET          (MC_CHM_FIRST + 14)

#define MC_CHM_SETAXISOFFSET          (MC_CHM_FIRST + 15)

#define MC_CHM_SETTOOLTIPS            (MC_CHM_FIRST + 16)

#define MC_CHM_GETTOOLTIPS            (MC_CHM_FIRST + 17)

#define MC_CHM_GETAXISLEGENDW         (MC_CHM_FIRST + 18)

#define MC_CHM_GETAXISLEGENDA         (MC_CHM_FIRST + 19)

#define MC_CHM_SETAXISLEGENDW         (MC_CHM_FIRST + 20)

#define MC_CHM_SETAXISLEGENDA         (MC_CHM_FIRST + 21)

#define MC_CHN_GETDISPINFO            (MC_CHN_FIRST + 0)

'
#ifdef UNICODE
     #define MC_CHM_GETDATASETLEGEND   MC_CHM_GETDATASETLEGENDW
     #define MC_CHM_SETDATASETLEGEND   MC_CHM_SETDATASETLEGENDW
     #define MC_CHM_GETAXISLEGEND   MC_CHM_GETAXISLEGENDW
     #define MC_CHM_SETAXISLEGEND   MC_CHM_SETAXISLEGENDW
#else
     #define MC_CHM_GETDATASETLEGEND   MC_CHM_GETDATASETLEGENDA
     #define MC_CHM_SETDATASETLEGEND   MC_CHM_SETDATASETLEGENDA
     #define MC_CHM_GETAXISLEGEND   MC_CHM_GETAXISLEGENDA
     #define MC_CHM_SETAXISLEGEND   MC_CHM_SETAXISLEGENDA
#endif


'#ifdef __cplusplus
'}  /* extern "C" */
'#endif
'
'#endif  /* MCTRL_CHART_H */
