require("config")

-- These will be overwritten when loading a save.
global.list = global.list or {}
global.count = global.count or {personal_tesseract = {}, tesseract = {}, vehicle_tesseract = {}}

-- Debug functionality. Stolen from... somewhere... and modified.
local debug = function(...)
	debug_mem = debug_mem or {}

	local t = {}
	local i, v

	for i, v in ipairs({...}) do
		t[#t + 1] = tostring(v)
	end
	local s = table.concat(t, "")

	if game ~= nil and #game.players > 0 then
		if #debug_mem > 0 then	-- Could potentially optimize this.
			for _, m in pairs( debug_mem ) do
				for _, player in pairs(game.players) do
					if player.connected then 
						player.print(m)
					end
				end
			end

			debug_mem = {}
		end
		
		if s ~= nil then 
			for _, player in pairs(game.players) do
				if player.connected and player.mod_settings["wp-config-debug"].value then
					player.print(s)
				end
			end
		end
	else
		table.insert( debug_mem, s )
	end
end

local exception = function(...)
	debug_mem = debug_mem or {}

	local t = {}
	local i, v

	for i, v in ipairs({...}) do
		t[#t + 1] = tostring(v)
	end
	local s = table.concat(t, "")

	if game ~= nil and #game.players > 0 then
		if #debug_mem > 0 then	-- Could potentially optimize this.
			for _, m in pairs( debug_mem ) do
				for _, player in pairs(game.players) do
					if player.connected then 
						player.print(m)
					end
				end
			end

			debug_mem = {}
		end
		
		if s ~= nil then 
			for _, player in pairs(game.players) do
				if player.connected then
					player.print(s)
				end
			end
		end
	else
		table.insert( debug_mem, s )
	end
end

-----------------------------------------------------------------------------------------------------------------------
function table_remove(t, should_keep)
    local j, n = 1, #t;

    for i=1,n do
        if (should_keep(t, i, j)) then
            -- Move i's kept value to j's position, if it's not already there.
            if (i ~= j) then
                t[j] = t[i];
                t[i] = nil;
            end
            j = j + 1; -- Increment position of where we'll place the next kept value.
        else
            t[i] = nil;
        end
    end

    return t;
end


function get_tesseract_config( name )
	if name == nil then
		return nil
	end

	config = get_config(wp.tesseract, name, "in_label")
	if config == nil then
		config = get_config(wp.tesseract, name, "out_label")
	end
	if config == nil then
		config = get_config(wp.personal_tesseract, name, "label")
	end
	if config == nil then
		config = get_config(wp.vehicle_tesseract, name, "label")
	end
	return config
end


function get_config( t, name, key )
	for i = 1, #t do
		if name == t[i][key] then
			return t[i]
		end
	end
	return nil
end


function is_input(tesseract, name)
	return name == tesseract.in_label
end


function set_energy_level(t, new_energy)
	local entity = t.entity
	-- This was previously used to make sure accumulator animations played. Now that we use EEI, this doesn't do anything.
	--[[local old_energy = t.energy
	-- Ignore very small changes in power.
	if math.abs(new_energy - entity.energy) < 5 then
		return
	end
	-- Used to force animations while tesseracts are working.
	if new_energy == old_energy then
		if new_energy > 0 then
			new_energy = new_energy - 1
		else
			new_energy = new_energy + 1
		end
	end]]--
	entity.energy = new_energy
	t.energy = new_energy
end

-----------------------------------------------------------------------------------------------------------------------
function new_entity( entity )
	local config = get_tesseract_config(entity.name)
	if config then
		global.count[config.key][config.index] = (global.count[config.key][config.index] or 0) + 1
		table.insert( global.list, {entity = entity, energy = entity.energy, key = config.key, index = config.index, name = entity.name, warmup_cost = config.warmup_cost } )
		debug(global.list[#global.list].name, " created. Warmup: ", display_number(global.list[#global.list].warmup_cost), "J.")
	end
end

function on_entity_creation( event )
	new_entity( event.created_entity )
end
script.on_event(defines.events.on_built_entity, on_entity_creation )
script.on_event(defines.events.on_robot_built_entity, on_entity_creation )

function on_equipment_creation( event )
	new_entity( event.equipment )
end
script.on_event(defines.events.on_player_placed_equipment, on_equipment_creation )

function on_scripted_entity_creation( event )
	if event and event.entity and event.entity.name then
		new_entity( event.entity )
	end
end
script.on_event(defines.events.script_raised_revive, on_scripted_entity_creation )
script.on_event(defines.events.script_raised_built, on_scripted_entity_creation )

-----------------------------------------------------------------------------------------------------------------------
function on_tick(event)
	-- Check for global drain from tesseracts.
	local highest_active = nil
	local count = global.count.tesseract
	
	for i=#wp.tesseract, 1, -1 do
		if count[i] and count[i] > 0 then
			highest_active = wp.tesseract[i]
			break
		end
	end

	if highest_active ~= nil then	-- Skip all calculations if we have no possibility of input energy.
		-- Assumption: higher tiers of tesseract always have higher costs.
		local drain_total = highest_active.global_drain_per_tick
		local req_total = 0
		local sup_total = 0
		local list = global.list

		-- Figure out what is being requested.
		for i=1, #list do
			if list[i] == nil then
				break
			end

			local entry = list[i]
			local entity = entry.entity
			local config = wp[entry.key][entry.index]

			if not entity.valid then	-- Check for removed entities.
				debug((entry.name or (entry.key .. "-" .. entry.index)), " removed.")
				global.count[entry.key][entry.index] = global.count[entry.key][entry.index] - 1
				list[i] = list[#list]
				list[#list] = nil
				i = i - 1 -- i is now a different entity; need to evaluate again.
			elseif entity.name == config.in_label then	-- Check if we're dealing with an input tesseract.
				if entry.warmup_cost ~= nil then
					if entry.warmup_cost >= entity.energy then
						entry.warmup_cost = entry.warmup_cost - entity.energy
						entity.energy = 0
					else
						entity.energy = entity.energy - entry.warmup_cost
						entry.warmup_cost = nil
					end
				end
				sup_total = sup_total + (entity.energy * config.individual_efficiency)
			else
				local req = ((entity[config.buffer_key] - entity.energy) / config.individual_efficiency)
				if req > 0 then
					req_total = req_total + req
					drain_total = drain_total + config.individual_drain_per_tick
				end
			end
		end

		-- Calculate transfer ratios.
		local req_ratio = (sup_total - drain_total) / req_total
		if req_ratio ~= req_ratio then	-- Test for NaN, which happens when we have no request
			req_ratio = 0
		end
		req_ratio = math.max(0, math.min(req_ratio, 1))

		local sup_ratio = (req_total + drain_total) / sup_total
		if sup_ratio ~= sup_ratio then	-- Test for NaN, which happens when we have no supply
			sup_ratio = 0
		end
		sup_ratio = math.max(0, math.min(sup_ratio, 1))

		-- Keep in mind that drain_total is in terms of tesseract power (eg, after it has been charged 
		-- the efficiency cost from input tesseracts), but the specified drain in config is in terms 
		-- of input power. Thus, this drain number will be slightly off from what might be expected.
		if game.tick % (5 * 60) == 0 then
			debug("Total request: ", display_number(req_total * 60), "W, Request ratio: ", math.floor(req_ratio * 100 + 0.5), "%, Total supply: ", display_number(sup_total * 60), "W, Supply ratio: ", math.floor(sup_ratio * 100 + 0.5), "%, Drain: " .. display_number(drain_total * 60), "W")
		end
		
		-- Perform transfers.
		for i=1, #list do
			local entry = list[i]
			local entity = entry.entity
			local config = wp[entry.key][entry.index]

			if not entity.valid then
				if list[i].valid ~= false then
					exception(entry.name, " does not exist anymore, but is still somehow on the global list.")
				end
				list[i].valid = false
			elseif entity.name == config.in_label then	-- Check if we're dealing with an input tesseract.
				set_energy_level(list[i], entity.energy * (1 - sup_ratio))
			else										-- Otherwise, we're dealing with an output tesseract.
				local next_energy = entity.energy + ((entity[config.buffer_key] - entity.energy) * req_ratio)
				if entry.warmup_cost ~= nil then
					if entry.warmup_cost >= next_energy then
						entry.warmup_cost = entry.warmup_cost - next_energy
						next_energy = 0
					else
						next_energy = next_energy - entry.warmup_cost
						entry.warmup_cost = nil
					end
				end
				set_energy_level(list[i], next_energy)
			end
		end
	end
end
script.on_event(defines.events.on_tick, on_tick)