# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

import bpy

import csv
import os
import math

import mathutils

from .sets.factories import sets_factories, obj_collection_factories, obj_simple_factories, get_sets_mgr_by_id
from .preferences import get_prefs
from .vlog import Log
from .progress import start_progress, update_progress, end_progress
from .blender_zen_utils import ZenPolls


class AddonTestError(Exception):
    """ Critical error in test """
    pass


class AddonTestManual(Exception):
    """ Manual test """
    pass


def _set_object_mode(context):
    if context.mode != 'OBJECT':
        bpy.ops.object.mode_set(mode='OBJECT')

    if context.mode != 'OBJECT':
        raise AddonTestError('PREPARE> Object mode is expected!')


def _prepare_test(context):
    try:
        if context.mode != 'EDIT_MESH':
            if bpy.ops.object.mode_set.poll():
                bpy.ops.object.mode_set(mode='EDIT')

        bpy.ops.zsts.global_cleanup()
    except Exception:
        pass

    _set_object_mode(context)

    bpy.ops.zsts.global_cleanup(delete_all_collections=True)

    for p_obj in bpy.data.objects:
        bpy.data.objects.remove(p_obj)

    if len(bpy.data.objects):
        raise AddonTestError('PREPARE> Expect empty scene without objects!')

    context.scene.zen_object_collections_mode = 'obj_sets'

    bpy.ops.mesh.primitive_cube_add(enter_editmode=False)
    bpy.ops.mesh.primitive_cube_add(enter_editmode=True, location=(-4, 0, 0))

    if context.mode != 'EDIT_MESH':
        raise AddonTestError('PREPARE> Edit mode is expected!')

    if context.active_object is None:
        raise AddonTestError('PREPARE> Active object was not created!')


def _check_mgr_mode_prepared(p_cls_mgr, context, p_scene):
    p_scene.zen_sets_active_mode = p_cls_mgr.id_group
    p_cls_mgr.set_mesh_select_mode(context)

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.reveal(select=False)


def _check_group_prepared(p_cls_mgr, context, p_scene, p_obj):

    _check_mgr_mode_prepared(p_cls_mgr, context, p_scene)

    p_list = p_cls_mgr.get_list(p_scene)
    if len(p_list) != 1:
        raise AddonTestError('TEST> List length != 1')

    p_obj_list = p_cls_mgr.get_list(p_obj)
    if len(p_obj_list) != 1:
        raise AddonTestError('TEST> List length != 1')

    sel_count = p_cls_mgr.get_selected_count(p_obj)
    if sel_count:
        raise AddonTestError('TEST> Selection was not cleared!')


def _check_object_mgr_mode_prepared(p_cls_mgr, p_scene):
    p_scene.zen_object_collections_mode = p_cls_mgr.id_group

    bpy.ops.object.select_all(action='DESELECT')
    bpy.ops.object.hide_view_clear(select=False)


def _check_object_group_prepared(p_cls_mgr, p_scene):

    _check_object_mgr_mode_prepared(p_cls_mgr, p_scene)

    i_expected_length = 2 if p_scene.zen_object_collections_mode == 'obj_sets' else 1

    p_list = p_cls_mgr.get_list(p_scene)
    if len(p_list) != i_expected_length:
        raise AddonTestError(f'TEST> List length != {i_expected_length}')


def _check_scene_list_count(p_cls_mgr, p_scene, i_expected_count):
    i_scene_list_count = len(p_cls_mgr.get_list(p_scene))
    if i_scene_list_count != i_expected_count:
        raise AddonTestError(f'TEST> Scene list count must be {i_expected_count} but is {i_scene_list_count}!')


def _check_list_count(p_cls_mgr, p_scene, p_obj, i_expected_count):
    i_scene_list_count = len(p_cls_mgr.get_list(p_scene))
    if i_scene_list_count != i_expected_count:
        raise AddonTestError(f'TEST> Scene list count must be {i_expected_count} but is {i_scene_list_count}!')

    i_obj_list_count = len(p_cls_mgr.get_list(p_obj))
    if i_obj_list_count != i_expected_count:
        raise AddonTestError(f'TEST> Object list count must be {i_expected_count} but is {i_obj_list_count}!')


def _check_group_count(p_cls_mgr, p_scene, p_obj, i_expected_count):
    p_group = p_cls_mgr._get_scene_group(p_scene)

    if p_group is None:
        raise AddonTestError('TEST> Can not get Scene active group!')

    p_obj_group = p_cls_mgr._get_group_by_layer(p_obj, p_group.layer_name)
    if p_obj_group is None:
        raise AddonTestError('TEST> Can not get Object active group!')

    if p_obj_group.group_count != i_expected_count:
        raise AddonTestError(f'TEST> Group count mismatch! Expected {i_expected_count} but got {p_obj_group.group_count}')


def _check_object_group_count(p_cls_mgr, context, i_expected_count):
    p_group = p_cls_mgr._get_scene_group(context.scene)

    if p_group is None:
        raise AddonTestError('TEST> Can not get Scene active group!')

    p_objects = p_group.get_objects(context)
    i_obj_count = len(p_objects)
    if i_obj_count != i_expected_count:
        raise AddonTestError(f'TEST> Group count mismatch! Expected {i_expected_count} but got {i_obj_count}')


def _check_selected_count(p_cls_mgr, p_obj, p_bm_items, expected_indices):
    i_expected_count = len(expected_indices)
    i_sel_count = p_cls_mgr.get_selected_count(p_obj)
    if i_sel_count != i_expected_count:
        raise AddonTestError(f'TEST> Expected selected count {i_expected_count} but got {i_sel_count}')

    for i in expected_indices:
        if not p_bm_items[i].select:
            raise AddonTestError(f'TEST> BMesh Element:{i} is not selected!')


def _check_hidden_count(p_cls_mgr, p_obj, p_bm_items, expected_indices):
    i_expected_count = len(expected_indices)
    i_hidden_count = len([item for item in p_bm_items if item.hide])
    if i_hidden_count != i_expected_count:
        raise AddonTestError(f'TEST> Expected hidden count {i_expected_count} but got {i_hidden_count}')

    for i in expected_indices:
        if not p_bm_items[i].hide:
            raise AddonTestError(f'TEST> BMesh Element:{i} is not hidden!')


def Test_zsts_new_group(context):
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        p_scene.zen_sets_active_mode = p_cls_mgr.id_group
        p_cls_mgr.set_mesh_select_mode(context)

        bpy.ops.mesh.select_all(action='DESELECT')

        p_list = p_cls_mgr.get_list(p_scene)

        if len(p_list):
            raise AddonTestError('TEST> Expect empty list!')

        sel_count = p_cls_mgr.get_selected_count(p_obj)
        if sel_count:
            raise AddonTestError('TEST> Selection was not cleared!')

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[0].select_set(True)
        items[1].select_set(True)

        bpy.ops.zsts.new_group()

        _check_list_count(p_cls_mgr, p_scene, p_obj, 1)

        _check_group_count(p_cls_mgr, p_scene, p_obj, 2)


def Test_zsts_append_to_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ add 2,3 - elements and total must be 4 """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[2].select_set(True)
        items[3].select_set(True)

        bpy.ops.zsts.append_to_group()

        _check_group_count(p_cls_mgr, p_scene, p_obj, 4)


def Test_zsts_set_active_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ add 2,3 - elements and total must be 4 """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.set_active_group(
            mode='GROUP_NAME', group=p_cls_mgr.list_item_prefix)


def Test_zsts_remove_from_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ remove 1 element and total must be 1 """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[1].select_set(True)

        bpy.ops.zsts.remove_from_group()

        _check_group_count(p_cls_mgr, p_scene, p_obj, 1)


def Test_zsts_assign_to_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ assign 2,3,4 - elements and total must be 3 """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[2].select_set(True)
        items[3].select_set(True)
        items[4].select_set(True)

        bpy.ops.zsts.assign_to_group()

        _check_group_count(p_cls_mgr, p_scene, p_obj, 3)


def Test_zsto_assign_to_group(context):

    Test_zsto_new_group(context)

    p_scene = context.scene

    bpy.ops.mesh.primitive_cube_add(location=(0, 0, 10))
    obj1 = context.active_object
    bpy.ops.mesh.primitive_cube_add(location=(0, 0, 10))
    obj2 = context.active_object

    for f in obj_simple_factories:
        p_cls_mgr = f.get_mgr()
        _check_object_group_prepared(p_cls_mgr, p_scene)

        obj1.select_set(True)
        obj2.select_set(True)

        bpy.ops.zsto.assign_to_group()

        _check_object_group_count(p_cls_mgr, context, 2)


def Test_zsts_del_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ delete group and total groups must be 0 """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.del_group()

        _check_list_count(p_cls_mgr, p_scene, p_obj, 0)


def Test_zsts_delete_empty_groups(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ delete empty groups """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.mesh.select_all(action='SELECT')
        bpy.ops.zsts.remove_from_group()
        _check_group_count(p_cls_mgr, p_scene, p_obj, 0)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[2].select_set(True)
        items[3].select_set(True)

        bpy.ops.zsts.new_group()

        _check_list_count(p_cls_mgr, p_scene, p_obj, 2)

        bpy.ops.zsts.delete_empty_groups()
        _check_list_count(p_cls_mgr, p_scene, p_obj, 1)


def Test_zsts_delete_groups_combo(context):
    raise AddonTestManual


def Test_zsts_delete_groups_internal(context):
    raise AddonTestManual


def Test_zsts_select_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ select group and check 0,1 are selected """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.select_group(clear_selection=True)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        _check_selected_count(p_cls_mgr, p_obj, items, [0, 1])


def Test_zsts_select_ungroupped(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ select group and check 0,1 are selected """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.select_ungroupped()


def Test_zsts_select_only(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ select group and check 0,1 are selected """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.select_only()

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        _check_selected_count(p_cls_mgr, p_obj, items, [0, 1])


def Test_zsts_internal_hide_group_by_index(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ select group and check 0,1 are selected """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.internal_hide_group_by_index(group_index=0)


def Test_zsts_select_only_one_group(context):
    raise AddonTestManual


def Test_zsts_show_only_one_group(context):
    raise AddonTestManual


def Test_zsts_mode_selector(context):
    raise AddonTestManual


def Test_zsts_group_selector(context):
    raise AddonTestManual


def Test_zsts_show_tool_help(context):
    raise AddonTestManual


def Test_zsts_assign_tool_pinned(context):
    raise AddonTestManual


def Test_zsts_assign_to_pinned_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        if not p_cls_mgr.is_unique:
            _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

            bm = p_cls_mgr._get_bm(p_obj)
            items = p_cls_mgr.get_bm_items(bm)

            items[2].select_set(True)
            items[3].select_set(True)
            items[4].select_set(True)

            bpy.ops.zsts.assign_to_pinned_group(group_name="Pinned")


def Test_zsts_assign_uv_borders_to_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        if p_cls_mgr.id_group == 'edge':
            _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

            bm = p_cls_mgr._get_bm(p_obj)
            items = p_cls_mgr.get_bm_items(bm)

            items[2].select_set(True)
            items[3].select_set(True)
            items[4].select_set(True)

            bpy.ops.zsts.assign_uv_borders_to_group()


def Test_zsts_select_append(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ select group and check 0,1,2,3,4 are selected """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[2].select_set(True)
        items[3].select_set(True)
        items[4].select_set(True)

        bpy.ops.zsts.select_append()

        _check_selected_count(p_cls_mgr, p_obj, items, [0, 1, 2, 3, 4])


def Test_zsts_deselect_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ deselect group """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[0].select_set(True)
        items[1].select_set(True)
        items[2].select_set(True)

        _check_selected_count(p_cls_mgr, p_obj, items, [0, 1, 2])

        bpy.ops.zsts.deselect_group()

        _check_selected_count(p_cls_mgr, p_obj, items, [2])


def Test_zsts_intersect_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ intersect_group """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[1].select_set(True)
        items[2].select_set(True)

        _check_selected_count(p_cls_mgr, p_obj, items, [1, 2])

        bpy.ops.zsts.intersect_group()

        _check_selected_count(p_cls_mgr, p_obj, items, [1])


def Test_zsts_hide_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ hide_group """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        _check_hidden_count(p_cls_mgr, p_obj, items, [])

        bpy.ops.zsts.hide_group()

        _check_hidden_count(p_cls_mgr, p_obj, items, [0, 1])


def Test_zsts_unhide_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ unhide group """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        _check_hidden_count(p_cls_mgr, p_obj, items, [])

        bpy.ops.zsts.hide_group()

        _check_hidden_count(p_cls_mgr, p_obj, items, [0, 1])

        bpy.ops.zsts.unhide_group()

        _check_hidden_count(p_cls_mgr, p_obj, items, [])


def Test_zsts_invert_hide_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ invert_hide_group """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        _check_hidden_count(p_cls_mgr, p_obj, items, [])

        bpy.ops.zsts.hide_group('INVOKE_DEFAULT')

        _check_hidden_count(p_cls_mgr, p_obj, items, [0, 1])

        bpy.ops.zsts.invert_hide_group('INVOKE_DEFAULT')

        i_hidden_count = len([item for item in items if item.hide])
        if i_hidden_count != (len(items) - 2):
            raise AddonTestError('TEST> Expect that only 2 elements are visible')


def Test_zsts_draw_highlight(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ draw_highlight """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.draw_highlight()


def Test_zsts_export_groups_to_vertex_colors(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_groups_to_vertex_colors """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)
        if not p_cls_mgr.is_blender:
            bpy.ops.zsts.export_groups_to_vertex_colors()


def Test_zsts_import_vertex_colors_to_groups(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_groups_to_vertex_colors """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        if not p_cls_mgr.is_blender:
            bpy.ops.zsts.export_groups_to_vertex_colors()

            bpy.ops.zsts.import_vertex_colors_to_groups()


def Test_zsts_global_cleanup(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ global_cleanup """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zsts.global_cleanup()

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_mgr_mode_prepared(p_cls_mgr, context, p_scene)
        _check_list_count(p_cls_mgr, p_scene, p_obj, 0)


def Test_zsts_object_cleanup(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ object_cleanup """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zsts.object_cleanup(obj_name=p_obj.name)

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_mgr_mode_prepared(p_cls_mgr, context, p_scene)

        p_scene_list = p_cls_mgr.get_list(p_scene)
        if len(p_scene_list) != 1:
            raise AddonTestError(f'TEST> Scene list count must be {1} but is {len(p_scene_list)}')

        p_obj_list = p_cls_mgr.get_list(p_obj)
        if len(p_obj_list) != 0:
            raise AddonTestError(f'TEST> Object list count must be {0} but is {len(p_scene_list)}')


def Test_zsts_group_linked(context):
    """ group_linked """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()

        if p_cls_mgr.id_group in {'face_u', 'blgroup_u'}:
            _check_mgr_mode_prepared(p_cls_mgr, context, p_scene)

            bpy.ops.zsts.group_linked()

            _check_list_count(p_cls_mgr, p_scene, p_obj, 1)


def Test_zsts_internal_change_weight_by_index(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        if p_cls_mgr.id_group == 'blgroup':
            _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)
            bpy.ops.zsts.internal_change_weight_by_index()


def Test_zsts_add_vertex_weight_preset(context):
    raise AddonTestManual


def Test_zsts_smart_select(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ smart_select """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        _check_selected_count(p_cls_mgr, p_obj, items, [])

        items[1].select_set(True)

        bpy.ops.zsts.smart_select()

        _check_selected_count(p_cls_mgr, p_obj, items, [0, 1])


def Test_zsts_smart_isolate(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ smart_select """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        _check_selected_count(p_cls_mgr, p_obj, items, [])

        items[1].select_set(True)

        bpy.ops.zsts.smart_isolate()

        _check_selected_count(p_cls_mgr, p_obj, items, [0, 1])


def _check_seams_count(items, i_expected_count):
    i_seam_count = len([item for item in items if item.seam])
    if i_seam_count != i_expected_count:
        raise AddonTestError(f'TEST> Expect {i_expected_count} but got {i_seam_count}')


def Test_zsts_mark_seams(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ mark seams """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()

        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        if p_cls_mgr.id_element == 'edge':
            bm = p_cls_mgr._get_bm(p_obj)
            items = p_cls_mgr.get_bm_items(bm)

            for item in items:
                item.seam = False

            _check_seams_count(items, 0)
            bpy.ops.zsts.mark_seams()
            _check_seams_count(items, 2)


def Test_zsts_clear_seams(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ clear seams """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()

        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        if p_cls_mgr.id_element == 'edge':
            bm = p_cls_mgr._get_bm(p_obj)
            items = p_cls_mgr.get_bm_items(bm)

            for item in items:
                item.seam = False

            _check_seams_count(items, 0)
            bpy.ops.zsts.mark_seams()
            _check_seams_count(items, 2)
            bpy.ops.zsts.clear_seams()
            _check_seams_count(items, 0)


def Test_zsts_move_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ move_group """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bm = p_cls_mgr._get_bm(p_obj)
        items = p_cls_mgr.get_bm_items(bm)

        items[2].select_set(True)
        items[3].select_set(True)

        bpy.ops.zsts.new_group()

        _check_list_count(p_cls_mgr, p_scene, p_obj, 2)

        p_list = p_cls_mgr.get_list(p_scene)
        was_group_names = [g.name for g in p_list]

        bpy.ops.zsts.move_group(direction='UP')

        for i, g in enumerate(p_list):
            if g.name == was_group_names[i]:
                raise AddonTestError(f'TEST> Group {g.name} was not moved up!')


def _check_scene_objects(p_scene, i_expected_count):
    i_count = len(bpy.data.objects)
    if i_count != i_expected_count:
        raise AddonTestError(f'TEST> Scene objects count must be {i_expected_count} but is {i_count}')


def Test_zsts_split_to_objects(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ split_to_objects """
    p_obj = context.active_object

    p_scene = context.scene

    f = sets_factories[0]

    p_cls_mgr = f.get_mgr()
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    _check_scene_objects(p_scene, 2)
    bpy.ops.zsts.split_to_objects()
    _check_scene_objects(p_scene, 3)


def Test_zsts_select_scene_objects(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ select_scene_objects """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.select_all(action='SELECT')

        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.mesh.select_all(action='SELECT')
        bpy.ops.zsts.new_group()

        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.select_all(action='DESELECT')

        for obj in bpy.data.objects:
            if obj.type == 'MESH':
                obj.select_set(True)
                bpy.context.view_layer.objects.active = obj
                break

        bpy.ops.object.mode_set(mode='EDIT')
        bpy.ops.zsts.select_scene_objects()


def Test_zsts_pie_caller(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.pie_caller(cmd_default='zsts.global_cleanup')


def Test_zsts_reset_preferences(context):
    # bpy.ops.zsts.reset_preferences()
    raise AddonTestManual


def Test_zsts_pie_menu(context):
    raise AddonTestManual


def Test_zsts_show_keymaps(context):
    raise AddonTestManual


def Test_zsts_internal_build_lookup(context):
    bpy.ops.zsts.internal_build_lookup()


def Test_zsts_copy_to_clipboard(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ copy_to_clipboard """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.copy_to_clipboard()


def Test_zsts_paste_from_clipboard(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ paste_from_clipboard """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        _check_list_count(p_cls_mgr, p_scene, p_obj, 1)

        bpy.ops.zsts.copy_to_clipboard(group_mode='ALL')

        p_cls_mgr.delete_scene_groups(context)

        _check_list_count(p_cls_mgr, p_scene, p_obj, 0)

        bpy.ops.zsts.paste_from_clipboard()

        _check_list_count(p_cls_mgr, p_scene, p_obj, 1)


def Test_zsts_restore_preference(context):
    bpy.ops.zsts.restore_preference(mode='3D_VERT')
    bpy.ops.zsts.restore_preference(mode='3D_EDGE')
    bpy.ops.zsts.restore_preference(mode='3D_FACE')


def Test_zsts_rename_groups(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ rename_groups """
    p_obj = context.active_object

    p_scene = context.scene

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        bpy.ops.zsts.rename_groups(find=p_cls_mgr.id_group, replace='test', match_case=False)


def Test_zen_edge_u_list_split_group_edges(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ split_group_edges """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('edge_u')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_edge_u_list.split_group_edges(group_mode='ALL')


def Test_zen_face_list_import_active_face_map(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ import_active_face_map """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('face')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_face_u_list.export_active_to_face_map()

    bpy.ops.zen_face_list.import_active_face_map()


def Test_zen_face_list_import_all_face_maps(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ import_all_face_maps """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('face')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_face_u_list.export_all_to_face_maps()

    bpy.ops.zen_face_list.import_all_face_maps()


def Test_zen_face_u_list_import_active_face_map(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ import_active_face_map """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('face_u')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_face_u_list.export_active_to_face_map()

    bpy.ops.zen_face_u_list.import_active_face_map()


def Test_zen_face_u_list_import_all_face_maps(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ import_all_face_maps """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('face_u')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_face_u_list.export_all_to_face_maps()

    bpy.ops.zen_face_u_list.import_all_face_maps()


def Test_zen_face_u_list_assign_materials_to_groups(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ assign_materials_to_groups """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('face_u')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_face_u_list.assign_materials_to_groups()


def Test_zsto_assign_materials_to_groups(context):
    Test_zsto_new_group(context)

    p_cls_mgr = get_sets_mgr_by_id('obj_simple_parts')

    _check_object_group_prepared(p_cls_mgr, context.scene)

    bpy.ops.zsto.assign_materials_to_groups(assign_to_faces=True)


def Test_zsto_group_linked(context):
    Test_zsto_new_group(context)

    p_cls_mgr = get_sets_mgr_by_id('obj_simple_parts')

    _check_object_group_prepared(p_cls_mgr, context.scene)

    bpy.ops.zsto.group_linked()


def Test_zsto_remove_linked_objects(context):
    Test_zsto_new_group(context)

    p_cls_mgr = get_sets_mgr_by_id('obj_simple_parts')

    _check_object_group_prepared(p_cls_mgr, context.scene)

    bpy.ops.zsto.remove_linked_objects(group_mode='ALL')


def Test_zen_face_u_list_export_active_to_face_map(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_active_to_face_map """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('face_u')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_face_u_list.export_active_to_face_map()


def Test_zen_face_u_list_export_all_to_face_maps(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_all_to_face_maps """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('face_u')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_face_u_list.export_all_to_face_maps()


def Test_zen_vert_list_export_active_to_vertex_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_active_to_vertex_group """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('vert')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_vert_list.export_active_to_vertex_group()


def Test_zen_vert_list_export_all_to_vertex_groups(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_all_to_vertex_groups """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('vert')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_vert_list.export_all_to_vertex_groups()


def Test_zen_vert_u_list_export_active_to_vertex_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_active_to_vertex_group """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('vert_u')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_vert_u_list.export_active_to_vertex_group()


def Test_zen_vert_u_list_export_all_to_vertex_groups(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_all_to_vertex_groups """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('vert')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_vert_u_list.export_all_to_vertex_groups()


def Test_zen_vert_list_import_active_vertex_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ import_active_vertex_group """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('vert')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_vert_list.export_active_to_vertex_group()

    bpy.ops.zen_vert_list.import_active_vertex_group()


def Test_zen_vert_list_import_all_vertex_groups(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ import_all_vertex_groups """
    p_obj = context.active_object

    p_scene = context.scene

    p_cls_mgr = get_sets_mgr_by_id('vert')
    _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

    bpy.ops.zen_vert_list.export_all_to_vertex_groups()

    bpy.ops.zen_vert_list.import_all_vertex_groups()


def _create_new_palette(context):
    was_count = len(bpy.data.palettes)
    bpy.ops.zsts.palette_add_new()
    if len(bpy.data.palettes) != was_count + 1:
        raise AddonTestError('Can not create palette!')

    if context.scene.zsts_color_palette.palette is None:
        raise AddonTestError('Can not initialize [scene.zsts_color_palette.palette] !')


def _add_new_color_to_palette(context):
    pal = context.scene.zsts_color_palette.palette
    was_count = len(pal.colors)
    bpy.ops.zsts.palette_color_add()
    if len(pal.colors) != was_count + 1:
        raise AddonTestError('Can not add color!')


def _remove_color_from_palette(context):
    pal = context.scene.zsts_color_palette.palette
    was_count = len(pal.colors)
    bpy.ops.zsts.palette_color_remove()
    if len(pal.colors) != was_count - 1:
        raise AddonTestError('Can not remove color!')


def Test_zsts_palette_color_remove(context):
    _create_new_palette(context)

    _add_new_color_to_palette(context)

    _remove_color_from_palette(context)


def Test_zsts_palette_color_add(context):
    _create_new_palette(context)

    _add_new_color_to_palette(context)


def Test_zsts_palette_color_sort(context):
    if context.area.type != 'VIEW_3D':
        return

    _create_new_palette(context)

    pal = context.scene.zsts_color_palette.palette
    _add_new_color_to_palette(context)
    col = mathutils.Color(pal.colors[-1].color)
    col.h = 0.75
    pal.colors[-1].color = col

    _add_new_color_to_palette(context)
    col = mathutils.Color(pal.colors[-1].color)
    col.h = 0.5
    pal.colors[-1].color = col

    bpy.ops.zsts.palette_color_sort(type='HSV')
    col = mathutils.Color(pal.colors[-1].color)
    if math.fabs(col.h - 0.75) > 0.001:
        raise AddonTestError('Can not sort colors! Expect hue: 1.0 but got:', col.h)


def Test_zsts_palette_color_move(context):
    if context.area.type != 'VIEW_3D':
        return

    Test_zsts_palette_color_sort(context)

    pal = context.scene.zsts_color_palette.palette
    pal.colors.active = pal.colors[-1]

    was_color = mathutils.Color(pal.colors[-1].color)

    bpy.ops.zsts.palette_color_move(type='UP')

    if was_color != mathutils.Color(pal.colors[-2].color):
        raise AddonTestError('Can not move color!')


def Test_zsts_palette_color_assign_to_group(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ palette_color_assign_to_group """
    p_obj = context.active_object

    p_scene = context.scene

    _create_new_palette(context)

    pal = context.scene.zsts_color_palette.palette
    _add_new_color_to_palette(context)

    template_color = mathutils.Color((0.5, 0.5, 0.5))

    pal.colors[-1].color = template_color

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        p_list = p_cls_mgr.get_list(p_scene)

        bpy.ops.zsts.palette_color_assign_to_group(
            group_layer=p_list[-1].layer_name, group_mode='ACTIVE', palette_mode='PAL_SEQ')

        if p_list[-1].group_color != template_color:
            raise AddonTestError('Can not assign palette color to group!')


def Test_zsts_palette_select_color(context):
    _create_new_palette(context)

    _add_new_color_to_palette(context)
    _add_new_color_to_palette(context)

    bpy.ops.zsts.palette_select_color(color_index=0)

    pal = context.scene.zsts_color_palette.palette
    if pal.colors.active != pal.colors[0]:
        raise AddonTestError('Can not select palette color!')


def Test_zsts_palette_add_new(context):
    _create_new_palette(context)


def Test_zsts_export_group_colors_to_palette(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    """ export_group_colors_to_palette """
    p_obj = context.active_object

    p_scene = context.scene

    _create_new_palette(context)

    pal = context.scene.zsts_color_palette.palette

    template_color = mathutils.Color((0.5, 0.5, 0.5))

    for f in sets_factories:
        p_cls_mgr = f.get_mgr()
        _check_group_prepared(p_cls_mgr, context, p_scene, p_obj)

        p_list = p_cls_mgr.get_list(p_scene)

        for g in p_list:
            g.group_color = template_color

        bpy.ops.zsts.export_group_colors_to_palette()

        for col in pal.colors:
            if col.color != template_color:
                raise AddonTestError('Can not export group colors! Expect:', template_color[:],
                                     'but got:', col.color[:])

        break


def Test_zsts_reset_draw_cache(context):
    bpy.ops.zsts.reset_draw_cache()


def Test_zsts_set_workspace_tool(context: bpy.types.Context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.set_workspace_tool()

    if not ZenPolls.poll_zen_tool_in_workspace(context):
        raise AddonTestError('TEST> Can not set Zen Sets workspace tool!')


def Test_zsts_pie_caller_bottom(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.pie_caller_bottom(cmd_enum='zsts.set_workspace_tool', is_menu=False)


def Test_zsts_pie_caller_bottom_left(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.pie_caller_bottom_left()


def Test_zsts_pie_caller_bottom_right(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.pie_caller_bottom_right()


def Test_zsts_pie_caller_right(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.pie_caller_right(cmd_enum='zsts.select_group', is_menu=False)


def Test_zsts_pie_caller_left(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.pie_caller_left(cmd_enum='zsts.deselect_group', is_menu=False)


def Test_zsts_pie_caller_top_left(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.pie_caller_top_left(cmd_enum='zsts.del_group', is_menu=False)


def Test_zsts_pie_caller_top(context):
    """ create 1 group with 0,1 - elements """
    Test_zsts_new_group(context)

    bpy.ops.zsts.pie_caller_top(
        cmd_enum='zsts.smart_select', is_menu=True)


def Test_zsts_remove_weight_from_group(context):
    raise AddonTestManual


def Test_zsts_draw_show_tool_help(context):
    raise AddonTestManual


def Test_zsts_assign_vertex_weight_to_group_color(context):
    raise AddonTestManual


def Test_zsto_delete_groups_internal(context):
    _set_object_mode(context)
    # bpy.ops.zsto.delete_groups_internal(delete_mode='EMPTY', without_nested_objects=False)
    raise AddonTestManual


def Test_zsto_delete_groups_combo(context):
    _set_object_mode(context)
    # bpy.ops.zsto.delete_groups_combo(delete_mode='EMPTY', without_nested_objects=False)
    raise AddonTestManual


def Test_zsto_pie_caller_top_left(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.pie_caller_top_left(cmd_enum='zsto.del_group_only', is_menu=True)


def Test_zsto_new_group_link(context):
    _set_object_mode(context)
    bpy.ops.zsto.new_group_link()


def Test_zsto_delete_empty_groups(context):

    Test_zsto_new_group(context)

    bpy.ops.zsto.delete_empty_groups(without_nested_objects=False)


def Test_zsto_new_group(context):
    _set_object_mode(context)

    p_scene = context.scene

    for f in obj_collection_factories:
        p_cls_mgr = f.get_mgr()
        p_scene.zen_object_collections_mode = p_cls_mgr.id_group

        bpy.ops.object.select_all(action='DESELECT')

        p_list = p_cls_mgr.get_list(p_scene)

        i_expected_length = 1 if p_scene.zen_object_collections_mode == 'obj_sets' else 0

        if len(p_list) != i_expected_length:
            raise AddonTestError('TEST> Expect empty list!')

        sel_count = len(context.selected_objects)
        if sel_count:
            raise AddonTestError('TEST> Selection was not cleared!')

        bpy.ops.object.select_all(action='SELECT')

        sel_count = len(context.selected_objects)
        if sel_count != 2:
            raise AddonTestError(f'Expect 2 selected objects, but got {sel_count}')

        bpy.ops.zsto.new_group()

        p_cls_mgr.update_collections(p_scene)

        _check_scene_list_count(p_cls_mgr, p_scene, i_expected_length + 1)

        _check_object_group_count(p_cls_mgr, context, 2)


def Test_zsto_append_to_group(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.append_to_group(move=True, keep_nested=True)


def Test_zsto_remove_from_group(context):
    Test_zsto_new_group(context)

    for f in obj_collection_factories:
        p_cls_mgr = f.get_mgr()
        _check_object_group_prepared(p_cls_mgr, context.scene)

        bpy.ops.object.select_all(action='SELECT')

        bpy.ops.zsto.remove_from_group(keep_nested=False)


def Test_zsto_select_objects_with(context):
    _set_object_mode(context)

    bpy.ops.zsto.select_objects_with()


def Test_zsto_pie_caller_top_right(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.pie_caller_top_right(cmd_enum='zsto.new_group_link', is_menu=True)


def Test_zsto_duplicate_as_intance(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.duplicate_as_intance()


def Test_zsto_duplicate_collection_linked(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.duplicate_collection_linked()


def Test_zsto_pie_caller_left(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.pie_caller_left(
        cmd_enum='zsts.deselect_group', is_menu=True)


def Test_zsto_hide_viewport_by_index(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.hide_viewport_by_index(group_index=0, mode='DEFAULT', select=True, nested=True)


def Test_zsto_internal_select_objects_by_index(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.internal_select_objects_by_index(group_index=0, mode='SELECT', nested=True, move=True, keep_nested=True)


def Test_zsto_hide_select_by_index(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.hide_select_by_index(group_index=0, mode='DEFAULT', nested=True)


def Test_zsto_convert_collection_to_object(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.convert_collection_to_object()


def Test_zsto_invert_hide_group(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.invert_hide_group(select=True, nested=True)


def Test_zsto_convert_object_to_collection(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.convert_collection_to_object()

    bpy.ops.zsto.convert_object_to_collection()


def Test_zsto_pie_caller_right(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.pie_caller_right(cmd_enum='zsts.select_group', is_menu=True)


def Test_zsto_del_group_only(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.del_group_only()


def Test_zsto_unhide_all(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.unhide_all(select=True)


def Test_zsto_pie_caller_bottom(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.pie_caller_bottom(
        cmd_enum='bpy.ops.wm.tool_set_by_id(name="zsts.select_object_tool")', is_menu=True)


def Test_zsto_exclude_toggle_group(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.exclude_toggle_group()


def Test_zsto_del_group(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.del_group(hierarchy=False)


def Test_zsto_exclude_group_by_index(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.exclude_group_by_index(group_index=0, mode='DEFAULT', nested=True)


def Test_zsto_disable_in_renders_by_index(context):
    raise AddonTestManual


def Test_zsto_hide_render_by_index(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.hide_render_by_index(group_index=0, mode='DEFAULT', nested=True)


def Test_zsto_disable_viewport_by_index(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.disable_viewport_by_index(group_index=0, mode='DEFAULT', nested=True)


def Test_zsto_duplicate_collection(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.duplicate_collection()


def Test_zsto_new_group_move(context):
    _set_object_mode(context)
    bpy.ops.zsto.new_group_move(new_group_name="Test")


def Test_zsto_delete_hierarchy(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.delete_hierarchy()


def Test_zsto_pie_caller_top(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.pie_caller_top()


def Test_zsts_copy_python_cmd_to_auto_groups(context):
    raise AddonTestManual


def Test_zsts_select_mode(context):
    raise AddonTestManual


def Test_zsto_pie_caller_bottom_left(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.pie_caller_bottom_left(
        cmd_enum='zsto.remove_from_group("INVOKE_DEFAULT", keep_nested=False)',
        is_menu=True)


def Test_zsto_pie_caller_bottom_right(context):
    Test_zsto_new_group(context)

    bpy.ops.zsto.pie_caller_bottom_right(
        cmd_enum='zsto.append_to_group("INVOKE_DEFAULT", keep_nested=False)',
        is_menu=True)


def Test_zsto_rename_as_parent_collection(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.rename_as_parent_collection()


def Test_zsto_rename_collection_as_object(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.zsto.rename_collection_as_object()


def Test_zsto_sync_parent_pos_with_child(context):
    _set_object_mode(context)

    bpy.ops.zsto.new_group()

    bpy.ops.object.empty_add(type='PLAIN_AXES')
    p_obj_parent = context.active_object

    bpy.ops.mesh.primitive_cube_add()
    p_child = context.active_object

    p_child.parent = p_obj_parent

    bpy.ops.zsto.sync_parent_pos_with_child()


def Test_zsto_colorize_selected(context):
    raise AddonTestManual


class ZsAddonTest(bpy.types.Operator):
    bl_idname = 'zsts_test.addon_test'
    bl_label = 'Zen Sets Addon Testing'
    bl_description = 'Executes list of Zen Sets Tests'
    bl_options = {'REGISTER'}

    stop_on_fail: bpy.props.BoolProperty(name='Stop on fail', default=True)
    report_undefined: bpy.props.BoolProperty(name='Report about undefined tests', default=True)

    @classmethod
    def poll(cls, context):
        return get_prefs().common.developer_mode

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)

    def execute(self, context):
        try:
            _prepare_test(context)

            csv_dirname = os.path.dirname(__file__)
            csv_filename = os.path.join(csv_dirname, 'report.csv')

            with open(csv_filename, 'w', newline='') as file:
                writer = csv.writer(file)
                writer.writerow(["No", "Module", "Operator", "Result", "Description"])

                operators_list = set()

                for op_module_name in dir(bpy.ops):

                    def is_zen_module(name):
                        elements = ('vert', 'edge', 'face')
                        for elem in elements:
                            if name.startswith('zen_' + elem):
                                return True
                        return False

                    if op_module_name in {'zsts', 'zsto'} or is_zen_module(op_module_name):
                        op_module = getattr(bpy.ops, op_module_name)
                        start_progress(context, 0, 100, high_priority=True)
                        zsts_operators = dir(op_module)
                        for i, op_submodule_name in enumerate(zsts_operators):
                            op = getattr(op_module, op_submodule_name)
                            text = repr(op)
                            if text.split("\n")[-1].startswith("bpy.ops."):
                                operators_list.add((text, op_module_name, op_submodule_name))

                tot = 0
                start_progress(context, 0, 100, high_priority=True)
                for i, op_data in enumerate(operators_list):

                    text, op_module_name, op_submodule_name = op_data

                    result = 'UNKNOWN'
                    description = ''
                    try:
                        Log.info('Starting:', text)

                        func_name = f'Test_{op_module_name}_{op_submodule_name}'
                        func_name_exec = f'{func_name}(context)'

                        exec(f'_func_check = {func_name}')
                        _prepare_test(context)
                        exec(func_name_exec)

                        result = 'SUCCESS'
                    except NameError:
                        result = 'NO TEST FOUND'
                        description = f'{func_name_exec} - not found!'
                        Log.warn(text, description)
                    except AddonTestManual:
                        result = 'MANUAL'
                        description = 'MANUAL!'
                        Log.warn(text, description)
                    except AddonTestError as e:
                        result = 'FAILED'
                        description = func_name_exec + ': ' + str(e)
                        Log.error(text, description)
                    except Exception as e:
                        result = 'ERROR EXCEPTION'
                        description = func_name_exec + ': ' + str(e)
                        Log.error(text, description)

                    tot += 1

                    if self.report_undefined or result != 'NO TEST FOUND':
                        writer.writerow([tot, op_module_name, op_submodule_name, result, description])

                        valid_results = ['SUCCESS', 'MANUAL']

                        if result not in valid_results:
                            if self.stop_on_fail:
                                raise AddonTestError(result, description)
                            else:
                                self.report({'ERROR'}, description)
                    i_percent = int(i / len(operators_list) * 100)
                    Log.debug(f'Completed:{text} - {i_percent}%')
                    update_progress(context, i_percent, high_priority=True)
                end_progress(context, high_priority=True)
        except Exception as e:
            self.report({'ERROR'}, str(e))

        end_progress(context, high_priority=True)

        try:
            if bpy.ops.object.mode_set.poll():
                bpy.ops.object.mode_set(mode='EDIT')
        except Exception as e:
            print(e)

        return {'FINISHED'}


def menu_addon_test_func(self, context):
    if get_prefs().common.developer_mode:
        layout = self.layout
        layout.separator()
        layout.operator('zsts_test.addon_test')


def register():
    bpy.utils.register_class(ZsAddonTest)
    bpy.types.ZSTS_MT_GroupMenu.append(menu_addon_test_func)


def unregister():
    bpy.types.ZSTS_MT_GroupMenu.remove(menu_addon_test_func)
    bpy.utils.unregister_class(ZsAddonTest)
