# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

""" Zen Blender Utils """
import bpy
import mathutils

import ctypes
import platform
import re
import numpy as np

from enum import IntEnum

from .vlog import Log

_ATTR_DEPSGRAPH_UPDATE_LOCK = 'zen_sets_depsgraph_update_lock'
_ATTR_DEPSGRAPH_UPDATE_ONE_LOCK = 'zen_sets_depsgraph_update_one_lock'
_ATTR_LAY_COL_UPDATE_ONE_LOCK = 'zen_sets_lay_col_update_one_lock'
_ATTR_NOTIFY_EDIT_LOCK = 'zen_sets_notify_edit_lock'
_ATTR_SCENE_GROUP_INDEX_LOCK = 'zen_sets_index_lock'
_ATTR_LOOKUP_BUILD_LOCK = 'zen_sets_lookup_build_lock'
_ATTR_DRAW_LOCK = 'zen_sets_draw_lock'
_ATTR_DRAW_TOOL_HELP_FLAG = 'zen_sets_draw_tool_flag'


_ATTR_OBJ_COLOR_STATE = 'zen_sets_object_color_state'


class ZenLocks:
    @staticmethod
    def is_depsgraph_update_locked():
        return (
            _ATTR_DEPSGRAPH_UPDATE_LOCK in bpy.app.driver_namespace.keys()) \
            and bpy.app.driver_namespace[_ATTR_DEPSGRAPH_UPDATE_LOCK]

    @staticmethod
    def lock_depsgraph_update():
        bpy.app.driver_namespace[_ATTR_DEPSGRAPH_UPDATE_LOCK] = True

    @staticmethod
    def unlock_depsgraph_update():
        bpy.app.driver_namespace[_ATTR_DEPSGRAPH_UPDATE_LOCK] = False

    @staticmethod
    def is_depsgraph_update_one_locked():
        return (
            _ATTR_DEPSGRAPH_UPDATE_ONE_LOCK in bpy.app.driver_namespace.keys()) \
            and bpy.app.driver_namespace[_ATTR_DEPSGRAPH_UPDATE_ONE_LOCK]

    @staticmethod
    def lock_depsgraph_update_one():
        bpy.app.driver_namespace[_ATTR_DEPSGRAPH_UPDATE_ONE_LOCK] = True

    @staticmethod
    def unlock_depsgraph_update_one():
        bpy.app.driver_namespace[_ATTR_DEPSGRAPH_UPDATE_ONE_LOCK] = False

    @staticmethod
    def is_lay_col_update_one_locked():
        return (
            _ATTR_LAY_COL_UPDATE_ONE_LOCK in bpy.app.driver_namespace.keys()) \
            and bpy.app.driver_namespace[_ATTR_LAY_COL_UPDATE_ONE_LOCK]

    @staticmethod
    def lock_lay_col_update_one():
        bpy.app.driver_namespace[_ATTR_LAY_COL_UPDATE_ONE_LOCK] = True

    @staticmethod
    def unlock_lay_col_update_one():
        bpy.app.driver_namespace[_ATTR_LAY_COL_UPDATE_ONE_LOCK] = False

    @staticmethod
    def is_notify_edit_locked():
        return (
            _ATTR_NOTIFY_EDIT_LOCK in bpy.app.driver_namespace.keys()) \
            and bpy.app.driver_namespace[_ATTR_NOTIFY_EDIT_LOCK]

    @staticmethod
    def lock_notify_edit():
        bpy.app.driver_namespace[_ATTR_NOTIFY_EDIT_LOCK] = True

    @staticmethod
    def unlock_notify_edit():
        bpy.app.driver_namespace[_ATTR_NOTIFY_EDIT_LOCK] = False

    @staticmethod
    def is_lookup_build_locked():
        return (
            _ATTR_LOOKUP_BUILD_LOCK in bpy.app.driver_namespace.keys()) \
            and bpy.app.driver_namespace[_ATTR_LOOKUP_BUILD_LOCK]

    @staticmethod
    def lock_lookup_build():
        bpy.app.driver_namespace[_ATTR_LOOKUP_BUILD_LOCK] = True

    @staticmethod
    def unlock_lookup_build():
        bpy.app.driver_namespace[_ATTR_LOOKUP_BUILD_LOCK] = False

    @staticmethod
    def get_draw_overlay_locked():
        if _ATTR_DRAW_LOCK in bpy.app.driver_namespace.keys():
            return bpy.app.driver_namespace[_ATTR_DRAW_LOCK]
        return 1.0

    @staticmethod
    def set_lock_draw_overlay(percent):
        bpy.app.driver_namespace[_ATTR_DRAW_LOCK] = percent

    @staticmethod
    def is_group_index_locked(id_group):
        return (
            (_ATTR_SCENE_GROUP_INDEX_LOCK + id_group) in bpy.app.driver_namespace.keys()) \
            and bpy.app.driver_namespace[_ATTR_SCENE_GROUP_INDEX_LOCK + id_group]

    @staticmethod
    def lock_group_index(id_group):
        bpy.app.driver_namespace[_ATTR_SCENE_GROUP_INDEX_LOCK + id_group] = True

    @staticmethod
    def unlock_group_index(id_group):
        bpy.app.driver_namespace[_ATTR_SCENE_GROUP_INDEX_LOCK + id_group] = False

    @staticmethod
    def is_draw_tool_help_flag(space_data: str):
        return bpy.app.driver_namespace.get(_ATTR_DRAW_TOOL_HELP_FLAG + space_data, False)

    @staticmethod
    def set_draw_tool_help_flag(space_data: str, state: bool):
        bpy.app.driver_namespace[_ATTR_DRAW_TOOL_HELP_FLAG + space_data] = state

    @staticmethod
    def clear_all_draw_tool_help_flags():
        for space_type in {'VIEW_3D', 'IMAGE_EDITOR'}:
            bpy.app.driver_namespace[_ATTR_DRAW_TOOL_HELP_FLAG + space_type] = False


class ZenStates:
    @staticmethod
    def set_obj_color_state():
        bpy.app.driver_namespace[_ATTR_OBJ_COLOR_STATE] = True

    @staticmethod
    def unset_obj_color_state():
        bpy.app.driver_namespace[_ATTR_OBJ_COLOR_STATE] = False

    @staticmethod
    def is_obj_color_state():
        return (
            _ATTR_OBJ_COLOR_STATE in bpy.app.driver_namespace.keys()) \
            and bpy.app.driver_namespace[_ATTR_OBJ_COLOR_STATE]


class ZsViewLayerStoredType(IntEnum):
    HideSelect = 0,
    HideViewport = 1,
    HideGet = 2,
    SelectGet = 3,
    WasEditable = 4


def save_viewlayer_layers_state(parent_layer, layers_state):
    for child_layer in parent_layer.children:
        layers_state[child_layer.name] = (child_layer.exclude, child_layer.hide_viewport)
        save_viewlayer_layers_state(child_layer, layers_state)


def show_all_viewlayers(parent_layer):
    for child_layer in parent_layer.children:
        child_layer.exclude = False
        child_layer.hide_viewport = False
        show_all_viewlayers(child_layer)


def restore_viewlayer_layers(parent_layer, layers_state):
    for child_layer in parent_layer.children:
        if child_layer.name in layers_state:
            is_excluded, is_viewport_hidden = layers_state[child_layer.name]
            if child_layer.exclude != is_excluded:
                child_layer.exclude = is_excluded
            if child_layer.hide_viewport != is_viewport_hidden:
                child_layer.hide_viewport = is_viewport_hidden
        restore_viewlayer_layers(child_layer, layers_state)


def save_viewlayer_objects_state():
    view_layer = bpy.context.view_layer
    act_obj_name = view_layer.objects.active.name if view_layer.objects.active else ''
    were_objects = {obj.name: (obj.hide_select,
                               obj.hide_viewport,
                               obj.hide_get(),
                               obj.select_get(),
                               obj.data.is_editmode if obj.type == 'MESH' else (act_obj_name == obj.name))
                    for obj in view_layer.objects}
    return (were_objects, act_obj_name)


def restore_viewlayer_objects(were_objects):
    obj_list = bpy.context.view_layer.objects
    for obj_name, obj_data in were_objects.items():
        obj = obj_list.get(obj_name)
        if obj:
            if obj.hide_get() != obj_data[ZsViewLayerStoredType.HideGet]:
                obj.hide_set(obj_data[ZsViewLayerStoredType.HideGet])

            if obj.hide_select != obj_data[ZsViewLayerStoredType.HideSelect]:
                obj.hide_select = obj_data[ZsViewLayerStoredType.HideSelect]
            if obj.hide_viewport != obj_data[ZsViewLayerStoredType.HideViewport]:
                obj.hide_viewport = obj_data[ZsViewLayerStoredType.HideViewport]

            if obj.select_get() != obj_data[ZsViewLayerStoredType.SelectGet]:
                obj.select_set(obj_data[ZsViewLayerStoredType.SelectGet])
        else:
            Log.error('Can not restore:', obj_name)


def ensure_object_in_viewlayer(obj, parent_layer):
    if obj.hide_viewport:
        obj.hide_viewport = False
    if obj.hide_select:
        obj.hide_select = False

    for child_layer in parent_layer.children:
        if obj.name in child_layer.collection.all_objects:
            if child_layer.exclude:
                child_layer.exclude = False
            if child_layer.hide_viewport:
                child_layer.hide_viewport = False
            if child_layer.collection.hide_select:
                child_layer.collection.hide_select = False

        ensure_object_in_viewlayer(obj, child_layer)


def unhide_and_select_all_viewlayer_objects(act_obj_name):
    view_layer = bpy.context.view_layer
    for obj in view_layer.objects:
        if obj.type == 'MESH':
            obj.hide_viewport = False
            obj.hide_select = False
            obj.hide_set(False)
            obj.select_set(True)
        if obj.name == act_obj_name:
            view_layer.objects.active = obj


def prepare_stored_objects_for_edit(were_objects, act_obj_name):
    view_layer = bpy.context.view_layer
    for obj in view_layer.objects:
        if obj.name in were_objects:
            was_in_edit_mode = were_objects[obj.name][ZsViewLayerStoredType.WasEditable]
            if was_in_edit_mode:
                try:
                    obj.hide_viewport = False
                    obj.hide_select = False
                    obj.hide_set(False)
                    obj.select_set(True)
                except Exception:
                    pass
            else:
                try:
                    obj.hide_viewport = were_objects[obj.name][ZsViewLayerStoredType.HideViewport]
                    obj.hide_select = were_objects[obj.name][ZsViewLayerStoredType.HideSelect]
                    obj.hide_set(were_objects[obj.name][ZsViewLayerStoredType.HideGet])
                    obj.select_set(False)
                except Exception:
                    pass

        if obj.name == act_obj_name:
            view_layer.objects.active = obj


class ZenModeSwitcher:
    def __init__(self, mode='OBJECT'):
        self.were_objects, self.act_obj_name = save_viewlayer_objects_state()
        ctx_mode = bpy.context.mode
        if ctx_mode == 'OBJECT':
            self.was_mode = 'OBJECT'
        else:
            self.was_mode = 'EDIT'
        if mode != self.was_mode:
            if bpy.ops.object.mode_set.poll():
                bpy.ops.object.mode_set(mode=mode)

    def return_to_edit_mode(self):
        prepare_stored_objects_for_edit(self.were_objects, self.act_obj_name)
        bpy.ops.object.mode_set(mode='EDIT')
        restore_viewlayer_objects(self.were_objects)

    def return_to_mode(self):
        if self.was_mode == 'EDIT':
            prepare_stored_objects_for_edit(self.were_objects, self.act_obj_name)
        if bpy.ops.object.mode_set.poll():
            bpy.ops.object.mode_set(mode=self.was_mode)
        restore_viewlayer_objects(self.were_objects)


def fix_undo_push_edit_mode(msg):
    try:
        ZenLocks.lock_depsgraph_update()

        active_object = bpy.context.view_layer.objects.active
        if active_object and active_object.mode == "EDIT":
            try:
                ZenLocks.lock_notify_edit()
                mode_switcher = ZenModeSwitcher()

                bpy.ops.mesh.primitive_cube_add(enter_editmode=False)
                bpy.ops.object.delete(use_global=True, confirm=False)

                bpy.ops.object.select_all(action='DESELECT')

                mode_switcher.return_to_edit_mode()
            except Exception as e:
                Log.error('FIX_UNDO:', e)

        bpy.ops.ed.undo_push(message=msg)
    finally:
        ZenLocks.unlock_depsgraph_update()
    pass


class ZenPolls:

    version_greater_3_2_0 = False

    @classmethod
    def is_collection_mode(cls, context: bpy.types.Context):
        return context.scene.zen_object_collections_mode == 'obj_sets'

    @classmethod
    def is_object_and_collection_mode(cls, context: bpy.types.Context):
        return context.mode == 'OBJECT' and cls.is_collection_mode(context)

    @classmethod
    def is_object_and_simple_mode(cls, context: bpy.types.Context):
        return context.mode == 'OBJECT' and not cls.is_collection_mode(context)

    @classmethod
    def is_vertex_group_mode(cls, context: bpy.types.Context):
        return context.scene.zen_sets_active_mode == 'blgroup'

    @classmethod
    def is_edit_and_vertex_group_mode(cls, context: bpy.types.Context):
        return context.mode == 'EDIT_MESH' and cls.is_vertex_group_mode(context)

    @classmethod
    def is_view3d_space_data(cls, context):
        return hasattr(context, 'space_data') and context.space_data and context.space_data.type == 'VIEW_3D'

    @classmethod
    def poll_edit_or_uv(cls, context):
        if context.mode == 'EDIT_MESH':
            space = context.space_data
            if space:
                space_type = space.type
                if space_type == 'VIEW_3D' or (space.type == 'IMAGE_EDITOR' and getattr(space, 'show_uvedit', False)):
                    return True
        return False

    @classmethod
    def poll_object_edit_or_uv(cls, context):
        if context.mode == 'OBJECT':
            return True

        if cls.poll_edit_or_uv(context):
            return True

        return False

    @classmethod
    def poll_edit_or_uv_with_objs_in_mode(cls, context):
        return cls.poll_edit_or_uv(context) and (len(context.objects_in_mode) != 0)

    @classmethod
    def poll_object_edit_or_uv_with_objs_in_mode(cls, context):
        return context.mode == 'OBJECT' or cls.poll_edit_or_uv_with_objs_in_mode(context)

    @classmethod
    def poll_zen_tool_in_workspace(cls, context):
        if context.space_data.type == 'IMAGE_EDITOR':
            _id = getattr(context.workspace.tools.from_space_image_mode('UV', create=False), 'idname', None)
            if isinstance(_id, str) and _id.startswith('zsts.select_'):
                return True
        else:
            _id = getattr(context.workspace.tools.from_space_view3d_mode(context.mode, create=False), 'idname', None)
            if isinstance(_id, str) and _id.startswith('zsts.select_'):
                return True

        return False


class ZenSelectionStats:
    def __init__(self) -> None:
        self.was_mesh_sel_count = 0
        self.was_uv_sel_count = 0
        self.was_object_sel_count = 0

        self.new_mesh_sel_count = 0
        self.new_uv_sel_count = 0
        self.new_object_sel_count = 0

        self.selection_changed = False


def update_view3d_in_all_screens():
    context = bpy.context

    for window in context.window_manager.windows:
        for area in window.screen.areas:
            if area.type == 'VIEW_3D':
                area.tag_redraw()


def update_imageeditor_in_all_screens():
    context = bpy.context

    for window in context.window_manager.windows:
        for area in window.screen.areas:
            if area.type == 'IMAGE_EDITOR':
                area.tag_redraw()


def update_areas_in_all_screens():
    context = bpy.context

    for window in context.window_manager.windows:
        for area in window.screen.areas:
            if area.type in {'VIEW_3D', 'IMAGE_EDITOR'}:
                area.tag_redraw()


def is_any_uv_editor_opened(context):
    for window in context.window_manager.windows:
        for area in window.screen.areas:
            if area.type == 'IMAGE_EDITOR':
                for space in area.spaces:
                    if getattr(space, 'show_uvedit', False):
                        return True
    return False


def is_property_default(prop, obj_prop):
    if hasattr(prop, "default"):
        is_array = getattr(prop, "is_array", False)
        if is_array:
            default_array = [p for p in prop.default_array]
            current_value = [p for p in getattr(obj_prop, prop.identifier)]
            if default_array != current_value:
                return True
        elif getattr(prop, 'is_enum_flag', False):
            current = getattr(obj_prop, prop.identifier)
            if current != prop.default_flag:
                return True
        else:
            current = getattr(obj_prop, prop.identifier)
            if current != prop.default:
                return True
    return False


def is_property_modified(obj_prop):
    for prop in obj_prop.bl_rna.properties:
        if is_property_default(prop, obj_prop):
            return True
    return False


def reset_property_modified(obj_prop):
    for prop in obj_prop.bl_rna.properties:
        if hasattr(prop, "default"):
            is_array = getattr(prop, "is_array", False)
            if is_array:
                cur_array = getattr(obj_prop, prop.identifier)

                default_values = [(i, p) for i, p in enumerate(prop.default_array)]
                current_values = [(i, p) for i, p in enumerate(cur_array)]
                if default_values != current_values:
                    if len(default_values) == len(current_values):
                        for i, p in default_values:
                            cur_array[i] = p
            elif getattr(prop, 'is_enum_flag', False):
                current = getattr(obj_prop, prop.identifier)
                if current != prop.default_flag:
                    setattr(obj_prop, prop.identifier, prop.default_flag)
            else:
                current = getattr(obj_prop, prop.identifier)
                if current != prop.default:
                    setattr(obj_prop, prop.identifier, prop.default)


class ZsOperatorAttrs:
    @classmethod
    def get_operator_attr(cls, op_name, attr_name, default=None):
        wm = bpy.context.window_manager
        op_last = wm.operator_properties_last(op_name)
        if op_last:
            return getattr(op_last, attr_name, default)
        return default

    @classmethod
    def set_operator_attr(cls, op_name, attr_name, value):
        wm = bpy.context.window_manager
        op_last = wm.operator_properties_last(op_name)
        if op_last:
            setattr(op_last, attr_name, value)

    @classmethod
    def get_operator_attr_int_enum(cls, op_name, attr_name, default, p_items):
        p_val = cls.get_operator_attr(op_name, attr_name, default)
        for i, item in enumerate(p_items):
            if item[0] == p_val:
                return i
        return 0

    @classmethod
    def set_operator_attr_int_enum(cls, op_name, attr_name, value, p_items):
        for i, item in enumerate(p_items):
            if i == value:
                cls.set_operator_attr(op_name, attr_name, item[0])


def get_mimic_bl_rna_layout(p_instance, p_bl_rna, layout, attr_name, factor=0.3, expand=False):
    row = layout.row(align=True)
    r = row.split(factor=factor, align=True)
    r.alignment = 'RIGHT'
    p_prop = p_bl_rna.properties[attr_name]
    if p_prop.type != 'BOOLEAN':
        r.label(text=p_prop.name)
        r.prop(p_instance, attr_name, text=None if expand else "", expand=expand)
    else:
        r.separator()
        r.prop(p_instance, attr_name)

    return r


def get_mimic_operator_layout(operator: bpy.types.Operator, layout, attr_name, factor=0.3, expand=False):
    return get_mimic_bl_rna_layout(
        operator, operator.properties.bl_rna, layout, attr_name, factor, expand)


def draw_ex_last_operator_properties(context: bpy.types.Context, op_id, layout: bpy.types.UILayout):
    wm = context.window_manager
    op_last = wm.operator_properties_last(op_id)
    if op_last:
        op = bpy.types.Operator.bl_rna_get_subclass_py(op_last.__class__.__name__)
        if op:
            op.draw_ex(op_last, layout, context)


def draw_last_operator_properties(
        context: bpy.types.Context, op_id, layout: bpy.types.UILayout,
        blacklist_props=set(), whitelist_props=set(),
        factor=0.3):
    wm = context.window_manager
    op_last = wm.operator_properties_last(op_id)
    if op_last:
        props = op_last.bl_rna.properties
        for k in props.keys():
            if len(whitelist_props) != 0 and k not in whitelist_props:
                continue

            if k not in blacklist_props and k != 'rna_type' and not props[k].is_hidden:
                get_mimic_bl_rna_layout(op_last, op_last.bl_rna, layout, k, factor=factor)


def ireplace(text, find, repl):
    return re.sub('(?i)' + re.escape(find), lambda m: repl, text)


def win_emulate_escape():
    try:
        if platform.system() == 'Windows':
            VK_ESCAPE = 0x1B
            ctypes.windll.user32.keybd_event(VK_ESCAPE)
    except Exception as e:
        Log.error('Can not emulate ESC!', e)


class CallerCmdProps:
    bl_op_cls = None
    bl_label = None
    bl_desc = None
    op_cmd_short = ''
    cmd = None


def get_command_props(cmd, context=bpy.context) -> CallerCmdProps:

    op_props = CallerCmdProps()

    if cmd:
        op_cmd = cmd
        op_args = '()'

        override_args = {}

        args = []

        cmd_split = cmd.split("(", 1)
        if len(cmd_split) == 2:
            op_cmd = cmd_split[0]
            r_part = cmd_split[1].rstrip(')')
            r_part_split = r_part.split(",")
            for part in r_part_split:
                if '=' in part:
                    pair = part.split("=", 1)
                    override_args[pair[0].strip()] = pair[1]
                else:
                    args.append(part)

        op_props.op_cmd_short = op_cmd.replace("bpy.ops.", "", 1)
        op_cmd = f"bpy.ops.{op_props.op_cmd_short}"

        try:
            wm = context.window_manager
            op_last = wm.operator_properties_last(op_props.op_cmd_short)
            if op_last:
                props = op_last.bl_rna.properties
                keys = set(props.keys()) - {'rna_type'}
                args += [
                    (k + '=' + (override_args[k] if k in override_args else repr(getattr(op_last, k))))
                    for k in dir(op_last)
                    if k in keys and (k in override_args or (not props[k].is_readonly and not props[k].is_skip_save))]

            if len(args):
                op_args = f'({",".join(args)})'

            op_props.bl_op_cls = eval(op_cmd)
            op_props.cmd = op_cmd + op_args

            try:
                rna = op_props.bl_op_cls.get_rna().rna_type \
                    if hasattr(op_props.bl_op_cls, "get_rna") \
                    else op_props.bl_op_cls.get_rna_type()

                op_props.bl_label = rna.name
                op_props.bl_desc = rna.description
            except Exception as ex:
                Log.warn('Description error:', ex)

        except Exception as ex:
            op_props = CallerCmdProps()
            Log.error('Eval error:', ex, 'cmd:', cmd)

    return op_props


def dump_context_objects(context):
    def _dump_object(idx, p_obj, s_category):
        print('\tCategory:', s_category, '[', idx, ']:', p_obj.name)

    categories = (
        'visible_objects',
        'selectable_objects',
        'selected_objects',
        'editable_objects',
        'selected_editable_objects',
        'objects_in_mode',
        'objects_in_mode_unique_data'
    )

    for s_cat in categories:
        print('===========> Category:', s_cat, '<=============')
        for i, obj in enumerate(getattr(context, s_cat, [])):
            _dump_object(i, obj, s_cat)


def vertex_weight_to_color(weight):
    weight_color = mathutils.Color()
    weight_color.hsv = (2/3 * weight, 1, 1)
    return mathutils.Color((weight_color.b, weight_color.g, weight_color.r))


def color_to_vertex_weight(color):
    color = mathutils.Color(color)
    weight_color = mathutils.Color((color.b, color.g, color.r))
    return np.clip(weight_color.h / (2/3), 0, 1)


class ZsDrawConstans:
    DEFAULT_VERT_ACTIVE_ALPHA = 80
    DEFAULT_VERT_INACTIVE_ALPHA = 60
    DEFAULT_VERT_ACTIVE_POINT_SIZE = 10
    DEFAULT_VERT_INACTIVE_POINT_SIZE = 6
    DEFAULT_VERT_USE_ZOOM_FACTOR = False

    DEFAULT_EDGE_ACTIVE_ALPHA = 60
    DEFAULT_EDGE_INACTIVE_ALPHA = 40
    DEFAULT_EDGE_ACTIVE_LINE_WIDTH = 3
    DEFAULT_EDGE_INACTIVE_LINE_WIDTH = 2

    DEFAULT_FACE_ACTIVE_ALPHA = 60
    DEFAULT_FACE_INACTIVE_ALPHA = 40

    DEFAULT_OBJECT_ACTIVE_ALPHA = 40
    DEFAULT_OBJECT_INACTIVE_ALPHA = 20
    DEFAULT_OBJECT_COLLECTION_BOUNDBOX_WIDTH = 2
    DEFAULT_OBJECT_COLLECTION_LABEL_SIZE = 12

    BGL_ACTIVE_FONT_COLOR = (0.157, 0.643, 0.408)
    BGL_INACTIVE_FONT_COLOR = (0.8, 0.8, 0.8)


class ZsUiConstants:
    ZSTS_PANEL_CATEGORY = "Zen Sets"
    ZSTS_REGION_TYPE = "UI"
    ZSTS_CONTEXT = "mesh_edit"
    ZSTS_SPACE_TYPE = "VIEW_3D"

    ZSTS_SYNC_MODE_ICON = 'LINKED'
