# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2022, Alex Zhornyak

import re
import sys

from functools import cmp_to_key


def make_sort_parts(s, mode):
    t_digits = []
    t_texts = []
    if s:
        i_int_pos = 0
        for c in re.split('([0-9]+)', s):
            if c:
                if c.isdigit():
                    if mode == 'HUMAN_LAST_INTEGER':
                        t_digits.insert(0, c)
                    elif mode == 'HUMAN_FIRST_INTEGER':
                        t_digits.insert(i_int_pos, c)
                        i_int_pos += 1
                    else:
                        t_texts.append(c)
                else:
                    t_texts.append(c)
    return t_digits + t_texts


def alphanum_key(s, prop_name, case_insensitive, suffix, prefix, mode):
    if prop_name:
        s = getattr(s, prop_name)

    if case_insensitive:
        s = s.lower()

    s_suffix = ''
    s_prefix = ''

    if suffix:
        t_parts = s.split(suffix)
        if len(t_parts) > 1:
            s_suffix = t_parts[-1]
            s = ''.join(t_parts[:-1])

    if prefix:
        t_parts = s.split(prefix)
        if len(t_parts) > 1:
            s_prefix = t_parts[0]
            s = ''.join(t_parts[1:])

    if mode == 'ALPHA_SORT':
        return (s_suffix, s_prefix, s)
    else:
        return (make_sort_parts(s_suffix, mode), make_sort_parts(s_prefix, mode), make_sort_parts(s, mode))


def cmp(a, b):
    return (a > b) - (a < b)


def cmp_tables(t_a, t_b, b_alpha):
    n_a_size = len(t_a)
    n_b_size = len(t_b)

    res = 0

    if n_a_size != 0 or n_a_size != n_b_size:
        for idx, k1 in enumerate(t_a):
            if idx in range(n_b_size):
                k2 = t_b[idx]
                b_k1_digit = k1.isdigit()
                b_k2_digit = k2.isdigit()
                if not b_alpha and (b_k1_digit or b_k2_digit):
                    i_1 = int(k1) if b_k1_digit else sys.maxsize
                    i_2 = int(k2) if b_k2_digit else sys.maxsize
                    res = cmp(i_1, i_2)
                else:
                    res = cmp(k1, k2)
                if res != 0:
                    break
            else:
                res = -1
                break

        if res == 0 and n_a_size < n_b_size:
            res = 1

    return res


def human_sort_copy(p_list, prop_name, case_insensitive=True, suffix='', prefix='', mode='HUMAN'):
    """
    Sort a list in the way that humans expect.
    """

    def cmp_items(a, b, prop_name, case_insensitive, suffix, prefix):
        t_a_group = alphanum_key(a, prop_name, case_insensitive, suffix, prefix, mode)
        t_b_group = alphanum_key(b, prop_name, case_insensitive, suffix, prefix, mode)

        res = 0

        for i, _ in enumerate(t_a_group):
            res = cmp_tables(t_a_group[i], t_b_group[i], mode == 'ALPHA_SORT')
            if res != 0:
                break

        # deep debug:
        # print('A1:', t_a_group, 'B1:', t_b_group, '--->', res)

        return res

    p_cmp_items = cmp_to_key(lambda a, b: cmp_items(a, b, prop_name, case_insensitive, suffix, prefix))

    return sorted(p_list, key=p_cmp_items)
