# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# <pep8 compliant>

# Partially taken from Oleg Stepanov (DotBow)
# Copyright 2022, Alex Zhornyak

import bpy
import blf
import rna_keymap_ui

from .keymap import _keymap, _KEY_READABLE_MAP
from .labels import ZsLabels
from .blender_zen_utils import ZenLocks, ZenPolls, ZsDrawConstans

from .bl_ui_widgets.bl_ui_button import BL_UI_Button
from .bl_ui_widgets.bl_ui_label import BL_UI_Label
from .bl_ui_widgets.bl_ui_drag_panel import BL_UI_Drag_Panel
from .bl_ui_widgets.bl_ui_draw_op import BL_UI_OT_draw_operator

addon_keymap = []


def get_hotkey_entry_item(km, kmi_name, kmi_value, handled_kmi):
    for km_item in km.keymap_items:
        if km_item in handled_kmi:
            continue
        if km_item.idname == kmi_name:
            if kmi_value is None:
                return km_item
            elif ('name' in km_item.properties) and (km_item.properties.name == kmi_value):
                return km_item

    return None


def draw_keymaps(context, layout):
    wm = context.window_manager
    kc = wm.keyconfigs.user
    box = layout.box()
    split = box.split()
    col = split.column()
    col.label(text='Setup Keymap')

    km_tree = {}
    for _km, _kmi in addon_keymap:
        if _km.name not in km_tree:
            km_tree[_km.name] = []

        km_tree[_km.name].append((_kmi.idname, _kmi.properties.name if 'name' in _kmi.properties else None))

    """ Special case for Tool keymap """
    for _km in kc.keymaps:
        if 'Zen Sets' in _km.name:
            if _km.name not in km_tree:
                km_tree[_km.name] = []

            for _kmi in _km.keymap_items:
                km_tree[_km.name].append(_kmi)

    handled_kmi = set()
    for km_name, kmi_items in km_tree.items():
        km = kc.keymaps.get(km_name)
        if km:
            col.context_pointer_set("keymap", km)
            col.separator()
            row = col.row(align=True)
            row.label(text=km_name)

            if km.is_user_modified:
                subrow = row.row()
                subrow.alignment = 'RIGHT'
                subrow.operator("preferences.keymap_restore", text="Restore")

            for kmi_node in kmi_items:
                kmi = get_hotkey_entry_item(km, kmi_node[0], kmi_node[1], handled_kmi) if isinstance(kmi_node, tuple) else kmi_node
                col.separator()

                if kmi:
                    handled_kmi.add(kmi)
                    rna_keymap_ui.draw_kmi([], kc, km, kmi, col, 0)
                else:
                    row = col.row(align=True)
                    row.separator(factor=2.0)
                    row.label(text=f"Keymap item for '{kmi_node[0]} ' in '{km_name}' not found",
                              icon='ERROR')


def draw_keymaps_tool(context, layout):
    wm = context.window_manager
    kc = wm.keyconfigs.user

    km_tree = {}

    """ Special case for Tool keymap """
    for _km in kc.keymaps:
        if 'Zen Sets' in _km.name:
            if _km.name not in km_tree:
                km_tree[_km.name] = []

            for _kmi in _km.keymap_items:
                km_tree[_km.name].append(_kmi)

    s_key = ''
    if context.area.type == 'IMAGE_EDITOR':
        s_key = "image editor"
    else:
        if context.mode == 'OBJECT':
            s_key = "object"
        elif context.mode == 'EDIT_MESH':
            s_key = "edit mesh"

    handled_kmi = set()
    for km_name, kmi_items in km_tree.items():
        km = kc.keymaps.get(km_name)
        if km:
            if s_key not in km_name.lower():
                continue

            layout.label(text=km_name)
            box = layout.box()
            col = box.column()

            col.context_pointer_set("keymap", km)

            for kmi_node in kmi_items:
                kmi = get_hotkey_entry_item(km, kmi_node[0], kmi_node[1], handled_kmi) if isinstance(kmi_node, tuple) else kmi_node

                if kmi:
                    handled_kmi.add(kmi)

                    if kmi.active:
                        key_text = kmi.name

                        if key_text == ZsLabels.OT_ASSIGN_ITEM_PINNED_LABEL:
                            from .sets.factories import get_sets_mgr
                            p_cls_mgr = get_sets_mgr(context.scene)
                            if not p_cls_mgr or p_cls_mgr.is_unique:
                                continue

                        col.separator()

                        text = []
                        if kmi.shift:
                            text.append('Shift')
                        if kmi.ctrl:
                            text.append('Ctrl')
                        if kmi.alt:
                            text.append('Alt')

                        text.append(_KEY_READABLE_MAP.get(kmi.type, kmi.type))

                        row1 = col.row(align=True)

                        keep_active_group = getattr(kmi.properties, 'keep_active_group', None)
                        if keep_active_group is not None:
                            if keep_active_group:
                                key_text += ' [Add]'

                        mode = getattr(kmi.properties, 'mode', None)
                        if mode is not None:
                            if mode == 'ADD':
                                key_text += ' [Add]'
                            elif mode == 'SUB':
                                key_text += ' [Sub]'

                        row1.label(text=key_text)

                        row1.label(text='+'.join(text))


class ZSTS_OT_ToolHelpDrawOperator(BL_UI_OT_draw_operator):

    bl_idname = "zsts.draw_show_tool_help"
    bl_label = "Zen Sets Tool Help"
    bl_description = "Help for Zen Sets Workspace Tool"
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return ZenPolls.poll_object_edit_or_uv(context)

    def tool_poll(self, context: bpy.types.Context):
        # hide in other tools
        if context.space_data == self.space_data:
            return ZenPolls.poll_zen_tool_in_workspace(context)
        return False

    def draw_callback_px(self, op, context: bpy.types.Context):
        if self.tool_poll(context):
            super().draw_callback_px(op, context)

    def handle_widget_events(self, context: bpy.types.Context, event: bpy.types.Event):
        if self.tool_poll(context):
            return super().handle_widget_events(context, event)
        return False

    def modal(self, context: bpy.types.Context, event: bpy.types.Event):
        if not ZenLocks.is_draw_tool_help_flag(self.space_data.type):
            self.cancel(context)

        return super().modal(context, event)

    def invoke(self, context, event):
        space_type = context.space_data.type
        if ZenLocks.is_draw_tool_help_flag(space_type):
            ZenLocks.set_draw_tool_help_flag(space_type, False)
            return {'CANCELLED'}

        ZenLocks.set_draw_tool_help_flag(space_type, True)
        res = super().invoke(context, event)

        return res

    def on_finish(self, context):
        ZenLocks.set_draw_tool_help_flag(self.space_data.type, False)
        super().on_finish(context)

    def on_area_resized(self, context):
        self.panel.update(self.panel.x_screen, self.panel.y_screen)
        self.panel.layout_widgets()

        super().on_area_resized(context)

    def create_widgets(self, context):
        from .preferences import get_prefs
        addon_prefs = get_prefs()

        widgets = []
        widgets_panel = []
        column_1 = []
        column_2 = []

        self.panel = BL_UI_Drag_Panel(100, 300, 360, 290)
        self.panel.bg_color = (0.2, 0.2, 0.2, 0.9)
        widgets.append(self.panel)
        ui_scale = context.preferences.view.ui_scale

        i_RAW_FONT_HEIGHT = addon_prefs.label_2d.tool_help_font_size
        i_FONT_DPI = round(72 * ui_scale)
        i_ROW_HEIGHT = round(i_RAW_FONT_HEIGHT * 2.0 * ui_scale)

        i_TEXT_OFFSET = 10 * ui_scale

        y_pos = 0

        wm = context.window_manager
        kc = wm.keyconfigs.user

        km_tree = {}

        max_col_1_w = 0
        max_col_2_w = 0

        """ Special case for Tool keymap """
        for _km in kc.keymaps:
            if 'Zen Sets' in _km.name:
                if _km.name not in km_tree:
                    km_tree[_km.name] = []

                for _kmi in _km.keymap_items:
                    km_tree[_km.name].append(_kmi)

        s_key = ''
        if context.area.type == 'IMAGE_EDITOR':
            s_key = "image editor"
        else:
            if context.mode == 'OBJECT':
                s_key = "object"
            elif context.mode == 'EDIT_MESH':
                s_key = "edit mesh"

        handled_kmi = set()
        for km_name, kmi_items in km_tree.items():
            km = kc.keymaps.get(km_name)
            if km:
                if s_key not in km_name.lower():
                    continue

                for kmi_node in kmi_items:
                    kmi = get_hotkey_entry_item(km, kmi_node[0], kmi_node[1], handled_kmi) if isinstance(kmi_node, tuple) else kmi_node

                    if kmi:
                        handled_kmi.add(kmi)

                        if kmi.active:
                            key_text = kmi.name

                            if key_text == ZsLabels.OT_ASSIGN_ITEM_PINNED_LABEL:
                                from .sets.factories import get_sets_mgr
                                p_cls_mgr = get_sets_mgr(context.scene)
                                if not p_cls_mgr or p_cls_mgr.is_unique:
                                    continue

                            text = []
                            if kmi.shift:
                                text.append('Shift')
                            if kmi.ctrl:
                                text.append('Ctrl')
                            if kmi.alt:
                                text.append('Alt')

                            text.append(_KEY_READABLE_MAP.get(kmi.type, kmi.type))

                            keep_active_group = getattr(kmi.properties, 'keep_active_group', None)
                            if keep_active_group is not None:
                                if keep_active_group:
                                    key_text += ' [Add]'

                            mode = getattr(kmi.properties, 'mode', None)
                            if mode is not None:
                                if mode == 'ADD':
                                    key_text += ' [Add]'
                                elif mode == 'SUB':
                                    key_text += ' [Sub]'

                            p_ui_label = BL_UI_Label(i_TEXT_OFFSET, y_pos, 170, i_ROW_HEIGHT)
                            p_ui_label.active_alpha = 1.0
                            p_ui_label.text = '+'.join(text)
                            p_ui_label.text_size = i_RAW_FONT_HEIGHT
                            p_ui_label.text_scale = ui_scale
                            p_ui_label.text_color = ZsDrawConstans.BGL_ACTIVE_FONT_COLOR + (1,)
                            column_1.append(p_ui_label)
                            blf.size(0, i_RAW_FONT_HEIGHT, i_FONT_DPI)
                            lbl_w, _ = blf.dimensions(0, p_ui_label.text)

                            p_ui_label_2 = BL_UI_Label(i_TEXT_OFFSET + 170, y_pos, 70, i_ROW_HEIGHT)
                            p_ui_label_2.text = key_text
                            p_ui_label_2.text_size = i_RAW_FONT_HEIGHT
                            p_ui_label_2.text_scale = ui_scale
                            p_ui_label_2.text_color = ZsDrawConstans.BGL_INACTIVE_FONT_COLOR + (1,)
                            column_2.append(p_ui_label_2)
                            lbl_2_w, _ = blf.dimensions(0, p_ui_label_2.text)

                            max_col_1_w = max(max_col_1_w, lbl_w)
                            max_col_2_w = max(max_col_2_w, lbl_2_w)

                            y_pos += i_ROW_HEIGHT

        y_pos += 10 * ui_scale

        i_BTN_WIDTH = 120 * ui_scale
        button1 = BL_UI_Button(i_TEXT_OFFSET, y_pos, i_BTN_WIDTH, i_ROW_HEIGHT)
        button1.bg_color = (0.4, 0.4, 0.4, 0.4)
        button1.hover_bg_color = (0.6, 0.6, 0.6, 1.0)
        button1.text = "Close"
        button1.text_scale = ui_scale
        button1.text_size = i_RAW_FONT_HEIGHT + 2
        button1.set_mouse_down(self.button1_press)

        widgets_panel.extend(column_1)
        widgets_panel.extend(column_2)
        widgets_panel.append(button1)

        y_pos += i_ROW_HEIGHT + 10 * ui_scale

        for widget in column_1:
            widget.width = max_col_1_w

        for widget in column_2:
            widget.x = i_TEXT_OFFSET + max_col_1_w + i_TEXT_OFFSET
            widget.width = max_col_2_w

        self.panel.width = max_col_1_w + max_col_2_w + i_TEXT_OFFSET * 3
        self.panel.height = y_pos

        widgets += widgets_panel

        self.init_widgets(context, widgets)

        self.panel.add_widgets(widgets_panel)

    def on_invoke(self, context: bpy.types.Context, event: bpy.types.Event):

        self.create_widgets(context)

        # Open the panel at the right bottom corner location
        i_area_height = context.region.height
        i_top = i_area_height - self.widgets[0].height - 10

        ui_scale = context.preferences.view.ui_scale

        widget_size = self.widgets[0].width

        n_panel_width = [region.width for region in context.area.regions if region.type == "UI"][0]
        i_left = context.region.width - n_panel_width - widget_size - 10 * ui_scale

        self.widgets[0].set_location(i_left, i_top)

    # Button press handlers
    def button1_press(self, widget):
        self.cancel(bpy.context)


def register():
    wm = bpy.context.window_manager
    kc = wm.keyconfigs.addon

    if kc:
        addon_keymap.extend(_keymap())

    bpy.utils.register_class(ZSTS_OT_ToolHelpDrawOperator)


def unregister():

    ZenLocks.clear_all_draw_tool_help_flags()
    ZSTS_OT_ToolHelpDrawOperator.unregister_all_handlers()

    bpy.utils.unregister_class(ZSTS_OT_ToolHelpDrawOperator)

    for km, kmi in addon_keymap:
        km.keymap_items.remove(kmi)

    addon_keymap.clear()
