# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

""" Zen Sets Addon Properties module """
import bpy
import addon_utils

import functools

from .vlog import Log
from .labels import ZsLabels
from .keymap_manager import draw_keymaps
from .ico import zs_icon_get, ZIconsType
from .blender_zen_utils import (
    ZsDrawConstans,
    ZsOperatorAttrs,
    ZenPolls,
    update_areas_in_all_screens,
    update_imageeditor_in_all_screens,
    reset_property_modified, is_property_modified,
    get_mimic_bl_rna_layout)


_CACHE_ADDON_VERSION = None


def get_addon_version():
    global _CACHE_ADDON_VERSION
    if _CACHE_ADDON_VERSION is None:
        for addon in addon_utils.modules():
            if addon.bl_info['name'] == 'Zen Sets':
                ver = addon.bl_info['version']
                _CACHE_ADDON_VERSION = '%i.%i.%i' % (ver[0], ver[1], ver[2])
                break

    return _CACHE_ADDON_VERSION if _CACHE_ADDON_VERSION else '0.0.0'


def rsetattr(obj, attr, val):
    pre, _, post = attr.rpartition('.')
    return setattr(rgetattr(obj, pre) if pre else obj, post, val)


def rgetattr(obj, attr, *args):
    def _getattr(obj, attr):
        return getattr(obj, attr, *args)
    return functools.reduce(_getattr, [obj] + attr.split('.'))


def get_annotated_prop(p_instance, p_prop_name):
    p_prop = p_instance.__annotations__[p_prop_name]
    return p_prop.keywords if bpy.app.version >= (2, 93, 0) else p_prop[1]


def get_instanced_int_enum_property(p_instance, p_path_prefix, p_prop_name):
    p_prop = get_annotated_prop(p_instance, p_prop_name)

    return bpy.props.EnumProperty(
        name=p_prop.get('name', ''),
        description=p_prop.get('description', ''),
        items=p_prop.get('items', []),
        get=lambda self: int(rgetattr(get_prefs(), p_path_prefix + '.' + p_prop_name)),
        set=lambda self, value: rsetattr(get_prefs(), p_path_prefix + '.' + p_prop_name, str(value))
    )


def get_instanced_bool_property(p_instance, p_path_prefix, p_prop_name):
    p_prop = get_annotated_prop(p_instance, p_prop_name)

    return bpy.props.BoolProperty(
        name=p_prop.get('name', ''),
        description=p_prop.get('description', ''),
        get=lambda self: rgetattr(get_prefs(), p_path_prefix + '.' + p_prop_name),
        set=lambda self, value: rsetattr(get_prefs(), p_path_prefix + '.' + p_prop_name, value)
    )


t_ZEN_SETS_ELEMENT_LINKS = {
    'vert': 'blgroup',
    'blgroup': 'vert',

    'face_u': 'blgroup_u',
    'blgroup_u': 'face_u'
}


def is_element_mode_enabled(addon_prefs, element_mode):
    mode_enabled = getattr(addon_prefs.modes, f'enable_{element_mode}')
    if not mode_enabled:
        p_linked = t_ZEN_SETS_ELEMENT_LINKS.get(element_mode, '')
        if p_linked:
            mode_enabled = getattr(addon_prefs.modes, f'enable_{p_linked}')

    return mode_enabled


def display_element_modes(element_layout, blender_layout, unique_layout, context, display_mode_labels=False):
    addon_prefs = get_prefs()

    p_scene = context.scene

    element_mode = p_scene.zen_sets_active_mode.replace('_u', '')

    if display_mode_labels:
        element_layout.label(text='Element Mode:')

    for item in p_scene.bl_rna.properties['zen_sets_element_mode'].enum_items:
        if item.identifier not in {'blgroup', 'blgroup_u'}:
            sets_enabled = getattr(addon_prefs.modes, f'enable_{item.identifier}')
            parts_enabled = getattr(addon_prefs.modes, f'enable_{item.identifier}_u')
            if (element_mode == item.identifier) or sets_enabled or parts_enabled:
                element_layout.prop_enum(p_scene, "zen_sets_element_mode", item.identifier)

    if display_mode_labels:
        unique_layout.label(text='Group Mode')

    if blender_layout:
        if addon_prefs.modes.enable_blgroup or p_scene.zen_sets_active_mode == 'blgroup':
            blender_layout.prop_enum(p_scene, 'zen_sets_active_mode', 'blgroup', text='Vert Groups')
        if addon_prefs.modes.enable_blgroup_u or p_scene.zen_sets_active_mode == 'blgroup_u':
            blender_layout.prop_enum(p_scene, 'zen_sets_active_mode', 'blgroup_u', text='Face Maps')

    sets_enabled = is_element_mode_enabled(addon_prefs, element_mode)
    parts_enabled = is_element_mode_enabled(addon_prefs, element_mode + '_u')

    if sets_enabled or (element_mode == p_scene.zen_sets_active_mode):
        unique_layout.prop_enum(p_scene, "zen_sets_unique_mode", 'SETS')
    if parts_enabled or (element_mode + '_u' == p_scene.zen_sets_active_mode):
        unique_layout.prop_enum(p_scene, "zen_sets_unique_mode", 'PARTS')


def display_object_element_modes(element_layout, context, display_mode_labels=False):
    addon_prefs = get_prefs()
    element_mode = context.scene.zen_object_collections_mode

    modes = []

    if addon_prefs.modes.enable_obj_simple_sets or element_mode == 'obj_simple_sets':
        modes.append('obj_simple_sets')
    if addon_prefs.modes.enable_obj_simple_parts or element_mode == 'obj_simple_parts':
        modes.append('obj_simple_parts')

    if len(modes) > 0:
        if display_mode_labels:
            element_layout.label(text='Element Mode:')
        # Collection mode mustn't be removed!
        element_layout.prop_enum(context.scene, "zen_object_collections_mode", 'obj_sets')

        for mode in modes:
            element_layout.prop_enum(context.scene, "zen_object_collections_mode", mode)


def display_compact_element_modes(layout, context, ids):
    addon_prefs = get_prefs()
    prefix = ''
    for k, v in enumerate(ids):
        new_prefix = v[:3]
        mode_enabled = getattr(addon_prefs.modes, f'enable_{v}')
        if mode_enabled or context.scene.zen_sets_active_mode == v:
            if k != 0 and new_prefix != prefix:
                layout.separator()

            layout.prop_enum(context.scene, "zen_sets_active_mode", v, text="")
            prefix = new_prefix


ALIGNMENT_2D_ITEMS = (
    ('TOP_LEFT', "Top Left", ""),
    ('TOP_RIGHT', "Top Right", ""),
    ('BOTTOM_LEFT', "Bottom Left", ""),
    ('BOTTOM_RIGHT', "Bottom Right", ""),
)


class ZSTS_OT_Keymaps(bpy.types.Operator):
    bl_idname = "zsts.show_keymaps"
    bl_label = "Keymap"
    bl_options = {'REGISTER'}
    bl_description = "Set Shortcuts for Zen Sets Menus"

    def execute(self, context):

        addon_utils.modules_refresh()

        try:
            mod = addon_utils.addons_fake_modules.get("ZenSets")
            info = addon_utils.module_bl_info(mod)
            info["show_expanded"] = True  # or False to Collapse
        except Exception as e:
            Log.error(e)

        addon_prefs = get_prefs()
        addon_prefs.tabs = 'KEYMAP'

        context.preferences.active_section = "ADDONS"
        bpy.ops.screen.userpref_show("INVOKE_DEFAULT")
        bpy.data.window_managers['WinMan'].addon_search = "Zen Sets"

        return {'FINISHED'}


class ZsDisplay2DSettings(bpy.types.PropertyGroup):
    size: bpy.props.IntProperty(
        name="Font Size", description="Overlay Label Font size",
        default=11, min=8, max=50)
    color: bpy.props.FloatVectorProperty(
        name="Color", description="Overlay Label Color",
        default=(1, 1, 1, 1), subtype='COLOR', size=4, min=0, max=1)
    alignment: bpy.props.EnumProperty(
        name="Alignment",
        description="Overlay Label Alignment",
        items=ALIGNMENT_2D_ITEMS,
        default='TOP_LEFT')
    offset_x: bpy.props.IntProperty(
        name="Offset X", description="Offset from area edges",
        subtype='PERCENTAGE', default=50, min=0, max=100)
    offset_y: bpy.props.IntProperty(
        name="Offset Y", description="Offset from area edges",
        subtype='PERCENTAGE', default=0, min=0, max=100)

    tool_help_font_size: bpy.props.IntProperty(
        name="Font Size", description="Zen Sets Tool Help Font size",
        default=11, min=6, max=30)

    def draw(self, layout):
        row = layout.row(align=False)
        row.prop(self, "color", text="")
        row.prop(self, "size")

        row = layout.row(align=False)
        row.prop(self, "alignment", text="")
        row.prop(self, "offset_x")
        row.prop(self, "offset_y")

    def draw_tool_options(self, layout):
        row = layout.row(align=False)
        row.label(text='Tool Help Panel')
        row.prop(self, 'tool_help_font_size')


class PropDisplay3D:
    @classmethod
    def is_modified(cls, options, id_element):
        return getattr(options, f'{id_element}_active_alpha') != \
            getattr(ZsDrawConstans, f'DEFAULT_{id_element.upper()}_ACTIVE_ALPHA') or \
            getattr(options, f'{id_element}_inactive_alpha') != \
            getattr(ZsDrawConstans, f'DEFAULT_{id_element.upper()}_INACTIVE_ALPHA')

    @classmethod
    def draw_element_settings(cls, options, layout, id_element):
        row = layout.row()
        row.label(text=f'{id_element.capitalize()} display settings')
        if cls.is_modified(options, id_element):
            subrow = row.row()
            subrow.alignment = 'RIGHT'
            op = subrow.operator("zsts.restore_preference", text="Restore")
            op.mode = f'3D_{id_element}'.upper()
        row = layout.row()
        row.prop(options, f'{id_element}_active_alpha')
        row.prop(options, f'{id_element}_inactive_alpha')

    @classmethod
    def restore(cls, options, id_element):
        f_alpha_1 = getattr(ZsDrawConstans, f'DEFAULT_{id_element.upper()}_ACTIVE_ALPHA')
        setattr(options, f'{id_element}_active_alpha', f_alpha_1)
        f_alpha_2 = getattr(ZsDrawConstans, f'DEFAULT_{id_element.upper()}_INACTIVE_ALPHA')
        setattr(options, f'{id_element}_inactive_alpha', f_alpha_2)


class VertPropDisplay3D(PropDisplay3D):
    @classmethod
    def is_modified(cls, options, id_element):
        modified = super().is_modified(options, id_element)
        if modified:
            return True

        return options.vert_active_point_size != ZsDrawConstans.DEFAULT_VERT_ACTIVE_POINT_SIZE or \
            options.vert_inactive_point_size != ZsDrawConstans.DEFAULT_VERT_INACTIVE_POINT_SIZE or \
            options.vert_use_zoom_factor != ZsDrawConstans.DEFAULT_VERT_USE_ZOOM_FACTOR

    @classmethod
    def draw_element_settings(cls, options, layout, id_element):
        super().draw_element_settings(options, layout, id_element)

        row = layout.row()
        row.prop(options, 'vert_active_point_size')
        row.prop(options, 'vert_inactive_point_size')
        row = layout.row()
        row.prop(options, 'vert_use_zoom_factor')

    @classmethod
    def restore(cls, options, id_element):
        super().restore(options, id_element)
        options.vert_active_point_size = ZsDrawConstans.DEFAULT_VERT_ACTIVE_POINT_SIZE
        options.vert_inactive_point_size = ZsDrawConstans.DEFAULT_VERT_INACTIVE_POINT_SIZE
        options.vert_use_zoom_factor = ZsDrawConstans.DEFAULT_VERT_USE_ZOOM_FACTOR


class EdgePropDisplay3D(PropDisplay3D):
    @classmethod
    def is_modified(cls, options, id_element):
        modified = super().is_modified(options, id_element)
        if modified:
            return True

        return options.edge_active_line_width != ZsDrawConstans.DEFAULT_EDGE_ACTIVE_LINE_WIDTH or \
            options.edge_inactive_line_width != ZsDrawConstans.DEFAULT_EDGE_INACTIVE_LINE_WIDTH or \
            options.edge_z_fight_compensation is False

    @classmethod
    def draw_element_settings(cls, options, layout, id_element):
        super().draw_element_settings(options, layout, id_element)

        row = layout.row()
        row.prop(options, 'edge_active_line_width')
        row.prop(options, 'edge_inactive_line_width')

        layout.prop(options, 'edge_z_fight_compensation')

    @classmethod
    def restore(cls, options, id_element):
        super().restore(options, id_element)
        options.edge_active_line_width = ZsDrawConstans.DEFAULT_EDGE_ACTIVE_LINE_WIDTH
        options.edge_inactive_line_width = ZsDrawConstans.DEFAULT_EDGE_INACTIVE_LINE_WIDTH


class ObjectPropDisplay3D(PropDisplay3D):
    @classmethod
    def is_modified(cls, options, id_element):
        modified = super().is_modified(options, id_element)
        if modified:
            return True

        return (options.object_collection_line_width != ZsDrawConstans.DEFAULT_OBJECT_COLLECTION_BOUNDBOX_WIDTH or
                options.object_collection_label_size != ZsDrawConstans.DEFAULT_OBJECT_COLLECTION_LABEL_SIZE or
                not options.object_collection_label_background_enabled)

    @classmethod
    def draw_element_settings(cls, options, layout, id_element):
        super().draw_element_settings(options, layout, id_element)

        layout.label(text='Collection Bound Box')
        row = layout.row()
        row.prop(options, 'object_collection_line_width')
        row.prop(options, 'object_collection_label_size')
        row.prop(options, 'object_collection_label_background_enabled')

    @classmethod
    def restore(cls, options, id_element):
        super().restore(options, id_element)
        options.object_collection_line_width = ZsDrawConstans.DEFAULT_OBJECT_COLLECTION_BOUNDBOX_WIDTH
        options.object_collection_label_size = ZsDrawConstans.DEFAULT_OBJECT_COLLECTION_LABEL_SIZE
        options.object_collection_label_background_enabled = True


_DISPLAY_3D_PROPS = {'vert': VertPropDisplay3D,
                     'edge': EdgePropDisplay3D,
                     'face': PropDisplay3D,
                     'object': ObjectPropDisplay3D}


class ZsDisplay3DSettings(bpy.types.PropertyGroup):
    vert_active_point_size: bpy.props.IntProperty(
        name="Active Group Vertex Point Size",
        subtype='PIXEL', default=ZsDrawConstans.DEFAULT_VERT_ACTIVE_POINT_SIZE, min=4, max=40)
    vert_inactive_point_size: bpy.props.IntProperty(
        name="Inactive Group Vertex Point Size",
        subtype='PIXEL', default=ZsDrawConstans.DEFAULT_VERT_INACTIVE_POINT_SIZE, min=2, max=30)

    vert_active_alpha: bpy.props.IntProperty(
        name="Active Group Opacity", description="Percent of alpha in Active Group Color",
        subtype='PERCENTAGE', default=ZsDrawConstans.DEFAULT_VERT_ACTIVE_ALPHA, min=40, max=100)
    vert_inactive_alpha: bpy.props.IntProperty(
        name="Inactive Group Opacity", description="Percent of alpha in Inactive Group Color",
        subtype='PERCENTAGE', default=ZsDrawConstans.DEFAULT_VERT_INACTIVE_ALPHA, min=10, max=100)

    vert_use_zoom_factor: bpy.props.BoolProperty(
        name="Size in UV by Zoom", description="Vertex size depends on Zoom Factor in UV Editor",
        default=ZsDrawConstans.DEFAULT_VERT_USE_ZOOM_FACTOR
    )

    edge_active_line_width: bpy.props.IntProperty(
        name="Active Group Edge Line Width",
        subtype='PIXEL', default=ZsDrawConstans.DEFAULT_EDGE_ACTIVE_LINE_WIDTH, min=2, max=20)
    edge_inactive_line_width: bpy.props.IntProperty(
        name="Inactive Group Edge Line Width",
        subtype='PIXEL', default=ZsDrawConstans.DEFAULT_EDGE_INACTIVE_LINE_WIDTH, min=1, max=10)
    edge_z_fight_compensation: bpy.props.BoolProperty(
        name='Compensate Z-Fight',
        default=True
    )

    edge_active_alpha: bpy.props.IntProperty(
        name="Active Group Opacity", description="Percent of alpha in Active Group Color",
        subtype='PERCENTAGE', default=ZsDrawConstans.DEFAULT_EDGE_ACTIVE_ALPHA, min=40, max=100)
    edge_inactive_alpha: bpy.props.IntProperty(
        name="Inactive Group Opacity", description="Percent of alpha in Inactive Group Color",
        subtype='PERCENTAGE', default=ZsDrawConstans.DEFAULT_EDGE_INACTIVE_ALPHA, min=10, max=100)

    face_active_alpha: bpy.props.IntProperty(
        name="Active Group Opacity", description="Percent of alpha in Active Group Color",
        subtype='PERCENTAGE', default=ZsDrawConstans.DEFAULT_FACE_ACTIVE_ALPHA, min=40, max=100)
    face_inactive_alpha: bpy.props.IntProperty(
        name="Inactive Group Opacity", description="Percent of alpha in Inactive Group Color",
        subtype='PERCENTAGE', default=ZsDrawConstans.DEFAULT_FACE_INACTIVE_ALPHA, min=10, max=100)

    object_active_alpha: bpy.props.IntProperty(
        name="Active Group Opacity", description="Percent of alpha in Active Group Color",
        subtype='PERCENTAGE', default=ZsDrawConstans.DEFAULT_OBJECT_ACTIVE_ALPHA, min=40, max=100)
    object_inactive_alpha: bpy.props.IntProperty(
        name="Inactive Group Opacity", description="Percent of alpha in Inactive Group Color",
        subtype='PERCENTAGE', default=ZsDrawConstans.DEFAULT_OBJECT_INACTIVE_ALPHA, min=10, max=100)

    object_collection_line_width: bpy.props.IntProperty(
        name="Line Width",
        subtype='PIXEL', default=ZsDrawConstans.DEFAULT_OBJECT_COLLECTION_BOUNDBOX_WIDTH, min=1, max=10)
    object_collection_label_size: bpy.props.IntProperty(
        name="Font Size",
        subtype='PIXEL', default=ZsDrawConstans.DEFAULT_OBJECT_COLLECTION_LABEL_SIZE, min=6, max=40)
    object_collection_label_background_enabled: bpy.props.BoolProperty(name='Draw Background', default=True)

    def draw(self, layout):
        for elem, prop in _DISPLAY_3D_PROPS.items():
            box = layout.box()
            prop.draw_element_settings(self, box, elem)


class ZsShowPanelSettings(bpy.types.PropertyGroup):
    show_import_export: bpy.props.BoolProperty(
        name="Import-Export",
        description="Show Import-Export Panel",
        default=True)

    show_tools: bpy.props.BoolProperty(
        name="Tools",
        description="Show Tools Panel",
        default=True)

    show_preferences: bpy.props.BoolProperty(
        name="Preferences",
        description="Show Preferences Panel",
        default=True)

    show_help: bpy.props.BoolProperty(
        name="Help",
        description="Show Help Panel",
        default=True)

    show_overlay: bpy.props.BoolProperty(
        name="Overlay",
        description="Show Overlay Panel",
        default=True)

    def draw(self, layout):
        box = layout.box()

        for key in self.__annotations__.keys():
            box.prop(self, key)


def _on_update_display_all_scene_group(self, context):
    bpy.ops.zsts.internal_build_lookup()


def _on_update_selection_follow_act_group(self, context):
    if bpy.ops.zsts.select_only.poll():
        bpy.ops.zsts.select_only()


def _on_update_display_mesh_mode(self, context):
    if not self.display_mesh_mode:
        self.display_mesh_mode = {'CAGE'}


class ZsCommonSettings(bpy.types.PropertyGroup):
    auto_highlight: bpy.props.BoolProperty(
        name=ZsLabels.PROP_AUTO_HIGHLIGHT_LABEL,
        description=ZsLabels.PROP_AUTO_HIGHLIGHT_DESC,
        default=True
    )

    display_all_parts: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DISPLAY_ALL_PARTS_LABEL,
        description=ZsLabels.PROP_DISPLAY_ALL_PARTS_DESC,
        default=True
    )

    auto_update_draw_cache: bpy.props.BoolProperty(
        name=ZsLabels.PROP_AUTO_UPDATE_DRAW_CACHE_LABEL,
        description=ZsLabels.PROP_AUTO_UPDATE_DRAW_CACHE_LABEL,
        default=True,
        options={'SKIP_SAVE'}
    )

    display_mesh_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_DISPLAY_MODIFIED_LABEL,
        description='Mode',
        items=[
            ('CAGE', 'Standard', 'Display Mesh Cage', 'MESH_DATA', 1),
            ('MOD', 'Modified', 'Display Modified Mesh', 'MODIFIER', 2)
        ],
        options={'SKIP_SAVE', 'ENUM_FLAG'},
        default={'CAGE'},
        update=_on_update_display_mesh_mode
    )

    object_shading: bpy.props.BoolProperty(
        name=ZsLabels.PROP_OBJECTS_SHADING_LABEL,
        description=ZsLabels.PROP_OBJECTS_SHADING_DESC,
        default=True
    )

    sync_with_mesh_selection: bpy.props.BoolProperty(
        name=ZsLabels.PROP_SYNC_MESH_SEL_LABEL,
        description=ZsLabels.PROP_SYNC_MESH_SEL_DESC,
        default=True
    )

    compact_mode: bpy.props.BoolProperty(
        name=ZsLabels.PROP_COMPACT_MODE_LABEL,
        description=ZsLabels.PROP_COMPACT_MODE_DESC,
        options={'HIDDEN', 'SKIP_SAVE'},
        default=False
    )

    pie_assist: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DISPLAY_PIE_ASSIST_LABEL,
        description=ZsLabels.PROP_DISPLAY_PIE_ASSIST_DESC,
        default=True
    )

    pie_assist_font_size: bpy.props.IntProperty(
        name=ZsLabels.PROP_DISPLAY_PIE_ASSIST_FONTSIZE_LABEL,
        description=ZsLabels.PROP_DISPLAY_PIE_ASSIST_FONTSIZE_DESC,
        min=8,
        max=16,
        default=11
    )

    developer_mode: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DEVELOPER_MODE_LABEL,
        description=ZsLabels.PROP_DEVELOPER_MODE_DESC,
        # Do not modify the next line !!!
        default=False  # KEY_DEVELOPER_MODE
    )

    def draw(self, layout):
        box = layout.box()

        for key in self.__annotations__.keys():
            # compact_mode - will be returned back later !
            if key == 'display_mesh_mode':
                row = box.row(align=True)
                row.prop(self, key, expand=True)
            elif key != 'compact_mode':
                box.prop(self, key)


_was_tree_view_state = None


def _on_collection_list_mode_update(self, context):
    global _was_tree_view_state
    if self.collection_list_mode != 'DEFAULT':
        if _was_tree_view_state is None:
            _was_tree_view_state = self.tree_view
        if self.tree_view:
            self.tree_view = False
    else:
        if _was_tree_view_state is not None:
            if self.tree_view != _was_tree_view_state:
                self.tree_view = _was_tree_view_state
            _was_tree_view_state = None


class ZsListOptions(bpy.types.PropertyGroup):
    display_all_scene_groups: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DISPLAY_ALL_SCENE_GROUPS_NAME,
        description=ZsLabels.PROP_DISPLAY_ALL_SCENE_GROUPS_DESC,
        default=False,
        update=_on_update_display_all_scene_group
    )

    display_hidden_groups_info: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DISPLAY_HIDDEN_GROUPS_INFO_NAME,
        description=ZsLabels.PROP_DISPLAY_HIDDEN_GROUPS_INFO_DESC,
        default=True,
    )

    display_excluded_groups_info: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DISPLAY_EXCLUDED_GROUPS_INFO_NAME,
        description=ZsLabels.PROP_DISPLAY_EXCLUDED_GROUPS_INFO_DESC,
        default=True,
    )

    display_hide_renders_info: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DISPLAY_DISABLED_IN_RENDERS_NAME,
        description=ZsLabels.PROP_DISPLAY_DISABLED_IN_RENDERS_DESC,
        default=True,
    )

    display_objects_info: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DISPLAY_OBJECTS_INFO_NAME,
        description=ZsLabels.PROP_DISPLAY_OBJECTS_INFO_DESC,
        default=True,
    )

    display_vertex_groups_weights_info: bpy.props.BoolProperty(
        name=ZsLabels.PROP_DISPLAY_VERTEX_GROUPS_WEIGHTS_NAME,
        description=ZsLabels.PROP_DISPLAY_VERTEX_GROUPS_WEIGHTS_DESC,
        default=True,
    )

    selection_follow_act_group: bpy.props.BoolProperty(
        name=ZsLabels.PROP_SELECTION_FOLLOW_ACT_GROUP_NAME,
        description=ZsLabels.PROP_SELECTION_FOLLOW_ACT_GROUP_DESC,
        default=True,
        update=_on_update_selection_follow_act_group
    )

    auto_frame_selected: bpy.props.BoolProperty(
        name=ZsLabels.PROP_AUTO_FRAME_SELECTED_NAME,
        description=ZsLabels.PROP_AUTO_FRAME_SELECTED_DESC,
        default=True
    )

    tree_view: bpy.props.BoolProperty(
        name='TreeView',
        description='Display List as TreeView in Object Mode',
        default=True
    )

    collection_list_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_COLLECTION_LIST_MODE_LABEL,
        description=ZsLabels.PROP_COLLECTION_LIST_MODE_DESC,
        items=[
            ("DEFAULT", "Default", "Default display collections"),
            ("SEL_OBJS", "Selected Objects", "Display collections that contain of Selected Objects"),
            ("SEL_OBJS_PARENTS", "Selected Parents", "Display collections with all parents of Selected Objects"),
            ("SEL_COLLECTIONS", "Selected Collections", "Display collections that are considered as Selected")
        ],
        default="DEFAULT",
        update=_on_collection_list_mode_update
    )

    def draw(self, layout):
        box = layout.box()

        for key in self.__annotations__.keys():
            if key == 'collection_list_mode':
                col = box.column(align=True)
                col.label(text=ZsLabels.PROP_COLLECTION_LIST_MODE_LABEL)
                col.prop(self, key, text='')
            else:
                box.prop(self, key)


def _on_update_uv_options(self, context):
    from .sets.draw_sets import check_update_cache, mark_groups_modified
    from .sets.factories import get_sets_mgr

    if self.display_uv:
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            for p_obj in context.objects_in_mode:
                mark_groups_modified(p_cls_mgr, p_obj, modes={'UV'})
                check_update_cache(p_cls_mgr, p_obj)

            update_imageeditor_in_all_screens()


class ZsUvOptions(bpy.types.PropertyGroup):
    display_uv: bpy.props.BoolProperty(
        name='Display UV',
        description='Display groups in UV Editor',
        default=True,
        update=_on_update_uv_options
    )

    selected_only: bpy.props.BoolProperty(
        name='Visible UV only',
        description='Display only visible UV elements in UV Editor',
        default=True,
        update=_on_update_uv_options
    )

    def draw(self, layout):
        box = layout.box()

        for key in self.__annotations__.keys():
            box.prop(self, key)


i_MAX_WEIGHT_PRESET_SIZE = 30
i_DEFAULT_WEIGHT_PRESET_SIZE = 11


def get_default_weights():
    return [(0.1*i if i < i_DEFAULT_WEIGHT_PRESET_SIZE else 0.0) for i in range(i_MAX_WEIGHT_PRESET_SIZE)]


class ZsVertexGroupsOptions(bpy.types.PropertyGroup):
    weight_preset: bpy.props.FloatVectorProperty(
        name='Weights Preset',
        size=i_MAX_WEIGHT_PRESET_SIZE,
        min=0.0,
        max=1.0,
        soft_min=0.0,
        soft_max=1.0,
        precision=3,
        default=get_default_weights()
    )

    weight_preset_size: bpy.props.IntProperty(
        name='Weights Preset Size',
        min=2,
        max=i_MAX_WEIGHT_PRESET_SIZE,
        default=i_DEFAULT_WEIGHT_PRESET_SIZE
    )

    mask_modifier: bpy.props.BoolProperty(
        name='Mask Modifier',
        description='Hide | Unhide Vertex Group in Object Mode using Mask Modifier',
        default=False
    )

    mask_delete_after: bpy.props.BoolProperty(
        name='Clear Mask after Off',
        description='Delete Mask Modifier after turning Mask off',
        default=True
    )

    def draw_mask(self, layout, context):
        if ZenPolls.is_edit_and_vertex_group_mode(context):
            box = layout.box()
            box.prop(self, 'mask_modifier')
            row = box.row(align=True)
            row.enabled = self.mask_modifier
            row.prop(self, 'mask_delete_after')


def _on_sync_with_collection_color_tag(self, context):
    from .sets.factories import get_sets_object_mgr

    p_cls_mgr = get_sets_object_mgr(context.scene)
    if p_cls_mgr:
        if self.sync_with_collection_color_tag:
            p_cls_mgr.sync_all_collections_icons(context)
        else:
            p_cls_mgr.reset_all_collection_icons(context)


def draw_collection_toolbar(self, context):
    if context.mode == 'OBJECT':
        addon_prefs = get_prefs()
        if addon_prefs.object_options.display_collection_toolbar_in_view3d_header:
            from .sets.factories import get_sets_object_mgr
            p_cls_mgr = get_sets_object_mgr(context.scene)
            if p_cls_mgr:
                p_cls_mgr._do_draw_collection_toolbar(context, self.layout, False)


def update_collection_toolbar_header_position(self, context):
    bpy.types.VIEW3D_HT_header.remove(draw_collection_toolbar)

    if self.toolbar_in_view3d_header_position == 'RIGHT':
        bpy.types.VIEW3D_HT_header.append(draw_collection_toolbar)
    else:
        bpy.types.VIEW3D_HT_header.prepend(draw_collection_toolbar)


class ZsObjectOptions(bpy.types.PropertyGroup):
    sync_with_collection_color_tag: bpy.props.BoolProperty(
        name=ZsLabels.PROP_SYNC_COLLECTION_COLOR_TAG_NAME,
        description=ZsLabels.PROP_SYNC_COLLECTION_COLOR_TAG_DESC,
        default=False,
        update=_on_sync_with_collection_color_tag
    )

    hide_collection_with_objects: bpy.props.BoolProperty(
        name=ZsLabels.PROP_HIDE_COLLECTIONS_WITH_OBJECTS_NAME,
        description=ZsLabels.PROP_HIDE_COLLECTIONS_WITH_OBJECTS_DESC,
        default=False
    )

    display_collection_toolbar_in_view3d_header: bpy.props.BoolProperty(
        name='Toolbar in View3D Header',
        description='Display Collection Toolbar in View3D Header',
        default=True
    )

    toolbar_in_view3d_header_position: bpy.props.EnumProperty(
        name='Toolbar Position',
        description='Position of Collection Toolbar in View3D Header',
        items=[
            ('LEFT', 'Left', ''),
            ('RIGHT', 'Right', '')
        ],
        default='RIGHT',
        update=update_collection_toolbar_header_position
    )

    display_active_object_collections: bpy.props.BoolProperty(
        name='Collections in Active Object',
        description='Display active Object Collections names at the corner of its Bound box',
        default=True
    )

    display_collection_caption: bpy.props.BoolProperty(
        name='Collection Caption',
        description='Display Collection Name at the corner of its Bound box',
        default=True
    )

    display_collection_bbox: bpy.props.BoolProperty(
        name='Collection Bound Box',
        description='Display Collection Bound Box',
        default=True
    )

    display_collection_caption_mode: bpy.props.EnumProperty(
        name='Collection Caption Mode',
        description='Select Active or All Collections to display caption',
        items=[
            ('ACTIVE', 'Active', 'Display caption of Active Collection'),
            ('ALL', 'All', 'Display caption of All Collections')
        ],
        default='ACTIVE'
    )

    object_display_transform_affect_options: bpy.props.BoolProperty(
        name='Transform Affect State',
        description='Display Transform affect options State(s)',
        default=True)

    def draw(self, layout):
        box = layout.box()

        for key in self.__annotations__.keys():
            if key == 'toolbar_in_view3d_header_position':
                row = box.row(align=True)
                row.prop(self, key, expand=True)
            elif key == 'display_collection_caption_mode':
                col = box.column(align=True)
                col.label(text='Collection Caption Mode')
                row = col.row(align=True)
                row.prop(self, key, expand=True)
            else:
                box.prop(self, key)

    def draw_collection_captions(self, layout):
        layout.prop(self, 'display_collection_bbox')
        if self.display_collection_bbox:
            layout.prop(self, 'display_collection_caption')
            if self.display_collection_caption:
                row = layout.row(align=True)
                row.prop(self, 'display_collection_caption_mode', expand=True)

        layout.prop(self, 'display_active_object_collections')


class ZsGroupSelectorOptions(bpy.types.PropertyGroup):
    mode: bpy.props.EnumProperty(
        name='Mode',
        items=[
            ('ALL', 'All', 'All Groups'),
            ('VISIBLE', 'Visible', 'Visible Groups Only'),
            ('ISOLATE', 'Isolate', 'Isolate Selected Group')
        ],
        default='ALL'
    )

    skip_empty: bpy.props.BoolProperty(
        name='Skip Empty Groups',
        description='Skip Groups that do not contain elements',
        default=False
    )

    select: bpy.props.BoolProperty(
        name='Select',
        default=False)

    frame_selected: bpy.props.BoolProperty(
        name='Frame Selected',
        description='Move the view to the selection center',
        default=False)


class ZsModeEnableSettings(bpy.types.PropertyGroup):
    enable_vert: bpy.props.BoolProperty(
        name='Vert Sets',
        description='Enable Vert Sets Mode',
        default=True
    )

    enable_vert_u: bpy.props.BoolProperty(
        name='Vert Parts',
        description='Enable Vert Parts Mode',
        default=True
    )

    enable_edge: bpy.props.BoolProperty(
        name='Edge Sets',
        description='Enable Edge Sets Mode',
        default=True
    )

    enable_edge_u: bpy.props.BoolProperty(
        name='Edge Parts',
        description='Enable Edge Parts Mode',
        default=True
    )

    enable_face: bpy.props.BoolProperty(
        name='Face Sets',
        description='Enable Face Sets Mode',
        default=True
    )

    enable_face_u: bpy.props.BoolProperty(
        name='Face Parts',
        description='Enable Face Parts Mode',
        default=True
    )

    enable_blgroup: bpy.props.BoolProperty(
        name='Vertex Groups',
        description='Enable Blender Vertex Groups Mode',
        default=True
    )

    enable_blgroup_u: bpy.props.BoolProperty(
        name='Face Maps',
        description='Enable Blender Face Maps Mode',
        default=True
    )

    enable_obj_simple_sets: bpy.props.BoolProperty(
        name='Object Sets',
        description='Enable Object Sets Mode',
        default=True
    )

    enable_obj_simple_parts: bpy.props.BoolProperty(
        name='Object Parts',
        description='Enable Object Parts Mode',
        default=True
    )

    def draw(self, layout):
        box = layout.box()

        for i, key in enumerate(self.__annotations__.keys()):
            if i != 0 and i % 2 == 0:
                box.separator(factor=0.1)
            box.prop(self, key)


t_empty_display_type_items = [
    ('PLAIN_AXES', 'Plain Axes', '', 'EMPTY_AXIS', 0),
    ('ARROWS', 'Arrows', '', 'EMPTY_ARROWS', 1),
    ('SINGLE_ARROW', 'Single Arrow', '', 'EMPTY_SINGLE_ARROW', 2),
    ('CIRCLE', 'Circle', '', 'MESH_CIRCLE', 3),
    ('CUBE', 'Cube', '', 'MESH_CUBE', 4),
    ('SPHERE', 'Sphere', '', 'SPHERE', 5),
    ('CONE', 'Cone', '', 'CONE', 6),
    ('IMAGE', 'Image', '', 'IMAGE', 7)
]


class ZsOperatorOptions(bpy.types.PropertyGroup):
    show_hide_group_select: bpy.props.BoolProperty(name="Select", default=False)

    tool_assign_pinned_clear_select: bpy.props.BoolProperty(name="Clear Selection", default=True)
    tool_assign_pinned_group_name: bpy.props.StringProperty(name="Pinned", default='Pinned')
    tool_assign_pinned_group_color: bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_DEFAULT_COLOR_NAME,
        subtype='COLOR_GAMMA',
        size=3,
        min=0, max=1,
        default=(0.3, 0.8, 0.3)
    )

    def get_tool_assign_pinned_group_name():
        return bpy.props.StringProperty(
            name=ZsLabels.PROP_GROUP_NAME,
            description=ZsLabels.PROP_PINNED_DESC,
            get=lambda self: get_prefs().op_options.tool_assign_pinned_group_name,
            set=lambda self, value: setattr(get_prefs().op_options, 'tool_assign_pinned_group_name', value)
        )

    def get_tool_assign_pinned_group_color(options={'ANIMATABLE'}):
        return bpy.props.FloatVectorProperty(
            name=ZsLabels.PROP_DEFAULT_COLOR_NAME,
            description=ZsLabels.PROP_DEFAULT_PINNED_COLOR_DESC,
            subtype='COLOR_GAMMA',
            size=3,
            min=0, max=1,
            get=lambda self: get_prefs().op_options.tool_assign_pinned_group_color,
            set=lambda self, value: setattr(get_prefs().op_options, 'tool_assign_pinned_group_color', value),
            options=options)

    convert_parent_to_collection_skip_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_CONVERT_OBJ_TO_COL_SKIP_EMPTIES_LABEL,
        description=ZsLabels.PROP_CONVERT_OBJ_TO_COL_SKIP_EMPTIES_DESC,
        items=[
            ("0",
             ZsLabels.PROP_CONVERT_OBJ_TO_COL_SKIP_EMPTIES_AUTO_LABEL,
             ZsLabels.PROP_CONVERT_OBJ_TO_COL_SKIP_EMPTIES_AUTO_DESC),
            ("1",
             ZsLabels.PROP_CONVERT_OBJ_TO_COL_SKIP_EMPTIES_SKIP_ALL_LABEL,
             ZsLabels.PROP_CONVERT_OBJ_TO_COL_SKIP_EMPTIES_SKIP_ALL_DESC),
            ("2",
             ZsLabels.PROP_CONVERT_OBJ_TO_COL_SKIP_EMPTIES_LOAD_ALL_LABEL,
             ZsLabels.PROP_CONVERT_OBJ_TO_COL_SKIP_EMPTIES_LOAD_ALL_DESC),
        ],
        default="0"
    )

    convert_collection_to_parent_select: bpy.props.BoolProperty(
        name="Select All", description=ZsLabels.PROP_CONVERT_COL_TO_OBJ_SELECT_ALL_DESC, default=False)

    calculate_col_center_default_position: bpy.props.EnumProperty(
        name=ZsLabels.PROP_CONVERT_COL_TO_OBJ_DEFAULT_CENTER_POS_LABEL,
        description=ZsLabels.PROP_CONVERT_COL_TO_OBJ_DEFAULT_CENTER_POS_DESC,
        items=[
            ("0",
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_ZERO_POINT_LABEL,
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_ZERO_POINT_DESC),
            ("1",
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_CURSOR_3D_LABEL,
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_CURSOR_3D_DESC),
        ],
        default="0"
    )

    calculate_col_center_calculate_center: bpy.props.EnumProperty(
        name=ZsLabels.PROP_CONVERT_COL_TO_OBJ_CALC_CENTER_POS_LABEL,
        description=ZsLabels.PROP_CONVERT_COL_TO_OBJ_CALC_CENTER_POS_DESC,
        items=[
            ("0",
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_OBJECTS_CENTER_DEFAULT_LABEL,
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_OBJECTS_CENTER_DEFAULT_DESC),
            ("1",
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_OBJECTS_CENTER_LABEL,
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_OBJECTS_CENTER_DESC),
            ("2",
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_OBJECTS_NESTED_CENTER_LABEL,
             ZsLabels.PROP_CONVERT_COL_TO_OBJ_OBJECTS_NESTED_CENTER_DESC),
        ],
        default="1"
    )

    calculate_col_center_calculate_center_with_rot: bpy.props.BoolProperty(
        name=ZsLabels.PROP_CONVERT_COL_TO_OBJ_CALC_CENTER_USE_ROT_LABEL,
        description=ZsLabels.PROP_CONVERT_COL_TO_OBJ_CALC_CENTER_USE_ROT_DESC,
        default=True
    )

    calculate_col_center_calculate_center_with_scale: bpy.props.BoolProperty(
        name=ZsLabels.PROP_CONVERT_COL_TO_OBJ_CALC_CENTER_USE_SCALE_LABEL,
        description=ZsLabels.PROP_CONVERT_COL_TO_OBJ_CALC_CENTER_USE_SCALE_DESC,
        default=True
    )

    def get_empty_display_size(default=0, get=None, set=None, options={'ANIMATABLE'}):
        return bpy.props.FloatProperty(
            name="Size",
            description="Size of display for empties in the viewport",
            min=0.0001,
            max=1000,
            unit='LENGTH',
            default=default,
            get=get,
            set=set,
            options=options
        )

    empty_display_size: get_empty_display_size(
        get=lambda self: ZsOperatorAttrs.get_operator_attr('zsto.convert_collection_to_object', 'empty_display_size', 1.0),
        set=lambda self, value: ZsOperatorAttrs.set_operator_attr('zsto.convert_collection_to_object', 'empty_display_size', value),
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    def get_empty_display_type(default=0, get=None, set=None, options={'ANIMATABLE'}):
        return bpy.props.EnumProperty(
            name="Display As",
            description="Viewport display style for empties",
            items=t_empty_display_type_items,
            default='PLAIN_AXES')

    empty_display_type: get_empty_display_type(
        get=lambda self: ZsOperatorAttrs.get_operator_attr_int_enum(
            'zsto.convert_collection_to_object', 'empty_display_type', 0, t_empty_display_type_items),
        set=lambda self, value: ZsOperatorAttrs.set_operator_attr_int_enum(
            'zsto.convert_collection_to_object', 'empty_display_type', value, t_empty_display_type_items),
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    def draw_calculate_center(self, layout):

        get_mimic_bl_rna_layout(self, self.bl_rna, layout, 'calculate_col_center_default_position', factor=0.4)

        box = layout.box()
        for key in self.__annotations__.keys():
            if key.startswith("calculate_col_center") and key != 'calculate_col_center_default_position':
                _ = get_mimic_bl_rna_layout(self, self.bl_rna, box, key, factor=0.4)

    def draw_convert_collection_to_parent(self, layout: bpy.types.UILayout):
        self.draw_calculate_center(layout)

        get_mimic_bl_rna_layout(self, self.bl_rna, layout, "convert_collection_to_parent_select", factor=0.4)
        get_mimic_bl_rna_layout(self, self.bl_rna, layout, "empty_display_type", factor=0.4)
        get_mimic_bl_rna_layout(self, self.bl_rna, layout, "empty_display_size", factor=0.4)

    def draw_convert_parent_to_collection(self, layout):
        get_mimic_bl_rna_layout(self, self.bl_rna, layout, "convert_parent_to_collection_skip_mode", factor=0.4)

        box = layout.box()
        if self.convert_parent_to_collection_skip_mode == "0":
            self.draw_calculate_center(box)
        else:
            box.label(text='No options')

    def draw_assign_to_pinned(self, layout: bpy.types.UILayout):
        get_mimic_bl_rna_layout(self, self.bl_rna, layout, "tool_assign_pinned_group_name", factor=0.5)
        get_mimic_bl_rna_layout(self, self.bl_rna, layout, "tool_assign_pinned_group_color", factor=0.5)


class ZsInstanceCalculateCenterCollectionProps:
    calculate_col_center_default_position: get_instanced_int_enum_property(
        ZsOperatorOptions, "op_options", "calculate_col_center_default_position"
    )

    calculate_col_center_calculate_center: get_instanced_int_enum_property(
        ZsOperatorOptions, "op_options", "calculate_col_center_calculate_center"
    )

    calculate_col_center_calculate_center_with_rot: get_instanced_bool_property(
        ZsOperatorOptions, "op_options", "calculate_col_center_calculate_center_with_rot"
    )

    calculate_col_center_calculate_center_with_scale: get_instanced_bool_property(
        ZsOperatorOptions, "op_options", "calculate_col_center_calculate_center_with_scale"
    )


class ZsInstanceCollectionToParent(ZsInstanceCalculateCenterCollectionProps):
    select: get_instanced_bool_property(
        ZsOperatorOptions, "op_options", "convert_collection_to_parent_select"
    )


class ZsInstanceParentToCollection(ZsInstanceCalculateCenterCollectionProps):
    convert_parent_to_collection_skip_mode: get_instanced_int_enum_property(
        ZsOperatorOptions, "op_options", "convert_parent_to_collection_skip_mode"
    )


class ZsMinMaxGroup(bpy.types.PropertyGroup):
    def get_min(self):
        return self.get('min', 0)

    def set_min(self, value):
        p_min = self.get('min', 0)
        if p_min != value:
            self['min'] = value
            if value > self.max:
                self['max'] = value

    def get_max(self):
        return self.get('max', 0)

    def set_max(self, value):
        p_max = self.get('max', 0)
        if p_max != value:
            self['max'] = value
            if self.min > value:
                self['min'] = value

    min: bpy.props.FloatProperty(
        name='Range Min',
        get=get_min,
        set=set_min,
        min=0.0,
        max=1.0,
        soft_min=0.0,
        soft_max=1.0,
        precision=3
    )

    max: bpy.props.FloatProperty(
        name='Range Max',
        get=get_max,
        set=set_max,
        min=0.0,
        max=1.0,
        soft_min=0.0,
        soft_max=1.0,
        precision=3
    )

    def draw(self, layout, context):
        layout.prop(self, 'min')
        layout.prop(self, 'max')


class ZSTS_AddonPreferences(bpy.types.AddonPreferences):

    bl_idname = ZsLabels.ADDON_NAME

    def draw(self, context):
        layout = self.layout

        row = layout.row()
        row.prop(self, "tabs", expand=True)

        if self.tabs == 'KEYMAP':
            draw_keymaps(context, layout)
        elif self.tabs == 'DISPLAY':
            box = layout.box()
            row = box.row()
            row.label(text="Overlay label")

            if is_property_modified(self.label_2d):
                subrow = row.row()
                subrow.alignment = 'RIGHT'
                op = subrow.operator("zsts.restore_preference", text="Restore")
                op.mode = 'label_2d'

            self.label_2d.draw(box)

            box = layout.box()
            self.label_2d.draw_tool_options(box)

            box = layout.box()
            box.label(text='3D display settings')
            self.display_3d.draw(box)

            box = layout.box()
            box.label(text="Panels")
            self.show_panels.draw(box)

        elif self.tabs == 'COMMON':
            r = layout.row()
            r = r.split(factor=0.5)
            col1 = r.column(align=False)
            col2 = r.column(align=False)

            # - 1 column

            row = col1.row()
            row.label(text="Common Options")
            if is_property_modified(self.common):
                subrow = row.row()
                subrow.alignment = 'RIGHT'
                op = subrow.operator("zsts.restore_preference", text="Restore")
                op.mode = 'common'
            self.common.draw(col1)

            row = col1.row()
            row.label(text="Groups Object Options")

            if is_property_modified(self.object_options):
                subrow = row.row()
                subrow.alignment = 'RIGHT'
                op = subrow.operator("zsts.restore_preference", text="Restore")
                op.mode = 'object_options'
            self.object_options.draw(col1)

            # - 2 column

            row = col2.row()
            row.label(text="Groups List Options")
            if is_property_modified(self.list_options):
                subrow = row.row()
                subrow.alignment = 'RIGHT'
                op = subrow.operator("zsts.restore_preference", text="Restore")
                op.mode = 'list_options'
            self.list_options.draw(col2)

            row = col2.row()
            row.label(text="UV Options")
            if is_property_modified(self.uv_options):
                subrow = row.row()
                subrow.alignment = 'RIGHT'
                op = subrow.operator("zsts.restore_preference", text="Restore")
                op.mode = 'uv_options'
            self.uv_options.draw(col2)

        elif self.tabs == 'MODES':
            box = layout.box()
            box.label(text="Modes")

            self.modes.draw(box)

        elif self.tabs == 'HELP':
            box = layout.box()
            box.operator(
                "wm.url_open",
                text=ZsLabels.PANEL_HELP_DOC_LABEL,
                icon="HELP"
            ).url = ZsLabels.PANEL_HELP_DOC_LINK
            box.operator(
                "wm.url_open",
                text=ZsLabels.PANEL_HELP_DISCORD_LABEL,
                icon_value=zs_icon_get(ZIconsType.DiscordLogo)
            ).url = ZsLabels.PANEL_HELP_DISCORD_LINK

        box_products = layout.box()
        box_products.label(text='Zen Add-ons')
        box_products.operator(
            "wm.url_open",
            text=ZsLabels.PREF_ZEN_UV_URL_DESC,
            icon_value=zs_icon_get(ZIconsType.AddonZenUVLogo)
        ).url = ZsLabels.PREF_ZEN_UV_URL_LINK
        box_products.operator(
            "wm.url_open",
            text=ZsLabels.PREF_ZEN_UV_BBQ_URL_DESC,
            icon_value=zs_icon_get(ZIconsType.AddonZenBBQ)
        ).url = ZsLabels.PREF_ZEN_UV_BBQ_URL_LINK
        box_products.operator(
            "wm.url_open",
            text=ZsLabels.PREF_ZEN_UV_CHECKER_URL_DESC,
            icon_value=zs_icon_get(ZIconsType.AddonZenChecker)
        ).url = ZsLabels.PREF_ZEN_UV_CHECKER_URL_LINK
        box_products.operator(
            "wm.url_open",
            text=ZsLabels.PREF_ZEN_UV_TOPMOST_URL_DESC,
            icon='WINDOW',
        ).url = ZsLabels.PREF_ZEN_UV_TOPMOST_URL_LINK

    tabs: bpy.props.EnumProperty(
        items=[
            ("COMMON", "Common", ""),
            ("KEYMAP", "Keymap", ""),
            ("DISPLAY", "Display", ""),
            ("MODES", "Modes", ""),
            ("HELP", "Help", ""),
        ],
        default="COMMON"
    )

    common: bpy.props.PointerProperty(type=ZsCommonSettings)

    list_options: bpy.props.PointerProperty(type=ZsListOptions)

    object_options: bpy.props.PointerProperty(type=ZsObjectOptions)

    uv_options: bpy.props.PointerProperty(type=ZsUvOptions)

    label_2d: bpy.props.PointerProperty(type=ZsDisplay2DSettings)

    display_3d: bpy.props.PointerProperty(type=ZsDisplay3DSettings)

    show_panels: bpy.props.PointerProperty(type=ZsShowPanelSettings)

    modes: bpy.props.PointerProperty(type=ZsModeEnableSettings)

    group_selector_options: bpy.props.PointerProperty(type=ZsGroupSelectorOptions)

    op_options: bpy.props.PointerProperty(type=ZsOperatorOptions)

    vgroups_options: bpy.props.PointerProperty(type=ZsVertexGroupsOptions)


class ZSTS_OT_RestorePreference(bpy.types.Operator):
    """ Zen Sets Build Lookup """
    bl_idname = 'zsts.restore_preference'
    bl_label = 'Restore'
    bl_options = {'REGISTER'}

    mode: bpy.props.EnumProperty(
        items=[('label_2d', 'label_2d', ''),
               ('list_options', 'list_options', ''),
               ('common', 'common', ''),
               ('uv_options', 'uv_options', ''),
               ('object_options', 'object_options', ''),

               ("3D_VERT", "3D_VERT", ""),
               ("3D_EDGE", "3D_EDGE", ""),
               ("3D_FACE", "3D_FACE", ""),
               ("3D_OBJECT", "3D_OBJECT", ""),

               ('NONE', 'NONE', '')],

        default='NONE')

    def execute(self, context):
        addon_prefs = get_prefs()

        unique_modes_3d = (f'3D_{item.upper()}' for item in _DISPLAY_3D_PROPS.keys())
        prop_group_modes = ('label_2d', 'common', 'uv_options', 'list_options', 'object_options')

        if self.mode in unique_modes_3d:
            id_element = self.mode.replace('3D_', '').lower()
            _DISPLAY_3D_PROPS[id_element].restore(addon_prefs.display_3d, id_element)
        elif self.mode in prop_group_modes:
            reset_property_modified(getattr(addon_prefs, self.mode))

        update_areas_in_all_screens()
        return {'FINISHED'}


classes = (
    ZsDisplay2DSettings,
    ZsDisplay3DSettings,
    ZsShowPanelSettings,
    ZsListOptions,
    ZsObjectOptions,
    ZsGroupSelectorOptions,
    ZsUvOptions,
    ZsOperatorOptions,
    ZsCommonSettings,
    ZsModeEnableSettings,
    ZsVertexGroupsOptions,
    ZsMinMaxGroup,

    ZSTS_OT_Keymaps,
    ZSTS_OT_RestorePreference,

    ZSTS_AddonPreferences
)


def get_prefs() -> ZSTS_AddonPreferences:
    """ Return Zen Sets Properties obj """
    return bpy.context.preferences.addons['ZenSets'].preferences


def register():
    for c in classes:
        bpy.utils.register_class(c)

    try:
        addon_prefs = get_prefs()
        update_collection_toolbar_header_position(addon_prefs.object_options, bpy.context)
    except Exception as e:
        Log.error(e)


def unregister():
    global _CACHE_ADDON_VERSION
    _CACHE_ADDON_VERSION = None

    try:
        bpy.types.VIEW3D_HT_header.remove(draw_collection_toolbar)
    except Exception as e:
        Log.error(e)

    for c in reversed(classes):
        bpy.utils.unregister_class(c)
