# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2022, Alex Zhornyak

""" Zen Sets Right menu inject """
import bpy

from .sets import ZSTSSceneListGroup
from .sets.elements.bl_vgroup_sets import ZSBLVGSceneListGroup
from .sets.elements.bl_fmaps_unisets import ZSUFMSceneListGroup

from .sets.objects.collection_object_sets import ZSTSCollectionListGroup
from .sets.objects.simple.simple_object_sets import ZSOS_SceneListGroup
from .sets.objects.simple.simple_object_parts import ZSOP_SceneListGroup

from .sets.factories import get_sets_mgr

from .ico import zs_icon_get, ZIconsType


scene_list_classes = (
    ZSTSSceneListGroup,
    ZSBLVGSceneListGroup,
    ZSUFMSceneListGroup,

    ZSTSCollectionListGroup,
    ZSOS_SceneListGroup,
    ZSOP_SceneListGroup
)


def is_scene_list_type(ptr):
    for cls in scene_list_classes:
        if isinstance(ptr, cls):
            return True
    return False


def _dump_object(obj, text):
    print('-' * 40, text, '-' * 40)
    s_type = type(obj)
    print('Type ====>', s_type)
    for attr in dir(obj):
        if hasattr(obj, attr):
            print(f".{attr} = {getattr(obj, attr, 'None')}")


def _dump_context(context):
    if hasattr(context, 'button_pointer'):
        btn = context.button_pointer
        _dump_object(btn, 'button_pointer')

    if hasattr(context, 'button_prop'):
        prop = context.button_prop
        _dump_object(prop, 'button_prop')

    if hasattr(context, 'button_operator'):
        op = context.button_operator
        _dump_object(op, 'button_operator')


class ZSTS_OT_InternalRmbTest(bpy.types.Operator):
    """Right click entry test"""
    bl_idname = "zsts.internal_rmb_test"
    bl_label = "Execute custom action"

    @classmethod
    def poll(cls, context):
        return context.active_object is not None

    def execute(self, context):

        _dump_context(context)

        return {'FINISHED'}


class ZSTS_OT_PythonCmdCopyToAutoGroups(bpy.types.Operator):
    """ Copy to Auto Groups """
    bl_idname = "zsts.copy_python_cmd_to_auto_groups"
    bl_label = "Copy command to Auto Groups"
    bl_description = "Copy python command to Zen Sets Auto Groups Operator"
    bl_options = {'REGISTER', 'UNDO'}

    python_command: bpy.props.StringProperty(
        name='Python Command',
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    @classmethod
    def poll(cls, context):
        if context.mode in {'EDIT_MESH', 'OBJECT'}:

            op_prefix = 'zsts' if context.mode == 'EDIT_MESH' else 'zsto'

            wm = context.window_manager
            op_props = wm.operator_properties_last(op_prefix + '.group_linked')
            return op_props is not None
        return False

    def execute(self, context):
        if context.mode in {'EDIT_MESH', 'OBJECT'}:
            op_prefix = 'zsts' if context.mode == 'EDIT_MESH' else 'zsto'

            wm = context.window_manager
            op_props = wm.operator_properties_last(op_prefix + '.group_linked')
            if op_props:
                op_props.custom_delimiter = self.python_command
                if 'CUSTOM' not in op_props.delimit:
                    op_props.delimit = {'CUSTOM'}
                    return {'FINISHED'}

        return {'CANCELLED'}


# This class has to be exactly named like that to insert an entry in the right click menu
class WM_MT_button_context(bpy.types.Menu):
    bl_label = "Zen Sets Injector Menu"

    def draw(self, context):
        pass


def zen_sets_menu_func(self, context: bpy.types.Context):
    layout = self.layout

    # DEBUG
    # _dump_context(context)

    if hasattr(context, 'button_pointer'):
        btn = context.button_pointer
        if is_scene_list_type(btn):
            layout.separator()
            layout.operator('zsts.palette_color_assign_to_group', icon='EYEDROPPER').group_layer = btn.layer_name

            if hasattr(context, 'scene'):
                p_cls_mgr = get_sets_mgr(context.scene)
                if p_cls_mgr:
                    if p_cls_mgr.id_group == 'blgroup':
                        layout.operator('zsts.assign_vertex_weight_to_group_color', icon='EYEDROPPER').group_layer = btn.layer_name

            layout.separator()
            layout.operator("zsts.export_group_colors_to_palette", icon='COLOR')

    if hasattr(context, 'button_operator'):
        op = context.button_operator

        bl_rna = getattr(op, 'bl_rna', None)
        if bl_rna is not None:
            id = getattr(bl_rna, "identifier", None)
            if id is not None:
                if id == 'WM_OT_tool_set_by_id':
                    if op.name.startswith('zsts.select_'):
                        layout.separator()
                        layout.operator('zsts.draw_show_tool_help', icon_value=zs_icon_get(ZIconsType.AddonLogoPng))
                else:
                    if context.mode in {'EDIT_MESH', 'OBJECT'}:

                        op_prefix = 'zsts' if context.mode == 'EDIT_MESH' else 'zsto'

                        wm = context.window_manager
                        op_props = wm.operator_properties_last(op_prefix + '.group_linked')
                        if op_props:
                            op_module, op_name = id.split("_OT_")
                            if 'select' in op_name.lower():
                                props = op.bl_rna.properties
                                keys = set(props.keys()) - {'rna_type'}
                                kwargs = [
                                    k + '=' + repr(getattr(op, k))
                                    for k in dir(op)
                                    if k in keys and not props[k].is_readonly and not props[k].is_skip_save]
                                kwargs = ','.join(kwargs)

                                op_module = op_module.lower()

                                layout.separator()
                                op = layout.operator('zsts.copy_python_cmd_to_auto_groups', icon_value=zs_icon_get(ZIconsType.AddonLogoPng))
                                op.python_command = f'{op_module}.{op_name}({kwargs})'


def menu_object_zen_sets_draw(self, context):

    ops = (
        bpy.ops.zsto.rename_as_parent_collection,
        bpy.ops.zsto.rename_collection_as_object,
        bpy.ops.zsto.sync_parent_pos_with_child
    )

    poll_results = [op.idname_py() for op in ops if op.poll()]

    if len(poll_results) != 0:
        layout = self.layout
        layout.separator()

        for op_name in poll_results:
            layout.operator(op_name)


classes = (
    # DEBUG
    # ZSTS_OT_InternalRmbTest,
    ZSTS_OT_PythonCmdCopyToAutoGroups,

    WM_MT_button_context,
)


def register():
    for cls in classes:
        bpy.utils.register_class(cls)
    bpy.types.WM_MT_button_context.append(zen_sets_menu_func)
    bpy.types.VIEW3D_MT_object_context_menu.append(menu_object_zen_sets_draw)
    bpy.types.OUTLINER_MT_object.append(menu_object_zen_sets_draw)


def unregister():
    bpy.types.OUTLINER_MT_object.remove(menu_object_zen_sets_draw)
    bpy.types.VIEW3D_MT_object_context_menu.remove(menu_object_zen_sets_draw)
    bpy.types.WM_MT_button_context.remove(zen_sets_menu_func)
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)
