# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

""" Zen Sets Init """

import bpy
from bpy.app.handlers import persistent

import functools
from timeit import default_timer as timer

from ..labels import ZsLabels
from ..preferences import get_prefs, t_ZEN_SETS_ELEMENT_LINKS
from ..overlay_msgbox import cleanup_overlay_handles
from ..ico import zs_icon_get, ZIconsType
from ..vlog import Log
from ..blender_zen_utils import (
    ZenLocks,
    ZenPolls,
    update_areas_in_all_screens)

from .draw_sets import (
    check_update_cache,
    mark_groups_modified,
    is_draw_active,
    is_draw_handler_enabled,
    remove_all_handlers3d,
    remove_draw_handler,
    reset_all_draw_cache)

from .basic_interface import ZsInterfaceFactory
from .objects.collection_interface import ZsObjectsInterfaceFactory

from .basic_prop_list import (
    ZSTSObjectListGroup, ZSTSSceneListGroup)
from .pie import ZsPieFactory
from .tools import (
    ZsBoxWorkSpaceTool,
    ZsCircleWorkSpaceTool,
    ZsLassoWorkSpaceTool,
    ZsObjectWorkSpaceTool,
    ZsUvWorkSpaceTool,
    ZsToolFactory
)
from .color_palette import ZsColorPaletteFactory

from .factories import (
    sets_factories, obj_collection_factories,
    get_sets_mgr, get_sets_ids,
    get_sets_edit_mgr, get_sets_object_mgr)

from .edit_mesh_handlers import _ObjectCacheStore

from .elements.bl_vgroup_sets import vertexGroupsIndexSync, ZsVGroupLayerManager
from .elements.bl_fmaps_unisets import faceMapsIndexSync, ZsFaceMapsLayerManager


classes = (
    ZSTSObjectListGroup,
    ZSTSSceneListGroup,
)

_factories = (
    ZsPieFactory,
    ZsInterfaceFactory,
    ZsObjectsInterfaceFactory,
    ZsToolFactory,
    ZsColorPaletteFactory
)

ELEMENT_MODES = (('vert', 'vert_u', 'blgroup'),
                 ('edge', 'edge_u'),
                 ('face', 'face_u', 'blgroup_u'))

_handle_subcribe_to = None
_key_subscribe_to_edit_mode = bpy.types.Object, "mode"

i_UPDATE_COUNT = 0
g_LAST_UPDATE = {}

_LAST_MESH_SELECT = [False, False, False]
_ACTIVE_MODE_LOCK = False
_LAST_ACTIVE_MODE = ''
_LAST_OBJECT_ACTIVE_MODE = ''
_zen_sets_group_items = []

_USE_SINGLE_TOOL = True

_notify_tool_cache = {'EDIT_MESH': '', 'OBJECT': ''}

_zen_sets_active_mode_ids = None


def clear_notify_tool_cache():
    for k in _notify_tool_cache.keys():
        _notify_tool_cache[k] = ''


def _get_group_ids(scene, context):
    global _zen_sets_active_mode_ids
    # do not cache because custom icon values are not displayed !!!
    _zen_sets_active_mode_ids = []

    for i, f in enumerate(sets_factories):
        p_cls_mgr = f.get_mgr()
        if p_cls_mgr.id_group == 'blgroup':
            icon_id = 'GROUP_VERTEX'
        elif p_cls_mgr.id_group == 'blgroup_u':
            icon_id = 'FACE_MAPS'
        else:
            icon_id = zs_icon_get(ZIconsType.Vert + i)
        _zen_sets_active_mode_ids.append((p_cls_mgr.id_group, p_cls_mgr.list_item_prefix, p_cls_mgr.list_item_prefix, icon_id, i))

    return _zen_sets_active_mode_ids


def _get_zen_sets_groups(p_scene, context):
    global _zen_sets_group_items

    p_new_items = []

    p_cls_mgr = get_sets_mgr(p_scene)
    if p_cls_mgr:
        p_new_items = [
            (g.layer_name, g.name, '', i)
            for i, g in p_cls_mgr.get_current_group_pairs(context)]

    if _zen_sets_group_items != p_new_items:
        _zen_sets_group_items = p_new_items

    return _zen_sets_group_items


def _get_enum_group(self):
    p_scene = self
    p_cls_mgr = get_sets_mgr(p_scene)

    if p_cls_mgr:
        p_group_pair = p_cls_mgr.get_current_group_pair(bpy.context)
        if p_group_pair:
            global _zen_sets_group_items
            for item in _zen_sets_group_items:
                if item[0] == p_group_pair[1].layer_name:
                    return item[3]
    return -1


def _set_enum_group(self, value):
    p_scene = self
    p_cls_mgr = get_sets_mgr(p_scene)
    if p_cls_mgr:
        p_cls_mgr.set_list_index(p_scene, value)


def _activate_highlight():
    bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')


def _on_update_object_mode(self, context: bpy.types.Context):
    p_scene = self
    mode = p_scene.zen_object_collections_mode

    global _LAST_OBJECT_ACTIVE_MODE
    b_was_draw_enabled = False
    if _LAST_OBJECT_ACTIVE_MODE != mode:
        b_was_draw_enabled = is_draw_active()
        remove_all_handlers3d()
        _LAST_OBJECT_ACTIVE_MODE = mode

    if mode:
        p_mode_name = bpy.types.UILayout.enum_item_name(p_scene, 'zen_object_collections_mode', mode)
        bpy.ops.ed.undo_push(message='Zen Mode: ' + p_mode_name)

    if b_was_draw_enabled:
        bpy.app.timers.register(_activate_highlight)


def _on_update_mode(self, context: bpy.types.Context):
    p_scene = self

    mode = p_scene.zen_sets_active_mode

    global _LAST_ACTIVE_MODE
    b_was_draw_enabled = False
    if _LAST_ACTIVE_MODE != mode:
        b_was_draw_enabled = is_draw_active()
        remove_all_handlers3d()
        _LAST_ACTIVE_MODE = mode

    for e in ELEMENT_MODES:
        if mode in e:
            setattr(p_scene, f'zen_sets_last_{e[0]}_mode', mode)
            break

    addon_prefs = get_prefs()
    if addon_prefs.common.sync_with_mesh_selection:
        global _ACTIVE_MODE_LOCK
        if _ACTIVE_MODE_LOCK is False:
            p_cls_mgr = get_sets_mgr(p_scene)
            if p_cls_mgr:
                if context.tool_settings.mesh_select_mode[:] != p_cls_mgr.get_mesh_select_mode():
                    _ACTIVE_MODE_LOCK = True
                    p_cls_mgr.set_mesh_select_mode(context)
                    _ACTIVE_MODE_LOCK = False

    if mode:
        p_mode_name = bpy.types.UILayout.enum_item_name(p_scene, 'zen_sets_active_mode', mode)
        bpy.ops.ed.undo_push(message='Zen Mode: ' + p_mode_name)

    if b_was_draw_enabled:
        bpy.app.timers.register(_activate_highlight)


def _get_enum_element_mode(self):
    p_scene = self
    p_cls_mgr = get_sets_edit_mgr(p_scene)
    if p_cls_mgr:
        for item in p_scene.bl_rna.properties['zen_sets_element_mode'].enum_items:
            if item.identifier == p_cls_mgr.id_display_element:
                return item.value
    else:
        return 0


def _set_enum_element_mode(self, value):
    p_scene = self

    new_mode = None

    for item in p_scene.bl_rna.properties['zen_sets_element_mode'].enum_items:
        if item.value == value:
            addon_prefs = get_prefs()

            if item.identifier in {'blgroup', 'blgroup_u'}:
                if getattr(addon_prefs.modes, f'enable_{item.identifier}'):
                    new_mode = item.identifier
            else:
                new_mode = _get_direct_element_mode_by_index(p_scene, value)

            break

    if new_mode is not None:
        if p_scene.zen_sets_active_mode != new_mode:
            p_scene.zen_sets_active_mode = new_mode


def _get_direct_element_mode_by_index(p_scene, index):
    if index >= 0 and index < len(ELEMENT_MODES):
        addon_prefs = get_prefs()

        b_enabled_sets = getattr(addon_prefs.modes, f'enable_{ELEMENT_MODES[index][0]}')
        b_enabled_parts = getattr(addon_prefs.modes, f'enable_{ELEMENT_MODES[index][1]}')

        id_unique = _get_unique_mode(p_scene)
        if (id_unique == 1 and b_enabled_parts) or (b_enabled_parts and not b_enabled_sets):
            return ELEMENT_MODES[index][1]
        if (id_unique == 0 or b_enabled_sets):
            return ELEMENT_MODES[index][0]

    return None


def _get_last_element_mode_by_index(p_scene, index):
    if index >= 0 and index < len(ELEMENT_MODES):
        addon_prefs = get_prefs()

        enabled_sets = []
        enabled_parts = []
        for v in ELEMENT_MODES[index]:
            if getattr(addon_prefs.modes, f'enable_{v}'):
                if v.endswith('_u'):
                    enabled_parts.append(v)
                else:
                    enabled_sets.append(v)

        # use 0 index, because 'last element' property uses only 0
        last_mode = getattr(p_scene, f'zen_sets_last_{ELEMENT_MODES[index][0]}_mode')

        b_enabled_parts = len(enabled_parts) > 0
        b_enabled_sets = len(enabled_sets) > 0

        if ((last_mode in enabled_parts and b_enabled_parts) or
           (b_enabled_parts and not b_enabled_sets)):
            if last_mode == '' and len(enabled_parts) > 0:
                last_mode = enabled_parts[0]
            return last_mode
        elif last_mode in enabled_sets or b_enabled_sets:
            if last_mode == '' and len(enabled_sets) > 0:
                last_mode = enabled_sets[0]
            return last_mode
    return None


def _get_unique_mode(self):
    p_scene = self
    p_cls_mgr = get_sets_mgr(p_scene)
    if p_cls_mgr:
        return 1 if p_cls_mgr.is_unique else 0
    else:
        return 0


def _enqueue_unique_mode(p_scene, value):

    mode = p_scene.zen_sets_active_mode

    if value == 1:
        if not mode.endswith('_u'):
            mode += '_u'
    else:
        if mode.endswith('_u'):
            mode = mode.replace('_u', '')

    if mode:
        addon_prefs = get_prefs()
        if not getattr(addon_prefs.modes, f'enable_{mode}'):
            p_linked = t_ZEN_SETS_ELEMENT_LINKS.get(mode, '')
            if p_linked:
                if getattr(addon_prefs.modes, f'enable_{p_linked}'):
                    mode = p_linked

        if mode != p_scene.zen_sets_active_mode:
            p_scene.zen_sets_active_mode = mode


def _set_unique_mode(self, value):
    p_scene = self
    p_cls_mgr = get_sets_mgr(p_scene)
    if p_cls_mgr:
        bpy.app.timers.register(functools.partial(_enqueue_unique_mode, p_scene, value))


_was_mode = None
_was_draw_cache_edit = None
_was_draw_cache_object = None


def _notify_edit_mode(handle):
    # may be set only once !
    if ZenLocks.is_notify_edit_locked():
        ZenLocks.unlock_notify_edit()
    else:
        Log.debug('OBJECT MODE:', bpy.context.mode)
        global _was_mode
        global _was_draw_cache_edit
        global _was_draw_cache_object
        new_mode = bpy.context.mode

        ZenLocks.clear_all_draw_tool_help_flags()

        if _was_mode == 'EDIT_MESH':
            if not is_draw_active():
                _was_draw_cache_edit = None
        elif _was_mode == 'OBJECT':
            if not is_draw_active():
                _was_draw_cache_object = None

        addon_prefs = get_prefs()

        p_scene = bpy.context.scene

        p_cur_mgr = get_sets_mgr(p_scene)

        for f in sets_factories:
            p_cls_mgr = f.get_mgr()
            if new_mode == 'EDIT_MESH':
                if getattr(addon_prefs.modes, f'enable_{p_cls_mgr.id_group}'):
                    if p_cls_mgr == p_cur_mgr:
                        p_cls_mgr.check_for_new_layers(p_scene, bpy.context.objects_in_mode)
                    else:
                        _ObjectCacheStore.add_pending_check_mesh_layers_manager(p_cls_mgr)

                    if p_cls_mgr.id_group == _was_draw_cache_edit:
                        _activate_highlight()
            else:
                if is_draw_handler_enabled(p_cls_mgr):
                    _was_draw_cache_edit = p_cls_mgr.id_group

                remove_draw_handler(p_cls_mgr, bpy.context)

        for f in obj_collection_factories:
            p_cls_mgr = f.get_mgr()
            if new_mode == 'OBJECT':
                p_cls_mgr.check_update_list(bpy.context.scene)
                if p_cls_mgr.id_group == _was_draw_cache_object:
                    _activate_highlight()
            else:
                if is_draw_handler_enabled(p_cls_mgr):
                    _was_draw_cache_object = p_cls_mgr.id_group

                remove_draw_handler(p_cls_mgr, bpy.context)

        _was_mode = new_mode


def _notify_tool_changed(handle):
    try:
        ctx = bpy.context
        s_mode = ctx.mode
        if s_mode in {'EDIT_MESH', 'OBJECT'}:
            global _notify_tool_cache

            b_draw_highlight = False
            b_draw_uv = False

            if s_mode == 'EDIT_MESH':
                _id_uv = getattr(ctx.workspace.tools.from_space_image_mode('UV', create=False), 'idname', None)
                was_uv = _notify_tool_cache.get('UV', None)
                if was_uv != _id_uv:
                    # switched to UV tab
                    if was_uv is None and _id_uv is not None:
                        p_cls_mgr = get_sets_mgr(ctx.scene)
                        if p_cls_mgr.is_uv_display_active():
                            for p_obj in ctx.objects_in_mode:
                                mark_groups_modified(p_cls_mgr, p_obj, modes={'UV'})
                                check_update_cache(p_cls_mgr, p_obj)
                            update_areas_in_all_screens()

                    _notify_tool_cache['UV'] = _id_uv
                    if isinstance(_id_uv, str) and _id_uv.startswith('zsts.select_'):
                        b_draw_highlight = True
                        b_draw_uv = True

            _id = getattr(ctx.workspace.tools.from_space_view3d_mode(s_mode, create=False), 'idname', None)
            if _notify_tool_cache[s_mode] != _id:
                _notify_tool_cache[s_mode] = _id
                if isinstance(_id, str) and _id.startswith('zsts.select_'):
                    b_draw_highlight = True

            if b_draw_highlight and get_prefs().common.auto_highlight and is_draw_active() is False:
                if b_draw_uv:
                    addon_prefs = get_prefs()
                    addon_prefs.uv_options.display_uv = True
                bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')
                update_areas_in_all_screens()
    except Exception as e:
        Log.error('TOOL CHANGED:', e)


def _subscribe_rna_common_types():
    bpy.msgbus.subscribe_rna(
        key=_key_subscribe_to_edit_mode,
        owner=_handle_subcribe_to,
        args=(_handle_subcribe_to,),
        notify=_notify_edit_mode,
        options={"PERSISTENT", }
    )

    bpy.msgbus.subscribe_rna(
        key=(bpy.types.WorkSpace, 'tools'),
        owner=_handle_subcribe_to,
        args=(_handle_subcribe_to,),
        notify=_notify_tool_changed,
        options={"PERSISTENT", }
    )


@persistent
def _load_scene_handler(dummy):
    Log.debug("Load Scene Handler:", bpy.data.filepath)
    remove_all_handlers3d()
    reset_all_draw_cache()
    clear_notify_tool_cache()

    _subscribe_rna_common_types()

    addon_prefs = get_prefs()
    addon_prefs.common.display_mesh_mode = {'CAGE'}
    addon_prefs.common.auto_update_draw_cache = True

    p_scene = bpy.context.scene
    if p_scene:
        if getattr(addon_prefs.modes, f'enable_{p_scene.zen_sets_active_mode}') is False:
            for id in get_sets_ids():
                if getattr(get_prefs().modes, f'enable_{id}'):
                    p_scene.zen_sets_active_mode = id
                    break
        p_sets_mgr = get_sets_edit_mgr(p_scene)
        if p_sets_mgr:
            p_sets_mgr.build_lookup_table(bpy.context)

        # Update Collection UIList Cache
        from .objects.collection_object_sets import ZsCollectionLayerManager

        ZsCollectionLayerManager.update_collections(p_scene)

        if addon_prefs.object_options.sync_with_collection_color_tag:
            ZsCollectionLayerManager.sync_all_collections_icons(bpy.context)


_was_active_view_layer_collection = None
_LOCK_SET_LAYER_COLLECTION = False


def _on_object_is_updated(scene):
    Log.debug('OBJECT Update:', timer())

    from .objects.collection_object_sets import ZsCollectionLayerManager
    p_cls_mgr = ZsCollectionLayerManager
    if p_cls_mgr:

        p_cls_mgr.check_update_list(scene)

        global _LOCK_SET_LAYER_COLLECTION
        if not _LOCK_SET_LAYER_COLLECTION:
            _LOCK_SET_LAYER_COLLECTION = True
            try:
                global _was_active_view_layer_collection
                v_layer = bpy.context.view_layer
                if v_layer.active_layer_collection != _was_active_view_layer_collection:
                    Log.debug(
                        'On layer changed =====>',
                        v_layer.active_layer_collection.name if v_layer.active_layer_collection else 'NONE',
                        ':', timer())
                    _was_active_view_layer_collection = v_layer.active_layer_collection

                    if not ZenLocks.is_lay_col_update_one_locked():
                        was_index = p_cls_mgr.get_list_index(scene)
                        if _was_active_view_layer_collection:
                            new_index = p_cls_mgr._index_of_layer_collection(v_layer, _was_active_view_layer_collection)
                        else:
                            new_index = -1

                        if new_index != was_index:
                            p_cls_mgr.set_list_index(scene, new_index)
                    else:
                        ZenLocks.unlock_lay_col_update_one()
            finally:
                _LOCK_SET_LAYER_COLLECTION = False


@persistent
def _depsgraph_update_post(scene):
    if ZenLocks.is_depsgraph_update_locked():
        return

    if ZenLocks.is_depsgraph_update_one_locked():
        ZenLocks.unlock_depsgraph_update_one()
        return

    addon_prefs = get_prefs()
    if addon_prefs.common.sync_with_mesh_selection:
        """ change mode if 'mesh_select_mode' was changed """
        global _ACTIVE_MODE_LOCK
        if _ACTIVE_MODE_LOCK is False:

            new_mode = None

            curr_mesh_select = bpy.context.tool_settings.mesh_select_mode
            global _LAST_MESH_SELECT
            for i, v in enumerate(curr_mesh_select):
                if v and (_LAST_MESH_SELECT is None or v != _LAST_MESH_SELECT[i]):
                    new_mode = _get_last_element_mode_by_index(scene, i)
                    break

            for i, v in enumerate(curr_mesh_select):
                _LAST_MESH_SELECT[i] = v

            # DO NOT CHANGE CHECK TYPE
            if new_mode:
                is_mode_enabled = getattr(addon_prefs.modes, f'enable_{new_mode}')
                if is_mode_enabled:
                    _ACTIVE_MODE_LOCK = True
                    if scene.zen_sets_active_mode != new_mode:
                        scene.zen_sets_active_mode = new_mode
                    _ACTIVE_MODE_LOCK = False

    if bpy.context.mode == 'EDIT_MESH':
        _ObjectCacheStore.on_edit_mesh_is_updated(scene)
        p_cls_mgr = get_sets_mgr(scene)
        if p_cls_mgr == ZsVGroupLayerManager:
            vertexGroupsIndexSync.set_active_object(bpy.context)
        elif p_cls_mgr == ZsFaceMapsLayerManager:
            faceMapsIndexSync.set_active_object(bpy.context)
    elif bpy.context.mode == 'OBJECT':
        _on_object_is_updated(scene)


_select_group_context = {}


def _zen_sets_idle_select_group():

    global _select_group_context
    if len(_select_group_context) == 0:
        return

    context = bpy.context

    p_cls_mgr = get_sets_mgr(bpy.context.scene)
    if p_cls_mgr:

        addon_prefs = get_prefs()

        was_selection = {}

        mode = _select_group_context.get('mode', None)
        if mode is None:
            return

        is_object = mode == 'OBJECT'
        is_mesh_edit = mode == 'EDIT_MESH'

        area = _select_group_context.get('area', None)
        if area is None:
            return

        b_is_uv_area = area is not None and area.type == 'IMAGE_EDITOR'
        b_is_uv_area_not_sync = b_is_uv_area and not p_cls_mgr.is_uv_sync()

        p_group_pair = p_cls_mgr.get_current_group_pair(context)
        if p_group_pair is None:
            return

        _, p_group = p_group_pair
        s_layer_name = p_group.layer_name

        if addon_prefs.list_options.selection_follow_act_group:
            if bpy.ops.zsts.select_only.poll():
                bpy.ops.zsts.select_only(_select_group_context)
        else:
            if s_layer_name != '':
                was_selection = p_cls_mgr.store_selection(bpy.context, b_is_uv_area_not_sync)

        if addon_prefs.list_options.auto_frame_selected:
            was_locked = ZenLocks.is_depsgraph_update_locked()
            try:
                ZenLocks.lock_depsgraph_update()
                if len(was_selection):
                    if is_mesh_edit:
                        # do not use Blender deselect operators
                        for p_obj, v in was_selection.items():
                            bm = v[0]
                            uv_layer = v[1]
                            if b_is_uv_area_not_sync and uv_layer is None:
                                continue
                            p_cls_mgr._do_set_or_append_group_to_selection(
                                p_obj, bm, s_layer_name, b_is_uv_area_not_sync, uv_layer, True)
                    elif is_object:
                        p_objects = p_group.get_objects(context)
                        bpy.ops.object.select_all(action='DESELECT')
                        for p_obj in p_objects:
                            try:
                                if p_obj.visible_get():
                                    p_obj.select_set(True)
                            except Exception:
                                pass
                if area:
                    try:
                        for region in area.regions:
                            if region.type == 'WINDOW':
                                override = _select_group_context.copy()
                                override['region'] = region

                                if b_is_uv_area:
                                    bpy.ops.image.view_selected(override)
                                else:
                                    bpy.ops.view3d.view_selected(override)

                                break
                    except Exception as e:
                        Log.error(e)

                if len(was_selection):
                    p_cls_mgr.restore_selection(
                        context, b_is_uv_area_not_sync, was_selection)
            finally:
                if not was_locked:
                    ZenLocks.unlock_depsgraph_update()


def _on_scene_group_index_changed(scene, context, id_group):

    p_cls_mgr = get_sets_mgr(scene)

    global _LOCK_SET_LAYER_COLLECTION
    if not _LOCK_SET_LAYER_COLLECTION:
        if ZenPolls.is_object_and_collection_mode(context):
            _LOCK_SET_LAYER_COLLECTION = True
            try:
                Log.debug('On index changed ======>', scene.zen_sets_object_list_index, ':', timer())
                if p_cls_mgr:
                    idx = p_cls_mgr.get_list_index(scene)
                    p_list = p_cls_mgr.get_list(scene)
                    if idx in range(len(p_list)):
                        lay_col = p_cls_mgr.get_internal_layer_collection(bpy.context, idx)
                        if lay_col:
                            if not lay_col.exclude:
                                if bpy.context.view_layer.active_layer_collection != lay_col:
                                    bpy.context.view_layer.active_layer_collection = lay_col
                                    global _was_active_view_layer_collection
                                    _was_active_view_layer_collection = lay_col
            finally:
                _LOCK_SET_LAYER_COLLECTION = False

    b_is_edit_mode = context.mode == 'EDIT_MESH'

    if p_cls_mgr == ZsVGroupLayerManager:
        vertexGroupsIndexSync.set_active_list_index(context, p_cls_mgr.get_list_index(scene))
    elif p_cls_mgr == ZsFaceMapsLayerManager:
        faceMapsIndexSync.set_active_list_index(context, p_cls_mgr.get_list_index(scene))

    if ZenLocks.is_group_index_locked(id_group):
        ZenLocks.unlock_group_index(id_group)
        return

    addon_prefs = get_prefs()
    if b_is_edit_mode or ZenPolls.is_object_and_simple_mode(context):
        if (addon_prefs.list_options.selection_follow_act_group or
                addon_prefs.list_options.auto_frame_selected):
            if bpy.app.timers.is_registered(_zen_sets_idle_select_group):
                bpy.app.timers.unregister(_zen_sets_idle_select_group)
            global _select_group_context
            _select_group_context.clear()
            _select_group_context.update(context.copy())
            bpy.app.timers.register(_zen_sets_idle_select_group, first_interval=0.04)

    update_areas_in_all_screens()


def _closure_on_scene_group_index_changed(id_group):
    return lambda scene, context: _on_scene_group_index_changed(scene, context, id_group)


def _on_collection_zen_color_update(self, context):
    addon_prefs = get_prefs()
    if addon_prefs.object_options.sync_with_collection_color_tag:
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.set_collection_icon_from_color(self)


def _on_addon_started():
    ctx = bpy.context
    if ctx:
        p_scene = bpy.context.scene
        if p_scene:
            p_cls_mgr = get_sets_object_mgr(p_scene)
            if p_cls_mgr:
                p_cls_mgr.update_collections(p_scene)


def register():
    """ Register classes """

    for c in classes:
        bpy.utils.register_class(c)

    for f in _factories:
        if hasattr(f, 'register'):
            f.register()

        for c in f.classes:
            bpy.utils.register_class(c)

    for f in sets_factories:
        for c in f.classes:
            bpy.utils.register_class(c)

        p_cls_mgr = f.get_mgr()

        p_scene_list_type = f.get_ui_list() if hasattr(f, 'get_ui_list') else ZSTSSceneListGroup
        p_object_list_type = f.get_obj_ui_list() if hasattr(f, 'get_obj_ui_list') else ZSTSObjectListGroup

        setattr(bpy.types.Object, p_cls_mgr.list_prop_name(),
                bpy.props.CollectionProperty(type=p_object_list_type))
        setattr(bpy.types.Scene, p_cls_mgr.list_prop_name(),
                bpy.props.CollectionProperty(type=p_scene_list_type))
        setattr(bpy.types.Scene, p_cls_mgr.active_list_prop_name(),
                bpy.props.IntProperty(name=f"Group name, {p_cls_mgr.id_element}(s), object(s)",
                default=-1,
                update=_closure_on_scene_group_index_changed(p_cls_mgr.id_group)))

    bpy.types.Scene.zen_object_collections_mode = bpy.props.EnumProperty(
        name='Object Collections Mode',
        items=(
            ('obj_sets', 'Collections', 'Object Collections', 'OUTLINER_COLLECTION', 0),
            ('obj_simple_sets', 'Sets', 'Object Sets', zs_icon_get(ZIconsType.Sets), 1),
            ('obj_simple_parts', 'Parts', 'Object Parts', zs_icon_get(ZIconsType.Parts), 2),
        ),
        default='obj_sets',
        update=_on_update_object_mode
    )
    bpy.types.Object.zen_tag = bpy.props.StringProperty(name='Zen Sets Tag', default="")
    bpy.types.Object.zen_color = bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_COLOR_NAME,
        subtype='COLOR_GAMMA',
        size=3,
        default=(0.0, 0.0, 0.0),
        min=0, max=1,
    )
    bpy.types.Collection.zen_expanded = bpy.props.BoolProperty(name='Expanded', default=True)
    bpy.types.Collection.zen_color = bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_GROUP_COLOR_NAME,
        subtype='COLOR_GAMMA',
        size=3,
        default=(0.0, 0.0, 0.0),
        min=0, max=1,
        update=_on_collection_zen_color_update
    )
    for f in obj_collection_factories:
        for c in f.classes:
            bpy.utils.register_class(c)

        if hasattr(f, 'register'):
            f.register()

        p_cls_mgr = f.get_mgr()

        object_options = {'SKIP_SAVE'} if p_cls_mgr.id_group == 'obj_sets' else {'ANIMATABLE'}

        setattr(bpy.types.Scene, p_cls_mgr.list_prop_name(),
                bpy.props.CollectionProperty(type=f.get_ui_list(), options=object_options))
        setattr(bpy.types.Scene, p_cls_mgr.active_list_prop_name(),
                bpy.props.IntProperty(name=f"Group name, {p_cls_mgr.id_element}(s)",
                default=-1,
                update=_closure_on_scene_group_index_changed(p_cls_mgr.id_group),
                options=object_options))

    bpy.types.Scene.zen_sets_active_mode = bpy.props.EnumProperty(
        name=ZsLabels.PROP_ACTIVE_MODE_NAME,
        description=ZsLabels.PROP_ACTIVE_MODE_DESC,
        items=_get_group_ids,
        update=_on_update_mode)

    bpy.types.Scene.zen_sets_last_vert_mode = bpy.props.StringProperty(
        name="Last Vert Mode",
        description="Last Vert Mode of Zen Sets")

    bpy.types.Scene.zen_sets_last_edge_mode = bpy.props.StringProperty(
        name="Last Edge Mode",
        description="Last Edge Mode of Zen Sets")

    bpy.types.Scene.zen_sets_last_face_mode = bpy.props.StringProperty(
        name="Last Face Mode",
        description="Last Face Mode of Zen Sets")

    bpy.types.Scene.zen_sets_groups = bpy.props.EnumProperty(
        name="Groups",
        description="Groups of Zen Sets",
        items=_get_zen_sets_groups,
        get=_get_enum_group,
        set=_set_enum_group,
        default=None,
        options={'ANIMATABLE', 'SKIP_SAVE'}
    )

    bpy.types.Scene.zen_sets_element_mode = bpy.props.EnumProperty(
        name=ZsLabels.PROP_ELEMENT_MODE_NAME,
        description=ZsLabels.PROP_ELEMENT_MODE_DESC,
        items=[
            ("vert", "Vertex", "", zs_icon_get(ZIconsType.Vert), 0),
            ("edge", "Edge", "", zs_icon_get(ZIconsType.Edge), 1),
            ("face", "Face", "", zs_icon_get(ZIconsType.Face), 2),

            ("blgroup", "Vertex Groups", "", 'GROUP_VERTEX', 3),
            ("blgroup_u", "Face Maps", "", 'FACE_MAPS', 4),
        ],
        get=_get_enum_element_mode,
        set=_set_enum_element_mode,
        options={'ANIMATABLE', 'SKIP_SAVE'}
    )

    bpy.types.Scene.zen_sets_unique_mode = bpy.props.EnumProperty(
        name=ZsLabels.PROP_UNIQUE_MODE_NAME,
        description=ZsLabels.PROP_UNIQUE_MODE_DESC,
        items=[
            ("SETS", "Sets", "", zs_icon_get(ZIconsType.Sets), 0),
            ("PARTS", "Parts", "", zs_icon_get(ZIconsType.Parts), 1),
        ],
        get=_get_unique_mode,
        set=_set_unique_mode,
        options={'ANIMATABLE', 'SKIP_SAVE'}
    )

    bpy.types.Scene.zen_sets_last_smart_layer = bpy.props.StringProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    ZenLocks.set_lock_draw_overlay(1.0)

    try:
        bpy.utils.register_tool(ZsObjectWorkSpaceTool)
        bpy.utils.register_tool(ZsUvWorkSpaceTool)
        if _USE_SINGLE_TOOL:
            bpy.utils.register_tool(ZsBoxWorkSpaceTool)
        else:
            bpy.utils.register_tool(ZsBoxWorkSpaceTool, group=True)
            bpy.utils.register_tool(ZsCircleWorkSpaceTool, after=ZsBoxWorkSpaceTool.bl_idname)
            bpy.utils.register_tool(ZsLassoWorkSpaceTool, after=ZsBoxWorkSpaceTool.bl_idname)
    except Exception as e:
        Log.error('Register tool:', e)

    global _handle_subcribe_to
    if _handle_subcribe_to is None:
        _handle_subcribe_to = object()
        _subscribe_rna_common_types()

    if _load_scene_handler not in bpy.app.handlers.load_post:
        bpy.app.handlers.load_post.append(_load_scene_handler)

    if _depsgraph_update_post not in bpy.app.handlers.depsgraph_update_post:
        bpy.app.handlers.depsgraph_update_post.append(_depsgraph_update_post)

    bpy.app.timers.register(_on_addon_started, persistent=True)


def unregister():
    """ Unregister classes """

    cleanup_overlay_handles()

    remove_all_handlers3d()

    if _load_scene_handler in bpy.app.handlers.load_post:
        bpy.app.handlers.load_post.remove(_load_scene_handler)

    if _depsgraph_update_post in bpy.app.handlers.depsgraph_update_post:
        bpy.app.handlers.depsgraph_update_post.remove(_depsgraph_update_post)

    try:
        if _USE_SINGLE_TOOL:
            bpy.utils.unregister_tool(ZsBoxWorkSpaceTool)
        else:
            bpy.utils.unregister_tool(ZsBoxWorkSpaceTool)
            bpy.utils.unregister_tool(ZsCircleWorkSpaceTool)
            bpy.utils.unregister_tool(ZsLassoWorkSpaceTool)
        bpy.utils.unregister_tool(ZsUvWorkSpaceTool)
        bpy.utils.unregister_tool(ZsObjectWorkSpaceTool)
    except Exception as e:
        Log.error('Unregister tool:', e)

    for f in reversed(_factories):
        for c in reversed(f.classes):
            bpy.utils.unregister_class(c)

        if hasattr(f, 'unregister'):
            f.unregister()

    for f in reversed(sets_factories):
        p_cls_mgr = f.get_mgr()

        p_group_prop = getattr(bpy.types.Object, p_cls_mgr.list_prop_name())
        if p_group_prop:
            del p_group_prop

        p_group_prop = getattr(bpy.types.Scene, p_cls_mgr.list_prop_name())
        if p_group_prop:
            del p_group_prop

        p_index_prop = getattr(bpy.types.Scene, p_cls_mgr.active_list_prop_name())
        if p_index_prop:
            del p_index_prop

        for c in reversed(f.classes):
            bpy.utils.unregister_class(c)

    for f in reversed(obj_collection_factories):
        p_cls_mgr = f.get_mgr()

        if hasattr(f, 'unregister'):
            f.unregister()

        p_group_prop = getattr(bpy.types.Scene, p_cls_mgr.list_prop_name())
        if p_group_prop:
            del p_group_prop

        p_index_prop = getattr(bpy.types.Scene, p_cls_mgr.active_list_prop_name())
        if p_index_prop:
            del p_index_prop

        for c in reversed(f.classes):
            bpy.utils.unregister_class(c)

    for c in reversed(classes):
        bpy.utils.unregister_class(c)

    del bpy.types.Scene.zen_sets_active_mode
    del bpy.types.Scene.zen_sets_last_vert_mode
    del bpy.types.Scene.zen_sets_last_edge_mode
    del bpy.types.Scene.zen_sets_last_face_mode
    del bpy.types.Scene.zen_sets_groups
    del bpy.types.Scene.zen_sets_element_mode
    del bpy.types.Scene.zen_sets_unique_mode

    del bpy.types.Scene.zen_object_collections_mode
    del bpy.types.Scene.zen_sets_last_smart_layer

    del bpy.types.Collection.zen_expanded
    del bpy.types.Collection.zen_color

    del bpy.types.Object.zen_color
    del bpy.types.Object.zen_tag

    global _handle_subcribe_to
    if _handle_subcribe_to is not None:
        bpy.msgbus.clear_by_owner(_handle_subcribe_to)
        _handle_subcribe_to = None


if __name__ == "__main__":
    pass
