# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

# blender
import bpy
import bmesh

from typing import Tuple
from timeit import default_timer as timer

# local
from ..labels import ZsLabels
from ..vlog import Log

from .factories import get_sets_mgr, get_sets_ids, sets_factories, obj_simple_factories
from .draw_sets import (
    check_update_cache_on_change,
    is_draw_active,
    mark_groups_modified,
    remove_draw_handler,
    is_draw_handler_enabled,
    check_update_cache,
    _do_add_draw_handler,
    _do_remove_draw_handler,
    reset_all_draw_cache)
from .pie import ZsPieFactory

from ..preferences import (
    get_prefs,
    get_addon_version,
    display_element_modes, display_compact_element_modes, display_object_element_modes,
    ZsOperatorOptions)
from ..ico import zs_icon_get, ZIconsType
from ..blender_zen_utils import (
    ZenPolls,
    fix_undo_push_edit_mode,
    is_property_modified,
    update_areas_in_all_screens,
    get_mimic_operator_layout,
    get_command_props,
    draw_last_operator_properties,
    ZenLocks,
    ZsUiConstants)
from ..progress import start_progress, update_progress, end_progress

from .smart_rename import get_last_selected_smart_group, set_last_selected_smart_group

from .export_utils import export_zen_group_to_vertex_color


def message_box(context, message="", title="Information", icon='INFO'):

    def draw(self, context):
        self.layout.label(text=message)

    context.window_manager.popup_menu(draw, title=title, icon=icon)


def warning_box(context, message=''):
    message_box(context=context, message=message, title="WARNING", icon="ERROR")


def error_box(context, message=''):
    message_box(context=context, message=message, title="ERROR", icon="ERROR")


""" POLL FUNCTIONS """


def _poll_context_by_current_groups_len(context):
    p_cls_mgr = get_sets_mgr(context.scene)
    return True if (p_cls_mgr and len(p_cls_mgr.get_current_group_pairs(context))) else False


def _poll_context_has_selected_group(context):
    p_cls_mgr = get_sets_mgr(context.scene)
    return p_cls_mgr.get_current_list_index(context) != -1 if p_cls_mgr else False


def _poll_context_has_selected_mesh_elements(context):
    p_cls_mgr = get_sets_mgr(context.scene)
    return (
        p_cls_mgr.get_current_list_index(context) != -1 and
        p_cls_mgr.get_context_selected_count(context)) if p_cls_mgr else False


def _poll_context_has_active_group(context):
    p_cls_mgr = get_sets_mgr(context.scene)
    return p_cls_mgr.get_current_list_index(context) != -1 if p_cls_mgr else False


""" API CLASSES """


class ZsElementGroup(bpy.types.PropertyGroup):
    item: bpy.props.IntProperty(name='Item')


""" OPERATORS """


class ZSTS_OT_NewItem(bpy.types.Operator):
    """Add a new item to the list."""
    bl_idname = 'zsts.new_group'
    bl_description = ZsLabels.OT_NEW_ITEM_DESC
    bl_label = ZsLabels.OT_NEW_ITEM_LABEL
    bl_options = {'REGISTER'}

    # UNDO with UILists can not work properly !!!

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        res = {'CANCELLED'}
        if p_cls_mgr:
            res = p_cls_mgr.execute_NewGroup(self, context)
        return res


class ZSTS_OT_DeleteGroup(bpy.types.Operator):
    """Delete the selected item from the list."""
    bl_idname = 'zsts.del_group'
    bl_description = ZsLabels.OT_DEL_ITEM_DESC
    bl_label = ZsLabels.OT_DEL_ITEM_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_DeleteGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_Append(bpy.types.Operator):
    bl_idname = 'zsts.append_to_group'
    bl_label = ZsLabels.OT_APPEND_ITEM_LABEL
    bl_description = ZsLabels.OT_APPEND_ITEM_DESC
    bl_options = {'REGISTER', 'UNDO'}

    def draw_ex(self, layout, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr.id_group == 'blgroup':
            row = layout.row()
            p_cls_mgr._do_draw_vertex_weight(context, row)

    def draw(self, context):
        self.draw_ex(self.layout, context)

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_mesh_elements(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_AppendToGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_AssignUVBorders(bpy.types.Operator):
    bl_idname = 'zsts.assign_uv_borders_to_group'
    bl_label = ZsLabels.OT_ASSIGN_UV_BORDERS_LABEL
    bl_description = ZsLabels.OT_ASSIGN_UV_BORDERS_DESC
    bl_options = {'REGISTER', 'UNDO'}

    use_pinned_group: bpy.props.BoolProperty(
        name="Pinned",
        description='Store UV Borders in the pinned Group',
        default=True)
    pinned_name: bpy.props.StringProperty(
        name="Name",
        description='Name of the pinned Group for storing UV Borders',
        default='UV Borders')

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH'

    def execute(self, context):
        p_scene = context.scene
        p_cls_mgr = get_sets_mgr(p_scene)
        if p_cls_mgr:
            p_group = p_cls_mgr._get_scene_group(p_scene)

            if self.use_pinned_group:
                p_list = p_cls_mgr.get_list(p_scene)
                idx = p_cls_mgr._index_of_group_name(p_list, self.pinned_name)
                if idx == -1:
                    layer_name = p_cls_mgr.create_unique_layer_name()
                    p_group = p_cls_mgr.ensure_group_in_scene(
                        p_scene,
                        layer_name,
                        self.pinned_name,
                        (0.3, 1.0, 0.3))

                    p_cls_mgr.set_list_index(p_scene, p_cls_mgr._index_of_layer(p_list, layer_name))

            if p_group:
                p_cls_mgr.set_mesh_select_mode(context)

                for p_obj in context.objects_in_mode:
                    p_cls_mgr.set_uv_borders_to_edge_group(p_obj, p_group)
                    mark_groups_modified(p_cls_mgr, p_obj)

                p_cls_mgr._add_draw_handler()
            else:
                self.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

            return {'FINISHED'}

        return {'CANCELLED'}


class ZSTS_OT_MoveItem(bpy.types.Operator):
    """Move an item in the list."""
    bl_idname = 'zsts.move_group'
    bl_description = ZsLabels.OT_MOVE_ITEM_DESC
    bl_label = ZsLabels.OT_MOVE_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    direction: bpy.props.EnumProperty(
        name='Direction',
        items=(
            ('UP', 'Up', ""),
            ('DOWN', 'Down', ""),
        )
    )

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_MoveItem(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_AssignToPinned(bpy.types.Operator):
    bl_idname = 'zsts.assign_to_pinned_group'
    bl_label = ZsLabels.OT_ASSIGN_ITEM_PINNED_LABEL
    bl_description = ZsLabels.OT_ASSIGN_ITEM_PINNED_DESC
    bl_options = {'REGISTER', 'UNDO'}

    group_name: ZsOperatorOptions.get_tool_assign_pinned_group_name()

    group_color: ZsOperatorOptions.get_tool_assign_pinned_group_color(options={'HIDDEN', 'SKIP_SAVE'})

    @classmethod
    def poll(cls, context):
        if context.mode == 'EDIT_MESH':
            p_cls_mgr = get_sets_mgr(context.scene)
            if p_cls_mgr:
                if not p_cls_mgr.is_unique:
                    return True
        return False

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_AssignToGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_Assign(bpy.types.Operator):
    bl_idname = 'zsts.assign_to_group'
    bl_label = ZsLabels.OT_ASSIGN_ITEM_LABEL
    bl_description = ZsLabels.OT_ASSIGN_ITEM_DESC
    bl_options = {'REGISTER', 'UNDO'}

    identifier: bpy.props.StringProperty(
        name=ZsLabels.PROP_LAYER_NAME,
        description=ZsLabels.PROP_LAYER_NAME_DESC,
        default="",
        options={"HIDDEN", "SKIP_SAVE"}
    )

    group_name: bpy.props.StringProperty(
        name=ZsLabels.PROP_GROUP_NAME,
        description=ZsLabels.PROP_GROUP_NAME_DESC,
        default="",
        options={"HIDDEN", "SKIP_SAVE"}
    )

    group_color: bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_COLOR_NAME,
        subtype='COLOR_GAMMA',
        size=3,
        default=(0.0, 0.0, 0.0),
        min=0, max=1,
        options={"HIDDEN", "SKIP_SAVE"}
    )

    group_indices: bpy.props.CollectionProperty(
        name='Group Indices',
        type=ZsElementGroup,
        options={"HIDDEN", "SKIP_SAVE"}
    )

    group_mode: bpy.props.EnumProperty(
        name='Group Mode',
        items=[
            ('SELECTED', 'Selected', 'Selected Elements'),
            ('INDICES', 'Indices', 'Elements by Indices')
        ],
        options={"HIDDEN", "SKIP_SAVE"},
        default='SELECTED'
    )

    def draw_ex(self, layout: bpy.types.UILayout, context: bpy.types.Context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr.id_group == 'blgroup':
            row = layout.row()
            p_cls_mgr._do_draw_vertex_weight(context, row)

    def draw(self, context):
        self.draw_ex(self.layout, context)

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH'

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_AssignToGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_SelectOnlyGroup(bpy.types.Operator):
    """ Select only items marked with current group """
    bl_idname = 'zsts.select_only'
    bl_description = ZsLabels.OT_SELECT_ACTIVE_ITEM_DESC
    bl_label = ZsLabels.OT_SELECT_ACTIVE_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_SelectOnlyGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_SelectAppendGroup(bpy.types.Operator):
    """ Select group """
    bl_idname = 'zsts.select_append'
    bl_description = ZsLabels.OT_SELECT_ITEM_DESC
    bl_label = ZsLabels.OT_SELECT_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    # Object Props
    nested: bpy.props.BoolProperty(name='Nested', default=True)

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def draw(self, context):
        layout = self.layout
        if context.mode == 'OBJECT':
            layout.prop(self, 'nested')

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_SelectAppendGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_SelectGroup(bpy.types.Operator):
    """ Select only items marked with current group """
    bl_idname = 'zsts.select_group'
    bl_description = ZsLabels.OT_SELECT_GROUP_DESC
    bl_label = ZsLabels.OT_SELECT_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    # All
    clear_selection: bpy.props.BoolProperty(name="Clear Selection", default=True)
    # Object Props
    nested: bpy.props.BoolProperty(name='Nested', default=True)

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    def draw_ex(self, layout: bpy.types.UILayout, context: bpy.types.Context):
        layout.use_property_split = True
        layout.prop(self, 'clear_selection')
        if ZenPolls.is_object_and_collection_mode(context):
            layout.prop(self, 'nested')

    def draw(self, context):
        self.draw_ex(self.layout, context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            if self.clear_selection:
                return p_cls_mgr.execute_SelectOnlyGroup(self, context)
            else:
                return p_cls_mgr.execute_SelectAppendGroup(self, context)

        return {'CANCELLED'}


class ZSTS_OT_DeselectGroup(bpy.types.Operator):
    """ Remove group from mesh selection """
    bl_idname = 'zsts.deselect_group'
    bl_description = ZsLabels.OT_DESELECT_ITEM_DESC
    bl_label = ZsLabels.OT_DESELECT_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    # Object Props
    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw_ex(self, layout: bpy.types.UILayout, context: bpy.types.Context):
        layout.use_property_split = True
        if ZenPolls.is_object_and_collection_mode(context):
            layout.prop(self, 'nested')

    def draw(self, context):
        self.draw_ex(self.layout, context)

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_DeselectGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_SelectUngroupped(bpy.types.Operator):
    """ Select nongroupped items """
    bl_idname = 'zsts.select_ungroupped'
    bl_description = ZsLabels.OT_SELECT_UNGROUPPED_DESC
    bl_label = ZsLabels.OT_SELECT_UNGROUPPED_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return not ZenPolls.is_object_and_collection_mode(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_SelectUngroupped(self, context)

        return {'CANCELLED'}


class ZSTS_OT_IntersectGroup(bpy.types.Operator):
    """ Intersect group with mesh selection """
    bl_idname = 'zsts.intersect_group'
    bl_description = ZsLabels.OT_INTERSECT_ITEM_DESC
    bl_label = ZsLabels.OT_INTERSECT_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    # Object Props
    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw(self, context):
        layout = self.layout
        if ZenPolls.is_object_and_collection_mode(context):
            get_mimic_operator_layout(self, layout, 'nested')

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_IntersectGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_GroupSiblings(bpy.types.Operator):
    bl_idname = 'zsts.smart_select'
    bl_label = ZsLabels.OT_SIBLINGS_ITEM_LABEL
    bl_description = ZsLabels.OT_SIBLINGS_ITEM_DESC
    bl_options = {'REGISTER', 'UNDO'}

    select_only_active_group: bpy.props.BoolProperty(
        name=ZsLabels.PROP_SEL_ONLY_ONE_GROUP_NAME,
        description=ZsLabels.PROP_SEL_ONLY_ONE_GROUP_DESC,
        default=False
    )

    keep_active_group: bpy.props.BoolProperty(
        name=ZsLabels.PROP_KEEP_ACTIVE_GROUP_NAME,
        description=ZsLabels.PROP_KEEP_ACTIVE_GROUP_DESC,
        default=False
    )

    last_selected_group: bpy.props.StringProperty(
        name='Last selected',
        description='Last selected Group name',
        get=get_last_selected_smart_group,
        set=set_last_selected_smart_group,
        options={'SKIP_SAVE'}
    )

    @classmethod
    def poll(cls, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.get_context_selected_count(context) if p_cls_mgr else False

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_GroupSiblings(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_SmartIsolate(bpy.types.Operator):
    bl_idname = 'zsts.smart_isolate'
    bl_label = 'Smart Isolate'
    bl_description = 'Isolate smart selected Elements | Unhide All'
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def draw(self, context: bpy.types.Context):
        layout = self.layout
        draw_last_operator_properties(context, 'zsts.smart_select', layout)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_SmartIsolate(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_HideGroup(bpy.types.Operator):
    """ Hide Group """
    bl_idname = 'zsts.hide_group'
    bl_description = ZsLabels.OT_HIDE_GROUP_DESC
    bl_label = ZsLabels.OT_HIDE_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    # Object Props
    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw_ex(self, layout: bpy.types.UILayout, context: bpy.types.Context):
        if ZenPolls.is_object_and_collection_mode(context):
            layout.prop(self, 'nested')

        addon_prefs = get_prefs()
        addon_prefs.vgroups_options.draw_mask(layout, context)

    def draw(self, context):
        self.draw_ex(self.layout, context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            hide_changed, has_elements = p_cls_mgr.set_active_group_hidden_state(context, True, self)
            if not hide_changed:
                if has_elements:
                    self.report({'WARNING'}, 'Group is already hidden!')
                else:
                    if context.mode == 'EDIT_MESH':
                        self.report({'WARNING'}, 'Can not hide empty group!')
                    else:
                        self.report({'WARNING'}, 'Can not hide group!')
            return {'FINISHED'}
        else:
            return {'CANCELLED'}


class ZSTS_OT_UnhideGroup(bpy.types.Operator):
    """ Unhide Group """
    bl_idname = 'zsts.unhide_group'
    bl_description = ZsLabels.OT_UNHIDE_GROUP_DESC
    bl_label = ZsLabels.OT_UNHIDE_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    select: bpy.props.BoolProperty(name='Select', default=True)

    # Object Props
    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw_ex(self, layout: bpy.types.UILayout, context: bpy.types.Context):
        if ZenPolls.is_object_and_collection_mode(context):
            layout.prop(self, 'nested')

        layout.prop(self, 'select')

        addon_prefs = get_prefs()
        addon_prefs.vgroups_options.draw_mask(layout, context)

    def draw(self, context):
        self.draw_ex(self.layout, context)

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            hide_changed, has_elements = p_cls_mgr.set_active_group_hidden_state(context, False, self)
            if not hide_changed:
                if has_elements:
                    if ZenPolls.is_object_and_simple_mode(context):
                        self.report({'WARNING'}, 'Group is not hidden or Check Collections!')
                    else:
                        self.report({'WARNING'}, 'Group is not hidden!')
                else:
                    if context.mode == 'EDIT_MESH':
                        self.report({'WARNING'}, 'Can not unhide empty group!')
                    else:
                        self.report({'WARNING'}, 'Can not unhide group!')
            return {'FINISHED'}
        else:
            return {'CANCELLED'}


t_invert_hide_status = {}


class ZSTS_OT_InvertHideGroup(bpy.types.Operator):
    """ Invert Hide Group """
    bl_idname = 'zsts.invert_hide_group'
    bl_description = ZsLabels.OT_INVERT_HIDE_GROUP_DESC
    bl_label = ZsLabels.OT_INVERT_HIDE_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    select: bpy.props.BoolProperty(name='Select', default=True)

    # FIX: bug in Blender with EnumProperty and Last Op Props
    def update_submode(self, context: bpy.types.Context):
        global t_invert_hide_status
        t_invert_hide_status[context.as_pointer()] = self.submode

    submode: bpy.props.EnumProperty(
        items=[
            ('UNDEFINED', 'Undefined', ''),
            ('ISOLATE', 'Isolate', ''),
            ('UNHIDE_ALL', 'Unhide All', '')],
        default='UNDEFINED',
        update=update_submode)

    def draw_ex(self, layout: bpy.types.UILayout, context: bpy.types.Context):
        s_submode_id = t_invert_hide_status.get(context.as_pointer(), '')
        if s_submode_id:
            layout.label(text=layout.enum_item_name(self, 'submode', s_submode_id))

        layout.prop(self, 'select')

        addon_prefs = get_prefs()
        addon_prefs.vgroups_options.draw_mask(layout, context)

    def draw(self, context: bpy.types.Context):
        self.draw_ex(self.layout, context)

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr and self.submode != 'UNDEFINED':
            if self.submode == 'ISOLATE':
                p_cls_mgr.set_active_group_invert_hide(context, self)
            elif self.submode == 'UNHIDE_ALL':
                b_is_uv = p_cls_mgr.is_uv_area_and_not_sync()
                p_cls_mgr.unhide_all(context, self.select or b_is_uv, self)
            return {'FINISHED'}
        else:
            return {'CANCELLED'}

    def invoke(self, context: bpy.types.Context, event: bpy.types.Event):
        self.submode = 'UNDEFINED'
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            hide_changed, has_elements = p_cls_mgr.set_active_group_invert_hide(context, self)
            if not hide_changed:
                if not has_elements:
                    self.report({'WARNING'}, 'Can not isolate empty group, so unhide all!')
                self.submode = 'UNHIDE_ALL'
                b_is_uv = p_cls_mgr.is_uv_area_and_not_sync()
                p_cls_mgr.unhide_all(context, self.select or b_is_uv, self)
            else:
                self.submode = 'ISOLATE'

            return {'FINISHED'}
        else:
            return {'CANCELLED'}


class ZSTS_OT_RemoveSelectionFromGroup(bpy.types.Operator):
    """ Remove mesh selected items from group """
    bl_idname = 'zsts.remove_from_group'
    bl_label = ZsLabels.OT_REMOVE_SEL_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def description(cls, context, properties):
        return 'Remove selected elements from ' + ('Active Group' if properties.mode == 'ACTIVE' else 'All Groups')

    mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', 'Remove selected Elements from Active Group'),
            ('ALL', 'All', 'Remove selected Elements from All Groups'),
        ],
        options={'HIDDEN', 'SKIP_SAVE'},
        default='ACTIVE')

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_mesh_elements(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_RemoveSelectionFromGroup(self, context)
        return {'CANCELLED'}


class ZSTS_OT_DeleteEmptyGroups(bpy.types.Operator):
    """ Delete empty groups """
    bl_idname = 'zsts.delete_empty_groups'
    bl_description = ZsLabels.OT_DELETE_EMPTY_GROUP_DESC
    bl_label = ZsLabels.OT_DELETE_EMPTY_GROUP_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            return len(p_cls_mgr.get_list(context.scene))
        return False

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_DeleteEmptyGroups(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_DeleteGroupsCombo(bpy.types.Operator):
    """ Delete all groups or delete empty """
    bl_idname = 'zsts.delete_groups_combo'
    bl_description = ZsLabels.OT_DELETE_GROUPS_COMBO_DESC
    bl_label = ZsLabels.OT_DELETE_GROUPS_COMBO_LABEL
    bl_options = {'REGISTER'}

    delete_mode: bpy.props.EnumProperty(name="Delete Mode",
                                        items=[
                                            ('EMPTY', 'Empty Groups', ''),
                                            ('ALL', 'All Groups', ''),
                                        ],
                                        default='EMPTY')

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def execute(self, context):
        return bpy.ops.zsts.delete_groups_internal(
            'INVOKE_DEFAULT',
            delete_mode=self.delete_mode)


class ZSTS_OT_DeleteGroupsInternal(bpy.types.Operator):
    """ Internal delete combo groups """
    bl_idname = 'zsts.delete_groups_internal'
    bl_label = 'Choose what to Delete and press OK'
    bl_options = {'REGISTER'}

    delete_mode: bpy.props.EnumProperty(
        name="Delete Mode",
        items=[
            ('EMPTY', 'Empty Groups', ''),
            ('ALL', 'All Groups', ''),
        ],
        default='EMPTY')

    def execute(self, context):
        op_delete = context.window_manager.operator_properties_last('zsts.delete_groups_combo')
        if op_delete:
            op_delete.delete_mode = self.delete_mode

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            if self.delete_mode == 'ALL':
                return p_cls_mgr.execute_DeleteAllGroups(self, context)
            else:
                if ZSTS_OT_DeleteEmptyGroups.poll(context):
                    return p_cls_mgr.execute_DeleteEmptyGroups(self, context)

        return {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "delete_mode", expand=True)
        layout.separator()


class ZSTS_OT_DrawHighlight(bpy.types.Operator):
    bl_idname = 'zsts.draw_highlight'
    bl_label = ZsLabels.OT_DRAW_HIGHLIGHT_LABEL
    bl_description = ZsLabels.OT_DRAW_HIGHLIGHT_DESCRIPTION
    bl_options = {'REGISTER'}

    mode: bpy.props.EnumProperty(
        items=[("NONE", "None", ""),
               ("ON", "On", ""),
               ("OFF", "Off", "")],
        default='NONE',
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    def modal(self, context, event):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            return {'PASS_THROUGH'}
        else:
            return {'CANCELLED'}

    def cancel(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            remove_draw_handler(p_cls_mgr, context)

    def invoke(self, context, event):
        if context.mode in {'EDIT_MESH', 'OBJECT'}:
            p_cls_mgr = get_sets_mgr(context.scene)
            if p_cls_mgr:
                b_modal = False
                if not is_draw_handler_enabled(p_cls_mgr):
                    if self.mode != 'OFF':
                        interval = timer()
                        depsgraph = context.evaluated_depsgraph_get()
                        for p_obj in context.objects_in_mode:
                            Log.debug('Checking cache:', p_obj.name)
                            interval_obj = timer()
                            p_obj_eval = p_obj.evaluated_get(depsgraph)
                            check_update_cache_on_change(p_cls_mgr, p_obj, p_obj_eval)
                            Log.debug('Check cache completed:', p_obj.name, timer() - interval_obj)
                        _do_add_draw_handler(p_cls_mgr, context)

                        update_areas_in_all_screens()

                        context.window_manager.modal_handler_add(self)
                        b_modal = True
                        elapsed = timer() - interval
                        Log.debug('ZSTS_OT_DrawHighlight - ON:', elapsed)
                else:
                    if self.mode != 'ON':
                        _do_remove_draw_handler(p_cls_mgr.id_group, context)

                return {'RUNNING_MODAL'} if b_modal else {'CANCELLED'}

        return {'CANCELLED'}


class ZSTS_OT_MarkSeams(bpy.types.Operator):
    """ Mark Seams """
    bl_idname = 'zsts.mark_seams'
    bl_description = ZsLabels.OT_MARK_SEAMS_IN_GROUP_DESK
    bl_label = ZsLabels.OT_MARK_SEAMS_IN_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_MarkSeams(self, context, True) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_ClearSeams(bpy.types.Operator):
    """ Clear Seams """
    bl_idname = 'zsts.clear_seams'
    bl_description = ZsLabels.OT_CLEAR_SEAMS_IN_GROUP_DESK
    bl_label = ZsLabels.OT_CLEAR_SEAMS_IN_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_has_active_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_MarkSeams(self, context, False) if p_cls_mgr else {'CANCELLED'}


class ZSTS_OT_GroupLinked(bpy.types.Operator):
    """ Zen Sets Group Linked """
    bl_idname = 'zsts.group_linked'
    bl_description = ZsLabels.OT_GROUP_LINKED_DESK
    bl_label = ZsLabels.OT_GROUP_LINKED_LABEL
    bl_options = {'REGISTER'}
    bl_ui_units_x = 14

    delimit: bpy.props.EnumProperty(
        name='Delimit',
        items=[
            ('NORMAL', 'Normal', ''),
            ('MATERIAL', 'Material', ''),
            ('SEAM', 'Seam', ''),
            ('SHARP', 'Sharp', ''),
            ('UV', 'UVs', ''),
            # User Custom Delimiter
            ('CUSTOM', 'Custom', ''),
        ],
        options={'ENUM_FLAG'},
        default={'NORMAL'})

    def get_custom_delimiter_name(self):
        if self.custom_delimiter != '':
            s_NOT_FOUND = '(NOT_FOUND)'
            op_props = get_command_props(self.custom_delimiter)
            if op_props.bl_op_cls:
                try:
                    if op_props.bl_label:
                        return op_props.bl_label
                    elif op_props.bl_desc:
                        return op_props.bl_desc
                    else:
                        return op_props.bl_op_cls.idname_py()
                except Exception:
                    return s_NOT_FOUND
            else:
                return s_NOT_FOUND
        return ''

    custom_delimiter_name: bpy.props.StringProperty(
        name='Custom Operator',
        description='Python Operator cmd that will be used as mesh selector',
        get=get_custom_delimiter_name
    )

    custom_delimiter: bpy.props.StringProperty(
        name='Custom Delimiter Operator'
    )

    min_group_size: bpy.props.IntProperty(
        name='Min Group Size',
        min=1,
        default=1
    )

    def get_delimits(self) -> Tuple[set, bool]:
        return set(self.delimit).difference(['CUSTOM']), 'CUSTOM' in self.delimit

    def is_zen_uv_similar_present(self):
        try:
            return (
                bpy.context.scene.tool_settings.use_uv_select_sync and
                bpy.ops.uv.zenuv_select_similar.poll()
            )
        except Exception:
            return False

    @classmethod
    def poll(cls, context):
        return len(context.objects_in_mode)

    def modal(self, context, event):
        if event.type in {'RIGHTMOUSE', 'ESC'}:
            self.cancel(context)
            return {'CANCELLED'}

        if event.type == 'TIMER':
            if not self.process_step(context):
                if not self.prepare_object(context):
                    self.cancel(context)
                    return {'FINISHED'}
            pass

        return {'RUNNING_MODAL'}

    def execute_custom_delimiter(self):
        op_props = get_command_props(self.custom_delimiter)
        eval(op_props.cmd)

    def prepare_object(self, context):
        edit_objs = context.objects_in_mode
        for i, obj in enumerate(edit_objs):
            if self.obj is None:
                self.obj = obj
                break
            if self.obj == obj:
                bm = self.p_cls_mgr._get_bm(obj)
                bm.faces.ensure_lookup_table()
                for item in bm.faces:
                    item.hide_set(item.index in self.hidden_indices)

                bmesh.update_edit_mesh(self.obj.data, loop_triangles=False, destructive=False)
                self.p_cls_mgr.update_all_obj_groups_count(self.obj, no_lookup=True)

                if i < len(edit_objs) - 1:
                    self.obj = edit_objs[i + 1]
                else:
                    self.obj = None
                break

        if self.obj is not None:
            bm = self.p_cls_mgr._get_bm(obj)
            bm.faces.ensure_lookup_table()
            self.hidden_indices = set(item.index for item in bm.faces if item.hide)

        self.i_index = 0
        return False if self.obj is None else True

    def process_step(self, context):
        p_scene = context.scene
        bm = self.p_cls_mgr._get_bm(self.obj)
        i_face_count = len(bm.faces)

        i_was_index = self.i_index

        me = self.obj.data

        result = False

        default_delimits, use_custom = self.get_delimits()

        for self.i_index in range(self.i_index, i_face_count, 1):
            if bm.faces[self.i_index].hide:
                continue

            bm.faces[self.i_index].select_set(True)

            was_sel = me.total_face_sel

            if len(default_delimits):
                bpy.ops.mesh.select_linked(delimit=default_delimits)

            try:
                if use_custom:
                    self.execute_custom_delimiter()

                sel_dif = me.total_face_sel - was_sel
                if sel_dif >= self.min_group_size:
                    s_layer_name = self.p_cls_mgr.create_unique_layer_name()

                    p_color = self.p_cls_mgr._gen_new_color(self.p_scene_list)
                    i_list_index = self.p_cls_mgr.add_layer_to_list(p_scene, s_layer_name, p_color)
                    self.p_cls_mgr.set_selection_to_new_group(self.obj, self.p_scene_list[i_list_index], self.i_index, i_face_count)
                elif me.total_face_sel == 0:
                    bm.faces[self.i_index].select_set(True)

                bpy.ops.mesh.hide(unselected=False)

                bm = self.p_cls_mgr._get_bm(self.obj)
                bm.faces.ensure_lookup_table()
            except Exception as e:
                Log.error(e)
                self.last_error = 'Auto Groups failed! See console for details!'
                self.i_index = i_face_count

            result = True
            break

        self.i_processed += (self.i_index - i_was_index)

        i_percent = int((self.i_processed + 1) / self.i_total_count * 100)
        update_progress(context, i_percent)

        return result

    def execute(self, context):

        custom_active = 'CUSTOM' in self.delimit
        custom_alert = (
            self.custom_delimiter_name == '(NOT_FOUND)' or
            self.custom_delimiter_name == '' and custom_active)
        if custom_alert:
            self.report({'ERROR'}, 'Custom Operator has errors! Check properties!')
            return {'CANCELLED'}

        p_scene = context.scene
        self.p_cls_mgr = get_sets_mgr(p_scene)
        if self.p_cls_mgr:
            self.i_total_count = 0
            self.obj = None
            self.p_scene_list = self.p_cls_mgr.get_list(p_scene)
            self.i_list_index = -1
            self.i_processed = 0
            self.last_error = ''

            for p_obj in context.objects_in_mode:
                bm = self.p_cls_mgr._get_bm(p_obj)
                bm.faces.ensure_lookup_table()
                self.i_total_count += len(bm.faces)

            if self.i_total_count:

                if self.i_total_count < 15000:
                    return self.p_cls_mgr.execute_GroupLinked(self, context)

                bpy.ops.mesh.select_all(action='DESELECT')
                if self.prepare_object(context):
                    ZenLocks.lock_lookup_build()
                    ZenLocks.lock_depsgraph_update()

                    addon_prefs = get_prefs()
                    self.was_auto_highlight = addon_prefs.common.auto_highlight
                    addon_prefs.common.auto_highlight = False
                    bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='OFF')

                    wm = context.window_manager
                    self._timer = wm.event_timer_add(0.001, window=context.window)
                    wm.modal_handler_add(self)

                    start_progress(context)
                    self.report({'INFO'}, "Press 'ESC' to cancel ...")

                    return {'RUNNING_MODAL'}

        return {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def cancel(self, context):
        if hasattr(self, '_timer'):
            wm = context.window_manager
            wm.event_timer_remove(self._timer)

            ZenLocks.unlock_lookup_build()
            ZenLocks.unlock_depsgraph_update()

            bpy.ops.mesh.reveal(select=False)
            self.p_cls_mgr.build_lookup_table(context)

            self.p_cls_mgr.set_list_index(context.scene, self.i_list_index)

            fix_undo_push_edit_mode('Auto Groups')
            for p_obj in context.objects_in_mode:
                mark_groups_modified(self.p_cls_mgr, p_obj)
                check_update_cache(self.p_cls_mgr, p_obj)

            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')
            get_prefs().common.auto_highlight = self.was_auto_highlight

            end_progress(context)

            last_err = getattr(self, 'last_error', '')
            if last_err != '':
                self.report({'ERROR'}, last_err)
                self.last_error = ''
            else:
                self.report({'INFO'}, 'Auto Groups completed!')

            update_areas_in_all_screens()

    def draw(self, context):
        layout = self.layout

        _ = get_mimic_operator_layout(self, layout, 'delimit', expand=True, factor=0.5)

        get_mimic_operator_layout(self, layout, 'min_group_size', factor=0.5)

        box = layout.box()
        box.active = 'CUSTOM' in self.delimit
        box.alert = (
            self.custom_delimiter_name == '(NOT_FOUND)' or
            self.custom_delimiter_name == '' and box.active)

        get_mimic_operator_layout(self, box, 'custom_delimiter_name', factor=0.5)
        box.prop(self, 'custom_delimiter', text='')


class ZSTS_OT_SplitToObjects(bpy.types.Operator):
    """ Zen Sets Split to objects """
    bl_idname = 'zsts.split_to_objects'
    bl_description = ZsLabels.OT_SPLIT_TO_OBJ_DESK
    bl_label = ZsLabels.OT_SPLIT_TO_OBJ_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', ''),
            ('ALL', 'All', ''),
        ],
        default='ACTIVE')
    is_mesh_cut: bpy.props.BoolProperty(name=ZsLabels.PROP_IS_MESH_CUT, default=True)
    join: bpy.props.BoolProperty(
        name=ZsLabels.PROP_JOIN_MESH_LABEL,
        description=ZsLabels.PROP_JOIN_MESH_DESC,
        default=True)

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_SplitToObjects(self, context) if p_cls_mgr else {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "group_mode", expand=True)

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            if p_cls_mgr.is_unique:
                layout.prop(self, "is_mesh_cut")
        layout.prop(self, "join")


class ZSTS_OT_GlobalCleanup(bpy.types.Operator):
    """ Zen Sets Global Cleanup """
    bl_idname = 'zsts.global_cleanup'
    bl_description = ZsLabels.OT_GLOBAL_CLEANUP_DESC
    bl_label = ZsLabels.OT_GLOBAL_CLEANUP_LABEL
    bl_options = {'REGISTER'}

    delete_all_collections: bpy.props.BoolProperty(
        name='Delete All Collections',
        default=True
    )

    @classmethod
    def poll(self, context):
        return context.mode in {'EDIT_MESH', 'OBJECT'}

    def execute(self, context):
        p_cur_cls_mgr = get_sets_mgr(context.scene)
        if p_cur_cls_mgr:
            remove_draw_handler(p_cur_cls_mgr, context)

            p_scene = context.scene

            operator = self

            if context.mode == 'EDIT_MESH':
                def _internal_global_cleanup(self, context):
                    for f in sets_factories:
                        p_cls_mgr = f.get_mgr()
                        p_list = p_cls_mgr.get_list(p_scene)
                        for obj in context.objects_in_mode:
                            operator.report({'INFO'}, f'Start cleaning obj:[{obj.name}] mode:[{p_cls_mgr.id_group}]')
                            p_cls_mgr._do_cleanup_object(obj)
                        p_list.clear()
                        p_cls_mgr.set_list_index(p_scene, -1)

                p_cur_cls_mgr._perform_all_scene_edit(context, _internal_global_cleanup)

                for f in sets_factories:
                    p_cls_mgr = f.get_mgr()
                    p_cls_mgr.build_lookup_table(context)

                fix_undo_push_edit_mode('Global Cleanup')
                for p_obj in context.objects_in_mode:
                    mark_groups_modified(p_cur_cls_mgr, p_obj)
                    check_update_cache(p_cur_cls_mgr, p_obj)
            elif context.mode == 'OBJECT':

                from .objects.collection_object_sets import ZsCollectionLayerManager

                p_scene_collection = p_scene.collection
                all_collections = set(
                    it for it, _ in ZsCollectionLayerManager._iterate_layer_tree(p_scene_collection, None)
                    if it != p_scene_collection)

                for p_obj in p_scene.objects:
                    if self.delete_all_collections:
                        in_scene = False
                        for collection in p_obj.users_collection:
                            if collection != p_scene_collection:
                                collection.objects.unlink(p_obj)
                            else:
                                in_scene = True
                        if not in_scene:
                            p_scene_collection.objects.link(p_obj)

                    p_obj.zsos_uuid.clear()
                    p_obj.zsop_uuid = ''

                if self.delete_all_collections:
                    for it in all_collections:
                        if bpy.data.collections.get(it.name):
                            bpy.data.collections.remove(it)

                for f in obj_simple_factories:
                    p_cls_mgr = f.get_mgr()
                    p_list = p_cls_mgr.get_list(p_scene)
                    p_list.clear()
                    p_cls_mgr.set_list_index(p_scene, -1)

                bpy.ops.ed.undo_push(message='Global Cleanup')

            operator.report({'INFO'}, 'Cleanup completed!')

        return {'FINISHED'}

    def invoke(self, context: bpy.types.Context, event: bpy.types.Event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)


class ZSTS_OT_ObjectCleanup(bpy.types.Operator):
    """ Zen Sets Object Cleanup """
    bl_idname = 'zsts.object_cleanup'
    bl_description = ZsLabels.OT_OBJECT_CLEANUP_DESC
    bl_label = ZsLabels.OT_OBJECT_CLEANUP_LABEL
    bl_options = {'INTERNAL'}

    obj_name: bpy.props.StringProperty()

    def execute(self, context):
        if self.obj_name == '':
            return {'CANCELLED'}

        for p_obj in context.scene.objects:
            if p_obj.name == self.obj_name:
                for f in sets_factories:
                    p_cls_mgr = f.get_mgr()
                    p_cls_mgr._do_cleanup_object(p_obj)
                break
        return {'FINISHED'}


class ZSTS_OT_ExportGroupToVertexColor(bpy.types.Operator):
    """ Export Group To Vertex Colors """
    bl_idname = 'zsts.export_groups_to_vertex_colors'
    bl_description = ZsLabels.OT_EXPORT_GROUP_TO_VERT_COL_DESC
    bl_label = ZsLabels.OT_EXPORT_GROUP_TO_VERT_COL_LABEL
    bl_options = {'REGISTER'}

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', ''),
            ('ALL', 'All', ''),
        ],
        default='ACTIVE')

    generate_new_layer: bpy.props.BoolProperty(name="Generate New Vertex Color Layer", default=True)

    name_prefix: bpy.props.StringProperty(
        name='Name Prefix',
        description='Vertex Color Layer Prefix will be inserted in the Group name',
        default='Col.'
    )

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:

            b_generate_new_layer = self.generate_new_layer if p_cls_mgr.is_unique else True

            for p_obj in context.objects_in_mode:
                if self.group_mode == 'ACTIVE':
                    p_group_pair = p_cls_mgr.get_current_group_pair(context)
                    if p_group_pair:
                        export_zen_group_to_vertex_color(p_cls_mgr, p_obj, p_group_pair[1], b_generate_new_layer, self.name_prefix)
                else:
                    for i, p_group in p_cls_mgr.get_current_group_pairs(context):
                        export_zen_group_to_vertex_color(p_cls_mgr, p_obj, p_group, b_generate_new_layer, self.name_prefix)

                me = p_obj.data
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
                ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "group_mode", expand=True)

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            if p_cls_mgr.is_unique:
                layout.prop(self, "generate_new_layer")

        layout.prop(self, 'name_prefix')


class ZSTS_OT_ImportVertexColorToGroup(bpy.types.Operator):
    """ Import Vertex Colors To Group """
    bl_idname = 'zsts.import_vertex_colors_to_groups'
    bl_description = ZsLabels.OT_IMPORT_VERT_COL_TO_GROUP_DESC
    bl_label = ZsLabels.OT_IMPORT_VERT_COL_TO_GROUP_LABEL
    bl_options = {'REGISTER'}

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', ''),
            ('ALL', 'All', ''),
        ],
        default='ACTIVE')
    ignored_color: bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_IGNORED_COLOR_NAME,
        description=ZsLabels.PROP_IGNORED_COLOR_DESC,
        subtype='COLOR',
        size=3,
        default=(1.0, 1.0, 1.0),
        min=0, max=1,
    )
    use_ignored_color: bpy.props.BoolProperty(
        name=ZsLabels.PROP_IMPORT_IGNORED_COLOR_NAME,
        default=True)

    name_prefix: bpy.props.StringProperty(
        name='Name Prefix',
        description='Vertex Color Layer Prefix will be inserted in the Group name',
        default='Col.'
    )

    @classmethod
    def poll(cls, context):
        if context.mode == 'EDIT_MESH':
            p_obj = context.active_object
            if p_obj and p_obj.type == 'MESH':
                p_cls_mgr = get_sets_mgr(context.scene)
                if p_cls_mgr:
                    bm = p_cls_mgr._get_bm(context.active_object)
                    return len(bm.loops.layers.color)
        return False

    def execute(self, context):
        p_obj = context.active_object
        if p_obj:
            p_cls_mgr = get_sets_mgr(context.scene)
            if p_cls_mgr:
                p_cls_mgr.import_vertex_colors_to_zen_groups(
                    context.scene,
                    p_obj,
                    self.group_mode == 'ACTIVE',
                    self.ignored_color if self.use_ignored_color else None,
                    self.name_prefix
                )

                me = p_obj.data
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
                ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "group_mode", expand=True)
        row = layout.row(align=True)
        row.prop(self, 'use_ignored_color')
        if self.use_ignored_color:
            row.prop(self, "ignored_color", text='')

        layout.prop(self, 'name_prefix')


class ZSTS_OT_CopyToClipboard(bpy.types.Operator):
    """ Export Group To Clipboard """
    bl_idname = 'zsts.copy_to_clipboard'
    bl_description = ZsLabels.OT_COPY_TO_CLIPBOARD_DESC
    bl_label = ZsLabels.OT_COPY_TO_CLIPBOARD_LABEL
    bl_options = {'REGISTER'}

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', ''),
            ('ALL', 'All', ''),
        ],
        default='ACTIVE')

    filter_name: bpy.props.StringProperty(
        name='Filter By Name',
        description='Copy only Groups with names matching the Filter'
    )

    filter_case_sensitive: bpy.props.BoolProperty(
        name='Case Sensitive',
        description='Case Sensitive comparison',
        default=False
    )

    filter_type: bpy.props.EnumProperty(
        name='Filter Type',
        items=[
            ('WILDCARD', 'Wildcard', ''),
            ('REGEX', 'Regex', ''),
        ],
        default='WILDCARD')

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            try:
                active_layer_name = ''
                if self.group_mode == 'ACTIVE':
                    p_group_pair = p_cls_mgr.get_current_group_pair(context)
                    if p_group_pair:
                        active_layer_name = p_group_pair[1].layer_name

                bpy.context.window_manager.clipboard = p_cls_mgr.export_to_json(context, self, active_layer_name)

                return {'FINISHED'}
            except Exception as e:
                self.report({'ERROR'}, str(e))

        return {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "group_mode", expand=True)

        if self.group_mode == 'ALL':
            box = layout.box()
            box.use_property_split = True
            box.active = self.filter_name != ''
            box.prop(self, 'filter_name')
            row = box.row(align=True)
            row.prop(self, 'filter_type', expand=True)
            box.prop(self, 'filter_case_sensitive')


class ZSTS_OT_PasteClipboard(bpy.types.Operator):
    """ Import Group From Clipboard """
    bl_idname = 'zsts.paste_from_clipboard'
    bl_description = ZsLabels.OT_PASTE_FROM_CLIPBOARD_DESC
    bl_label = ZsLabels.OT_PASTE_FROM_CLIPBOARD_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return context.window_manager.clipboard.startswith('{"mode":')

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            try:
                p_cls_mgr.import_from_json(context, context.window_manager.clipboard)
            except Exception as e:
                Log.error(e)
                self.report({'ERROR'}, 'Clipboard import error!')

        return {'FINISHED'}


class ZSTS_OT_SelectObjectsInGroup(bpy.types.Operator):
    """ Zen Sets Select Objects in Group """
    bl_idname = 'zsts.select_scene_objects'
    bl_description = ZsLabels.OT_SELECT_OBJS_IN_GROUPS_DESC
    bl_label = ZsLabels.OT_SELECT_OBJS_IN_GROUPS_LABEL
    bl_options = {'REGISTER'}

    prop_unhide: bpy.props.BoolProperty(
        name=ZsLabels.PROP_UNHIDE_OBJECTS_NAME,
        description=ZsLabels.PROP_UNHIDE_OBJECTS_DESC,
        default=True)

    prop_make_selectable: bpy.props.BoolProperty(
        name=ZsLabels.PROP_MAKE_SELECTABLE_OBJECTS_NAME,
        description=ZsLabels.PROP_MAKE_SELECTABLE_OBJECTS_DESC,
        default=True
    )

    @classmethod
    def poll(cls, context):

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            if len(p_cls_mgr.get_current_group_pairs(context)) != 0 \
               and not p_cls_mgr.are_all_group_objects_selected(context):
                return True
        return False

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_SelectSceneObjectsInGroup(self, context) if p_cls_mgr else {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        col = layout.column(align=False)
        col.prop(self, "prop_unhide", expand=True)
        col.prop(self, "prop_make_selectable", expand=True)


class ZSTS_OT_RenameGroups(bpy.types.Operator):
    """ Zen Sets Rename Groups """
    bl_idname = 'zsts.rename_groups'
    bl_description = ZsLabels.OT_RENAME_GROUPS_DESC
    bl_label = ZsLabels.OT_RENAME_GROUPS_LABEL
    bl_options = {'REGISTER'}

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('VISIBLE', 'Visible', ''),
            ('ALL', 'All', ''),
        ],
        default='ALL')
    find: bpy.props.StringProperty(name='Find')
    replace: bpy.props.StringProperty(name='Replace')
    match_case: bpy.props.BoolProperty(name='Match Case', default=True)
    use_counter: bpy.props.BoolProperty(name='Counter', default=False)
    start_from: bpy.props.IntProperty(name='Start from', default=1)

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        return p_cls_mgr.execute_RenameGroups(self, context) if p_cls_mgr else {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "group_mode", expand=True)
        layout.prop(self, "find")
        layout.prop(self, "replace")
        layout.prop(self, "match_case")
        row = layout.row()
        row.prop(self, "use_counter")
        row.prop(self, "start_from")


class ZSTS_OT_InternalBuildLookup(bpy.types.Operator):
    """ Zen Sets Build Lookup """
    bl_idname = 'zsts.internal_build_lookup'
    bl_label = 'Zen Sets Build Lookup'
    bl_options = {'INTERNAL'}

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.build_lookup_table(context)
        return {'FINISHED'}


class ZSTS_OT_InternalHideGroupByIndex(bpy.types.Operator):
    """ Zen Sets Hide Group By Index """
    bl_idname = 'zsts.internal_hide_group_by_index'
    bl_label = 'Hide in Viewport'
    bl_description = (
        'Default - Hide | Unhide Group\n' +
        'CTRL - Isolate | Restore Group By Index\n' +
        'SHIFT - Unhide All')
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(name='Hide Group By Index', options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        items=[
            ('HIDE', 'Hide | Unhide', ''),
            ('ISOLATE', 'Isolate | Restore', ''),
            ('UNHIDE_ALL', 'Unhide All', '')],
        default='HIDE')

    submode: bpy.props.EnumProperty(
        items=[
            ('UNDEFINED', 'Undefined', ''),
            ('HIDE', 'Hide', ''),
            ('UNHIDE', 'Unhide', ''),
            ('ISOLATE', 'Isolate', ''),
            ('UNHIDE_ALL', 'Unhide All', '')],
        default='UNDEFINED')

    select: bpy.props.BoolProperty(name='Select', default=True)

    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw(self, context):
        layout = self.layout
        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))
        layout.prop(self, 'select')
        if ZenPolls.is_object_and_collection_mode(context):
            layout.prop(self, 'nested')

        addon_prefs = get_prefs()
        addon_prefs.vgroups_options.draw_mask(layout, context)

    def invoke_execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.check_validate_list(context)
            p_list = p_cls_mgr.get_list(context.scene)
            if self.group_index in range(len(p_list)):
                b_is_uv = p_cls_mgr.is_uv_area_and_not_sync()
                if self.mode == 'ISOLATE':
                    hide_changed, has_elements = p_cls_mgr.set_group_pair_invert_hide(
                        context, (self.group_index, p_list[self.group_index]), self)
                    self.submode = 'ISOLATE'
                    if not hide_changed:
                        if has_elements:
                            self.submode = 'UNHIDE_ALL'
                            p_cls_mgr.unhide_all(context, self.select or b_is_uv, self)
                        else:
                            self.report({'WARNING'}, 'Can not isolate empty group!')
                elif self.mode == 'UNHIDE_ALL':
                    self.submode = 'UNHIDE_ALL'
                    p_cls_mgr.unhide_all(context, self.select or b_is_uv, self)
                else:
                    hide_changed, has_elements, hide_state = p_cls_mgr.hide_group_by_index(context, self.group_index, self)

                    self.submode = 'HIDE' if hide_state else 'UNHIDE'

                    if not hide_changed:
                        status = 'hide' if hide_state else 'unhide'
                        status2 = 'hidden' if hide_state else 'unhidden'
                        if has_elements:
                            if ZenPolls.is_object_and_simple_mode(context):
                                self.report({'WARNING'}, f'Can not {status} Group! It may be already {status2} or Check Collections!')
                            else:
                                self.report({'WARNING'}, f'Can not {status} Group! It may be already {status2}')
                        else:
                            self.report({'WARNING'}, f'Can not {status} empty group!')
                return {'FINISHED'}
        return {'CANCELLED'}

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr and self.submode != 'UNDEFINED':
            p_cls_mgr.check_validate_list(context)
            p_list = p_cls_mgr.get_list(context.scene)
            if self.group_index in range(len(p_list)):
                b_is_uv = p_cls_mgr.is_uv_area_and_not_sync()
                if self.submode == 'ISOLATE':
                    hide_changed, has_elements = p_cls_mgr.set_group_pair_invert_hide(
                        context, (self.group_index, p_list[self.group_index]), self)
                elif self.submode == 'UNHIDE_ALL':
                    p_cls_mgr.unhide_all(context, self.select or b_is_uv, self)
                elif self.submode in {'HIDE', 'UNHIDE'}:
                    hide_state = self.submode == 'HIDE'
                    hide_changed, has_elements = p_cls_mgr.set_group_pair_hidden_state(
                        context, (self.group_index, p_list[self.group_index]), hide_state, self)
                return {'FINISHED'}
        return {'CANCELLED'}

    def invoke(self, context, event):
        self.mode = 'HIDE'
        self.submode = 'UNDEFINED'
        if event.type == 'LEFTMOUSE':
            if event.ctrl:
                self.mode = 'ISOLATE'
            elif event.shift:
                self.mode = 'UNHIDE_ALL'
        return self.invoke_execute(context)


class ZSTS_OT_ExportGroupColorsToPalette(bpy.types.Operator):
    """ Zen Sets Export Group Colors To Palette """
    bl_idname = 'zsts.export_group_colors_to_palette'
    bl_label = ZsLabels.OT_PALETTE_EXPORT_GROUP_COLORS_LABEL
    bl_description = ZsLabels.OT_PALETTE_EXPORT_GROUP_COLORS_DESC
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def _is_color_present(self, color, colors):
        for v_col in colors.values():
            if v_col.color == color:
                return True
        return False

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_scene = context.scene
            pal = p_scene.zsts_color_palette.palette
            if pal is None:
                bpy.ops.zsts.palette_add_new()
            pal = p_scene.zsts_color_palette.palette
            if pal:
                pairs = p_cls_mgr.get_current_group_pairs(context)
                if len(pairs):
                    for i, group in pairs:
                        if not self._is_color_present(group.group_color, pal.colors):
                            new_color = pal.colors.new()
                            new_color.color = group.group_color
                    if p_scene.zsts_color_palette.mode == 'AUTO':
                        p_scene.zsts_color_palette.mode = 'PAL_SEQ'
                    return {'FINISHED'}

        return {'CANCELLED'}


class ZSTS_OT_AssignPaletteColorToGroup(bpy.types.Operator):
    bl_idname = "zsts.palette_color_assign_to_group"
    bl_label = ZsLabels.OT_PALETTE_ASSIGN_COLOR_TO_GROUP_LABEL
    bl_description = ZsLabels.OT_PALETTE_ASSIGN_COLOR_TO_GROUP_DESC
    bl_options = {'UNDO'}

    group_layer: bpy.props.StringProperty(
        options={'HIDDEN', 'SKIP_SAVE'},
        default='')

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', ''),
            ('ALL', 'All', ''),
        ],
        default='ACTIVE')

    palette_mode: bpy.props.EnumProperty(
        name='Palette Mode',
        items=(
            ('PAL_SEQ', 'Sequence', 'Palette Sequence Color'),
            ('PAL_RND', 'Random', 'Palette Random Color')
        ),
        default='PAL_SEQ'
    )

    @classmethod
    def poll(cls, context):
        pal = context.scene.zsts_color_palette.palette
        return pal and len(pal.colors)

    def draw(self, context):
        p_scene = context.scene
        pal = p_scene.zsts_color_palette.palette
        if pal:
            layout = self.layout
            row = layout.row(align=True)
            row.prop(self, 'group_mode', expand=True)
            if self.group_mode == 'ALL':
                row = layout.row(align=True)
                row.prop(self, 'palette_mode', expand=True)
            box = layout.box()
            box.alignment = 'LEFT'
            col = layout
            cols_per_row = 10
            for i, color in enumerate(pal.colors):
                if i % cols_per_row == 0:
                    flow = box.column_flow(columns=cols_per_row, align=True)

                is_color_active = pal.colors.active == color
                op_icon = "LAYER_ACTIVE" if is_color_active else "LAYER_USED"
                col = flow.column(align=True)
                col.prop(color, 'color', text='', emboss=True)
                col.operator('zsts.palette_select_color',
                             text='', emboss=is_color_active, icon=op_icon).color_index = i

    def execute(self, context):
        p_scene = context.scene
        pal = p_scene.zsts_color_palette.palette
        if pal and pal.colors.active:
            p_cls_mgr = get_sets_mgr(p_scene)
            if p_cls_mgr:
                if self.group_mode == 'ACTIVE':
                    p_group = p_cls_mgr._get_group_by_layer(p_scene, self.group_layer)
                    if p_group:
                        p_group.group_color = pal.colors.active.color
                        return {'FINISHED'}
                else:
                    p_group_pairs = p_cls_mgr.get_current_group_pairs(context)
                    if len(p_group_pairs):
                        p_temp_list = []
                        for i, group in p_cls_mgr.get_current_group_pairs(context):
                            if self.palette_mode == 'PAL_SEQ':
                                group.group_color = p_cls_mgr._gen_new_color_seq(p_temp_list, pal.colors)
                            else:
                                group.group_color = p_cls_mgr._gen_new_color_rnd(p_temp_list, pal.colors)
                            p_temp_list.append(group)
                        return {'FINISHED'}

        return {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)


class ZSTS_OT_SetSculptMask(bpy.types.Operator):
    bl_idname = "mesh.zsts_set_sculpt_mask"
    bl_label = 'Set Sculpt Mask'
    bl_description = 'Set Sculpt Mask, Isolate or Clear All'
    bl_options = {'REGISTER', 'UNDO'}

    mode: bpy.props.EnumProperty(
        name='Mask Mode',
        items=[
            ('MASK', 'Mask', ''),
            ('UNMASK', 'Unmask', ''),
            ('ISOLATE', 'Isolate', ''),
            ('UNMASK_ALL', 'Unmask All', ''),
        ],
        default='MASK')

    submode: bpy.props.EnumProperty(
        name='Vertices Mode',
        items=[
            ('ACTIVE_GROUP', 'Active Group', 'Operation will be applied to Vertices in Active Group'),
            ('SELECTED', 'Selected', 'Operation will be applied to Selected Vertices')
        ],
        default='ACTIVE_GROUP')

    mask_value: bpy.props.FloatProperty(
        name='Mask Value',
        description='Mask Value will be set to Paint Mask mesh Layer',
        min=0.1,
        max=1.0,
        soft_min=0.1,
        soft_max=1.0,
        default=1.0
    )

    def draw(self, context: bpy.types.Context):
        layout = self.layout

        layout.use_property_split = True

        col = layout.column(align=True)
        col.prop(self, 'mode', expand=True)

        if self.mode != 'UNMASK_ALL':
            col = layout.column(align=True)
            col.prop(self, 'submode', expand=True)

            layout.prop(self, 'mask_value')

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_SetSculptMask(self, context)

        return {'FINISHED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)


class ZSTS_OT_ResetDrawCache(bpy.types.Operator):
    bl_idname = "zsts.reset_draw_cache"
    bl_label = 'Reset Draw Cache'
    bl_description = 'Reset draw cache for all mesh sets, collections and objects'
    bl_options = {'REGISTER'}

    def execute(self, context):

        was_draw_active = is_draw_active()
        if was_draw_active:
            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='OFF')

        reset_all_draw_cache()

        if was_draw_active:
            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')

        self.report({'INFO'}, 'Draw Cache has been reset!')

        return {'FINISHED'}


class ZSTS_OT_SelectMode(bpy.types.Operator):
    bl_idname = "zsts.select_mode"
    bl_label = 'Select Mode'
    bl_description = 'Select Zen Sets active Mode'
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        ZsPieFactory.mark_pie_cancelled()

        bpy.ops.wm.call_panel(name='ZSTS_PT_SelectModeMenu')

        return {'FINISHED'}


class ZSTS_OT_SetActiveGroup(bpy.types.Operator):
    bl_idname = "zsts.set_active_group"
    bl_label = 'Set Active Group'
    bl_description = 'Set active Zen Sets Group'
    bl_options = {'REGISTER', 'UNDO'}

    mode: bpy.props.EnumProperty(
        name='Mode',
        description='Set Active Group Mode',
        items=[
            ('GROUP_NAME', 'Group Name', 'Zen Sets Group Name'),
            ('IDENTIFIER', 'Identifier', 'Zen Sets Group Identifier')
        ],
        default='GROUP_NAME'
    )

    group: bpy.props.StringProperty(
        name='Group',
        description='Group Name or Identifier regarding to Set Mode',
        default=''
    )

    @classmethod
    def poll(cls, context):
        return context.mode in {'EDIT_MESH', 'OBJECT'}

    def execute(self, context):
        p_scene = context.scene
        p_cls_mgr = get_sets_mgr(p_scene)
        if p_cls_mgr:
            p_list = p_cls_mgr.get_list(p_scene)
            idx = -1
            if self.mode == 'GROUP_NAME':
                idx = p_cls_mgr._index_of_group_name(p_list, self.group)
            elif self.mode == 'IDENTIFIER':
                idx = p_cls_mgr._index_of_layer(p_list, self.group)

            if idx != -1:
                p_cls_mgr.set_list_index(p_scene, idx, lock=True)
                return {'FINISHED'}

        return {'CANCELLED'}


""" PANELS """


class ZsBasePanel:

    def draw_header(self, context):
        p_scene = context.scene
        custom_text = ''
        p_cls_mgr = get_sets_mgr(p_scene)
        if p_cls_mgr:
            custom_text = p_cls_mgr.get_draw_panel_header(context)

        layout = self.layout
        layout.label(text="Zen Sets" + custom_text)

    def draw(self, context):
        layout = self.layout
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            addon_prefs = get_prefs()

            is_edit_mesh = context.mode == 'EDIT_MESH'
            is_object_collection_mode = ZenPolls.is_object_and_collection_mode(context)
            if is_edit_mesh:
                if addon_prefs.common.compact_mode:
                    # !!! DEPRECATED !!!
                    row = layout.row(align=True)
                    row.alignment = 'EXPAND'

                    display_compact_element_modes(row, context, get_sets_ids())
                else:
                    row_element = layout.row(align=True)
                    row_unique = layout.row(align=True)

                    b_is_blender = p_cls_mgr.is_blender or addon_prefs.modes.enable_blgroup or addon_prefs.modes.enable_blgroup_u
                    row_blender = None
                    if b_is_blender:
                        row_bl = layout.row(align=True)
                        row_bl.label(text='Blender Groups')
                        row_blender = layout.row(align=True)

                    display_element_modes(row_element, row_blender, row_unique, context)
            else:
                row = layout.row(align=True)
                display_object_element_modes(row, context)

                if is_object_collection_mode:
                    row = layout.row(align=True)
                    row.prop(addon_prefs.list_options, "collection_list_mode", expand=False, text="")

            row_toolbar = layout.row(align=False)

            r_left = row_toolbar.row(align=True)
            b_is_highlight_enabled = is_draw_handler_enabled(p_cls_mgr)
            r_left.operator('zsts.draw_highlight', text='', depress=b_is_highlight_enabled, icon='OVERLAY')

            r_right = row_toolbar.row(align=True)
            r_right.alignment = 'RIGHT'
            if is_edit_mesh:
                r_sync = r_left.row(align=True)
                r_sync.prop(addon_prefs.common, 'sync_with_mesh_selection', icon_only=True, icon=ZsUiConstants.ZSTS_SYNC_MODE_ICON)
                r_sync.active = not p_cls_mgr.is_uv_area_and_not_sync()

            if not is_object_collection_mode:
                r_right.prop(addon_prefs.list_options, 'selection_follow_act_group', text='', icon='RESTRICT_SELECT_OFF')
                r_right.prop(addon_prefs.list_options, 'auto_frame_selected', text='', icon='FULLSCREEN_EXIT')
                r_right.separator()

            if is_object_collection_mode:
                r_right.prop(addon_prefs.list_options, 'tree_view', text='', icon='OUTLINER')
            r_right.prop(addon_prefs.list_options, 'display_objects_info', text='', icon='OUTLINER_OB_GROUP_INSTANCE')
            if is_object_collection_mode:
                r_right.prop(addon_prefs.list_options, 'display_excluded_groups_info', text='', icon='CHECKBOX_HLT')

            r_right.prop(addon_prefs.list_options, 'display_hidden_groups_info', text='', icon='HIDE_OFF')

            if not is_edit_mesh:
                r_right.prop(addon_prefs.list_options, 'display_hide_renders_info', text='', icon='RESTRICT_RENDER_OFF')

            if p_cls_mgr.id_group == 'blgroup':
                r_right.prop(addon_prefs.list_options, 'display_vertex_groups_weights_info', text='', icon='MOD_VERTEX_WEIGHT')
            r_right.separator(factor=1.5)

            if is_edit_mesh:
                r_right.prop(addon_prefs.list_options, 'display_all_scene_groups', text='', icon='XRAY')
            else:
                r_right.label(icon='BLANK1')

            p_cls_mgr.execute_Draw(context, layout)


class ZSTS_PT_ComboSets(bpy.types.Panel, ZsBasePanel):
    """  Zen Sets Groups Panel """
    bl_context = ''
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY
    bl_label = ""
    # bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        return context.mode in {'EDIT_MESH', 'OBJECT'}


class ZSTS_PT_ComboUiSets(bpy.types.Panel, ZsBasePanel):
    """  Zen Sets Groups Panel UI """
    bl_context = ''
    bl_space_type = 'IMAGE_EDITOR'
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY
    bl_label = ""
    # bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH' and context.space_data and getattr(context.space_data, 'show_uvedit')


class ZSTS_PT_Help(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_idname = "ZSTS_PT_Help"
    bl_label = "Help"
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY

    @classmethod
    def poll(cls, context):
        if context.mode not in {'EDIT_MESH', 'OBJECT'}:
            return False

        addon_prefs = get_prefs()
        return addon_prefs.show_panels.show_help

    def draw(self, context):
        layout = self.layout

        col = layout.column(align=False)

        row = col.row(align=True)
        row.operator(
            "wm.url_open",
            text=ZsLabels.PANEL_HELP_DOC_LABEL,
            icon="HELP"
        ).url = ZsLabels.PANEL_HELP_DOC_LINK
        row = col.row(align=True)
        row.operator(
            "wm.url_open",
            text=ZsLabels.PANEL_HELP_DISCORD_LABEL,
            icon_value=zs_icon_get(ZIconsType.DiscordLogo)
        ).url = ZsLabels.PANEL_HELP_DISCORD_LINK
        try:
            row = col.row(align=True)
            row.label(text='Version: ' + get_addon_version())
        except Exception:
            Log.error('Zen Sets: No version found. There may be several versions installed. Try uninstalling everything and installing the latest version.')


class ZSTS_PT_Preferences(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = ZsLabels.PANEL_PREFERENCES_LABEL
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        if context.mode not in {'EDIT_MESH', 'OBJECT'}:
            return False

        addon_prefs = get_prefs()
        return addon_prefs.show_panels.show_preferences

    def draw(self, context):
        layout = self.layout
        layout.operator(
            'zsts.show_keymaps',
            icon_value=zs_icon_get(ZIconsType.AddonLogoPng))
        layout.operator('zsts.reset_preferences')
        layout.operator('zsts.global_cleanup')


class ZSTS_PT_SubCommon(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = "Common"
    bl_parent_id = "ZSTS_PT_Preferences"
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        addon_prefs = get_prefs()
        addon_prefs.common.draw(self.layout)

        def _display_options(addon_prefs, layout_box, layout_label, attr_name):
            if is_property_modified(getattr(addon_prefs, attr_name)):
                subrow = layout_label.row()
                subrow.alignment = 'RIGHT'
                op = subrow.operator("zsts.restore_preference", text="Restore")
                op.mode = attr_name
            p_options = getattr(addon_prefs, attr_name)
            p_options.draw(layout_box)

        box = self.layout.box()
        row = box.row()
        row.label(text="Groups List Options")
        _display_options(addon_prefs, box, row, 'list_options')

        if context.mode == 'OBJECT':
            row = box.row()
            row.label(text="Groups Object Options")
            _display_options(addon_prefs, box, row, 'object_options')


class ZSTS_PT_SubPanels(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = "Panels"
    bl_parent_id = "ZSTS_PT_Preferences"
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        addon_prefs = get_prefs()
        addon_prefs.show_panels.draw(self.layout)


class ZSTS_PT_SubModes(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = "Modes"
    bl_parent_id = 'ZSTS_PT_Preferences'
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        addon_prefs = get_prefs()
        addon_prefs.modes.draw(self.layout)


class ZSTS_OT_ResetPreferences(bpy.types.Operator):
    """ Reset Zen Sets Preferences """
    bl_idname = "zsts.reset_preferences"
    bl_label = ZsLabels.RESET_LABEL
    bl_description = ZsLabels.RESET_DESC
    bl_options = {"REGISTER", "INTERNAL"}

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout
        layout.label(text=ZsLabels.RESET_MES)
        layout.separator()

    def execute(self, context):
        addon_prefs = get_prefs()
        items = addon_prefs.__annotations__.keys()
        for pref in items:
            addon_prefs.property_unset(pref)

        return {'FINISHED'}


class ZSTS_PT_ImportExport(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = "Import-Export"
    # bl_context = ZsUiConstants.ZSTS_CONTEXT
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        addon_prefs = get_prefs()
        return context.mode in {'EDIT_MESH', 'OBJECT'} and addon_prefs.show_panels.show_import_export

    def draw(self, context):
        layout = self.layout
        row = layout.row(align=True)
        row.operator("zsts.copy_to_clipboard", text='Copy', icon='COPYDOWN')
        row.operator("zsts.paste_from_clipboard", text='Paste', icon='PASTEDOWN')

        if context.mode == 'EDIT_MESH':
            addon_prefs = get_prefs()
            if addon_prefs.common.developer_mode is False:
                p_cls_mgr = get_sets_mgr(context.scene)
                if p_cls_mgr:
                    box = layout.box()
                    p_cls_mgr.execute_DrawImport(box, context)
                    box = layout.box()
                    p_cls_mgr.execute_DrawExport(box, context)


class ZSTS_PT_SubImport(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = "Import"
    bl_parent_id = 'ZSTS_PT_ImportExport'
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY
    # bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        addon_prefs = get_prefs()
        return addon_prefs.common.developer_mode

    def draw(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.execute_DrawImport(self.layout, context)


class ZSTS_PT_SubExport(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = "Export"
    bl_parent_id = 'ZSTS_PT_ImportExport'
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        addon_prefs = get_prefs()
        return addon_prefs.common.developer_mode

    def draw(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.execute_DrawExport(self.layout, context)


class ZSTS_PT_Tools(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = "Tools"
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY
    # bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(cls, context):
        addon_prefs = get_prefs()
        return addon_prefs.show_panels.show_tools and context.mode in {'EDIT_MESH', 'OBJECT'}

    def draw(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.execute_DrawTools(self, context)


class ZSTS_PT_DrawHighlight_Overlay(bpy.types.Panel):
    bl_label = "Zen Sets Overlay"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'
    bl_parent_id = "VIEW3D_PT_overlay"

    @classmethod
    def poll(cls, context):
        addon_prefs = get_prefs()
        return addon_prefs.show_panels.show_overlay and context.mode in {'EDIT_MESH', 'OBJECT'}

    def draw(self, context):
        layout = self.layout

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr._do_draw_highlight(context, layout)


class ZSTS_PT_DrawHighlight_UIOverlay(bpy.types.Panel):
    bl_label = "Zen Sets Overlay"
    bl_space_type = 'IMAGE_EDITOR'
    bl_region_type = 'HEADER'
    bl_parent_id = "IMAGE_PT_overlay"

    @classmethod
    def poll(cls, context):
        addon_prefs = get_prefs()
        return addon_prefs.show_panels.show_overlay and context.mode in {'EDIT_MESH'}

    def draw(self, context):
        layout = self.layout

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr._do_draw_highlight(context, layout)


class ZSTS_PT_OverlayFilter(bpy.types.Panel):
    bl_label = "Zen Sets Overlay Filter"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH'

    def draw(self, context):
        layout = self.layout

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr._do_draw_overlay_filter(context, layout)


class ZSTS_PT_AssignToPinnedOptions(bpy.types.Panel):
    bl_label = "Zen Sets Assign to Pinned"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH'

    def draw(self, context):
        layout = self.layout
        addon_prefs = get_prefs()
        addon_prefs.op_options.draw_assign_to_pinned(layout)


class ZSTS_PT_SelectModeMenu(bpy.types.Panel):
    bl_label = "Select Zen Sets Mode"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'

    def draw(self, context):
        layout = self.layout

        if context.mode == 'OBJECT':
            display_object_element_modes(layout, context)
        elif context.mode == 'EDIT_MESH':
            row_element = layout.row(align=True)
            row_blender = layout.row(align=True)
            row_mode = layout.row(align=True)

            display_element_modes(row_element, row_blender, row_mode, context)


""" MENUS """


class ZSTS_MT_GroupMenu(bpy.types.Menu):
    bl_label = "Zen Sets Menu"

    def draw(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)

        if p_cls_mgr:
            p_cls_mgr.execute_DrawMenu(self, context)


class ZSTS_MT_ImportMenu(bpy.types.Menu):
    bl_label = "Import"

    def draw(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)

        if p_cls_mgr:
            p_cls_mgr.execute_DrawImport(self.layout, context, is_menu=True)


class ZSTS_MT_ExportMenu(bpy.types.Menu):
    bl_label = "Export"

    def draw(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)

        if p_cls_mgr:
            p_cls_mgr.execute_DrawExport(self.layout, context, is_menu=True)


class ZSTS_MT_AssignMenu(bpy.types.Menu):
    bl_label = ""

    def draw(self, context):
        layout = self.layout
        op_prefix = 'zsto' if context.mode == 'OBJECT' else 'zsts'

        layout.operator(op_prefix + '.assign_to_group')


class ZSTS_MT_RemoveMenu(bpy.types.Menu):
    bl_label = ""

    def draw(self, context):
        layout = self.layout

        op_prefix = 'zsto' if context.mode == 'OBJECT' else 'zsts'

        op = layout.operator(op_prefix + '.remove_from_group', text='Remove from All Groups')
        op.mode = 'ALL'

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr and p_cls_mgr.id_group == 'blgroup':
            layout.separator()
            layout.operator('zsts.remove_weight_from_group')


class ZSTS_MT_IsolateMenu(bpy.types.Menu):
    bl_label = ""

    def draw(self, context):
        layout = self.layout
        layout.operator('zsts.smart_isolate')


""" FACTORY """


class ZsInterfaceFactory:
    classes = (
        ZsElementGroup,

        ZSTS_OT_NewItem,
        ZSTS_OT_DeleteGroup,
        ZSTS_OT_Append,
        ZSTS_OT_Assign,
        ZSTS_OT_MoveItem,
        ZSTS_OT_SelectGroup,
        ZSTS_OT_SelectUngroupped,
        ZSTS_OT_SelectOnlyGroup,
        ZSTS_OT_IntersectGroup,
        ZSTS_OT_SelectAppendGroup,
        ZSTS_OT_DeselectGroup,
        ZSTS_OT_GroupSiblings,
        ZSTS_OT_SmartIsolate,
        ZSTS_OT_HideGroup,
        ZSTS_OT_UnhideGroup,
        ZSTS_OT_InvertHideGroup,
        ZSTS_OT_RemoveSelectionFromGroup,
        ZSTS_OT_DeleteEmptyGroups,
        ZSTS_OT_DeleteGroupsCombo,
        ZSTS_OT_DeleteGroupsInternal,
        ZSTS_OT_DrawHighlight,
        ZSTS_OT_MarkSeams,
        ZSTS_OT_ClearSeams,
        ZSTS_OT_ResetPreferences,
        ZSTS_OT_GroupLinked,
        ZSTS_OT_SplitToObjects,
        ZSTS_OT_GlobalCleanup,
        ZSTS_OT_ObjectCleanup,
        ZSTS_OT_ExportGroupToVertexColor,
        ZSTS_OT_ImportVertexColorToGroup,
        ZSTS_OT_SelectObjectsInGroup,
        ZSTS_OT_InternalBuildLookup,
        ZSTS_OT_CopyToClipboard,
        ZSTS_OT_PasteClipboard,
        ZSTS_OT_InternalHideGroupByIndex,
        ZSTS_OT_RenameGroups,
        ZSTS_OT_ExportGroupColorsToPalette,
        ZSTS_OT_AssignPaletteColorToGroup,
        ZSTS_OT_AssignUVBorders,
        ZSTS_OT_AssignToPinned,
        ZSTS_OT_ResetDrawCache,
        ZSTS_OT_SetActiveGroup,
        ZSTS_OT_SelectMode,
        ZSTS_OT_SetSculptMask,

        ZSTS_PT_ComboSets,
        ZSTS_PT_ComboUiSets,

        ZSTS_PT_ImportExport,
        ZSTS_PT_SubImport,
        ZSTS_PT_SubExport,
        ZSTS_PT_Tools,
        ZSTS_PT_Preferences,
        ZSTS_PT_SubCommon,
        ZSTS_PT_SubPanels,
        ZSTS_PT_SubModes,
        ZSTS_PT_Help,
        ZSTS_PT_DrawHighlight_Overlay,
        ZSTS_PT_OverlayFilter,
        ZSTS_PT_DrawHighlight_UIOverlay,
        ZSTS_PT_AssignToPinnedOptions,
        ZSTS_PT_SelectModeMenu,

        ZSTS_MT_GroupMenu,
        ZSTS_MT_ImportMenu,
        ZSTS_MT_ExportMenu,
        ZSTS_MT_AssignMenu,
        ZSTS_MT_RemoveMenu,
        ZSTS_MT_IsolateMenu
    )
