# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

""" Zen List Groups System """
# blender
import bmesh

from collections import defaultdict

from .basic_sets import ZsLayerManager

from ..blender_zen_utils import ZenLocks, ZenPolls, ZenSelectionStats
from .draw_sets import check_update_cache, mark_groups_modified

from .uv_utils import is_uv_seam


class ZsListLayerManager(ZsLayerManager):

    @classmethod
    def set_selection_to_group(self, p_obj, p_scene_group, indices):
        layerName = p_scene_group.layer_name

        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        b_indices_mode = indices is not None
        if b_indices_mode:
            i_selected_count = len(indices)
        else:
            b_is_uv = self.is_uv_area_and_not_sync()
            if b_is_uv:
                uv_sel = self.fetch_uv_selections(bm)
                i_selected_count = len(uv_sel)
            else:
                i_selected_count = self.get_selected_count(p_obj)

        mesh_layer = self.ensure_mesh_layer(p_obj, bm, layerName) if i_selected_count else self.get_mesh_layer(p_obj, bm, layerName)
        if mesh_layer:
            i_group_count = 0
            for item in self.get_bm_items(bm):
                is_selected = (
                    (item.index in indices) if b_indices_mode
                    else (item.select and (not b_is_uv or (item.index in uv_sel)))
                )
                self.set_bm_item(item, mesh_layer, is_selected)
                if is_selected:
                    i_group_count += 1

            if i_group_count:
                p_group = self.ensure_group_in_object(p_obj, p_scene_group)
                p_group.group_count = i_group_count
                p_group.group_hide_count = 0

            bm.select_flush_mode()
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

    @classmethod
    def set_selection_to_new_group(self, p_obj, p_scene_group, i_start, i_end):
        layerName = p_scene_group.layer_name

        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        i_selected_count = self.get_selected_count(p_obj)

        mesh_layer = self.ensure_mesh_layer(p_obj, bm, layerName) if i_selected_count else self.get_mesh_layer(p_obj, bm, layerName)
        if mesh_layer and i_selected_count != 0:
            i_group_count = 0
            bm_items = self.get_bm_items(bm)
            for idx in range(i_start, i_end, 1):
                item = bm_items[idx]
                is_selected = item.select
                self.set_bm_item(item, mesh_layer, is_selected)
                if is_selected:
                    i_group_count += 1
                    i_selected_count -= 1
                    if i_selected_count == 0:
                        break

            if i_group_count:
                p_group = self.ensure_group_in_object(p_obj, p_scene_group)
                p_group.group_count = i_group_count
                p_group.group_hide_count = 0
            bm.select_flush_mode()
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

    @classmethod
    def set_uv_borders_to_edge_group(self, p_obj, p_scene_group):
        layerName = p_scene_group.layer_name

        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        uv_layer = bm.loops.layers.uv.active
        if uv_layer:
            mesh_layer = self.ensure_mesh_layer(p_obj, bm, layerName)

            bm.edges.ensure_lookup_table()
            bm.faces.ensure_lookup_table()

            visited_edges = set()

            was_modified = False

            for face in bm.faces:
                for loop in face.loops:
                    edge = loop.edge
                    if edge not in visited_edges:
                        visited_edges.add(edge)

                        uv = loop[uv_layer]
                        next_uv = loop.link_loop_next[uv_layer]

                        is_seam = False
                        if edge.is_boundary:
                            is_seam = True
                        else:
                            other_loop = loop.link_loop_radial_next
                            other_uv = other_loop[uv_layer]
                            other_uv_next = other_loop.link_loop_next[uv_layer]

                            if is_uv_seam(uv.uv, next_uv.uv, other_uv.uv, other_uv_next.uv):
                                is_seam = True

                        if edge[mesh_layer] != is_seam:
                            edge[mesh_layer] = is_seam
                            was_modified = True

            if was_modified:
                self.ensure_group_in_object(p_obj, p_scene_group)
                bm.select_flush_mode()
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
                self.update_obj_group_count(p_obj, layerName)

    @classmethod
    def select_ungroupped(self, p_obj, p_group_pairs):
        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        layers = [self.get_mesh_layer(p_obj, bm, g.layer_name) for _, g in p_group_pairs]

        b_is_uv = self.is_uv_area_and_not_sync()
        uv_layer = bm.loops.layers.uv.active

        for item in self.get_bm_items(bm):
            b_select = not any(self.is_bm_item_set(item, layer) for layer in layers if layer)
            if b_is_uv and uv_layer:
                for loop in self.get_item_loops(item):
                    loop[uv_layer].select = b_select
                    if ZenPolls.version_greater_3_2_0:
                        loop[uv_layer].select_edge = b_select
            else:
                item.select = b_select

        bm.select_flush_mode()
        bmesh.update_edit_mesh(p_obj.data, loop_triangles=False, destructive=False)

        if self.is_uv_force_update():
            mark_groups_modified(self, p_obj, modes={'UV'})
            check_update_cache(self, p_obj)
        ZenLocks.lock_depsgraph_update_one()

    @classmethod
    def _do_cleanup_object(self, obj):
        super()._do_cleanup_object(obj)

        bmesh.update_edit_mesh(obj.data, loop_triangles=False, destructive=False)

    @classmethod
    def smart_select(self, context, select_active_group_only, keep_active_group) -> ZenSelectionStats:
        selection_stats = ZenSelectionStats()
        selection_stats.was_mesh_sel_count = self.get_context_selected_count(context)
        if selection_stats.was_mesh_sel_count == 0:
            return selection_stats

        p_scene = context.scene
        p_group_pairs = self.get_current_group_pairs(context)

        self.set_mesh_select_mode(context)

        # an element may belong to many groups, so make a set of such groups
        sel_mesh_layers = set()
        map_mesh_layer = dict()
        map_obj_sel_layers = defaultdict(dict)
        map_non_sel = set()

        b_is_uv = self.is_uv_area_and_not_sync()
        b_is_face = self.id_element == 'face'

        selection_stats.was_uv_sel_count = 0

        p_mesh_objects = dict()

        for p_obj in context.objects_in_mode_unique_data:
            if p_obj.type != 'MESH':
                continue

            me = p_obj.data
            bm = bmesh.from_edit_mesh(me)
            items = self.get_bm_items(bm)

            p_mesh_objects[p_obj] = bm

            map_mesh_layer[p_obj] = self.get_mesh_layers_in_group_pairs(p_obj, bm, p_group_pairs)

            if b_is_uv:
                uv_sel = self.fetch_uv_selections(bm)
                i_cur_uv_selected = len(uv_sel)
                selection_stats.was_uv_sel_count += i_cur_uv_selected
                if i_cur_uv_selected == 0:
                    continue

            for item in items:
                if item.select and (not b_is_uv or (item.index in uv_sel)):
                    is_nongroupped = True
                    for layer, layer_name in map_mesh_layer[p_obj]:
                        if self.is_bm_item_set(item, layer):
                            sel_mesh_layers.add(layer_name)
                            is_nongroupped = False
                    if is_nongroupped:
                        map_non_sel.add(p_obj)

        if b_is_uv and selection_stats.was_uv_sel_count == 0:
            return selection_stats

        modified_modes = set()

        modified_objects_data = set()

        for p_obj, p_layers in map_mesh_layer.items():
            for p_layer, p_layer_name in p_layers:
                if p_layer_name in sel_mesh_layers:
                    map_obj_sel_layers[p_obj][p_layer_name] = p_layer

        # 1) selected elemenents do not belong to any group
        if len(sel_mesh_layers) == 0:
            for p_obj, bm in p_mesh_objects.items():
                items = self.get_bm_items(bm)
                uv_layer = bm.loops.layers.uv.active

                selected_loops = set()
                unselected_loops = set()

                b_obj_selection_changed = False

                for item in items:
                    is_present = not any(self.is_bm_item_set(item, layer) for layer, _ in map_mesh_layer[p_obj])
                    was_selected = item.select
                    self._smart_process_item(
                        item, b_is_uv, uv_layer, is_present, b_is_face,
                        selected_loops, unselected_loops)

                    if is_present:
                        selection_stats.new_mesh_sel_count += 1

                    if item.select != was_selected:
                        b_obj_selection_changed = True

                if b_is_uv:
                    all_loops = selected_loops.union(unselected_loops)
                    for loop in all_loops:
                        b_select = loop in selected_loops
                        was_selected = loop[uv_layer].select
                        loop[uv_layer].select = b_select
                        if ZenPolls.version_greater_3_2_0 and b_is_face:
                            loop[uv_layer].select_edge = b_select
                        selection_stats.new_uv_sel_count += 1

                        if was_selected != b_select:
                            b_obj_selection_changed = True

                if b_obj_selection_changed:
                    selection_stats.selection_changed = True
                    modified_objects_data.add(p_obj.data)
                    bm.select_flush_mode()
                    bmesh.update_edit_mesh(p_obj.data, loop_triangles=True, destructive=False)

            self.set_list_index(p_scene, -1)
            self._do_set_last_smart_select('')
        else:
            sel_active_layer_name = ''
            sel_last_layer_name = sel_active_layer_name
            p_group_pair = self.get_current_group_pair(context)
            p_scene_layer_name = p_group_pair[1].layer_name if p_group_pair else ''
            # check if we have a group present in selection to keep it remaining
            if p_scene_layer_name in sel_mesh_layers:
                sel_active_layer_name = p_scene_layer_name
                sel_last_layer_name = sel_active_layer_name

            # by default we will select first group
            if sel_active_layer_name == '':
                sel_active_layer_name = next(iter(sel_mesh_layers))
                sel_last_layer_name = sel_active_layer_name

            if context.active_object:
                p_obj = context.active_object
                bm = p_mesh_objects.get(p_obj, None)
                if bm:
                    items = self.get_bm_items(bm)
                    if len(items):
                        active_element = bm.select_history.active
                        if active_element and (type(items[0]) is type(active_element)):
                            for layer_name, layer in map_obj_sel_layers[p_obj].items():
                                if self.is_bm_item_set(active_element, layer):
                                    if not keep_active_group:
                                        sel_active_layer_name = layer_name
                                    sel_last_layer_name = layer_name
                                    break

            for p_obj, bm in p_mesh_objects.items():
                items = self.get_bm_items(bm)
                uv_layer = bm.loops.layers.uv.active

                selected_loops = set()
                unselected_loops = set()

                b_obj_selection_changed = False

                if select_active_group_only:
                    cur_mesh_layer = map_obj_sel_layers[p_obj].get(sel_active_layer_name, None)
                    if cur_mesh_layer:
                        for item in items:
                            was_selected = item.select
                            is_present = self.is_bm_item_set(item, cur_mesh_layer)
                            self._smart_process_item(
                                item, b_is_uv, uv_layer, is_present,
                                b_is_face,
                                selected_loops, unselected_loops)
                            if is_present:
                                selection_stats.new_mesh_sel_count += 1
                            if item.select != was_selected:
                                b_obj_selection_changed = True
                else:
                    has_nongroupped = p_obj in map_non_sel
                    for item in items:
                        is_present = any(self.is_bm_item_set(item, layer) for layer in map_obj_sel_layers[p_obj].values())
                        if not is_present and has_nongroupped and not any(self.is_bm_item_set(item, layer) for layer, _ in map_mesh_layer[p_obj]):
                            is_present = True

                        was_selected = item.select
                        self._smart_process_item(
                            item, b_is_uv, uv_layer, is_present,
                            b_is_face, selected_loops, unselected_loops)

                        if is_present:
                            selection_stats.new_mesh_sel_count += 1

                        if item.select != was_selected:
                            b_obj_selection_changed = True

                if b_is_uv:
                    all_loops = selected_loops.union(unselected_loops)
                    for loop in all_loops:
                        b_select = loop in selected_loops
                        was_selected = loop[uv_layer].select
                        loop[uv_layer].select = b_select
                        if ZenPolls.version_greater_3_2_0 and b_is_face:
                            loop[uv_layer].select_edge = b_select
                        selection_stats.new_uv_sel_count += 1

                        if was_selected != b_select:
                            b_obj_selection_changed = True

                if b_obj_selection_changed:
                    selection_stats.selection_changed = True
                    modified_objects_data.add(p_obj.data)
                    bm.select_flush_mode()
                    bmesh.update_edit_mesh(p_obj.data, loop_triangles=True, destructive=False)

            for i, g in p_group_pairs:
                layer_name = g.layer_name
                if sel_active_layer_name:
                    if layer_name == sel_active_layer_name:
                        i_list_index = self.get_list_index(p_scene)
                        if i_list_index != i:
                            self.set_list_index(p_scene, i)
                            modified_modes.update({'UV', 'MESH'})
                        if sel_last_layer_name == sel_active_layer_name:
                            self._do_set_last_smart_select(layer_name)
                            sel_last_layer_name = None
                        sel_active_layer_name = None
                if sel_last_layer_name:
                    if layer_name == sel_active_layer_name:
                        self._do_set_last_smart_select(layer_name)
                        sel_last_layer_name = None
                if sel_active_layer_name is None and sel_last_layer_name is None:
                    break

        if selection_stats.selection_changed:
            if self.is_uv_force_update():
                modified_modes.add('UV')
            if modified_modes:
                for p_obj in context.objects_in_mode:
                    if p_obj.data in modified_objects_data:
                        mark_groups_modified(self, p_obj, modes=modified_modes)
                        check_update_cache(self, p_obj)
            ZenLocks.lock_depsgraph_update_one()

        return selection_stats

    @classmethod
    def import_vertex_colors_to_zen_groups(self, p_scene, p_obj, b_active, p_ignored_color, s_prefix):
        bm = self._get_bm(p_obj)
        p_scene_list = self.get_list(p_scene)
        bm_items = self.get_bm_items(bm)
        i_group_index = -1
        act_name = bm.loops.layers.color.active.name if bm.loops.layers.color.active else ''
        for p_vert_color_layer in bm.loops.layers.color.values():
            if b_active and p_vert_color_layer.name != act_name:
                continue

            vc_name = p_vert_color_layer.name.replace(s_prefix, '', 1)

            b_is_vert = self.id_element == 'vert'
            b_is_edge = self.id_element == 'edge'

            colors = set((loop[p_vert_color_layer][0], loop[p_vert_color_layer][1], loop[p_vert_color_layer][2])
                         for item in bm_items
                         for loop in (item.link_loops if (b_is_vert or b_is_edge) else item.loops)
                         if p_ignored_color is None or (loop[p_vert_color_layer][0] != p_ignored_color.r and
                                                        loop[p_vert_color_layer][1] != p_ignored_color.g and
                                                        loop[p_vert_color_layer][2] != p_ignored_color.b)
                         )

            for i, color in enumerate(colors):
                group_name = vc_name
                if len(colors) > 1:
                    group_name += f' [{i+1}]'

                i_group_index = self._index_of_group_name(p_scene_list, group_name)
                if i_group_index == -1:
                    s_layer_name = self.create_unique_layer_name()
                    i_group_index = self.add_layer_to_list(p_scene, s_layer_name, color)
                    p_scene_list[i_group_index].name = group_name

                    i_obj_group_index = self.add_layer_to_list(p_obj, s_layer_name, color)
                    p_obj_list = self.get_list(p_obj)
                    p_obj_list[i_obj_group_index].name = group_name
                else:
                    self.ensure_group_in_object(p_obj, p_scene_list[i_group_index])

                if p_scene_list[i_group_index].group_color != color:
                    p_scene_list[i_group_index].group_color = color

                layer = self.ensure_mesh_layer(p_obj, bm, p_scene_list[i_group_index].layer_name)
                for item in bm_items:
                    loops = item.link_loops if (b_is_vert or b_is_edge) else item.loops
                    has_color = False
                    for loop in loops:
                        if loop[p_vert_color_layer][0] == color[0] and \
                           loop[p_vert_color_layer][1] == color[1] and \
                           loop[p_vert_color_layer][2] == color[2]:

                            has_color = True
                            break

                    item[layer] = has_color

        self.update_all_obj_groups_count(p_obj)

        self.set_list_index(p_scene, i_group_index)

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

        layout = menu.layout

        layout.operator('zsts.assign_to_pinned_group')

    @classmethod
    def execute_DrawTools(cls, tools, context):
        super().execute_DrawTools(tools, context)

        layout = tools.layout

        row = layout.row(align=True)
        row.operator('zsts.assign_to_pinned_group')
        row.popover(
            panel="ZSTS_PT_AssignToPinnedOptions",
            text="",
            icon='DOWNARROW_HLT')
