# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# blender
import bpy
import bmesh
# python
import uuid
import random
import json
from mathutils import Color
from timeit import default_timer as timer
from collections import Counter, defaultdict
import numpy as np
from typing import Tuple
import fnmatch
import re

# local
from ..labels import ZsLabels
from ..preferences import ZSTS_AddonPreferences, get_prefs
from ..vlog import Log
from ..ico import zs_icon_get, ZIconsType
from ..progress import start_progress, update_progress, end_progress
from ..blender_zen_utils import (
    ZsOperatorAttrs,
    fix_undo_push_edit_mode,
    save_viewlayer_objects_state,
    prepare_stored_objects_for_edit,
    save_viewlayer_layers_state,
    show_all_viewlayers,
    unhide_and_select_all_viewlayer_objects,
    restore_viewlayer_layers,
    restore_viewlayer_objects,
    ensure_object_in_viewlayer,
    ireplace,
    is_any_uv_editor_opened,
    ZenLocks, ZenPolls, ZenSelectionStats,
    update_areas_in_all_screens)

from .draw_sets import check_update_cache, is_draw_handler_enabled, remove_draw_handler, update_cache_group, mark_groups_modified
from .basic_colors import CONST_GROUP_COLORS, ColorUtils
from .lookup_utils import ZsLookup, get_lookup_table, get_lookup_table_check


_rnd = random.Random()


class Zs_UL_BaseList(bpy.types.UIList):
    """ Zen Groups UIList """

    def filter_items(self, context, data, propname):
        addon_prefs = get_prefs()
        if addon_prefs.list_options.display_all_scene_groups:
            return [], []

        p_scene_list = getattr(data, propname)

        filters = self.get_lookup_item(context, ZsLookup.Filter)

        # Default return values.
        flt_flags = [self.bitflag_filter_item] * len(p_scene_list)

        if self.filter_name:
            flt_flags = bpy.types.UI_UL_list.filter_items_by_name(
                self.filter_name, self.bitflag_filter_item, p_scene_list,
                reverse=self.use_filter_sort_reverse)

        for idx in filters:
            flt_flags[idx] &= ~self.bitflag_filter_item

        flt_neworder = []

        return flt_flags, flt_neworder

    def draw_item(self, context, layout, data, item, icon, active_data, active_propname, index):
        addon_prefs = get_prefs()

        t_info = self.get_lookup_extra_info_by_index(context, index)
        n_obj_count = t_info.get('objects', 0)
        n_hide_count = t_info.get('hide_count', 0)

        if n_obj_count == 0:
            layout.enabled = False
        else:
            layout.active = n_hide_count == 0

        layout.separator(factor=0.1)

        col = layout.column(align=False)
        col.ui_units_x = 0.7
        col.separator(factor=0.8)
        col.scale_y = 0.6
        col.prop(item, 'group_color', text='')

        r = layout.split(factor=0.5)
        r.prop(item, 'name', text='', emboss=False, icon='NONE')

        r = r.row()
        r.alignment = 'RIGHT'
        self.execute_DrawListItemRightProps(
            r, context, item, index,
            t_info, addon_prefs)

        r.separator(factor=0.1)


class ZsLayerManager:

    id_group = ''           # edge, vert, face ...
    id_mask = ''            # ZSEG, ZSVG, ZSFG ...
    list_item_prefix = ''   # Edges, Verts, Faces ...
    id_element = ''         # edge, vert, face
    is_unique = False
    is_blender = False

    @classmethod
    def list_prop_name(cls):
        return f'zen_{cls.id_group}_list'

    @classmethod
    def active_list_prop_name(cls):
        return f'zen_{cls.id_group}_list_index'

    @classmethod
    def list_layer_prefix(cls):
        return f'zen_{cls.id_group}_layer'

    @classmethod
    def layer_hash_name(cls):
        return f'zen_{cls.id_group}_hash_layer'

    # PROTECTED
    @classmethod
    def _ensure_mesh_layer(self, layerType, layerName):
        """ Return layer int type or create new """
        layer = layerType.get(layerName)
        if not layer:
            layer = layerType.new(layerName)
        return layer

    @classmethod
    def _get_mesh_layer(self, layerType, layerName):
        """ Return mesh layer or None """
        return layerType.get(layerName)

    @classmethod
    def _remove_mesh_layer(self, layerType, layerName):
        layer = layerType.get(layerName)
        if layer:
            layerType.remove(layer)

    @classmethod
    def _gen_new_color2(self, p_list):
        i_color_index = -1
        for i, v in reversed(list(enumerate(p_list))):
            if i_color_index == -1:
                for k, v2 in enumerate(CONST_GROUP_COLORS):
                    if v.group_color == v2:
                        i_color_index = k + 1
                        break
            else:
                break

        if i_color_index == -1 or i_color_index >= len(CONST_GROUP_COLORS):
            i_color_index = 0

        return CONST_GROUP_COLORS[i_color_index]

    @classmethod
    def _gen_new_color_seq(self, p_list, colors):
        if len(colors) == 0:
            return Color()

        i_color_index = -1
        for i, v in reversed(list(enumerate(p_list))):
            if i_color_index == -1:
                for k, v2 in enumerate(colors):
                    if v.group_color == v2.color:
                        i_color_index = k + 1
                        break
            else:
                break

        if i_color_index == -1 or i_color_index >= len(colors):
            i_color_index = 0

        return colors[i_color_index].color

    @classmethod
    def _gen_new_color_rnd(self, p_list, colors):
        if len(colors) == 0:
            return Color()

        i_color_count = len(colors)
        idx = _rnd.randrange(0, i_color_count)
        if len(p_list) > 1 and i_color_count > 1:
            i_iteration = 0
            while p_list[-1].group_color == colors[idx].color and i_iteration < 10:
                idx = _rnd.randrange(0, i_color_count)
                i_iteration += 1

        return colors[idx].color

    @classmethod
    def _gen_new_color(self, p_list):
        p_scene = bpy.context.scene
        pal = p_scene.zsts_color_palette.palette
        if pal:
            if len(pal.colors):
                if p_scene.zsts_color_palette.mode == 'PAL_SEQ':
                    return self._gen_new_color_seq(p_list, pal.colors)
                elif p_scene.zsts_color_palette.mode == 'PAL_RND':
                    return self._gen_new_color_rnd(p_list, pal.colors)

        return self._gen_new_color_auto(p_list)

    @classmethod
    def _gen_new_color_auto(self, p_list):
        return ColorUtils.gen_new_color(p_list, 'group_color')

    @classmethod
    def _add_list_layer(self, p_list, layerName, sItemPrefix, p_color):
        p_list.add()

        i = len(p_list) - 1

        p_list[-1].name = sItemPrefix + (f"_{i + 1}" if i > 0 else "")
        p_list[-1].layer_name = layerName
        p_list[-1].group_color = p_color

        return i

    @classmethod
    def _create_unique_layer_name(self, layerPrefix):
        return layerPrefix + f"_{str(uuid.uuid4())}"

    @classmethod
    def _get_bm(self, p_obj):
        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)
        return bm

    @classmethod
    def _get_scene_group(self, p_scene):
        p_list = self.get_list(p_scene)
        i_items_count = len(p_list)
        i_list_index = self.get_list_index(p_scene)

        if i_list_index in range(i_items_count):
            return p_list[i_list_index]
        else:
            return None

    @classmethod
    def _index_of_layer(self, p_list, layerName):
        i_items_count = len(p_list)
        for i in range(i_items_count):
            if layerName == p_list[i].layer_name:
                return i

        return -1

    @classmethod
    def _index_of_group_name(self, p_list, groupName):
        i_items_count = len(p_list)
        for i in range(i_items_count):
            if groupName == p_list[i].name:
                return i

        return -1

    @classmethod
    def get_group_pair_by_name(self, p_group_pairs, groupName):
        for p_group_pair in p_group_pairs:
            if groupName == p_group_pair[1].name:
                return p_group_pair

        return None

    @classmethod
    def get_group_pair_by_layer(self, p_group_pairs, layerName):
        for p_group_pair in p_group_pairs:
            if layerName == p_group_pair[1].layer_name:
                return p_group_pair

        return None

    @classmethod
    def _get_group_by_layer(self, p_scene_or_obj, layerName):
        p_list = self.get_list(p_scene_or_obj)
        for group in p_list:
            if group.layer_name == layerName:
                return group

        return None

    @classmethod
    def _get_layer_name_by_index(self, p_scene, i_list_index):
        p_list = self.get_list(p_scene)
        i_items_count = len(p_list)

        if i_list_index in range(i_items_count):
            return p_list[i_list_index].layer_name
        else:
            return ''

    @classmethod
    def get_current_layer_name(self, context):
        p_group_pair = self.get_current_group_pair(context)
        return p_group_pair[1].layer_name if p_group_pair else ''

    @classmethod
    def _get_layer_name(self, p_scene):
        i_list_index = self.get_list_index(p_scene)
        return self._get_layer_name_by_index(p_scene, i_list_index)

    @classmethod
    def _get_group_color(self, p_scene):
        p_item = self._get_scene_group(p_scene)

        if p_item is not None:
            return p_item.group_color
        else:
            return None

    @classmethod
    def has_object_empty_groups(self, context, p_obj):
        p_list = self.get_list(context.scene)
        for g in p_list:
            p_group = self._get_group_by_layer(p_obj, g.layer_name)
            if p_group and p_group.group_count == 0:
                return True
        return False

    @classmethod
    def _get_layer_extra_info(self, context, layerName):
        n_group_count = 0
        n_obj_count = 0
        n_hide_count = 0
        for obj in context.objects_in_mode:
            group = self._get_group_by_layer(obj, layerName)
            n_current_count = -1
            n_current_hide_count = 0
            if group is not None:
                n_current_count = group.group_count
                n_current_hide_count = group.group_hide_count

            if n_current_count > -1:
                n_group_count = n_group_count + n_current_count
                n_obj_count = n_obj_count + 1
                n_hide_count = n_hide_count + n_current_hide_count

        p_extra_info = {
            'count': n_group_count,
            'objects': n_obj_count,
            'hide_count': n_hide_count
        }
        return p_extra_info

    @classmethod
    def _get_extra_info(self, context, index):
        p_scene = context.scene
        s_layer_name = self._get_layer_name_by_index(p_scene, index)
        if s_layer_name:
            return self._get_layer_extra_info(context, s_layer_name)
        else:
            return {}

    @classmethod
    def get_obj_highlighted_groups(self, p_obj):
        p_list = self.get_list(bpy.context.scene)
        result = []

        def _internal_display_groups(self):
            return [p_list[i] for i in self.get_lookup_item(bpy.context, ZsLookup.ObjectHighlightedGroups)[p_obj]]

        try:
            result = _internal_display_groups(self)
        except Exception:
            self.build_lookup_table(bpy.context)
            result = _internal_display_groups(self)

        return result

    @classmethod
    def is_update_elements_monitor_enabled(cls, p_addon_prefs: ZSTS_AddonPreferences):
        return p_addon_prefs.list_options.display_hidden_groups_info

    @classmethod
    def is_group_highlighted_in_obj(self, p_obj, p_group):
        p_obj_group = self._get_group_by_layer(p_obj, p_group.layer_name)
        return p_obj_group and p_obj_group.group_count != 0

    @classmethod
    def is_current_group_active(self, context):
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            p_info = self.get_lookup_extra_info_layer(context)[p_group_pair[1].layer_name]
            if p_info.get('objects', 0) != 0:
                return True
        return False

    @classmethod
    def get_obj_group_count(self, p_obj, layerName):
        me = p_obj.data
        if me and p_obj.type == 'MESH':
            if me.is_editmode:
                bm = self._get_bm(p_obj)
                layer = self.get_mesh_layer(p_obj, bm, layerName)
                if layer:
                    p_items = [item.hide for item in self.get_bm_items(bm) if self.is_bm_item_set(item, layer)]
                    p_count = len(p_items)
                    return {
                        'count': p_count,
                        'hide_count': p_items.count(True)
                    }
        return {}

    @classmethod
    def _do_set_or_append_group_to_selection(self, p_obj, bm, layerName, b_is_uv_not_sync, uv_layer, b_set):
        layer = self.get_mesh_layer(p_obj, bm, layerName)
        for item in self.get_bm_items(bm):
            b_select = ((layer is not None) and self.is_bm_item_set(item, layer))
            if b_set or b_select:
                if b_is_uv_not_sync and uv_layer:
                    for loop in self.get_item_loops(item):
                        loop[uv_layer].select = b_select
                        if ZenPolls.version_greater_3_2_0:
                            loop[uv_layer].select_edge = b_select
                else:
                    item.select = b_select

    @classmethod
    def _set_or_append_group_to_selection(self, p_obj, layerName, b_set):
        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        b_is_uv = self.is_uv_area_and_not_sync()
        uv_layer = bm.loops.layers.uv.active

        self._do_set_or_append_group_to_selection(p_obj, bm, layerName, b_is_uv, uv_layer, b_set)

        bm.select_flush_mode()
        bmesh.update_edit_mesh(p_obj.data, loop_triangles=False, destructive=False)

        if self.is_uv_force_update():
            mark_groups_modified(self, p_obj, modes={'UV'})
            check_update_cache(self, p_obj)
        ZenLocks.lock_depsgraph_update_one()

    @classmethod
    def set_group_to_selection(self, p_obj, layerName):
        self._set_or_append_group_to_selection(p_obj, layerName, True)

    @classmethod
    def append_group_to_selection(self, p_obj, layerName):
        self._set_or_append_group_to_selection(p_obj, layerName, False)

    @classmethod
    def store_selection(self, context: bpy.types.Context, b_is_not_sync):
        was_selection = {}
        for p_obj in context.objects_in_mode_unique_data:
            if p_obj.type != 'MESH':
                continue
            bm = self._get_bm(p_obj)
            bm_items = self.get_bm_items(bm)
            uv_layer = bm.loops.layers.uv.active
            selected_items = []
            if uv_layer:
                if b_is_not_sync:
                    selected_items = [
                        (loop, loop[uv_layer].select_edge if ZenPolls.version_greater_3_2_0 else None)
                        for item in bm_items for loop in self.get_item_loops(item)
                        if not item.hide and loop[uv_layer].select
                    ]
                else:
                    selected_items = [
                        item
                        for item in bm_items
                        if not item.hide and item.select
                    ]
            was_selection[p_obj] = (bm, uv_layer, selected_items)
        return was_selection

    @classmethod
    def restore_selection(self, context: bpy.types.Context, b_is_not_sync, was_selection: dict):
        if b_is_not_sync:
            for v in was_selection.values():
                uv_layer = v[1]
                if uv_layer:
                    for loop, select_edge in v[2]:
                        loop[uv_layer].select = True
                        if ZenPolls.version_greater_3_2_0:
                            loop[uv_layer].select_edge = select_edge
                    bm = v[0]
                    bm.select_flush_mode()
        else:
            for v in was_selection.values():
                for item in v[2]:
                    item.select = True
                bm = v[0]
                bm.select_flush_mode()

    # public abstract
    # this methods must be overrided in all derived classes !!!
    @classmethod
    def get_selected_count(self, p_obj):
        Log.error("ABSTRACT> 'get_selected_count'")
        return 0

    @classmethod
    def get_item_loops(self, p_item):
        Log.error("ABSTRACT> 'get_item_loops'")
        return None

    @classmethod
    def fetch_uv_selections(self, bm) -> set:
        return set()

    @classmethod
    def set_selection_to_group(self, p_obj, p_scene_group, indices):
        Log.error("ABSTRACT> 'set_selection_to_group'")

    @classmethod
    def set_selection_to_new_group(self, p_obj, p_scene_group, i_start, i_end):
        Log.error("ABSTRACT> 'set_selection_to_new_group'")

    @classmethod
    def append_selection_to_group(self, p_obj, p_scene_group):
        layerName = p_scene_group.layer_name

        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        b_is_uv = self.is_uv_area_and_not_sync()
        if b_is_uv:
            uv_sel = self.fetch_uv_selections(bm)
            i_selected_count = len(uv_sel)
        else:
            i_selected_count = self.get_selected_count(p_obj)

        mesh_layer = self.ensure_mesh_layer(p_obj, bm, layerName) if i_selected_count else self.get_mesh_layer(p_obj, bm, layerName)
        if mesh_layer:
            was_modified = False
            for item in self.get_bm_items(bm):
                if item.select and (not b_is_uv or (item.index in uv_sel)):
                    self.set_bm_item(item, mesh_layer, item.select)
                    was_modified = True

            if was_modified:
                self.ensure_group_in_object(p_obj, p_scene_group)
                bm.select_flush_mode()
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

                self.update_obj_group_count(p_obj, layerName)

    @classmethod
    def remove_selection_from_group(self, p_obj, layerName):
        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)
        layer = self.get_mesh_layer(p_obj, bm, layerName)
        if layer:
            was_modified = False
            b_is_uv = self.is_uv_area_and_not_sync()
            if b_is_uv:
                uv_sel = self.fetch_uv_selections(bm)
            for item in self.get_bm_items(bm):
                if item.select and (not b_is_uv or (item.index in uv_sel)):
                    self.set_bm_item(item, layer, False)
                    was_modified = True

            if was_modified:
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
                bm.select_flush_mode()
                self.update_obj_group_count(p_obj, layerName)

    @classmethod
    def remove_group_from_selection(self, p_obj, layerName):
        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        layer = self.get_mesh_layer(p_obj, bm, layerName)
        if layer:
            b_is_uv = self.is_uv_area_and_not_sync()
            uv_layer = bm.loops.layers.uv.active

            for item in self.get_bm_items(bm):
                if item.select and self.is_bm_item_set(item, layer):
                    if b_is_uv and uv_layer:
                        for loop in self.get_item_loops(item):
                            loop[uv_layer].select = False
                            if ZenPolls.version_greater_3_2_0:
                                loop[uv_layer].select_edge = False
                    else:
                        item.select = False

            bm.select_flush_mode()
            bmesh.update_edit_mesh(p_obj.data, loop_triangles=False, destructive=False)

            if self.is_uv_force_update():
                mark_groups_modified(self, p_obj, modes={'UV'})
                check_update_cache(self, p_obj)
            ZenLocks.lock_depsgraph_update_one()

    @classmethod
    def intersect_selection_with_group(self, p_obj, layerName):
        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        layer = self.get_mesh_layer(p_obj, bm, layerName)
        b_is_uv = self.is_uv_area_and_not_sync()
        uv_layer = bm.loops.layers.uv.active

        for item in self.get_bm_items(bm):
            b_select = item.select and ((layer is not None) and self.is_bm_item_set(item, layer))
            if b_is_uv and uv_layer:
                for loop in self.get_item_loops(item):
                    loop[uv_layer].select = b_select
                    if ZenPolls.version_greater_3_2_0:
                        loop[uv_layer].select_edge = b_select
            else:
                item.select = b_select

        bm.select_flush_mode()
        bmesh.update_edit_mesh(p_obj.data, loop_triangles=False, destructive=False)

        if self.is_uv_force_update():
            mark_groups_modified(self, p_obj, modes={'UV'})
            check_update_cache(self, p_obj)
        ZenLocks.lock_depsgraph_update_one()

    @classmethod
    def get_cacher(self):
        Log.error("ABSTRACT> 'get_cacher'")

    @classmethod
    def get_bm_items(self, bm):
        Log.error("ABSTRACT> 'get_bm_items'")
        return None

    @classmethod
    def select_ungroupped(self, p_obj, p_group_pairs):
        Log.error("ABSTRACT> 'select_ungroupped'")

    @classmethod
    def is_uv_area(self):
        ctx = bpy.context
        if ctx.area is None:
            return False
        return ctx.area.type == 'IMAGE_EDITOR' and ctx.area.ui_type == 'UV'

    @classmethod
    def is_uv_area_and_not_sync(self):
        return not self.is_uv_sync() and self.is_uv_area()

    @classmethod
    def is_uv_sync(self):
        return bpy.context.scene.tool_settings.use_uv_select_sync

    @classmethod
    def check_uv_select_mode(self) -> bool:
        b_changed = False
        if bpy.context.tool_settings.uv_select_mode != self.id_uv_select_mode:
            bpy.context.tool_settings.uv_select_mode = self.id_uv_select_mode
            b_changed = True
        return b_changed

    @classmethod
    def check_uv_not_sync(self):
        b_is_uv_not_sync = self.is_uv_area_and_not_sync()
        if b_is_uv_not_sync:
            if self.id_element in {'vert', 'edge'}:
                raise RuntimeError(f'Element {self.id_uv_select_mode} is restricted in UV Non Sync Mode!')
        return b_is_uv_not_sync

    @classmethod
    def is_uv_display_active(self):
        addon_prefs = get_prefs()
        return (
            is_draw_handler_enabled(self) and
            addon_prefs.uv_options.display_uv and
            is_any_uv_editor_opened(bpy.context))

    @classmethod
    def is_uv_editor_open_and_enabled(self) -> bool:
        addon_prefs = get_prefs()
        return (
            addon_prefs.uv_options.display_uv and
            is_any_uv_editor_opened(bpy.context))

    @classmethod
    def is_uv_force_update(self, only_selection=True) -> bool:
        b_is_draw_active = is_draw_handler_enabled(self)

        b_is_uv_not_sync = self.is_uv_area_and_not_sync()
        if only_selection:
            return b_is_draw_active and b_is_uv_not_sync is False

        addon_prefs = get_prefs()
        b_force_uv_update = (
            is_draw_handler_enabled(self) and
            addon_prefs.uv_options.display_uv and
            addon_prefs.uv_options.selected_only and
            not self.is_uv_sync() and
            is_any_uv_editor_opened(bpy.context))
        return b_force_uv_update

    @classmethod
    def ensure_mesh_layer(self, p_obj, p_bm, layerName):
        return self._ensure_mesh_layer(self.get_bm_items(p_bm).layers.int, layerName)

    @classmethod
    def get_mesh_layer(self, p_obj, p_bm, layerName):
        return self._get_mesh_layer(self.get_bm_items(p_bm).layers.int, layerName)

    @classmethod
    def get_mesh_layers_in_group_pairs(self, p_obj, p_bm, p_group_pairs):
        return [(layer, layer.name) for layer in [self.get_mesh_layer(p_obj, p_bm, g.layer_name) for _, g in p_group_pairs] if layer]

    @classmethod
    def remove_mesh_layer(self, p_obj, p_bm, layerName, cleanup=False):
        self._remove_mesh_layer(self.get_bm_items(p_bm).layers.int, layerName)

    @classmethod
    def is_bm_item_set(self, p_bm_item, p_layer):
        return bool(p_bm_item[p_layer])

    @classmethod
    def set_bm_item(self, p_bm_item, p_layer, p_val):
        p_bm_item[p_layer] = p_val

    # PUBLIC
    @classmethod
    def get_bm_layer_items(self, p_obj, bm, layerName):
        layer = self.get_mesh_layer(p_obj, bm, layerName)
        return [item for item in self.get_bm_items(bm) if self.is_bm_item_set(item, layer)] if layer else []

    @classmethod
    def get_bm_nonlayered_items(self, p_obj, bm, p_group_pairs):
        items = self.get_bm_items(bm)
        layers = [self.get_mesh_layer(p_obj, bm, g.layer_name) for _, g in p_group_pairs]

        return [item for item in items for layer in layers if not layer or not self.is_bm_item_set(item, layer)]

    @classmethod
    def remove_obj_list_layer(self, p_obj, layerName):
        p_list = self.get_list(p_obj)
        for i, group in enumerate(p_list):
            if group.layer_name == layerName:
                p_list.remove(i)
                break

    @classmethod
    def is_layer_present_in_scene(self, layerName):
        for obj in bpy.context.scene.objects:
            if obj.type == 'MESH':
                if self._get_group_by_layer(obj, layerName):
                    return True
        return False

    @classmethod
    def remove_scene_layer(self, p_scene, layerName):
        p_list = self.get_list(p_scene)
        i_layer_index = self._index_of_layer(p_list, layerName)
        if i_layer_index != -1:
            if not self.is_layer_present_in_scene(layerName):
                p_list.remove(i_layer_index)
                i_layer_index = i_layer_index - 1
        self.build_lookup_table(bpy.context)

        p_context_group_pairs = self.get_context_group_pairs(bpy.context)

        i_new_index = -1
        for i in range(len(p_context_group_pairs) - 1, -1, -1):
            if p_context_group_pairs[i][0] <= i_layer_index:
                i_new_index = p_context_group_pairs[i][0]
                break

        if i_new_index == -1 and len(p_context_group_pairs):
            i_new_index = p_context_group_pairs[0][0]

        self.set_list_index(p_scene, i_new_index)

    @classmethod
    def create_unique_layer_name(self):
        return self._create_unique_layer_name(self.list_layer_prefix())

    @classmethod
    def add_layer_to_list(self, p_obj, layerName, p_color):
        p_list = self.get_list(p_obj)
        return self._add_list_layer(p_list, layerName, self.list_item_prefix, p_color)

    @classmethod
    def ensure_group_in_scene(self, p_scene, layerName, groupName, groupColor):
        p_list = self.get_list(p_scene)
        return self._ensure_group_in_list(p_list, layerName, groupName, groupColor)

    @classmethod
    def ensure_group_in_object(self, p_obj, p_group):
        p_list = self.get_list(p_obj)
        return self._ensure_group_in_list(p_list, p_group.layer_name, p_group.name, p_group.group_color)

    @classmethod
    def _ensure_group_in_list(self, p_list, layerName, groupName, groupColor):
        index = -1
        for i, g in enumerate(p_list):
            if g.layer_name == layerName:
                index = i
                break

        if index == -1:
            p_list.add()
            index = len(p_list) - 1
            p_list[index].layer_name = layerName

        if p_list[index].name != groupName:
            p_list[index].name = groupName

        if p_list[index].group_color != groupColor:
            p_list[index].group_color = groupColor

        return p_list[index]

    @classmethod
    def get_list(self, p_scene_or_obj):
        return getattr(p_scene_or_obj, self.list_prop_name()) if p_scene_or_obj else []

    @classmethod
    def get_current_group_pairs(self, context):
        return [(i, g) for i, g in enumerate(self.get_list(context.scene))] \
            if get_prefs().list_options.display_all_scene_groups \
            else self.get_context_group_pairs(context)

    @classmethod
    def get_scene_group_pair(self, context):
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        i_items_count = len(p_list)
        i_list_index = self.get_list_index(p_scene)

        if i_list_index in range(i_items_count):
            return (i_list_index, p_list[i_list_index])
        else:
            return None

    @classmethod
    def get_current_group_pair(self, context):
        if get_prefs().list_options.display_all_scene_groups:
            return self.get_scene_group_pair(context)
        else:
            return self.get_context_group_pair(context)

    @classmethod
    def get_context_group_pair(self, context):
        p_scene = context.scene
        p_scene_list = self.get_list(p_scene)
        i_list_index = self.get_list_index(p_scene)
        if i_list_index in range(len(p_scene_list)):
            p_group = p_scene_list[i_list_index]
            p_info = self.get_lookup_extra_info_by_index(context, i_list_index)
            if p_info.get('objects', 0) != 0:
                return (i_list_index, p_group)
        return None

    @classmethod
    def get_context_group_pairs(self, context):
        return self.get_lookup_item(context, ZsLookup.CurrentGroupPairs)

    @classmethod
    def set_list_index(self, p_scene, i_list_index, lock=True):
        if lock:
            ZenLocks.lock_group_index(self.id_group)

        if hasattr(p_scene, self.active_list_prop_name()):
            setattr(p_scene, self.active_list_prop_name(), i_list_index)

    @classmethod
    def get_list_index(self, p_scene):
        return getattr(p_scene, self.active_list_prop_name()) \
            if (p_scene and hasattr(p_scene, self.active_list_prop_name())) else -1

    @classmethod
    def get_current_list_index(self, context):
        p_group_pair = self.get_current_group_pair(context)
        return p_group_pair[0] if p_group_pair else -1

    @classmethod
    def get_current_group_pair_index(self, context):
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            p_group_pairs = self.get_current_group_pairs(context)
            for i, pair in enumerate(p_group_pairs):
                if pair[0] == p_group_pair[0]:
                    return i
        return -1

    @classmethod
    def set_current_group_pair_index(self, context, i_pair_index, lock=True):
        p_group_pairs = self.get_current_group_pairs(context)
        if i_pair_index in range(len(p_group_pairs)):
            return self.set_list_index(context.scene, p_group_pairs[i_pair_index][0], lock=lock)

    @classmethod
    def set_mesh_select_mode(self, context):
        select_mode = self.get_mesh_select_mode()
        if context.tool_settings.mesh_select_mode[:] != select_mode:
            if bpy.ops.mesh.select_mode.poll():
                bpy.ops.mesh.select_mode(type=self.id_element.upper())
        if self.is_uv_area_and_not_sync():
            self.check_uv_select_mode()

    @classmethod
    def get_mesh_select_mode(self):
        # 0 - vert, 1 - edge, 2 - face
        Log.error('ABSTRACT> get_mesh_select_mode')
        return (False, False, False)

    @classmethod
    def check_for_new_layers(self, p_scene, p_objects):
        p_scene_list = self.get_list(p_scene)

        scene_group_map = {g.layer_name: i for i, g in enumerate(p_scene_list)}

        for p_obj in p_objects:
            if p_obj.type == 'MESH':
                handled_layers = []
                p_obj_list = self.get_list(p_obj)

                for i in range(len(p_obj_list)):
                    s_obj_layer = p_obj_list[i].layer_name
                    p_color = p_obj_list[i].group_color
                    i_scene_index = scene_group_map.get(s_obj_layer, -1)
                    # layer is not present
                    if i_scene_index == -1:
                        self.ensure_group_in_scene(p_scene, s_obj_layer, p_obj_list[i].name, p_color)
                    else:
                        if p_obj_list[i].name != p_scene_list[i_scene_index].name:
                            p_obj_list[i].name = p_scene_list[i_scene_index].name
                        if p_obj_list[i].group_color != p_scene_list[i_scene_index].group_color:
                            p_obj_list[i].group_color = p_scene_list[i_scene_index].group_color
                    handled_layers.append(s_obj_layer)

                if not self.is_blender and not self.is_unique:
                    bm = self._get_bm(p_obj)
                    items = self.get_bm_items(bm)

                    for mesh_layer_name in items.layers.int.keys():
                        if mesh_layer_name.startswith(self.list_layer_prefix()):
                            if mesh_layer_name not in handled_layers:
                                i_scene_index = scene_group_map.get(mesh_layer_name, -1)
                                if i_scene_index != -1:
                                    Log.warn(f'Object:[{p_obj.name}] - Layer:[{mesh_layer_name}] will be restored!')
                                    self.ensure_group_in_object(p_obj, p_scene_list[i_scene_index])
                                    self.update_obj_group_count(p_obj, mesh_layer_name)
                else:
                    # will be performed in 'basic_map_sets' update_all_obj_groups_count
                    pass

        self.build_lookup_table(bpy.context)

    @classmethod
    def is_lookup_table_valid(self, lookup_table, context):
        if ZsLookup.MetaInfo not in lookup_table:
            return False
        p_list = self.get_list(context.scene)
        if len(p_list) != lookup_table[ZsLookup.MetaInfo][0]:
            return False

        if len(p_list):
            if lookup_table[ZsLookup.MetaInfo][1] != p_list[0].layer_name:
                return False

        return True

    @classmethod
    def get_lookup_extra_info_layer(self, context):
        return self.get_lookup_item(context, ZsLookup.ExtraInfoLayer)

    @classmethod
    def get_lookup_item(self, context, id):
        lookup_table = get_lookup_table_check(self, context)

        if id not in lookup_table:
            self.build_lookup_table(context)

        return lookup_table[id]

    @classmethod
    def get_lookup_extra_info_by_index(self, context, index):
        try:
            t_info = self.get_lookup_item(context, ZsLookup.ExtraInfoIndex)[index]
            return t_info
        except Exception:
            return {}

    @classmethod
    def build_lookup_table(self, context):
        if ZenLocks.is_lookup_build_locked():
            return

        addon_prefs = get_prefs()
        lookup_table = get_lookup_table(self)

        interval = timer()
        p_scene_list = self.get_list(context.scene)

        i_list_count = len(p_scene_list)

        lookup_table[ZsLookup.Filter] = []
        lookup_table[ZsLookup.ExtraInfoIndex] = {}
        lookup_table[ZsLookup.ExtraInfoLayer] = {}
        lookup_table[ZsLookup.CurrentGroupPairs] = []
        lookup_table[ZsLookup.MetaInfo] = (i_list_count, p_scene_list[0].layer_name if i_list_count else '')
        lookup_table[ZsLookup.ObjectHighlightedGroups] = {}

        for obj in context.objects_in_mode:
            if obj not in lookup_table[ZsLookup.ObjectHighlightedGroups]:
                lookup_table[ZsLookup.ObjectHighlightedGroups][obj] = []

        for i, g in enumerate(p_scene_list):
            p_extra_info = self._do_get_lookup_extrainfo(context, (i, g), lookup_table)
            lookup_table[ZsLookup.ExtraInfoIndex][i] = p_extra_info
            lookup_table[ZsLookup.ExtraInfoLayer][g.layer_name] = p_extra_info

            n_obj_count = p_extra_info.get('objects', 0)

            if n_obj_count == 0 and addon_prefs.list_options.display_all_scene_groups is False:
                lookup_table[ZsLookup.Filter].append(i)
            else:
                lookup_table[ZsLookup.CurrentGroupPairs].append((i, g))
        Log.debug(f'LOOKUP> Mode:[{self.id_group}] completed:', timer() - interval)

    @classmethod
    def _do_get_lookup_extrainfo(cls, context: bpy.types.Context, p_group_pair, p_lookup_table):
        idx, p_group = p_group_pair

        n_group_count = 0
        n_obj_count = 0
        n_hide_count = 0
        for obj in context.objects_in_mode:
            group = cls._get_group_by_layer(obj, p_group.layer_name)
            n_current_count = -1
            n_current_hide_count = 0
            if group is not None:
                n_current_count = group.group_count
                n_current_hide_count = group.group_hide_count

            if n_current_count > -1:
                n_group_count = n_group_count + n_current_count
                n_obj_count = n_obj_count + 1
                n_hide_count = n_hide_count + n_current_hide_count
                if n_current_count > 0:
                    p_lookup_table[ZsLookup.ObjectHighlightedGroups][obj].append(idx)

        p_extra_info = {
            'count': n_group_count,
            'objects': n_obj_count,
            'hide_count': n_hide_count
        }

        return p_extra_info

    @classmethod
    def check_validate_list(cls, context):
        pass

    @classmethod
    def update_all_groups_count(self, p_objects):
        b_need_update = False
        for p_obj in p_objects:
            if self.update_all_obj_groups_count(p_obj, no_lookup=True):
                b_need_update = True
        if b_need_update:
            self.build_lookup_table(bpy.context)

    @classmethod
    def update_all_obj_groups_count(self, p_obj, no_lookup=False):
        p_obj_list = self.get_list(p_obj)
        b_need_update = False

        dic = defaultdict(list)
        layers = set()
        bm = self._get_bm(p_obj)
        for group in p_obj_list:
            layer = self.get_mesh_layer(p_obj, bm, group.layer_name)
            if layer:
                layers.add(layer)

        for item in self.get_bm_items(bm):
            for layer in layers:
                if self.is_bm_item_set(item, layer):
                    dic[layer.name].append(item.hide)

        for group in p_obj_list:
            i_group_count = 0
            i_hide_count = 0
            if group.layer_name in dic:
                arr = np.fromiter(dic[group.layer_name], 'b')
                i_group_count = len(arr)
                i_hide_count = np.count_nonzero(arr)
            if i_group_count != group.group_count or i_hide_count != group.group_hide_count:
                group.group_count = i_group_count
                group.group_hide_count = i_hide_count
                b_need_update = True

        if b_need_update and not no_lookup:
            self.build_lookup_table(bpy.context)
        return b_need_update

    @classmethod
    def update_obj_group_count(self, p_obj, layerName):
        group = self._get_group_by_layer(p_obj, layerName)
        if group is not None:
            p_info = self.get_obj_group_count(p_obj, group.layer_name)
            i_group_count = p_info.get('count', 0)
            i_hide_count = p_info.get('hide_count', 0)
            if i_group_count != group.group_count or i_hide_count != group.group_hide_count:
                group.group_count = i_group_count
                group.group_hide_count = i_hide_count
                self.build_lookup_table(bpy.context)

    @classmethod
    def get_context_selected_count(self, context):
        n_selected = 0
        for obj in context.objects_in_mode:
            n_selected = n_selected + self.get_selected_count(obj)
        return n_selected

    @classmethod
    def _add_draw_handler(self):
        if get_prefs().common.auto_highlight and not is_draw_handler_enabled(self):
            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')

    @classmethod
    def get_context_group_count(self, context, layerName):
        n_count = 0
        for obj in context.objects_in_mode:
            group = self._get_group_by_layer(obj, layerName)
            if group is not None:
                n_count = n_count + group.group_count
        return n_count

    @classmethod
    def _do_set_last_smart_select(self, layerName):
        bpy.context.scene.zen_sets_last_smart_layer = layerName

    @classmethod
    def _smart_process_item(self, item, b_is_uv, uv_layer,
                            is_present, b_is_face,
                            selected_loops,
                            unselected_loops):
        if b_is_uv and uv_layer:
            for loop in self.get_item_loops(item):
                if b_is_face or loop.face.select:
                    if is_present:
                        selected_loops.add(loop)
                    else:
                        unselected_loops.add(loop)
        else:
            item.select = is_present

    @classmethod
    def invert_context_uv_selection(self, context: bpy.types.Context):
        if self.id_element == 'face':
            bpy.ops.mesh.select_all(action='INVERT')
        else:
            for p_obj in context.objects_in_mode_unique_data:
                if p_obj.type == 'MESH':
                    bm = self._get_bm(p_obj)
                    bm.faces.ensure_lookup_table()
                    for face in bm.faces:
                        if not face.hide:
                            face.select = not face.select
                    bmesh.update_edit_mesh(p_obj.data, loop_triangles=True)

    @classmethod
    def _iterate_layer_tree(self, p_lay_col, p_lay_col_parent):
        yield (p_lay_col, p_lay_col_parent)
        for p_child in p_lay_col.children:
            yield from self._iterate_layer_tree(p_child, p_lay_col)

    @classmethod
    def smart_select(self, context, select_active_group_only, keep_active_group) -> ZenSelectionStats:
        Log.error('ABSTRACT> smart_select')
        return ZenSelectionStats()

    @classmethod
    def hide_group_by_index(self, context, group_index, props):
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        hidden_state = True
        if group_index in range(len(p_list)):
            t_info = self._get_layer_extra_info(context, p_list[group_index].layer_name)
            hide_count = t_info.get('hide_count', 0)
            hidden_state = 0 == hide_count

            status = self.set_group_pair_hidden_state(context, (group_index, p_list[group_index]), hidden_state, props)
            return status[0], status[1], hidden_state
        else:
            return False, False, hidden_state

    @classmethod
    def set_group_pair_hidden_state(self, context, p_group_pair, hidden_state, props):
        b_rebuild_lookup = False
        b_has_elements = False
        b_is_uv = self.is_uv_area_and_not_sync()
        b_select = getattr(props, 'select', False)
        p_group = p_group_pair[1]
        for p_obj in context.objects_in_mode:
            me = p_obj.data
            bm = bmesh.from_edit_mesh(me)
            items = self.get_bm_layer_items(p_obj, bm, p_group.layer_name)
            if len(items):
                b_has_elements = True

                for item in items:
                    item.hide_set(hidden_state)
                    if (b_select or b_is_uv) and not hidden_state:
                        item.select = True
                        if b_is_uv and b_select:
                            uv_layer = bm.loops.layers.uv.active
                            if uv_layer:
                                for loop in self.get_item_loops(item):
                                    loop[uv_layer].select = True
                                    if ZenPolls.version_greater_3_2_0:
                                        loop[uv_layer].select_edge = True

                bm.select_flush_mode()

                p_obj_group = self._get_group_by_layer(p_obj, p_group.layer_name)
                if p_obj_group:
                    i_count = len(items) if hidden_state else 0
                    if i_count != p_obj_group.group_hide_count:
                        p_obj_group.group_hide_count = i_count
                        b_rebuild_lookup = True

                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

        if b_rebuild_lookup:
            self.build_lookup_table(context)

        return b_rebuild_lookup, b_has_elements

    @classmethod
    def set_active_group_hidden_state(self, context, hidden_state, props):
        p_group_pair = self.get_scene_group_pair(context)
        return self.set_group_pair_hidden_state(context, p_group_pair, hidden_state, props) if p_group_pair else (False, False)

    @classmethod
    def set_group_pair_invert_hide(self, context, p_group_pair, props) -> Tuple[bool, bool]:
        b_rebuild_lookup = False
        b_has_elements = False
        b_was_inverted = False
        p_group = p_group_pair[1]
        b_select = props.select
        b_is_uv = self.is_uv_area_and_not_sync()
        for p_obj in context.objects_in_mode:
            me = p_obj.data
            bm = bmesh.from_edit_mesh(me)
            items = self.get_bm_items(bm)
            if len(items):
                layer_items = self.get_bm_layer_items(p_obj, bm, p_group.layer_name)
                if len(layer_items):
                    b_has_elements = True

                for item in self.get_bm_items(bm):
                    hidden_state = item not in layer_items
                    if not b_was_inverted:
                        if hidden_state and not item.hide:
                            b_was_inverted = True
                    item.hide_set(hidden_state)
                    if (b_select or b_is_uv) and not hidden_state:
                        item.select = True
                        if b_is_uv and b_select:
                            uv_layer = bm.loops.layers.uv.active
                            if uv_layer:
                                for loop in self.get_item_loops(item):
                                    loop[uv_layer].select = True
                                    if ZenPolls.version_greater_3_2_0:
                                        loop[uv_layer].select_edge = True

                bm.select_flush_mode()

                p_obj_group = self._get_group_by_layer(p_obj, p_group.layer_name)
                if p_obj_group:
                    if 0 != p_obj_group.group_hide_count:
                        p_obj_group.group_hide_count = 0
                        b_rebuild_lookup = True

                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
        if b_rebuild_lookup:
            self.build_lookup_table(context)

        return (b_was_inverted or b_rebuild_lookup), b_has_elements

    @classmethod
    def set_active_group_invert_hide(self, context, props):
        p_group_pair = self.get_scene_group_pair(context)
        return self.set_group_pair_invert_hide(context, p_group_pair, props) if p_group_pair else (False, False)

    @classmethod
    def unhide_all(self, context, b_select, props):
        bpy.ops.mesh.reveal(select=b_select)

    @classmethod
    def new_group(self, context, props):
        self.set_mesh_select_mode(context)
        p_scene = context.scene

        s_layer_name = self.create_unique_layer_name()
        p_scene_list = self.get_list(p_scene)
        p_color = self._gen_new_color(p_scene_list)
        i_list_index = self.add_layer_to_list(p_scene, s_layer_name, p_color)
        self.set_list_index(p_scene, i_list_index)

        p_scene.zen_sets_last_smart_layer = s_layer_name

        self.set_mesh_select_mode(context)

        b_indices_mode = getattr(props, "group_mode", None) == 'INDICES'
        indices_dict = defaultdict(set)
        if b_indices_mode:
            for idx in props.group_indices:
                indices_dict[idx.name].add(idx.item)

        for p_obj in context.objects_in_mode:
            if b_indices_mode and p_obj.name not in indices_dict:
                continue

            self.set_selection_to_group(p_obj, p_scene_list[i_list_index], indices_dict.get(p_obj.name, None))
            self.update_all_obj_groups_count(p_obj, no_lookup=True)
            mark_groups_modified(self, p_obj)

        self.build_lookup_table(context)

    # PUBLIC FUNCTIONS FOR OPERATORS
    @classmethod
    def execute_NewGroup(self, operator, context):
        try:
            self.check_uv_not_sync()

            self.new_group(context, operator)

            self._add_draw_handler()

            i_total_count = 0
            for p_obj in context.objects_in_mode:
                mark_groups_modified(self, p_obj)
                check_update_cache(self, p_obj)
                i_total_count += self.get_selected_count(p_obj)

            if i_total_count != 0 and self.is_uv_area_and_not_sync():
                layer_name = self.get_current_layer_name(context)
                i_total_count = self.get_context_group_count(context, layer_name)

            fix_undo_push_edit_mode('New Group')

            if i_total_count == 0:
                operator.report({'INFO'}, 'Group is Empty! Check in all Groups!')

                addon_prefs = get_prefs()
                if not addon_prefs.list_options.display_all_scene_groups:
                    addon_prefs.list_options.display_all_scene_groups = True
        except Exception as e:
            operator.report({'WARNING'}, str(e))
            return {'CANCELLED'}

        return {'FINISHED'}

    @classmethod
    def delete_group(self, context, layerName):
        p_scene = context.scene

        if get_prefs().list_options.display_all_scene_groups:
            self._do_delete_scene_group(context, layerName)
        else:
            self._do_delete_context_group(context, layerName)

        self.remove_scene_layer(p_scene, layerName)

        fix_undo_push_edit_mode('Delete Group')
        for p_obj in context.objects_in_mode:
            mark_groups_modified(self, p_obj)
            check_update_cache(self, p_obj)

        p_list = self.get_list(p_scene)
        if len(p_list) == 0 and is_draw_handler_enabled(self):
            remove_draw_handler(self, context)

    @classmethod
    def _do_delete_object_group(self, p_obj, layerName, update=True):
        if p_obj:
            bm = self._get_bm(p_obj)
            mesh_cleanup = False
            if self.is_unique:
                obj_layer = self._get_group_by_layer(p_obj, layerName)
                if obj_layer and obj_layer.group_count != 0:
                    mesh_cleanup = True

            self.remove_mesh_layer(p_obj, bm, layerName, mesh_cleanup)
            self.remove_obj_list_layer(p_obj, layerName)

            if update:
                bmesh.update_edit_mesh(p_obj.data, loop_triangles=False, destructive=False)

    @classmethod
    def _do_delete_context_group(self, context, layerName):
        for obj in context.objects_in_mode:
            self._do_delete_object_group(obj, layerName)

    @classmethod
    def _do_delete_scene_group(self, context, layerName):
        def _delete_group_internal(self, context):
            self._do_delete_context_group(context, layerName)

        self._perform_all_scene_edit(context, _delete_group_internal)

    @classmethod
    def execute_DeleteGroup(self, operator, context):
        s_layer_name = self.get_current_layer_name(context)
        if not s_layer_name == '':
            self.delete_group(context, s_layer_name)
            return {'FINISHED'}
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'CANCELLED'}

    @classmethod
    def get_api_group(self, context, props):
        p_scene = context.scene

        p_group = None

        layer_name = getattr(props, "identifier", "")
        group_name = getattr(props, "group_name", "")
        group_color = getattr(props, "group_color", (0, 0, 0))[:]

        p_list = self.get_list(p_scene)
        idx = None

        if layer_name != "":
            idx = self._index_of_layer(p_list, layer_name)
        elif props.group_name != "":
            idx = self._index_of_group_name(p_list, group_name)
        else:
            p_group = self._get_scene_group(p_scene)

        if idx is not None:
            if idx != -1:
                p_group = p_list[idx]
                if group_color != (0, 0, 0):
                    if p_group.group_color[:] != group_color:
                        p_group.group_color = group_color
                if group_name != '' and p_group.name != group_name:
                    p_group.name = group_name
            else:
                if layer_name == "":
                    layer_name = self.create_unique_layer_name()

                if group_color == (0, 0, 0):
                    group_color = self._gen_new_color(p_list)

                idx = self.add_layer_to_list(p_scene, layer_name, group_color)

                if group_name != "":
                    p_list[idx].name = group_name

            self.set_list_index(p_scene, idx)

            p_group = p_list[idx]

        return p_group

    @classmethod
    def execute_AssignToGroup(self, operator, context):
        try:
            self.check_uv_not_sync()

            p_group = self.get_api_group(context, operator)

            if p_group:
                self.set_mesh_select_mode(context)
                self._add_draw_handler()

                b_indices_mode = getattr(operator, "group_mode", None) == 'INDICES'
                indices_dict = defaultdict(set)
                if b_indices_mode:
                    for idx in operator.group_indices:
                        indices_dict[idx.name].add(idx.item)

                for p_obj in context.objects_in_mode:
                    if b_indices_mode and p_obj.name not in indices_dict:
                        continue

                    self.set_selection_to_group(p_obj, p_group, indices_dict.get(p_obj.name, None))
                    self.update_all_obj_groups_count(p_obj, no_lookup=True)
                    mark_groups_modified(self, p_obj)
                self.build_lookup_table(context)

                return {'FINISHED'}
            else:
                return self.execute_NewGroup(operator, context)
        except Exception as e:
            operator.report({'WARNING'}, str(e))

        return {'CANCELLED'}

    @classmethod
    def execute_AppendToGroup(self, operator, context):
        try:
            self.check_uv_not_sync()

            p_scene = context.scene

            p_group = self._get_scene_group(p_scene)

            if p_group:
                self.set_mesh_select_mode(context)
                self._add_draw_handler()

                for p_obj in context.objects_in_mode:
                    self.append_selection_to_group(p_obj, p_group)
                    mark_groups_modified(self, p_obj)
                return {'FINISHED'}
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_SelectUngroupped(self, operator, context):
        try:
            self.check_uv_not_sync()

            p_group_pairs = self.get_current_group_pairs(context)
            if len(p_group_pairs):
                self.set_mesh_select_mode(context)

                for p_obj in context.objects_in_mode:
                    self.select_ungroupped(p_obj, p_group_pairs)
            else:
                bpy.ops.mesh.select_all(action='SELECT')

            return {'FINISHED'}
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_SelectOnlyGroup(self, operator, context):
        try:
            self.check_uv_not_sync()
            s_layer_name = self.get_current_layer_name(context)
            if not s_layer_name == '':
                self.set_mesh_select_mode(context)
                for obj in context.objects_in_mode:
                    self.set_group_to_selection(obj, s_layer_name)
                return {'FINISHED'}
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_SelectAppendGroup(self, operator, context):
        try:
            self.check_uv_not_sync()
            s_layer_name = self.get_current_layer_name(context)
            if not s_layer_name == '':
                self.set_mesh_select_mode(context)
                for obj in context.objects_in_mode:
                    self.append_group_to_selection(obj, s_layer_name)
                return {'FINISHED'}
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_DeselectGroup(self, operator, context):
        try:
            self.check_uv_not_sync()
            s_layer_name = self.get_current_layer_name(context)
            if not s_layer_name == '':
                self.set_mesh_select_mode(context)
                for obj in context.objects_in_mode:
                    self.remove_group_from_selection(obj, s_layer_name)
                return {'FINISHED'}
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_RemoveSelectionFromGroup(self, operator, context):
        try:
            self.check_uv_not_sync()

            p_pairs = self.get_current_group_pairs(context) if operator.mode == 'ALL' else [self.get_current_group_pair(context)]
            b_modified = False
            for p_group_pair in p_pairs:
                if p_group_pair[1].layer_name != '':
                    self.set_mesh_select_mode(context)
                    for p_obj in context.objects_in_mode:
                        self.remove_selection_from_group(p_obj, p_group_pair[1].layer_name)
                        update_cache_group(self, p_obj, p_group_pair[1])
                        b_modified = True
            if b_modified:
                return {'FINISHED'}

            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_IntersectGroup(self, operator, context):
        try:
            self.check_uv_not_sync()

            s_layer_name = self.get_current_layer_name(context)
            if not s_layer_name == '':
                self.set_mesh_select_mode(context)
                for obj in context.objects_in_mode:
                    self.intersect_selection_with_group(obj, s_layer_name)
                return {'FINISHED'}
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_RenameGroups(self, operator, context):
        if operator.replace == '' and operator.find == '':
            operator.report({'WARNING'}, 'Nothing was defined to replace!')
            return {'CANCELLED'}

        p_context_group_pairs = self.get_context_group_pairs(context)
        i_start = operator.start_from
        is_modified = False

        try:
            for i, p_group in p_context_group_pairs:
                if operator.group_mode == 'VISIBLE':
                    t_info = self.get_lookup_extra_info_by_index(context, i)
                    if t_info.get('hide_count') != 0:
                        continue

                new_name = p_group.name
                if operator.find != '':
                    if operator.match_case:
                        new_name = new_name.replace(operator.find, operator.replace)
                    else:
                        new_name = ireplace(new_name, operator.find, operator.replace)
                else:
                    new_name = operator.replace

                if operator.use_counter:
                    new_name = new_name + str(i_start)
                    i_start += 1

                if p_group.name != new_name:
                    p_group.name = new_name
                    is_modified = True
        except Exception as e:
            operator.report({'ERROR'}, str(e))
            is_modified = True

        if is_modified:
            if context.mode == 'EDIT_MESH':
                fix_undo_push_edit_mode('Rename Groups')
            else:
                update_areas_in_all_screens()
                bpy.ops.ed.undo_push(message='Rename Groups')

            return {'FINISHED'}
        else:
            operator.report({'WARNING'}, 'No replace matches found!')

        return {'CANCELLED'}

    @classmethod
    def execute_DeleteEmptyGroups(self, operator, context):
        was_draw_enabled = is_draw_handler_enabled(self)
        remove_draw_handler(self, context)

        p_scene = context.scene

        p_list = self.get_list(p_scene)

        active_layer_name = ''
        p_group = self._get_scene_group(p_scene)
        if p_group:
            active_layer_name = p_group.layer_name

        def _internal_delete_all_objects_empty_groups(self, context):
            del_layers = set()
            was_modified = False
            for obj in context.objects_in_mode:
                b_require_update = False
                for g in reversed(self.get_list(obj)):
                    if g.group_count == 0:
                        del_layers.add(g.layer_name)
                        self._do_delete_object_group(obj, g.layer_name, update=False)
                        b_require_update = True
                if b_require_update:
                    was_modified = True
                    bmesh.update_edit_mesh(obj.data, loop_triangles=False, destructive=False)

            i_scene_group_count = len(p_list)
            for i in range(i_scene_group_count - 1, -1, -1):
                g = p_list[i]
                if g.layer_name in del_layers:
                    if not self.is_layer_present_in_scene(g.layer_name):
                        p_list.remove(i)
                        was_modified = True
                else:
                    if not self.is_layer_present_in_scene(g.layer_name):
                        p_list.remove(i)
                        was_modified = True
            return was_modified

        """ Set all objects in edit and execute """
        was_modified = self._perform_all_scene_edit(context, _internal_delete_all_objects_empty_groups)

        if was_modified:
            self.build_lookup_table(context)

        p_context_group_pairs = self.get_context_group_pairs(context)
        p_was_group_pair = self.get_group_pair_by_layer(p_context_group_pairs, active_layer_name)
        if p_was_group_pair:
            self.set_list_index(p_scene, p_was_group_pair[0])
        elif len(p_context_group_pairs):
            self.set_list_index(p_scene, p_context_group_pairs[0][0])
        else:
            self.set_list_index(p_scene, -1)

        if was_modified:
            fix_undo_push_edit_mode('Delete Empty Groups')
            for p_obj in context.objects_in_mode:
                mark_groups_modified(self, p_obj)
                check_update_cache(self, p_obj)

        if was_draw_enabled:
            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')

        return {'FINISHED'}

    @classmethod
    def delete_context_groups(self, context):
        remove_draw_handler(self, context)
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        p_context_group_pairs = self.get_context_group_pairs(context)

        for obj in context.objects_in_mode:
            self._do_cleanup_object(obj)

        self.set_list_index(p_scene, -1)

        del_indices = []
        for i, g in p_context_group_pairs:
            del_indices.append(i)

        del_indices.sort(reverse=True)
        for i in del_indices:
            if not self.is_layer_present_in_scene(p_list[i].layer_name):
                p_list.remove(i)

        self.build_lookup_table(context)

    @classmethod
    def are_all_group_objects_selected(self, context):
        p_scene = context.scene
        p_group = self._get_scene_group(p_scene)
        if not p_group:
            return True

        sel_objects = []
        was_objects = [obj.name for obj in context.objects_in_mode]
        for obj in bpy.context.scene.objects:
            if obj.type == 'MESH':
                p_obj_list = self.get_list(obj)
                if self._index_of_layer(p_obj_list, p_group.layer_name) != -1:
                    sel_objects.append(obj.name)

        return (not len(sel_objects)) or (Counter(was_objects) == Counter(sel_objects))

    @classmethod
    def execute_SelectSceneObjectsWith(self, operator, context):

        def is_with_vgroups_object(context: bpy.types.Context, p_obj: bpy.types.Object):
            return len(p_obj.vertex_groups) != 0

        def is_with_fmaps_object(context: bpy.types.Context, p_obj: bpy.types.Object):
            return len(p_obj.face_maps) != 0

        def is_with_modifiers_object(context: bpy.types.Context, p_obj: bpy.types.Object):
            return len(p_obj.modifiers) != 0

        fn_map = {
            'WITH_VGROUPS': is_with_vgroups_object,
            'WITH_FMAPS': is_with_fmaps_object,
            'WITH_MODIFIERS': is_with_modifiers_object,
        }

        return self._do_select_scene_objects_in_group(operator, context, fn_map.get(operator.mode))

    @classmethod
    def execute_SelectSceneObjectsInGroup(self, operator, context):
        p_scene = context.scene
        p_group = self._get_scene_group(p_scene)
        if not p_group:
            return True

        sel_objects = []
        was_active_obj = context.active_object.name if context.active_object else ''
        was_objects = [obj.name for obj in context.objects_in_mode]
        for obj in bpy.context.scene.objects:
            if obj.type == 'MESH':
                p_obj_list = self.get_list(obj)
                if self._index_of_layer(p_obj_list, p_group.layer_name) != -1:
                    sel_objects.append(obj.name)

        if len(sel_objects) and (Counter(was_objects) != Counter(sel_objects)):
            bpy.ops.object.mode_set(mode='OBJECT')
            bpy.ops.object.select_all(action='DESELECT')
            objs = []
            view_layer = bpy.context.view_layer
            for obj in bpy.context.scene.objects:
                if obj.name in sel_objects:
                    try:
                        if obj.hide_select:
                            if not operator.prop_make_selectable:
                                continue
                            else:
                                obj.hide_select = False
                        if obj.hide_get():
                            if not operator.prop_unhide:
                                continue
                            else:
                                obj.hide_set(False)
                    except Exception:
                        pass

                    # --- condition when Collection is excluded or hidden ---
                    if operator.prop_make_selectable and operator.prop_unhide:
                        ensure_object_in_viewlayer(obj, view_layer.layer_collection)

                    try:
                        obj.select_set(True)
                        objs.append(obj)
                        if obj.name == was_active_obj:
                            view_layer.objects.active = obj
                    except Exception as e:
                        Log.error(e)
                        continue
            if len(objs) and (view_layer.objects.active not in objs):
                view_layer.objects.active = objs[0]
            if bpy.ops.object.mode_set.poll():
                bpy.ops.object.mode_set(mode='EDIT')
            else:
                operator.report({'WARNING'}, 'Can not edit all selected objects!')

        return {'FINISHED'}

    @classmethod
    def _perform_all_scene_edit(self, context, onEditCallback):
        if len(bpy.context.scene.objects):
            were_objects, act_obj_name = save_viewlayer_objects_state()
            scene_objects = [obj.name for obj in bpy.context.scene.objects if obj.type == 'MESH']
            objs_in_mode = [obj.name for obj in context.objects_in_mode if obj.type == 'MESH']
            if Counter(objs_in_mode) != Counter(scene_objects):
                result = None
                try:
                    bpy.ops.object.mode_set(mode='OBJECT')

                    layers_state = {}
                    save_viewlayer_layers_state(bpy.context.view_layer.layer_collection, layers_state)
                    show_all_viewlayers(bpy.context.view_layer.layer_collection)
                    unhide_and_select_all_viewlayer_objects(act_obj_name)

                    bpy.ops.object.mode_set(mode='EDIT')

                    result = onEditCallback(self, context)

                    bpy.ops.object.mode_set(mode='OBJECT')
                    bpy.ops.object.select_all(action='DESELECT')

                    prepare_stored_objects_for_edit(were_objects, act_obj_name)
                    restore_viewlayer_layers(bpy.context.view_layer.layer_collection, layers_state)
                    bpy.ops.object.mode_set(mode='EDIT')
                    restore_viewlayer_objects(were_objects)
                except Exception as e:
                    Log.error('ALL_SCENE_EDIT:', e)

                return result

        """ Default edit mode """
        return onEditCallback(self, context)

    @classmethod
    def delete_scene_groups(self, context):
        remove_draw_handler(self, context)
        p_scene = context.scene
        p_list = self.get_list(p_scene)

        def _internal_delete_groups(self, context):
            for obj in context.objects_in_mode:
                self._do_cleanup_object(obj)

        self._perform_all_scene_edit(context, _internal_delete_groups)

        p_list.clear()
        self.set_list_index(p_scene, -1)
        self.build_lookup_table(context)

    @classmethod
    def execute_DeleteAllGroups(self, operator, context):
        if get_prefs().list_options.display_all_scene_groups:
            self.delete_scene_groups(context)
        else:
            self.delete_context_groups(context)

        fix_undo_push_edit_mode('Delete Empty Groups')
        for p_obj in context.objects_in_mode:
            mark_groups_modified(self, p_obj)
            check_update_cache(self, p_obj)

        return {'FINISHED'}

    @classmethod
    def execute_GroupLinked(self, operator: bpy.types.Operator, context: bpy.types.Context):
        addon_prefs = get_prefs()
        was_auto_highlight = addon_prefs.common.auto_highlight
        addon_prefs.common.auto_highlight = False
        bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='OFF')

        interval = timer()

        try:
            p_scene = context.scene
            p_scene_list = self.get_list(p_scene)
            i_list_index = -1

            ZenLocks.lock_lookup_build()
            ZenLocks.lock_depsgraph_update()

            default_delimits, use_custom = operator.get_delimits()

            for p_obj in context.objects_in_mode_unique_data:
                start_progress(context)
                if p_obj.type != 'MESH':
                    continue

                bm = self._get_bm(p_obj)
                bm.faces.ensure_lookup_table()

                me = p_obj.data

                if len(bm.faces):
                    hidden_indices = set(item.index for item in bm.faces if item.hide)

                    bpy.ops.mesh.select_all(action='DESELECT')

                    i_face_count = len(bm.faces)
                    for i_index in range(0, i_face_count, 1):
                        if bm.faces[i_index].hide:
                            continue

                        bm.faces[i_index].select_set(True)

                        was_sel = me.total_face_sel

                        if len(default_delimits):
                            bpy.ops.mesh.select_linked(delimit=default_delimits)

                        try:
                            if use_custom:
                                operator.execute_custom_delimiter()

                            sel_dif = me.total_face_sel - was_sel
                            if sel_dif >= operator.min_group_size:
                                s_layer_name = self.create_unique_layer_name()

                                p_color = self._gen_new_color(p_scene_list)
                                i_list_index = self.add_layer_to_list(p_scene, s_layer_name, p_color)
                                self.set_selection_to_new_group(p_obj, p_scene_list[i_list_index], i_index, i_face_count)
                            elif me.total_face_sel == 0:
                                bm.faces[i_index].select_set(True)

                            bpy.ops.mesh.hide(unselected=False)

                            bm = self._get_bm(p_obj)
                            bm.faces.ensure_lookup_table()
                        except Exception as e:
                            Log.error(e)
                            operator.report({'ERROR'}, 'Auto Groups failed! See console for details!')
                            break

                        Log.debug('Remaining:', i_face_count - i_index - 1, 'of:', i_face_count)
                        update_progress(context, int((i_face_count - i_index - 1) / i_face_count * 100))

                    bm = self._get_bm(p_obj)
                    bm.faces.ensure_lookup_table()
                    for item in bm.faces:
                        item.hide_set(item.index in hidden_indices)

                    bmesh.update_edit_mesh(p_obj.data, loop_triangles=False, destructive=False)
                    self.update_all_obj_groups_count(p_obj, no_lookup=True)

                    end_progress(context)

            self.set_list_index(p_scene, i_list_index)

        finally:
            ZenLocks.unlock_lookup_build()
            ZenLocks.unlock_depsgraph_update()
            self.build_lookup_table(context)

            fix_undo_push_edit_mode('Auto Groups')
            for p_obj in context.objects_in_mode:
                mark_groups_modified(self, p_obj)
                check_update_cache(self, p_obj)

            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')
            addon_prefs.common.auto_highlight = was_auto_highlight

            Log.debug('AutoGroups completed ==========>', timer() - interval)

        return {'FINISHED'}

    @classmethod
    def execute_GroupSiblings(self, operator, context):
        try:
            self.check_uv_not_sync()
            self.smart_select(context, operator.select_only_active_group, operator.keep_active_group)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
            return {'CANCELLED'}
        return {'FINISHED'}

    @classmethod
    def is_anyone_hidden(self, context):
        b_is_uv = self.is_uv_area_and_not_sync()

        for p_obj in context.objects_in_mode_unique_data:
            bm = self._get_bm(p_obj)
            bm_items = self.get_bm_items(bm)
            if b_is_uv:
                if any((item.hide or not item.select) for item in bm_items):
                    return True
            else:
                if any(item.hide for item in bm_items):
                    return True

        return False

    @classmethod
    def execute_SmartIsolate(self, operator, context):
        try:
            select_only_active_group = ZsOperatorAttrs.get_operator_attr('zsts.smart_select', 'select_only_active_group', False)
            keep_active_group = ZsOperatorAttrs.get_operator_attr('zsts.smart_select', 'keep_active_group', False)

            self.check_uv_not_sync()
            selection_stats = self.smart_select(context, select_only_active_group, keep_active_group)

            b_is_uv_area = self.is_uv_area()

            if selection_stats.selection_changed or (self.get_context_selected_count(context) != 0 and not self.is_anyone_hidden(context)):

                if context.mode == 'EDIT_MESH':
                    if b_is_uv_area:
                        bpy.ops.uv.hide(unselected=True)
                    else:
                        bpy.ops.mesh.hide(unselected=True)
                elif context.mode == 'OBJECT':
                    bpy.ops.object.hide_view_set(unselected=True)
            else:
                if context.mode == 'EDIT_MESH':
                    if b_is_uv_area:
                        bpy.ops.uv.reveal(select=False)
                    else:
                        bpy.ops.mesh.reveal(select=False)
                elif context.mode == 'OBJECT':
                    bpy.ops.object.hide_view_clear(select=False)

        except Exception as e:
            operator.report({'WARNING'}, str(e))
            return {'CANCELLED'}
        return {'FINISHED'}

    @classmethod
    def _move_index(self, p_scene, direction):
        """ Move index of an item render queue while clamping it. """
        index = self.get_list_index(p_scene)
        list_length = len(self.get_list(p_scene)) - 1
        # (index starts at 0)
        new_index = index + (-1 if direction == 'UP' else 1)
        self.set_list_index(p_scene, max(0, min(new_index, list_length)))

    @classmethod
    def execute_MoveItem(self, operator, context):
        direction = operator.direction
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        if len(p_list):
            p_group_pairs = self.get_current_group_pairs(context)
            i_old_index = self.get_current_group_pair_index(context)
            i_groups_count = len(p_group_pairs)
            if i_groups_count > 0 and i_old_index != -1:
                i_new_index = i_old_index
                if direction == 'DOWN':
                    i_new_index = i_old_index + 1
                    if i_new_index > i_groups_count - 1:
                        i_new_index = 0
                else:
                    i_new_index = i_old_index - 1
                    if i_new_index < 0:
                        i_new_index = i_groups_count - 1

                if i_new_index != i_old_index:
                    p_list.move(p_group_pairs[i_old_index][0], p_group_pairs[i_new_index][0])
                    self.set_current_group_pair_index(context, i_new_index)
                    self.build_lookup_table(context)

        return {'FINISHED'}

    @classmethod
    def execute_MarkSeams(self, operator, context, mark):
        if self.id_element == 'edge':
            p_group = self._get_scene_group(context.scene)
            if p_group:
                for obj in context.objects_in_mode:
                    bm = self._get_bm(obj)
                    for item in self.get_bm_layer_items(obj, bm, p_group.layer_name):
                        item.seam = mark
                    bmesh.update_edit_mesh(obj.data, loop_triangles=False, destructive=False)
                    ZenLocks.lock_depsgraph_update_one()
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
            return {'FINISHED'}
        else:
            operator.report({'ERROR'}, "Requires 'edge' based group!")
            return {'CANCELLED'}

    @classmethod
    def execute_SetSculptMask(self, operator: bpy.types.Operator, context: bpy.types.Context):
        p_group_pair = self.get_current_group_pair(context)

        b_is_uv = self.is_uv_area_and_not_sync()

        was_locked = ZenLocks.is_depsgraph_update_locked()

        def set_item(p_item, p_layer, p_value, p_handled_verts):
            if self.id_element == 'vert':
                if p_value == 0.0:
                    if p_item.index in p_handled_verts:
                        return
                else:
                    p_handled_verts.add(p_item.index)
                p_item[p_layer] = p_value
            else:
                for p_vert in p_item.verts:
                    if p_value == 0.0:
                        if p_vert.index in p_handled_verts:
                            return
                    else:
                        p_handled_verts.add(p_vert.index)
                    p_vert[p_layer] = p_value

        try:
            ZenLocks.lock_depsgraph_update()

            for p_obj in context.objects_in_mode_unique_data:
                if p_obj.type == 'MESH':
                    bm = self._get_bm(p_obj)
                    p_layer = bm.verts.layers.paint_mask.verify()

                    p_mesh_layer = None
                    if p_group_pair:
                        p_mesh_layer = self.get_mesh_layer(p_obj, bm, p_group_pair[1].layer_name)

                    p_selected_indices = set()
                    if operator.submode == 'SELECTED':
                        if b_is_uv:
                            p_selected_indices = self.fetch_uv_selections(bm)

                    p_handled_verts = set()
                    for item in self.get_bm_items(bm):
                        if operator.mode == 'UNMASK_ALL':
                            set_item(item, p_layer, 0.0, p_handled_verts)
                        else:
                            if operator.submode == 'SELECTED':
                                b_is_selected = not item.hide and (item.index in p_selected_indices if b_is_uv else item.select)
                            else:
                                if p_mesh_layer is None:
                                    continue

                                b_is_selected = self.is_bm_item_set(item, p_mesh_layer)

                            if operator.mode == 'MASK':
                                if b_is_selected:
                                    set_item(item, p_layer, operator.mask_value, p_handled_verts)
                            elif operator.mode == 'UNMASK':
                                if b_is_selected:
                                    set_item(item, p_layer, 0.0, p_handled_verts)
                            elif operator.mode == 'ISOLATE':
                                p_val = 0.0 if b_is_selected else operator.mask_value
                                set_item(item, p_layer, p_val, p_handled_verts)

                    bmesh.update_edit_mesh(p_obj.data, loop_triangles=False, destructive=False)
        finally:
            if not was_locked:
                ZenLocks.unlock_depsgraph_update()

        return {'FINISHED'}

    @classmethod
    def _do_cleanup_object_no_update(self, obj):
        bm = self._get_bm(obj)
        p_obj_list = self.get_list(obj)

        list_len = len(p_obj_list)
        for i in range(list_len - 1, -1, -1):
            g = p_obj_list[i]
            if len(bm.verts) > 10000 and list_len > 10:
                Log.info(f'{i+1} of {list_len} - Cleaning obj:[{obj.name}] group:[{g.name}]')
            self.remove_mesh_layer(obj, bm, g.layer_name)
            self.remove_obj_list_layer(obj, g.layer_name)

        bm_items = self.get_bm_items(bm)
        bm_layers = bm_items.layers.int
        layer_prefix = self.list_layer_prefix()
        for mesh_layer in reversed(bm_layers.values()):
            mesh_layer_name = mesh_layer.name
            if mesh_layer_name.startswith(layer_prefix):
                bm_layers.remove(mesh_layer)
                Log.debug('Removed extra mesh layer:', mesh_layer_name)

        p_obj_list.clear()

    @classmethod
    def _do_cleanup_object(self, obj):
        self._do_cleanup_object_no_update(obj)

    @classmethod
    def _do_separate_group_to_object(self, context: bpy.types.Context, p_ctx_objects, p_group, is_mesh_cut, b_join):
        if p_group is None:
            return set()

        p_new_objects = []

        for obj in p_ctx_objects:

            bpy.ops.mesh.select_all(action='DESELECT')

            all_objs = set(bpy.data.objects)

            self.set_group_to_selection(obj, p_group.layer_name)

            if self.get_selected_count(obj):
                if is_mesh_cut is False:
                    bpy.ops.mesh.duplicate()

                bpy.ops.mesh.separate(type='SELECTED')

                bmesh.update_edit_mesh(obj.data, loop_triangles=False)

                # new objects can be created till this moment
                all_objs = set(bpy.data.objects) - all_objs

                for p_new_obj in all_objs:
                    if b_join:
                        if len(p_new_objects) == 0:
                            p_new_obj.name = p_group.name
                    else:
                        p_new_obj.name = obj.name + "." + p_group.name

                    p_new_objects.append(p_new_obj)

        if b_join:
            if len(p_new_objects) > 1:
                if ZenPolls.version_greater_3_2_0:
                    with context.temp_override(active_object=p_new_objects[0], selected_editable_objects=p_new_objects):
                        bpy.ops.object.join()
                        return set([p_new_objects[0]])
                else:
                    override = context.copy()
                    override['active_object'] = p_new_objects[0]
                    override['selected_editable_objects'] = p_new_objects
                    bpy.ops.object.join(override)
                    return set([p_new_objects[0]])

        return set(p_new_objects)

    @classmethod
    def execute_SplitToObjects(self, op: bpy.types.Operator, context):

        self.set_mesh_select_mode(context)

        bpy.ops.mesh.reveal(select=False)

        created_objects = set()

        is_active_group = op.group_mode == 'ACTIVE'

        b_is_mesh_cut = op.is_mesh_cut if self.is_unique else False

        p_ctx_objects = tuple(context.objects_in_mode_unique_data)

        if is_active_group:
            p_group = self._get_scene_group(context.scene)
            if p_group:
                new_objects = self._do_separate_group_to_object(context, p_ctx_objects, p_group, b_is_mesh_cut, op.join)
                created_objects.update(new_objects)
        else:
            remove_draw_handler(self, context)

            p_list = self.get_list(context.scene)
            if len(p_list):
                start_progress(context)
                for i, g in enumerate(p_list):
                    new_objects = self._do_separate_group_to_object(context, p_ctx_objects, g, b_is_mesh_cut, op.join)
                    created_objects.update(new_objects)
                    update_progress(context, int(100 - (i / len(p_list) * 100)))

                end_progress(context)

        if b_is_mesh_cut:
            for obj in p_ctx_objects:
                if obj.type == 'MESH':
                    bm = self._get_bm(obj)
                    if len(bm.verts) == 0:
                        p_data = obj.data
                        for p_obj in context.view_layer.objects:
                            if p_obj.data == p_data:
                                bpy.data.objects.remove(p_obj)

        if len(created_objects):
            were_objects = {}
            act_obj_name = ''
            was_object_mode = False
            if context.mode == 'EDIT_MESH':
                were_objects, act_obj_name = save_viewlayer_objects_state()
                bpy.ops.object.mode_set(mode='OBJECT')
            elif context.mode == 'OBJECT':
                was_object_mode = True

            if context.mode == 'OBJECT':
                if len(were_objects):
                    prepare_stored_objects_for_edit(were_objects, act_obj_name)

                created_objects.intersection_update(bpy.context.view_layer.objects)
                for p_obj in created_objects:
                    p_obj.select_set(True)
                    if bpy.context.view_layer.objects.active is None:
                        bpy.context.view_layer.objects.active = p_obj

            if bpy.ops.object.mode_set.poll():
                bpy.ops.object.mode_set(mode='EDIT')
                if len(were_objects):
                    restore_viewlayer_objects(were_objects)

                self.execute_DeleteEmptyGroups(op, context)

                if was_object_mode:
                    bpy.ops.object.mode_set(mode='OBJECT')

        return {'FINISHED'}

    @classmethod
    def string_filter_match(self, text, props):
        if props.filter_name:
            if props.filter_type == 'WILDCARD':
                return (
                    fnmatch.fnmatchcase(text, props.filter_name)
                    if props.filter_case_sensitive else
                    fnmatch.fnmatch(text, props.filter_name))
            elif props.filter_type == 'REGEX':
                return (
                    re.match(props.filter_name, text)
                    if props.filter_case_sensitive else
                    re.match(props.filter_name, text, re.IGNORECASE)
                )
        return True

    @classmethod
    def export_to_json(self, context, props, active_layer_name=''):
        json_data = {}

        json_data['mode'] = self.id_group
        json_data['element'] = self.id_element
        json_data['unique'] = self.is_unique

        i_max = 0
        for p_obj in context.objects_in_mode:
            i_max = i_max + len(self.get_list(p_obj))

        json_data['progress_max'] = i_max

        start_progress(context)
        i_count = 0
        for p_obj in context.objects_in_mode:
            data = {}
            self._get_bm(p_obj)
            data['groups'] = {}
            p_obj_list = self.get_list(p_obj)
            bm = self._get_bm(p_obj)
            for group in p_obj_list:
                i_count = i_count + 1
                update_progress(context, int(i_count / i_max * 100))
                if active_layer_name != '':
                    if active_layer_name != group.layer_name:
                        continue
                else:
                    if not self.string_filter_match(group.name, props):
                        continue

                group_data = {}
                group_data['name'] = group.name
                group_data['color'] = (group.group_color.r, group.group_color.g, group.group_color.b)
                group_data['items'] = [item.index for item in self.get_bm_layer_items(p_obj, bm, group.layer_name)]
                data['groups'][group.layer_name] = group_data
            json_data[p_obj.name] = data
        end_progress(context)
        # DO NOT USE IDENT !!!
        return json.dumps(json_data)

    @classmethod
    def import_from_json(self, context, json_data):
        json_data = json.loads(json_data)

        try:
            ZenLocks.lock_lookup_build()
            start_progress(context)

            i_max = json_data['progress_max']
            p_last_group = None
            i_count = 0
            for p_obj in context.objects_in_mode:
                if p_obj.name in json_data:
                    obj_data = json_data[p_obj.name]
                    bm = self._get_bm(p_obj)
                    source_items = None
                    if json_data['element'] == 'vert':
                        source_items = bm.verts
                    elif json_data['element'] == 'edge':
                        source_items = bm.edges
                    elif json_data['element'] == 'face':
                        source_items = bm.faces
                    else:
                        raise Exception('JSON> Data element is not detected! (Possible: vert, edge, face')

                    bm_items = self.get_bm_items(bm)
                    if source_items is not None:
                        source_items.ensure_lookup_table()

                    for key, val in obj_data['groups'].items():
                        i_count = i_count + 1
                        update_progress(context, int(i_count / i_max * 100))

                        if self.is_blender:
                            layerName = val['name']
                        else:
                            layerName = key.replace(json_data['mode'], self.id_element)

                        p_group = self.ensure_group_in_scene(context.scene, layerName, val['name'], Color(val['color']))
                        p_last_group = p_group
                        self.ensure_group_in_object(p_obj, p_group)

                        layer = self.ensure_mesh_layer(p_obj, bm, layerName)
                        for elem in bm_items:
                            if self.is_bm_item_set(elem, layer):
                                self.set_bm_item(elem, layer, False)

                        for source_index in val['items']:
                            if type(source_items) == type(bm_items):
                                self.set_bm_item(bm_items[source_index], layer, True)
                            else:
                                if isinstance(bm_items, bmesh.types.BMVertSeq):
                                    if isinstance(source_items, bmesh.types.BMFaceSeq):
                                        for v in source_items[source_index].verts:
                                            self.set_bm_item(v, layer, True)
                                    elif isinstance(source_items, bmesh.types.BMEdgeSeq):
                                        for v in source_items[source_index].verts:
                                            self.set_bm_item(v, layer, True)
                                elif isinstance(bm_items, bmesh.types.BMEdgeSeq):
                                    if isinstance(source_items, bmesh.types.BMFaceSeq):
                                        for e in source_items[source_index].edges:
                                            self.set_bm_item(e, layer, True)
                                    elif isinstance(source_items, bmesh.types.BMVertSeq):
                                        for e in source_items[source_index].link_edges:
                                            self.set_bm_item(e, layer, True)
                                elif isinstance(bm_items, bmesh.types.BMFaceSeq):
                                    if isinstance(source_items, bmesh.types.BMEdgeSeq):
                                        for f in source_items[source_index].link_faces:
                                            self.set_bm_item(f, layer, True)
                                    elif isinstance(source_items, bmesh.types.BMVertSeq):
                                        for f in source_items[source_index].link_faces:
                                            self.set_bm_item(f, layer, True)
                    self.update_all_obj_groups_count(p_obj, no_lookup=True)

                    bmesh.update_edit_mesh(p_obj.data, loop_triangles=False, destructive=False)
            if p_last_group:
                p_scene = context.scene
                p_scene_list = self.get_list(p_scene)
                i_list_index = self._index_of_layer(p_scene_list, p_last_group.layer_name)
                self.set_list_index(p_scene, i_list_index)
        finally:
            ZenLocks.unlock_lookup_build()
            self.build_lookup_table(context)
            fix_undo_push_edit_mode('Paste Groups')
            for p_obj in context.objects_in_mode:
                mark_groups_modified(self, p_obj)
                check_update_cache(self, p_obj)
            end_progress(context)

    @classmethod
    def get_color_button_layout(self, layout):
        r = layout.row()
        r.ui_units_x = 1.2

        col = r.column()
        col.separator(factor=0.8)

        col.scale_y = 0.6
        return col

    @classmethod
    def _do_draw_color_icon(self, layout, p_group):
        col = self.get_color_button_layout(layout)
        col.prop(p_group, 'group_color', text='')

    @classmethod
    def _do_draw_highlight(self, context, layout, type='NONE'):
        p_scene = context.scene
        p_addon_prefs = get_prefs()
        row = layout.row()
        b_is_highlight_enabled = is_draw_handler_enabled(self)
        if b_is_highlight_enabled:
            p_item = self._get_scene_group(p_scene)
            if p_item:
                self._do_draw_color_icon(row, p_item)

        is_tool = type == 'TOOL'
        is_tool_but_not_header = is_tool and context.region.type != 'TOOL_HEADER'

        r = row.row(align=True if is_tool_but_not_header else False)
        l_col = r.column(align=True)
        r_col = r.column(align=True)
        r_col.alignment = 'RIGHT'
        l_col.operator('zsts.draw_highlight',
                       text=ZsLabels.OT_DRAW_HIGHLIGHT_LABEL,
                       depress=b_is_highlight_enabled, icon='OVERLAY')
        subrow = None
        if is_tool_but_not_header:
            subrow = layout.column(align=True)
        else:
            if is_tool:
                subrow = r_col.row(align=True)
            else:
                subrow = r_col

        if context.mode == 'OBJECT':
            if is_tool_but_not_header:
                self._do_draw_overlay_filter(context, subrow)
            else:
                subrow.popover(
                    panel="ZSTO_PT_OverlayFilter",
                    text="",
                    icon='FILTER',
                )
        else:
            subrow.active = b_is_highlight_enabled

            row_props = subrow

            row_props = subrow.row(align=True)
            row_props.alignment = 'EXPAND' if is_tool_but_not_header else 'RIGHT'

            row_props.prop(p_addon_prefs.uv_options, 'display_uv', icon='UV_DATA', icon_only=True)
            row_props.separator()

            row_props.prop(p_addon_prefs.common, 'display_mesh_mode', expand=True, icon_only=True)

            row_props.separator()

            row_props.popover(
                panel="ZSTS_PT_OverlayFilter",
                text="",
                icon='FILTER',
            )

    @classmethod
    def _do_draw_overlay_filter(self, context, layout):
        addon_prefs = get_prefs()
        if self.is_unique:
            layout.prop(addon_prefs.common, 'display_all_parts')

        if self.is_uv_editor_open_and_enabled():
            layout.label(text='UV Editor Options')
            layout.prop(addon_prefs.uv_options, 'selected_only')
            if self.id_element == 'vert':
                layout.prop(addon_prefs.display_3d, 'vert_use_zoom_factor')

        layout.prop(addon_prefs.common, 'auto_highlight')

        layout.label(text='Draw Cache')
        layout.prop(addon_prefs.common, 'auto_update_draw_cache')
        layout.operator('zsts.reset_draw_cache', icon='FILE_REFRESH')

    @classmethod
    def _do_draw_groups_list(self, layout, p_scene):
        layout.template_list(
            self.id_mask + '_UL_List',      # list type name
            "name",                         # list id
            p_scene,                        # dataptr
            self.list_prop_name(),          # prop name
            p_scene,                        # active_dataptr
            self.active_list_prop_name(),   # active_propname
            rows=5                          # default and min rows to display
        )

    @classmethod
    def get_draw_panel_header(self, context):
        custom_text = ''
        p_scene = context.scene
        addon_prefs = get_prefs()
        p_scene_list = self.get_list(p_scene)
        n_visible_groups = 0
        total_count = len(p_scene_list)

        if addon_prefs.list_options.display_all_scene_groups is False:
            lookup_extrainfo = self.get_lookup_extra_info_layer(context)
            for g in p_scene_list:
                try:
                    t_info = lookup_extrainfo[g.layer_name]
                    if t_info.get('objects', 0) > 0:
                        n_visible_groups = n_visible_groups + 1
                except KeyError:
                    pass
            if n_visible_groups != total_count:
                custom_text = f' - {n_visible_groups} of {total_count} Group(s)'

        if custom_text == '' and total_count > 0:
            custom_text = f' - {total_count} Group(s)'

        return custom_text

    @classmethod
    def execute_Draw(self, context: bpy.types.Context, layout: bpy.types.UILayout):
        p_scene = context.scene

        is_object_mode = context.mode == 'OBJECT'
        is_object_and_collection_mode = is_object_mode and ZenPolls.is_collection_mode(context)
        op_prefix = 'zsto' if is_object_mode else 'zsts'

        row = layout.row()
        col = row.column()
        self._do_draw_groups_list(col, p_scene)

        col = row.column(align=True)
        col.operator(op_prefix + '.new_group', text="", icon='ADD')
        col.operator(op_prefix + '.del_group', text="", icon='REMOVE')
        col.separator()

        col.menu('ZSTS_MT_GroupMenu', icon='DOWNARROW_HLT', text="")

        if not is_object_and_collection_mode and not self.is_blender:
            op_up = col.operator('zsts.move_group', text="", icon='TRIA_UP')
            op_up.direction = 'UP'
            op_down = col.operator('zsts.move_group', text="", icon='TRIA_DOWN')
            op_down.direction = 'DOWN'

        col.separator()
        col.operator(op_prefix + '.delete_groups_combo', text="", icon='TRASH')

        if self.id_group == 'blgroup':
            row = layout.row(align=True)

            self._do_draw_vertex_weight(context, row)

            row.separator()
            row.popover(panel='ZSBLVG_PT_WeightPresets', text='', icon='PREFERENCES')

        row = layout.row(align=True)
        row.label(text="Selection to group")

        row = layout.row(align=True)
        row.operator(op_prefix + '.append_to_group')
        if not is_object_and_collection_mode:
            row.menu('ZSTS_MT_AssignMenu', text="", icon='DOWNARROW_HLT')
            row.separator()

        op = row.operator(op_prefix + '.remove_from_group')
        op.mode = 'ACTIVE'
        row.menu('ZSTS_MT_RemoveMenu', text="", icon='DOWNARROW_HLT')

        row = layout.row()
        row.label(text="Group to selection")

        col = layout.column(align=True)
        row = col.row(align=True)
        row.operator('zsts.select_group')
        row.operator('zsts.deselect_group')
        row.operator('zsts.intersect_group')

        if not is_object_and_collection_mode:
            row = col.row(align=True)
            row.operator('zsts.select_ungroupped')

        row = col.row(align=True)
        row.operator('zsts.smart_select', icon_value=zs_icon_get(ZIconsType.SmartSelect))
        # row.menu('ZSTS_MT_IsolateMenu', text="", icon='DOWNARROW_HLT')

        layout.label(text="Display group")
        col = layout.column(align=True)

        row = col.row(align=True)
        row.operator('zsts.hide_group', text="Hide")
        row.operator('zsts.unhide_group', text="Unhide")

        b_emboss = True
        b_depress = False
        if is_object_and_collection_mode:
            p_group_pair = self.get_current_group_pair(context)
            if p_group_pair:
                p_collection = p_group_pair[1].collection
                if p_collection and p_collection != context.scene.collection:
                    wm = context.window_manager
                    op_props = wm.operator_properties_last('zsto.hide_viewport_by_index')

                    from .objects.collection_prop_set import ZsHideViewportCollectionProperty

                    has_changed, has_elements = ZsHideViewportCollectionProperty.isolate_group_pair_property_state(
                        self, context, p_group_pair, op_props, fake=True)
                    b_emboss = not has_changed and has_elements
                    b_depress = b_emboss
        row.operator(op_prefix + '.invert_hide_group', text="Isolate", depress=b_depress)

        row = col.row(align=True)
        row.operator('zsts.smart_isolate')

        self._do_draw_highlight(context, layout)

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        layout = menu.layout

        layout.operator("zsts.select_scene_objects")

        layout.separator()
        layout.operator("zsts.rename_groups")

        layout.separator()
        layout.operator("zsts.copy_to_clipboard", icon='COPYDOWN')
        layout.operator("zsts.paste_from_clipboard", icon='PASTEDOWN')

        layout.separator()
        layout.menu('ZSTS_MT_ImportMenu')
        layout.menu('ZSTS_MT_ExportMenu')

        layout.separator()
        layout.operator('zsts.global_cleanup')

        layout.separator()
        layout.operator('zsts.split_to_objects')  # icon='OBJECT_DATA'
        layout.operator('mesh.zsts_set_sculpt_mask')

    @classmethod
    def execute_DrawImport(cls, layout, context, is_menu=False):
        layout.operator('zsts.import_vertex_colors_to_groups')

    @classmethod
    def execute_DrawExport(cls, layout, context, is_menu=False):
        layout.operator('zsts.export_groups_to_vertex_colors')  # icon='GROUP_VCOL'

    @classmethod
    def execute_DrawTools(cls, tools, context):
        layout = tools.layout

        layout.operator('zsts.split_to_objects')  # icon='OBJECT_DATA'
        layout.operator('mesh.zsts_set_sculpt_mask')
        # layout.operator('zsts.import_vertex_colors_to_groups')
        # layout.operator('zsts.export_groups_to_vertex_colors')  # icon='GROUP_VCOL'

    @classmethod
    def execute_DrawListItemRightProps(
            cls, layout: bpy.types.UILayout, context: bpy.types.Context,
            item, index,
            p_info, addon_prefs):

        n_group_count = p_info.get('count', 0)
        n_obj_count = p_info.get('objects', 0)
        n_hide_count = p_info.get('hide_count', 0)

        layout.label(text=str(n_group_count) if n_obj_count else '-')
        if addon_prefs.list_options.display_objects_info:
            layout.label(text=str(n_obj_count) if n_obj_count else '-')

        if addon_prefs.list_options.display_hidden_groups_info:
            if n_obj_count:
                icon_hide_id = 'HIDE_OFF' if n_hide_count == 0 else ('HIDE_ON' if n_hide_count == n_group_count else 'RADIOBUT_OFF')
                op = layout.operator('zsts.internal_hide_group_by_index', text='', icon=icon_hide_id, emboss=False)
                op.group_index = index
            else:
                layout.label(text='-')
