# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

import bpy
import mathutils

import math
from collections import defaultdict

from ..blender_zen_utils import ZenLocks
from ..vlog import Log
from .basic_prop_list import (
    _get_prop_list_group_color, _on_update_group_hide_count,
    _on_update_color)


class ZsBlenderSets:

    @classmethod
    def get_bl_groups(self, p_obj: bpy.types.Object):
        Log.error('ABSTRACT> get_bl_groups')
        return []

    @classmethod
    def update_list(self, context: bpy.types.Context):

        b_was_locked_update = ZenLocks.is_depsgraph_update_locked()
        b_was_locked_lookup = ZenLocks.is_lookup_build_locked()

        b_is_vgroups = self.id_group == 'blgroup'

        try:
            ZenLocks.lock_lookup_build()
            ZenLocks.lock_depsgraph_update()

            p_scene = context.scene
            p_scene_list = self.get_list(context.scene)

            d_name_color = {}
            d_new_name_color = {}
            d_index_color = {}
            for i, p_group in enumerate(p_scene_list):
                p_color = mathutils.Color(p_group.group_color)
                d_name_color[p_group.name] = p_color
                d_index_color[i] = p_color

            d = defaultdict(list)

            p_active_name = ''
            if context.active_object:
                p_groups = self.get_bl_groups(context.active_object)
                p_active_group = p_groups.active
                if p_active_group:
                    p_active_name = p_active_group.name

            for p_obj in p_scene.objects:
                for i, p_group in enumerate(self.get_bl_groups(p_obj)):
                    d[p_group.name].append(p_obj)

            if tuple(d_name_color.keys()) != tuple(d.keys()):
                p_scene_list.clear()

                b_length_same = len(d_name_color) == len(d)

                idx = -1

                for i, k in enumerate(d.keys()):
                    p_color = d_name_color.get(k, None)
                    if p_color is None and b_length_same:
                        p_color = d_index_color[i]

                    if p_color is None:
                        p_color = self._gen_new_color(p_scene_list)

                    d_new_name_color[k] = p_color
                    self.add_layer_to_list(p_scene, k, p_color)

                    if k == p_active_name:
                        idx = i

                was_index = self.get_list_index(p_scene)
                if was_index != idx:
                    self.set_list_index(p_scene, idx)

            for p_obj in p_scene.objects:
                blgroups = [g.name for g in self.get_bl_groups(p_obj)]

                p_obj_list = self.get_list(p_obj)
                obj_groups = [g.layer_name for g in p_obj_list]

                if blgroups != obj_groups:
                    p_obj_list.clear()

                    for g in self.get_bl_groups(p_obj):
                        p_color = d_new_name_color.get(g.name, (0.5, 0.5, 0.5))
                        self.add_layer_to_list(p_obj, g.name, p_color)

                if p_obj in context.objects_in_mode:
                    for p_group in p_obj_list:
                        p_info = self.get_obj_group_count(p_obj, p_group.layer_name)
                        i_group_count = p_info.get('count', 0)
                        i_hide_count = p_info.get('hide_count', 0)
                        if i_group_count != p_group.group_count:
                            p_group.group_count = i_group_count
                        if i_hide_count != p_group.group_hide_count:
                            p_group.group_hide_count = i_hide_count

                        # special weight handling for vertex groups
                        if b_is_vgroups:
                            d_weight = p_info.get('weight', 0)
                            if not math.isclose(d_weight, p_group.group_weight, abs_tol=1e-06):
                                p_group.group_weight = d_weight

            s_last_layer = self._get_layer_name(p_scene)
            if p_scene.zen_sets_last_smart_layer != s_last_layer:
                p_scene.zen_sets_last_smart_layer = s_last_layer
        finally:
            if not b_was_locked_lookup:
                ZenLocks.unlock_lookup_build()
            if not b_was_locked_update:
                ZenLocks.unlock_depsgraph_update()


class ZsBlenderObjectPropList():
    def get_group_name(self):
        return self.get('name', '')

    def set_group_name(self, value):
        self['name'] = value
        self['layer_name'] = value

    def get_layer_name(self):
        return self.get('name', '')

    def set_layer_name(self, value):
        self['name'] = value
        self['layer_name'] = value

    name: bpy.props.StringProperty(
            name="Name",
            get=get_group_name,
            set=set_group_name
        )
    layer_name: bpy.props.StringProperty(
        name="LayerName",
        description="",
        get=get_layer_name,
        set=set_layer_name
    )
    group_color: _get_prop_list_group_color(on_update=None)
    group_count: bpy.props.IntProperty(name="GroupCount")
    group_hide_count: bpy.props.IntProperty(name="GroupHiddenCount", update=_on_update_group_hide_count)
    group_weight: bpy.props.FloatProperty(name='Group Weight')


class ZsBlenderScenePropList():
    def get_object_groups(self, p_obj: bpy.types.Object):
        Log.error('ABSTRACT> get_object_groups')
        return None

    def get_update_method(self):
        Log.error('ABSTRACT> get_update_method')
        return None

    def get_group_name(self):
        return self.get('name', '')

    def set_group_name(self, value):
        p_old_name = self.get('name', '')
        if p_old_name != value:
            self['name'] = value
            self['layer_name'] = value

            was_locked = ZenLocks.is_depsgraph_update_locked()
            was_lookup_locked = ZenLocks.is_lookup_build_locked()
            try:
                ZenLocks.lock_depsgraph_update()
                ZenLocks.lock_lookup_build()

                b_need_update = False

                for p_obj in bpy.context.objects_in_mode:
                    p_object_groups = self.get_object_groups(p_obj)
                    p_group = p_object_groups.get(p_old_name, None)
                    if p_group:
                        p_group.name = value
                        b_need_update = True

                p_update_method = self.get_update_method()

                if b_need_update:
                    if bpy.app.timers.is_registered(p_update_method):
                        bpy.app.timers.unregister(p_update_method)

                    bpy.app.timers.register(p_update_method)
            finally:
                if not was_lookup_locked:
                    ZenLocks.unlock_lookup_build()
                if not was_locked:
                    ZenLocks.unlock_depsgraph_update()

    def get_layer_name(self):
        return self.get('name', '')

    def set_layer_name(self, value):
        self['name'] = value
        self['layer_name'] = value

    name: bpy.props.StringProperty(
        name="Name",
        get=get_group_name,
        set=set_group_name
    )
    layer_name: bpy.props.StringProperty(
        name="LayerName",
        description="",
        get=get_layer_name,
        set=set_layer_name
    )
    group_color: _get_prop_list_group_color(on_update=_on_update_color)


class ZsBlenderGroupIndexSync:
    def __init__(self) -> None:
        self.active_object = None
        self.active_blgroup_index = None
        self.active_name = ''
        self.lock = False

    def get_cls_mgr(self):
        Log.error('ABSTRACT> get_cls_mgr')
        return None

    def set_active_object(self, context: bpy.types.Context):
        if not self.lock:
            try:
                self.lock = True

                p_obj = context.active_object
                if p_obj:
                    p_scene = context.scene
                    p_cls_mgr = self.get_cls_mgr()
                    p_blender_groups = p_cls_mgr.get_bl_groups(p_obj)
                    bl_idx = p_blender_groups.active_index
                    p_list = p_cls_mgr.get_list(p_scene)
                    was_list_idx = p_cls_mgr.get_list_index(p_scene)

                    p_blender_name = ''
                    if bl_idx in range(len(p_blender_groups)):
                        p_bl_group = p_blender_groups[bl_idx]
                        p_blender_name = p_bl_group.name

                    if self.active_object != p_obj or self.active_blgroup_index != bl_idx or self.active_name != p_blender_name:
                        self.active_object = p_obj
                        self.active_blgroup_index = bl_idx
                        self.active_name = p_blender_name

                        idx = p_cls_mgr._index_of_layer(p_list, p_blender_name)
                        if was_list_idx != idx:
                            p_cls_mgr.set_list_index(p_scene, idx)
                            p_cls_mgr._do_set_last_smart_select(p_blender_name)

                            Log.debug(p_cls_mgr.list_item_prefix + '> set active list index:', idx)
            finally:
                self.lock = False

    def set_active_list_index(self, context: bpy.types.Context, idx: int):
        if not self.lock:
            try:
                self.lock = True

                p_obj = context.active_object
                if p_obj:
                    self.active_object = p_obj
                    p_scene = context.scene
                    p_cls_mgr = self.get_cls_mgr()
                    p_blender_groups = p_cls_mgr.get_bl_groups(p_obj)
                    p_list = p_cls_mgr.get_list(p_scene)
                    if idx in range(len(p_list)):
                        p_name = p_list[idx].layer_name
                        self.active_blgroup_index = p_blender_groups.find(p_name)
                        if self.active_blgroup_index != -1:
                            self.active_name = p_name
                        else:
                            self.active_name = ''
                    else:
                        self.active_blgroup_index = -1
                        self.active_name = ''

                    if p_blender_groups.active_index != self.active_blgroup_index:
                        p_blender_groups.active_index = self.active_blgroup_index
                        Log.debug(p_cls_mgr.list_item_prefix + '> set active vertex groups index:', self.active_blgroup_index)

            finally:
                self.lock = False
