# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

import bpy

from .basic_colors import ColorUtils

from ..blender_zen_utils import (
    ZsUiConstants,
    ZenModeSwitcher,
    ZenLocks)
from ..labels import ZsLabels


def _gen_palette_color(colors):
    return ColorUtils.gen_new_color(colors, 'color')


class ZSTS_OT_PaletteSelectColor(bpy.types.Operator):
    bl_idname = "zsts.palette_select_color"
    bl_label = ""
    bl_description = ZsLabels.OT_PALETTE_SELECT_COLOR_DESC
    bl_options = {'UNDO'}

    color_index: bpy.props.IntProperty()

    def execute(self, context):
        p_scene = context.scene
        pal = p_scene.zsts_color_palette.palette
        if pal:
            if self.color_index in range(len(pal.colors)):
                pal.colors.active = pal.colors[self.color_index]

        return {'FINISHED'}


class ZSTS_OT_PaletteAddNew(bpy.types.Operator):
    bl_idname = "zsts.palette_add_new"
    bl_label = ZsLabels.OT_PALETTE_ADD_NEW_PALETTE_LABEL
    bl_description = ZsLabels.OT_PALETTE_ADD_NEW_PALETTE_DESC
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        p_scene = context.scene
        pal = bpy.data.palettes.new('Zen Palette')
        if pal:
            p_scene.zsts_color_palette.palette = pal
            return {'FINISHED'}

        return {'CANCELLED'}


class ZSTS_OT_PaletteColorAdd(bpy.types.Operator):
    bl_idname = "zsts.palette_color_add"
    bl_label = ZsLabels.OT_PALETTE_COLOR_ADD_LABEL
    bl_description = ZsLabels.OT_PALETTE_COLOR_ADD_DESC
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.scene.zsts_color_palette.palette is not None

    def execute(self, context):
        pal = context.scene.zsts_color_palette.palette
        if pal:
            color_value = _gen_palette_color(pal.colors)
            new_color = pal.colors.new()
            new_color.color = color_value
            new_color.strength = 1.0
            pal.colors.active = new_color
            return {'FINISHED'}
        else:
            return {'CANCELLED'}


class ZSTS_OT_PaletteColorRemove(bpy.types.Operator):
    bl_idname = "zsts.palette_color_remove"
    bl_label = ZsLabels.OT_PALETTE_COLOR_REMOVE_LABEL
    bl_description = ZsLabels.OT_PALETTE_COLOR_REMOVE_DESC
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        pal = context.scene.zsts_color_palette.palette
        return True if (pal and pal.colors.active) else False

    def execute(self, context):
        pal = context.scene.zsts_color_palette.palette
        if pal and pal.colors.active:
            pal.colors.remove(pal.colors.active)
            return {'FINISHED'}
        else:
            return {'CANCELLED'}


class ZSTS_OT_PaletteColorMove(bpy.types.Operator):
    bl_idname = "zsts.palette_color_move"
    bl_label = ZsLabels.OT_PALETTE_COLOR_MOVE_LABEL
    bl_description = ZsLabels.OT_PALETTE_COLOR_MOVE_DESC
    bl_options = {'REGISTER', 'UNDO'}

    type: bpy.props.EnumProperty(
        items=(
            ('UP', 'Up', ''),
            ('DOWN', 'Down', '')
        ))

    @classmethod
    def poll(cls, context):
        pal = context.scene.zsts_color_palette.palette
        return True if (pal and pal.colors.active) else False

    def execute(self, context):
        pal = context.scene.zsts_color_palette.palette
        if pal and pal.colors.active:
            try:
                ZenLocks.lock_depsgraph_update()
                ZenLocks.lock_notify_edit()

                switch = ZenModeSwitcher(mode='VERTEX_PAINT')
                vpaint = context.tool_settings.vertex_paint
                was_palette = vpaint.palette
                vpaint.palette = pal
                if bpy.ops.palette.color_move.poll():
                    bpy.ops.palette.color_move(type=self.type)
                    vpaint.palette = was_palette
                switch.return_to_mode()
            finally:
                ZenLocks.unlock_depsgraph_update()
            return {'FINISHED'}
        else:
            return {'CANCELLED'}


class ZSTS_OT_PaletteColorSort(bpy.types.Operator):
    bl_idname = "zsts.palette_color_sort"
    bl_label = ZsLabels.OT_PALETTE_COLOR_SORT_LABEL
    bl_description = ZsLabels.OT_PALETTE_COLOR_SORT_DESC
    bl_options = {'REGISTER', 'UNDO'}

    type: bpy.props.EnumProperty(
        name='Sort Palette Colors',
        description='Sort By: Hue, Saturation, Value, Luminance',
        items=(
            ('HSV', 'Hue', ''),
            ('SVH', 'Saturation', ''),
            ('VHS', 'Value', ''),
            ('LUMINANCE', 'Luminance', '')
        ))

    @classmethod
    def poll(cls, context):
        pal = context.scene.zsts_color_palette.palette
        return pal is not None

    def execute(self, context):
        pal = context.scene.zsts_color_palette.palette
        if pal:
            try:
                ZenLocks.lock_depsgraph_update()
                ZenLocks.lock_notify_edit()

                switch = ZenModeSwitcher(mode='VERTEX_PAINT')
                vpaint = context.tool_settings.vertex_paint
                was_palette = vpaint.palette
                vpaint.palette = pal
                if bpy.ops.palette.color_move.poll():
                    bpy.ops.palette.sort(type=self.type)
                    vpaint.palette = was_palette
                switch.return_to_mode()
            finally:
                ZenLocks.unlock_depsgraph_update()
            return {'FINISHED'}
        else:
            return {'CANCELLED'}


class ZSTS_PT_ColorPresets(bpy.types.Panel):
    bl_space_type = ZsUiConstants.ZSTS_SPACE_TYPE
    bl_label = ""
    bl_parent_id = 'ZSTS_PT_ComboSets'
    bl_region_type = ZsUiConstants.ZSTS_REGION_TYPE
    bl_category = ZsUiConstants.ZSTS_PANEL_CATEGORY
    bl_options = {'DEFAULT_CLOSED'}

    def draw_header(self, context):
        layout = self.layout
        s_mode = layout.enum_item_name(context.scene.zsts_color_palette, 'mode', context.scene.zsts_color_palette.mode)
        layout.label(text=f'Color Management - {s_mode}')

    @classmethod
    def poll(cls, context):
        return True

    def draw(self, context):
        p_scene = context.scene

        layout = self.layout

        width = context.region.width
        ui_scale = context.preferences.system.ui_scale

        cols_per_row = max(int(width / (30 * ui_scale)), 1)

        row = layout.row(align=True)
        row.prop(p_scene.zsts_color_palette, 'mode', expand=True)

        if p_scene.zsts_color_palette.mode != 'AUTO':
            pal_layout = layout
            pal_layout.active = p_scene.zsts_color_palette.mode != 'AUTO'

            pal_layout.template_ID(p_scene.zsts_color_palette, 'palette', new=ZSTS_OT_PaletteAddNew.bl_idname)
            pal = p_scene.zsts_color_palette.palette
            if pal:
                row = pal_layout.row(align=True)
                row.alignment = 'LEFT'
                row.operator(ZSTS_OT_PaletteColorAdd.bl_idname, text='', icon='ADD')
                row.operator(ZSTS_OT_PaletteColorRemove.bl_idname, text='', icon='REMOVE')
                row.operator(ZSTS_OT_PaletteColorMove.bl_idname, text='', icon='TRIA_UP').type = 'UP'
                row.operator(ZSTS_OT_PaletteColorMove.bl_idname, text='', icon='TRIA_DOWN').type = 'DOWN'
                row.operator_menu_enum(ZSTS_OT_PaletteColorSort.bl_idname, 'type', text='', icon='SORTSIZE')

                col = pal_layout

                for i, color in enumerate(pal.colors):
                    if i % cols_per_row == 0:
                        flow = pal_layout.column_flow(columns=cols_per_row, align=True)

                    is_color_active = pal.colors.active == color
                    op_icon = "LAYER_ACTIVE" if is_color_active else "LAYER_USED"
                    col = flow.column(align=True)
                    # col.active = is_color_active
                    col.prop(color, 'color', text='', emboss=True)
                    col.operator('zsts.palette_select_color',
                                 text='', emboss=is_color_active, icon=op_icon).color_index = i


class ZsColorPaletteGroup(bpy.types.PropertyGroup):
    palette: bpy.props.PointerProperty(name='Color Palette', type=bpy.types.Palette)
    mode: bpy.props.EnumProperty(
        name='Generate New Color Mode',
        items=(
            ('AUTO', 'Auto', 'Auto Color'),
            ('PAL_SEQ', 'Sequence', 'Palette Sequence Color'),
            ('PAL_RND', 'Random', 'Palette Random Color')
        ),
        default='AUTO'
    )


class ZsColorPaletteFactory:
    classes = (
        ZSTS_OT_PaletteSelectColor,
        ZSTS_OT_PaletteAddNew,
        ZSTS_OT_PaletteColorAdd,
        ZSTS_OT_PaletteColorRemove,
        ZSTS_OT_PaletteColorMove,
        ZSTS_OT_PaletteColorSort,

        ZSTS_PT_ColorPresets,
    )

    @classmethod
    def register(cls):
        bpy.utils.register_class(ZsColorPaletteGroup)
        bpy.types.Scene.zsts_color_palette = bpy.props.PointerProperty(type=ZsColorPaletteGroup)

    @classmethod
    def unregister(cls):
        del bpy.types.Scene.zsts_color_palette
        bpy.utils.unregister_class(ZsColorPaletteGroup)
