# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

import bpy
import bpy_extras.view3d_utils
import bgl
import blf
import gpu
import bmesh
import uuid
from gpu_extras.batch import batch_for_shader

from timeit import default_timer as timer
from collections import defaultdict
from mathutils import Vector
import numpy as np
import math

from ..blender_zen_utils import ZenLocks, is_any_uv_editor_opened

from ..vlog import Log
from ..preferences import get_prefs
from ..hash_utils import hash32


LINE_VERTEX_SHADER = '''
    uniform mat4 viewProjectionMatrix;
    uniform float z_offset = 0.0;

    in vec3 pos;

    void main()
    {
        gl_Position = viewProjectionMatrix * vec4(pos, 1.0f);
        gl_Position.z -= z_offset;
    }
'''

LINE_FRAGMENT_SHADER = '''
    uniform vec4 color;
    out vec4 FragColor;

    void main()
    {
        FragColor = color;
    }
'''


class BasicCacher:
    def __init__(self) -> None:
        self.obj = None
        self.last_bbox_data = ()
        self.volume = 0
        self.layer_name = ''
        self.batch = None
        self.mtx = None
        self.z_order = 1
        self.force_rebuild_eval_cache = False
        self.bound_box = None
        self.me = None
        self.bm = None
        self.mesh_statistics = ''
        self.mesh_verts_stats = None
        self.last_bm_data = ''
        self.mesh_uuid = None
        self.mesh_cache_uuid = None

        self.uv_batch = None
        self.uv_shader = gpu.shader.from_builtin('2D_UNIFORM_COLOR')
        self.uv_cache_uuid = None
        self.uv_sync_mode = None
        self.uv_layer_name = None
        self.uv_sel_only = None
        pass

    def get_bm_data(self, bm: bmesh.types.BMesh):
        first_vert = (0, 0, 0)
        last_vert = (0, 0, 0)
        v_size = len(bm.verts)
        if v_size != 0:
            bm.verts.ensure_lookup_table()
            first_vert = bm.verts[0].co.to_tuple()
            last_vert = bm.verts[v_size - 1].co.to_tuple()
        return (len(bm.verts), len(bm.edges), len(bm.faces)), (first_vert, last_vert)

    def get_mesh_data(self, mesh: bpy.types.Mesh):
        first_vert = (0, 0, 0)
        last_vert = (0, 0, 0)
        v_size = len(mesh.vertices)
        if v_size != 0:
            first_vert = mesh.vertices[0].co.to_tuple()
            last_vert = mesh.vertices[v_size - 1].co.to_tuple()
        return (len(mesh.vertices), len(mesh.edges), len(mesh.polygons)), (first_vert, last_vert)

    def get_bound_box(self, p_obj):
        return [Vector(v) for v in p_obj.bound_box]

    def ensure_mesh(self, p_obj):
        if self.me is None:
            self.me = p_obj.to_mesh()

    def get_bm(self, p_obj):
        if p_obj.is_evaluated:
            if not self.bm:
                self.bm = bmesh.new(use_operators=False)
                self.ensure_mesh(p_obj)
                self.bm.from_mesh(self.me)
            return self.bm
        else:
            return bmesh.from_edit_mesh(p_obj.data)

    def cleanup_mesh(self, p_obj):
        if self.me:
            p_obj.to_mesh_clear()
            self.me = None
        if self.bm:
            self.bm.free()
            self.bm = None

    def get_cache_vol(self, p_obj):
        # Do not combine into single line! Much worse performance!
        if p_obj.is_evaluated:
            self.ensure_mesh(p_obj)
            self.mesh_statistics, self.mesh_verts_stats = self.get_mesh_data(self.me)
            vol = [f.area for f in self.me.polygons if not f.hide]
            return sum(vol)
        else:
            bm = bmesh.from_edit_mesh(p_obj.data)
            self.mesh_statistics, self.mesh_verts_stats = self.get_bm_data(bm)
            vol = [f.calc_area() for f in bm.faces if not f.hide]
            total = sum(vol)
            return total

    def recalc_mesh(self, p_mgr_cls, p_obj, volume=None):
        self.obj = p_obj

        self.volume = self.get_cache_vol(p_obj) if volume is None else volume
        self.last_bbox_data = self.get_bound_box(p_obj)
        self.last_bm_data = self.mesh_statistics, self.mesh_verts_stats
        self.layer_name = ''
        self.mesh_uuid = uuid.uuid4()

    def is_bmesh_valid(self, p_obj: bpy.types.Object):
        if self.obj != p_obj:
            return False

        # 1) check that verts, edges, faces, loops are the same
        new_bm_data = ''

        if p_obj.is_evaluated and self.last_bbox_data != self.get_bound_box(p_obj):
            # print('Modified bbox: Last', self.last_bbox_data)
            # print('Modified bbox: New', self.get_bound_box(p_obj))
            return False

        if p_obj.is_evaluated:
            self.ensure_mesh(p_obj)
            new_bm_data = self.get_mesh_data(self.me)
        else:
            bm = bmesh.from_edit_mesh(p_obj.data)
            new_bm_data = self.get_bm_data(bm)
        if self.last_bm_data != new_bm_data:
            # print('Modified elems: Last', self.last_bm_data)
            # print('Modified elems: New', new_bm_data)
            return False

        return True

    def is_bmesh_same(self, p_obj):
        if not self.is_bmesh_valid(p_obj):
            return False, None

        # 2) check that volume remains same
        vol = self.get_cache_vol(p_obj)

        if self.volume != vol:
            # print('Modified vol: Last:', self.volume, 'New:', vol)
            return False, vol

        return True, vol

    def check_draw_cache_prepared(self, p_mgr_cls, p_obj, p_group, p_display_groups, modes={}, b_calc_vol=False):
        if b_calc_vol:
            result, vol = self.is_bmesh_same(p_obj)
            if result is False:
                self.recalc_mesh(p_mgr_cls, p_obj, volume=vol)
        else:
            if not self.is_bmesh_valid(p_obj):
                self.recalc_mesh(p_mgr_cls, p_obj)

        if not p_mgr_cls.is_unique:
            s_layer_name = p_group.layer_name if p_group else ''
            if self.layer_name != s_layer_name:
                self.mesh_cache_uuid = None
            if self.uv_layer_name != s_layer_name:
                self.uv_cache_uuid = None

        if 'MESH' in modes:
            if self.mesh_uuid != self.mesh_cache_uuid:
                try:
                    self.build_cache(p_mgr_cls, p_obj, p_group, p_display_groups)
                except ReferenceError as e:
                    Log.error('BUILD CACHE:', e)
                    self.recalc_mesh(p_mgr_cls, p_obj, volume=self.volume)
                    self.build_cache(p_mgr_cls, p_obj, p_group, p_display_groups)

        if 'UV' in modes:
            addon_prefs = get_prefs()
            sel_uv_only = addon_prefs.uv_options.selected_only
            if addon_prefs.uv_options.display_uv and \
                not p_obj.is_evaluated and (('MESH' not in modes) or is_any_uv_editor_opened(bpy.context)) and \
                (b_calc_vol or self.uv_sync_mode != p_mgr_cls.is_uv_sync() or
                 self.mesh_uuid != self.uv_cache_uuid or self.uv_sel_only != sel_uv_only):

                self.build_uv_batch(p_mgr_cls, p_obj, p_group, p_display_groups, sel_uv_only)

    def build_cache(self, p_mgr_cls, p_obj, p_active_group, p_display_groups):
        self.batch = None
        self.mesh_cache_uuid = self.mesh_uuid
        self.force_rebuild_eval_cache = True
        if p_active_group:
            self.build_group_cache(p_mgr_cls, p_obj, p_active_group, p_display_groups)

    def build_group_cache(self, p_mgr_cls, p_obj, p_group, p_display_groups):
        self.force_rebuild_eval_cache = True

    def ensure_loops(self, bm):
        try:
            if self.loops is None:
                self.loops = bm.calc_loop_triangles()
            else:
                if len(self.loops):
                    looptris = self.loops[0]
                    if len(looptris):
                        loop = looptris[0]
                        loop.face
        except Exception as e:
            Log.debug('ENSURE LOOPS:', e)
            self.loops = bm.calc_loop_triangles()
        return self.loops

    def get_edge_z_offset(self, option_enabled):
        if option_enabled is False:
            return 0.0

        try:
            object_volume = sum([self.obj.dimensions.x, self.obj.dimensions.y, self.obj.dimensions.z]) / 3
            return object_volume / 5000 if object_volume < 5 else 0.0001
        except Exception:
            return 0.0

    def get_points_zoom_size(self):
        zoom_factor = bpy.context.space_data.zoom[:]

        pt_size_1 = 2.0 * zoom_factor[0]
        pt_size_2 = 1.0 * zoom_factor[0]

        if pt_size_1 < 1.0:
            pt_size_1 = 1.0
        if pt_size_2 < 1.0:
            pt_size_2 = 1.0

        return (pt_size_1, pt_size_2)


class BasicSetsCacher(BasicCacher):
    def build_group_cache(self, p_mgr_cls, p_obj, p_group, p_display_groups):
        super().build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)
        self.layer_name = p_group.layer_name if p_group else ''

    def draw(self, p_mgr_cls, p_obj, p_display_groups):
        p_group_pair = p_mgr_cls.get_current_group_pair(bpy.context)
        if p_group_pair and p_group_pair[1] in p_display_groups:
            try:
                addon_prefs = get_prefs()

                if addon_prefs.common.auto_update_draw_cache:
                    self.check_draw_cache_prepared(
                        p_mgr_cls, p_obj, p_group_pair[1], p_display_groups, modes={'MESH'})

                if self.batch is not None:
                    alpha = getattr(addon_prefs.display_3d, f'{p_mgr_cls.id_element}_active_alpha') / 100
                    alpha = alpha * ZenLocks.get_draw_overlay_locked()
                    self.mtx = (bpy.context.region_data.perspective_matrix @ p_obj.matrix_world).copy()
                    self._internal_draw(self.mtx, p_group_pair[1].group_color, alpha)
            finally:
                self.cleanup_mesh(p_obj)

    def draw_uv(self, p_mgr_cls, p_obj, p_display_groups):
        p_group_pair = p_mgr_cls.get_current_group_pair(bpy.context)
        if p_group_pair and p_group_pair[1] in p_display_groups:

            addon_prefs = get_prefs()

            if addon_prefs.common.auto_update_draw_cache:
                self.check_draw_cache_prepared(p_mgr_cls, p_obj, p_group_pair[1], p_display_groups, modes={'UV'})

            if self.uv_batch is not None:
                alpha = getattr(addon_prefs.display_3d, f'{p_mgr_cls.id_element}_active_alpha') / 100
                alpha = alpha * ZenLocks.get_draw_overlay_locked()
                p_color = p_group_pair[1].group_color[:] + (alpha,)

                points_size = (
                    addon_prefs.display_3d.vert_active_point_size,
                    addon_prefs.display_3d.vert_inactive_point_size
                )

                if p_mgr_cls.id_element == 'vert':
                    if addon_prefs.display_3d.vert_use_zoom_factor:
                        points_size = self.get_points_zoom_size()

                bgl.glEnable(bgl.GL_BLEND)
                bgl.glEnable(bgl.GL_LINE_SMOOTH)
                bgl.glBlendFunc(bgl.GL_SRC_ALPHA,
                                bgl.GL_ONE_MINUS_SRC_ALPHA)

                bgl.glLineWidth(addon_prefs.display_3d.edge_active_line_width)
                bgl.glPointSize(points_size[0])

                self.uv_shader.bind()
                self.uv_shader.uniform_float("color", p_color)
                self.uv_batch.draw(self.uv_shader)

                bgl.glLineWidth(1.0)
                bgl.glPointSize(1.0)
                bgl.glDisable(bgl.GL_LINE_SMOOTH)
                bgl.glDisable(bgl.GL_BLEND)
                bgl.glBlendFunc(bgl.GL_ONE, bgl.GL_ZERO)

    def _internal_draw(self, p_matrix, p_color, p_alpha):
        Log.error("ABSTRACT> '_internal_draw'")

    def build_uv_batch(self, p_mgr_cls, p_obj, p_active_group, p_display_groups, sel_uv_only):
        self.uv_cache_uuid = self.mesh_uuid
        self.uv_sync_mode = p_mgr_cls.is_uv_sync()
        self.uv_batch = None
        self.uv_layer_name = ''
        self.uv_sel_only = sel_uv_only
        if p_active_group:
            self.uv_layer_name = p_active_group.layer_name
            bm = self.get_bm(p_obj)
            layer = p_mgr_cls.get_mesh_layer(p_obj, bm, self.uv_layer_name)
            uv_layer = bm.loops.layers.uv.active
            if uv_layer and layer:
                interval = timer()
                is_uv_sync = p_mgr_cls.is_uv_sync()
                self.build_uv_layer(p_mgr_cls, bm, layer, uv_layer, sel_uv_only, is_uv_sync)
                Log.debug(p_obj.name, "- Build UV Batch:", p_active_group.name, timer() - interval)

    def build_uv_layer(self, p_mgr_cls, bm, mesh_layer, uv_layer, sel_uv_only, is_uv_sync):
        Log.error('ABSTRACT> build_uv_layer')


# MAPS
class BaseBatchCache:
    def __init__(self) -> None:
        self.batch_verts = []
        self.batch_indices = []
        self.batch = None


class BaseBatchCacheUV:
    def __init__(self) -> None:
        self.uv_batch = None
        self.uv_coords = []
        self.uv_indices = []


class BasicUniqueCacher(BasicCacher):
    def __init__(self) -> None:
        super().__init__()

        self.verts = []
        self.group_batches = {}
        self.group_batches_uv = {}

    def recalc_mesh(self, p_mgr_cls, p_obj, volume=None):
        interval = timer()
        super().recalc_mesh(p_mgr_cls, p_obj, volume=volume)
        self.fill_verts(p_obj)
        self.group_batches = {}
        self.group_batches_uv = {}
        Log.debug(p_obj.name, f'[{p_mgr_cls.list_item_prefix}] mesh recalculated:', self.mesh_statistics, 'secs:', timer() - interval)

    def fill_verts(self, p_obj):
        if p_obj.is_evaluated:
            self.ensure_mesh(p_obj)
            self.verts = np.empty((len(self.me.vertices), 3), 'f')
            self.me.vertices.foreach_get("co", np.reshape(self.verts, len(self.me.vertices) * 3))
        else:
            bm = bmesh.from_edit_mesh(p_obj.data)
            self.verts = [v.co.to_tuple() for v in bm.verts]

        if len(self.verts):
            format = gpu.types.GPUVertFormat()
            format.attr_add(id="pos", comp_type='F32', len=3, fetch_mode='FLOAT')
            self.vbo = gpu.types.GPUVertBuf(format, len=len(self.verts))
            self.vbo.attr_fill(0, data=self.verts)
        else:
            self.vbo = None

    def get_hash_list(self, p_mgr_cls, p_obj):
        Log.error('ABSTRACT> get_hash_list')
        return None

    def prepare_batch_data(self, p_group_batch, p_indices):
        Log.error('ABSTRACT> prepare_batch_data')

    def get_alphas(self, p_mgr_cls):
        addon_prefs = get_prefs()
        act_alpha = getattr(addon_prefs.display_3d, f'{p_mgr_cls.id_element}_active_alpha') / 100
        inact_alpha = getattr(addon_prefs.display_3d, f'{p_mgr_cls.id_element}_inactive_alpha') / 100
        return (act_alpha * ZenLocks.get_draw_overlay_locked(), inact_alpha * ZenLocks.get_draw_overlay_locked())

    def get_hash_scene_list(self, p_mgr_cls, p_obj, p_display_groups):
        return [p_mgr_cls.get_hash_from_str(p_obj, group.layer_name) for group in p_display_groups]

    def build_cache(self, p_mgr_cls, p_obj, p_active_group, p_display_groups):
        interval = timer()

        self.mesh_cache_uuid = self.mesh_uuid
        self.force_rebuild_eval_cache = True
        self.group_batches = {}

        hashes = self.get_hash_scene_list(p_mgr_cls, p_obj, p_display_groups)

        dic = self.get_hash_list(p_mgr_cls, p_obj)

        Log.debug(f'Obj:{p_obj.name} Hash dictionary:[{len(dic)}] created in:', timer() - interval)

        for i, k in enumerate(hashes):
            group_batch = BaseBatchCache()
            if k in dic:
                self.prepare_batch_data(group_batch, dic[k])
                self.build_batch(group_batch)
            self.group_batches[p_display_groups[i].layer_name] = group_batch

        s_eval = ''
        if p_obj.is_evaluated:
            s_eval = '[EVAL]'
        Log.debug(f'Completed cache build for obj:[{p_obj.name}]{s_eval} in:', timer() - interval)

    def draw_group(self, p_group, p_mgr_cls, p_obj, p_display_groups, p_alphas, is_active):
        p_group_batch = self.group_batches.get(p_group.layer_name, None)
        if p_group_batch is None:
            p_group_batch = self.build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)

        self.mtx = (bpy.context.region_data.perspective_matrix @ p_obj.matrix_world).copy()
        self.draw_batch(self.mtx, p_group_batch, p_alphas, is_active, p_group.group_color)

    def draw_batch(self, p_matrix, p_group_batch, p_alphas, is_active, p_color):
        if p_group_batch and p_group_batch.batch:
            self.shader.bind()
            f_alpha = p_alphas[0] if is_active else p_alphas[1]
            self.shader.uniform_float("ModelViewProjectionMatrix", p_matrix)
            self.shader.uniform_float("color", (p_color.r, p_color.g, p_color.b, f_alpha))
            p_group_batch.batch.draw(self.shader)

    def do_draw_groups(self, p_mgr_cls, p_obj, p_display_groups):
        try:
            addon_prefs = get_prefs()

            if addon_prefs.common.auto_update_draw_cache:
                self.check_draw_cache_prepared(p_mgr_cls, p_obj, None, p_display_groups, modes={'MESH'})

            ctx = bpy.context

            p_scene = ctx.scene
            alphas = self.get_alphas(p_mgr_cls)
            active_layer_name = p_mgr_cls.get_current_layer_name(ctx)
            if get_prefs().common.display_all_parts:
                for p_group in p_display_groups:
                    is_active = active_layer_name == p_group.layer_name
                    self.draw_group(p_group, p_mgr_cls, p_obj, p_display_groups, alphas, is_active)
            else:
                p_group = p_mgr_cls._get_scene_group(p_scene)
                if p_group and (p_group in p_display_groups):
                    is_active = active_layer_name == p_group.layer_name
                    self.draw_group(p_group, p_mgr_cls, p_obj, p_display_groups, alphas, is_active)
        finally:
            self.cleanup_mesh(p_obj)

    def draw_uv_group(self, p_group, p_mgr_cls, p_obj, p_display_groups, params, is_active):
        p_group_batch_uv = self.group_batches_uv.get(p_group.layer_name, None)
        if p_group_batch_uv:
            p_batch_uv = p_group_batch_uv.uv_batch
            if p_batch_uv is not None:
                p_color = p_group.group_color

                p_alpha = params[0][0] if is_active else params[0][1]
                pt_size = params[1][0] if is_active else params[1][1]
                line_size = params[2][0] if is_active else params[2][1]

                bgl.glLineWidth(line_size)
                bgl.glPointSize(pt_size)

                self.uv_shader.uniform_float(
                    "color", (p_color.r, p_color.g, p_color.b, p_alpha))
                p_batch_uv.draw(self.uv_shader)

    def get_uv_hash_list(self, p_mgr_cls, bm, sel_uv_only, is_sync, hash_layer, uv_layer):
        Log.error('ABSTRACT> get_uv_hash_list')
        return None

    def build_uv_group_batch(self, uvs, p_group_batch):
        Log.error('ABSTRACT> build_uv_group_batch')

    def build_uv_batch(self, p_mgr_cls, p_obj, p_active_group, p_display_groups, sel_uv_only):
        self.uv_cache_uuid = self.mesh_uuid
        self.uv_sync_mode = p_mgr_cls.is_uv_sync()
        self.uv_sel_only = sel_uv_only
        bm = self.get_bm(p_obj)
        hash_layer = p_mgr_cls.get_hash_layer(bm)
        uv_layer = bm.loops.layers.uv.active
        self.group_batches_uv = {}
        if hash_layer and uv_layer:
            is_uv_sync = p_mgr_cls.is_uv_sync()

            dic = self.get_uv_hash_list(p_mgr_cls, bm, sel_uv_only, is_uv_sync, hash_layer, uv_layer)

            for p_group in p_display_groups:
                p_group_batch_uv = self.group_batches_uv.get(p_group.layer_name, None)
                if p_group_batch_uv is None:
                    p_group_batch_uv = BaseBatchCacheUV()
                    self.group_batches_uv[p_group.layer_name] = p_group_batch_uv

                p_bytes = p_mgr_cls.get_hash_from_str(p_obj, p_group.layer_name)
                if p_bytes in dic:
                    self.build_uv_group_batch(dic[p_bytes], p_group_batch_uv)

    def draw_uv(self, p_mgr_cls, p_obj, p_display_groups):
        ctx = bpy.context
        p_scene = ctx.scene
        alphas = self.get_alphas(p_mgr_cls)
        active_layer_name = p_mgr_cls.get_current_layer_name(ctx)

        addon_prefs = get_prefs()

        if addon_prefs.common.auto_update_draw_cache:
            self.check_draw_cache_prepared(p_mgr_cls, p_obj, None, p_display_groups, modes={'UV'})

        if len(self.group_batches_uv):
            points_size = (
                addon_prefs.display_3d.vert_active_point_size,
                addon_prefs.display_3d.vert_inactive_point_size)
            lines_size = (
                addon_prefs.display_3d.edge_active_line_width,
                addon_prefs.display_3d.edge_inactive_line_width)

            if p_mgr_cls.id_element == 'vert':
                if addon_prefs.display_3d.vert_use_zoom_factor:
                    points_size = self.get_points_zoom_size()

            self.uv_shader.bind()
            bgl.glEnable(bgl.GL_BLEND)
            bgl.glEnable(bgl.GL_LINE_SMOOTH)
            bgl.glBlendFunc(bgl.GL_SRC_ALPHA, bgl.GL_ONE_MINUS_SRC_ALPHA)

            try:
                params = (
                    alphas,
                    points_size,
                    lines_size
                )

                if addon_prefs.common.display_all_parts:
                    for p_group in p_display_groups:
                        is_active = active_layer_name == p_group.layer_name
                        self.draw_uv_group(p_group, p_mgr_cls, p_obj, p_display_groups, params, is_active)
                else:
                    p_group = p_mgr_cls._get_scene_group(p_scene)
                    if p_group and (p_group in p_display_groups):
                        is_active = active_layer_name == p_group.layer_name
                        self.draw_uv_group(p_group, p_mgr_cls, p_obj, p_display_groups, params, is_active)
            finally:
                # Restore OpenGL settings
                bgl.glLineWidth(1.0)
                bgl.glPointSize(1.0)
                bgl.glDisable(bgl.GL_BLEND)
                bgl.glDisable(bgl.GL_LINE_SMOOTH)
                bgl.glBlendFunc(bgl.GL_ONE, bgl.GL_ZERO)


# VERTS
class VertsCacher(BasicSetsCacher):
    def __init__(self) -> None:
        BasicSetsCacher.__init__(self)

        self.verts = []
        self.shader = gpu.shader.from_builtin('3D_UNIFORM_COLOR')
        pass

    def is_bmesh_valid(self, p_obj):
        if super().is_bmesh_valid(p_obj):
            try:
                if not p_obj.is_evaluated:
                    bm = bmesh.from_edit_mesh(p_obj.data)
                    bm.verts.ensure_lookup_table()
                return True
            except ReferenceError as e:
                Log.error('BMESH is invalid:', e)
        return False

    def recalc_mesh(self, p_mgr_cls, p_obj, volume=None):
        interval = timer()
        super().recalc_mesh(p_mgr_cls, p_obj, volume=volume)
        Log.debug(p_obj.name, '[Vert Sets] mesh recalculated:', self.mesh_statistics, 'secs:', timer() - interval)

    def build_group_cache(self, p_mgr_cls, p_obj, p_group, p_display_groups):
        super().build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)

        if p_group in p_display_groups:
            if p_obj.is_evaluated:
                self.ensure_mesh(p_obj)
                self.layer = self.me.vertex_layers_int.get(self.layer_name)
                self.verts = [v.co.to_tuple() for v in self.me.vertices
                              if not v.hide and self.layer.data[v.index].value] if self.layer else []
            else:
                bm = bmesh.from_edit_mesh(p_obj.data)
                self.layer = p_mgr_cls.get_mesh_layer(p_obj, bm, self.layer_name)
                self.verts = [v.co.to_tuple() for v in bm.verts
                              if not v.hide and p_mgr_cls.is_bm_item_set(v, self.layer)] if self.layer else []
            self.build_batch()
        else:
            self.layer = None
            self.verts = []
            self.batch = None

    def build_uv_layer(self, p_mgr_cls, bm, mesh_layer, uv_layer, sel_uv_only, is_uv_sync):
        display_all = not sel_uv_only or is_uv_sync
        uvs = [loop[uv_layer].uv.to_tuple(5)
               for v in bm.verts for loop in v.link_loops
               if ((display_all and (not sel_uv_only or not loop.face.hide)) or (v.select and loop.face.select)) and
               v.hide is False and p_mgr_cls.is_bm_item_set(v, mesh_layer)]

        if len(uvs):
            uv_verts, _ = np.unique(uvs, return_inverse=True, axis=0)
            self.uv_coords = uv_verts.tolist()

            self.uv_batch = batch_for_shader(
                self.uv_shader, 'POINTS', {"pos": self.uv_coords})

    def build_batch(self):
        if len(self.verts):
            self.batch = batch_for_shader(
                self.shader, 'POINTS',
                {"pos": self.verts}
            )
        else:
            self.batch = None

    def _internal_draw(self, p_matrix, p_color, p_alpha):
        try:
            bgl.glPointSize(get_prefs().display_3d.vert_active_point_size)
            bgl.glEnable(bgl.GL_DEPTH_TEST)
            bgl.glEnable(bgl.GL_BLEND)

            self.shader.bind()
            self.shader.uniform_float("ModelViewProjectionMatrix", p_matrix)
            self.shader.uniform_float("color", (p_color.r, p_color.g, p_color.b, p_alpha))
            self.batch.draw(self.shader)
        finally:
            bgl.glDisable(bgl.GL_BLEND)
            bgl.glDisable(bgl.GL_DEPTH_TEST)
            bgl.glPointSize(1)


class VertsUniqueCacher(BasicUniqueCacher):
    def __init__(self) -> None:
        super().__init__()
        self.shader = gpu.shader.from_builtin('3D_UNIFORM_COLOR')

    def is_bmesh_valid(self, p_obj):
        if super().is_bmesh_valid(p_obj):
            try:
                if not p_obj.is_evaluated:
                    bm = bmesh.from_edit_mesh(p_obj.data)
                    bm.verts.ensure_lookup_table()
                return True
            except ReferenceError as e:
                Log.error('BMESH is invalid:', e)
        return False

    def get_hash_list(self, p_mgr_cls, p_obj):
        if self.obj.is_evaluated:
            self.ensure_mesh(self.obj)
            self.layer = self.me.vertex_layers_int.get(p_mgr_cls.layer_hash_name())

            if not self.layer:
                return []

            dic = defaultdict(list)
            for v in self.me.vertices:
                if not v.hide and self.layer.data[v.index].value != 0:
                    dic[self.layer.data[v.index].value].append(v.index)
            return dic
        else:
            bm = bmesh.from_edit_mesh(p_obj.data)
            self.layer = p_mgr_cls.get_hash_layer(bm)
            if not self.layer:
                return []

            dic = defaultdict(list)
            for v in bm.verts:
                if not v.hide and v[self.layer] != 0:
                    dic[v[self.layer]].append(v.index)
            return dic

    def prepare_batch_data(self, p_group_batch, p_indices):
        p_group_batch.batch_indices = gpu.types.GPUIndexBuf(type="POINTS", seq=p_indices) if len(p_indices) else None

    def build_group_cache(self, p_mgr_cls, p_obj, p_group, p_display_groups):
        super().build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)

        group_batch = BaseBatchCache()

        if p_obj.is_evaluated:
            self.ensure_mesh(self.obj)
            self.layer = self.me.vertex_layers_int.get(p_mgr_cls.layer_hash_name())
            if p_group in p_display_groups:
                p_bytes = hash32(p_group.layer_name)

                self.prepare_batch_data(
                    group_batch,
                    [v.index for v in self.me.vertices
                     if not v.hide and self.layer.data[v.index].value == p_bytes] if self.layer else []
                )
                self.build_batch(group_batch)
        else:
            bm = bmesh.from_edit_mesh(p_obj.data)
            self.layer = p_mgr_cls.get_hash_layer(bm)
            if p_group in p_display_groups:
                p_bytes = hash32(p_group.layer_name)

                self.prepare_batch_data(
                    group_batch,
                    [v.index for v in bm.verts
                     if not v.hide and v[self.layer] == p_bytes] if self.layer else []
                )
                self.build_batch(group_batch)

        self.group_batches[p_group.layer_name] = group_batch

        return group_batch

    def get_uv_hash_list(self, p_mgr_cls, bm, sel_uv_only, is_uv_sync, hash_layer, uv_layer):
        dic = defaultdict(list)
        b_display_all = not sel_uv_only or is_uv_sync
        for v in bm.verts:
            if (b_display_all or v.select) and v.hide is False and v[hash_layer] != 0:
                for loop in v.link_loops:
                    if ((b_display_all and (not sel_uv_only or not loop.face.hide)) or loop.face.select):
                        uv = loop[uv_layer].uv
                        dic[v[hash_layer]].append(uv.to_tuple(5))
        return dic

    def build_uv_group_batch(self, uvs, p_group_batch):
        if len(uvs):
            uv_verts, _ = np.unique(uvs, return_inverse=True, axis=0)
            p_group_batch.uv_coords = uv_verts.tolist()

            p_group_batch.uv_batch = batch_for_shader(
                self.uv_shader, 'POINTS', {"pos": p_group_batch.uv_coords})

    def build_batch(self, p_group_batch):
        if self.vbo and p_group_batch.batch_indices:
            p_group_batch.batch = gpu.types.GPUBatch(type="POINTS", buf=self.vbo, elem=p_group_batch.batch_indices)
        else:
            p_group_batch.batch = None

    def draw_batch(self, p_matrix, p_group_batch, p_alphas, is_active, p_color):
        if p_group_batch and p_group_batch.batch:
            addon_prefs = get_prefs()
            pt_size = addon_prefs.display_3d.vert_active_point_size if is_active \
                else addon_prefs.display_3d.vert_inactive_point_size
            bgl.glPointSize(pt_size)
            super().draw_batch(p_matrix, p_group_batch, p_alphas, is_active, p_color)
            bgl.glPointSize(1)

    def draw(self, p_mgr_cls, p_obj, p_display_groups):
        bgl.glEnable(bgl.GL_DEPTH_TEST)
        bgl.glEnable(bgl.GL_BLEND)

        self.do_draw_groups(p_mgr_cls, p_obj, p_display_groups)

        bgl.glDisable(bgl.GL_BLEND)
        bgl.glDisable(bgl.GL_DEPTH_TEST)


# EDGES
class EdgesCacher(BasicSetsCacher):
    def __init__(self) -> None:
        BasicSetsCacher.__init__(self)

        self.verts = []
        self.indices = []
        self.shader = gpu.types.GPUShader(LINE_VERTEX_SHADER, LINE_FRAGMENT_SHADER)
        pass

    def is_bmesh_valid(self, p_obj):
        if super().is_bmesh_valid(p_obj):
            try:
                if not p_obj.is_evaluated:
                    bm = bmesh.from_edit_mesh(p_obj.data)
                    bm.verts.ensure_lookup_table()
                    bm.edges.ensure_lookup_table()
                return True
            except ReferenceError as e:
                Log.error('BMESH is invalid:', e)
        return False

    def recalc_mesh(self, p_mgr_cls, p_obj, volume=None):
        interval = timer()
        super().recalc_mesh(p_mgr_cls, p_obj, volume=volume)
        self.verts = []
        Log.debug(p_obj.name, '[Edge Sets] mesh recalculated:', self.mesh_statistics, 'secs:', timer() - interval)

    def build_group_cache(self, p_mgr_cls, p_obj, p_group, p_display_groups):
        super().build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)

        if p_group in p_display_groups:
            bm = self.get_bm(p_obj)
            self.layer = p_mgr_cls.get_mesh_layer(p_obj, bm, self.layer_name)
            self.verts = [v.co.to_tuple() for e in bm.edges for v in e.verts
                          if e.hide is not True and e[self.layer]] if self.layer else []
            self.build_batch()
        else:
            self.layer = None
            self.verts = []
            self.batch = None

    def build_batch(self):
        if len(self.verts):
            self.batch = batch_for_shader(
                self.shader, 'LINES', {"pos": self.verts})
        else:
            self.batch = None

    def build_uv_layer(self, p_mgr_cls, bm, mesh_layer, uv_layer, sel_uv_only, is_uv_sync):
        display_all = not sel_uv_only or is_uv_sync
        uvs = []
        for e in bm.edges:
            if (display_all or e.select) and e.hide is False and e[mesh_layer] != 0:
                for loop in e.link_loops:
                    next_loop = loop.link_loop_next
                    if ((display_all and (not sel_uv_only or not loop.face.hide)) or (loop.face.select and next_loop.face.select)):
                        uv = loop[uv_layer].uv
                        next_uv = next_loop[uv_layer].uv
                        uvs.append(uv.to_tuple(5))
                        uvs.append(next_uv.to_tuple(5))

        if len(uvs):
            uv_verts, uv_indices = np.unique(uvs, return_inverse=True, axis=0)
            self.uv_coords = uv_verts.tolist()
            self.uv_indices = uv_indices.astype(np.int32)

            self.uv_batch = batch_for_shader(
                self.uv_shader, 'LINES', {"pos": self.uv_coords}, indices=self.uv_indices)

    def _internal_draw(self, p_matrix, p_color, p_alpha):
        addon_prefs = get_prefs()

        bgl.glEnable(bgl.GL_BLEND)
        bgl.glEnable(bgl.GL_LINE_SMOOTH)
        bgl.glEnable(bgl.GL_DEPTH_TEST)
        bgl.glLineWidth(addon_prefs.display_3d.edge_active_line_width)

        try:
            self.shader.bind()
            self.shader.uniform_float("viewProjectionMatrix", p_matrix)
            self.shader.uniform_float("color", (p_color.r, p_color.g, p_color.b, p_alpha))
            z_offset = self.get_edge_z_offset(addon_prefs.display_3d.edge_z_fight_compensation)
            self.shader.uniform_float('z_offset', z_offset)
            self.batch.draw(self.shader)

        finally:
            # restore opengl defaults
            bgl.glLineWidth(1)
            bgl.glDisable(bgl.GL_BLEND)
            bgl.glDisable(bgl.GL_LINE_SMOOTH)
            bgl.glDisable(bgl.GL_DEPTH_TEST)


class EdgesUniqueCacher(BasicUniqueCacher):
    def __init__(self) -> None:
        super().__init__()

        self.shader = gpu.types.GPUShader(LINE_VERTEX_SHADER, LINE_FRAGMENT_SHADER)

    def is_bmesh_valid(self, p_obj):
        if super().is_bmesh_valid(p_obj):
            try:
                if not p_obj.is_evaluated:
                    bm = bmesh.from_edit_mesh(p_obj.data)
                    bm.verts.ensure_lookup_table()
                    bm.edges.ensure_lookup_table()
                return True
            except ReferenceError as e:
                Log.error('BMESH is invalid:', e)
        return False

    def get_hash_list(self, p_mgr_cls, p_obj):
        bm = self.get_bm(p_obj)
        self.layer = p_mgr_cls.get_hash_layer(bm)
        if not self.layer:
            return []

        dic = defaultdict(list)
        for e in bm.edges:
            if not e.hide and e[self.layer] != 0:
                dic[e[self.layer]].append([v.index for v in e.verts])
        return dic

    def prepare_batch_data(self, p_group_batch, p_indices):
        p_group_batch.batch_indices = gpu.types.GPUIndexBuf(type="LINES", seq=p_indices) if len(p_indices) else None

    def build_group_cache(self, p_mgr_cls, p_obj, p_group, p_display_groups):
        super().build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)

        bm = self.get_bm(p_obj)
        self.layer = p_mgr_cls.get_hash_layer(bm)
        group_batch = BaseBatchCache()
        if p_group in p_display_groups:
            p_bytes = hash32(p_group.layer_name)
            self.prepare_batch_data(group_batch,
                                    [[v.index for v in e.verts]
                                     for e in bm.edges
                                     if e.hide is False and e[self.layer] == p_bytes] if self.layer else [])

            self.build_batch(group_batch)

        self.group_batches[p_group.layer_name] = group_batch

        return group_batch

    def build_batch(self, p_group_batch):
        if self.vbo and p_group_batch.batch_indices:
            p_group_batch.batch = gpu.types.GPUBatch(type="LINES", buf=self.vbo, elem=p_group_batch.batch_indices)
        else:
            p_group_batch.batch = None

    def draw_batch(self, p_matrix, p_group_batch, p_alphas, is_active, p_color):
        if p_group_batch and p_group_batch.batch:
            addon_prefs = get_prefs()
            line_size = addon_prefs.display_3d.edge_active_line_width if is_active \
                else addon_prefs.display_3d.edge_inactive_line_width
            try:
                bgl.glLineWidth(line_size)
                self.shader.bind()
                self.shader.uniform_float("viewProjectionMatrix", p_matrix)
                f_alpha = p_alphas[0] if is_active else p_alphas[1]
                self.shader.uniform_float("color", (p_color.r, p_color.g, p_color.b, f_alpha))
                z_offset = self.get_edge_z_offset(addon_prefs.display_3d.edge_z_fight_compensation)
                self.shader.uniform_float('z_offset', z_offset)
                p_group_batch.batch.draw(self.shader)
            finally:
                bgl.glLineWidth(1.0)

    def draw(self, p_mgr_cls, p_obj, p_display_groups):
        try:
            bgl.glEnable(bgl.GL_BLEND)
            bgl.glEnable(bgl.GL_LINE_SMOOTH)
            bgl.glEnable(bgl.GL_DEPTH_TEST)

            self.do_draw_groups(p_mgr_cls, p_obj, p_display_groups)
        finally:
            bgl.glLineWidth(1)
            bgl.glDisable(bgl.GL_BLEND)
            bgl.glDisable(bgl.GL_LINE_SMOOTH)
            bgl.glDisable(bgl.GL_DEPTH_TEST)

    def get_uv_hash_list(self, p_mgr_cls, bm, sel_uv_only, is_uv_sync, hash_layer, uv_layer):
        dic = defaultdict(list)
        b_display_all = not sel_uv_only or is_uv_sync
        for e in bm.edges:
            if (b_display_all or e.select) and not e.hide and e[hash_layer] != 0:
                for loop in e.link_loops:
                    next_loop = loop.link_loop_next
                    if ((b_display_all and (not sel_uv_only or not loop.face.hide)) or (loop.face.select and next_loop.face.select)):
                        uv = loop[uv_layer].uv
                        next_uv = next_loop[uv_layer].uv
                        dic[e[hash_layer]].append(uv.to_tuple(5))
                        dic[e[hash_layer]].append(next_uv.to_tuple(5))
        return dic

    def build_uv_group_batch(self, uvs, p_group_batch):
        if len(uvs):
            uv_verts, uv_indices = np.unique(uvs, return_inverse=True, axis=0)
            p_group_batch.uv_coords = uv_verts.tolist()
            p_group_batch.uv_indices = uv_indices.astype(np.int32)

            p_group_batch.uv_batch = batch_for_shader(
                self.uv_shader, 'LINES', {"pos": p_group_batch.uv_coords}, indices=p_group_batch.uv_indices)


# FACES
class FacesCacher(BasicSetsCacher):
    def __init__(self) -> None:
        BasicSetsCacher.__init__(self)

        self.verts = []
        self.indices = []
        self.loops = None
        self.shader = gpu.shader.from_builtin('3D_UNIFORM_COLOR')
        self.uv_shader = gpu.shader.from_builtin('2D_UNIFORM_COLOR')
        pass

    def is_bmesh_valid(self, p_obj):
        if super().is_bmesh_valid(p_obj):
            try:
                if not p_obj.is_evaluated:
                    bm = bmesh.from_edit_mesh(p_obj.data)
                    bm.verts.ensure_lookup_table()
                    bm.faces.ensure_lookup_table()
                return True
            except ReferenceError as e:
                Log.error('BMESH is invalid:', e)
        return False

    def recalc_mesh(self, p_mgr_cls, p_obj, volume=None):
        interval = timer()
        super().recalc_mesh(p_mgr_cls, p_obj, volume=volume)
        self.loops = None
        self.verts = []
        Log.debug(p_obj.name, '[Face Sets] mesh recalculated:', self.mesh_statistics, 'secs:', timer() - interval)

    def build_group_cache(self, p_mgr_cls, p_obj, p_group, p_display_groups):
        super().build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)

        if p_group in p_display_groups:
            if p_obj.is_evaluated:
                self.ensure_mesh(self.obj)
                self.me.calc_loop_triangles()
                self.loops = self.me.loop_triangles
                self.layer = self.me.polygon_layers_int.get(self.layer_name)
                self.verts = [self.me.vertices[vi].co.to_tuple()
                              for looptris in self.me.loop_triangles for vi in looptris.vertices
                              if not self.me.polygons[looptris.polygon_index].hide and
                              self.layer.data[looptris.polygon_index].value] if self.layer else []
            else:
                bm = bmesh.from_edit_mesh(p_obj.data)
                self.layer = p_mgr_cls.get_mesh_layer(p_obj, bm, self.layer_name)
                self.verts = [loop.vert.co.to_tuple()
                              for looptris in self.ensure_loops(bm) for loop in looptris
                              if looptris[0].face.hide is False and looptris[0].face[self.layer]] if self.layer else []
            self.build_batch()
        else:
            self.layer = None
            self.verts = []
            self.batch = None

    def build_uv_layer(self, p_mgr_cls, bm, mesh_layer, uv_layer, sel_uv_only, is_uv_sync):
        display_all = not sel_uv_only or is_uv_sync
        uvs = [loop[uv_layer].uv.to_tuple(5)
               for looptris in self.ensure_loops(bm) for loop in looptris
               if (display_all or looptris[0].face.select) and
               looptris[0].face.hide is False and looptris[0].face[mesh_layer]]

        if len(uvs):
            uv_verts, uv_indices = np.unique(uvs, return_inverse=True, axis=0)
            self.uv_coords = uv_verts.tolist()
            self.uv_indices = uv_indices.astype(np.int32)

            self.uv_batch = batch_for_shader(
                self.uv_shader, 'TRIS', {"pos": self.uv_coords}, indices=self.uv_indices)

    def build_batch(self):
        if len(self.verts):
            self.batch = batch_for_shader(
                self.shader, 'TRIS',
                {"pos": self.verts})
        else:
            self.batch = None

    def _internal_draw(self, p_matrix, p_color, p_alpha):
        try:
            bgl.glEnable(bgl.GL_BLEND)
            bgl.glEnable(bgl.GL_LINE_SMOOTH)
            bgl.glEnable(bgl.GL_DEPTH_TEST)
            bgl.glEnable(bgl.GL_POLYGON_OFFSET_FILL)
            bgl.glPolygonOffset(-1, -1 * self.z_order)

            self.shader.bind()
            self.shader.uniform_float("ModelViewProjectionMatrix", p_matrix)
            self.shader.uniform_float("color", (p_color.r, p_color.g, p_color.b, p_alpha))
            self.batch.draw(self.shader)
        finally:
            # restore opengl defaults
            bgl.glLineWidth(1)
            bgl.glDisable(bgl.GL_BLEND)
            bgl.glDisable(bgl.GL_LINE_SMOOTH)
            bgl.glDisable(bgl.GL_POLYGON_OFFSET_FILL)
            bgl.glPolygonOffset(0, 0)


class FaceUniqueCacher(BasicUniqueCacher):
    def __init__(self) -> None:
        super().__init__()

        self.loops = None
        self.shader = gpu.shader.from_builtin('3D_UNIFORM_COLOR')

    def is_bmesh_valid(self, p_obj):
        if super().is_bmesh_valid(p_obj):
            try:
                if not p_obj.is_evaluated:
                    bm = bmesh.from_edit_mesh(p_obj.data)
                    bm.verts.ensure_lookup_table()
                    bm.faces.ensure_lookup_table()
                return True
            except ReferenceError as e:
                Log.error('BMESH is invalid:', e)
        return False

    def set_eval_mesh_layer(self, p_mgr_cls, p_obj):
        self.layer = self.me.polygon_layers_int.get(p_mgr_cls.layer_hash_name())

    def get_hash_list(self, p_mgr_cls, p_obj):
        p_zero_val = p_mgr_cls.get_bm_item_zero_val()
        if self.obj.is_evaluated:
            self.ensure_mesh(self.obj)
            self.me.calc_loop_triangles()
            self.loops = self.me.loop_triangles
            self.set_eval_mesh_layer(p_mgr_cls, p_obj)

            if not self.layer:
                return []

            dic = defaultdict(list)
            for looptris in self.me.loop_triangles:
                if not self.me.polygons[looptris.polygon_index].hide \
                   and self.layer.data[looptris.polygon_index].value != p_zero_val:
                    dic[self.layer.data[looptris.polygon_index].value].append(looptris.vertices)
            return dic
        else:
            bm = self.get_bm(p_obj)
            self.layer = p_mgr_cls.get_hash_layer(bm)
            if not self.layer:
                return []

            dic = defaultdict(list)
            for looptris in self.ensure_loops(bm):
                if looptris[0].face.hide is False and looptris[0].face[self.layer] != p_zero_val:
                    dic[looptris[0].face[self.layer]].append([loop.vert.index for loop in looptris])
            return dic

    def recalc_mesh(self, p_mgr_cls, p_obj, volume=None):
        super().recalc_mesh(p_mgr_cls, p_obj, volume=volume)
        self.loops = None

    def prepare_batch_data(self, p_group_batch, p_indices):
        p_group_batch.batch_indices = gpu.types.GPUIndexBuf(type="TRIS", seq=p_indices) if len(p_indices) else None

    def build_group_cache(self, p_mgr_cls, p_obj, p_group, p_display_groups):
        super().build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)

        group_batch = BaseBatchCache()
        if p_obj.is_evaluated:
            self.ensure_mesh(self.obj)
            self.me.calc_loop_triangles()
            self.loops = self.me.loop_triangles
            self.set_eval_mesh_layer(p_mgr_cls, p_obj)

            if self.layer:
                if p_group in p_display_groups:
                    p_bytes = p_mgr_cls.get_hash_from_str(p_obj, p_group.layer_name)
                    self.prepare_batch_data(group_batch,
                                            [looptris.vertices
                                             for looptris in self.me.loop_triangles
                                             if not self.me.polygons[looptris.polygon_index].hide and
                                             self.layer.data[looptris.polygon_index].value == p_bytes])
                    self.build_batch(group_batch)
        else:
            bm = self.get_bm(p_obj)
            self.layer = p_mgr_cls.get_hash_layer(bm)

            if self.layer:
                if p_group in p_display_groups:
                    p_bytes = p_mgr_cls.get_hash_from_str(p_obj, p_group.layer_name)
                    self.prepare_batch_data(group_batch,
                                            [[loop.vert.index for loop in looptris]
                                             for looptris in self.ensure_loops(bm)
                                             if looptris[0].face.hide is False and looptris[0].face[self.layer] == p_bytes])
                    self.build_batch(group_batch)

        self.group_batches[p_group.layer_name] = group_batch
        return group_batch

    def build_batch(self, p_group_batch):
        if self.vbo and p_group_batch.batch_indices:
            p_group_batch.batch = gpu.types.GPUBatch(type="TRIS", buf=self.vbo, elem=p_group_batch.batch_indices)
        else:
            p_group_batch.batch = None

    def get_uv_hash_list(self, p_mgr_cls, bm, sel_uv_only, is_uv_sync, hash_layer, uv_layer):
        dic = defaultdict(list)
        b_display_all = not sel_uv_only or is_uv_sync
        p_zero_val = p_mgr_cls.get_bm_item_zero_val()
        for looptris in self.ensure_loops(bm):
            if (b_display_all or looptris[0].face.select) and \
                    looptris[0].face.hide is False and \
                    looptris[0].face[hash_layer] != p_zero_val:
                for loop in looptris:
                    dic[looptris[0].face[hash_layer]].append(loop[uv_layer].uv.to_tuple(5))
        return dic

    def build_uv_group_batch(self, uvs, p_group_batch):
        if len(uvs):
            uv_verts, uv_indices = np.unique(uvs, return_inverse=True, axis=0)
            p_group_batch.uv_coords = uv_verts.tolist()
            p_group_batch.uv_indices = uv_indices.astype(np.int32)

            p_group_batch.uv_batch = batch_for_shader(
                self.uv_shader, 'TRIS', {"pos": p_group_batch.uv_coords}, indices=p_group_batch.uv_indices)

    def draw(self, p_mgr_cls, p_obj, p_display_groups):
        interval = timer()

        try:
            bgl.glEnable(bgl.GL_BLEND)
            bgl.glEnable(bgl.GL_LINE_SMOOTH)
            bgl.glEnable(bgl.GL_DEPTH_TEST)
            bgl.glEnable(bgl.GL_POLYGON_OFFSET_FILL)
            bgl.glPolygonOffset(-1, -1 * self.z_order)

            self.do_draw_groups(p_mgr_cls, p_obj, p_display_groups)
        finally:
            bgl.glLineWidth(1)
            bgl.glDisable(bgl.GL_BLEND)
            bgl.glDisable(bgl.GL_LINE_SMOOTH)
            bgl.glDisable(bgl.GL_DEPTH_TEST)
            bgl.glDisable(bgl.GL_POLYGON_OFFSET_FILL)
            bgl.glPolygonOffset(0, 0)

            elapsed = timer() - interval
            if elapsed > 0.1:
                Log.warn(f'Draw obj:[{p_obj.name}] mode:[Face Parts] low performance:', elapsed)


class FaceMapsUniqueCacher(FaceUniqueCacher):
    def set_eval_mesh_layer(self, p_mgr_cls, p_obj):
        self.layer = self.me.face_maps.active


_CUBE_INDICES = (
    (0, 1), (1, 2), (2, 3), (3, 0),  # Front
    (4, 5), (5, 6), (6, 7), (7, 4),  # Back
    (0, 4), (1, 5), (2, 6), (3, 7)
)


class BasicObjectsCacher:
    def __init__(self) -> None:
        self.volume = None
        self.batch = None
        self.shader = gpu.shader.from_builtin('3D_UNIFORM_COLOR')
        self.mesh_data = (0, 0, 0)

    def is_same(self, p_obj, p_volume):
        if self.volume != p_volume:
            return False

        mesh = p_obj.data
        if mesh and isinstance(mesh, bpy.types.Mesh):
            if self.mesh_data[0] != len(mesh.vertices):
                return False
            if self.mesh_data[1] != len(mesh.edges):
                return False
            if self.mesh_data[2] != len(mesh.polygons):
                return False

        return True

    def build_batch(self, p_obj, p_volume):
        self.volume = p_volume

        mesh = p_obj.data

        clear_to_mesh = False
        if mesh:
            if not isinstance(mesh, bpy.types.Mesh):
                try:
                    mesh = p_obj.to_mesh()
                    clear_to_mesh = True
                except Exception:
                    mesh = None
        if mesh:
            self.mesh_data = (len(mesh.vertices), len(mesh.edges), len(mesh.polygons))

            mesh.calc_loop_triangles()

            vert_count = len(mesh.vertices)
            tris_count = len(mesh.loop_triangles)

            coords = np.empty((vert_count, 3), 'f')
            indices = np.empty((tris_count, 3), 'i')

            mesh.vertices.foreach_get(
                "co", np.reshape(coords, vert_count * 3))
            mesh.loop_triangles.foreach_get(
                "vertices", np.reshape(indices, tris_count * 3))

            self.batch = batch_for_shader(self.shader, 'TRIS', {"pos": coords}, indices=indices)
        else:
            e_len = 1.0 / 5.0
            coords = ((-e_len, -e_len, -e_len), (-e_len, -e_len, e_len),
                      (-e_len, e_len, e_len), (-e_len, e_len, -e_len),
                      (e_len, -e_len, -e_len), (e_len, -e_len, e_len),
                      (e_len, e_len, e_len), (e_len, e_len, -e_len))

            self.batch = batch_for_shader(self.shader, 'LINES', {"pos": coords}, indices=_CUBE_INDICES)

        if clear_to_mesh:
            p_obj.to_mesh_clear()

    @classmethod
    def start_bgl_draw(cls):
        bgl.glEnable(bgl.GL_BLEND)
        bgl.glEnable(bgl.GL_LINE_SMOOTH)
        bgl.glEnable(bgl.GL_DEPTH_TEST)
        bgl.glEnable(bgl.GL_POLYGON_OFFSET_FILL)
        bgl.glPolygonOffset(-1, -1)

    @classmethod
    def finish_bgl_draw(cls):
        bgl.glLineWidth(1)
        bgl.glDisable(bgl.GL_BLEND)
        bgl.glDisable(bgl.GL_LINE_SMOOTH)
        bgl.glDisable(bgl.GL_DEPTH_TEST)
        bgl.glDisable(bgl.GL_POLYGON_OFFSET_FILL)
        bgl.glPolygonOffset(0, 0)

    def draw(self, p_mgr_cls, p_obj, p_color, is_active, p_alpha=None):
        interval = timer()

        p_volume = [Vector(v) for v in p_obj.bound_box]
        if not self.is_same(p_obj, p_volume) or self.batch is None:
            self.build_batch(p_obj, p_volume)

        self.shader.bind()
        matrix = bpy.context.region_data.perspective_matrix
        self.shader.uniform_float("ModelViewProjectionMatrix", matrix @ p_obj.matrix_world)
        if p_alpha is None:
            addon_prefs = get_prefs()
            p_alpha = (
                addon_prefs.display_3d.object_active_alpha
                if is_active else addon_prefs.display_3d.object_inactive_alpha) * 0.01
        self.shader.uniform_float("color", (p_color.r, p_color.g, p_color.b, p_alpha))

        self.batch.draw(self.shader)

        elapsed = timer() - interval
        if elapsed > 0.1:
            Log.warn(f'Draw obj:[{p_obj.name}] mode:[Object] low performance:', elapsed)


def get_object_bounds(p_obj, mtx):
    mesh = p_obj.data
    coords = []
    if mesh and tuple(p_obj.dimensions) != (0, 0, 0):
        coords = [mtx @ Vector(corner) for corner in p_obj.bound_box]
    else:
        e_len = 1.0 / 5.0
        coords = ((-e_len, -e_len, -e_len), (-e_len, -e_len, e_len),
                  (-e_len, e_len, e_len), (-e_len, e_len, -e_len),
                  (e_len, -e_len, -e_len), (e_len, -e_len, e_len),
                  (e_len, e_len, e_len), (e_len, e_len, -e_len))
        coords = [mtx @ Vector(corner) for corner in coords]
    return coords


def get_object_volume(p_obj, mtx):
    coords = get_object_bounds(p_obj, mtx)
    x, y, z = zip(*coords)
    return min(x), min(y), min(z), max(x), max(y), max(z)


def get_position_bounds_2d(context: bpy.types.Context, p_bounds):
    font_position = None
    if p_bounds:
        rgn2d = context.region
        rgn_width = rgn2d.width
        rgn_height = rgn2d.height
        point_2d = (rgn_width / 4, -rgn_height / 2)
        min_dist = None

        for v_pt in p_bounds:
            vec_pos = bpy_extras.view3d_utils.location_3d_to_region_2d(
                context.region, context.space_data.region_3d, v_pt)
            if vec_pos:
                margin = 50
                clip_bounds = (margin, margin, rgn_width - margin, rgn_height - margin)
                if (vec_pos[0] > clip_bounds[0] and vec_pos[0] < clip_bounds[2] and
                        vec_pos[1] > clip_bounds[1] and vec_pos[1] < clip_bounds[3]):
                    dist_to_point = math.sqrt(sum((px - qx) ** 2.0 for px, qx in zip(vec_pos, point_2d)))
                    if min_dist is None or dist_to_point < min_dist:
                        min_dist = dist_to_point
                        font_position = vec_pos

        # --- Do we need to display in the Center ? ---
        # if font_position is None:
        #     font_position = (rgn_width / 2, rgn_height / 2)

    return font_position


class BasicCollectionCacher:
    def __init__(self) -> None:
        self.obj = None
        self.volume = None
        self.batch = None
        self.shader = gpu.shader.from_builtin('3D_UNIFORM_COLOR')
        self.mesh_data = (0, 0, 0)
        self.bounds = None

        self.back_batch = None
        self.back_pos = None
        self.back_text = None
        self.back_font_size = None
        self.back_font_dpi = None
        self.back_shader = gpu.shader.from_builtin('2D_UNIFORM_COLOR')

    def get_recursive_volume(self, p_depsgraph, p_objects, p_mtx, p_check_volume):
        for p_obj in p_objects:
            mtx = p_obj.matrix_world
            if p_mtx is not None:
                mtx = p_mtx @ mtx
            if p_obj.instance_collection:
                self.get_recursive_volume(p_depsgraph, p_obj.instance_collection.all_objects, mtx, p_check_volume)
            else:
                p_check_volume.append(
                    get_object_volume(p_obj.evaluated_get(p_depsgraph), mtx))

    def is_same(self, p_objects, p_depsgraph):

        check_volume = []
        self.get_recursive_volume(p_depsgraph, p_objects, None, check_volume)

        if self.volume is None or self.volume != check_volume:
            return False, check_volume

        return True, check_volume

    def build_batch(self, p_volume):
        self.volume = p_volume

        if len(self.volume):
            min_axis_x, min_axis_y, min_axis_z, max_axis_x, max_axis_y, max_axis_z = zip(*self.volume)

            min_x = min(min_axis_x)
            max_x = max(max_axis_x)

            min_y = min(min_axis_y)
            max_y = max(max_axis_y)

            min_z = min(min_axis_z)
            max_z = max(max_axis_z)

            self.bounds = ((min_x, min_y, min_z), (min_x, min_y, max_z),
                           (min_x, max_y, max_z), (min_x, max_y, min_z),
                           (max_x, min_y, min_z), (max_x, min_y, max_z),
                           (max_x, max_y, max_z), (max_x, max_y, min_z))

            self.batch = batch_for_shader(self.shader, 'LINES', {"pos": self.bounds}, indices=_CUBE_INDICES)
        else:
            self.batch = None
            self.bounds = None

    def get_position_2d(self, context):
        return get_position_bounds_2d(context, self.bounds)

    def draw_caption_background(self, ui_scale, font_size: int, s_text, font_pos):
        if s_text == '':
            return

        dpi = round(ui_scale * 72)

        if (self.back_batch is None or
           self.back_text != s_text or
           self.back_pos != font_pos or
           self.back_font_size != font_size or
           self.back_font_dpi != dpi):
            # --- BUILD BATCH PROCESS ---
            self.back_pos = font_pos
            self.back_text = s_text
            self.back_font_size = font_size
            self.back_font_dpi = dpi

            blf.size(0, font_size, dpi)
            text_width, text_height = blf.dimensions(0, s_text)

            i_width = text_width + 4
            i_height = text_height + 2

            i_x = font_pos[0] - 2
            i_y = font_pos[1] - 2

            indices = ((0, 1, 2), (0, 2, 3))

            # bottom left, top left, top right, bottom right
            vertices = (
                (i_x, i_y),
                (i_x, i_y + i_height),
                (i_x + i_width, i_y + i_height),
                (i_x + i_width, i_y))

            self.back_batch = batch_for_shader(self.back_shader, 'TRIS', {"pos": vertices}, indices=indices)

        self.back_shader.bind()
        self.back_shader.uniform_float("color", (0.0, 0.0, 0.0, 0.5))

        self.back_batch.draw(self.back_shader)

    def draw(self, p_mgr_cls, p_objects, p_group, p_depsgraph):
        interval = timer()

        try:
            bgl.glEnable(bgl.GL_BLEND)
            bgl.glEnable(bgl.GL_LINE_SMOOTH)
            bgl.glDisable(bgl.GL_DEPTH_TEST)

            res, _volume = self.is_same(p_objects, p_depsgraph)
            if not res:
                self.build_batch(_volume)

            if self.batch is not None:
                addon_prefs = get_prefs()

                bgl.glLineWidth(addon_prefs.display_3d.object_collection_line_width)

                self.shader.bind()
                matrix = bpy.context.region_data.perspective_matrix
                self.shader.uniform_float("ModelViewProjectionMatrix", matrix)
                p_alpha = addon_prefs.display_3d.object_active_alpha * 0.01
                self.shader.uniform_float("color", (p_group.group_color.r, p_group.group_color.g, p_group.group_color.b, p_alpha))

                self.batch.draw(self.shader)

        finally:
            bgl.glLineWidth(1)
            bgl.glDisable(bgl.GL_BLEND)
            bgl.glDisable(bgl.GL_LINE_SMOOTH)
            bgl.glDisable(bgl.GL_DEPTH_TEST)
            bgl.glDisable(bgl.GL_POLYGON_OFFSET_FILL)
            bgl.glPolygonOffset(0, 0)

            elapsed = timer() - interval
            if elapsed > 0.1:
                Log.warn(f'Draw obj:[{p_group.name}] mode:[Collection] low performance:', elapsed)
