# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

from collections import defaultdict
import bpy
import bgl
import blf
import gpu
import bpy_extras

from timeit import default_timer as timer

from mathutils import Color, Matrix

from .draw_cache import BasicCollectionCacher, BasicObjectsCacher, get_position_bounds_2d, get_object_bounds

from ..vlog import Log
from ..preferences import get_prefs
from ..blender_zen_utils import ZenLocks, ZenPolls, ZenStates, ZsDrawConstans, update_areas_in_all_screens

_cache = {}
_collection_cache = {}
_draw_handlers = {}


def is_object_cage_visible(p_obj):
    return all(not it.show_on_cage for it in p_obj.modifiers
               if it.is_active and it.show_viewport and it.show_in_editmode)


def is_draw_active():
    return len(_draw_handlers) > 0


def is_draw_handler_enabled(p_mgr_cls):
    if p_mgr_cls is None:
        return None

    return p_mgr_cls.id_group in _draw_handlers


def remove_draw_handler(p_mgr_cls, context):
    s_idname = p_mgr_cls.id_group
    if s_idname in _draw_handlers:
        _do_remove_draw_handler(s_idname, context)


def _do_remove_draw_handler(s_idname, context):

    for i, h in enumerate(_draw_handlers[s_idname]):
        if i < 2:
            bpy.types.SpaceView3D.draw_handler_remove(h, 'WINDOW')
        else:
            bpy.types.SpaceImageEditor.draw_handler_remove(h, 'WINDOW')

    del _draw_handlers[s_idname]
    Log.debug('Clear handler:', s_idname)

    update_areas_in_all_screens()


def _do_add_draw_handler(p_mgr_cls, context):
    s_idname = p_mgr_cls.id_group

    args = (context, )
    _draw_handlers[s_idname] = (
        bpy.types.SpaceView3D.draw_handler_add(draw_zensets_callback_3d, args, 'WINDOW', 'POST_VIEW'),
        bpy.types.SpaceView3D.draw_handler_add(draw_zensets_callback_2d, args, 'WINDOW', 'POST_PIXEL'),

        bpy.types.SpaceImageEditor.draw_handler_add(draw_zensets_uveditor_3d, args, 'WINDOW', 'POST_VIEW'),
        bpy.types.SpaceImageEditor.draw_handler_add(draw_zensets_callback_2d, args, 'WINDOW', 'POST_PIXEL')
    )
    Log.debug('Add draw_handler:', s_idname)

    # optional: switch others
    for key in list(_draw_handlers):
        if not key == s_idname:
            _do_remove_draw_handler(key, context)


def remove_all_handlers3d():
    for handles in _draw_handlers.values():
        for i, h in enumerate(handles):
            if i < 2:
                bpy.types.SpaceView3D.draw_handler_remove(h, 'WINDOW')
            else:
                bpy.types.SpaceImageEditor.draw_handler_remove(h, 'WINDOW')
    _draw_handlers.clear()


def reset_all_draw_cache():
    global _cache
    _cache.clear()
    global _collection_cache
    _collection_cache.clear()


def get_mgr_cache(p_mgr_cls):
    global _cache
    if p_mgr_cls.id_group not in _cache:
        _cache[p_mgr_cls.id_group] = {}

    return _cache[p_mgr_cls.id_group]


def get_obj_cache(p_mgr_cls, p_obj):
    global _cache
    p_mgr_cache = get_mgr_cache(p_mgr_cls)
    if p_obj not in p_mgr_cache:
        p_mgr_cache[p_obj] = p_mgr_cls.get_cacher()

    return p_mgr_cache[p_obj]


def reset_obj_cache(p_mgr_cls, p_obj):
    global _cache
    p_mgr_cache = get_mgr_cache(p_mgr_cls)
    p_mgr_cache[p_obj] = p_mgr_cls.get_cacher()


def check_update_cache(p_mgr_cls, p_obj):
    p_display_groups = p_mgr_cls.get_obj_highlighted_groups(p_obj)
    if len(p_display_groups):
        p_group_pair = p_mgr_cls.get_current_group_pair(bpy.context)
        p_group = p_group_pair[1] if p_group_pair else None
        p_obj_cache = get_obj_cache(p_mgr_cls, p_obj)
        p_obj_cache.check_draw_cache_prepared(
            p_mgr_cls, p_obj, p_group, p_display_groups, modes={'MESH', 'UV'}, b_calc_vol=False)


def check_update_cache_on_change(p_mgr_cls, p_obj, p_obj_eval):
    p_display_groups = p_mgr_cls.get_obj_highlighted_groups(p_obj)
    if len(p_display_groups):
        p_group_pair = p_mgr_cls.get_current_group_pair(bpy.context)
        p_group = p_group_pair[1] if p_group_pair else None
        p_obj_cache = get_obj_cache(p_mgr_cls, p_obj)
        p_obj_cache.check_draw_cache_prepared(
            p_mgr_cls, p_obj, p_group, p_display_groups, modes={'MESH', 'UV'}, b_calc_vol=True)

        if p_obj_eval != p_obj:
            addon_prefs = get_prefs()
            if not is_object_cage_visible(p_obj) or ('MOD' in addon_prefs.common.display_mesh_mode):
                if p_obj_eval and p_obj_eval.data:
                    p_obj_cache_eval = get_obj_cache(p_mgr_cls, p_obj_eval)
                    if p_obj_cache.force_rebuild_eval_cache:
                        p_obj_cache_eval.mesh_cache_uuid = None
                        p_obj_cache.force_rebuild_eval_cache = False
                    p_obj_cache_eval.check_draw_cache_prepared(
                        p_mgr_cls, p_obj_eval, p_group, p_display_groups, modes={'MESH'}, b_calc_vol=False)
                    p_obj_cache_eval.cleanup_mesh(p_obj_eval)


def update_cache_group(p_mgr_cls, p_obj, p_group):
    p_obj_cache = get_obj_cache(p_mgr_cls, p_obj)

    try:
        p_display_groups = p_mgr_cls.get_obj_highlighted_groups(p_obj)
        p_obj_cache.check_draw_cache_prepared(p_mgr_cls, p_obj, p_group, p_display_groups, modes={'MESH', 'UV'})
        p_obj_cache.build_group_cache(p_mgr_cls, p_obj, p_group, p_display_groups)
    except ReferenceError as e:
        Log.error('update_cache_group', e)


def mark_groups_modified(p_mgr_cls, p_obj, modes={'MESH', 'UV'}):
    p_obj_cache = get_obj_cache(p_mgr_cls, p_obj)
    if 'MESH' in modes:
        p_obj_cache.mesh_cache_uuid = None
    if 'UV' in modes:
        p_obj_cache.uv_cache_uuid = None


def draw_zensets_uveditor_3d(context):
    if ZenLocks.get_draw_overlay_locked() == 0.0:
        return

    addon_prefs = get_prefs()
    if not addon_prefs.uv_options.display_uv:
        return

    from ..sets.factories import get_sets_mgr
    p_mgr_cls = get_sets_mgr(context.scene)
    if p_mgr_cls is None:
        return

    if (context.mode == 'EDIT_MESH' and
            context.area.type == 'IMAGE_EDITOR' and
            context.space_data and
            getattr(context.space_data, 'show_uvedit', False)):
        viewport_info = bgl.Buffer(bgl.GL_INT, 4)
        bgl.glGetIntegerv(bgl.GL_VIEWPORT, viewport_info)

        width = viewport_info[2]
        height = viewport_info[3]

        region = context.region

        uv_to_view = region.view2d.view_to_region

        origin_x, origin_y = uv_to_view(0, 0, clip=False)
        top_x, top_y = uv_to_view(1.0, 1.0, clip=False)
        axis_x = top_x - origin_x
        axis_y = top_y - origin_y

        matrix = Matrix((
            [axis_x / width * 2, 0, 0, 2.0 * -
                ((width - origin_x - 0.5 * width)) / width],
            [0, axis_y / height * 2, 0, 2.0 * -
                ((height - origin_y - 0.5 * height)) / height],
            [0, 0, 1.0, 0],
            [0, 0, 0, 1.0]))

        identiy = Matrix.Identity(4)

        with gpu.matrix.push_pop():
            gpu.matrix.load_matrix(matrix)

            with gpu.matrix.push_pop_projection():
                gpu.matrix.load_projection_matrix(identiy)

                try:
                    for p_obj in context.objects_in_mode:
                        p_display_groups = p_mgr_cls.get_obj_highlighted_groups(p_obj)
                        if len(p_display_groups):
                            p_obj_cache = get_obj_cache(p_mgr_cls, p_obj)
                            p_obj_cache.draw_uv(p_mgr_cls, p_obj, p_display_groups)
                except Exception as e:
                    Log.error(e)

            bgl.glViewport(*tuple(viewport_info))


def draw_zensets_callback_2d(context: bpy.types.Context) -> None:
    from ..sets.factories import get_sets_mgr
    p_mgr_cls = get_sets_mgr(context.scene)
    if p_mgr_cls is None:
        return

    addon_prefs = get_prefs()
    b_is_uv_area = p_mgr_cls.is_uv_area()
    b_is_edit_mesh = context.mode == 'EDIT_MESH'
    b_is_uv_enabled = b_is_edit_mesh and p_mgr_cls.is_uv_display_active()
    if b_is_uv_area:
        if not b_is_uv_enabled:
            return

    p_group = p_mgr_cls._get_scene_group(context.scene)
    s_display_group_name = 'None'
    s_active_group_name = ''  # original name
    b_group_active = False
    if p_group:
        s_display_group_name = p_group.name
        s_active_group_name = s_display_group_name
        b_group_active = True

    text_parts = ()
    is_object_mode = context.mode == 'OBJECT'
    if is_object_mode:
        if not context.space_data or context.space_data.type != 'VIEW_3D':
            return

    b_is_processing = ZenLocks.get_draw_overlay_locked() != 1.0
    if b_is_processing:
        text_parts = ('Zen Sets - Processing ...', )
    else:
        if p_mgr_cls.is_blender:
            mode = "Maps" if p_mgr_cls.is_unique else "Groups"
        else:
            mode = "Parts" if p_mgr_cls.is_unique else "Sets"

        if is_object_mode:
            element = 'Collection'
            if not ZenPolls.is_collection_mode(context):
                element = f'Object {mode}'
            text_parts = (f'Zen Sets - {element}: [', s_display_group_name, ']')
        else:
            element = p_mgr_cls.id_element.title()
            text_parts = (f"Zen Sets - {element} {mode}: [", s_display_group_name, "]")

    i_area_width = context.area.width
    i_area_height = context.area.height

    ui_scale = context.preferences.system.ui_scale

    i_FONT_2D_SIZE = addon_prefs.label_2d.size

    blf.size(0, addon_prefs.label_2d.size, context.preferences.system.dpi)
    i_FONT_W, i_FONT_H = blf.dimensions(0, ' '.join(text_parts))

    n_left_offset = sum([
        region.width for region in context.area.regions
        if region.alignment == "LEFT" and region.width > 1
    ])
    n_top_offset = sum([
        region.height for region in context.area.regions
        if region.alignment == "TOP"
    ])
    n_right_offset = sum([
        region.width for region in context.area.regions
        if region.alignment == "RIGHT" and region.width > 1
    ])
    n_bottom_offset = sum([
        region.height for region in context.area.regions
        if region.alignment == "BOTTOM"
    ])

    i_vis_area_width = max(
        i_area_width - n_right_offset - n_left_offset - i_FONT_W - 10 * ui_scale, 200 * ui_scale)
    i_vis_area_height = max(
        i_area_height - n_top_offset - n_bottom_offset - i_FONT_H - 10 * ui_scale, 100 * ui_scale)

    d_x_ratio = i_vis_area_width / 100
    d_y_ratio = i_vis_area_height / 100

    p_align = addon_prefs.label_2d.alignment

    i_x = (
        addon_prefs.label_2d.offset_x
        if p_align in {'TOP_LEFT', 'BOTTOM_LEFT'}
        else (100 - addon_prefs.label_2d.offset_x)) * d_x_ratio
    i_y = (
        addon_prefs.label_2d.offset_y
        if p_align in {'TOP_LEFT', 'TOP_RIGHT'}
        else (100 - addon_prefs.label_2d.offset_y)) * d_y_ratio
    i_x += n_left_offset
    i_top = i_y
    i_y = i_vis_area_height - i_top

    position = (i_x, i_y)

    bgl.glEnable(bgl.GL_BLEND)

    def draw_text_2d(position, size, color, text):
        font_id = 0
        blf.enable(0, blf.SHADOW)
        blf.shadow(0, 3, 0.0, 0.0, 0.0, 1.0)
        blf.shadow_offset(0, 1, -1)
        blf.position(font_id, position[0], position[1], 0)
        blf.color(font_id, color[0], color[1], color[2], color[3])
        blf.size(font_id, size, round(72 * ui_scale))
        blf.draw(font_id, text)
        blf.disable(0, blf.SHADOW)

    def get_text_width_pos(text, position):
        text_w, _ = blf.dimensions(0, text)
        position = position[0] + text_w + 3 * ui_scale, position[1]
        return position

    if is_object_mode:
        if addon_prefs.object_options.object_display_transform_affect_options and context.active_object:
            if ((context.tool_settings.use_transform_skip_children and
                len(context.active_object.children) != 0) or
                    context.tool_settings.use_transform_pivot_point_align or
                    context.tool_settings.use_transform_data_origin):

                vec_pos = bpy_extras.view3d_utils.location_3d_to_region_2d(
                    context.region, context.space_data.region_3d,
                    context.active_object.matrix_world.translation)
                vec_pos[0] += 2 * ui_scale
                vec_pos[1] -= 10 * ui_scale

                font_size = addon_prefs.display_3d.object_collection_label_size
                transform_text = []
                if context.tool_settings.use_transform_data_origin:
                    transform_text.append("O")
                if context.tool_settings.use_transform_pivot_point_align:
                    transform_text.append("L")
                if (context.tool_settings.use_transform_skip_children and
                        len(context.active_object.children) != 0):
                    transform_text.append("P")

                draw_text_2d(vec_pos, font_size, (0, 1, 1, 1), " ".join(transform_text))

        p_act_obj = context.active_object
        b_act_obj_collections = addon_prefs.object_options.display_active_object_collections and p_act_obj is not None
        b_is_collection_mode = ZenPolls.is_collection_mode(context)

        font_size = addon_prefs.display_3d.object_collection_label_size
        d_y_offset = (font_size + 1) * ui_scale

        p_handled_objects = defaultdict(list)

        if b_act_obj_collections:
            p_bounds = get_object_bounds(p_act_obj, p_act_obj.matrix_world)
            font_collection_pos = get_position_bounds_2d(context, p_bounds)
            if font_collection_pos:
                p_handled_objects[p_act_obj].append(-1)

                font_collection_pos[1] -= (d_y_offset * (len(p_handled_objects[p_act_obj]) + 1))
                if p_act_obj.visible_get():
                    font_color = list(bpy.context.preferences.themes[0].view_3d.object_active)
                    font_color.append(1.0)
                else:
                    font_color = (0.5, 0.5, 0.5, 1.0)
                draw_text_2d(font_collection_pos, font_size, font_color, " * " + p_act_obj.name)

        if p_group and addon_prefs.object_options.display_collection_bbox and addon_prefs.object_options.display_collection_caption:
            if p_group.layer_name in _collection_cache:
                p_collection_cache = _collection_cache[p_group.layer_name]
                font_collection_pos = p_collection_cache.get_position_2d(context)
                if font_collection_pos:
                    font_color = Color(p_group.group_color)
                    font_color.v = 1.0
                    if font_color.s > 0.5:
                        font_color.s = 0.5

                    font_size = addon_prefs.display_3d.object_collection_label_size

                    font_collection_pos[1] -= font_size * ui_scale

                    if addon_prefs.display_3d.object_collection_label_background_enabled:
                        p_collection_cache.draw_caption_background(ui_scale, font_size, s_display_group_name, font_collection_pos)

                    font_color = list(font_color)
                    font_color.append(1.0)
                    draw_text_2d(font_collection_pos, font_size, font_color, s_display_group_name)

            if addon_prefs.object_options.display_collection_caption_mode == 'ALL':
                b_act_obj_collections = False

                p_handled_sizes = set()
                p_handled_rects = set()
                p_visible_objects = set(context.visible_objects)
                p_group_pairs = p_mgr_cls.get_current_group_pairs(context)

                p_handled_cols = set()

                for idx, group in p_group_pairs:
                    p_objects = set()
                    p_all_objects = set()
                    if b_is_collection_mode:
                        p_collection = group.collection
                        if p_collection is None:
                            continue

                        if p_collection in p_handled_cols:
                            continue

                        p_handled_cols.add(p_collection)

                        p_objects = p_visible_objects.intersection(p_collection.objects)
                        p_all_objects = set(p_collection.all_objects)
                    else:
                        p_objects = p_visible_objects.intersection(group.get_objects(context))
                        p_all_objects = p_objects

                    p_first_obj = (
                        p_act_obj
                        if (p_act_obj in p_all_objects)
                        else (p_objects.pop() if (len(p_objects) != 0 and group.name != s_active_group_name) else None))

                    if p_first_obj is None:
                        continue

                    p_bounds = get_object_bounds(p_first_obj, p_first_obj.matrix_world)
                    font_collection_pos = get_position_bounds_2d(context, p_bounds)
                    if font_collection_pos:
                        p_handled_objects[p_first_obj].append(idx)

                        font_color = Color(group.group_color)
                        font_color.v = 1.0
                        if font_color.s > 0.5:
                            font_color.s = 0.5

                        font_collection_pos[1] -= (d_y_offset * (len(p_handled_objects[p_first_obj]) + 1))

                        s_group_name = ''
                        if group.name == s_active_group_name:
                            s_group_name = '> '

                        s_group_name += group.name

                        d_w, d_h = blf.dimensions(0, s_group_name)
                        from ..simple_geometry import Rectangle

                        if p_first_obj != p_act_obj:
                            if any(Rectangle(r[0], r[1], r[2], r[3]).intersects(Rectangle(int(font_collection_pos[0]), int(font_collection_pos[1]), int(d_w), int(d_h)))
                                    for r in p_handled_rects):
                                continue

                        while (int(font_collection_pos[0]), int(font_collection_pos[1])) in set(p_handled_sizes):
                            font_collection_pos[1] -= d_y_offset

                        p_handled_sizes.add((int(font_collection_pos[0]), int(font_collection_pos[1])))
                        p_handled_rects.add((int(font_collection_pos[0]), int(font_collection_pos[1]), int(d_w), int(d_h)))

                        font_color = list(font_color)
                        font_color.append(1.0)
                        draw_text_2d(font_collection_pos, font_size, font_color, s_group_name)

        if b_act_obj_collections:
            p_handled_sizes = set()
            p_handled_cols = set()

            p_group_pairs = p_mgr_cls.get_current_group_pairs(context)
            for idx, group in p_group_pairs:
                p_objects = set()
                if b_is_collection_mode:
                    p_col = group.collection
                    if p_col is None:
                        continue

                    if p_col in p_handled_cols:
                        continue

                    p_handled_cols.add(p_col)

                    p_objects = set(p_col.all_objects)
                else:
                    p_objects = group.get_objects(context)

                if p_act_obj not in p_objects:
                    continue

                p_bounds = get_object_bounds(p_act_obj, p_act_obj.matrix_world)
                font_collection_pos = get_position_bounds_2d(context, p_bounds)
                if font_collection_pos:
                    p_handled_objects[p_act_obj].append(idx)

                    font_collection_pos[1] -= (d_y_offset * (len(p_handled_objects[p_act_obj]) + 1))

                    font_color = Color(group.group_color)
                    font_color.v = 1.0
                    if font_color.s > 0.5:
                        font_color.s = 0.5

                    while (int(font_collection_pos[0]), int(font_collection_pos[1])) in set(p_handled_sizes):
                        font_collection_pos[1] -= d_y_offset

                    p_handled_sizes.add((int(font_collection_pos[0]), int(font_collection_pos[1])))

                    font_color = list(font_color)
                    font_color.append(1.0)

                    s_group_name = ''
                    if group.name == s_active_group_name:
                        s_group_name = '> '

                    s_group_name += group.name

                    draw_text_2d(font_collection_pos, font_size, font_color, s_group_name)

    for i, text in enumerate(text_parts):
        cl_label = addon_prefs.label_2d.color
        if i == 1:
            cl_label = (ZsDrawConstans.BGL_ACTIVE_FONT_COLOR + (1,)) if b_group_active else (ZsDrawConstans.BGL_INACTIVE_FONT_COLOR + (0.5,))
        draw_text_2d(position, i_FONT_2D_SIZE, cl_label, text)
        position = get_text_width_pos(text, position)

    if not addon_prefs.common.auto_update_draw_cache:
        s_lock = 'LOCK'
        cl_label = (1.0, 1.0, 0.0, 1)
        draw_text_2d(position, i_FONT_2D_SIZE, cl_label, s_lock)
        position = get_text_width_pos(s_lock, position)

    extra_draws = (
        'MOD' in addon_prefs.common.display_mesh_mode,
        b_is_uv_enabled
    )

    if any(extra_draws):

        if extra_draws[0]:
            text = 'MOD*'
            draw_text_2d(position, int(addon_prefs.label_2d.size * 0.8), (1.0, 0, 0, 1), text)

            position = get_text_width_pos(text, position)

        if extra_draws[1]:
            text = 'UV'
            if not addon_prefs.uv_options.selected_only:
                text += "*"
            draw_text_2d(position, int(addon_prefs.label_2d.size * 0.8), (0.0, 1, 1, 1), text)

            # Uncomment for other extra draw ...
            # position = get_text_width_pos(text, position)

    # restore opengl defaults
    bgl.glLineWidth(1)
    bgl.glDisable(bgl.GL_BLEND)
    bgl.glEnable(bgl.GL_DEPTH_TEST)


def draw_zensets_callback_3d(context):
    if ZenLocks.get_draw_overlay_locked() == 0.0:
        return

    from ..sets.factories import get_sets_mgr
    p_mgr_cls = get_sets_mgr(context.scene)
    if p_mgr_cls is None:
        return

    addon_prefs = get_prefs()

    if context.mode == 'EDIT_MESH':
        start = timer()

        for p_obj in context.objects_in_mode:
            p_display_groups = p_mgr_cls.get_obj_highlighted_groups(p_obj)
            if len(p_display_groups):
                b_draw_cage = is_object_cage_visible(p_obj)

                p_obj_cache = get_obj_cache(p_mgr_cls, p_obj)
                p_obj_cache_eval = None

                depsgraph = context.evaluated_depsgraph_get()

                if not b_draw_cage or ('MOD' in addon_prefs.common.display_mesh_mode):
                    p_obj_eval = p_obj.evaluated_get(depsgraph)
                    if p_obj_eval and p_obj_eval.data:
                        p_obj_cache_eval = get_obj_cache(p_mgr_cls, p_obj_eval)
                        if p_obj_cache.force_rebuild_eval_cache:
                            p_obj_cache_eval.mesh_cache_uuid = None
                            p_obj_cache.force_rebuild_eval_cache = False
                        p_obj_cache_eval.draw(p_mgr_cls, p_obj_eval, p_display_groups)

                if b_draw_cage and ('CAGE' in addon_prefs.common.display_mesh_mode):
                    if (p_obj_cache_eval is None or p_obj_cache.volume != p_obj_cache_eval.volume or
                            p_obj_cache.mesh_verts_stats != p_obj_cache_eval.mesh_verts_stats):
                        p_obj_cache.z_order = 1 if p_obj_cache_eval is None else 2
                        p_obj_cache.draw(p_mgr_cls, p_obj, p_display_groups)

        timespan = timer() - start
        if timespan > 0.2:
            Log.debug('Draw BGL 3D:', p_obj.name, timespan)
    elif context.mode == 'OBJECT':
        p_scene = context.scene
        depsgraph = context.evaluated_depsgraph_get()

        if ZenStates.is_obj_color_state():
            for obj in context.selected_objects:
                p_obj = obj.evaluated_get(depsgraph)
                p_obj_cache = get_obj_cache(p_mgr_cls, p_obj)
                is_active = obj == context.active_object
                p_alpha = 0.8 if is_active else 0.6
                p_obj_cache.draw(p_mgr_cls, p_obj, obj.zen_color, is_active, p_alpha=p_alpha)
        else:
            if addon_prefs.common.object_shading:
                p_group_pairs = p_mgr_cls.get_highlighted_group_pairs(context)
                p_active_group = p_mgr_cls._get_scene_group(p_scene)

                b_bgl_init = False

                for idx, group in p_group_pairs:
                    p_objects = p_mgr_cls.get_highlighted_objects(context, group)
                    for obj in p_objects:
                        try:
                            p_obj = obj.evaluated_get(depsgraph)
                            p_obj_cache = get_obj_cache(p_mgr_cls, p_obj)
                            is_active = group == p_active_group
                            if not b_bgl_init:
                                b_bgl_init = True
                                BasicObjectsCacher.start_bgl_draw()
                            p_obj_cache.draw(p_mgr_cls, p_obj, group.group_color, is_active)
                        except Exception as e:
                            Log.error(e)

                if b_bgl_init:
                    BasicObjectsCacher.finish_bgl_draw()
                    b_bgl_init = False

        if addon_prefs.object_options.display_collection_bbox:
            global _collection_cache
            p_group = p_mgr_cls._get_scene_group(p_scene)
            if p_group:

                p_objects = set()

                if ZenPolls.is_collection_mode(context):
                    p_active_collection = p_group.collection
                    if p_active_collection:
                        p_objects = p_group.collection.all_objects
                else:
                    p_objects = p_group.get_objects(context)

                if p_group.layer_name not in _collection_cache:
                    _collection_cache[p_group.layer_name] = BasicCollectionCacher()

                if len(p_objects) != 0:
                    _collection_cache[p_group.layer_name].draw(
                        p_mgr_cls, p_objects, p_group, depsgraph)


if __name__ == "__main__":
    pass
