# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2022, Alex Zhornyak

import bpy
import bmesh
from timeit import default_timer as timer

import mathutils

from collections import defaultdict

from ..vlog import Log
from ..preferences import get_prefs

from .factories import (
    sets_factories,
    get_sets_mgr)

from ..blender_zen_utils import (
    ZenLocks,
    update_areas_in_all_screens)

from .draw_sets import (
    check_update_cache_on_change,
    is_draw_handler_enabled)


from .inject import is_modal_procedure


i_TIMER_DELAY_INTERVAL = 0.2


class FastObjectCache:
    def __init__(self) -> None:
        self.bm_data = None
        self.obj_bbox = None


class FastObjectCacheStore:

    def __init__(self) -> None:
        self._cache = {}
        self._pending_changed_objects = defaultdict(dict)
        self._pending_check_mesh_layer_managers = set()
        self._pending_draw_objects = set()
        self._obj_in_mode_cache = set()
        self._i_update_count = 0
        self._use_uv_select_sync = None
        pass

    def __del__(self):
        self.stop_events()

    def _get_bm_data(self, bm):
        return (len(bm.verts), len(bm.edges), len(bm.faces))

    def _get_bound_box(self, p_obj):
        return [mathutils.Vector(v) for v in p_obj.bound_box]

    def _verify_obj_in_cache(self, p_obj):
        if p_obj not in self._cache:
            self._cache[p_obj] = FastObjectCache()

    def check_object(self, p_obj):
        self._verify_obj_in_cache(p_obj)

        p_bm = bmesh.from_edit_mesh(p_obj.data)
        bm_data = self._get_bm_data(p_bm)
        is_mesh_changed = False
        if bm_data != self._cache[p_obj].bm_data:
            if self._cache[p_obj].bm_data is not None:
                is_mesh_changed = True
            self._cache[p_obj].bm_data = bm_data
        vert_count = bm_data[0]

        is_mesh_resized = False
        obj_bbox = self._get_bound_box(p_obj)
        if obj_bbox != self._cache[p_obj].obj_bbox:
            if self._cache[p_obj].obj_bbox is not None:
                is_mesh_resized = True
            self._cache[p_obj].obj_bbox = obj_bbox

        return (is_mesh_changed, is_mesh_resized, vert_count)

    def check_all_objects_for_hide_in_mode(self, context: bpy.types.Context):
        if bpy.app.timers.is_registered(_delayed_updated_zensets_groups):
            bpy.app.timers.unregister(_delayed_updated_zensets_groups)

        for p_obj in context.objects_in_mode:
            self._set_pending_object_state(p_obj, 0)

        bpy.app.timers.register(_delayed_updated_zensets_groups, first_interval=i_TIMER_DELAY_INTERVAL)

    def _set_pending_object_state(self, p_obj, state):
        for f in sets_factories:
            p_cls_mgr = f.get_mgr()
            if p_obj not in self._pending_changed_objects[p_cls_mgr.id_group]:
                self._pending_changed_objects[p_cls_mgr.id_group][p_obj] = state
            else:
                self._pending_changed_objects[p_cls_mgr.id_group][p_obj] |= state

    def add_pending_check_mesh_layers_manager(self, p_cls_mgr):
        self._pending_check_mesh_layer_managers.add(p_cls_mgr)

    def on_edit_mesh_is_updated(self, scene):

        i_mesh_vieport_verts_count = 0
        b_flag_pending_updates = False
        is_any_obj_modified = False

        ctx = bpy.context

        p_cls_mgr = get_sets_mgr(ctx.scene)
        if not p_cls_mgr:
            return

        addon_prefs = get_prefs()

        is_draw_enabled = addon_prefs.common.auto_update_draw_cache and is_draw_handler_enabled(p_cls_mgr)

        depsgraph = ctx.evaluated_depsgraph_get()

        b_force_uv_update = p_cls_mgr.is_uv_force_update(only_selection=False)

        unique_datas = {p_obj.data: p_obj for p_obj in ctx.objects_in_mode_unique_data}

        for update in depsgraph.updates:
            if not isinstance(update.id, bpy.types.Mesh):
                continue

            p_obj = unique_datas.get(update.id.original, None)
            if p_obj and p_obj.data.is_editmode:
                # print(
                #     "[" + p_obj.name + "] ",
                #     'UPDATE - Geometry:', update.is_updated_geometry,
                #     'Shading:', update.is_updated_shading,
                #     'Force UV:', b_force_uv_update)

                if update.is_updated_geometry or (update.is_updated_shading and b_force_uv_update):
                    is_obj_changed, is_obj_resized, vert_count = self.check_object(p_obj)

                    i_mesh_vieport_verts_count += vert_count

                    if not b_flag_pending_updates:
                        b_flag_pending_updates = True

                    if is_obj_resized or is_obj_changed:
                        is_any_obj_modified = True

                    if is_obj_changed:
                        self._set_pending_object_state(p_obj, 1)
                    else:
                        if p_cls_mgr.is_update_elements_monitor_enabled(addon_prefs) and update.is_updated_geometry:
                            self._set_pending_object_state(p_obj, 0)

                    if is_draw_enabled:
                        self._pending_draw_objects.add(p_obj)

        if p_cls_mgr in self._pending_check_mesh_layer_managers:
            if bpy.app.timers.is_registered(_delayed_check_zensets_for_new_layer):
                bpy.app.timers.unregister(_delayed_check_zensets_for_new_layer)

            bpy.app.timers.register(_delayed_check_zensets_for_new_layer)

        objs_in_mode = set(obj.name for obj in ctx.objects_in_mode)
        if objs_in_mode != self._obj_in_mode_cache:
            if p_cls_mgr:
                p_cls_mgr.build_lookup_table(ctx)
            self._obj_in_mode_cache = objs_in_mode

        # use boolean 'or' because it may be switched to different set of objects
        if b_flag_pending_updates or len(self._pending_changed_objects[p_cls_mgr.id_group]) > 0:
            if bpy.app.timers.is_registered(_delayed_updated_zensets_groups):
                bpy.app.timers.unregister(_delayed_updated_zensets_groups)

            bpy.app.timers.register(_delayed_updated_zensets_groups, first_interval=i_TIMER_DELAY_INTERVAL)

        # use boolean 'and' because only single set of draw objects
        if b_flag_pending_updates and len(self._pending_draw_objects) > 0:
            was_registered_draw = bpy.app.timers.is_registered(_delayed_pending_zensets_draw_update)
            if was_registered_draw:
                bpy.app.timers.unregister(_delayed_pending_zensets_draw_update)

            if was_registered_draw or i_mesh_vieport_verts_count > 15000 or is_any_obj_modified or is_modal_procedure(ctx):
                bpy.app.timers.register(_delayed_pending_zensets_draw_update, first_interval=i_TIMER_DELAY_INTERVAL)
                ZenLocks.set_lock_draw_overlay(0.0)
            else:
                self._check_pending_draw_objects()
                ZenLocks.set_lock_draw_overlay(1.0)

    def _check_pending_draw_objects(self):
        if len(self._pending_draw_objects):

            p_cls_draw_mgr = None
            for f in sets_factories:
                if is_draw_handler_enabled(f.get_mgr()):
                    p_cls_draw_mgr = f.get_mgr()
                    break

            if p_cls_draw_mgr:
                depsgraph = bpy.context.evaluated_depsgraph_get()

                pending_objects = self._pending_draw_objects.intersection(bpy.context.objects_in_mode)

                for p_obj in pending_objects:
                    p_obj_eval = p_obj.evaluated_get(depsgraph)
                    interval = timer()
                    check_update_cache_on_change(p_cls_draw_mgr, p_obj, p_obj_eval)
                    elapsed = timer() - interval

                    self._i_update_count += 1
                    Log.debug(self._i_update_count, p_obj.name, '- DRAW CHECKED:', elapsed)

            self._pending_draw_objects.clear()

    def stop_events(self):
        states = (
            _delayed_updated_zensets_groups,
            _delayed_pending_zensets_draw_update
        )
        for state in states:
            if bpy.app.timers.is_registered(state):
                bpy.app.timers.unregister(state)

        return states


_ObjectCacheStore = FastObjectCacheStore()


def _delayed_check_zensets_for_new_layer():
    ctx = bpy.context

    if ctx.mode == 'EDIT_MESH':
        p_scene = ctx.scene

        p_cls_mgr = get_sets_mgr(p_scene)
        if p_cls_mgr in _ObjectCacheStore._pending_check_mesh_layer_managers:
            _ObjectCacheStore._pending_check_mesh_layer_managers.remove(p_cls_mgr)

            p_cls_mgr.check_for_new_layers(p_scene, ctx.objects_in_mode)


def _delayed_updated_zensets_groups():
    # print('Check objects:', _ObjectCacheStore._pending_changed_objects)
    ctx = bpy.context

    if is_modal_procedure(ctx):
        return i_TIMER_DELAY_INTERVAL

    if ctx.mode == 'EDIT_MESH':
        p_cls_mgr = get_sets_mgr(ctx.scene)
        if p_cls_mgr and len(_ObjectCacheStore._pending_changed_objects[p_cls_mgr.id_group]) > 0:
            interval = timer()

            addon_prefs = get_prefs()
            if getattr(addon_prefs.modes, f'enable_{p_cls_mgr.id_group}'):
                p_objects_in_mode = set(ctx.objects_in_mode)
                b_need_build_lookup = False
                for p_obj, state in _ObjectCacheStore._pending_changed_objects[p_cls_mgr.id_group].items():
                    if p_obj in p_objects_in_mode:
                        if state == 1:
                            Log.debug(p_obj.name, ' - BMESH WAS CHANGED!')

                        p_cls_mgr.update_all_obj_groups_count(p_obj, no_lookup=True)
                        b_need_build_lookup = True

                if b_need_build_lookup:
                    p_cls_mgr.build_lookup_table(ctx)

                Log.debug('Hidden state was updated, secs:', timer() - interval)

            _ObjectCacheStore._pending_changed_objects[p_cls_mgr.id_group].clear()


def _delayed_pending_zensets_draw_update():
    ctx = bpy.context
    if is_modal_procedure(ctx):
        ZenLocks.set_lock_draw_overlay(0.0)
        return i_TIMER_DELAY_INTERVAL

    _ObjectCacheStore._check_pending_draw_objects()

    percent = ZenLocks.get_draw_overlay_locked()

    if percent != 1.0:
        percent += 0.1
    if percent > 1.0:
        percent = 1.0
    ZenLocks.set_lock_draw_overlay(percent)

    if percent > 0.1:
        update_areas_in_all_screens()

    res = None if percent == 1.0 else 0.05
    # print('res:', res, 'percent:', percent)

    return res
