# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

""" Zen Vert Groups System """
# blender
import bpy
import bmesh
from bl_operators.presets import AddPresetBase

from collections import defaultdict
from statistics import mean
import numpy as np
import math
from typing import Tuple

from ..basic_sets import Zs_UL_BaseList
from ..bl_sets import ZsBlenderSets, ZsBlenderScenePropList, ZsBlenderObjectPropList, ZsBlenderGroupIndexSync
from ..basic_list_sets import ZsListLayerManager
from ..draw_cache import VertsCacher
from ..draw_sets import mark_groups_modified
from ..lookup_utils import ZsLookup

from ...blender_zen_utils import ZenPolls, color_to_vertex_weight, update_areas_in_all_screens, vertex_weight_to_color
from ...preferences import ZSTS_AddonPreferences, get_prefs, i_MAX_WEIGHT_PRESET_SIZE, ZsMinMaxGroup
from ...labels import ZsLabels


class ZsVGroupLayerManager(ZsListLayerManager, ZsBlenderSets):

    list_item_prefix = 'VGroups'
    id_group = 'blgroup'
    id_mask = 'ZSBLVG'
    id_element = 'vert'
    id_display_element = 'blgroup'
    id_uv_select_mode = 'VERTEX'
    is_blender = True

    """ Verts section """
    @classmethod
    def get_bm_items(self, bm):
        bm.verts.ensure_lookup_table()
        return bm.verts

    @classmethod
    def get_selected_count(self, p_obj):
        me = p_obj.data
        return me.total_vert_sel

    @classmethod
    def get_mesh_select_mode(self):
        # 0 - vert, 1 - edge, 2 - face
        return (True, False, False)

    @classmethod
    def get_item_loops(self, p_item):
        return p_item.link_loops

    @classmethod
    def get_bl_groups(self, p_obj: bpy.types.Object):
        return p_obj.vertex_groups

    @classmethod
    def is_update_elements_monitor_enabled(cls, p_addon_prefs: ZSTS_AddonPreferences):
        return (
            super().is_update_elements_monitor_enabled(p_addon_prefs) or
            p_addon_prefs.list_options.display_vertex_groups_weights_info)

    @classmethod
    def set_selection_to_group(self, p_obj, p_scene_group, indices):
        layerName = p_scene_group.layer_name

        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        b_indices_mode = indices is not None
        if b_indices_mode:
            i_selected_count = len(indices)
        else:
            b_is_uv = self.is_uv_area_and_not_sync()
            if b_is_uv:
                uv_sel = self.fetch_uv_selections(bm)
                i_selected_count = len(uv_sel)
            else:
                i_selected_count = self.get_selected_count(p_obj)

        mesh_layer = self.ensure_mesh_layer(p_obj, bm, layerName) if i_selected_count else self.get_mesh_layer(p_obj, bm, layerName)
        if mesh_layer:
            i_group_count = 0
            p_weights = set()
            p_deform, idx = mesh_layer
            for item in self.get_bm_items(bm):
                is_selected = (
                    (item.index in indices) if b_indices_mode
                    else (item.select and (not b_is_uv or (item.index in uv_sel)))
                )
                self.set_bm_item(item, mesh_layer, is_selected)
                if is_selected:
                    p_weights.add(item[p_deform][idx])
                    i_group_count += 1

            if i_group_count:
                p_group = self.ensure_group_in_object(p_obj, p_scene_group)
                p_group.group_count = i_group_count
                p_group.group_hide_count = 0
                p_group.group_weight = self.get_weight_from_set(p_weights)

            bm.select_flush_mode()
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

    @classmethod
    def set_selection_to_new_group(self, p_obj, p_scene_group, i_start, i_end):
        layerName = p_scene_group.layer_name

        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        i_selected_count = self.get_selected_count(p_obj)

        mesh_layer = self.ensure_mesh_layer(p_obj, bm, layerName) if i_selected_count else self.get_mesh_layer(p_obj, bm, layerName)
        if mesh_layer and i_selected_count != 0:
            i_group_count = 0
            bm_items = self.get_bm_items(bm)
            p_weights = set()
            p_deform, idx = mesh_layer
            for idx in range(i_start, i_end, 1):
                item = bm_items[idx]
                is_selected = item.select
                self.set_bm_item(item, mesh_layer, is_selected)
                if is_selected:
                    i_group_count += 1
                    i_selected_count -= 1
                    p_weights.add(item[p_deform][idx])
                    if i_selected_count == 0:
                        break

            if i_group_count:
                p_group = self.ensure_group_in_object(p_obj, p_scene_group)
                p_group.group_count = i_group_count
                p_group.group_hide_count = 0
                p_group.group_weight = self.get_weight_from_set(p_weights)
            bm.select_flush_mode()
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

    @classmethod
    def update_deform_in_group(self, p_obj, p_scene_group):
        layerName = p_scene_group.layer_name

        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)

        mesh_layer = self.get_mesh_layer(p_obj, bm, layerName)
        if mesh_layer:
            i_group_count = 0
            i_group_hide_count = 0
            for item in self.get_bm_items(bm):
                if self.is_bm_item_set(item, mesh_layer):
                    self.set_bm_item(item, mesh_layer, True)
                    i_group_count += 1
                    if item.hide:
                        i_group_hide_count += 1

            if i_group_count:
                p_group = self.ensure_group_in_object(p_obj, p_scene_group)
                p_group.group_count = i_group_count
                p_group.group_hide_count = i_group_hide_count
                p_group.group_weight = bpy.context.tool_settings.vertex_group_weight

            bm.select_flush_mode()
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

    @classmethod
    def update_all_obj_groups_count(self, p_obj: bpy.types.Object, no_lookup=False):
        p_obj_list = self.get_list(p_obj)
        if [g.layer_name for g in p_obj_list] != [vg.name for vg in p_obj.vertex_groups]:
            self.update_list(bpy.context)
            if not no_lookup:
                self.build_lookup_table(bpy.context)
            return True

        b_need_update = False
        b_need_update_draw = False

        dic = defaultdict(list)
        dic_weight = defaultdict(set)
        bm = self._get_bm(p_obj)

        p_deform = bm.verts.layers.deform.active
        if p_deform:
            for item in self.get_bm_items(bm):
                for idx in item[p_deform].keys():
                    dic[idx].append(item.hide)
                    dic_weight[idx].add(item[p_deform][idx])

        for idx, group in enumerate(p_obj_list):
            i_group_count = 0
            i_hide_count = 0
            d_weight = 0
            if idx in dic:
                arr = np.fromiter(dic[idx], 'b')
                i_group_count = len(arr)
                i_hide_count = np.count_nonzero(arr)
            if idx in dic_weight:
                d_weight = self.get_weight_from_set(dic_weight[idx])

            b_elements_changed = i_group_count != group.group_count or i_hide_count != group.group_hide_count
            b_weights_changed = not math.isclose(d_weight, group.group_weight, abs_tol=1e-06)

            if b_elements_changed or b_weights_changed:
                group.group_count = i_group_count
                group.group_hide_count = i_hide_count
                group.group_weight = d_weight
                b_need_update = True

            if b_elements_changed:
                b_need_update_draw = True

        if b_need_update and not no_lookup:
            self.build_lookup_table(bpy.context)

        if b_need_update_draw and no_lookup:
            mark_groups_modified(self, p_obj)

        return b_need_update

    @classmethod
    def _do_get_lookup_extrainfo(cls, context: bpy.types.Context, p_group_pair, p_lookup_table):
        idx, p_group = p_group_pair

        n_group_count = 0
        n_obj_count = 0
        n_hide_count = 0

        p_weights = set()
        b_is_mean_weight = False

        for obj in context.objects_in_mode:
            group = cls._get_group_by_layer(obj, p_group.layer_name)
            n_current_count = -1
            n_current_hide_count = 0
            n_current_weight = 0
            if group is not None:
                n_current_count = group.group_count
                n_current_hide_count = group.group_hide_count
                n_current_weight = group.group_weight

            if n_current_count > -1:
                n_group_count = n_group_count + n_current_count
                n_obj_count = n_obj_count + 1
                n_hide_count = n_hide_count + n_current_hide_count

                if n_current_count > 0:
                    if n_current_weight < 0:
                        n_current_weight *= -1
                        b_is_mean_weight = True
                    p_weights.add(n_current_weight)

                if n_current_count > 0:
                    p_lookup_table[ZsLookup.ObjectHighlightedGroups][obj].append(idx)

        d_weight = cls.get_weight_from_set(p_weights)
        if b_is_mean_weight and d_weight > 0:
            d_weight *= -1

        p_extra_info = {
            'count': n_group_count,
            'objects': n_obj_count,
            'hide_count': n_hide_count,
            'weight': d_weight
        }

        return p_extra_info

    @classmethod
    def get_weight_from_set(self, p_weights):
        i_weights_count = len(p_weights)
        if i_weights_count == 1:
            return p_weights.pop()
        elif i_weights_count > 0:
            return mean(p_weights) * -1
        return 0

    @classmethod
    def get_obj_group_count(self, p_obj, layerName):
        me = p_obj.data
        if me and p_obj.type == 'MESH':
            if me.is_editmode:
                bm = self._get_bm(p_obj)
                layer = self.get_mesh_layer(p_obj, bm, layerName)
                p_items = []
                p_weights = []
                if layer:
                    p_zip_items = [
                        (item.hide, item[layer[0]][layer[1]])
                        for item in self.get_bm_items(bm) if self.is_bm_item_set(item, layer)]
                    if len(p_zip_items) > 0:
                        p_items, p_weights = zip(*p_zip_items)
                    p_weights = set(p_weights)

                d_weight = self.get_weight_from_set(p_weights)

                return {
                    'count': len(p_items),
                    'hide_count': p_items.count(True),
                    'weight': d_weight
                }
        return {}

    @classmethod
    def ensure_mesh_layer(self, p_obj, p_bm, layerName):
        p_layer = self.get_mesh_layer(p_obj, p_bm, layerName)
        if p_layer is None:

            p_obj.vertex_groups.new(name=layerName)
            p_deform = p_bm.verts.layers.deform.verify()
            idx = len(p_obj.vertex_groups) - 1

            return (p_deform, idx)

        return p_layer

    @classmethod
    def get_mesh_layer(self, p_obj, p_bm, layerName):
        idx = p_obj.vertex_groups.find(layerName)
        if idx != -1:
            p_bm_items = self.get_bm_items(p_bm)
            p_deform = p_bm_items.layers.deform.active
            if p_deform:
                return (p_deform, idx)

        return None

    @classmethod
    def get_mesh_layers_in_group_pairs(self, p_obj, p_bm, p_group_pairs):
        return [(layer, p_obj.vertex_groups[layer[1]].name) for layer in [self.get_mesh_layer(p_obj, p_bm, g.layer_name) for _, g in p_group_pairs] if layer]

    @classmethod
    def is_bm_item_set(self, p_bm_item, p_layer):
        p_deform, idx = p_layer
        return p_bm_item[p_deform].get(idx) is not None

    @classmethod
    def set_bm_item(self, p_bm_item, p_layer, p_val):
        p_deform, idx = p_layer
        if p_val:
            p_bm_item[p_deform][idx] = bpy.context.tool_settings.vertex_group_weight
        else:
            p_deform_item = p_bm_item[p_deform].get(idx)
            if p_deform_item is not None:
                del p_bm_item[p_deform][idx]

    @classmethod
    def fetch_uv_selections(self, bm):
        uv_selected = set()

        uv_layer = bm.loops.layers.uv.active
        if uv_layer:

            uv_selected = set(
                item.index
                for item in self.get_bm_items(bm)
                if not item.hide and
                all(
                    loop[uv_layer].select and loop.face.select
                    for loop in item.link_loops)
            )

        return uv_selected

    @classmethod
    def create_unique_layer_name(self):
        p_scene = bpy.context.scene
        p_scene_list = self.get_list(p_scene)
        i_count = 1
        s_layer_name = self.list_item_prefix + f'.{i_count:03d}'
        while self._index_of_layer(p_scene_list, s_layer_name) != -1:
            s_layer_name = self.list_item_prefix + f'.{i_count:03d}'
            i_count += 1
        return s_layer_name

    @classmethod
    def remove_mesh_layer(self, p_obj, p_bm, layerName, cleanup=False):
        p_group = p_obj.vertex_groups.get(layerName)
        if p_group:
            p_obj.vertex_groups.remove(p_group)

    @classmethod
    def _add_list_layer(self, p_list, layerName, sItemPrefix, p_color):
        p_list.add()

        i = len(p_list) - 1

        p_list[-1]['name'] = layerName
        p_list[-1]['layer_name'] = layerName
        p_list[-1].group_color = p_color

        return i

    @classmethod
    def get_cacher(self):
        return VertsCacher()

    @classmethod
    def _do_draw_vertex_weight(self, context: bpy.types.Context, layout: bpy.types.UILayout):
        wm = context.window_manager
        op_last = wm.operator_properties_last('zsts.internal_change_weight_by_index')

        layout.prop(context.tool_settings, "vertex_group_weight", text="Weight")

        if op_last:
            r = layout.row(align=True)
            r.ui_units_x = 1
            r.prop(op_last, 'weight_color', text='')
            layout.prop_menu_enum(op_last, 'weights', text='', icon='DOWNARROW_HLT')

    @classmethod
    def remove_weight_from_group(self, p_obj, layerName, props):
        me = p_obj.data
        bm = bmesh.from_edit_mesh(me)
        layer = self.get_mesh_layer(p_obj, bm, layerName)
        if layer:
            was_modified = False
            p_deform, idx = layer
            for item in self.get_bm_items(bm):
                p_weight = item[p_deform].get(idx)
                if p_weight is not None:
                    if p_weight >= props.min_max.min and p_weight <= props.min_max.max:
                        del item[p_deform][idx]
                        was_modified = True

            if was_modified:
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
                bm.select_flush_mode()
                self.update_obj_group_count(p_obj, layerName)

                mark_groups_modified(self, p_obj)

    @classmethod
    def execute_RemoveWeightFromGroup(self, operator: bpy.types.Operator, context: bpy.types.Context):
        try:
            p_pairs = self.get_current_group_pairs(context) if operator.mode == 'ALL' else [self.get_current_group_pair(context)]
            b_modified = False
            for p_group_pair in p_pairs:
                if p_group_pair[1].layer_name != '':
                    for p_obj in context.objects_in_mode:
                        self.remove_weight_from_group(p_obj, p_group_pair[1].layer_name, operator)
                        b_modified = True
            if b_modified:
                return {'FINISHED'}

            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'FINISHED'}

    @classmethod
    def unhide_all(self, context, b_select, props):
        super().unhide_all(context, b_select, props)

        addon_prefs = get_prefs()

        if addon_prefs.vgroups_options.mask_modifier:
            for p_obj in context.objects_in_mode:
                for p_mod in p_obj.modifiers:
                    if p_mod.type == 'MASK' and p_mod.mode == 'VERTEX_GROUP':
                        if addon_prefs.vgroups_options.mask_delete_after:
                            p_obj.modifiers.remove(p_mod)
                        else:
                            p_mod.show_viewport = False

    @classmethod
    def set_group_pair_invert_hide(self, context, p_group_pair, props) -> Tuple[bool, bool]:
        p_res = super().set_group_pair_invert_hide(context, p_group_pair, props)

        if p_group_pair:
            addon_prefs = get_prefs()
            if addon_prefs.vgroups_options.mask_modifier:
                _, p_group = p_group_pair
                p_mask_mod = None
                for p_obj in context.objects_in_mode:
                    for p_mod in p_obj.modifiers:
                        if p_mod.type == 'MASK' and p_mod.mode == 'VERTEX_GROUP':
                            if p_mod.vertex_group == p_group.name:
                                p_mask_mod = p_mod
                            else:
                                if addon_prefs.vgroups_options.mask_delete_after:
                                    p_obj.modifiers.remove(p_mod)
                                else:
                                    p_mod.show_viewport = False

                if p_mask_mod is None:
                    p_mask_mod = p_obj.modifiers.new('ZenMask.' + p_group.name, 'MASK')

                p_mask_mod.mode = 'VERTEX_GROUP'
                p_mask_mod.vertex_group = p_group.name
                p_mask_mod.invert_vertex_group = False
                p_mask_mod.show_viewport = True

        return p_res

    @classmethod
    def set_group_pair_mask_state(self, context: bpy.types.Context, p_group_pair, p_hidden, props):
        _, p_group = p_group_pair

        p_pairs = self.get_current_group_pairs(context)

        t_hidden_groups = {}

        for idx, p_it_group in p_pairs:
            t_info = self.get_lookup_extra_info_by_index(context, idx)
            n_count = t_info.get('count', 0)
            n_hide_count = t_info.get('hide_count', 0)

            if p_it_group.name != p_group.name:
                t_hidden_groups[p_it_group.name] = n_count > 0 and n_hide_count == n_count

        addon_prefs = get_prefs()

        def process_group(b_hide, p_mod, vg_name):
            if b_hide:
                if p_mod is None:
                    p_mod = p_obj.modifiers.new('ZenMask.' + vg_name, 'MASK')

                p_mod.mode = 'VERTEX_GROUP'
                p_mod.vertex_group = vg_name
                p_mod.invert_vertex_group = True
                p_mod.show_viewport = True
            else:
                if p_mod:
                    if addon_prefs.vgroups_options.mask_delete_after:
                        p_obj.modifiers.remove(p_mod)
                    else:
                        p_mod.show_viewport = False

        for p_obj in context.objects_in_mode:

            p_mask_mod = None

            p_handled_names = set()

            for p_mod in p_obj.modifiers:
                if p_mod.type == 'MASK' and p_mod.mode == 'VERTEX_GROUP':

                    p_handled_names.add(p_mod.vertex_group)

                    if p_mod.vertex_group == p_group.name:
                        p_mask_mod = p_mod
                    else:
                        b_hide = t_hidden_groups.get(p_mod.vertex_group, False)
                        process_group(b_hide, p_mod, p_mod.vertex_group)

            for p_name, state in t_hidden_groups.items():
                if p_name not in p_handled_names:
                    process_group(state, None, p_name)

            process_group(p_hidden, p_mask_mod, p_group.name)

    @classmethod
    def set_group_pair_hidden_state(self, context: bpy.types.Context, p_group_pair, hidden_state, props):
        p_result = super().set_group_pair_hidden_state(context, p_group_pair, hidden_state, props)

        addon_prefs = get_prefs()
        if addon_prefs.vgroups_options.mask_modifier:
            self.set_group_pair_mask_state(context, p_group_pair, hidden_state, props)

        return p_result

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

        layout = menu.layout
        layout.operator('zsts.remove_weight_from_group')

    @classmethod
    def execute_DrawImport(cls, layout, context, is_menu=False):
        super().execute_DrawImport(layout, context, is_menu)

        # if is_menu:
        #     layout.separator()
        # col = layout.column(align=True)

    @classmethod
    def execute_DrawExport(cls, layout, context, is_menu=False):
        super().execute_DrawExport(layout, context, is_menu)

        # if is_menu:
        #     layout.separator()
        # col = layout.column(align=True)

    @classmethod
    def execute_DrawTools(cls, tools, context):
        super().execute_DrawTools(tools, context)

        layout = tools.layout
        layout.operator('zsts.remove_weight_from_group')

    @classmethod
    def execute_DrawListItemRightProps(
            cls, layout: bpy.types.UILayout, context: bpy.types.Context,
            item, index,
            p_info, addon_prefs):

        n_group_count = p_info.get('count', 0)
        n_obj_count = p_info.get('objects', 0)
        n_hide_count = p_info.get('hide_count', 0)

        layout.label(text=str(n_group_count) if n_obj_count else '-')
        if addon_prefs.list_options.display_objects_info:
            layout.label(text=str(n_obj_count) if n_obj_count else '-')

        if addon_prefs.list_options.display_hidden_groups_info:
            if n_obj_count:
                b_alert = any(
                    True

                    for p_obj in context.objects_in_mode
                    for p_mod in p_obj.modifiers
                    if p_mod.type == 'MASK' and p_mod.mode == 'VERTEX_GROUP' and p_mod.show_viewport and
                    (
                        (p_mod.vertex_group == item.name and p_mod.invert_vertex_group) or
                        (p_mod.vertex_group != item.name and not p_mod.invert_vertex_group)
                    )
                )

                r = layout
                if b_alert:
                    r = layout.row(align=True)
                    r.alignment = 'RIGHT'
                    r.alert = True

                icon_hide_id = 'HIDE_OFF' if n_hide_count == 0 else ('HIDE_ON' if n_hide_count == n_group_count else 'RADIOBUT_OFF')
                op = r.operator('zsts.internal_hide_group_by_index', text='', icon=icon_hide_id, emboss=False)
                op.group_index = index
            else:
                layout.label(text='-')

        if addon_prefs.list_options.display_vertex_groups_weights_info:
            d_weight = p_info.get('weight')
            r = layout.row(align=True)
            r.alignment = 'RIGHT'
            r.alert = d_weight < 0
            d_weight = math.fabs(d_weight)
            op = r.operator(
                 'zsts.internal_change_weight_by_index', text=f'{d_weight:.3f}')
            op.group_index = index

    @classmethod
    def _do_draw_overlay_filter(self, context: bpy.types.Context, layout: bpy.types.UILayout):
        super()._do_draw_overlay_filter(context, layout)

        if hasattr(context, 'space_data') and context.space_data and context.space_data.type == 'VIEW_3D':
            layout.separator()
            layout.prop(context.space_data.overlay, 'show_weight')
            row = layout.row(align=True)
            row.active = context.space_data.overlay.show_weight
            row.prop(context.tool_settings, 'vertex_group_user', expand=True)


_cache_weights_preset = []


class ZSBLVG_OT_InternalChangeWeightByIndex(bpy.types.Operator, ZsVGroupLayerManager):
    """ Zen Sets Hide Group By Index """
    bl_idname = 'zsts.internal_change_weight_by_index'
    bl_label = 'Vertex Group Weight'
    bl_description = 'Click to change Vertex Group Weight'
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(name='Group Index', options={'HIDDEN', 'SKIP_SAVE'})

    def get_weights_preset(self, context):
        global _cache_weights_preset

        addon_prefs = get_prefs()

        p_new_weights = []
        p_weights = []

        for i in range(addon_prefs.vgroups_options.weight_preset_size):
            p_weights.append(addon_prefs.vgroups_options.weight_preset[i])
        arr = np.fromiter(p_weights, dtype='f')
        arr = arr.round(decimals=6)
        _, _cache_weights_preset_indices = np.unique(arr, return_index=True)

        for idx in _cache_weights_preset_indices:
            p_new_weights.append(
                (str(idx), str(arr[idx]), '')
            )

        # use only this construction otherwise 'invoke_props_dialog' works wrong!
        if _cache_weights_preset != p_new_weights:
            _cache_weights_preset = p_new_weights

        return _cache_weights_preset

    def update_weights(self, context: bpy.types.Context):
        idx = int(self.weights)
        addon_prefs = get_prefs()
        if idx in range(len(addon_prefs.vgroups_options.weight_preset)):
            p_weight = addon_prefs.vgroups_options.weight_preset[idx]
            if not math.isclose(p_weight, context.tool_settings.vertex_group_weight, abs_tol=1e-06):
                context.tool_settings.vertex_group_weight = p_weight

    weights: bpy.props.EnumProperty(
        name='Weights',
        items=get_weights_preset,
        update=update_weights
    )

    def get_weight_color(self):
        return vertex_weight_to_color(bpy.context.tool_settings.vertex_group_weight)

    def set_weight_color(self, value):
        p_weight = color_to_vertex_weight(value)
        if bpy.context.tool_settings.vertex_group_weight != p_weight:
            bpy.context.tool_settings.vertex_group_weight = p_weight

    weight_color: bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_COLOR_NAME,
        subtype='COLOR_GAMMA',
        size=3,
        get=get_weight_color,
        set=set_weight_color,
        min=0, max=1
    )

    def draw(self, context):
        layout = self.layout
        col = layout.column(align=True)
        row = col.row(align=True)
        row.prop(context.tool_settings, 'vertex_group_weight', text='')
        r = row.row(align=True)
        r.scale_x = 0.3
        r.prop(self, 'weight_color', text='')
        col.separator()
        col.prop(self, 'weights', expand=True)

    def execute(self, context):
        try:
            p_scene = context.scene
            p_group = None
            p_list = self.get_list(p_scene)
            if self.group_index in range(len(p_list)):
                p_group = p_list[self.group_index]

            if p_group:
                for p_obj in context.objects_in_mode:
                    self.update_deform_in_group(p_obj, p_group)

                self.build_lookup_table(context)

                update_areas_in_all_screens()

                return {'FINISHED'}
            else:
                self.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            self.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    def invoke(self, context, event):
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        if self.group_index in range(len(p_list)):
            # ? Do we need to see the current value ?
            # t_info = self.get_lookup_extra_info_by_index(context, self.group_index)
            # context.tool_settings.vertex_group_weight = t_info['weight']

            wm = context.window_manager
            res = wm.invoke_props_dialog(self, width=150)
            return res
        else:
            return {'CANCELLED'}


def zen_sets_update_vertex_groups_list():
    ctx = bpy.context
    if ctx.mode == 'EDIT_MESH':
        from ..factories import get_sets_mgr
        p_cls_mgr = get_sets_mgr(ctx.scene)
        if p_cls_mgr == ZsVGroupLayerManager:
            p_cls_mgr.update_list(ctx)
            p_cls_mgr.build_lookup_table(ctx)

            update_areas_in_all_screens()


s_WEIGHT_PRESET_SUBDIR = 'zen_sets/weight_presets'


class ZSBLVG_MT_StoreWeightPresets(bpy.types.Menu):
    bl_label = 'Store Weight Preset'
    preset_subdir = s_WEIGHT_PRESET_SUBDIR
    preset_operator = 'script.execute_preset'
    draw = bpy.types.Menu.draw_preset


class ZSBLVG_OT_StoreAddWeightPreset(AddPresetBase, bpy.types.Operator):
    bl_idname = 'zsts.add_vertex_weight_preset'
    bl_label = 'Add|Remove Preset'
    preset_menu = 'ZSBLVG_MT_StoreWeightPresets'

    @classmethod
    def description(cls, context, properties):
        return ('Remove' if properties.remove_active else 'Add') + ' Vertex Weight Preset'

    # Common variable used for all preset values
    preset_defines = [
                        'prefs = bpy.context.preferences.addons["ZenSets"].preferences.vgroups_options'
                     ]

    def _get_preset_values():
        val = ['prefs.weight_preset_size']
        for i in range(i_MAX_WEIGHT_PRESET_SIZE):
            val.append(f'prefs.weight_preset[{i}]')
        return val

    # Properties to store in the preset
    preset_values = _get_preset_values()

    # Directory to store the presets
    preset_subdir = s_WEIGHT_PRESET_SUBDIR


class ZSBLVG_OT_RemoveWeightFromGroup(bpy.types.Operator, ZsVGroupLayerManager):
    """ Remove weight from group """
    bl_idname = 'zsts.remove_weight_from_group'
    bl_label = 'Remove Weight'
    bl_description = 'Remove elements with Vertex Weight in given range from Active or All Group(s)'
    bl_options = {'REGISTER', 'UNDO'}

    mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', 'Remove Weight from Active Group'),
            ('ALL', 'All', 'Remove Weight from All Groups'),
        ],
        default='ACTIVE')

    min_max: bpy.props.PointerProperty(type=ZsMinMaxGroup)

    def execute(self, context):
        return self.execute_RemoveWeightFromGroup(self, context)

    def draw(self, context):
        layout = self.layout
        layout.use_property_decorate = True
        layout.use_property_split = True

        layout.prop(self, 'mode', expand=True)
        self.min_max.draw(layout, context)

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)


class ZSBLVG_OT_DisplayWeights(bpy.types.Operator):
    """ Display Vertex Group Weights """
    bl_idname = 'mesh.zsts_display_vertex_weights'
    bl_label = 'Display Weights'
    bl_description = 'Display Vertex Group Weights'
    bl_options = {'REGISTER', 'UNDO'}

    mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('UNDEFINED', 'Undefined', ''),
            ('SHOW', 'Show', ''),
            ('HIDE', 'Hide', ''),
        ],
        default='UNDEFINED')

    def execute(self, context):
        if ZenPolls.is_view3d_space_data(context) and self.mode != 'UNDEFINED':
            context.space_data.overlay.show_weight = self.mode == 'SHOW'
            return {'FINISHED'}

        return {'CANCELLED'}

    def draw_ex(self, layout, context):
        b_active = False
        if ZenPolls.is_view3d_space_data(context) and self.mode != 'UNDEFINED':
            b_active = context.space_data.overlay.show_weight

        row = layout.row(align=True)
        row.active = b_active
        row.prop(context.tool_settings, 'vertex_group_user', expand=True)

    def draw(self, context):
        self.draw_ex(self.layout, context)

    def invoke(self, context, event):
        self.mode = 'UNDEFINED'
        if ZenPolls.is_view3d_space_data(context):
            self.mode = 'HIDE' if context.space_data.overlay.show_weight else 'SHOW'
            return self.execute(context)

        return {'CANCELLED'}


class ZSBLVG_OT_AssignVertexWeightToGroupColor(bpy.types.Operator, ZsVGroupLayerManager):
    bl_idname = "zsts.assign_vertex_weight_to_group_color"
    bl_label = ZsLabels.OT_PALETTE_ASSIGN_COLOR_FROM_WEIGHT_LABEL
    bl_description = ZsLabels.OT_PALETTE_ASSIGN_COLOR_FROM_WEIGHT_DESC
    bl_options = {'REGISTER', 'UNDO'}

    group_layer: bpy.props.StringProperty(
        options={'HIDDEN', 'SKIP_SAVE'},
        default='')

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', ''),
            ('ALL', 'All', ''),
        ],
        default='ACTIVE')

    @classmethod
    def poll(cls, context):
        return cls.get_current_group_pair(context) is not None

    def draw(self, context):
        layout = self.layout
        row = layout.row(align=True)
        row.prop(self, 'group_mode', expand=True)

    def execute(self, context):

        def set_color(context, p_group_pair):
            idx, p_group = p_group_pair
            t_info = self.get_lookup_extra_info_by_index(context, idx)
            p_weight = math.fabs(t_info['weight'])
            p_group.group_color = vertex_weight_to_color(p_weight)

        p_group_pairs = self.get_current_group_pairs(context)
        if len(p_group_pairs):
            if self.group_mode == 'ACTIVE':
                p_group_pair = self.get_group_pair_by_layer(p_group_pairs, self.group_layer)
                if p_group_pair:
                    set_color(context, p_group_pair)
                    return {'FINISHED'}
            else:
                for p_group_pair in p_group_pairs:
                    set_color(context, p_group_pair)

                return {'FINISHED'}

        return {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)


class ZSBLVG_PT_WeightPresets(bpy.types.Panel):
    bl_label = "Weight Presets Settings"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'

    def draw(self, context):
        layout = self.layout
        addon_prefs = get_prefs()

        row = layout.row(align=True)
        row.menu("ZSBLVG_MT_StoreWeightPresets", text=ZSBLVG_MT_StoreWeightPresets.bl_label)
        row.operator("zsts.add_vertex_weight_preset", text="", icon="ADD")
        row.operator("zsts.add_vertex_weight_preset", text="", icon="REMOVE").remove_active = True

        box = layout.box()

        box.prop(addon_prefs.vgroups_options, 'weight_preset_size')

        col = box.column(align=True)
        for i in range(addon_prefs.vgroups_options.weight_preset_size):
            row = col.split(factor=0.1)
            row.label(text=str(i + 1))
            row.prop(addon_prefs.vgroups_options, 'weight_preset', index=i, text='')


class ZSBLVGObjectListGroup(ZsBlenderObjectPropList, bpy.types.PropertyGroup):
    """
    Group of properties representing
    an item in the zen sets groups for OBJECT
    """
    pass


class ZSBLVGSceneListGroup(ZsBlenderScenePropList, bpy.types.PropertyGroup):
    """
    Group of properties representing
    an item in the zen sets groups for SCENE
    """
    def get_object_groups(self, p_obj: bpy.types.Object):
        return p_obj.vertex_groups

    def get_update_method(self):
        return zen_sets_update_vertex_groups_list


class ZsVertexGroupIndexSync(ZsBlenderGroupIndexSync):
    def get_cls_mgr(self):
        return ZsVGroupLayerManager


vertexGroupsIndexSync = ZsVertexGroupIndexSync()


class ZSBLVG_UL_List(Zs_UL_BaseList, ZsVGroupLayerManager):
    pass


class ZSBLVG_Factory:
    classes = (
        ZSBLVGSceneListGroup,
        ZSBLVGObjectListGroup,
        ZSBLVG_OT_InternalChangeWeightByIndex,
        ZSBLVG_OT_RemoveWeightFromGroup,
        ZSBLVG_OT_AssignVertexWeightToGroupColor,
        ZSBLVG_OT_DisplayWeights,

        ZSBLVG_UL_List,

        ZSBLVG_PT_WeightPresets,
        ZSBLVG_MT_StoreWeightPresets,
        ZSBLVG_OT_StoreAddWeightPreset,
    )

    def get_mgr():
        return ZsVGroupLayerManager

    def get_ui_list():
        return ZSBLVGSceneListGroup

    def get_obj_ui_list():
        return ZSBLVGObjectListGroup
