# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

""" Zen Edge Groups System """
from ..basic_sets import Zs_UL_BaseList
from ..basic_list_sets import ZsListLayerManager
from ..draw_cache import EdgesCacher


class ZsEdgeLayerManager(ZsListLayerManager):

    list_item_prefix = 'Edges'
    id_group = 'edge'
    id_mask = 'ZSEG'
    id_element = 'edge'
    id_display_element = 'edge'
    id_uv_select_mode = 'EDGE'

    """ Edges section """
    @classmethod
    def get_bm_items(self, bm):
        bm.edges.ensure_lookup_table()
        return bm.edges

    @classmethod
    def get_selected_count(self, p_obj):
        me = p_obj.data
        return me.total_edge_sel

    @classmethod
    def get_mesh_select_mode(self):
        # 0 - vert, 1 - edge, 2 - face
        return (False, True, False)

    @classmethod
    def get_cacher(self):
        return EdgesCacher()

    @classmethod
    def get_item_loops(self, p_item):
        return [loop for v in p_item.verts for loop in v.link_loops]

    @classmethod
    def fetch_uv_selections(self, bm):
        uv_edge_selected = set()

        uv_layer = bm.loops.layers.uv.active
        if uv_layer:

            uv_edge_selected = set(
                item.index
                for item in self.get_bm_items(bm)
                if not item.hide and
                all(
                    loop[uv_layer].select and loop.face.select
                    for v in item.verts for loop in v.link_loops)
            )

        return uv_edge_selected

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

        layout = menu.layout

        layout.separator()
        layout.operator('zsts.mark_seams')
        layout.operator('zsts.clear_seams')
        layout.separator()
        layout.operator('zsts.assign_uv_borders_to_group')

    @classmethod
    def execute_DrawTools(cls, tools, context):
        super().execute_DrawTools(tools, context)

        layout = tools.layout

        row = layout.row(align=True)
        row.operator('zsts.mark_seams')
        row.operator('zsts.clear_seams')

        row = layout.row(align=True)
        row.operator('zsts.assign_uv_borders_to_group')


class ZSEG_UL_List(Zs_UL_BaseList, ZsEdgeLayerManager):
    pass


class ZSEG_Factory:
    classes = (
        ZSEG_UL_List,
    )

    def get_mgr():
        return ZsEdgeLayerManager
