# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

""" Zen Edge Unique Groups System """
import bpy
import bmesh

from ..basic_sets import Zs_UL_BaseList
from ..basic_map_sets import ZsMapLayerManager
from ..draw_cache import EdgesUniqueCacher

from ...labels import ZsLabels


class ZsEdgeUniqueLayerManager(ZsMapLayerManager):

    list_item_prefix = 'EdgesP'
    id_group = 'edge_u'
    id_mask = 'ZSUEG'
    id_element = 'edge'
    id_display_element = 'edge'
    id_uv_select_mode = 'EDGE'
    is_unique = True

    """ Edges section """
    @classmethod
    def get_bm_items(self, bm):
        bm.edges.ensure_lookup_table()
        return bm.edges

    @classmethod
    def get_selected_count(self, p_obj):
        me = p_obj.data
        return me.total_edge_sel

    @classmethod
    def get_mesh_select_mode(self):
        # 0 - vert, 1 - edge, 2 - face
        return (False, True, False)

    @classmethod
    def get_item_loops(self, p_item):
        return [loop for v in p_item.verts for loop in v.link_loops]

    @classmethod
    def fetch_uv_selections(self, bm):
        uv_edge_selected = set()

        uv_layer = bm.loops.layers.uv.active
        if uv_layer:

            uv_edge_selected = set(
                item.index
                for item in self.get_bm_items(bm)
                if not item.hide and
                all(
                    loop[uv_layer].select and loop.face.select
                    for v in item.verts for loop in v.link_loops)
            )

        return uv_edge_selected

    @classmethod
    def get_cacher(self):
        return EdgesUniqueCacher()

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

        layout = menu.layout

        layout.separator()
        layout.operator('zsts.mark_seams')
        layout.operator('zsts.clear_seams')

    @classmethod
    def execute_DrawTools(cls, tools, context):
        super().execute_DrawTools(tools, context)

        layout = tools.layout
        layout.operator(ZSUEG_OT_SplitGroupEdges.bl_idname)

        row = layout.row(align=True)
        row.operator('zsts.mark_seams')
        row.operator('zsts.clear_seams')


class ZSUEG_UL_List(Zs_UL_BaseList, ZsEdgeUniqueLayerManager):
    pass


class ZSUEG_OT_SplitGroupEdges(bpy.types.Operator, ZsEdgeUniqueLayerManager):
    """ Import native groups """
    bl_idname = ZsEdgeUniqueLayerManager.list_prop_name() + '.split_group_edges'
    bl_description = ZsLabels.OT_SPLIT_GROUP_EDGES_DESC
    bl_label = ZsLabels.OT_SPLIT_GROUP_EDGES_LABEL
    bl_options = {'REGISTER'}

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', ''),
            ('ALL', 'All', ''),
        ],
        default='ACTIVE')

    @classmethod
    def poll(cls, context):
        return cls.get_list_index(context.scene) != -1

    def _split_group_edges(self, p_obj, p_group):
        bm = self._get_bm(p_obj)

        p_edges = [item for item in self.get_bm_layer_items(p_obj, bm, p_group.layer_name)]
        bmesh.ops.split_edges(bm, edges=p_edges)

    def execute(self, context):
        p_scene = context.scene

        p_obj = context.active_object
        if p_obj:
            if self.group_mode == 'ACTIVE':
                p_group = self._get_scene_group(p_scene)
                if p_group:
                    self._split_group_edges(p_obj, p_group)
            else:
                p_scene_list = self.get_list(p_scene)
                for p_group in p_scene_list:
                    self._split_group_edges(p_obj, p_group)

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False)

        return {'FINISHED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "group_mode", expand=True)


class ZSUEG_Factory:
    classes = (
        ZSUEG_UL_List,

        ZSUEG_OT_SplitGroupEdges
    )

    def get_mgr():
        return ZsEdgeUniqueLayerManager
