# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

""" Zen Face Groups System """
# blender
import bpy
import bmesh

from ...labels import ZsLabels
from ...blender_zen_utils import fix_undo_push_edit_mode

from ..basic_sets import Zs_UL_BaseList
from ..basic_list_sets import ZsListLayerManager
from ..draw_cache import FacesCacher


class ZsFaceLayerManager(ZsListLayerManager):

    list_item_prefix = 'Faces'
    id_group = 'face'
    id_mask = 'ZSFG'
    id_element = 'face'
    id_display_element = 'face'
    id_uv_select_mode = 'FACE'

    """ Faces section """
    @classmethod
    def get_bm_items(self, bm):
        bm.faces.ensure_lookup_table()
        return bm.faces

    @classmethod
    def get_selected_count(self, p_obj):
        me = p_obj.data
        return me.total_face_sel

    @classmethod
    def get_mesh_select_mode(self):
        # 0 - vert, 1 - edge, 2 - face
        return (False, False, True)

    @classmethod
    def get_item_loops(self, p_item):
        return p_item.loops

    @classmethod
    def fetch_uv_selections(self, bm):
        uv_selected = set()

        uv_layer = bm.loops.layers.uv.active
        if uv_layer:

            uv_selected = set(
                item.index
                for item in self.get_bm_items(bm)
                if not item.hide and
                all(
                    (loop[uv_layer].select and loop.face.select)
                    for loop in item.loops)
            )

        return uv_selected

    @classmethod
    def check_uv_select_mode(self) -> bool:
        b_changed = False
        if bpy.context.tool_settings.uv_select_mode not in {self.id_uv_select_mode, 'ISLAND'}:
            bpy.context.tool_settings.uv_select_mode = self.id_uv_select_mode
            b_changed = True
        return b_changed

    @classmethod
    def get_cacher(self):
        return FacesCacher()

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

        # layout = menu.layout

        # layout.separator()
        # layout.operator('zsts.group_linked')  # icon='LINKED'

    @classmethod
    def execute_DrawImport(cls, layout, context, is_menu=False):
        super().execute_DrawImport(layout, context, is_menu)

        if is_menu:
            layout.separator()
        col = layout.column(align=True)
        col.operator(ZSFG_OT_ImportAllFaceMaps.bl_idname)
        col.operator(ZSFG_OT_ImportActiveFaceMap.bl_idname)

    @classmethod
    def execute_DrawTools(cls, tools, context):
        # layout = tools.layout
        # layout.operator('zsts.group_linked')  # icon='LINKED'

        super().execute_DrawTools(tools, context)


class ZSFG_UL_List(Zs_UL_BaseList, ZsFaceLayerManager):
    pass


def _import_face_map(self, p_scene, p_scene_list, p_obj, p_face_map, p_fm_layer, p_bm, select=False):
    i_group_index = self._index_of_group_name(p_scene_list, p_face_map.name)
    if i_group_index == -1:
        s_layer_name = self.create_unique_layer_name()
        p_color = self._gen_new_color(p_scene_list)
        i_group_index = self.add_layer_to_list(p_scene, s_layer_name, p_color)
        p_scene_list[i_group_index].name = p_face_map.name

        i_obj_group_index = self.add_layer_to_list(p_obj, s_layer_name, p_color)
        p_obj_list = self.get_list(p_obj)
        p_obj_list[i_obj_group_index].name = p_face_map.name

    layer = self.ensure_mesh_layer(p_obj, p_bm, p_scene_list[i_group_index].layer_name)

    for face in p_bm.faces:
        face[layer] = face[p_fm_layer] == p_face_map.index

        if select:
            face.select = face[layer]

    self.update_obj_group_count(p_obj, layer.name)

    if select:
        self.set_list_index(p_scene, i_group_index)


class ZSFG_OT_ImportAllFaceMaps(bpy.types.Operator, ZsFaceLayerManager):
    """ Import native groups """
    bl_idname = ZsFaceLayerManager.list_prop_name() + '.import_all_face_maps'
    bl_description = ZsLabels.OT_IMPORT_ALL_FACE_MAPS_DESC
    bl_label = ZsLabels.OT_IMPORT_ALL_FACE_MAPS_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        p_obj = context.active_object
        return p_obj and len(p_obj.face_maps) > 0

    def execute(self, context):
        p_obj = context.active_object
        if p_obj and len(p_obj.face_maps) > 0:
            p_scene = context.scene
            p_scene_list = self.get_list(context.scene)

            bm = self._get_bm(p_obj)

            fm = bm.faces.layers.face_map.verify()

            for g in p_obj.face_maps:
                _import_face_map(self, p_scene, p_scene_list, p_obj, g, fm, bm)

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

            fix_undo_push_edit_mode('Import All Face Maps')

        return {'FINISHED'}


class ZSFG_OT_ImportActiveFaceMap(bpy.types.Operator, ZsFaceLayerManager):
    """ Import native groups """
    bl_idname = ZsFaceLayerManager.list_prop_name() + '.import_active_face_map'
    bl_description = ZsLabels.OT_IMPORT_ACT_FACE_MAP_DESC
    bl_label = ZsLabels.OT_IMPORT_ACT_FACE_MAP_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return context.active_object and context.active_object.face_maps.active

    def execute(self, context):
        p_obj = context.active_object
        if p_obj and p_obj.face_maps.active:
            g = p_obj.face_maps.active

            p_scene = context.scene
            p_scene_list = self.get_list(context.scene)

            bm = self._get_bm(p_obj)

            fm = bm.faces.layers.face_map.verify()

            _import_face_map(self, p_scene, p_scene_list, p_obj, g, fm, bm, select=True)

            bm.select_flush_mode()

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

            fix_undo_push_edit_mode('Import Active Face Map')

        return {'FINISHED'}


class ZSFG_Factory:
    classes = [
        ZSFG_UL_List,
        ZSFG_OT_ImportAllFaceMaps,
        ZSFG_OT_ImportActiveFaceMap
    ]

    def get_mgr():
        return ZsFaceLayerManager
