# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

""" Zen Part Groups System """
# blender
import bpy
import bmesh

from ...labels import ZsLabels
from ...hash_utils import hash32
from ...blender_zen_utils import ZenLocks, fix_undo_push_edit_mode, ZenModeSwitcher

from ..basic_sets import Zs_UL_BaseList
from ..basic_map_sets import ZsMapLayerManager
from ..draw_cache import FaceUniqueCacher


class ZsUniqueFacesLayerManager(ZsMapLayerManager):

    id_group = 'face_u'
    list_item_prefix = 'FacesP'
    id_mask = 'ZSUFG'
    id_element = 'face'
    id_display_element = 'face'
    id_uv_select_mode = 'FACE'
    is_unique = True

    """ Parts section """
    @classmethod
    def get_bm_items(self, bm):
        bm.faces.ensure_lookup_table()
        return bm.faces

    @classmethod
    def get_selected_count(self, p_obj):
        me = p_obj.data
        return me.total_face_sel

    @classmethod
    def get_mesh_select_mode(self):
        # 0 - vert, 1 - edge, 2 - face
        return (False, False, True)

    @classmethod
    def get_item_loops(self, p_item):
        return p_item.loops

    @classmethod
    def fetch_uv_selections(self, bm):
        uv_selected = set()

        uv_layer = bm.loops.layers.uv.active
        if uv_layer:

            uv_selected = set(
                item.index
                for item in self.get_bm_items(bm)
                if not item.hide and
                all(
                    (loop[uv_layer].select and loop.face.select)
                    for loop in item.loops)
            )

        return uv_selected

    @classmethod
    def check_uv_select_mode(self) -> bool:
        b_changed = False
        if bpy.context.tool_settings.uv_select_mode not in {self.id_uv_select_mode, 'ISLAND'}:
            bpy.context.tool_settings.uv_select_mode = self.id_uv_select_mode
            b_changed = True
        return b_changed

    @classmethod
    def get_cacher(self):
        return FaceUniqueCacher()

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

        layout = menu.layout

        layout.separator()
        layout.operator('zsts.group_linked')  # icon='LINKED'
        layout.operator(ZSUFG_OT_AssignMaterialsToGroups.bl_idname)  # icon='MATERIAL'

    @classmethod
    def execute_DrawImport(cls, layout, context, is_menu=False):
        super().execute_DrawImport(layout, context, is_menu)

        if is_menu:
            layout.separator()
        col = layout.column(align=True)
        col.operator(ZSUFG_OT_ImportAllFaceMaps.bl_idname)
        col.operator(ZSUFG_OT_ImportActiveFaceMap.bl_idname)

    @classmethod
    def execute_DrawExport(cls, layout, context, is_menu=False):
        super().execute_DrawExport(layout, context, is_menu)

        if is_menu:
            layout.separator()
        col = layout.column(align=True)
        col.operator(ZSUFG_OT_ExportAllToFaceMaps.bl_idname)
        col.operator(ZSUFG_OT_ExportActiveToFaceMap.bl_idname)

    @classmethod
    def execute_DrawTools(cls, tools, context):
        layout = tools.layout
        layout.operator('zsts.group_linked')

        super().execute_DrawTools(tools, context)

        layout.operator(ZSUFG_OT_AssignMaterialsToGroups.bl_idname)


class ZSUFG_UL_List(Zs_UL_BaseList, ZsUniqueFacesLayerManager):
    pass


def _import_face_map(self, p_scene, p_scene_list, p_obj, p_face_map, p_fm_layer, p_bm, select=False):
    i_group_index = self._index_of_group_name(p_scene_list, p_face_map.name)
    if i_group_index == -1:
        s_layer_name = self.create_unique_layer_name()
        p_color = self._gen_new_color(p_scene_list)
        i_group_index = self.add_layer_to_list(p_scene, s_layer_name, p_color)
        p_scene_list[i_group_index].name = p_face_map.name

        i_obj_group_index = self.add_layer_to_list(p_obj, s_layer_name, p_color)
        p_obj_list = self.get_list(p_obj)
        p_obj_list[i_obj_group_index].name = p_face_map.name
    else:
        self.ensure_group_in_object(p_obj, p_scene_list[i_group_index])

    layer = self.ensure_hash_layer(p_bm)
    p_bytes = hash32(p_scene_list[i_group_index].layer_name)

    for face in p_bm.faces:
        b_is_active = face[p_fm_layer] == p_face_map.index
        if b_is_active:
            face[layer] = p_bytes
        else:
            if face[layer] == p_bytes:
                face[layer] = 0

        if select:
            face.select = b_is_active

    if select:
        self.set_list_index(p_scene, i_group_index)


class ZSUFG_OT_ImportAllFaceMaps(bpy.types.Operator, ZsUniqueFacesLayerManager):
    """ Import native groups """
    bl_idname = ZsUniqueFacesLayerManager.list_prop_name() + '.import_all_face_maps'
    bl_description = ZsLabels.OT_IMPORT_ALL_FACE_MAPS_DESC
    bl_label = ZsLabels.OT_IMPORT_ALL_FACE_MAPS_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        p_obj = context.active_object
        return p_obj and len(p_obj.face_maps) > 0

    def execute(self, context):
        p_obj = context.active_object
        if p_obj and len(p_obj.face_maps) > 0:
            p_scene = context.scene
            p_scene_list = self.get_list(context.scene)

            bm = self._get_bm(p_obj)

            fm = bm.faces.layers.face_map.verify()

            for g in p_obj.face_maps:
                _import_face_map(self, p_scene, p_scene_list, p_obj, g, fm, bm)

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

            self.update_all_obj_groups_count(p_obj)

            fix_undo_push_edit_mode('Import All Face Maps')

        return {'FINISHED'}


class ZSUFG_OT_ImportActiveFaceMap(bpy.types.Operator, ZsUniqueFacesLayerManager):
    """ Import native groups """
    bl_idname = ZsUniqueFacesLayerManager.list_prop_name() + '.import_active_face_map'
    bl_description = ZsLabels.OT_IMPORT_ACT_FACE_MAP_DESC
    bl_label = ZsLabels.OT_IMPORT_ACT_FACE_MAP_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return context.active_object and context.active_object.face_maps.active

    def execute(self, context):
        p_obj = context.active_object
        if p_obj and p_obj.face_maps.active:
            g = p_obj.face_maps.active

            p_scene = context.scene
            p_scene_list = self.get_list(context.scene)

            bm = self._get_bm(p_obj)

            fm = bm.faces.layers.face_map.verify()

            _import_face_map(self, p_scene, p_scene_list, p_obj, g, fm, bm, select=True)

            bm.select_flush_mode()

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

            self.update_all_obj_groups_count(p_obj)

            fix_undo_push_edit_mode('Import Active Face Maps')

        return {'FINISHED'}


def _export_zen_group_to_native(p_cls_mgr, p_obj, p_group, select=False):
    bm = p_cls_mgr._get_bm(p_obj)
    hash_layer = p_cls_mgr.get_hash_layer(bm)
    if hash_layer:
        p_bytes = hash32(p_group.layer_name)
        f_group = p_obj.face_maps[p_group.name] if p_group.name in p_obj.face_maps else p_obj.face_maps.new(name=p_group.name)
        t_faces = []
        t_wrong_faces = []

        fm_layer = bm.faces.layers.face_map.verify()

        b_need_flush = False

        for f in bm.faces:
            b_is_exported = f[hash_layer] == p_bytes
            if b_is_exported:
                t_faces.append(f.index)
            else:
                if f[fm_layer] == f_group.index:
                    t_wrong_faces.append(f.index)

            if select:
                f.select = b_is_exported
                b_need_flush = True

        mode_switcher = ZenModeSwitcher()
        # we must remove faces that belonged previously, because Blender always append
        f_group.remove(t_wrong_faces)
        f_group.add(t_faces)
        mode_switcher.return_to_edit_mode()

        if b_need_flush:
            bm = p_cls_mgr._get_bm(p_obj)
            bm.select_flush_mode()


class ZSUFG_OT_ExportActiveToFaceMap(bpy.types.Operator, ZsUniqueFacesLayerManager):
    """ Import native groups """
    bl_idname = ZsUniqueFacesLayerManager.list_prop_name() + '.export_active_to_face_map'
    bl_description = ZsLabels.OT_EXPORT_ACT_TO_FACE_MAP_DESC
    bl_label = ZsLabels.OT_EXPORT_ACT_TO_FACE_MAP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return cls.get_current_list_index(context) != -1

    def execute(self, context):
        p_obj = context.active_object

        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            _export_zen_group_to_native(self, p_obj, p_group_pair[1], select=True)

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
            ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}


class ZSUFG_OT_ExportAllToFaceMaps(bpy.types.Operator, ZsUniqueFacesLayerManager):
    """ Import native groups """
    bl_idname = ZsUniqueFacesLayerManager.list_prop_name() + '.export_all_to_face_maps'
    bl_description = ZsLabels.OT_EXPORT_ALL_TO_FACE_MAPS_DESC
    bl_label = ZsLabels.OT_EXPORT_ALL_TO_FACE_MAPS_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.active_object and len(cls.get_current_group_pairs(context)) > 0

    def execute(self, context):
        p_obj = context.active_object
        if p_obj:
            p_group_pairs = self.get_current_group_pairs(context)

            for p_group_pair in p_group_pairs:
                _export_zen_group_to_native(self, p_obj, p_group_pair[1])

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
            ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}


class ZSUFG_OT_AssignMaterialsToGroups(bpy.types.Operator, ZsUniqueFacesLayerManager):
    """ Import native groups """
    bl_idname = ZsUniqueFacesLayerManager.list_prop_name() + '.assign_materials_to_groups'
    bl_description = ZsLabels.OT_ASSIGN_MATERIAL_TO_GROUP_DESC
    bl_label = ZsLabels.OT_ASSIGN_MATERIAL_TO_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        p_scene = context.scene
        p_list = cls.get_list(p_scene)
        return p_list and len(p_list) > 0

    def execute(self, context):
        p_group_pairs = self.get_current_group_pairs(context)

        if len(p_group_pairs):

            def_mat_name = 'Material.Default'

            def_mat = None
            if def_mat_name not in bpy.data.materials.keys():
                def_mat = bpy.data.materials.new(name=def_mat_name)
                material_color = (1, 1, 1, 1)
                def_mat.diffuse_color = material_color
                def_mat.use_nodes = True
                def_mat.use_fake_user = True
                pr_bsdf_node = def_mat.node_tree.nodes.get("Principled BSDF")
                if pr_bsdf_node:
                    pr_bsdf_node.inputs['Base Color'].default_value = material_color
            else:
                def_mat = bpy.data.materials.get(def_mat_name)

            update_objs = set()

            for p_obj in context.objects_in_mode:
                id_def_mat = p_obj.data.materials.find(def_mat_name)

                if id_def_mat == -1:
                    p_obj.data.materials.append(def_mat)
                    id_def_mat = len(p_obj.data.materials) - 1

                bm = self._get_bm(p_obj)
                nonlayered_items = self.get_bm_nonlayered_items(p_obj, bm, p_group_pairs)
                if len(nonlayered_items):
                    for item in nonlayered_items:
                        item.material_index = id_def_mat
                    update_objs.add(p_obj)

            for _, p_group in p_group_pairs:
                material_name = 'Material.' + p_group.name
                material_color = (p_group.group_color.r, p_group.group_color.g, p_group.group_color.b, 1)
                new_mat = bpy.data.materials.new(name=material_name) if material_name not in bpy.data.materials.keys() else bpy.data.materials.get(material_name)
                new_mat.diffuse_color = material_color
                new_mat.use_nodes = True
                new_mat.use_fake_user = True
                pr_bsdf_node = new_mat.node_tree.nodes.get("Principled BSDF")
                if pr_bsdf_node:
                    pr_bsdf_node.inputs['Base Color'].default_value = material_color

                for p_obj in context.objects_in_mode:
                    id_mat = p_obj.data.materials.find(material_name)

                    if id_mat == -1:
                        p_obj.data.materials.append(new_mat)
                        id_mat = len(p_obj.data.materials) - 1

                    bm = self._get_bm(p_obj)
                    items = self.get_bm_layer_items(p_obj, bm, p_group.layer_name)
                    if len(items):
                        for item in items:
                            item.material_index = id_mat
                        update_objs.add(p_obj)

            for p_obj in update_objs:
                me = p_obj.data
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
                ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}


class ZSUFG_Factory:
    classes = (
        ZSUFG_UL_List,
        ZSUFG_OT_ImportAllFaceMaps,
        ZSUFG_OT_ImportActiveFaceMap,
        ZSUFG_OT_ExportAllToFaceMaps,
        ZSUFG_OT_ExportActiveToFaceMap,
        ZSUFG_OT_AssignMaterialsToGroups,
    )

    def get_mgr():
        return ZsUniqueFacesLayerManager
