# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

""" Zen Vert Groups System """
# blender
import bpy
import bmesh

from ...labels import ZsLabels
from ...blender_zen_utils import ZenLocks

from ..basic_sets import Zs_UL_BaseList
from ..basic_list_sets import ZsListLayerManager
from ..draw_cache import VertsCacher
from ..export_utils import export_zen_group_to_vertex_group


class ZsVertLayerManager(ZsListLayerManager):

    list_item_prefix = 'Verts'
    id_group = 'vert'
    id_mask = 'ZSVG'
    id_element = 'vert'
    id_display_element = 'vert'
    id_uv_select_mode = 'VERTEX'

    """ Verts section """
    @classmethod
    def get_bm_items(self, bm):
        bm.verts.ensure_lookup_table()
        return bm.verts

    @classmethod
    def get_selected_count(self, p_obj):
        me = p_obj.data
        return me.total_vert_sel

    @classmethod
    def get_mesh_select_mode(self):
        # 0 - vert, 1 - edge, 2 - face
        return (True, False, False)

    @classmethod
    def get_item_loops(self, p_item):
        return p_item.link_loops

    @classmethod
    def fetch_uv_selections(self, bm):
        uv_selected = set()

        uv_layer = bm.loops.layers.uv.active
        if uv_layer:

            uv_selected = set(
                item.index
                for item in self.get_bm_items(bm)
                if not item.hide and
                all(
                    loop[uv_layer].select and loop.face.select
                    for loop in item.link_loops)
            )

        return uv_selected

    @classmethod
    def get_cacher(self):
        return VertsCacher()

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

    @classmethod
    def execute_DrawImport(cls, layout, context, is_menu=False):
        super().execute_DrawImport(layout, context, is_menu)

        if is_menu:
            layout.separator()
        col = layout.column(align=True)
        col.operator(ZSVG_OT_ImportAllVertexGroups.bl_idname)
        col.operator(ZSVG_OT_ImportActiveVertexGroup.bl_idname)

    @classmethod
    def execute_DrawExport(cls, layout, context, is_menu=False):
        super().execute_DrawExport(layout, context, is_menu)

        if is_menu:
            layout.separator()
        col = layout.column(align=True)
        col.operator(ZSVG_OT_ExportAllToVertexGroups.bl_idname)
        col.operator(ZSVG_OT_ExportActiveToVertexGroup.bl_idname)

    @classmethod
    def execute_DrawTools(cls, tools, context):
        super().execute_DrawTools(tools, context)


class ZSVG_UL_List(Zs_UL_BaseList, ZsVertLayerManager):
    pass


class ZSVG_OT_ImportAllVertexGroups(bpy.types.Operator, ZsVertLayerManager):
    """ Import vertex groups """
    bl_idname = ZsVertLayerManager.list_prop_name() + '.import_all_vertex_groups'
    bl_description = ZsLabels.OT_IMPORT_ALL_VERTEX_GROUPS_DESC
    bl_label = ZsLabels.OT_IMPORT_ALL_VERTEX_GROUPS_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        p_obj = context.active_object
        return p_obj and len(p_obj.vertex_groups) > 0

    def execute(self, context):
        p_scene = context.scene
        p_group_pairs = self.get_current_group_pairs(context)
        p_scene_list = self.get_list(p_scene)

        groups_lookup = {}

        p_obj = context.active_object
        if p_obj and len(p_obj.vertex_groups) > 0:
            for g in p_obj.vertex_groups:
                p_group_pair = self.get_group_pair_by_name(p_group_pairs, g.name)
                if not p_group_pair:
                    s_layer_name = self.create_unique_layer_name()
                    p_color = self._gen_new_color(p_scene_list)
                    i_group_index = self.add_layer_to_list(p_scene, s_layer_name, p_color)
                    p_scene_list[i_group_index].name = g.name

                    i_obj_group_index = self.add_layer_to_list(p_obj, s_layer_name, p_color)
                    p_obj_list = self.get_list(p_obj)
                    p_obj_list[i_obj_group_index].name = g.name

                    p_group_pair = (i_group_index, p_scene_list[i_group_index])

                groups_lookup[g.index] = p_group_pair[1].layer_name

            p_obj.update_from_editmode()

            bm = self._get_bm(p_obj)
            me = p_obj.data
            bm.verts.ensure_lookup_table()

            for vert in me.vertices:
                for g in p_obj.vertex_groups:
                    layer = self.ensure_mesh_layer(p_obj, bm, groups_lookup[g.index])
                    b_active = False
                    for elem in vert.groups:
                        if elem.group == g.index:
                            b_active = True
                            break
                    bm.verts[vert.index][layer] = b_active

            for layerName in groups_lookup.values():
                self.update_obj_group_count(p_obj, layerName)

            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

        return {'FINISHED'}


class ZSVG_OT_ImportActiveVertexGroup(bpy.types.Operator, ZsVertLayerManager):
    """ Import vertex groups """
    bl_idname = ZsVertLayerManager.list_prop_name() + '.import_active_vertex_group'
    bl_description = ZsLabels.OT_IMPORT_ACT_VERTEX_GROUP_DESC
    bl_label = ZsLabels.OT_IMPORT_ACT_VERTEX_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.active_object and context.active_object.vertex_groups.active

    def execute(self, context):
        p_scene = context.scene
        p_scene_list = self.get_list(context.scene)

        self.set_mesh_select_mode(context)

        p_obj = context.active_object
        if p_obj and p_obj.vertex_groups.active:
            g = p_obj.vertex_groups.active
            i_group_index = self._index_of_group_name(p_scene_list, g.name)
            if i_group_index == -1:
                s_layer_name = self.create_unique_layer_name()
                p_color = self._gen_new_color(p_scene_list)
                i_group_index = self.add_layer_to_list(p_scene, s_layer_name, p_color)
                p_scene_list[i_group_index].name = g.name

                i_obj_group_index = self.add_layer_to_list(p_obj, s_layer_name, p_color)
                p_obj_list = self.get_list(p_obj)
                p_obj_list[i_obj_group_index].name = g.name
            else:
                self.ensure_group_in_object(p_obj, p_scene_list[i_group_index])

            s_layer_name = p_scene_list[i_group_index].layer_name

            p_obj.update_from_editmode()

            bm = self._get_bm(p_obj)
            me = p_obj.data
            bm.verts.ensure_lookup_table()
            layer = self.ensure_mesh_layer(p_obj, bm, s_layer_name)
            for vert in bm.verts:
                b_active = False
                for elem in me.vertices[vert.index].groups:
                    if elem.group == g.index:
                        b_active = True
                        break
                vert[layer] = b_active
                vert.select = b_active

            self.update_obj_group_count(p_obj, s_layer_name)

            self.set_list_index(p_scene, i_group_index)

            bm.select_flush_mode()

            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)

        return {'FINISHED'}


class ZSVG_OT_ExportActiveToVertexGroup(bpy.types.Operator, ZsVertLayerManager):
    """ Import native groups """
    bl_idname = ZsVertLayerManager.list_prop_name() + '.export_active_to_vertex_group'
    bl_description = ZsLabels.OT_EXPORT_ACT_TO_VERTEX_GROUP_DESC
    bl_label = ZsLabels.OT_EXPORT_ACT_TO_VERTEX_GROUP_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return context.active_object and cls.get_current_list_index(context) != -1

    def execute(self, context):
        p_obj = context.active_object
        if p_obj:
            p_group_pair = self.get_current_group_pair(context)
            if p_group_pair:
                export_zen_group_to_vertex_group(self, p_obj, p_group_pair[1], select=True)

                me = p_obj.data
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
                ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}


class ZSVG_OT_ExportAllToVertexGroups(bpy.types.Operator, ZsVertLayerManager):
    """ Import native groups """
    bl_idname = ZsVertLayerManager.list_prop_name() + '.export_all_to_vertex_groups'
    bl_description = ZsLabels.OT_EXPORT_ALL_TO_VERTEX_GROUPS_DESC
    bl_label = ZsLabels.OT_EXPORT_ALL_TO_VERTEX_GROUPS_LABEL
    bl_options = {'REGISTER'}

    @classmethod
    def poll(cls, context):
        return context.active_object and len(cls.get_current_group_pairs(context))

    def execute(self, context):
        p_obj = context.active_object
        if p_obj:
            p_group_pairs = self.get_current_group_pairs(context)
            for p_group_pair in p_group_pairs:
                export_zen_group_to_vertex_group(self, p_obj, p_group_pair[1])

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
            ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}


class ZSVG_Factory:
    classes = (
        ZSVG_UL_List,
        ZSVG_OT_ImportAllVertexGroups,
        ZSVG_OT_ImportActiveVertexGroup,
        ZSVG_OT_ExportAllToVertexGroups,
        ZSVG_OT_ExportActiveToVertexGroup,
    )

    def get_mgr():
        return ZsVertLayerManager
