# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

""" Zen Vert Unique Groups System """
# blender
import bpy
import bmesh

from ...labels import ZsLabels
from ...blender_zen_utils import ZenLocks

from ..basic_sets import Zs_UL_BaseList
from ..basic_map_sets import ZsMapLayerManager
from ..draw_cache import VertsUniqueCacher
from ..export_utils import export_zen_group_to_vertex_group


class ZsUniqueVertLayerManager(ZsMapLayerManager):

    list_item_prefix = 'VertsP'
    id_group = 'vert_u'
    id_mask = 'ZSUVG'
    id_element = 'vert'
    id_display_element = 'vert'
    id_uv_select_mode = 'VERTEX'
    is_unique = True

    """ Verts section """
    @classmethod
    def get_bm_items(self, bm):
        bm.verts.ensure_lookup_table()
        return bm.verts

    @classmethod
    def get_selected_count(self, p_obj):
        me = p_obj.data
        return me.total_vert_sel

    @classmethod
    def get_mesh_select_mode(self):
        # 0 - vert, 1 - edge, 2 - face
        return (True, False, False)

    @classmethod
    def get_item_loops(self, p_item):
        return p_item.link_loops

    @classmethod
    def fetch_uv_selections(self, bm):
        uv_selected = set()

        uv_layer = bm.loops.layers.uv.active
        if uv_layer:

            uv_selected = set(
                item.index
                for item in self.get_bm_items(bm)
                if not item.hide and
                all(
                    loop[uv_layer].select and loop.face.select
                    for loop in item.link_loops)
            )

        return uv_selected

    @classmethod
    def get_cacher(self):
        return VertsUniqueCacher()

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

    @classmethod
    def execute_DrawTools(cls, tools, context):
        super().execute_DrawTools(tools, context)

    @classmethod
    def execute_DrawImport(cls, layout, context, is_menu=False):
        super().execute_DrawImport(layout, context, is_menu)

    @classmethod
    def execute_DrawExport(cls, layout, context, is_menu=False):
        super().execute_DrawExport(layout, context, is_menu)

        if is_menu:
            layout.separator()
        col = layout.column(align=True)
        col.operator(ZSUVG_OT_ExportAllToVertexGroups.bl_idname)
        col.operator(ZSUVG_OT_ExportActiveToVertexGroup.bl_idname)


class ZSUVG_UL_List(Zs_UL_BaseList, ZsUniqueVertLayerManager):
    pass


class ZSUVG_OT_ExportActiveToVertexGroup(bpy.types.Operator, ZsUniqueVertLayerManager):
    """ Import native groups """
    bl_idname = ZsUniqueVertLayerManager.list_prop_name() + '.export_active_to_vertex_group'
    bl_description = ZsLabels.OT_EXPORT_ACT_TO_VERTEX_GROUP_DESC
    bl_label = ZsLabels.OT_EXPORT_ACT_TO_VERTEX_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.active_object and cls.get_current_list_index(context) != -1

    def execute(self, context):
        p_obj = context.active_object
        if p_obj:
            p_group_pair = self.get_current_group_pair(context)
            if p_group_pair:
                export_zen_group_to_vertex_group(self, p_obj, p_group_pair[1], select=True)

                me = p_obj.data
                bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
                ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}


class ZSUVG_OT_ExportAllToVertexGroups(bpy.types.Operator, ZsUniqueVertLayerManager):
    """ Import native groups """
    bl_idname = ZsUniqueVertLayerManager.list_prop_name() + '.export_all_to_vertex_groups'
    bl_description = ZsLabels.OT_EXPORT_ALL_TO_VERTEX_GROUPS_DESC
    bl_label = ZsLabels.OT_EXPORT_ALL_TO_VERTEX_GROUPS_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.active_object and len(cls.get_current_group_pairs(context))

    def execute(self, context):
        p_obj = context.active_object
        if p_obj:
            p_group_pairs = self.get_current_group_pairs(context)
            for p_group_pair in p_group_pairs:
                export_zen_group_to_vertex_group(self, p_obj, p_group_pair[1])

            me = p_obj.data
            bmesh.update_edit_mesh(me, loop_triangles=False, destructive=False)
            ZenLocks.lock_depsgraph_update_one()

        return {'FINISHED'}


class ZSUVG_Factory:
    classes = (
        ZSUVG_UL_List,
        ZSUVG_OT_ExportAllToVertexGroups,
        ZSUVG_OT_ExportActiveToVertexGroup
    )

    def get_mgr():
        return ZsUniqueVertLayerManager
