# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

import bpy

import uuid
import mathutils

from typing import Tuple

from .collection_prop_set import (
    ZsExcludeCollectionProperty,
    ZsHideViewportCollectionProperty,
    ZsHideSelectCollectionProperty,
    ZsDisableViewportCollectionProperty,
    ZsHideRenderCollectionProperty
)

from ..draw_sets import is_draw_active
from ..smart_rename import get_last_selected_smart_group, set_last_selected_smart_group

from ...preferences import (
    get_prefs, get_annotated_prop,
    ZsOperatorOptions, ZsInstanceCollectionToParent, ZsInstanceParentToCollection)
from ...labels import ZsLabels
from ...blender_zen_utils import ZsOperatorAttrs, ZenStates, ZenPolls, draw_last_operator_properties, get_mimic_operator_layout, get_command_props

from ..factories import get_sets_object_mgr


def _poll_context_has_selected_group(context):
    p_scene = context.scene
    p_cls_mgr = get_sets_object_mgr(p_scene)
    if p_cls_mgr:
        if ZenPolls.is_object_and_collection_mode(context):
            p_collection = p_cls_mgr.get_active_collection(context)
            return (p_collection is not None and
                    p_collection != context.scene.collection)
        else:
            return p_cls_mgr.get_current_list_index(context) != -1
    return False


def _poll_context_by_current_groups_len(context):
    p_scene = context.scene
    p_cls_mgr = get_sets_object_mgr(p_scene)
    if p_cls_mgr:
        if ZenPolls.is_object_and_collection_mode(context):
            return len(p_cls_mgr.get_list(p_scene)) > 1
        else:
            return len(p_cls_mgr.get_list(p_scene)) != 0
    return False


class ZSTO_OT_NewItemMove(bpy.types.Operator):
    bl_idname = 'zsto.new_group_move'
    bl_description = ZsLabels.OT_NEW_ITEM_DESC + ' (Move objects)'
    bl_label = ZsLabels.OT_NEW_ITEM_LABEL + ' (Move)'
    bl_options = {'REGISTER', 'UNDO'}

    move = True

    new_group_name: bpy.props.StringProperty(
        name='New Group Name',
        description='New Group Name',
        get=get_last_selected_smart_group,
        set=set_last_selected_smart_group,
        options={'SKIP_SAVE'}
    )

    def draw(self, context: bpy.types.Context):
        layout = self.layout
        if ZenPolls.is_collection_mode(context):
            draw_last_operator_properties(context, 'zsto.new_group', layout, whitelist_props={'keep_nested'}, factor=0.4)
        get_mimic_operator_layout(self, layout, 'new_group_name', factor=0.4)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_NewGroup(self, context)
        return {'CANCELLED'}


class ZSTO_OT_NewItemLink(bpy.types.Operator):
    bl_idname = 'zsto.new_group_link'
    bl_description = ZsLabels.OT_NEW_ITEM_DESC + ' (Link objects)'
    bl_label = ZsLabels.OT_NEW_ITEM_LABEL + ' (Link)'
    bl_options = {'REGISTER', 'UNDO'}

    move = False

    new_group_name: bpy.props.StringProperty(
        name='New Group Name',
        description='New Group Name',
        get=get_last_selected_smart_group,
        set=set_last_selected_smart_group,
        options={'SKIP_SAVE'}
    )

    def draw(self, context: bpy.types.Context):
        layout = self.layout
        if ZenPolls.is_collection_mode(context):
            draw_last_operator_properties(context, 'zsto.new_group', layout, whitelist_props={'keep_nested'}, factor=0.4)
        get_mimic_operator_layout(self, layout, 'new_group_name', factor=0.4)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_NewGroup(self, context)
        else:
            return {'CANCELLED'}


class ZSTO_OT_NewItem(bpy.types.Operator):
    """ Add new group with choice """
    bl_idname = 'zsto.new_group'
    bl_label = ZsLabels.OT_NEW_ITEM_LABEL
    bl_description = 'Shortcut: Alt+F1 (Move) | Ctrl+F1 (Link)'
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def description(cls, context, properties) -> str:
        if ZenPolls.is_collection_mode(context):
            return cls.bl_description
        else:
            return ZsLabels.OT_NEW_ITEM_DESC

    move: bpy.props.BoolProperty(
        name=ZsLabels.PROP_MOVE_OBJECTS_NAME,
        description=ZsLabels.PROP_MOVE_OBJECTS_DESC,
        default=True)

    keep_nested: bpy.props.BoolProperty(
        name=ZsLabels.PROP_KEEP_NESTED_NAME,
        description=ZsLabels.PROP_KEEP_NESTED_DESC,
        default=True)

    new_group_name: bpy.props.StringProperty(
        name='New Group Name',
        description='New Group Name',
        get=get_last_selected_smart_group,
        set=set_last_selected_smart_group,
        options={'SKIP_SAVE'}
    )

    def draw_ex(self, layout: bpy.types.UILayout, context: bpy.types.Context):
        layout.use_property_split = True

        if ZenPolls.is_collection_mode(context):
            layout.prop(self, 'move')
            layout.prop(self, 'keep_nested')
        layout.prop(self, 'new_group_name')

    def draw(self, context: bpy.types.Context):
        self.draw_ex(self.layout, context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_NewGroup(self, context)
        return {'CANCELLED'}


class ZSTO_OT_DeleteGroupOnly(bpy.types.Operator):
    """Delete the selected item from the list."""
    bl_idname = 'zsto.del_group_only'
    bl_description = 'Delete selected Collection and shift all children to its Parent'
    bl_label = ZsLabels.OT_DEL_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_DeleteGroup(self, context)
        return {'CANCELLED'}


class ZSTO_OT_DeleteHierarchy(bpy.types.Operator):
    """ Delete Hierarchy """
    bl_idname = 'zsto.delete_hierarchy'
    bl_description = 'Delete selected Collection and all its children'
    bl_label = 'Delete Hierarchy'
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_DeleteHierarchy(self, context)
        return {'CANCELLED'}


class ZSTO_OT_DeleteGroup(bpy.types.Operator):
    """Delete the selected item from the list."""
    bl_idname = 'zsto.del_group'
    bl_description = ZsLabels.OT_DEL_ITEM_DESC
    bl_label = ZsLabels.OT_DEL_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    hierarchy: bpy.props.BoolProperty(
        name='Hierarchy',
        description='Delete selected Collection and all its children',
        default=False)

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            if self.hierarchy:
                return p_cls_mgr.execute_DeleteHierarchy(self, context)
            else:
                return p_cls_mgr.execute_DeleteGroup(self, context)
        return {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)


class ZSTO_OT_DeleteEmptyGroups(bpy.types.Operator):
    """ Delete empty groups """
    bl_idname = 'zsto.delete_empty_groups'
    bl_description = ZsLabels.OT_DELETE_UNLINK_EMPTY_GROUP_DESC
    bl_label = ZsLabels.OT_DELETE_EMPTY_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    without_nested_objects: bpy.props.BoolProperty(
        name=ZsLabels.PROP_WITHOUT_NESTED_OBJECTS_NAME,
        description=ZsLabels.PROP_WITHOUT_NESTED_OBJECTS_DESC,
        default=True)

    visible_only: bpy.props.BoolProperty(
        name='Visible only',
        default=True)

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_DeleteEmptyGroups(self, context)
        return {'CANCELLED'}


class ZSTO_OT_DeleteGroupsCombo(bpy.types.Operator):
    """ Delete all groups or delete empty """
    bl_idname = 'zsto.delete_groups_combo'
    bl_description = ZsLabels.OT_DELETE_UNLINK_GROUPS_COMBO_DESC
    bl_label = ZsLabels.OT_DELETE_GROUPS_COMBO_LABEL
    bl_options = {'REGISTER'}

    delete_mode: bpy.props.EnumProperty(name="Delete Mode",
                                        items=[
                                            ('EMPTY', 'Empty Groups', ''),
                                            ('ALL', 'All Groups', ''),
                                        ],
                                        default='EMPTY')

    without_nested_objects: bpy.props.BoolProperty(
        name=ZsLabels.PROP_WITHOUT_NESTED_OBJECTS_NAME,
        description=ZsLabels.PROP_WITHOUT_NESTED_OBJECTS_DESC,
        default=True)

    visible_only: bpy.props.BoolProperty(
        name='Visible only',
        default=True)

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def execute(self, context):
        return bpy.ops.zsto.delete_groups_internal(
            'INVOKE_DEFAULT',
            delete_mode=self.delete_mode,
            without_nested_objects=self.without_nested_objects,
            visible_only=self.visible_only)


class ZSTO_OT_DeleteGroupsInternal(bpy.types.Operator):
    """ Internal delete combo groups """
    bl_idname = 'zsto.delete_groups_internal'
    bl_label = 'Choose what to Delete (Unlink) and press OK'
    bl_options = {'REGISTER'}  # DO NOT USE UNDO HERE !

    delete_mode: bpy.props.EnumProperty(name="Delete Mode",
                                        items=[
                                            ('EMPTY', 'Empty Groups', ''),
                                            ('ALL', 'All Groups', ''),
                                        ],
                                        default='EMPTY')

    without_nested_objects: bpy.props.BoolProperty(
        name=ZsLabels.PROP_WITHOUT_NESTED_OBJECTS_NAME,
        description=ZsLabels.PROP_WITHOUT_NESTED_OBJECTS_DESC,
        default=True)

    visible_only: bpy.props.BoolProperty(
        name='Visible only',
        default=True)

    def execute(self, context):
        op_delete = context.window_manager.operator_properties_last('zsto.delete_groups_combo')
        if op_delete:
            op_delete.delete_mode = self.delete_mode
            op_delete.without_nested_objects = self.without_nested_objects
            op_delete.visible_only = self.visible_only

        res = {'CANCELLED'}

        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            if self.delete_mode == 'ALL':
                res = p_cls_mgr.execute_DeleteAllGroups(self, context)
            else:
                if ZSTO_OT_DeleteEmptyGroups.poll(context):
                    res = p_cls_mgr.execute_DeleteEmptyGroups(self, context)

            if 'CANCELLED' not in res:
                bpy.ops.ed.undo_push(message='Delete ' + self.delete_mode)

        return {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "delete_mode", expand=True)
        if ZenPolls.is_collection_mode(context):
            if self.delete_mode == 'EMPTY':
                layout.prop(self, "without_nested_objects")
            layout.prop(self, "visible_only")


class ZSTO_OT_Assign(bpy.types.Operator):
    bl_idname = 'zsto.assign_to_group'
    bl_label = ZsLabels.OT_ASSIGN_ITEM_LABEL
    bl_description = ZsLabels.OT_ASSIGN_ITEM_DESC
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return (
            ZenPolls.is_object_and_simple_mode(context) and
            _poll_context_has_selected_group(context))

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        return p_cls_mgr.execute_AssignToGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTO_OT_Append(bpy.types.Operator):
    bl_idname = 'zsto.append_to_group'
    bl_label = ZsLabels.OT_APPEND_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    move: bpy.props.BoolProperty(
        name=ZsLabels.PROP_MOVE_OBJECTS_NAME,
        description=ZsLabels.PROP_MOVE_OBJECTS_DESC,
        default=True)
    keep_nested: bpy.props.BoolProperty(
        name=ZsLabels.PROP_KEEP_NESTED_NAME,
        description=ZsLabels.PROP_KEEP_NESTED_DESC,
        default=True)

    @classmethod
    def description(cls, context, properties):
        if ZenPolls.is_collection_mode(context):
            return ZsLabels.OT_APPEND_ITEM_OBJ_DESC
        else:
            return ZsLabels.OT_APPEND_ITEM_DESC

    @classmethod
    def poll(cls, context):
        if ZenPolls.is_collection_mode(context):
            return len(context.selected_objects) != 0
        else:
            return (
                _poll_context_has_selected_group(context) and
                len(context.selected_objects) != 0)

    def draw(self, context):
        layout = self.layout
        layout.use_property_split = True

        if ZenPolls.is_collection_mode(context):
            layout.prop(self, 'move')
            layout.prop(self, 'keep_nested')

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_AppendToGroup(self, context)
        return {'CANCELLED'}


class ZSTO_OT_RemoveSelectionFromGroup(bpy.types.Operator):
    """ Remove mesh selected items from group """
    bl_idname = 'zsto.remove_from_group'
    bl_label = ZsLabels.OT_REMOVE_SEL_ITEM_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    keep_nested: bpy.props.BoolProperty(
        name=ZsLabels.PROP_KEEP_NESTED_NAME,
        description=ZsLabels.PROP_KEEP_NESTED_DESC,
        default=True)

    mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active Group', ZsLabels.OT_REMOVE_SEL_ITEM_OBJ_DESC),
            ('ALL', 'All Groups', ZsLabels.OT_REMOVE_SEL_ITEM_OBJ_ALL_DESC),
        ],
        options={'HIDDEN', 'SKIP_SAVE'},
        default='ACTIVE')

    @classmethod
    def description(cls, context, properties):
        return (
            ZsLabels.OT_REMOVE_SEL_ITEM_OBJ_DESC
            if properties.mode == 'ACTIVE'
            else ZsLabels.OT_REMOVE_SEL_ITEM_OBJ_ALL_DESC)

    def draw(self, context):
        layout = self.layout
        layout.use_property_decorate = True
        layout.use_property_split = True
        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))

        if ZenPolls.is_collection_mode(context):
            if self.mode != 'ALL':
                layout.prop(self, 'keep_nested')

    @classmethod
    def poll(cls, context):
        return len(context.selected_objects) != 0

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_RemoveSelectionFromGroup(self, context)
        return {'CANCELLED'}


class ZSTO_OT_DuplicateCollection(bpy.types.Operator):
    """ Duplicate Collection """
    bl_idname = 'zsto.duplicate_collection'
    bl_description = 'Duplicate Collection and select all its objects'
    bl_label = 'Duplicate Collection'
    bl_options = {'REGISTER', 'UNDO'}

    linked = False

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_DuplicateCollection(self, context)
        return {'CANCELLED'}


class ZSTO_OT_DuplicateCollectionLinked(bpy.types.Operator):
    """ Duplicate Collection Linked"""
    bl_idname = 'zsto.duplicate_collection_linked'
    bl_description = 'Duplicate Collection Linked and select all its objects'
    bl_label = 'Duplicate Collection Linked'
    bl_options = {'REGISTER', 'UNDO'}

    linked = True

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_DuplicateCollection(self, context)
        return {'CANCELLED'}


class ZSTO_OT_ConvertObjectToCollection(bpy.types.Operator, ZsInstanceParentToCollection):
    """ Convert Parent Object To Collection """
    bl_idname = 'zsto.convert_object_to_collection'
    bl_description = ZsLabels.OT_CONVERT_PARENT_TO_COL_DESC
    bl_label = ZsLabels.OT_CONVERT_PARENT_TO_COL_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    move = True

    @classmethod
    def poll(cls, context):
        # Do not change because operator properties are not shown
        return True

    @classmethod
    def fix_poll(cls, context):
        p_obj = context.active_object
        return p_obj is not None and len(p_obj.users_collection) != 0 and len(p_obj.children) != 0

    def execute(self, context):
        if self.fix_poll(context):
            p_cls_mgr = get_sets_object_mgr(context.scene)
            if p_cls_mgr:
                return p_cls_mgr.execute_ConvertObjectToCollection(self, context)

        self.report({'WARNING'}, 'Can not convert! Check params!')
        return {'CANCELLED'}


class ZSTO_OT_ConvertCollectionToParentObject(bpy.types.Operator, ZsInstanceCollectionToParent):
    """ Convert Collection To Parent Object """
    bl_idname = 'zsto.convert_collection_to_object'
    bl_description = ZsLabels.OT_CONVERT_COL_TO_PARENT_DESC
    bl_label = ZsLabels.OT_CONVERT_COL_TO_PARENT_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    empty_display_size: ZsOperatorOptions.get_empty_display_size(default=1.0)

    empty_display_type: ZsOperatorOptions.get_empty_display_type(default='PLAIN_AXES')

    @classmethod
    def poll(cls, context):
        # Do not change because operator properties are not shown
        return True

    @classmethod
    def fix_poll(cls, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr is None:
            return False

        p_collection = p_cls_mgr.get_active_collection(context)
        if p_collection is None:
            return False
        if (len(p_collection.children) == 0 and
                not any((p_obj.parent is None and p_obj.type != 'EMPTY')
                        for p_obj in p_collection.objects)):
            return False

        return True

    def execute(self, context):
        if self.fix_poll(context):
            p_cls_mgr = get_sets_object_mgr(context.scene)
            if p_cls_mgr:
                return p_cls_mgr.execute_ConvertCollectionToObject(self, context)

        self.report({'WARNING'}, 'Can not convert! Check params!')
        return {'CANCELLED'}


class ZSTO_OT_DuplicateAsInstance(bpy.types.Operator):
    """ Duplicate as instance """
    bl_idname = 'zsto.duplicate_as_intance'
    bl_description = 'Duplicate as instanced Object'
    bl_label = 'Duplicate as Instance'
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_DuplicateAsInstance(self, context)
        return {'CANCELLED'}


class ZSTO_OT_UnhideAll(bpy.types.Operator):
    """ Unhide all """
    bl_idname = 'zsto.unhide_all'
    bl_description = 'Unhide all Collections and Objects'
    bl_label = 'Unhide All'
    bl_options = {'REGISTER', 'UNDO'}

    select: bpy.props.BoolProperty(name='Select', default=True)

    @classmethod
    def poll(cls, context):
        return _poll_context_by_current_groups_len(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.unhide_all(context, self.select, self)
            return {'FINISHED'}
        return {'CANCELLED'}


t_invert_hide_status = {}


class ZSTO_OT_InvertHideGroup(bpy.types.Operator):
    """ Invert Hide Group """
    bl_idname = 'zsto.invert_hide_group'
    bl_description = ZsLabels.OT_INVERT_HIDE_GROUP_DESC
    bl_label = ZsLabels.OT_INVERT_HIDE_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    select: bpy.props.BoolProperty(name='Select', default=True)

    # Object Props
    nested: bpy.props.BoolProperty(name='Nested', default=True)

    # FIX: bug in Blender with EnumProperty and Last Op Props
    def update_submode(self, context: bpy.types.Context):
        global t_invert_hide_status
        t_invert_hide_status[context.as_pointer()] = self.submode

    submode: bpy.props.EnumProperty(
        items=[
            ('UNDEFINED', 'Undefined', ''),
            ('ISOLATE', 'Isolate', ''),
            ('UNHIDE_ALL', 'Unhide All', '')],
        options={'HIDDEN', 'SKIP_SAVE'},
        default='UNDEFINED',
        update=update_submode)

    @classmethod
    def poll(cls, context):
        return _poll_context_has_selected_group(context) and _poll_context_by_current_groups_len(context)

    def draw_ex(self, layout: bpy.types.UILayout, context: bpy.types.Context):
        s_submode_id = t_invert_hide_status.get(context.as_pointer(), '')
        if s_submode_id:
            layout.label(text=layout.enum_item_name(self, 'submode', s_submode_id))

        layout.prop(self, 'select')
        if ZenPolls.is_collection_mode(context):
            layout.prop(self, 'nested')

    def draw(self, context: bpy.types.Context):
        self.draw_ex(self.layout, context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr and self.submode != 'UNDEFINED':
            if self.submode == 'ISOLATE':
                p_cls_mgr.set_active_group_invert_hide(context, self)
            elif self.submode == 'UNHIDE_ALL':
                p_cls_mgr.unhide_all(context, self.select, self)

            if self.select:
                bpy.ops.object.select_all(action='SELECT')

            return {'FINISHED'}
        else:
            return {'CANCELLED'}

    def invoke(self, context: bpy.types.Context, event: bpy.types.Event):
        self.submode = 'UNDEFINED'
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            hide_changed, has_elements = p_cls_mgr.set_active_group_invert_hide(context, self)
            if not hide_changed:
                if not has_elements:
                    self.report({'WARNING'}, 'Can not isolate empty group, so unhide all!')
                self.submode = 'UNHIDE_ALL'
                p_cls_mgr.unhide_all(context, self.select, self)
            else:
                self.submode = 'ISOLATE'

            if self.select:
                bpy.ops.object.select_all(action='SELECT')

            return {'FINISHED'}
        else:
            return {'CANCELLED'}


class ZsInternalSetColPropByIndex(bpy.types.Operator):
    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.check_validate_list(context)
            p_list = p_cls_mgr.get_list(context.scene)
            if self.group_index in range(len(p_list)):
                p_group_pair = (self.group_index, p_list[self.group_index])
                if self.mode == 'ISOLATE':
                    prop_changed, has_elements = self.isolate_group_pair_property_state(
                        p_cls_mgr, context, p_group_pair, self)
                    if not prop_changed:
                        if has_elements:
                            p_root_pair = (0, p_list[0])
                            self.set_group_pair_property_state(
                                p_cls_mgr, context, p_root_pair, False, self, set_layer=False)
                        else:
                            self.report({'WARNING'}, 'Can not isolate empty group!')
                elif self.mode == 'UNSET_ALL':
                    p_root_pair = (0, p_list[0])
                    self.nested = True
                    self.set_group_pair_property_state(
                        p_cls_mgr, context, p_root_pair, False, self, set_layer=False)
                else:
                    prop_changed, has_elements, prop_state = self.toggle_group_pair(
                        p_cls_mgr, context, p_group_pair, self)
                    if not prop_changed:
                        if has_elements:
                            status = 'already' if prop_state else 'not'
                            self.report({'WARNING'}, f'Group is {status} {self.label_is_set}!')
                        else:
                            status = self.label_set if prop_state else self.label_unset
                            self.report({'WARNING'}, f'Can not {status} empty group!')
                return {'FINISHED'}
        return {'CANCELLED'}

    def invoke(self, context, event):
        self.mode = 'DEFAULT'
        if event.type == 'LEFTMOUSE':
            if event.ctrl:
                self.mode = 'ISOLATE'
            elif event.shift:
                self.mode = 'UNSET_ALL'
        return self.execute(context)


class ZSTO_OT_ExcludeGroupByIndex(ZsInternalSetColPropByIndex, ZsExcludeCollectionProperty):
    """ Zen Sets Exclude Group By Index """
    bl_idname = 'zsto.exclude_group_by_index'
    bl_label = ZsLabels.OT_EXCLUDE_FROM_VIEWLAYER_LABEL
    bl_description = (
        'Default - Exclude | Include Group\n' +
        'CTRL - Isolate | Restore Excluded Group\n' +
        'SHIFT - Restore All Excluded Groups')
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(name='Group Index', options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        items=[
            ('DEFAULT', 'Exclude | Include', ''),
            ('ISOLATE', 'Isolate | Restore', ''),
            ('UNSET_ALL', 'Restore (Include) All', '')],
        default='DEFAULT',
        options={'HIDDEN', 'SKIP_SAVE'})

    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw(self, context):
        layout = self.layout
        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))
        if self.mode != 'UNSET_ALL':
            layout.prop(self, "nested")


class ZSTO_OT_HideSelectByIndex(ZsInternalSetColPropByIndex, ZsHideSelectCollectionProperty):
    """ Zen Sets Hide Select By Index """
    bl_idname = 'zsto.hide_select_by_index'
    bl_label = 'Hide select in Viewport'
    bl_description = (
        'Default - Hide | Unhide Select in Group\n' +
        'CTRL - Isolate | Restore Unselectable Group\n' +
        'SHIFT - Restore All Unselectable Groups')
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(name='Group Index', options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        items=[
            ('DEFAULT', 'Hide | Unhide', ''),
            ('ISOLATE', 'Isolate | Restore', ''),
            ('UNSET_ALL', 'Restore (Unhide) All', '')],
        default='DEFAULT',
        options={'HIDDEN', 'SKIP_SAVE'})

    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw(self, context):
        layout = self.layout
        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))
        if self.mode != 'UNSET_ALL':
            layout.prop(self, "nested")


class ZSTO_OT_HideViewportByIndex(ZsInternalSetColPropByIndex, ZsHideViewportCollectionProperty):
    """ Zen Sets Hide Viewport By Index """
    bl_idname = 'zsto.hide_viewport_by_index'
    bl_label = 'Hide in ViewLayer'
    bl_description = (
        'Default - Hide | Unhide Group in ViewLayer\n' +
        'CTRL - Isolate | Restore Hidden Group\n' +
        'SHIFT - Restore All Hidden Groups')
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(name='Group Index', options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        items=[
            ('DEFAULT', 'Hide | Unhide', ''),
            ('ISOLATE', 'Isolate | Restore', ''),
            ('UNSET_ALL', 'Restore (Unhide) All', '')],
        default='DEFAULT',
        options={'HIDDEN', 'SKIP_SAVE'})

    select: bpy.props.BoolProperty(name='Select', default=True)

    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw(self, context):
        layout = self.layout
        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))
        layout.prop(self, "select")
        if self.mode != 'UNSET_ALL':
            layout.prop(self, "nested")


class ZSTO_OT_DisableViewportByIndex(ZsInternalSetColPropByIndex, ZsDisableViewportCollectionProperty):
    """ Zen Sets Hide Viewport By Index """
    bl_idname = 'zsto.disable_viewport_by_index'
    bl_label = 'Disable in Viewport'
    bl_description = (
        'Default - Enable | Disable Group in Viewport\n' +
        'CTRL - Isolate | Restore Disabled Group\n' +
        'SHIFT - Restore All Disabled Groups')
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(name='Group Index', options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        items=[
            ('DEFAULT', 'Enable | Disable', ''),
            ('ISOLATE', 'Isolate | Restore', ''),
            ('UNSET_ALL', 'Restore (Enable) All', '')],
        default='DEFAULT',
        options={'HIDDEN', 'SKIP_SAVE'})

    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw(self, context):
        layout = self.layout
        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))
        if self.mode != 'UNSET_ALL':
            layout.prop(self, "nested")


class ZSTO_OT_HideRenderByIndex(ZsInternalSetColPropByIndex, ZsHideRenderCollectionProperty):
    """ Zen Sets Hide Viewport By Index """
    bl_idname = 'zsto.hide_render_by_index'
    bl_label = 'Disable in Renders'
    bl_description = (
        'Default - Enable | Disable Group in Renders\n' +
        'CTRL - Isolate | Restore Disabled Group\n' +
        'SHIFT - Restore All Disabled Groups')
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(name='Group Index', options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        items=[
            ('DEFAULT', 'Enable | Disable', ''),
            ('ISOLATE', 'Isolate | Restore', ''),
            ('UNSET_ALL', 'Restore (Enable) All', '')],
        default='DEFAULT',
        options={'HIDDEN', 'SKIP_SAVE'})

    nested: bpy.props.BoolProperty(name='Nested', default=True)

    def draw(self, context):
        layout = self.layout
        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))
        if self.mode != 'UNSET_ALL':
            layout.prop(self, "nested")


class ZSTO_OT_ObjectsSetByIndex(bpy.types.Operator):
    """ Zen Sets Hide Group By Index """
    bl_idname = 'zsto.internal_select_objects_by_index'
    bl_label = 'Manage Objects in Collection'
    bl_description = ('Default - Select | Deselect Objects\n' +
                      'ALT - Hide | Unhide Objects\n' +
                      'CTRL - Append objects\n' +
                      'SHIFT - Remove objects')
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        items=[
            ('SELECT', 'Select | Deselect Objects', ''),
            ('APPEND', 'Append Objects', ''),
            ('REMOVE', 'Remove Objects', ''),
            ('HIDE', 'Hide | Unhide Objects', '')
        ],
        default='SELECT',
        options={'HIDDEN', 'SKIP_SAVE'})

    nested: bpy.props.BoolProperty(name='Nested', default=True)

    move: bpy.props.BoolProperty(
        name=ZsLabels.PROP_MOVE_OBJECTS_NAME,
        description=ZsLabels.PROP_MOVE_OBJECTS_DESC,
        default=True)

    keep_nested: bpy.props.BoolProperty(
        name=ZsLabels.PROP_KEEP_NESTED_NAME,
        description=ZsLabels.PROP_KEEP_NESTED_DESC,
        default=True)

    @classmethod
    def poll(cls, context):
        return context.scene.objects

    def draw(self, context):
        layout = self.layout

        is_append = self.mode == 'APPEND'
        is_remove = self.mode == 'REMOVE'
        is_select = self.mode == 'SELECT'
        is_hide = self.mode == 'HIDE'

        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))

        if is_append:
            layout.prop(self, 'move')

        if is_append or is_remove:
            layout.prop(self, 'keep_nested')

        if is_select or is_hide:
            layout.prop(self, 'nested')

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.check_validate_list(context)
            p_list = p_cls_mgr.get_list(context.scene)
            if self.group_index in range(len(p_list)):
                p_group = p_list[self.group_index]
                p_group_pair = self.group_index, p_group
                if self.mode == 'APPEND':
                    p_cls_mgr.append_to_group_pair(p_group_pair, self, context)
                elif self.mode == 'REMOVE':
                    p_cls_mgr.remove_selection_from_group_pair(p_group_pair, self, context)
                else:
                    p_collection = p_group.collection
                    p_objects = p_collection.all_objects if self.nested else p_collection.objects

                    if self.mode == 'HIDE':
                        hide_state = all(not p_obj.hide_get() for p_obj in p_objects)

                        for p_obj in p_objects:
                            try:
                                p_obj.hide_set(hide_state)
                            except Exception:
                                pass
                    else:
                        p_selectable = set(context.selectable_objects).intersection(p_objects)
                        if p_selectable:
                            if set(context.selected_objects).issuperset(p_selectable):
                                p_cls_mgr.deselect_group_pair(p_group_pair, self, context)
                            else:
                                p_cls_mgr.select_group_pair(p_group_pair, self, context)
                return {'FINISHED'}
        return {'CANCELLED'}

    def invoke(self, context, event):
        self.mode = 'SELECT'
        if event.type == 'LEFTMOUSE':
            if event.ctrl:
                self.mode = 'APPEND'
            elif event.shift:
                self.mode = 'REMOVE'
            elif event.alt:
                self.mode = 'HIDE'

        return self.execute(context)


_colorize_selected_items = []
_colorize_selected_ids = []


class ZSTO_OT_ColorizeSelected(bpy.types.Operator):
    """ Zen Sets ColorizeSelected """
    bl_idname = 'zsto.colorize_selected'
    bl_label = 'Get Colorized Selected Objects'
    bl_description = 'Colorize Selected Objects in the Viewport'
    bl_options = {'REGISTER'}
    bl_property = 'selected_objects'

    def get_selected_object_ids(self, context):
        global _colorize_selected_items
        global _colorize_selected_ids
        _colorize_selected_ids = [
            (it_obj.name, it_obj.name, '') for it_obj in _colorize_selected_items]
        return _colorize_selected_ids

    def update_selected_object(self, context):
        p_sel_obj = bpy.data.objects.get(self.selected_objects)
        if p_sel_obj and p_sel_obj != context.view_layer.objects.active:
            context.view_layer.objects.active = p_sel_obj

    selected_objects: bpy.props.EnumProperty(
        name='Selected Objects',
        description='List of Selected objects in the Viewport',
        items=get_selected_object_ids,
        update=update_selected_object,
        options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        name="Delete Mode",
        items=[
            ('SELECT', 'Select', ''),
            ('EDIT', 'Edit', ''),
        ],
        default='SELECT')

    @classmethod
    def poll(cls, context):
        return context.selected_objects

    def draw(self, context):
        layout = self.layout

        layout.prop(self, 'mode', expand=True)

        for p_obj in _colorize_selected_items:
            row = layout.row(align=False)
            try:
                row.label(text='', icon='OUTLINER_OB_' + p_obj.type)
            except Exception:
                pass
            s_row = row.split(factor=0.1, align=True)
            s_row.prop(p_obj, 'zen_color', icon_only=True)
            if self.mode == 'SELECT':
                s_row.prop_enum(self, 'selected_objects', p_obj.name)
            else:
                s_row.prop(p_obj, 'name')

    def execute(self, context):
        for p_obj in context.selected_objects:
            try:
                if p_obj.name != self.selected_objects:
                    p_obj.select_set(False)
                else:
                    if context.active_object != p_obj:
                        context.view_layer.objects.active = p_obj
            except Exception:
                pass
        self.cancel(context)
        return {'FINISHED'}

    def cancel(self, context):
        ZenStates.unset_obj_color_state()
        if not getattr(self, 'was_draw_enabled', False):
            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='OFF')

    def invoke(self, context, event):
        res = {'CANCELLED'}

        self.was_draw_enabled = is_draw_active()

        try:
            wm = context.window_manager

            ZenStates.set_obj_color_state()

            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')

            global _colorize_selected_items
            _colorize_selected_items.clear()
            _colorize_selected_items = [it_obj for it_obj in context.selected_objects]

            p_colors = bpy.context.preferences.themes[0].collection_color

            idx = 3  # more pleasant colors
            for it_obj in context.selected_objects:
                it_obj.zen_color = p_colors[idx].color
                idx += 1
                if idx == len(p_colors):
                    idx = 0

            if context.active_object and context.active_object in set(context.selected_objects):
                self.selected_objects = context.active_object.name

            res = wm.invoke_props_dialog(self)

            context.area.tag_redraw()
        except Exception:
            ZenStates.unset_obj_color_state()
            pass

        return res


class ZSTO_OT_ExcludeToggleGroup(bpy.types.Operator):
    """ Exclude Group """
    bl_idname = 'zsto.exclude_toggle_group'
    bl_label = ZsLabels.OT_EXCLUDE_TOGGLE_LABEL
    bl_description = ZsLabels.OT_EXCLUDE_TOGGLE_DESC
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            return p_cls_mgr.get_current_group_pair(context)
        return False

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_group_pair = p_cls_mgr.get_current_group_pair(context)
            if p_group_pair:
                idx, p_group = p_group_pair
                attr_nested = ZsOperatorAttrs.get_operator_attr('zsto.exclude_group_by_index', 'nested', default=True)
                return bpy.ops.zsto.exclude_group_by_index(
                    group_index=idx, mode='DEFAULT', nested=attr_nested)
        return {'CANCELLED'}


def get_rename_object_collection_pair(context: bpy.types.Context, p_obj: bpy.types.Object, p_handled_collections):
    if p_obj is not None:
        p_out_col = None
        p_out_obj = None

        for p_col in p_obj.users_collection:
            if p_col not in p_handled_collections:
                if (p_out_col is None or
                        (context.layer_collection is not None and
                            context.layer_collection.collection == p_col)):
                    p_out_col = p_col
                    p_out_obj = p_obj
        if p_out_col:
            return (p_out_col, p_out_obj)

    return None


class ZSTO_OT_RenameObjectAsParent(bpy.types.Operator):
    """ Rename Object As Parent Collection """
    bl_idname = 'zsto.rename_as_parent_collection'
    bl_label = "Rename Object(s) as Parent Collection"
    bl_description = "Rename active Object or Selected with the same name as Parent Collection"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.mode == 'OBJECT'

    def execute(self, context):

        p_handled_collections = [context.scene.collection]

        def rename_object(p_pair):
            p_col, p_obj = p_pair
            if p_obj.name != p_col.name:

                p_col_name = p_col.name
                p_start_name = p_col_name
                i_index = 2
                while bpy.data.objects.get(p_col_name) is not None:
                    p_col_name = p_start_name + f'_{i_index}'
                    i_index += 1

                p_obj.name = p_col_name

                return True
            return False

        b_modified = False

        if context.active_object:
            p_pair = get_rename_object_collection_pair(context, context.active_object, p_handled_collections)
            if p_pair:
                if rename_object(p_pair):
                    b_modified = True

        for p_obj in context.selected_objects:
            if p_obj == context.active_object:
                continue

            p_pair = get_rename_object_collection_pair(context, p_obj, p_handled_collections)
            if p_pair:
                if rename_object(p_pair):
                    b_modified = True

        if b_modified:
            return {'FINISHED'}

        return {'CANCELLED'}


class ZSTO_OT_RenameCollectionAsObject(bpy.types.Operator):
    """ Rename Object As Parent Collection """
    bl_idname = 'zsto.rename_collection_as_object'
    bl_label = "Rename Parent Collection as Object"
    bl_description = "Rename Parent Collection as active Object name"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return get_rename_object_collection_pair(context, context.active_object, [context.scene.collection]) is not None

    def execute(self, context):
        p_pair = get_rename_object_collection_pair(context, context.active_object, [context.scene.collection])
        if p_pair is not None:
            p_col, p_obj = p_pair
            if p_col.name != p_obj.name:
                was_name = p_col.name
                # Blender adds automatically prefix, so swap names
                exist_col = bpy.data.collections.get(p_obj.name)
                if exist_col:
                    exist_col.name = str(uuid.uuid4())

                p_col.name = p_obj.name

                if exist_col:
                    exist_col.name = was_name

                return {'FINISHED'}

        return {'CANCELLED'}


class ZSTO_OT_SyncParentPosWithChild(bpy.types.Operator):
    """ Sync parent pos with child  """
    bl_idname = 'zsto.sync_parent_pos_with_child'
    bl_label = "Sync Parent Transform"
    bl_description = "Synchronize Parent position, rotation, scale with Child Object"
    bl_options = {'REGISTER', 'UNDO'}

    use_position: bpy.props.BoolProperty(
        name='Use Position',
        description='Synchronize Parent position with Child position',
        default=True)

    use_rotation: bpy.props.BoolProperty(
        name='Use Rotation',
        description='Synchronize Parent rotation with Child rotation',
        default=False)

    use_scale: bpy.props.BoolProperty(
        name='Use Scale',
        description='Synchronize Parent scale with Child scale',
        default=False)

    @classmethod
    def poll(cls, context):
        return context.mode == 'OBJECT' and context.active_object and context.active_object.parent

    def execute(self, context):
        p_act_obj = context.active_object
        p_parent = p_act_obj.parent
        if p_act_obj and p_parent and (self.use_position or self.use_rotation or self.use_scale):
            was_children = set((p_obj, p_obj.matrix_world.copy().freeze()) for p_obj in p_parent.children)
            for p_child_obj, _ in was_children:
                p_child_obj.parent = None

            v_mtx = p_act_obj.matrix_world.copy()
            if not self.use_position or self.use_rotation or not self.use_scale:
                pos, rot, sca = v_mtx.decompose()

                p_pos, p_rot, p_sca = p_parent.matrix_world.decompose()
                if not self.use_position:
                    pos = p_pos
                if not self.use_rotation:
                    rot = p_rot
                if not self.use_scale:
                    sca = p_sca

                mtx_pos = mathutils.Matrix.Translation(pos)
                mtx_rot = rot.to_matrix().to_4x4()
                mtx_sca = mathutils.Matrix.Scale(sca.magnitude, 4)

                v_mtx = mtx_pos @ mtx_rot @ mtx_sca

            p_parent.matrix_world = v_mtx

            for p_child_obj, p_mtx in was_children:
                p_child_obj.parent = p_parent
                p_child_obj.matrix_world = p_mtx
            return {'FINISHED'}

        return {'CANCELLED'}


class ZSTO_OT_GroupLinked(bpy.types.Operator):
    """ Zen Sets Group Linked """
    bl_idname = 'zsto.group_linked'
    bl_description = 'Make groups by delimiters'
    bl_label = ZsLabels.OT_GROUP_LINKED_LABEL
    bl_options = {'REGISTER', 'UNDO'}
    bl_ui_units_x = 14

    delimit: bpy.props.EnumProperty(
        name='Delimit',
        items=[
            ('TYPE', 'Type', ''),
            ('LINKED_OBDATA', 'Object Data', ''),
            ('LINKED_MATERIAL', 'Material', ''),
            ('LINKED_DUPGROUP', ' Instanced Collection', ''),
            ('LINKED_PARTICLE', 'Particle System', ''),
            ('LINKED_LIBRARY', 'Library', ''),
            ('LINKED_LIBRARY_OBDATA', 'Library Object Data', ''),
            ('SINGLE_OBJECT', 'Single Object', ''),
            # User Custom Delimiter
            ('CUSTOM', 'Custom', ''),
        ],
        options={'ENUM_FLAG'},
        default={'TYPE'})

    def get_custom_delimiter_name(self):
        if self.custom_delimiter != '':
            s_NOT_FOUND = '(NOT_FOUND)'
            op_props = get_command_props(self.custom_delimiter)
            if op_props.bl_op_cls:
                try:
                    if op_props.bl_label:
                        return op_props.bl_label
                    elif op_props.bl_desc:
                        return op_props.bl_desc
                    else:
                        return op_props.bl_op_cls.idname_py()
                except Exception:
                    return s_NOT_FOUND
            else:
                return s_NOT_FOUND
        return ''

    custom_delimiter_name: bpy.props.StringProperty(
        name='Custom Operator',
        description='Python Operator cmd that will be used as mesh selector',
        get=get_custom_delimiter_name
    )

    custom_delimiter: bpy.props.StringProperty(
        name='Custom Delimiter Operator'
    )

    min_group_size: bpy.props.IntProperty(
        name='Min Group Size',
        min=1,
        default=1
    )

    def get_delimits(self) -> Tuple[set, bool]:
        return set(self.delimit).difference(['CUSTOM']), 'CUSTOM' in self.delimit

    @classmethod
    def poll(cls, context: bpy.types.Context):
        return ZenPolls.is_object_and_simple_mode(context) and len(context.selectable_objects) != 0

    def execute_custom_delimiter(self):
        op_props = get_command_props(self.custom_delimiter)
        eval(op_props.cmd)

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def execute(self, context):

        custom_active = 'CUSTOM' in self.delimit
        custom_alert = (
            self.custom_delimiter_name == '(NOT_FOUND)' or
            self.custom_delimiter_name == '' and custom_active)
        if custom_alert:
            self.report({'ERROR'}, 'Custom Operator has errors! Check properties!')
            return {'CANCELLED'}

        p_scene = context.scene
        p_cls_mgr = get_sets_object_mgr(p_scene)
        if p_cls_mgr:
            return p_cls_mgr.execute_GroupLinked(self, context)

        return {'CANCELLED'}

    def draw(self, context):
        layout = self.layout

        _ = get_mimic_operator_layout(self, layout, 'delimit', expand=True, factor=0.5)

        get_mimic_operator_layout(self, layout, 'min_group_size', factor=0.5)

        box = layout.box()
        box.active = 'CUSTOM' in self.delimit
        box.alert = (
            self.custom_delimiter_name == '(NOT_FOUND)' or
            self.custom_delimiter_name == '' and box.active)

        get_mimic_operator_layout(self, box, 'custom_delimiter_name', factor=0.5)
        box.prop(self, 'custom_delimiter', text='')


class ZSTO_OT_SelectObjectsWith(bpy.types.Operator):
    """ Zen Sets With Vertex Groups """
    bl_idname = 'zsto.select_objects_with'
    bl_description = ZsLabels.OT_SELECT_OBJS_WITH_IN_GROUPS_DESC
    bl_label = ZsLabels.OT_SELECT_OBJS_WITH_IN_GROUPS_LABEL
    bl_options = {'REGISTER'}

    prop_unhide: bpy.props.BoolProperty(
        name=ZsLabels.PROP_UNHIDE_OBJECTS_NAME,
        description=ZsLabels.PROP_UNHIDE_OBJECTS_DESC,
        default=True)

    prop_make_selectable: bpy.props.BoolProperty(
        name=ZsLabels.PROP_MAKE_SELECTABLE_OBJECTS_NAME,
        description=ZsLabels.PROP_MAKE_SELECTABLE_OBJECTS_DESC,
        default=True
    )

    mode: bpy.props.EnumProperty(
        name='Mode',
        description='Object Selection Mode',
        items=[
            ('WITH_VGROUPS', 'Vertex Groups', 'Select Objects with Vertex Groups'),
            ('WITH_FMAPS', 'Face Maps', 'Select Objects with Face Maps'),
            ('WITH_MODIFIERS', 'Modifiers', 'Select Objects with Modifiers'),
        ],
        default='WITH_VGROUPS')

    @classmethod
    def poll(cls, context):
        p_scene = context.scene
        p_cls_mgr = get_sets_object_mgr(p_scene)
        if p_cls_mgr:
            if (len(p_cls_mgr.get_current_group_pairs(context)) != 0 and
                    p_cls_mgr.get_current_list_index(context) != -1):
                return True
        return False

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        return p_cls_mgr.execute_SelectSceneObjectsWith(self, context) if p_cls_mgr else {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout

        row = layout.row(align=True)
        row.label(text='    ' + layout.enum_item_name(self, 'mode', self.mode))

        col = layout.column(align=False)
        col.prop(self, "prop_unhide", expand=True)
        col.prop(self, "prop_make_selectable", expand=True)


class ZSTO_OT_SortCollections(bpy.types.Operator):
    """ Zen Sets Sort Collections """
    bl_idname = 'object.zsto_sort_collections'
    bl_label = ZsLabels.OT_SORT_COLLECTIONS_LABEL
    bl_description = ZsLabels.OT_SORT_COLLECTIONS_DESC
    bl_options = {'REGISTER', 'UNDO'}

    def get_collection_name(self):
        context = bpy.context
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_group_pair = p_cls_mgr.get_current_group_pair(context)
            if p_group_pair:
                _, p_group = p_group_pair
                return p_group.name
        return ''

    collection_name: bpy.props.StringProperty(
        name='Collection',
        description='Collection that is used for sorting',
        get=get_collection_name,
        set=None)

    case_insensitive: bpy.props.BoolProperty(name='Case Insensitive', default=True)

    mode: bpy.props.EnumProperty(
        name='Mode',
        description='Sort Collections Mode',
        items=[
            ('HUMAN', 'Human Sort', 'Human sort (also known as natural sort)'),
            ('HUMAN_LAST_INTEGER', 'Human Last Integer', 'Human sort starting from last integer'),
            ('HUMAN_FIRST_INTEGER', 'Human First Integer', 'Human sort starting from first integer'),
            ('ALPHA_SORT', 'Alpha Sort', 'Sort in alphabetical order'),
            ('SHUFFLE', 'Shuffle', 'Shuffle Collection children'),
        ],
        default='HUMAN')

    reversed: bpy.props.BoolProperty(name='Reversed', default=False)

    prefix: bpy.props.StringProperty(name='Prefix')

    suffix: bpy.props.StringProperty(name='Suffix')

    nested: bpy.props.BoolProperty(name='Nested', default=True)

    @classmethod
    def poll(cls, context):
        return context.mode == 'OBJECT' and _poll_context_by_current_groups_len(context)

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        return p_cls_mgr.execute_SortCollections(self, context) if p_cls_mgr else {'CANCELLED'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self)


class ZSTO_OT_RemoveLinkedObjects(bpy.types.Operator):
    """ Remove Linked Objects """
    bl_idname = 'zsto.remove_linked_objects'
    bl_label = 'Remove Linked'
    bl_description = 'Remove Objects with Linked data from Groups'
    bl_options = {'REGISTER', 'UNDO'}

    group_mode: bpy.props.EnumProperty(
        name=ZsLabels.PROP_GROUP_MODE,
        items=[
            ('ACTIVE', 'Active', ''),
            ('ALL', 'All', ''),
        ],
        default='ALL')

    @classmethod
    def poll(cls, context):
        if ZenPolls.is_object_and_simple_mode(context):
            p_cls_mgr = get_sets_object_mgr(context.scene)
            if p_cls_mgr:
                p_scene = context.scene
                p_list = p_cls_mgr.get_list(p_scene)
                return p_list and len(p_list) > 0
        return False

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        return p_cls_mgr.execute_RemoveLinkedFromGroup(self, context) if p_cls_mgr else {'CANCELLED'}


class ZSTO_OT_SimpleDisableInRendersByIndex(bpy.types.Operator):
    """ Zen Sets Hide Group in Renders By Index """
    bl_idname = 'zsto.disable_in_renders_by_index'
    bl_label = 'Disable in Renders'
    bl_description = (
        'Default - Disable | Enable Group in Renders\n' +
        'CTRL - Isolate | Restore Group in Renders By Index\n' +
        'SHIFT - Enable All in Renders')
    bl_options = {'REGISTER', 'UNDO'}

    group_index: bpy.props.IntProperty(name='Hide Group By Index', options={'HIDDEN', 'SKIP_SAVE'})

    mode: bpy.props.EnumProperty(
        items=[
            ('HIDE', 'Disable | Enable', ''),
            ('ISOLATE', 'Isolate | Restore', ''),
            ('UNHIDE_ALL', 'Enable All', '')],
        default='HIDE')

    enable_collections: bpy.props.BoolProperty(
        name='Enable Collections',
        description='Enable Collections disabled in Renders if they contain Group objects',
        default=True)

    def draw(self, context):
        layout = self.layout
        layout.label(text=layout.enum_item_name(self, 'mode', self.mode))
        layout.prop(self, 'enable_collections')

    def execute(self, context):
        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr.check_validate_list(context)
            p_list = p_cls_mgr.get_list(context.scene)
            if self.group_index in range(len(p_list)):
                if self.mode == 'ISOLATE':
                    hide_changed, has_elements = p_cls_mgr.set_group_pair_renders_invert_hide(
                        context, (self.group_index, p_list[self.group_index]), self)
                    if not hide_changed:
                        if has_elements:
                            p_cls_mgr.unhide_in_renders_all(context, self.enable_collections)
                        else:
                            self.report({'WARNING'}, 'Can not isolate empty group!')
                elif self.mode == 'UNHIDE_ALL':
                    p_cls_mgr.unhide_in_renders_all(context, self.enable_collections)
                else:
                    hide_changed, has_elements, hide_state = p_cls_mgr.hide_group_in_renders_by_index(context, self.group_index, self)
                    if not hide_changed:
                        if has_elements:
                            status = 'already' if hide_state else 'not'
                            self.report({'WARNING'}, f'Group is {status} hidden! Check Collections!')
                        else:
                            status = 'hide' if hide_state else 'unhide'
                            self.report({'WARNING'}, f'Can not {status} empty group!')
                return {'FINISHED'}
        return {'CANCELLED'}

    def invoke(self, context, event):
        self.mode = 'HIDE'
        if event.type == 'LEFTMOUSE':
            if event.ctrl:
                self.mode = 'ISOLATE'
            elif event.shift:
                self.mode = 'UNHIDE_ALL'
        return self.execute(context)


class ZSTO_PT_OverlayFilter(bpy.types.Panel):
    bl_label = "Zen Sets Overlay Filter"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'

    @classmethod
    def poll(cls, context):
        return context.mode == 'OBJECT'

    def draw(self, context):
        layout = self.layout

        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_cls_mgr._do_draw_overlay_filter(context, layout)


class ZSTO_PT_ColToParentProps(bpy.types.Panel):
    bl_label = "Zen Sets Convert Collection To Parent"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'
    bl_ui_units_x = 14

    @classmethod
    def poll(cls, context):
        return context.mode == 'OBJECT'

    def draw(self, context):
        layout = self.layout
        addon_prefs = get_prefs()
        addon_prefs.op_options.draw_convert_collection_to_parent(layout)


class ZSTO_PT_ParentToColProps(bpy.types.Panel):
    bl_label = "Zen Sets Convert Parent To Collection"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'
    bl_ui_units_x = 14

    @classmethod
    def poll(cls, context):
        return context.mode == 'OBJECT'

    def draw(self, context):
        layout = self.layout
        addon_prefs = get_prefs()
        addon_prefs.op_options.draw_convert_parent_to_collection(layout)


class ZSTO_MT_SelectObjectsWith(bpy.types.Menu):
    bl_label = "Select Objects With"

    @classmethod
    def poll(cls, context):
        return context.mode == 'OBJECT'

    def draw(self, context):
        layout = self.layout

        p_cls_mgr = get_sets_object_mgr(context.scene)
        if p_cls_mgr:
            p_group_pair = p_cls_mgr.get_current_group_pair(context)
            if p_group_pair:
                layout.label(text=p_group_pair[1].name)
                layout.separator()

        p_prop = get_annotated_prop(ZSTO_OT_SelectObjectsWith, 'mode')
        for item in p_prop['items']:
            op = layout.operator(ZSTO_OT_SelectObjectsWith.bl_idname, text=item[1])
            op.mode = item[0]


class ZsObjectsInterfaceFactory:
    classes = (
        ZSTO_OT_Assign,
        ZSTO_OT_Append,
        ZSTO_OT_RemoveSelectionFromGroup,

        ZSTO_OT_NewItemMove,
        ZSTO_OT_NewItemLink,
        ZSTO_OT_NewItem,

        ZSTO_OT_DeleteEmptyGroups,
        ZSTO_OT_DeleteGroupsCombo,
        ZSTO_OT_DeleteGroupsInternal,

        ZSTO_OT_DeleteGroup,
        ZSTO_OT_DeleteGroupOnly,
        ZSTO_OT_DeleteHierarchy,

        ZSTO_OT_DuplicateCollection,
        ZSTO_OT_DuplicateCollectionLinked,
        ZSTO_OT_DuplicateAsInstance,

        ZSTO_OT_ConvertObjectToCollection,
        ZSTO_OT_ConvertCollectionToParentObject,

        ZSTO_OT_ExcludeToggleGroup,
        ZSTO_OT_UnhideAll,
        ZSTO_OT_InvertHideGroup,

        ZSTO_OT_ExcludeGroupByIndex,
        ZSTO_OT_ObjectsSetByIndex,
        ZSTO_OT_HideSelectByIndex,
        ZSTO_OT_HideViewportByIndex,
        ZSTO_OT_DisableViewportByIndex,
        ZSTO_OT_HideRenderByIndex,

        ZSTO_OT_ColorizeSelected,

        ZSTO_OT_RenameObjectAsParent,
        ZSTO_OT_RenameCollectionAsObject,
        ZSTO_OT_SyncParentPosWithChild,

        ZSTO_OT_GroupLinked,
        ZSTO_OT_SelectObjectsWith,
        ZSTO_OT_RemoveLinkedObjects,

        ZSTO_OT_SimpleDisableInRendersByIndex,
        ZSTO_OT_SortCollections,

        ZSTO_PT_OverlayFilter,
        ZSTO_PT_ParentToColProps,
        ZSTO_PT_ColToParentProps,

        ZSTO_MT_SelectObjectsWith
    )
