# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

import bpy
from mathutils import Color

from typing import Set
from timeit import default_timer as timer
from collections import OrderedDict, defaultdict
import math
import uuid
import mathutils
import json
import numpy as np

from .collection_prop_set import ZsHideViewportCollectionProperty

from ..basic_sets import ZsLayerManager
from ..draw_cache import BasicObjectsCacher
from ..draw_sets import remove_draw_handler, is_draw_handler_enabled

from ...labels import ZsLabels
from ...preferences import get_prefs
from ...vlog import Log
from ...blender_zen_utils import ZenLocks, ZenSelectionStats, ZsOperatorAttrs, ireplace, ensure_object_in_viewlayer
from ...progress import start_progress, update_progress, end_progress

_collections_map = {}

_i_last_seq_color = -1


def copy_objects(from_col, to_col, linked, dupe_lut):
    for o in from_col.objects:
        dupe = o.copy()
        if not linked and o.data:
            dupe.data = dupe.data.copy()
        to_col.objects.link(dupe)
        dupe_lut[o] = dupe


def copy_collection(parent, collection, linked=False):
    dupe_lut = defaultdict(lambda: None)

    def _copy(parent, collection, linked=False):
        cc = bpy.data.collections.new(collection.name)
        copy_objects(collection, cc, linked, dupe_lut)

        for c in collection.children:
            _copy(cc, c, linked)

        parent.children.link(cc)
        return cc

    cc = _copy(parent, collection, linked)
    for o, dupe in tuple(dupe_lut.items()):
        parent = dupe_lut[o.parent]
        if parent:
            dupe.parent = parent
        dupe.select_set(True)
    return cc


def _on_expanded_get(self):
    if self.collection:
        return getattr(self.collection, 'zen_expanded', True)
    return True


def _on_expanded_set(self, value):
    if self.collection:
        is_expanded = getattr(self.collection, 'zen_expanded', True)
        if is_expanded != value:
            setattr(self.collection, 'zen_expanded', value)


def _on_name_get(self):
    if self.collection:
        if self.collection == bpy.context.scene.collection:
            return 'Scene Collection'
        return self.collection.name
    return ''


def _on_name_set(self, value):
    if self.collection:
        if self.collection.name != value:
            self.collection.name = value


def _on_color_get(self):
    if self.collection:
        if self.collection == bpy.context.scene.collection:
            p_grid = bpy.context.preferences.themes[0].view_3d.grid
            return p_grid[:3]
        return getattr(self.collection, 'zen_color', (0, 0, 0))
    return (0, 0, 0)


def _on_color_set(self, value):
    if self.collection:
        if self.collection.zen_color != value:
            self.collection.zen_color = value


def _on_get_group_count(self):
    if self.collection:
        return len(self.collection.objects)
    return 0


def ui_select_object_operator(layout, context, p_collection):
    # set selection
    setsel = layout.row(align=True)
    icon = 'DOT'
    all_selected = False
    all_selectable = False

    p_all_objects = p_collection.all_objects

    if not p_all_objects:
        icon = 'RADIOBUT_OFF'
        setsel.active = False
    else:
        p_all_objects_set = set(p_collection.all_objects)
        p_selectable = set(context.selectable_objects)

        # test for all reachable
        p_collection_selectable = p_selectable.intersection(p_all_objects_set)
        if p_collection_selectable != p_all_objects_set:
            setsel.alert = True
            all_selectable = False

        if not p_collection_selectable:
            setsel.active = False
        else:
            icon = 'KEYFRAME'
            p_selected_objects = set(context.selected_objects)
            if p_selected_objects:
                p_intersected = p_selected_objects.intersection(p_collection_selectable)
                if p_intersected:
                    icon = 'KEYFRAME_HLT'
                    if p_intersected == p_collection_selectable:
                        all_selected = True

    t_info = {
        "op": setsel.operator(
            "zsto.internal_select_objects_by_index",
            text="",
            icon=icon,
            depress=all_selected,
            emboss=True),
        "all_selected": all_selected,
        "all_selectable": all_selectable
    }
    return t_info


class ZSTSCollectionListGroup(bpy.types.PropertyGroup):
    """
    Group of properties representing
    an item in the zen sets groups for COLLECTION
    """
    name: bpy.props.StringProperty(get=_on_name_get, set=_on_name_set)
    layer_name: bpy.props.StringProperty()
    group_color: bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_GROUP_COLOR_NAME,
        subtype='COLOR_GAMMA',
        size=3,
        min=0, max=1,
        get=_on_color_get,
        set=_on_color_set
    )
    group_count: bpy.props.IntProperty(name="GroupCount", get=_on_get_group_count)
    group_hide_count: bpy.props.IntProperty(
        name="GroupHiddenCount",
        # update=_on_update_group_hide_count
    )
    expanded: bpy.props.BoolProperty(
        get=_on_expanded_get,
        set=_on_expanded_set)
    level: bpy.props.IntProperty(default=0)
    has_children: bpy.props.BoolProperty(default=False)
    collection: bpy.props.PointerProperty(type=bpy.types.Collection)
    parent_collection: bpy.props.PointerProperty(type=bpy.types.Collection)
    parent_idx: bpy.props.IntProperty(default=-1)

    def get_objects(self, context) -> Set:
        return set(self.collection.objects)


class ZsBaseObjectUIList(bpy.types.UIList):
    def filter_items(self, context, data, propname):
        p_scene_list = getattr(data, propname)

        addon_prefs = get_prefs()
        if addon_prefs.list_options.collection_list_mode == 'SEL_OBJS_PARENTS':
            filters = self.get_selected_parents_filter(context, p_scene_list)
        elif addon_prefs.list_options.collection_list_mode == 'SEL_OBJS':
            filters = self.get_selected_objects_filter(context, p_scene_list)
        elif addon_prefs.list_options.collection_list_mode == 'SEL_COLLECTIONS':
            filters = self.get_selected_collections_filter(context, p_scene_list)
        else:
            filters = self.get_expand_filter(p_scene_list)

        # Default return values.
        flt_flags = [self.bitflag_filter_item] * len(p_scene_list)

        if self.filter_name:
            flt_flags = bpy.types.UI_UL_list.filter_items_by_name(
                self.filter_name, self.bitflag_filter_item, p_scene_list,
                reverse=self.use_filter_sort_reverse)

        for idx in filters:
            flt_flags[idx] &= ~self.bitflag_filter_item

        flt_neworder = []

        return flt_flags, flt_neworder

    def draw_item(self, context: bpy.types.Context, layout: bpy.types.UILayout, data, item, icon, active_data, active_propname, index):
        p_collection = item.collection
        lay_col = self.get_internal_layer_collection(context, index)
        b_is_active_lay_col = lay_col is not None and context.view_layer.active_layer_collection == lay_col
        addon_prefs = get_prefs()
        addon_prefs.list_options.tree_view

        if addon_prefs.list_options.tree_view:
            for _ in range(item.level):
                layout.label(icon='DOT')

        if item.has_children:
            s_icon = 'TRIA_DOWN' if item.expanded else 'TRIA_RIGHT'
            layout.prop(item, 'expanded', icon_only=True, icon=s_icon)
        else:
            if p_collection is None or p_collection.color_tag == 'NONE':
                layout.label(icon='OUTLINER_COLLECTION')
            else:
                layout.label(icon='COLLECTION_' + p_collection.color_tag)

        col = layout.column(align=False)
        col.ui_units_x = 0.7
        col.separator(factor=0.8)
        col.scale_y = 0.6
        col.prop(item, 'group_color', text='')

        r = layout.row(align=True)
        if item.level == 0:
            r1 = r.row(align=True)
            r1.enabled = False
            r1.prop(item, 'name', text='', emboss=b_is_active_lay_col, icon='NONE')
        else:
            r.prop(item, 'name', text='', emboss=b_is_active_lay_col, icon='NONE')

        if lay_col:
            layout.active = lay_col.is_visible

            addon_prefs = get_prefs()

            r = layout.row(align=True)
            r.alignment = 'RIGHT'

            if addon_prefs.list_options.display_objects_info:
                if p_collection:
                    r.label(text=str(len(p_collection.objects)))
                else:
                    r.label(text='-')

            if item.level != 0:

                p_all_objects = p_collection.all_objects
                p_all_objects_count = len(p_all_objects)

                if addon_prefs.list_options.display_excluded_groups_info:
                    r1 = r
                    all_selectable = all(
                        not it_col.exclude
                        for it_col, _ in self._iterate_layer_tree(lay_col, None)
                        if it_col != lay_col)
                    if not all_selectable:
                        r1 = r.row(align=True)
                        r1.alignment = 'RIGHT'
                        r1.alert = True
                    r1.prop(lay_col, 'exclude', emboss=False, icon_only=True)
                if addon_prefs.list_options.display_hidden_groups_info:
                    r1 = r
                    if p_all_objects_count > 0:
                        r1 = r.row(align=True)
                        r1.alignment = 'RIGHT'
                        r1.alert = not set(context.visible_objects).issuperset(p_collection.all_objects)

                    icon_hide_id = 'HIDE_ON' if lay_col.hide_viewport else 'HIDE_OFF'
                    op = r1.operator('zsts.internal_hide_group_by_index', text='', icon=icon_hide_id, emboss=False)
                    op.group_index = index
                if addon_prefs.list_options.display_hide_renders_info and p_collection:
                    r1 = r
                    all_selectable = all(
                        not it_col.hide_render
                        for it_col, _ in self._iterate_layer_tree(p_collection, None)
                        if it_col != p_collection)
                    if all_selectable:
                        if p_all_objects_count > 0:
                            arr = np.empty(p_all_objects_count, 'b')
                            p_all_objects.foreach_get('hide_render', np.reshape(arr, p_all_objects_count))
                            all_selectable = not np.any(arr)
                    if not all_selectable:
                        r1 = r.row(align=True)
                        r1.alignment = 'RIGHT'
                        r1.alert = True

                    icon_hide_id = 'RESTRICT_RENDER_ON' if p_collection.hide_render else 'RESTRICT_RENDER_OFF'
                    op = r1.operator('zsto.hide_render_by_index', text='', icon=icon_hide_id, emboss=False)
                    op.group_index = index

        r.separator()


class ZsObjectLayerManager(ZsLayerManager):

    prop_color_tags = ['NONE', 'COLOR_01', 'COLOR_02', 'COLOR_03', 'COLOR_04', 'COLOR_05', 'COLOR_06', 'COLOR_07', 'COLOR_08']

    @classmethod
    def list_prop_name(cls):
        return 'zen_sets_object_list'

    @classmethod
    def active_list_prop_name(cls):
        return 'zen_sets_object_list_index'

    @classmethod
    def list_layer_prefix(cls):
        return 'zen_sets_object_layer'

    @classmethod
    def get_context_selected_count(self, context):
        return len(context.selected_objects)

    @classmethod
    def get_current_group_pairs(self, context):
        return [(idx, g) for idx, g in enumerate(self.get_list(context.scene))]

    @classmethod
    def get_current_group_pair(self, context):
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        i_items_count = len(p_list)
        i_list_index = self.get_list_index(p_scene)

        if i_list_index in range(i_items_count):
            return (i_list_index, p_list[i_list_index])
        else:
            return None

    @classmethod
    def is_current_group_active(self, context):
        return self.get_active_collection(context) is not None

    @classmethod
    def list_collection_color(cls):
        return f'zen_{cls.id_group}_color'

    @classmethod
    def set_collection_icon_from_color(cls, p_collection):
        color = Color(p_collection.zen_color)
        p_colors = bpy.context.preferences.themes[0].collection_color
        i_tag = -1
        min_diff = 1.0
        for i, p_color in enumerate(p_colors):
            theme_color = Color(p_color.color)
            diff = math.fabs(theme_color.h - color.h)
            if diff < min_diff:
                min_diff = diff
                i_tag = i
        if i_tag != -1:
            new_color_tag = cls.prop_color_tags[i_tag + 1]
            if p_collection.color_tag != new_color_tag:
                p_collection.color_tag = new_color_tag

    @classmethod
    def set_collection_color_from_icon(cls, p_collection):
        try:
            idx = cls.prop_color_tags.index(p_collection.color_tag)
            if idx > 0:
                p_colors = bpy.context.preferences.themes[0].collection_color
                p_collection.zen_color = Color(p_colors[idx - 1].color)
        except Exception as e:
            Log.error('set_collection_color_from_icon:', e)

    @classmethod
    def sync_all_collections_icons(cls, context):
        for it_col, _ in cls._iterate_layer_tree(context.scene.collection, None):
            if it_col != context.scene.collection:
                cls.set_collection_icon_from_color(it_col)

    @classmethod
    def reset_all_collection_icons(cls, context):
        for it_col, _ in cls._iterate_layer_tree(context.scene.collection, None):
            if it_col != context.scene.collection:
                it_col.color_tag = 'NONE'

    @classmethod
    def get_collection_map(cls, p_view_layer):

        global _collections_map

        def _fill_collection_map(col_map, p_layer_collection):
            for p_child in p_layer_collection.children:
                col_map.append(p_child)
                _fill_collection_map(col_map, p_child)

        if p_view_layer not in _collections_map:
            _collections_map[p_view_layer] = [p_view_layer.layer_collection]
            _fill_collection_map(_collections_map[p_view_layer], p_view_layer.layer_collection)

        return _collections_map[p_view_layer]

    @classmethod
    def are_collections_modified(cls, p_scene):
        try:
            p_list = cls.get_list(p_scene)
            items_index = 0
            max_items = len(p_list)

            global _collections_map

            v_layer = bpy.context.view_layer
            if v_layer not in _collections_map:
                Log.debug(f'Layer:[{v_layer.name}] is not in map!')
                return True

            i_lay_col_count = len(_collections_map[v_layer])
            if max_items != i_lay_col_count:
                Log.debug(f'Layer count mismatch: {i_lay_col_count} != {max_items}')
                return True

            res, items_index = cls._has_list_difference(p_list, v_layer.layer_collection, 0, items_index, max_items)
            if res:
                return True

            if items_index != max_items:
                Log.debug(f'Handled layers mismatch: {items_index} != {max_items}')
                return True

            return False
        except Exception as e:
            Log.error(e)
            return True

    @classmethod
    def check_update_list(cls, p_scene):
        if cls.are_collections_modified(p_scene):
            cls.update_collections(p_scene)

    @classmethod
    def check_validate_list(cls, context):
        if not cls._validate_list_pointers(context):
            cls.update_collections(context.scene)

    @classmethod
    def _index_of_collection(cls, p_list, p_collection):
        for idx, group in enumerate(p_list):
            if group.collection == p_collection:
                return idx
        return -1

    @classmethod
    def _index_of_layer_collection(cls, p_view_layer, p_layer_collection):
        try:
            map = cls.get_collection_map(p_view_layer)
            return map.index(p_layer_collection)
        except Exception:
            pass
        return -1

    @classmethod
    def _has_list_difference(cls, p_list, p_layer_collection, level, index, max_items):
        if index >= max_items:
            Log.debug(f'Layer index:{index} is out of range:{max_items}')
            return True, index
        p_group = p_list[index]
        p_collection = p_layer_collection.collection
        if p_group.collection != p_collection or p_group.level != level:
            Log.debug(f'{index}: Collection: {p_collection.name} mismatch')
            return True, index

        index += 1
        for p_child in p_layer_collection.children:
            res, index = cls._has_list_difference(p_list, p_child, level + 1, index, max_items)
            if res:
                return True, index

        return False, index

    @classmethod
    def set_layer_collection_children_hide_state(cls, p_layer_collection_children, state):
        for p_child in p_layer_collection_children:
            if not p_child.exclude:
                if p_child.hide_viewport != state:
                    p_child.hide_viewport = state
                cls.set_layer_collection_children_hide_state(p_child.children, state)

    @classmethod
    def set_layer_collection_hide_state(cls, p_layer_collection, state):
        if not p_layer_collection.exclude:
            p_layer_collection.hide_viewport = state
            for p_child in p_layer_collection.children:
                cls.set_layer_collection_hide_state(p_child, state)

    @classmethod
    def update_collections(cls, p_scene):
        Log.debug('UPDATE COLLECTIONS =============>', timer())
        try:
            ZenLocks.lock_depsgraph_update()
            v_layer = bpy.context.view_layer

            global _collections_map
            _collections_map.clear()
            _collections_map[v_layer] = []

            p_list = cls.get_list(p_scene)

            last_idx = -1
            if p_scene.zen_sets_last_smart_layer != '':
                last_idx = cls._index_of_layer(p_list, p_scene.zen_sets_last_smart_layer)

            p_list.clear()

            cls._get_all_collections(_collections_map[v_layer], p_list, v_layer.layer_collection, -1, 0)

            if last_idx in range(len(p_list)):
                p_scene.zen_sets_last_smart_layer = p_list[last_idx].layer_name
        finally:
            ZenLocks.unlock_depsgraph_update()

    @classmethod
    def get_expand_filter(cls, p_list):
        addon_prefs = get_prefs()
        filters = []
        if addon_prefs.list_options.tree_view:
            old_level = -1
            expanded = True
            for idx, group in enumerate(p_list):
                if group.level < old_level:
                    expanded = True
                if not expanded:
                    filters.append(idx)
                else:
                    expanded = group.expanded
                    old_level = group.level
        return filters

    @classmethod
    def get_selected_parents_filter(cls, context, p_list):
        sel_collections = set(context.selected_objects)

        return [idx for idx, group in enumerate(p_list)
                if not group.collection or
                not set(group.collection.all_objects).intersection(sel_collections)]

    @classmethod
    def get_selected_objects_filter(cls, context: bpy.types.Context, p_list):
        sel_collections = set(context.selected_objects)

        return [idx for idx, group in enumerate(p_list)
                if not group.collection or
                not set(group.collection.objects).intersection(sel_collections)]

    @classmethod
    def get_selected_collections_filter(cls, context: bpy.types.Context, p_list):
        return [idx for idx, group in enumerate(p_list)
                if not group.collection or
                not cls.is_collection_selected(group.collection)]

    @classmethod
    def is_object_available(self, p_obj):
        return not p_obj.hide_select

    @classmethod
    def is_collection_selected(self, p_collection):
        result = False
        if p_collection:
            for p_obj in p_collection.all_objects:
                if self.is_object_available(p_obj):
                    if not p_obj.select_get():
                        return False
                    else:
                        result = True
        return result

    @classmethod
    def is_collection_unselected(self, p_collection):
        result = False
        if p_collection:
            for p_obj in p_collection.all_objects:
                if self.is_object_available(p_obj):
                    if p_obj.select_get():
                        return False
                    else:
                        result = True
        return result

    @classmethod
    def _get_collection_objects_by_select(cls, p_collection, selected, nested=True):
        return [p_obj
                for p_obj in (p_collection.all_objects if nested else p_collection.objects)
                if p_obj.visible_get() and (p_obj.select_get() == selected)]

    @classmethod
    def get_collection_selected_objects(cls, p_collection, nested=True):
        return cls._get_collection_objects_by_select(p_collection, True, nested=nested)

    @classmethod
    def get_collection_unselected_objects(cls, p_collection, nested=True):
        return cls._get_collection_objects_by_select(p_collection, False, nested=nested)

    @classmethod
    def _get_all_collections(cls, col_map, p_list, p_layer_collection, p_group_parent_idx, level):
        cur_idx = len(p_list)

        p_collection = p_layer_collection.collection
        col_map.append(p_layer_collection)
        new_item = p_list.add()
        new_item.level = level
        new_item.layer_name = cls.create_unique_layer_name()
        i_children_count = len(p_collection.children)
        new_item.has_children = i_children_count != 0
        new_item.collection = p_collection
        new_item.parent_collection = p_list[p_group_parent_idx].collection if p_group_parent_idx != -1 else None
        new_item.parent_idx = p_group_parent_idx

        default_color = Color((0, 0, 0))
        if p_collection.zen_color == default_color:
            cls.set_collection_color_from_icon(p_collection)
            if p_collection.zen_color == default_color:
                p_collection.zen_color = cls._gen_new_color(p_list)

        for p_child in p_layer_collection.children:
            cls._get_all_collections(col_map, p_list, p_child, cur_idx, level + 1)

    @classmethod
    def get_obj_highlighted_groups(self, p_obj):
        return self.get_list(bpy.context.scene)

    @classmethod
    def get_highlighted_group_pairs(self, context):
        if get_prefs().common.display_all_parts:
            return self.get_current_group_pairs(context)
        else:
            p_group_pair = self.get_current_group_pair(context)
            return [p_group_pair] if p_group_pair else []

    @classmethod
    def get_highlighted_objects(self, context, p_group):
        return set()

    @classmethod
    def _gen_new_color_seq(self, p_list, colors):
        if len(colors) == 0:
            return Color()

        global _i_last_seq_color
        _i_last_seq_color += 1

        if _i_last_seq_color == -1 or _i_last_seq_color >= len(colors):
            _i_last_seq_color = 0

        return colors[_i_last_seq_color].color

    @classmethod
    def get_layer_collection(cls, p_layer_collection, p_collection):
        if p_layer_collection is not None:
            if p_layer_collection.collection == p_collection:
                return p_layer_collection

            for lay_child in p_layer_collection.children:
                res = cls.get_layer_collection(lay_child, p_collection)
                if res:
                    return res

        return None

    @classmethod
    def get_internal_layer_collection(cls, context, p_group_index):
        try:
            map = cls.get_collection_map(context.view_layer)
            if p_group_index in range(len(map)):
                return map[p_group_index]
        except Exception as e:
            Log.error(f'INTERNAL COL ERROR:{p_group_index}:', e)

        return None

    @classmethod
    def _validate_list_pointers(self, context):
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        if len(p_list):
            p_view_layer = context.view_layer
            root_lay_col = p_view_layer.layer_collection
            if p_list[0].collection != root_lay_col:
                Log.debug('==========> List should be rehashed!!!')
                p_layer_list = [(p_lay_col, p_lay_col_parent)
                                for (p_lay_col, p_lay_col_parent)
                                in self._iterate_layer_tree(root_lay_col, None)]
                list_size = len(p_list)
                if p_view_layer in _collections_map and list_size == len(p_layer_list):
                    p_map = _collections_map[p_view_layer]
                    if list_size == len(p_map):
                        for idx in range(list_size):
                            p_lay_col, p_lay_col_parent = p_layer_list[idx]
                            p_map[idx] = p_lay_col
                            p_list[idx].collection = p_lay_col.collection
                            p_list[idx].parent_collection = p_lay_col_parent.collection if p_lay_col_parent else None
                        return True
        return False

    @classmethod
    def get_minimum_parent(self, p_lay_col, lay_cols):
        if not p_lay_col.exclude:
            if all(self.is_child_of_collection(p_lay_col, it) for it in lay_cols):
                result = p_lay_col
                if not self.is_collection_selected(p_lay_col.collection):
                    for it in p_lay_col.children:
                        res = self.get_minimum_parent(it, lay_cols)
                        if res:
                            if result != p_lay_col:
                                return p_lay_col
                            else:
                                result = res
                return result
        return None

    @classmethod
    def _do_get_selected_collections(self, context, selected):
        sel_collections = set()
        for p_obj in context.selected_objects:
            sel_collections.update(set(p_obj.users_collection))

        collections_in_mode = sel_collections

        root_lay_col = context.view_layer.layer_collection

        if not selected:
            collections_in_mode = set(
                it_col for it_col, _ in self._iterate_layer_tree(context.scene.collection, None)
                if it_col not in sel_collections)

        lay_cols_in_mode = set(
            it_lay_col for it_lay_col, _ in self._iterate_layer_tree(root_lay_col, None)
            if not it_lay_col.exclude and it_lay_col.collection in collections_in_mode)

        min_parent = self.get_minimum_parent(root_lay_col, lay_cols_in_mode)
        if min_parent is None:
            min_parent = root_lay_col

        return min_parent, lay_cols_in_mode

    @classmethod
    def _get_collections_and_objects_by_select(self, context, selected):
        dict_collections = OrderedDict()
        was_sel_objs = set(context.selected_objects) if selected else set(context.selectable_objects).difference(context.selected_objects)
        sel_objs = was_sel_objs.copy()
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        if len(p_list) == 0:
            return (None, None, None)

        p_scene_collection = p_scene.collection

        handled_level = -1

        for idx, group in enumerate(p_list):
            p_collection = group.collection
            if p_collection == p_scene_collection:
                continue
            if len(sel_objs) == 0:
                break

            if handled_level == -1 or group.level <= handled_level:
                handled_level = -1

                lay_col = self.get_internal_layer_collection(context, idx)
                if lay_col and not lay_col.exclude:
                    all_collection_in_state = self.is_collection_selected(p_collection) \
                        if selected \
                        else self.is_collection_unselected(p_collection)

                    if all_collection_in_state:
                        sel_col_objs = set(self._get_collection_objects_by_select(p_collection, selected, nested=True))

                        sel_objs = sel_objs.difference(sel_col_objs)
                        dict_collections[idx] = (group, lay_col)
                        handled_level = group.level

        root_parent_pair = (0, p_list[0])

        p_selected_collections = set(p_lay_col.collection for p_group, p_lay_col in dict_collections.values())
        if len(was_sel_objs) and p_scene_collection not in p_selected_collections:
            handled_level = -1
            for idx, group in enumerate(p_list):
                p_collection = group.collection
                if p_collection is None or p_collection == p_scene_collection:
                    continue

                if handled_level != -1 and idx <= handled_level:
                    break

                lay_col = self.get_internal_layer_collection(context, idx)
                if lay_col and not lay_col.exclude:
                    is_root_for_collections = all(
                        self.is_child_of_collection(p_collection, p_sel_col)
                        for p_sel_col in p_selected_collections)
                    if is_root_for_collections and was_sel_objs and was_sel_objs.issubset(p_collection.all_objects):
                        root_parent_pair = (idx, group)
                        handled_level = idx

        return (root_parent_pair, dict_collections, sel_objs)

    @classmethod
    def get_selected_collections_and_objects(self, context):
        return self._get_collections_and_objects_by_select(context, True)

    @classmethod
    def get_unselected_collections_and_objects(self, context):
        return self._get_collections_and_objects_by_select(context, False)

    @classmethod
    def get_selected_collections_and_objects_inside_group_pair(self, context, group_pair):
        dict_collections = OrderedDict()
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        if len(p_list) == 0:
            return (None, None, None)

        cur_idx, cur_group = group_pair
        start_level = cur_group.level
        sel_objs = set(p_obj for p_obj in context.selected_objects if p_obj.name in cur_group.collection.all_objects)
        handled_level = -1
        for idx in range(cur_idx + 1, len(p_list), 1):
            if len(sel_objs) == 0:
                break
            group = p_list[idx]
            if group.level <= start_level:
                break

            if handled_level == -1 or group.level <= handled_level:
                handled_level = -1

                lay_col = self.get_internal_layer_collection(context, idx)
                if lay_col and not lay_col.exclude:
                    if group.parent_idx not in dict_collections.keys():
                        p_collection = group.collection
                        if self.is_collection_selected(p_collection):
                            sel_col_objs = set(self.get_collection_selected_objects(p_collection))
                            sel_objs = sel_objs.difference(sel_col_objs)
                            dict_collections[idx] = (group, lay_col)
                            handled_level = idx

        root_parent_pair = (cur_group.parent_idx, p_list[cur_group.parent_idx]) \
            if cur_group.parent_idx in range(len(p_list)) \
            else (0, p_list[0])

        return (root_parent_pair, dict_collections, sel_objs)

    @classmethod
    def smart_select(self, context, select_active_group_only, keep_active_group) -> ZenSelectionStats:
        selection_stats = ZenSelectionStats()
        selection_stats.was_object_sel_count = self.get_context_selected_count(context)
        if selection_stats.was_object_sel_count == 0:
            return selection_stats

        self.check_validate_list(context)
        p_scene = context.scene

        sel_collections = set()
        was_sel_objects = set(context.selected_objects)
        for p_obj in context.selected_objects:
            sel_collections.update(set(p_obj.users_collection))

        if select_active_group_only:
            if bpy.context.collection in sel_collections:
                sel_collections = set([bpy.context.collection])

        bpy.ops.object.select_all(action='DESELECT')

        p_list = self.get_list(p_scene)
        i_list_index = self.get_list_index(p_scene)
        i_new_index = i_list_index

        for idx in range(len(p_list)):
            p_lay_col = self.get_internal_layer_collection(context, idx)
            if p_lay_col and not p_lay_col.exclude:
                p_collection = p_lay_col.collection
                if p_collection in sel_collections:
                    if len(p_collection.objects):
                        for col_obj in p_collection.objects:
                            if not col_obj.select_get():
                                try:
                                    col_obj.select_set(True)
                                except Exception:
                                    pass

                        self._do_set_last_smart_select(p_list[idx].layer_name)
                        i_new_index = idx

        if not keep_active_group and i_new_index != i_list_index:
            self.set_list_index(p_scene, i_new_index)

        selection_stats.new_object_sel_count = self.get_context_selected_count(context)
        selection_stats.selection_changed = was_sel_objects != set(context.selected_objects)
        return selection_stats

    @classmethod
    def is_anyone_hidden(self, context):
        p_list = self.get_list(context.scene)
        for idx, _ in enumerate(p_list):
            lay_col = self.get_internal_layer_collection(context, idx)
            if lay_col and not lay_col.exclude:
                if not lay_col.is_visible:
                    return True

        return any(p_obj.hide_get() for p_obj in context.view_layer.objects)

    @classmethod
    def relink_objects(self, root_lay_col, sel_objs, target_lay_col, move):
        target_col = target_lay_col.collection
        """ Process Link """
        for p_obj in sel_objs:
            try:
                target_col.objects.link(p_obj)
            except Exception:
                pass

        if move:
            def process_objects(p_lay_col):
                if not p_lay_col.exclude:
                    if p_lay_col != target_lay_col:
                        col = p_lay_col.collection
                        for it_obj in sel_objs.intersection(col.objects):
                            col.objects.unlink(it_obj)

                    for it_lay_col in p_lay_col.children:
                        process_objects(it_lay_col)

            """ Process Move """
            process_objects(root_lay_col)

    @classmethod
    def new_group(self, context, props):
        try:
            move = props.move
            keep_nested = ZsOperatorAttrs.get_operator_attr('zsto.new_group', 'keep_nested', True)

            self.check_validate_list(context)

            ZenLocks.lock_depsgraph_update()

            min_parent, lay_cols_in_mode = self._do_get_selected_collections(context, True)
            i_sel_cols_count = len(lay_cols_in_mode)
            if i_sel_cols_count != 0:
                if i_sel_cols_count == 1:
                    it_col = next(iter(lay_cols_in_mode))
                    if not self.is_collection_selected(it_col.collection):
                        min_parent = it_col
            else:
                p_group_pair = self.get_current_group_pair(context)
                if p_group_pair:
                    p_lay_col = self.get_internal_layer_collection(context, p_group_pair[0])
                    if p_lay_col and not p_lay_col.exclude:
                        min_parent = p_lay_col

            new_collection = bpy.data.collections.new(self.list_item_prefix)
            min_parent.collection.children.link(new_collection)
            new_lay_col = self.get_layer_collection(min_parent, new_collection)

            root_lay_col = context.view_layer.layer_collection

            sel_objs = set(context.selected_objects)

            if keep_nested:

                def iterate_selected(p_lay_col, p_lay_col_parent):
                    if p_lay_col != new_lay_col and not p_lay_col.exclude:
                        p_collection = p_lay_col.collection

                        all_collection_in_state = False
                        if p_lay_col != root_lay_col and not self.is_child_of_collection(p_lay_col, new_lay_col):
                            all_collection_in_state = self.is_collection_selected(p_collection)

                        def process_layer_collection(col_parent, lay_col):
                            try:
                                states = None
                                if move:
                                    states = self.unlink_collection(col_parent, lay_col)

                                self.link_with_unlock(new_lay_col, lay_col.collection, states)
                            except Exception as e:
                                Log.error('process_layer_collection:', e)

                        def process_objects(col):
                            for it_obj in sel_objs.intersection(col.objects):
                                try:
                                    new_lay_col.collection.objects.link(it_obj)
                                except Exception as e:
                                    Log.error('process_objects:', e)

                                col.objects.unlink(it_obj)

                        if all_collection_in_state:
                            process_layer_collection(p_lay_col_parent.collection, p_lay_col)
                        else:
                            for it_lay_col in p_lay_col.children:
                                iterate_selected(it_lay_col, p_lay_col)

                            process_objects(p_collection)

                """ Process Nested """
                iterate_selected(root_lay_col, None)
            else:
                self.relink_objects(root_lay_col, sel_objs, new_lay_col, move)

            p_list = self.get_list(context.scene)
            new_collection.zen_color = self._gen_new_color(p_list)

            try:
                context.view_layer.active_layer_collection = new_lay_col
            except Exception:
                pass

        finally:
            ZenLocks.unlock_depsgraph_update()

        p_scene = context.scene
        self.update_collections(p_scene)

        idx = self._index_of_layer_collection(context.view_layer, context.view_layer.active_layer_collection)
        if idx != -1:
            p_list = self.get_list(p_scene)
            p_scene.zen_sets_last_smart_layer = p_list[idx].layer_name
            self.set_list_index(p_scene, idx)

        self._add_draw_handler()

    @classmethod
    def execute_NewGroup(self, operator, context):
        self.new_group(context, operator)
        return {'FINISHED'}

    @classmethod
    def get_active_collection(self, context):
        group_pair = self.get_current_group_pair(context)
        if group_pair:
            return group_pair[1].collection
        return None

    @classmethod
    def get_active_layer_collection(self, context):
        p_collection = self.get_active_collection(context)
        if p_collection:
            lay_col = self.get_internal_layer_collection(context, p_collection)
            return lay_col
        return None

    @classmethod
    def execute_AppendToGroup(self, operator, context):
        self.check_validate_list(context)
        p_current_group_pair = self.get_current_group_pair(context)
        if p_current_group_pair:
            self.append_to_group_pair(p_current_group_pair, operator, context)

        return {'FINISHED'}

    @classmethod
    def is_child_of_collection(self, p_collection_parent, p_collection_child):
        for p_collection in p_collection_parent.children:
            if p_collection == p_collection_child:
                return True
            res = self.is_child_of_collection(p_collection, p_collection_child)
            if res:
                return res
        return False

    @classmethod
    def get_lay_col_states(self, p_layer_collection):
        return (p_layer_collection.exclude,
                p_layer_collection.hide_viewport,
                p_layer_collection.holdout,
                p_layer_collection.indirect_only)

    @classmethod
    def set_lay_col_states(self, p_layer_collection, p_states):
        p_layer_collection.exclude = p_states[0]
        p_layer_collection.hide_viewport = p_states[1]
        p_layer_collection.holdout = p_states[2]
        p_layer_collection.indirect_only = p_states[3]

    @classmethod
    def unlink_collection(self, p_collection_parent, p_lay_col):
        states = self.get_lay_col_states(p_lay_col)
        p_collection = p_lay_col.collection
        p_collection_parent.children.unlink(p_collection)
        return states

    @classmethod
    def link_collection(self, p_lay_col_parent, p_collection, states):
        p_collection_parent = p_lay_col_parent.collection
        if p_collection != p_collection_parent and p_collection.name not in p_collection_parent.children:
            p_collection_parent.children.link(p_collection)
            if states is not None:
                p_lay_col = self.get_layer_collection(p_lay_col_parent, p_collection)
                if p_lay_col:
                    self.set_lay_col_states(p_lay_col, states)
            return True
        return False

    @classmethod
    def link_with_unlock(self, p_lay_col_parent, p_collection, states) -> bpy.types.LayerCollection:
        p_lay_col = None
        if not self.link_collection(p_lay_col_parent, p_collection, states):
            p_lay_col = self.get_layer_collection(p_lay_col_parent, p_collection)
            if p_lay_col:
                self.unlock_layer_collection(p_lay_col)
        return p_lay_col

    @classmethod
    def finalize_hide_state(self, context):
        for idx, group in reversed(self.get_current_group_pairs(context)):
            p_collection = group.collection
            if p_collection is None or idx == 0:
                continue

            lay_col = self.get_internal_layer_collection(context, idx)
            if lay_col and not lay_col.exclude:
                if all(not p_obj.visible_get() for p_obj in p_collection.all_objects):
                    lay_col.hide_viewport = True

    @classmethod
    def unhide_all(self, context, b_select, props):
        self.set_layer_collection_children_hide_state(context.view_layer.layer_collection.children, False)
        if bpy.ops.object.hide_view_clear.poll():
            bpy.ops.object.hide_view_clear(select=b_select)

    @classmethod
    def remove_selection_from_group_pair(self, p_group_pair, operator, context):
        if not p_group_pair:
            operator.report({'INFO'}, 'Collection is not selected in Zen Sets List')
            return

        sel_objs = set(context.selected_objects)
        if not sel_objs:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
            return

        idx, p_group = p_group_pair
        p_cur_lay_col = self.get_internal_layer_collection(context, idx)
        if not p_cur_lay_col or p_cur_lay_col.exclude:
            operator.report({'INFO'}, 'Can not remove objects from excluded Collection!')
            return

        if idx == 0:
            operator.report({'INFO'}, 'Can not remove objects from Scene Collection')
            return

        p_cur_lay_col_parent = self.get_internal_layer_collection(context, p_group.parent_idx)
        if not p_cur_lay_col_parent:
            operator.report({'ERROR'}, 'Can not detect Collection parent')
            return

        if not set(p_cur_lay_col.collection.all_objects).intersection(sel_objs):
            operator.report({'INFO'}, 'Objects are not present in: ' + p_cur_lay_col.name)

        try:
            ZenLocks.lock_depsgraph_update()

            target_lay_col = p_cur_lay_col_parent

            if operator.keep_nested:

                def iterate_selected(p_lay_col, p_lay_col_parent):
                    p_collection = p_lay_col.collection

                    all_collection_in_state = (
                        self.is_collection_selected(p_collection)
                        if p_lay_col != p_cur_lay_col else False)

                    def process_layer_collection(col_parent, lay_col):
                        if lay_col == p_cur_lay_col:
                            return

                        try:
                            states = self.unlink_collection(col_parent, lay_col)

                            self.link_with_unlock(target_lay_col, lay_col.collection, states)
                        except Exception as e:
                            Log.error('process_layer_collection:', e)

                    def process_objects(col):
                        for it_obj in sel_objs.intersection(col.objects):
                            try:
                                target_lay_col.collection.objects.link(it_obj)
                            except Exception as e:
                                Log.error('process_objects', e)

                            col.objects.unlink(it_obj)

                    if all_collection_in_state:
                        process_layer_collection(p_lay_col_parent.collection, p_lay_col)
                    else:
                        for it_lay_col in p_lay_col.children:
                            if it_lay_col != target_lay_col and not it_lay_col.exclude:
                                iterate_selected(it_lay_col, p_lay_col)

                        process_objects(p_collection)

                """ Process Nested """
                iterate_selected(p_cur_lay_col, None)
            else:
                self.relink_objects(p_cur_lay_col, sel_objs, target_lay_col, True)

        finally:
            ZenLocks.unlock_depsgraph_update()

            self.update_collections(context.scene)

    @classmethod
    def execute_RemoveSelectionFromGroup(self, operator: bpy.types.Operator, context: bpy.types.Context):
        self.check_validate_list(context)

        if operator.mode == 'ALL':
            if bpy.ops.object.move_to_collection.poll():
                bpy.ops.object.move_to_collection(collection_index=0)
                return {'FINISHED'}
        else:
            p_group_pair = self.get_current_group_pair(context)
            if p_group_pair:
                _, p_group = p_group_pair
                p_collection = p_group.collection
                if p_collection and p_collection != context.scene.collection:
                    self.remove_selection_from_group_pair(p_group_pair, operator, context)
                    return {'FINISHED'}
        return {'CANCELLED'}

    @classmethod
    def execute_DeleteGroup(self, operator, context: bpy.types.Context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:

            was_active_obj = context.view_layer.objects.active

            idx, p_group = p_group_pair
            p_collection = p_group.collection
            if p_collection is not None and p_collection != context.scene.collection:
                p_lay_col_parent = self.get_internal_layer_collection(context, p_group.parent_idx)
                p_collection_parent = p_lay_col_parent.collection
                p_lay_col = self.get_internal_layer_collection(context, idx)
                for p_lay_col_child in p_lay_col.children:
                    p_child = p_lay_col_child.collection
                    states = self.unlink_collection(p_collection, p_lay_col_child)
                    self.link_with_unlock(p_lay_col_parent, p_child, states)

                for p_obj in p_collection.objects:
                    p_collection.objects.unlink(p_obj)
                    p_collection_parent.objects.link(p_obj)

                bpy.data.collections.remove(p_collection)

                try:
                    context.view_layer.objects.active = was_active_obj
                    if not context.view_layer.objects.active:
                        if context.selected_objects:
                            context.view_layer.objects.active = context.selected_objects[0]
                except Exception:
                    pass

                return {'FINISHED'}
            else:
                operator.report({'INFO'}, 'Can not delete Scene Collection!')
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'CANCELLED'}

    @classmethod
    def execute_DeleteAllGroups(self, operator, context):
        self.check_validate_list(context)
        p_scene_collection = context.scene.collection
        is_modified = False
        try:
            ZenLocks.lock_depsgraph_update()

            root_lay_col = context.view_layer.layer_collection
            excluded_collections = set(
                it_lay_col.collection for it_lay_col in self.get_children(root_lay_col, available_only=False)
                if not it_lay_col.is_visible)

            if operator.visible_only and excluded_collections:
                is_modified = True

                def iterate_selected(p_lay_col, p_last_parent):
                    for it_lay_col in p_lay_col.children:
                        p_collection = it_lay_col.collection
                        if it_lay_col.is_visible and p_collection not in excluded_collections:

                            iterate_selected(it_lay_col, p_lay_col, p_last_parent)

                            p_parent_collection = p_last_parent.collection
                            if p_collection != p_parent_collection:
                                for p_obj in p_collection.objects:
                                    p_collection.objects.unlink(p_obj)
                                    try:
                                        p_parent_collection.objects.link(p_obj)
                                    except Exception:
                                        pass
                            bpy.data.collections.remove(p_collection)
                        else:
                            states = self.get_lay_col_states(it_lay_col)

                            if it_lay_col not in set(p_last_parent.children):
                                states = self.unlink_collection(p_lay_col.collection, it_lay_col)
                                self.link_collection(p_last_parent, it_lay_col.collection, states)

                            p_child = p_last_parent.children.get(it_lay_col.name)
                            if p_child and p_child.is_visible:
                                self.set_lay_col_states(p_child, states)

                iterate_selected(root_lay_col, root_lay_col)
            else:
                for idx, group in reversed(self.get_current_group_pairs(context)):
                    p_collection = group.collection
                    if not p_collection or p_collection == p_scene_collection:
                        continue

                    for p_obj in p_collection.objects:
                        p_collection.objects.unlink(p_obj)
                        p_scene_collection.objects.link(p_obj)
                    bpy.data.collections.remove(p_collection)
                    is_modified = True
        finally:
            ZenLocks.unlock_depsgraph_update()

        if is_modified:
            self.update_collections(context.scene)
            return {'FINISHED'}
        else:
            return {'CANCELLED'}

    @classmethod
    def execute_DeleteEmptyGroups(self, operator, context):
        self.check_validate_list(context)
        is_modified = False
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        try:
            ZenLocks.lock_depsgraph_update()
            for idx in range(len(p_list) - 1, 0, -1):
                group = p_list[idx]
                p_collection = group.collection
                if p_collection and p_collection != p_scene.collection:
                    lay_col = self.get_internal_layer_collection(context, idx)
                    if lay_col and not lay_col.exclude:
                        if operator.visible_only and not lay_col.is_visible:
                            continue

                        if len(p_collection.objects) == 0:
                            if len(p_collection.children) != 0 and operator.without_nested_objects:
                                continue
                            if len(p_collection.all_objects) != 0:
                                p_group_parent_idx = group.parent_idx
                                while p_group_parent_idx != -1:
                                    p_group_parent = p_list[p_group_parent_idx]
                                    p_collection_parent = p_group_parent.collection
                                    if p_collection_parent and len(p_group_parent.collection.objects):
                                        break
                                    else:
                                        p_group_parent_idx = p_group_parent.parent_idx

                                if p_group_parent_idx == -1:
                                    p_group_parent_idx = 0

                                p_lay_col_parent = self.get_internal_layer_collection(context, p_group_parent_idx)
                                if p_lay_col_parent is None:
                                    p_lay_col_parent = context.view_layer.layer_collection

                                if p_lay_col_parent.exclude:
                                    p_lay_col_parent.exclude = False

                                for p_lay_col_child in lay_col.children:
                                    p_child = p_lay_col_child.collection
                                    states = self.unlink_collection(p_collection, p_lay_col_child)
                                    self.link_with_unlock(p_lay_col_parent, p_child, states)
                            bpy.data.collections.remove(p_collection)
                            is_modified = True
        finally:
            ZenLocks.unlock_depsgraph_update()
        if is_modified:
            self.update_collections(context.scene)
            return {'FINISHED'}
        else:
            # do not use 'CANCELLED' here, because props are not drawn
            return {'FINISHED'}

    @classmethod
    def unlock_layer_collection(self, p_layer_collection):
        if p_layer_collection:
            if p_layer_collection.exclude:
                p_layer_collection.exclude = False
            if p_layer_collection.hide_viewport:
                p_layer_collection.hide_viewport = False
            p_collection = p_layer_collection.collection
            if p_collection.hide_viewport:
                p_collection.hide_viewport = False
            if p_collection.hide_select:
                p_collection.hide_select = False

    @classmethod
    def hide_group_by_index(self, context, group_index, props):
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        hidden_state = True
        if group_index in range(len(p_list)):
            lay_col = self.get_internal_layer_collection(context, group_index)
            if lay_col:
                hidden_state = ZsHideViewportCollectionProperty.are_all_unset(self, context, lay_col, props)
                status = self.set_group_pair_hidden_state(context, (group_index, p_list[group_index]), hidden_state, props)
                return status[0], status[1], hidden_state
        else:
            return False, False, hidden_state

    @classmethod
    def set_group_pair_hidden_state(self, context, p_group_pair, hidden_state, props):
        return ZsHideViewportCollectionProperty.set_group_pair_property_state(
            self, context, p_group_pair, hidden_state, props)

    @classmethod
    def get_parents(self, context, p_group_pair, include_root=True, available_only=True):
        parents = []
        p_list = self.get_list(context.scene)
        self.get_group_pair_parent_groups(context, p_list, p_group_pair, p_parents=parents)
        result = []
        root = context.view_layer.layer_collection
        for it in parents:
            idx, p_group = it
            lay_col = self.get_internal_layer_collection(context, idx)
            if lay_col:
                if not include_root and lay_col == root:
                    continue
                if not available_only or not lay_col.exclude:
                    result.append(lay_col)
        return result

    @classmethod
    def get_children(self, p_lay_col, available_only=True):
        return [it for it, _ in self._iterate_layer_tree(p_lay_col, None)
                if (it != p_lay_col and not available_only or not it.exclude)]

    @classmethod
    def get_group_pair_parent_groups(self, context, p_list, p_group_pair, p_parents=[]):
        if p_group_pair:
            idx, p_group = p_group_pair
            idx_parent = p_group.parent_idx
            if idx_parent != -1:
                p_group_parent = (idx_parent, p_list[idx_parent])
                p_parents.append(p_group_parent)
                self.get_group_pair_parent_groups(context, p_list, p_group_parent, p_parents=p_parents)

    @classmethod
    def set_group_pair_invert_hide(self, context, p_group_pair, props):
        return ZsHideViewportCollectionProperty.isolate_group_pair_property_state(self, context, p_group_pair, props)

    @classmethod
    def convert_object_to_collection(self, context, parent_list, p_collection_parent, p_obj, props):
        p_link_to = p_collection_parent
        p_new_collection = None
        if len(p_obj.children):
            p_new_collection = bpy.data.collections.new(p_obj.name)
            p_collection_parent.children.link(p_new_collection)
            parent_list.append((p_obj, p_new_collection))

            if p_obj.zen_color[:] != (0, 0, 0):
                p_new_collection.zen_color = p_obj.zen_color

            p_link_to = p_new_collection

        p_link_to.objects.link(p_obj)

        if props.move:
            # remove from all other collections
            for collection in p_obj.users_collection:
                if collection != p_link_to:
                    collection.objects.unlink(p_obj)

        if p_new_collection:
            for p_child_obj in p_obj.children:
                self.convert_object_to_collection(context, parent_list, p_new_collection, p_child_obj, props)

        return p_link_to

    @classmethod
    def execute_ConvertObjectToCollection(self, operator, context):
        self.check_validate_list(context)
        p_obj = context.active_object
        if p_obj and len(p_obj.users_collection):
            p_collection_parent = p_obj.users_collection[0]

            parent_list = []
            p_new_collection = self.convert_object_to_collection(
                context, parent_list, p_collection_parent, p_obj, operator)
            if p_new_collection:
                bpy.ops.object.select_all(action='DESELECT')
                for p_obj in p_new_collection.all_objects:
                    try:
                        p_obj.select_set(True)
                    except Exception:
                        pass
                bpy.ops.object.parent_clear(type='CLEAR_KEEP_TRANSFORM')

                skip_objects = set()
                for parent, p_collection in reversed(parent_list):
                    if parent.type == 'EMPTY':
                        addon_prefs = get_prefs()
                        skip_mode = addon_prefs.op_options.convert_parent_to_collection_skip_mode
                        if skip_mode == '1':
                            bpy.data.objects.remove(parent)
                        elif skip_mode == '2':
                            continue
                        else:
                            skip_objects.add(parent)
                            center = self.get_collection_center(context, p_collection, addon_prefs.op_options, skip_objects=skip_objects)
                            c_pos, c_rot, c_sca = center.decompose()
                            p_pos, p_rot, p_sca = parent.matrix_world.decompose()

                            if (math.isclose((c_pos - p_pos).length, 0, abs_tol=1e-06) and
                                (not addon_prefs.op_options.calculate_col_center_calculate_center_with_rot or
                                 math.isclose((c_rot.to_exponential_map() - p_rot.to_exponential_map()).length, 0, abs_tol=1e-06)) and
                                (not addon_prefs.op_options.calculate_col_center_calculate_center_with_scale or
                                 math.isclose((c_sca - p_sca).length, 0, abs_tol=1e-06))):
                                # Mismatch ->
                                bpy.data.objects.remove(parent)

                try:
                    p_new_view_lay_col = self.get_layer_collection(context.view_layer.layer_collection, p_new_collection)
                    if p_new_view_lay_col:
                        context.view_layer.active_layer_collection = p_new_view_lay_col
                        if not context.active_object and len(p_new_view_lay_col.collection.all_objects) != 0:
                            context.view_layer.objects.active = p_new_view_lay_col.collection.all_objects[0]
                except Exception as e:
                    Log.warn(e)
                    pass

                return {'FINISHED'}

        return {'CANCELLED'}

    @classmethod
    def get_collection_center(self, context, p_collection, props, skip_objects=set()):
        all_objects = set()
        if props.calculate_col_center_calculate_center == '1':
            all_objects = set(p_collection.objects)
        elif props.calculate_col_center_calculate_center == '2':
            all_objects = set(p_collection.all_objects)

        all_objects.difference_update(skip_objects)

        obj_count = len(all_objects)
        if obj_count != 0:
            pos = []
            rot = []
            sca = []
            for obj in all_objects:
                mtx = None
                if obj.name == p_collection.name:
                    return obj.matrix_world.copy()

                if mtx is None:
                    mtx = obj.matrix_world
                it_pos, it_rot, it_sca = mtx.decompose()
                pos.append(it_pos)
                rot.append(it_rot.to_exponential_map())
                sca.append(it_sca)

            pos = sum(pos, mathutils.Vector()) / obj_count

            if props.calculate_col_center_calculate_center_with_rot:
                rot = sum(rot, mathutils.Vector()) / obj_count
                mtx_rot = mathutils.Quaternion(rot).to_matrix().to_4x4()
            else:
                mtx_rot = mathutils.Matrix()

            if props.calculate_col_center_calculate_center_with_scale:
                sca = sum(sca, mathutils.Vector()) / obj_count
                mtx_sca = mathutils.Matrix.Scale(sca.magnitude, 4)
            else:
                mtx_sca = mathutils.Matrix()

            return mathutils.Matrix.Translation(pos) @ mtx_rot @ mtx_sca
        else:
            if props.calculate_col_center_default_position == '1':
                return context.scene.cursor.matrix.copy()

            return mathutils.Matrix()

    @classmethod
    def execute_ConvertCollectionToObject(self, operator, context):
        try:
            self.check_validate_list(context)

            ZenLocks.lock_depsgraph_update()

            bpy.ops.object.select_all(action='DESELECT')

            p_group_pair = self.get_current_group_pair(context)
            if p_group_pair:
                idx, p_group = p_group_pair
                p_active_collection = p_group.collection
                p_active_parent_collection = p_group.parent_collection

                # special case for SceneCollection
                p_root_collection = context.scene.collection
                if p_active_collection == context.scene.collection:
                    p_active_parent_collection = p_root_collection

                    # special case if root has only 1 child
                    if len(p_active_collection.objects) == 0 and len(p_active_collection.children) == 1:
                        p_active_collection = p_active_collection.children[0]
                    else:
                        p_active_collection = bpy.data.collections.new('Scene Collection Object')
                        p_root_collection.children.link(p_active_collection)
                        # simple relink collections
                        for it_col in p_root_collection.children:
                            if it_col != p_active_collection:
                                p_root_collection.children.unlink(it_col)
                                p_active_collection.children.link(it_col)
                        for it_obj in p_root_collection.objects:
                            p_root_collection.objects.unlink(it_obj)
                            p_active_collection.objects.link(it_obj)

                if p_active_collection:
                    collection_chain = []

                    snap_dict = {}

                    addon_prefs = get_prefs()

                    def _iterate_children(p_collection, p_collection_parent, p_collection_chain):
                        snap_dict[p_collection.name] = self.get_collection_center(
                            context, p_collection, addon_prefs.op_options)
                        p_collection_chain.append((p_collection, p_collection_parent))
                        for p_child in p_collection.children:
                            _iterate_children(p_child, p_collection, p_collection_chain)

                    _iterate_children(p_active_collection, p_active_parent_collection, collection_chain)
                    collection_chain.reverse()

                    were_objects_map = {}
                    for p_obj in p_active_collection.all_objects:
                        p_obj.zen_tag = str(uuid.uuid1())
                        were_objects_map[p_obj.zen_tag] = p_obj.name

                    instance_obj = None
                    for p_collection, p_collection_parent in collection_chain:
                        try:
                            instance_obj = p_collection.objects.get(p_collection.name)
                            if not instance_obj:
                                instance_obj = bpy.data.objects.new(name=p_collection.name, object_data=None)
                                instance_obj.empty_display_size = operator.empty_display_size
                                instance_obj.empty_display_type = operator.empty_display_type
                            else:
                                p_collection.objects.unlink(instance_obj)
                                snap_dict[p_collection.name] = instance_obj.matrix_world.copy()
                                instance_obj.matrix_world = mathutils.Matrix()

                            p_collection_parent.objects.link(instance_obj)

                            instance_obj.select_set(True)
                            instance_obj.zen_color = p_collection.zen_color

                            instance_obj.zen_tag = str(uuid.uuid1())
                            were_objects_map[instance_obj.zen_tag] = p_collection.name

                            for p_obj in p_collection.objects:
                                if p_obj != instance_obj:
                                    p_collection.objects.unlink(p_obj)
                                    p_collection_parent.objects.link(p_obj)
                                    if p_obj.parent is None:
                                        p_obj.parent = instance_obj
                                        p_obj.select_set(True)

                            if p_collection_parent:
                                bpy.data.collections.remove(p_collection)
                        except Exception as e:
                            err_msg = str(e)
                            operator.report({'ERROR'}, 'Can not convert: ' + err_msg)

                    if instance_obj:
                        context.view_layer.objects.active = instance_obj

                        for k_uuid, v_mtx in snap_dict.items():
                            try:
                                p_obj = context.view_layer.objects[k_uuid]
                                was_children = set((p_obj, p_obj.matrix_world.copy().freeze()) for p_obj in p_obj.children)
                                for p_child_obj, _ in was_children:
                                    p_child_obj.parent = None

                                p_obj.matrix_world = v_mtx

                                for p_child_obj, p_mtx in was_children:
                                    p_child_obj.parent = p_obj
                                    p_child_obj.matrix_world = p_mtx

                            except Exception as e:
                                Log.error(e)

                        if not operator.select:
                            bpy.ops.object.select_all(action='DESELECT')
                        instance_obj.select_set(True)
        finally:
            ZenLocks.unlock_depsgraph_update()
            self.update_collections(context.scene)

        return {'FINISHED'}

    @classmethod
    def select_group_pair(self, p_group_pair, operator, context):
        idx, p_group = p_group_pair
        p_collection = p_group.collection
        if p_collection:
            # p_lay_col = self.get_internal_layer_collection(context, idx)
            # self.unlock_layer_collection(p_lay_col)

            b_nested = operator.nested

            bpy.ops.object.select_all(action='DESELECT')
            p_objects = p_collection.all_objects if b_nested else p_collection.objects
            for p_obj in p_objects:
                try:
                    if p_obj.visible_get():
                        p_obj.select_set(True)
                except Exception:
                    pass

    @classmethod
    def execute_SelectOnlyGroup(self, operator, context):
        self.check_validate_list(context)

        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            self.select_group_pair(p_group_pair, operator, context)
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'FINISHED'}

    @classmethod
    def select_append_group_pair(self, p_group_pair, operator, context):
        idx, p_group = p_group_pair
        p_collection = p_group.collection
        if p_collection:
            # p_lay_col = self.get_internal_layer_collection(context, idx)
            # self.unlock_layer_collection(p_lay_col)

            p_objects = p_collection.all_objects if operator.nested else p_collection.objects
            for p_obj in p_objects:
                try:
                    if p_obj.visible_get():
                        p_obj.select_set(True)
                except Exception:
                    pass

    @classmethod
    def execute_SelectAppendGroup(self, operator, context):
        self.check_validate_list(context)

        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            self.select_append_group_pair(p_group_pair, operator, context)
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'FINISHED'}

    @classmethod
    def deselect_group_pair(self, p_group_pair, operator, context):
        if p_group_pair:
            p_collection = p_group_pair[1].collection
            if p_collection:
                p_objects = p_collection.all_objects if operator.nested else p_collection.objects
                p_selected = set(context.selected_objects).intersection(p_objects)
                for p_obj in p_selected:
                    try:
                        p_obj.select_set(False)
                    except Exception:
                        pass

    @classmethod
    def execute_DeselectGroup(self, operator, context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            self.deselect_group_pair(p_group_pair, operator, context)
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'FINISHED'}

    @classmethod
    def execute_IntersectGroup(self, operator, context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            p_collection = p_group_pair[1].collection
            if p_collection:
                p_objects = p_collection.all_objects if operator.nested else p_collection.objects

                p_intersection = set(context.selected_objects).intersection(p_objects)

                bpy.ops.object.select_all(action='DESELECT')

                for p_obj in p_intersection:
                    try:
                        p_obj.select_set(True)
                    except Exception as e:
                        Log.error(e)
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'FINISHED'}

    @classmethod
    def are_all_group_objects_selected(self, context):
        p_scene = context.scene
        p_group = self._get_scene_group(p_scene)
        if not p_group:
            return True

        p_collection = p_group.collection
        if not p_collection:
            return True

        if not p_collection.all_objects:
            return False

        return set(p_collection.all_objects).issubset(context.selected_objects)

    @classmethod
    def _do_select_scene_objects_in_group(self, operator: bpy.types.Operator, context: bpy.types.Object, fn_condition=None):
        self.check_validate_list(context)
        p_collection = self.get_active_collection(context)
        if not p_collection:
            return {'CANCELLED'}

        s_collection_name = p_collection.name

        bpy.ops.object.select_all(action='DESELECT')

        for obj in p_collection.all_objects:
            try:
                if fn_condition:
                    if not fn_condition(context, obj):
                        continue

                if obj.hide_select:
                    if not operator.prop_make_selectable:
                        continue
                    else:
                        obj.hide_select = False
                if obj.hide_get():
                    if not operator.prop_unhide:
                        continue
                    else:
                        obj.hide_set(False)

                # --- condition when Collection is excluded or hidden ---
                if operator.prop_make_selectable and operator.prop_unhide:
                    ensure_object_in_viewlayer(obj, context.view_layer.layer_collection)
                    obj.select_set(True)
            except Exception as e:
                Log.error(e)
                continue

        if len(context.selected_objects) != 0:
            if not context.active_object or context.active_object not in context.selected_objects:
                context.view_layer.objects.active = context.selected_objects[0]
        else:
            operator.report({'INFO'}, f'Nothing was selected in - {s_collection_name} !')

        return {'FINISHED'}

    @classmethod
    def execute_SelectSceneObjectsInGroup(self, operator, context):
        return self._do_select_scene_objects_in_group(operator, context)

    @classmethod
    def execute_RenameGroups(self, operator, context):
        self.check_validate_list(context)
        if operator.replace == '' and operator.find == '':
            operator.report({'WARNING'}, 'Nothing was defined to replace!')
            return {'CANCELLED'}

        i_start = operator.start_from
        is_modified = False

        handled_collections = set()
        handled_collections.add(context.scene.collection)

        try:
            for idx, p_group in self.get_current_group_pairs(context):
                p_collection = p_group.collection
                if p_collection is None:
                    continue

                if p_collection in handled_collections:
                    continue
                else:
                    handled_collections.add(p_collection)

                lay_col = self.get_internal_layer_collection(context, idx)
                if lay_col and not lay_col.exclude:
                    if operator.group_mode == 'VISIBLE':
                        if lay_col.hide_viewport:
                            continue

                    new_name = p_group.name
                    if operator.find != '':
                        if operator.match_case:
                            new_name = new_name.replace(operator.find, operator.replace)
                        else:
                            new_name = ireplace(new_name, operator.find, operator.replace)
                    else:
                        new_name = operator.replace

                    if operator.use_counter:
                        new_name = new_name + str(i_start)
                        i_start += 1

                    if p_group.name != new_name:
                        p_group.name = new_name
                        is_modified = True
        except Exception as e:
            operator.report({'ERROR'}, str(e))
            is_modified = True

        if is_modified:
            bpy.ops.ed.undo_push(message=operator.bl_label)
        else:
            operator.report({'WARNING'}, 'No replace matches found!')

        return {'FINISHED'}

    @classmethod
    def append_to_group_pair(self, p_group_pair, operator, context):
        move = operator.move
        keep_nested = operator.keep_nested
        p_cur_idx, p_cur_group = p_group_pair
        p_current_lay_col = self.get_internal_layer_collection(context, p_cur_idx)
        was_active_obj = context.view_layer.objects.active
        if p_current_lay_col:
            try:
                ZenLocks.lock_depsgraph_update()

                root_lay_col = context.view_layer.layer_collection

                target_lay_col = p_current_lay_col

                sel_objs = set(context.selected_objects)

                if keep_nested:

                    target_all_objs = set(target_lay_col.collection.all_objects)

                    target_lay_col_children = set(target_lay_col.children)

                    def iterate_selected(p_lay_col, p_lay_col_parent):
                        if p_lay_col != target_lay_col and not p_lay_col.exclude:
                            p_collection = p_lay_col.collection

                            all_collection_in_state = (
                                self.is_collection_selected(p_collection)
                                if p_lay_col != root_lay_col else False)

                            def process_layer_collection(col_parent, lay_col):
                                if lay_col in target_lay_col_children:
                                    return

                                if target_all_objs.issuperset(lay_col.collection.all_objects):
                                    return

                                if self.is_child_of_collection(lay_col, target_lay_col):
                                    return

                                try:
                                    states = None
                                    if move:
                                        states = self.unlink_collection(col_parent, lay_col)

                                    self.link_with_unlock(target_lay_col, lay_col.collection, states)
                                except Exception:
                                    pass

                            def process_objects(col):
                                for it_obj in sel_objs.intersection(col.objects):
                                    try:
                                        target_lay_col.collection.objects.link(it_obj)
                                    except Exception:
                                        pass

                                    if move:
                                        col.objects.unlink(it_obj)

                            if all_collection_in_state:
                                process_layer_collection(p_lay_col_parent.collection, p_lay_col)
                            else:
                                for it_lay_col in p_lay_col.children:
                                    iterate_selected(it_lay_col, p_lay_col)

                                if p_lay_col != target_lay_col:
                                    process_objects(p_collection)

                    """ Process Nested """
                    iterate_selected(root_lay_col, None)
                else:
                    self.relink_objects(root_lay_col, sel_objs, target_lay_col, move)

                self._add_draw_handler()
            finally:
                ZenLocks.unlock_depsgraph_update()

                p_scene = context.scene
                self.update_collections(p_scene)
                try:
                    if not p_current_lay_col.exclude:
                        context.view_layer.active_layer_collection = p_current_lay_col
                    else:
                        idx = self._index_of_layer_collection(context.view_layer, p_current_lay_col)
                        if idx != -1 and idx != self.get_list_index(p_scene):
                            self.set_list_index(p_scene, idx)

                    context.view_layer.objects.active = was_active_obj
                    if not context.view_layer.objects.active:
                        if context.selected_objects:
                            context.view_layer.objects.active = context.selected_objects[0]
                except Exception:
                    pass

    @classmethod
    def execute_DuplicateCollection(self, operator, context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            idx, group = p_group_pair
            p_collection = group.collection
            if p_collection and p_collection != context.scene.collection:
                bpy.ops.object.select_all(action='DESELECT')
                p_new_col = copy_collection(group.parent_collection, p_collection, linked=operator.linked)
                p_parent_lay_col = self.get_internal_layer_collection(context, group.parent_idx)
                if p_parent_lay_col:
                    p_new_lay_col = self.get_layer_collection(p_parent_lay_col, p_new_col)
                    if p_new_lay_col:
                        context.view_layer.active_layer_collection = p_new_lay_col
                bpy.ops.wm.tool_set_by_id(name='builtin.move')
                return {'FINISHED'}
        return {'CANCELLED'}

    @classmethod
    def execute_DuplicateAsInstance(self, operator, context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            idx, group = p_group_pair
            p_collection = group.collection
            p_collection_parent = group.parent_collection
            if p_collection and p_collection_parent and p_collection != context.scene.collection:
                bpy.ops.object.select_all(action='DESELECT')

                instance_obj = bpy.data.objects.new(name=p_collection.name, object_data=None)
                instance_obj.instance_collection = p_collection
                instance_obj.instance_type = 'COLLECTION'

                p_collection_parent.objects.link(instance_obj)

                context.view_layer.objects.active = instance_obj
                instance_obj.select_set(True)

                bpy.ops.object.origin_clear()

                # Optional
                # bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY')

                bpy.ops.wm.tool_set_by_id(name='builtin.move')

                return {'FINISHED'}
        return {'CANCELLED'}

    @classmethod
    def execute_DeleteHierarchy(self, operator, context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            _, group = p_group_pair
            p_collection = group.collection
            if p_collection and p_collection != context.scene.collection:
                bpy.data.collections.remove(p_collection)
                return {'FINISHED'}
        return {'CANCELLED'}

    @classmethod
    def sort_collection(self, lay_col: bpy.types.LayerCollection, p_handled_cols: Set, props):
        from ...human_sort import human_sort_copy

        if lay_col is None:
            return

        p_collection = lay_col.collection

        if p_collection in p_handled_cols:
            return

        p_handled_cols.add(p_collection)

        p_children_size = len(lay_col.children)
        if p_children_size == 0:
            return

        children = list(lay_col.children)

        if props.mode == 'SHUFFLE':
            import random
            random.shuffle(children)
        else:
            children = human_sort_copy(
                children,
                'name',
                case_insensitive=props.case_insensitive,
                suffix=props.suffix,
                prefix=props.prefix,
                mode=props.mode
            )

            if props.reversed:
                children = reversed(children)

        b_modified = False

        for p_lay_col_child in children:
            if props.nested:
                if self.sort_collection(p_lay_col_child, p_handled_cols, props):
                    b_modified = True

            if p_children_size > 1:
                p_child = p_lay_col_child.collection
                states = self.unlink_collection(lay_col.collection, p_lay_col_child)
                self.link_collection(lay_col, p_child, states)
                b_modified = True

        return b_modified

    @classmethod
    def execute_SortCollections(self, operator: bpy.types.Operator, context: bpy.types.Context):

        if is_draw_handler_enabled(self):
            remove_draw_handler(self, context)

        bpy.ops.ed.undo_push(message='Fix Sort List Index')

        self.check_validate_list(context)

        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            b_need_update = False
            idx, p_group = p_group_pair
            try:
                ZenLocks.lock_depsgraph_update()

                p_cur_lay_col = self.get_internal_layer_collection(context, idx)
                if p_cur_lay_col:
                    p_handled_collections = set()
                    if self.sort_collection(p_cur_lay_col, p_handled_collections, operator):
                        b_need_update = True
            finally:
                ZenLocks.unlock_depsgraph_update()

                if b_need_update:
                    self.update_collections(context.scene)

                    return {'FINISHED'}
                else:
                    operator.report({'INFO'}, f'Nothing to sort in Collection: {p_group.name}')
        return {'CANCELLED'}

    @classmethod
    def export_to_json(self, context, props, active_layer_name=''):
        json_data = {}

        json_data['mode'] = self.id_group
        json_data['element'] = self.id_element
        json_data['unique'] = self.is_unique

        p_scene = context.scene

        start_progress(context)
        i_count = 0

        json_data['groups'] = {}
        p_list = self.get_list(p_scene)
        i_max = len(p_list)
        json_data['progress_max'] = i_max
        for group in p_list:
            i_count = i_count + 1
            update_progress(context, int(i_count / i_max * 100))
            if active_layer_name != '':
                if active_layer_name != group.layer_name:
                    continue
            else:
                if not self.string_filter_match(group.name, props):
                    continue

            p_collection = group.collection
            if p_collection and len(p_collection.objects) != 0:
                group_data = {}
                group_data['name'] = group.name
                group_data['color'] = (group.group_color.r, group.group_color.g, group.group_color.b)
                group_data['items'] = [obj.name for obj in p_collection.objects]
                json_data['groups'][group.layer_name] = group_data
        end_progress(context)
        # DO NOT USE IDENT !!!
        return json.dumps(json_data)

    @classmethod
    def import_from_json(self, context: bpy.types.Context, json_data):
        json_data = json.loads(json_data)

        try:
            start_progress(context)

            self.check_validate_list(context)

            ZenLocks.lock_depsgraph_update()

            i_max = json_data['progress_max']

            i_count = 0

            p_scene = context.scene

            min_parent = context.view_layer.active_layer_collection
            if min_parent is None:
                min_parent = context.view_layer.layer_collection

            root_lay_col = context.view_layer.layer_collection

            p_last_lay_col = context.view_layer.active_layer_collection

            for key, val in json_data['groups'].items():
                i_count = i_count + 1
                update_progress(context, int(i_count / i_max * 100))
                # layerName = key.replace(json_data['mode'], self.id_group)
                groupName = val['name']
                groupColor = mathutils.Color(val['color'])

                p_objects = set()
                for obj_name in val['items']:
                    p_obj = context.scene.objects.get(obj_name)
                    if p_obj is not None:
                        p_objects.add(p_obj)
                if len(p_objects) == 0:
                    continue

                new_collection = bpy.data.collections.get(groupName)
                if new_collection is None:
                    if groupName == 'Scene Collection':
                        new_collection = context.view_layer.layer_collection.collection
                        new_lay_col = context.view_layer.layer_collection
                    else:
                        new_collection = bpy.data.collections.new(groupName)
                        min_parent.collection.children.link(new_collection)
                        new_lay_col = self.get_layer_collection(min_parent, new_collection)
                else:
                    min_parent = context.view_layer.layer_collection
                    new_lay_col = self.get_layer_collection(min_parent, new_collection)

                new_collection.zen_color = groupColor

                move = True

                self.relink_objects(root_lay_col, p_objects, new_lay_col, move)

                p_last_lay_col = new_lay_col

            if p_last_lay_col is not None and p_last_lay_col != context.view_layer.active_layer_collection:
                context.view_layer.active_layer_collection = p_last_lay_col
        finally:
            end_progress(context)

            ZenLocks.unlock_depsgraph_update()

        self.update_collections(p_scene)

        bpy.ops.ed.undo_push(message='Import Groups')

    @classmethod
    def _do_draw_overlay_filter(self, context, layout):
        addon_prefs = get_prefs()
        layout.prop(addon_prefs.common, 'object_shading')
        layout.prop(addon_prefs.common, 'display_all_parts', text='All Collections')

        addon_prefs.object_options.draw_collection_captions(layout)

        layout.prop(addon_prefs.object_options, 'sync_with_collection_color_tag')
        layout.prop(addon_prefs.object_options, 'object_display_transform_affect_options')

        layout.operator('zsts.reset_draw_cache', icon='FILE_REFRESH')

    @classmethod
    def get_draw_panel_header(self, context):
        custom_text = ''
        p_scene = context.scene
        addon_prefs = get_prefs()
        p_scene_list = self.get_list(p_scene)
        total_count = len(p_scene_list)
        if addon_prefs.list_options.collection_list_mode == 'SEL_OBJS_PARENTS':
            filters = self.get_selected_parents_filter(context, p_scene_list)
            n_visible_groups = total_count - len(filters)
        elif addon_prefs.list_options.collection_list_mode == 'SEL_OBJS':
            filters = self.get_selected_objects_filter(context, p_scene_list)
            n_visible_groups = total_count - len(filters)
        elif addon_prefs.list_options.collection_list_mode == 'SEL_COLLECTIONS':
            filters = self.get_selected_collections_filter(context, p_scene_list)
            n_visible_groups = total_count - len(filters)
        else:
            all_layers = [it.exclude for it, _ in self._iterate_layer_tree(context.view_layer.layer_collection, None)]
            n_visible_groups = all_layers.count(False)
            total_count = len(all_layers)
        if n_visible_groups != total_count:
            custom_text = f' - {n_visible_groups} of {total_count} Group(s)'

        if custom_text == '' and total_count > 0:
            custom_text = f' - {total_count} Group(s)'

        return custom_text

    @classmethod
    def _do_draw_collection_toolbar(self, context: bpy.types.Context, layout, is_tool_header):
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            idx, p_group = p_group_pair
            lay_col = self.get_internal_layer_collection(context, idx)
            p_collection = p_group.collection
            if lay_col and p_collection:
                row = layout.row(align=True)

                p_all_objects = p_collection.all_objects
                p_all_objects_count = len(p_all_objects)

                t_info = {}

                if is_tool_header:
                    row.alignment = 'LEFT'
                    t_info = ui_select_object_operator(row, context, p_collection)
                    op = t_info.get("op")
                    op.group_index = idx
                    row.separator()
                else:
                    row.alignment = 'LEFT'

                row = row.box()
                row = row.row(align=True)
                if not is_tool_header:
                    t_info = ui_select_object_operator(row, context, p_collection)
                    op = t_info.get("op")
                    op.group_index = idx
                    row.separator()

                all_included = all(
                    not it_lay_col.exclude
                    for it_lay_col, _ in self._iterate_layer_tree(lay_col, None)
                    if it_lay_col != lay_col)
                subrow = row
                if not all_included:
                    subrow = row.row(align=True)
                    subrow.alert = True
                icon_exclude = 'CHECKBOX_DEHLT' if lay_col.exclude else 'CHECKBOX_HLT'
                op = subrow.operator('zsto.exclude_group_by_index', icon=icon_exclude, text='', emboss=False)
                op.group_index = idx

                all_visible = all(
                    not it_lay_col.hide_viewport
                    for it_lay_col, _ in self._iterate_layer_tree(lay_col, None)
                    if it_lay_col != lay_col)
                subrow = row
                if not all_visible:
                    subrow = row.row(align=True)
                    subrow.alert = True
                icon_hide_id = 'HIDE_ON' if lay_col.hide_viewport else 'HIDE_OFF'

                b_isolated = False
                if p_collection and p_collection != context.scene.collection:
                    wm = context.window_manager
                    op_props = wm.operator_properties_last('zsto.hide_viewport_by_index')

                    has_changed, has_elements = ZsHideViewportCollectionProperty.isolate_group_pair_property_state(
                        self, context, p_group_pair, op_props, fake=True)
                    b_isolated = not has_changed and has_elements
                op = subrow.operator(
                    'zsto.hide_viewport_by_index',
                    text='', icon=icon_hide_id, emboss=b_isolated, depress=b_isolated)
                op.group_index = idx

                all_selectable = all(
                    not it_col.hide_select
                    for it_col, _ in self._iterate_layer_tree(p_collection, None)
                    if it_col != p_collection)
                subrow = row
                if all_selectable:
                    if p_all_objects_count > 0:
                        arr = np.empty(p_all_objects_count, 'b')
                        p_all_objects.foreach_get('hide_select', np.reshape(arr, p_all_objects_count))
                        all_selectable = not np.any(arr)
                if not all_selectable:
                    subrow = row.row(align=True)
                    subrow.alert = True
                icon_select = 'RESTRICT_SELECT_ON' if p_collection.hide_select else 'RESTRICT_SELECT_OFF'
                op = subrow.operator('zsto.hide_select_by_index', text='', icon=icon_select, emboss=False)
                op.group_index = idx

                all_selectable = all(
                    not it_col.hide_viewport
                    for it_col, _ in self._iterate_layer_tree(p_collection, None)
                    if it_col != p_collection)
                subrow = row
                if all_selectable:
                    if p_all_objects_count > 0:
                        arr = np.empty(p_all_objects_count, 'b')
                        p_all_objects.foreach_get('hide_viewport', np.reshape(arr, p_all_objects_count))
                        all_selectable = not np.any(arr)
                if not all_selectable:
                    subrow = row.row(align=True)
                    subrow.alert = True
                icon_select = 'RESTRICT_VIEW_ON' if p_collection.hide_viewport else 'RESTRICT_VIEW_OFF'
                op = subrow.operator('zsto.disable_viewport_by_index', text='', icon=icon_select, emboss=False)
                op.group_index = idx

                all_selectable = all(
                    not it_col.hide_render
                    for it_col, _ in self._iterate_layer_tree(p_collection, None)
                    if it_col != p_collection)
                if all_selectable:
                    if p_all_objects_count > 0:
                        arr = np.empty(p_all_objects_count, 'b')
                        p_all_objects.foreach_get('hide_render', np.reshape(arr, p_all_objects_count))
                        all_selectable = not np.any(arr)
                subrow = row
                if not all_selectable:
                    subrow = row.row(align=True)
                    subrow.alert = True
                icon_select = 'RESTRICT_RENDER_ON' if p_collection.hide_render else 'RESTRICT_RENDER_OFF'
                op = subrow.operator('zsto.hide_render_by_index', text='', icon=icon_select, emboss=False)
                op.group_index = idx

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        layout = menu.layout

        layout.operator("zsts.select_scene_objects")
        layout.menu('ZSTO_MT_SelectObjectsWith')
        layout.operator("zsto.colorize_selected")

        layout.separator()
        layout.operator("zsts.rename_groups")
        layout.operator("zsto.rename_as_parent_collection")
        layout.operator("zsto.rename_collection_as_object")

        layout.separator()
        layout.operator('zsto.duplicate_collection')
        layout.operator('zsto.duplicate_collection_linked')
        layout.operator('zsto.duplicate_as_intance')

        layout.separator()
        layout.operator('zsto.convert_object_to_collection')
        layout.operator('zsto.convert_collection_to_object')

        layout.separator()
        layout.operator('object.zsto_sort_collections')

    @classmethod
    def execute_DrawTools(cls, tools, context):
        layout = tools.layout

        col = layout.column(align=True)
        col.operator("zsts.rename_groups")
        col.operator("zsto.rename_as_parent_collection", text='Rename Object(s) as Collection')
        col.operator("zsto.rename_collection_as_object", text='Rename Collection as Object')

        col = layout.column(align=True)
        col.operator('zsto.duplicate_collection')
        col.operator('zsto.duplicate_collection_linked')
        col.operator('zsto.duplicate_as_intance')

        from .collection_interface import (
            ZSTO_OT_ConvertCollectionToParentObject,
            ZSTO_OT_ConvertObjectToCollection
        )

        col = layout.column(align=False)
        row = col.row(align=True)
        row_op = row.row(align=True)
        row_op.enabled = ZSTO_OT_ConvertObjectToCollection.fix_poll(context)
        row_op.operator('zsto.convert_object_to_collection')
        row_pop = row.row(align=True)
        row_pop.popover(
            panel="ZSTO_PT_ParentToColProps",
            text="",
            icon='DOWNARROW_HLT')

        row = col.row(align=True)
        row_op = row.row(align=True)
        row_op.enabled = ZSTO_OT_ConvertCollectionToParentObject.fix_poll(context)
        row_op.operator('zsto.convert_collection_to_object')
        row_pop = row.row(align=True)
        row_pop.popover(
            panel="ZSTO_PT_ColToParentProps",
            text="",
            icon='DOWNARROW_HLT')

        layout.operator('object.zsto_sort_collections')

    @classmethod
    def execute_DrawImport(cls, layout, context, is_menu=False):
        pass

    @classmethod
    def execute_DrawExport(cls, layout, context, is_menu=False):
        pass


class ZsCollectionLayerManager(ZsObjectLayerManager):
    id_group = 'obj_sets'
    id_mask = 'ZSTO'
    is_unique = True
    id_element = 'object'

    list_item_prefix = 'Collection'

    @classmethod
    def get_cacher(self):
        return BasicObjectsCacher()

    @classmethod
    def get_highlighted_objects(self, context, p_group):
        p_scene = context.scene
        if p_group.collection and p_group.collection != p_scene.collection:
            return set(context.visible_objects).intersection(p_group.collection.objects)
        return set()


class ZSTO_UL_List(ZsBaseObjectUIList, ZsCollectionLayerManager):
    pass


class ZSTO_Factory:
    classes = (
        ZSTO_UL_List,
        ZSTSCollectionListGroup,
    )

    def get_mgr():
        return ZsCollectionLayerManager

    def get_ui_list():
        return ZSTSCollectionListGroup
