# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

from typing import Tuple

from ...blender_zen_utils import ZenLocks


class BasicCollectionProperty:

    prop_name = ''
    is_layer = False
    has_object_property = True
    available_only = True

    @classmethod
    def _do_set(cls, p_col, prop_disable, fake=False):
        was_set = getattr(p_col, cls.prop_name)
        if was_set != prop_disable:
            if not fake:
                setattr(p_col, cls.prop_name, prop_disable)
            return True
        return False

    @classmethod
    def _do_set_object_property(cls, p_obj, prop_disable, fake=False):
        return cls._do_set(p_obj, prop_disable, fake=fake)

    @classmethod
    def _do_finalize_lay_col(cls, context, p_lay_col, props, prop_state):
        pass

    @classmethod
    def _do_get_object_property(cls, p_obj):
        return getattr(p_obj, cls.prop_name)

    """ PUBLIC METHODS """

    @classmethod
    def set_group_pair_property_state(cls, p_cls_mgr, context, p_group_pair, prop_disable, props, set_layer=True):
        if p_group_pair:
            idx, p_group = p_group_pair
            p_lay_col = p_cls_mgr.get_internal_layer_collection(context, idx)
            if p_lay_col:

                p_col = p_lay_col if cls.is_layer else p_lay_col.collection

                b_nested = getattr(props, 'nested', True)

                has_changed = False
                root_lay_col = context.view_layer.layer_collection
                is_root = p_lay_col == root_lay_col

                has_elements = ((len(root_lay_col.children) or len(root_lay_col.collection.objects))
                                if is_root else True)

                if prop_disable:
                    if is_root:
                        if b_nested:
                            for it in root_lay_col.children:
                                if not it.exclude or not cls.available_only:
                                    it_col = it if cls.is_layer else it.collection
                                    if cls._do_set(it_col, prop_disable):
                                        has_changed = True
                        if cls.has_object_property:
                            for it_obj in p_lay_col.collection.objects:
                                if cls._do_set_object_property(it_obj, prop_disable):
                                    has_changed = True
                    else:
                        if cls._do_set(p_col, prop_disable):
                            has_changed = True
                else:
                    if not is_root:
                        if cls._do_set(p_col, prop_disable):
                            has_changed = True
                    else:
                        if not b_nested:
                            if cls.has_object_property:
                                for it_obj in p_lay_col.collection.objects:
                                    if cls._do_set_object_property(it_obj, prop_disable):
                                        has_changed = True

                    if b_nested:
                        parents = set(p_cls_mgr.get_parents(
                            context, p_group_pair,
                            include_root=False, available_only=cls.available_only))
                        children = set(p_cls_mgr.get_children(
                            p_lay_col, available_only=cls.available_only))
                        all_items = parents.union(children)

                        for it in all_items:
                            if it != root_lay_col and (not it.exclude or not cls.available_only):
                                it_col = it if cls.is_layer else it.collection
                                if cls._do_set(it_col, prop_disable):
                                    has_changed = True

                        if cls.has_object_property:
                            children.add(p_lay_col)
                            all_objects = set(
                                it_obj for it_col in children for it_obj in it_col.collection.objects)
                            for it_obj in all_objects:
                                if cls._do_set_object_property(it_obj, prop_disable):
                                    has_changed = True

                if set_layer:
                    try:
                        context.view_layer.active_layer_collection = p_lay_col
                    except Exception:
                        pass

                cls._do_finalize_lay_col(context, p_lay_col, props, prop_disable)

                return (has_changed, has_elements)

        return (False, False)

    @classmethod
    def isolate_group_pair_property_state(cls, p_cls_mgr, context, p_group_pair, props, fake=False) -> Tuple[bool, bool]:
        if p_group_pair:
            idx, p_group = p_group_pair
            p_lay_col = p_cls_mgr.get_internal_layer_collection(context, idx)
            if p_lay_col:
                b_nested = getattr(props, 'nested', True)

                has_changed = False

                root_lay_col = context.view_layer.layer_collection

                parents = set(p_cls_mgr.get_parents(
                    context, p_group_pair, include_root=False, available_only=cls.available_only))
                children = set(p_cls_mgr.get_children(p_lay_col, available_only=cls.available_only))

                has_elements = ((len(root_lay_col.children) or len(root_lay_col.collection.objects))
                                if p_lay_col == root_lay_col else True)

                for it, _ in p_cls_mgr._iterate_layer_tree(root_lay_col, None):
                    it_col = it if cls.is_layer else it.collection
                    is_self = it == p_lay_col
                    is_root = it == root_lay_col
                    is_in_child = b_nested and it in children
                    is_relative = it in parents or is_in_child
                    is_available = not it.exclude or not cls.available_only

                    if is_self or (is_available and not is_root):
                        b_isolate = is_self or is_relative
                        if cls._do_set(it_col, not b_isolate, fake=fake):
                            has_changed = True
                            if has_changed and fake:
                                break

                    if cls.has_object_property:
                        if is_self or is_root or (is_available and is_relative):
                            b_isolate = is_self or (is_available and is_in_child)
                            for it_obj in it.collection.objects:
                                if cls._do_set_object_property(it_obj, not b_isolate, fake=fake):
                                    has_changed = True
                                    if has_changed and fake:
                                        break

                if not fake:
                    cls._do_finalize_lay_col(context, p_lay_col, props, False)

                    if context.view_layer.active_layer_collection != p_lay_col:
                        try:
                            context.view_layer.active_layer_collection = p_lay_col
                        except Exception:
                            pass

                return (has_changed, has_elements)

        return (False, True)

    @classmethod
    def are_all_unset(cls, p_cls_mgr, context, p_lay_col, props):
        if context.view_layer.layer_collection == p_lay_col:
            b_nested = getattr(props, 'nested', True)
            objects_set = all(
                not cls._do_get_object_property(it_obj)
                for it_obj in p_lay_col.collection.objects) if cls.has_object_property else True
            if b_nested:
                children_set = all(
                    not getattr((it_lay_col if cls.is_layer else it_lay_col.collection), cls.prop_name)
                    for it_lay_col, _ in p_cls_mgr._iterate_layer_tree(p_lay_col, None)
                    if ((not it_lay_col.exclude or not cls.available_only) and it_lay_col != p_lay_col))
                return objects_set and children_set
            else:
                return objects_set
        else:
            p_col = p_lay_col if cls.is_layer else p_lay_col.collection
            return not getattr(p_col, cls.prop_name)

    @classmethod
    def toggle_group_pair(cls, p_cls_mgr, context, p_group_pair, props):
        disable_state = True
        if p_group_pair:
            idx, p_group = p_group_pair
            lay_col = p_cls_mgr.get_internal_layer_collection(context, idx)
            if lay_col:
                disable_state = cls.are_all_unset(p_cls_mgr, context, lay_col, props)
                status = cls.set_group_pair_property_state(
                    p_cls_mgr, context, p_group_pair, disable_state, props)
                return status[0], status[1], disable_state
        else:
            return False, False, disable_state


""" PUBLIC CLASSES """


class ZsHideViewportCollectionProperty(BasicCollectionProperty):
    prop_name = 'hide_viewport'
    is_layer = True

    label_set = 'hide in viewlayer'
    label_unset = 'unhide in viewlayer'
    label_is_set = 'hidden in viewlayer'

    @classmethod
    def _do_set_object_property(cls, p_obj, prop_disable, fake=False):
        was_set = p_obj.hide_get()
        if was_set != prop_disable:
            try:
                if not fake:
                    p_obj.hide_set(prop_disable)
                return True
            except Exception:
                pass
        return False

    @classmethod
    def _do_get_object_property(cls, p_obj):
        return p_obj.hide_get()

    @classmethod
    def _do_finalize_lay_col(cls, context, p_lay_col, props, prop_state):
        if prop_state is False:
            b_select = getattr(props, 'select')
            if b_select:
                b_nested = getattr(props, 'nested', True)
                all_objects = p_lay_col.collection.all_objects if b_nested else p_lay_col.collection.objects
                for it_obj in set(context.selectable_objects).intersection(all_objects):
                    try:
                        it_obj.select_set(True)
                    except Exception:
                        pass


class ZsExcludeCollectionProperty(BasicCollectionProperty):
    prop_name = 'exclude'
    is_layer = True
    has_object_property = False
    available_only = False

    label_set = 'exclude from viewlayer'
    label_unset = 'include in viewlayer'
    label_is_set = 'excluded from viewlayer'

    @classmethod
    def toggle_group_pair(cls, p_cls_mgr, context, p_group_pair, props):
        status = super().toggle_group_pair(p_cls_mgr, context, p_group_pair, props)
        is_disabled = status[2]
        if is_disabled:
            ZenLocks.lock_lay_col_update_one()
        return status


class ZsDisableViewportCollectionProperty(BasicCollectionProperty):
    prop_name = 'hide_viewport'
    is_layer = False

    label_set = 'disable in viewport'
    label_unset = 'enable in viewport'
    label_is_set = 'disabled in viewport'


class ZsHideSelectCollectionProperty(BasicCollectionProperty):
    prop_name = 'hide_select'
    is_layer = False

    label_set = 'hide select in viewport'
    label_unset = 'unhide select in viewport'
    label_is_set = 'hidden select in viewport'


class ZsHideRenderCollectionProperty(BasicCollectionProperty):
    prop_name = 'hide_render'
    is_layer = False

    label_set = 'disable in renders'
    label_unset = 'enable in renders'
    label_is_set = 'disabled in renders'
