# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2022, Alex Zhornyak

import bpy

import json
import mathutils

from typing import Set

from timeit import default_timer as timer

from ...basic_sets import ZsLayerManager
from ...draw_cache import BasicObjectsCacher
from ...draw_sets import (
    is_draw_handler_enabled, remove_draw_handler)

from ....blender_zen_utils import ZenLocks, ZenSelectionStats, update_view3d_in_all_screens, ensure_object_in_viewlayer
from ....progress import start_progress, update_progress, end_progress
from ....preferences import get_prefs
from ....labels import ZsLabels
from ....vlog import Log


class Zs_UL_SimpleObjectList(bpy.types.UIList):
    """ Zen Groups UIList """

    def draw_item(self, context, layout, data, item, icon, active_data, active_propname, index):
        addon_prefs = get_prefs()

        layout.separator(factor=0.1)
        layout.active = True

        col = layout.column(align=False)
        col.ui_units_x = 0.7
        col.separator(factor=0.8)
        col.scale_y = 0.6
        col.prop(item, 'group_color', text='')

        r = layout.split(factor=0.5)
        r.prop(item, 'name', text='', emboss=False, icon='NONE')

        r = r.row()
        r.alignment = 'RIGHT'

        p_objects = None

        if addon_prefs.list_options.display_objects_info:
            if p_objects is None:
                p_objects = item.get_objects(context)
                r.alert = not set(context.selectable_objects).issuperset(p_objects)

            n_obj_count = len(p_objects)

            r.label(text=str(n_obj_count) if n_obj_count else '0')

            if n_obj_count == 0:
                layout.active = False

        if addon_prefs.list_options.display_hidden_groups_info:
            r = r.row()
            r.alignment = 'RIGHT'
            if p_objects is None:
                p_objects = item.get_objects(context)
                r.alert = not set(context.selectable_objects).issuperset(p_objects)

            n_obj_count = len(p_objects)

            if n_obj_count == 0:
                r.label(text='', icon='BLANK1')
            else:
                p_hidden = p_objects - set(context.visible_objects)
                n_hide_count = len(p_hidden)

                icon_hide_id = 'HIDE_OFF' if n_hide_count == 0 else ('HIDE_ON' if n_hide_count == n_obj_count else 'RADIOBUT_OFF')
                op = r.operator('zsts.internal_hide_group_by_index', text='', icon=icon_hide_id, emboss=False)
                op.group_index = index

        if addon_prefs.list_options.display_hide_renders_info:
            if p_objects is None:
                p_objects = item.get_objects(context)

            r = r.row()
            r.alignment = 'RIGHT'

            n_obj_count = len(p_objects)

            if n_obj_count == 0:
                r.label(text='', icon='BLANK1')
            else:
                p_disabled_in_render = set()
                for it_col, _ in self._iterate_layer_tree(context.scene.collection, None):
                    if it_col.hide_render:
                        p_disabled_in_render.update(it_col.all_objects)

                p_disabled_in_render.intersection_update(p_objects)

                p_disabled_in_render_objs = set(p_obj for p_obj in p_objects if p_obj.hide_render)
                n_disable_count = len(p_disabled_in_render_objs)

                r.alert = len(p_disabled_in_render) > 0 or (n_disable_count != 0 and n_disable_count != n_obj_count)

                icon_hide_id = 'RESTRICT_RENDER_OFF' if n_disable_count == 0 else 'RESTRICT_RENDER_ON'
                op = r.operator('zsto.disable_in_renders_by_index', text='', icon=icon_hide_id, emboss=False)
                op.group_index = index

        r.separator(factor=0.1)


class ZsSimpleObjectBaseLayerManager(ZsLayerManager):
    id_group = 'obj_simple_sets'
    id_mask = 'ZSOS'
    is_unique = False
    id_element = 'object'

    list_item_prefix = 'Objects'

    @classmethod
    def get_cacher(self):
        return BasicObjectsCacher()

    @classmethod
    def check_update_list(cls, p_scene):
        pass

    @classmethod
    def _do_draw_collection_toolbar(self, context: bpy.types.Context, layout, is_tool_header):
        pass

    @classmethod
    def build_lookup_table(self, context):
        pass

    @classmethod
    def append_objects_to_group(self, p_objects, p_scene_group):
        Log.error('ABSTRACT> append_objects_to_group')

    @classmethod
    def assign_objects_to_group(self, context, p_objects, p_scene_group):
        Log.error('ABSTRACT> assign_objects_to_group')

    @classmethod
    def _do_draw_overlay_filter(self, context, layout):
        addon_prefs = get_prefs()
        layout.prop(addon_prefs.common, 'object_shading')
        if self.is_unique:
            layout.prop(addon_prefs.common, 'display_all_parts')

        addon_prefs.object_options.draw_collection_captions(layout)

        layout.prop(addon_prefs.object_options, 'object_display_transform_affect_options')

        layout.operator('zsts.reset_draw_cache', icon='FILE_REFRESH')

    @classmethod
    def get_lookup_extra_info_by_index(self, context, index):
        p_list = self.get_list(context.scene)
        if index in range(0, len(p_list)):
            p_objects = p_list[index].get_objects(context)
            n_count = len(p_objects)
            p_hidden = p_objects - set(context.visible_objects)
            n_hide_count = len(p_hidden)

            p_extra_info = {
                'count': n_count,
                'objects': n_count,
                'hide_count': n_hide_count
            }
            return p_extra_info

        return {}

    @classmethod
    def _get_layer_extra_info(self, context, layerName):
        group = self._get_group_by_layer(context.scene, layerName)
        if group is not None:
            p_objects = group.get_objects(context)
            n_count = len(p_objects)
            p_hidden = p_objects - set(context.visible_objects)
            n_hide_count = len(p_hidden)

            p_extra_info = {
                'count': n_count,
                'objects': n_count,
                'hide_count': n_hide_count
            }
            return p_extra_info

        return {}

    @classmethod
    def get_draw_panel_header(self, context):
        p_scene = context.scene
        p_scene_list = self.get_list(p_scene)

        total_count = len(p_scene_list)

        if total_count > 0:
            return f' - {total_count} Group(s)'
        else:
            return ''

    @classmethod
    def update_collections(cls, p_scene):
        pass

    @classmethod
    def ensure_active_object_in_selection(cls, context: bpy.types.Context):
        p_selected_objects = set(context.selected_objects)
        if len(p_selected_objects) != 0 and context.view_layer.objects.active not in p_selected_objects:
            context.view_layer.objects.active = context.selected_objects[0]

    @classmethod
    def get_highlighted_group_pairs(self, context):
        p_group_pair = self.get_current_group_pair(context)
        return [p_group_pair] if p_group_pair else []

    @classmethod
    def get_highlighted_objects(self, context, p_group):
        return set(context.visible_objects).intersection(p_group.get_objects(context))

    @classmethod
    def get_context_selected_count(self, context):
        return len(context.selected_objects)

    @classmethod
    def get_current_group_pairs(self, context):
        return [(idx, g) for idx, g in enumerate(self.get_list(context.scene))]

    @classmethod
    def get_context_group_pairs(self, context):
        return self.get_current_group_pairs(context)

    @classmethod
    def get_current_group_pair(self, context):
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        i_items_count = len(p_list)
        i_list_index = self.get_list_index(p_scene)

        if i_list_index in range(i_items_count):
            return (i_list_index, p_list[i_list_index])
        else:
            return None

    @classmethod
    def get_context_group_pair(self, context):
        return self.get_current_group_pair(context)

    @classmethod
    def is_current_group_active(self, context):
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            return True
        return False

    @classmethod
    def remove_scene_layer(self, p_scene, layerName):
        p_list = self.get_list(p_scene)
        i_layer_index = self._index_of_layer(p_list, layerName)
        if i_layer_index != -1:
            p_list.remove(i_layer_index)
            i_layer_index = i_layer_index - 1

        p_context_group_pairs = self.get_current_group_pairs(bpy.context)

        i_new_index = -1
        for i in range(len(p_context_group_pairs) - 1, -1, -1):
            if p_context_group_pairs[i][0] <= i_layer_index:
                i_new_index = p_context_group_pairs[i][0]
                break

        if i_new_index == -1 and len(p_context_group_pairs):
            i_new_index = p_context_group_pairs[0][0]

        self.set_list_index(p_scene, i_new_index)

    @classmethod
    def delete_group(self, context, layerName):
        p_scene = context.scene

        self.remove_scene_layer(p_scene, layerName)

        p_list = self.get_list(p_scene)
        if len(p_list) == 0 and is_draw_handler_enabled(self):
            remove_draw_handler(self, context)

        bpy.ops.ed.undo_push(message='Delete Group')

    @classmethod
    def export_to_json(self, context, props, active_layer_name=''):
        json_data = {}

        json_data['mode'] = self.id_group
        json_data['element'] = self.id_element
        json_data['unique'] = self.is_unique

        p_scene = context.scene

        start_progress(context)
        i_count = 0

        json_data['groups'] = {}
        p_list = self.get_list(p_scene)
        i_max = len(p_list)
        json_data['progress_max'] = i_max
        for group in p_list:
            i_count = i_count + 1
            update_progress(context, int(i_count / i_max * 100))
            if active_layer_name != '':
                if active_layer_name != group.layer_name:
                    continue
            else:
                if not self.string_filter_match(group.name, props):
                    continue
            group_data = {}
            group_data['name'] = group.name
            group_data['color'] = (group.group_color.r, group.group_color.g, group.group_color.b)
            group_data['items'] = [obj.name for obj in group.get_objects(context)]
            json_data['groups'][group.layer_name] = group_data
        end_progress(context)
        # DO NOT USE IDENT !!!
        return json.dumps(json_data)

    @classmethod
    def import_from_json(self, context, json_data):
        json_data = json.loads(json_data)

        try:
            start_progress(context)

            i_max = json_data['progress_max']
            p_last_group = None
            i_count = 0

            p_scene = context.scene

            for key, val in json_data['groups'].items():
                i_count = i_count + 1
                update_progress(context, int(i_count / i_max * 100))
                layerName = key.replace(json_data['mode'], self.id_group)
                p_group = self.ensure_group_in_scene(context.scene, layerName, val['name'], mathutils.Color(val['color']))
                p_last_group = p_group

                p_objects = set()
                for obj_name in val['items']:
                    p_obj = context.scene.objects.get(obj_name)
                    if p_obj is not None:
                        p_objects.add(p_obj)

                self.assign_objects_to_group(context, p_objects, p_group)

            if p_last_group:
                p_scene_list = self.get_list(p_scene)
                i_list_index = self._index_of_layer(p_scene_list, p_last_group.layer_name)
                self.set_list_index(p_scene, i_list_index)
        finally:
            end_progress(context)

        bpy.ops.ed.undo_push(message='Import Groups')

    @classmethod
    def set_group_pair_hidden_state(self, context, p_group_pair, hidden_state, props):
        b_changed = False
        b_select = getattr(props, 'select', False)
        p_group = p_group_pair[1]
        p_objects = p_group.get_objects(context)
        b_has_elements = len(p_objects) != 0
        for p_obj in p_objects:
            was_visible = p_obj.visible_get()
            try:
                p_obj.hide_set(hidden_state)
                if b_select and not hidden_state:
                    p_obj.select_set(True)
                if p_obj.visible_get() != was_visible:
                    b_changed = True
            except Exception:
                pass

        return b_changed, b_has_elements

    @classmethod
    def unhide_all(self, context, b_select, props):
        if bpy.ops.object.hide_view_clear.poll():
            bpy.ops.object.hide_view_clear(select=b_select)

    @classmethod
    def unhide_in_renders_all(self, context, b_enable_collections):
        if bpy.ops.object.hide_render_clear_all.poll():
            bpy.ops.object.hide_render_clear_all()

        if b_enable_collections:
            p_disabled_in_render = set(
                it_col.collection
                for it_col, _ in self._iterate_layer_tree(context.view_layer.layer_collection, None)
                if not it_col.exclude and it_col.collection.hide_render)
            for it_col in p_disabled_in_render:
                it_col.hide_render = False

    @classmethod
    def set_group_pair_invert_hide(self, context: bpy.types.Context, p_group_pair, props):
        b_changed = False
        b_select = getattr(props, 'select', False)
        p_group = p_group_pair[1]
        p_objects = p_group.get_objects(context)
        b_has_elements = len(p_objects) != 0

        for p_obj in context.view_layer.objects:
            hidden_state = p_obj not in p_objects

            try:
                if not b_changed:
                    if hidden_state and p_obj.visible_get():
                        b_changed = True
                p_obj.hide_set(hidden_state)

                if b_select and not hidden_state:
                    p_obj.select_set(True)
            except Exception:
                pass

        return b_changed, b_has_elements

    @classmethod
    def set_group_pair_renders_invert_hide(self, context: bpy.types.Context, p_group_pair, props):
        b_changed = False
        p_group = p_group_pair[1]
        p_objects = p_group.get_objects(context)
        b_has_elements = len(p_objects) != 0

        for p_obj in context.view_layer.objects:
            hidden_state = p_obj not in p_objects

            try:
                if not b_changed:
                    if hidden_state and not p_obj.hide_render:
                        b_changed = True
                p_obj.hide_render = hidden_state

            except Exception:
                pass

        return b_changed, b_has_elements

    @classmethod
    def set_group_pair_renders_state(self, context, p_group_pair, hidden_state, props):
        b_changed = False
        b_enable_collections = getattr(props, 'enable_collections', False)
        p_group = p_group_pair[1]
        p_objects = p_group.get_objects(context)
        b_has_elements = len(p_objects) != 0
        for p_obj in p_objects:
            was_visible = p_obj.hide_render
            try:
                p_obj.hide_render = hidden_state
                if p_obj.hide_render != was_visible:
                    b_changed = True
            except Exception:
                pass

        if b_enable_collections:
            for it_col, _ in self._iterate_layer_tree(context.view_layer.layer_collection, None):
                if not it_col.exclude:
                    p_col = it_col.collection
                    if p_col and p_col.hide_render and p_objects.intersection(p_col.all_objects):
                        p_col.hide_render = False

        return b_changed, b_has_elements

    @classmethod
    def hide_group_in_renders_by_index(self, context, group_index, props):
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        hidden_state = True
        if group_index in range(len(p_list)):
            p_objects = p_list[group_index].get_objects(context)
            hide_in_renders = set(p_obj for p_obj in p_objects if p_obj.hide_render)
            hidden_state = 0 == len(hide_in_renders)

            status = self.set_group_pair_renders_state(context, (group_index, p_list[group_index]), hidden_state, props)
            return status[0], status[1], hidden_state
        else:
            return False, False, hidden_state

    @classmethod
    def store_selection(self, context: bpy.types.Context, b_is_not_sync) -> Set:
        return set(context.selected_objects)

    @classmethod
    def restore_selection(self, context: bpy.types.Context, b_is_not_sync, was_selection: Set):
        bpy.ops.object.select_all(action='DESELECT')
        for p_obj in was_selection:
            p_obj.select_set(True)

    @classmethod
    def execute_AppendToGroup(self, operator, context: bpy.types.Context):
        try:
            p_scene = context.scene

            p_group = self._get_scene_group(p_scene)

            if p_group:
                t_info = self._get_layer_extra_info(context, p_group.layer_name)
                n_count = t_info.get('count', 0)
                n_hide_count = t_info.get('hide_count', 0)
                was_hidden = n_count != 0 and n_count == n_hide_count

                self.append_objects_to_group(context.selected_objects, p_group)

                if was_hidden and len(context.selected_objects) != 0:
                    bpy.ops.object.hide_view_set(unselected=False)

                update_view3d_in_all_screens()

                return {'FINISHED'}
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def are_all_group_objects_selected(self, context):
        p_scene = context.scene
        p_group = self._get_scene_group(p_scene)
        if not p_group:
            return True

        p_objects = p_group.get_objects(context)

        if len(p_objects) == 0:
            return False

        return p_objects.issubset(context.selected_objects)

    @classmethod
    def _do_select_scene_objects_in_group(self, operator, context: bpy.types.Object, fn_condition=None):
        self.check_validate_list(context)
        p_scene = context.scene
        p_group = self._get_scene_group(p_scene)
        if not p_group:
            return True

        s_group_name = p_group.name

        p_objects = p_group.get_objects(context)

        if len(p_objects) == 0:
            return {'CANCELLED'}

        bpy.ops.object.select_all(action='DESELECT')

        for obj in p_objects:
            try:
                if fn_condition:
                    if not fn_condition(context, obj):
                        continue

                if obj.hide_select:
                    if not operator.prop_make_selectable:
                        continue
                    else:
                        obj.hide_select = False
                if obj.hide_get():
                    if not operator.prop_unhide:
                        continue
                    else:
                        obj.hide_set(False)

                # --- condition when Collection is excluded or hidden ---
                if operator.prop_make_selectable and operator.prop_unhide:
                    ensure_object_in_viewlayer(obj, context.view_layer.layer_collection)

                    obj.select_set(True)
            except Exception as e:
                Log.error(e)
                continue

        if len(context.selected_objects) != 0:
            if not context.active_object or context.active_object not in context.selected_objects:
                context.view_layer.objects.active = context.selected_objects[0]
        else:
            operator.report({'INFO'}, f'Nothing was selected in - {s_group_name} !')

        return {'FINISHED'}

    @classmethod
    def execute_SelectSceneObjectsInGroup(self, operator, context):
        return self._do_select_scene_objects_in_group(operator, context)

    @classmethod
    def new_group(self, context: bpy.types.Context, props):
        p_scene = context.scene

        s_layer_name = self.create_unique_layer_name()
        p_scene_list = self.get_list(p_scene)
        p_color = self._gen_new_color(p_scene_list)
        i_list_index = self.add_layer_to_list(p_scene, s_layer_name, p_color)
        self.set_list_index(p_scene, i_list_index)

        p_scene.zen_sets_last_smart_layer = s_layer_name

        self.append_objects_to_group(context.selected_objects, p_scene_list[i_list_index])

        self._add_draw_handler()

    @classmethod
    def execute_NewGroup(self, operator, context):
        self.new_group(context, operator)

        return {'FINISHED'}

    @classmethod
    def execute_DeleteHierarchy(self, operator, context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            _, group = p_group_pair
            for p_obj in group.get_objects(context):
                bpy.data.objects.remove(p_obj)

            return {'FINISHED'}
        return {'CANCELLED'}

    @classmethod
    def execute_RemoveSelectionFromGroup(self, operator, context):
        self.check_validate_list(context)

        p_pairs = self.get_current_group_pairs(context) if operator.mode == 'ALL' else [self.get_current_group_pair(context)]
        b_modified = False
        for p_group_pair in p_pairs:
            if p_group_pair:
                self.remove_selection_from_group_pair(p_group_pair, operator, context)
                b_modified = True

        if b_modified:
            update_view3d_in_all_screens()

            return {'FINISHED'}

        return {'CANCELLED'}

    @classmethod
    def select_group_pair(self, p_group_pair, operator, context):
        _, p_group = p_group_pair
        p_objects = p_group.get_objects(context)

        bpy.ops.object.select_all(action='DESELECT')
        for p_obj in p_objects:
            try:
                if p_obj.visible_get():
                    p_obj.select_set(True)
            except Exception:
                pass

        self.ensure_active_object_in_selection(context)

    @classmethod
    def smart_select(self, context, select_active_group_only, keep_active_group):
        selection_stats = ZenSelectionStats()
        selection_stats.was_object_sel_count = self.get_context_selected_count(context)
        if selection_stats.was_object_sel_count == 0:
            return selection_stats

        # self.check_validate_list(context)
        p_scene = context.scene

        was_sel_objects = set(context.selected_objects)

        to_sel_objects = set()

        p_list = self.get_list(p_scene)
        i_list_index = self.get_list_index(p_scene)

        i_new_index = i_list_index

        groupped_objects = set()

        for idx, p_group in enumerate(p_list):
            p_objects = p_group.get_objects(context)

            groupped_objects.update(p_objects)

            if was_sel_objects.intersection(p_objects):

                self._do_set_last_smart_select(p_list[idx].layer_name)
                i_new_index = idx

                if select_active_group_only and idx == i_list_index:
                    to_sel_objects = p_objects
                    break
                else:
                    to_sel_objects.update(p_objects)

        bpy.ops.object.select_all(action='DESELECT')

        if len(to_sel_objects) != 0:
            for p_obj in p_scene.objects:
                if p_obj in to_sel_objects:
                    p_obj.select_set(True)
        else:
            i_new_index = -1

            for p_obj in p_scene.objects:
                if p_obj not in groupped_objects:
                    p_obj.select_set(True)

        if not keep_active_group and i_new_index != i_list_index:
            self.set_list_index(p_scene, i_new_index)

        selection_stats.new_object_sel_count = self.get_context_selected_count(context)
        selection_stats.selection_changed = was_sel_objects != set(context.selected_objects)
        return selection_stats

    @classmethod
    def is_anyone_hidden(self, context):
        return any(p_obj.hide_get() for p_obj in context.view_layer.objects)

    @classmethod
    def execute_SelectUngroupped(self, operator, context):
        try:
            self.check_validate_list(context)

            p_group_pairs = self.get_current_group_pairs(context)
            if len(p_group_pairs):
                groupped_objects = set()
                for _, p_group in p_group_pairs:
                    groupped_objects.update(p_group.get_objects(context))

                for p_obj in context.scene.objects:
                    b_select = p_obj not in groupped_objects
                    p_obj.select_set(b_select)
            else:
                bpy.ops.object.select_all(action='SELECT')

            return {'FINISHED'}

        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_SelectOnlyGroup(self, operator, context):
        self.check_validate_list(context)

        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            self.select_group_pair(p_group_pair, operator, context)
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'FINISHED'}

    @classmethod
    def select_append_group_pair(self, p_group_pair, operator, context):
        _, p_group = p_group_pair
        p_objects = p_group.get_objects(context)

        for p_obj in p_objects:
            try:
                p_obj.select_set(True)
            except Exception:
                pass

        self.ensure_active_object_in_selection(context)

    @classmethod
    def execute_SelectAppendGroup(self, operator, context):
        self.check_validate_list(context)

        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            self.select_append_group_pair(p_group_pair, operator, context)
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'FINISHED'}

    @classmethod
    def deselect_group_pair(self, p_group_pair, operator, context):
        if p_group_pair:
            _, p_group = p_group_pair
            p_objects = p_group.get_objects(context)
            p_selected = set(context.selected_objects).intersection(p_objects)
            for p_obj in p_selected:
                try:
                    p_obj.select_set(False)
                except Exception:
                    pass

            self.ensure_active_object_in_selection(context)

    @classmethod
    def execute_DeselectGroup(self, operator, context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            self.deselect_group_pair(p_group_pair, operator, context)
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'FINISHED'}

    @classmethod
    def execute_IntersectGroup(self, operator, context):
        self.check_validate_list(context)
        p_group_pair = self.get_current_group_pair(context)
        if p_group_pair:
            _, p_group = p_group_pair
            p_objects = p_group.get_objects(context)

            p_intersection = set(context.selected_objects).intersection(p_objects)

            bpy.ops.object.select_all(action='DESELECT')

            for p_obj in p_intersection:
                try:
                    p_obj.select_set(True)
                except Exception:
                    pass

            self.ensure_active_object_in_selection(context)
        else:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)

        return {'FINISHED'}

    @classmethod
    def execute_DeleteAllGroups(self, operator, context):
        p_scene = context.scene
        p_list = self.get_list(p_scene)

        if len(p_list) > 0:
            remove_draw_handler(self, context)

            self.set_list_index(p_scene, -1)

            p_list.clear()

            return {'FINISHED'}

        return {'CANCELLED'}

    @classmethod
    def execute_DeleteEmptyGroups(self, operator, context):
        self.check_validate_list(context)
        p_scene = context.scene
        p_list = self.get_list(p_scene)
        p_group_pairs = self.get_current_group_pairs(context)
        was_layer_name = self.get_current_layer_name(context)
        is_modified = False
        for idx, p_group_pairs in reversed(p_group_pairs):
            p_objects = p_group_pairs.get_objects(context)
            if len(p_objects) == 0:
                p_list.remove(idx)
                is_modified = True

        if is_modified:
            p_context_group_pairs = self.get_current_group_pairs(context)
            p_was_group_pair = self.get_group_pair_by_layer(p_context_group_pairs, was_layer_name)
            if p_was_group_pair:
                self.set_list_index(p_scene, p_was_group_pair[0])
            elif len(p_context_group_pairs):
                self.set_list_index(p_scene, p_context_group_pairs[0][0])
            else:
                self.set_list_index(p_scene, -1)

            return {'FINISHED'}

        return {'CANCELLED'}

    @classmethod
    def execute_GroupLinked(self, operator: bpy.types.Operator, context: bpy.types.Context):
        addon_prefs = get_prefs()
        was_auto_highlight = addon_prefs.common.auto_highlight
        addon_prefs.common.auto_highlight = False
        bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='OFF')

        interval = timer()

        try:
            p_scene = context.scene
            p_scene_list = self.get_list(p_scene)
            i_list_index = -1

            ZenLocks.lock_lookup_build()
            ZenLocks.lock_depsgraph_update()

            default_delimits, use_custom = operator.get_delimits()

            start_progress(context)

            bpy.ops.object.select_all(action='DESELECT')

            p_visible_objects = set(context.visible_objects)

            i_progress_max = len(context.selectable_objects)

            i_progress_index = 0

            was_active_object = context.view_layer.objects.active

            while (len(context.selectable_objects) != 0):
                p_obj = context.selectable_objects[0]

                was_sel = len(context.selected_objects)

                p_obj.select_set(True)
                context.view_layer.objects.active = p_obj

                group_prefix = ''
                group_name = ''

                if len(default_delimits):
                    for delim in default_delimits:
                        if delim.startswith('LINKED_'):
                            delim_type = delim.replace('LINKED_', '')
                            bpy.ops.object.select_linked(extend=True, type=delim_type)

                            if delim == 'LINKED_MATERIAL':
                                if p_obj.active_material is not None:
                                    group_name = p_obj.active_material.name
                            elif delim == 'LINKED_OBDATA':
                                if p_obj.data is not None:
                                    group_name = p_obj.data.name

                        elif delim == 'TYPE':
                            bpy.ops.object.select_grouped(extend=True, type='TYPE')
                            group_prefix = p_obj.type
                        elif delim == 'SINGLE_OBJECT':
                            group_name = p_obj.name

                try:
                    if use_custom:
                        operator.execute_custom_delimiter()

                    sel_dif = len(context.selected_objects) - was_sel
                    if sel_dif >= operator.min_group_size:
                        s_layer_name = self.create_unique_layer_name()

                        p_color = self._gen_new_color(p_scene_list)
                        i_list_index = self.add_layer_to_list(p_scene, s_layer_name, p_color)
                        self.append_objects_to_group(context.selected_objects, p_scene_list[i_list_index])

                        if group_name != '':
                            p_scene_list[i_list_index].name = group_name
                        if group_prefix != '':
                            p_scene_list[i_list_index].name = group_prefix.title() + (f"_{i_list_index + 1}" if i_list_index > 0 else "")

                    bpy.ops.object.hide_view_set(unselected=False)
                except Exception as e:
                    Log.error(e)
                    operator.report({'ERROR'}, 'Auto Groups failed! See console for details!')
                    break

                Log.debug('Remaining:', i_progress_max - i_progress_index - 1, 'of:', i_progress_max)
                update_progress(context, int((i_progress_max - i_progress_index - 1) / i_progress_max * 100))

                i_progress_index += 1

            end_progress(context)

            for p_obj in context.view_layer.objects:
                if p_obj in p_visible_objects:
                    try:
                        p_obj.hide_set(False)
                    except Exception:
                        pass

            context.view_layer.objects.active = was_active_object

            self.set_list_index(p_scene, i_list_index)

        finally:
            ZenLocks.unlock_lookup_build()
            ZenLocks.unlock_depsgraph_update()

            bpy.ops.zsts.draw_highlight('INVOKE_DEFAULT', mode='ON')
            addon_prefs.common.auto_highlight = was_auto_highlight

            Log.debug('AutoGroups completed ==========>', timer() - interval)

        return {'FINISHED'}

    @classmethod
    def execute_DrawTools(cls, tools, context):
        layout = tools.layout

        layout.operator("zsts.rename_groups")
        layout.operator("zsto.group_linked")

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        layout = menu.layout

        layout.operator("zsts.select_scene_objects")
        layout.menu("ZSTO_MT_SelectObjectsWith")
        layout.operator("zsto.colorize_selected")

        layout.separator()
        layout.operator("zsts.rename_groups")
        layout.operator("zsto.group_linked")

    @classmethod
    def execute_DrawImport(cls, layout, context, is_menu=False):
        pass

    @classmethod
    def execute_DrawExport(cls, layout, context, is_menu=False):
        pass
