# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2022, Alex Zhornyak

import bpy

import numpy as np

from typing import Set

from .simple_object_base import ZsSimpleObjectBaseLayerManager, Zs_UL_SimpleObjectList

from ...draw_cache import BasicObjectsCacher

from ....labels import ZsLabels
from ....blender_zen_utils import update_view3d_in_all_screens


class ZsObjectPartsLayerManager(ZsSimpleObjectBaseLayerManager):
    id_group = 'obj_simple_parts'
    id_mask = 'ZSOP'
    is_unique = True
    id_element = 'object'

    list_item_prefix = 'ObjectsP'

    @classmethod
    def get_cacher(self):
        return BasicObjectsCacher()

    @classmethod
    def check_update_list(cls, p_scene):
        pass

    @classmethod
    def _do_draw_collection_toolbar(self, context: bpy.types.Context, layout, is_tool_header):
        pass

    @classmethod
    def update_collections(cls, p_scene):
        pass

    @classmethod
    def get_highlighted_group_pairs(self, context):
        return self.get_current_group_pairs(context)

    @classmethod
    def remove_selection_from_group_pair(self, p_group_pair, operator, context):
        if not p_group_pair:
            operator.report({'INFO'}, 'Collection is not selected in Zen Sets List')
            return

        sel_objs = set(context.selected_objects)
        if not sel_objs:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
            return

        _, p_group = p_group_pair
        for p_obj in sel_objs:
            if p_obj.zsop_uuid == p_group.layer_name:
                p_obj.zsop_uuid = ''

    @classmethod
    def execute_AssignToGroup(self, operator, context):
        try:
            p_scene = context.scene

            p_group = self._get_scene_group(p_scene)

            if p_group:

                self.assign_objects_to_group(context, context.selected_objects, p_group)

                update_view3d_in_all_screens()

                return {'FINISHED'}
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_RemoveLinkedFromGroup(self, operator, context):
        p_scene = context.scene
        handled_objects_data = set()
        p_list = self.get_list(p_scene)
        i_current_index = self.get_list_index(p_scene)

        b_modified = False

        for idx, group in enumerate(p_list):
            if operator.group_mode == 'ACTIVE' and idx != i_current_index:
                continue

            p_objects = group.get_objects(context)
            for p_obj in p_objects:
                if p_obj.data is not None:
                    if p_obj.data in handled_objects_data:
                        p_obj.zsop_uuid = ''
                        b_modified = True
                    else:
                        handled_objects_data.add(p_obj.data)

        if b_modified:
            update_view3d_in_all_screens()
            return {'FINISHED'}
        else:
            return {'CANCELLED'}

    @classmethod
    def append_objects_to_group(self, p_objects, p_scene_group):
        for p_obj in p_objects:
            p_obj.zsop_uuid = p_scene_group.layer_name

    @classmethod
    def assign_objects_to_group(self, context, p_objects, p_scene_group):
        if not isinstance(p_objects, set):
            p_objects = set(p_objects)
        for p_obj in context.scene.objects:
            if p_obj in p_objects:
                p_obj.zsop_uuid = p_scene_group.layer_name
            else:
                if p_obj.zsop_uuid == p_scene_group.layer_name:
                    p_obj.zsop_uuid = ''

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

        layout = menu.layout

        layout.operator(ZSOP_OT_AssignMaterialsToGroups.bl_idname)  # icon='MATERIAL'
        layout.operator('zsto.remove_linked_objects')

    @classmethod
    def execute_DrawTools(cls, tools, context):
        super().execute_DrawTools(tools, context)

        layout = tools.layout
        layout.operator(ZSOP_OT_AssignMaterialsToGroups.bl_idname)
        layout.operator('zsto.remove_linked_objects')


class ZSOP_OT_AssignMaterialsToGroups(bpy.types.Operator, ZsObjectPartsLayerManager):
    """ Assign materials to groups """
    bl_idname = 'zsto.assign_materials_to_groups'
    bl_description = ZsLabels.OT_ASSIGN_MATERIAL_TO_GROUP_DESC
    bl_label = ZsLabels.OT_ASSIGN_MATERIAL_TO_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    assign_to_faces: bpy.props.BoolProperty(
        name='Assign To Faces',
        description='Assign Material to all Object Faces',
        default=True)

    @classmethod
    def poll(cls, context):
        if context.mode != 'OBJECT':
            return False

        p_scene = context.scene
        p_list = cls.get_list(p_scene)
        return p_list and len(p_list) > 0

    def execute(self, context):
        p_group_pairs = self.get_current_group_pairs(context)

        if len(p_group_pairs):

            def_mat_name = 'Material.Default'

            def_mat = None
            if def_mat_name not in bpy.data.materials.keys():
                def_mat = bpy.data.materials.new(name=def_mat_name)
                material_color = (1, 1, 1, 1)
                def_mat.diffuse_color = material_color
                def_mat.use_nodes = True
                def_mat.use_fake_user = True
                pr_bsdf_node = def_mat.node_tree.nodes.get("Principled BSDF")
                if pr_bsdf_node:
                    pr_bsdf_node.inputs['Base Color'].default_value = material_color
            else:
                def_mat = bpy.data.materials.get(def_mat_name)

            p_all_objects = set(context.scene.objects)

            update_objs = set()

            handled_mesh_data = set()

            for _, p_group in p_group_pairs:
                material_name = 'Material.' + p_group.name
                material_color = (p_group.group_color.r, p_group.group_color.g, p_group.group_color.b, 1)
                new_mat = bpy.data.materials.new(name=material_name) if material_name not in bpy.data.materials.keys() else bpy.data.materials.get(material_name)
                new_mat.diffuse_color = material_color
                new_mat.use_nodes = True
                new_mat.use_fake_user = True
                pr_bsdf_node = new_mat.node_tree.nodes.get("Principled BSDF")
                if pr_bsdf_node:
                    pr_bsdf_node.inputs['Base Color'].default_value = material_color

                p_group_objects = p_group.get_objects(context)
                update_objs.update(p_group_objects)
                for p_obj in p_group_objects:
                    if p_obj.type == 'MESH':
                        if p_obj.data in handled_mesh_data:
                            self.report({'WARNING'}, f'Object [{p_obj.name}] has linked data and will be skipped!')
                            continue
                        else:
                            handled_mesh_data.add(p_obj.data)

                        id_mat = p_obj.data.materials.find(material_name)

                        if id_mat == -1:
                            p_obj.data.materials.append(new_mat)
                            id_mat = len(p_obj.data.materials) - 1

                        if self.assign_to_faces:
                            indices = np.empty(len(p_obj.data.polygons), 'i')
                            indices.fill(id_mat)

                            p_obj.data.polygons.foreach_set('material_index', indices)

            for p_obj in (p_all_objects - update_objs):
                if p_obj.type == 'MESH':
                    if p_obj.data in handled_mesh_data:
                        self.report({'WARNING'}, f'Object [{p_obj.name}] has linked data and will be skipped!')
                        continue
                    else:
                        handled_mesh_data.add(p_obj.data)

                    id_def_mat = p_obj.data.materials.find(def_mat_name)

                    if id_def_mat == -1:
                        p_obj.data.materials.append(def_mat)
                        id_def_mat = len(p_obj.data.materials) - 1

                    if self.assign_to_faces:
                        indices = np.empty(len(p_obj.data.polygons), 'i')
                        indices.fill(id_def_mat)

                        p_obj.data.polygons.foreach_set('material_index', indices)

            return {'FINISHED'}

        return {'CANCELLED'}


class ZSOP_UL_List(Zs_UL_SimpleObjectList, ZsObjectPartsLayerManager):
    pass


class ZSOP_SceneListGroup(bpy.types.PropertyGroup):
    """
    Group of properties representing
    an item in the zen sets groups for SCENE
    """
    layer_name: bpy.props.StringProperty(
        name="LayerName",
        description="",
        default=""
    )
    group_color: bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_GROUP_COLOR_NAME,
        subtype='COLOR_GAMMA',
        size=3,
        default=(0.0, 0.5, 0.0),
        min=0, max=1,
    )

    def get_objects(self, context) -> Set:
        return set(obj for obj in context.scene.objects if obj.zsop_uuid == self.layer_name)


class ZSOP_Factory:
    classes = (
        ZSOP_UL_List,
        ZSOP_SceneListGroup,

        ZSOP_OT_AssignMaterialsToGroups,
    )

    def get_mgr():
        return ZsObjectPartsLayerManager

    def get_ui_list():
        return ZSOP_SceneListGroup

    @classmethod
    def register(cls):

        bpy.types.Object.zsop_uuid = bpy.props.StringProperty(
            name='Zen Parts Uuid'
        )

    @classmethod
    def unregister(cls):

        if hasattr(bpy.types.Object, 'zsop_uuid'):
            del bpy.types.Object.zsop_uuid
