# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2022, Alex Zhornyak


import bpy

from typing import Set

from ....labels import ZsLabels

from ...draw_cache import BasicObjectsCacher

from .simple_object_base import ZsSimpleObjectBaseLayerManager, Zs_UL_SimpleObjectList

from ....blender_zen_utils import update_view3d_in_all_screens


class ZsObjectSetsLayerManager(ZsSimpleObjectBaseLayerManager):
    id_group = 'obj_simple_sets'
    id_mask = 'ZSOS'
    is_unique = False
    id_element = 'object'

    list_item_prefix = 'Objects'

    @classmethod
    def get_cacher(self):
        return BasicObjectsCacher()

    @classmethod
    def check_update_list(cls, p_scene):
        pass

    @classmethod
    def check_validate_list(cls, context):
        p_scene = context.scene
        p_list = cls.get_list(p_scene)
        scene_layers = set(p_group.layer_name for p_group in p_list)
        for p_obj in context.scene.objects:
            for i in range(len(p_obj.zsos_uuid) - 1, -1, -1):
                elem = p_obj.zsos_uuid[i]
                if elem.name not in scene_layers:
                    p_obj.zsos_uuid.remove(i)

    @classmethod
    def _do_draw_collection_toolbar(self, context: bpy.types.Context, layout, is_tool_header):
        pass

    @classmethod
    def update_collections(cls, p_scene):
        pass

    @classmethod
    def get_highlighted_group_pairs(self, context):
        p_group_pair = self.get_current_group_pair(context)
        return [p_group_pair] if p_group_pair else []

    @classmethod
    def remove_selection_from_group_pair(self, p_group_pair, operator, context):
        if not p_group_pair:
            operator.report({'INFO'}, 'Collection is not selected in Zen Sets List')
            return

        sel_objs = set(context.selected_objects)
        if not sel_objs:
            operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
            return

        _, p_group = p_group_pair
        for p_obj in sel_objs:
            i_index = p_obj.zsos_uuid.find(p_group.layer_name)
            if i_index != -1:
                p_obj.zsos_uuid.remove(i_index)

    @classmethod
    def append_objects_to_group(self, p_objects, p_scene_group):
        for p_obj in p_objects:
            if p_obj.zsos_uuid.find(p_scene_group.layer_name) == -1:
                item = p_obj.zsos_uuid.add()
                item.name = p_scene_group.layer_name

    @classmethod
    def assign_objects_to_group(self, context, p_objects, p_scene_group):
        if not isinstance(p_objects, set):
            p_objects = set(p_objects)
        p_not_in_group = set(context.scene.objects) - p_objects
        for p_obj in p_not_in_group:
            i_index = p_obj.zsos_uuid.find(p_scene_group.layer_name)
            if i_index != -1:
                p_obj.zsos_uuid.remove(i_index)

        self.append_objects_to_group(p_objects, p_scene_group)

    @classmethod
    def execute_RemoveLinkedFromGroup(self, operator, context):
        p_scene = context.scene
        handled_objects_data = set()
        p_list = self.get_list(p_scene)
        i_current_index = self.get_list_index(p_scene)

        b_modified = False

        for idx, group in enumerate(p_list):
            if operator.group_mode == 'ACTIVE' and idx != i_current_index:
                continue

            p_objects = group.get_objects(context)
            for p_obj in p_objects:
                if p_obj.data is not None:
                    if p_obj.data in handled_objects_data:
                        i_index = p_obj.zsos_uuid.find(group.layer_name)
                        if i_index != -1:
                            p_obj.zsos_uuid.remove(i_index)
                        b_modified = True
                    else:
                        handled_objects_data.add(p_obj.data)

        if b_modified:
            update_view3d_in_all_screens()
            return {'FINISHED'}
        else:
            return {'CANCELLED'}

    @classmethod
    def execute_AssignToGroup(self, operator, context):
        try:
            p_scene = context.scene

            p_group = self._get_scene_group(p_scene)

            if p_group:
                self.assign_objects_to_group(context, context.selected_objects, p_group)

                update_view3d_in_all_screens()

                return {'FINISHED'}
            else:
                operator.report({'INFO'}, ZsLabels.OT_WARN_NOTHING_SELECTED)
        except Exception as e:
            operator.report({'WARNING'}, str(e))
        return {'CANCELLED'}

    @classmethod
    def execute_DrawMenu(cls, menu, context):
        super().execute_DrawMenu(menu, context)

        layout = menu.layout

        layout.operator('zsto.remove_linked_objects')

    @classmethod
    def execute_DrawTools(cls, tools, context):
        super().execute_DrawTools(tools, context)

        layout = tools.layout
        layout.operator('zsto.remove_linked_objects')


class ZSOS_UL_List(Zs_UL_SimpleObjectList, ZsObjectSetsLayerManager):
    pass


class ZSOS_SceneListGroup(bpy.types.PropertyGroup):
    """
    Group of properties representing
    an item in the zen sets groups for SCENE
    """
    layer_name: bpy.props.StringProperty(
        name="LayerName",
        description="",
        default=""
    )
    group_color: bpy.props.FloatVectorProperty(
        name=ZsLabels.PROP_GROUP_COLOR_NAME,
        subtype='COLOR_GAMMA',
        size=3,
        default=(0.0, 0.5, 0.0),
        min=0, max=1,
    )

    def get_objects(self, context) -> Set:
        return set(
            p_obj for p_obj in context.scene.objects
            if p_obj.zsos_uuid.find(self.layer_name) != -1)


class ZsUuidObjectElement(bpy.types.PropertyGroup):
    pass


class ZSOS_Factory:
    classes = (
        ZSOS_UL_List,
        ZSOS_SceneListGroup
    )

    def get_mgr():
        return ZsObjectSetsLayerManager

    def get_ui_list():
        return ZSOS_SceneListGroup

    @classmethod
    def register(cls):
        bpy.types.Object.zsos_uuid = bpy.props.CollectionProperty(type=bpy.types.PropertyGroup)

    @classmethod
    def unregister(cls):

        if hasattr(bpy.types.Object, 'zsos_uuid'):
            del bpy.types.Object.zsos_uuid
