# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

# blender
import bpy

from ..smart_rename import get_last_selected_smart_group, set_last_selected_smart_group

from ...blender_zen_utils import get_command_props, ZenPolls, ZsOperatorAttrs
from ...labels import ZsLabels


def _get_caller_description(template_items, properties):
    items = []
    for op_key, op_id in template_items.items():
        op_props = get_command_props(op_id)

        desc = op_id
        if op_props.bl_desc:
            desc = op_props.bl_desc
        elif op_props.bl_label:
            desc = op_props.bl_label
        items.append(f'{op_key} - {desc}')
    return '\n'.join(items)


class ZsBasicPieCaller(bpy.types.Operator):
    """ Zen Sets Caller Operator """

    bl_description = 'Click to open Popup'

    @classmethod
    def get_template_items(cls, context):
        return []

    def _exec_cmd(self):
        p_cmd = get_command_props(self.cmd_enum)
        return eval(p_cmd.cmd) if (p_cmd.bl_op_cls and p_cmd.bl_op_cls.poll()) else {'CANCELLED'}

    @classmethod
    def poll(cls, context):
        res = False
        for op_id in cls.get_template_items(context).values():
            op_props = get_command_props(op_id, context)
            if (op_props.bl_op_cls and op_props.bl_op_cls.poll()):
                res = True
                break
        return res

    @classmethod
    def description(cls, context, properties):
        return _get_caller_description(cls.get_template_items(context), properties)

    def get_template_display_name(self, context: bpy.types.Context):
        p_template_items = self.get_template_items(context)
        for k, v in p_template_items.items():
            if v == self.cmd_enum:
                p_items = self.get_template_display_items(context)
                return p_items.get(k, '')
        return ''

    def invoke(self, context, event):
        _template_items = self.get_template_items(context)

        if event.type == 'LEFTMOUSE' or event.value != 'RELEASE':
            # sequencies (Ctrl+Shift etc.) have priority
            for op_key, op_id in _template_items.items():
                if '+' in op_key:
                    keys = op_key.lower().split('+')

                    has_seq = False
                    for key in keys:
                        if getattr(event, key, False):
                            has_seq = True
                        else:
                            has_seq = False
                            break

                    if has_seq:
                        self.cmd_enum = op_id
                        return self.execute(context)

            # alt, ctrl, shift
            for op_key, op_id in _template_items.items():
                key = op_key.lower()
                if getattr(event, key, False):
                    self.cmd_enum = op_id
                    return self.execute(context)

        self.cmd_enum = _template_items['Default']

        return self.execute(context)

    def execute(self, context):
        return self._exec_cmd()


class ZsBasicBottomCaller(ZsBasicPieCaller):
    bl_label = "Group Selector | Display"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def get_template_items(cls, context):
        if context.mode == 'EDIT_MESH':
            t_items = {
                'Default': 'zsts.set_workspace_tool',
                'CTRL': 'bpy.ops.zsts.draw_highlight("INVOKE_DEFAULT")',
                'SHIFT': 'zsts.select_mode'
            }
            if ZenPolls.is_view3d_space_data(context):
                t_items['ALT'] = 'mesh.zsts_display_vertex_weights("INVOKE_DEFAULT")'
            return t_items
        else:
            return {
                'Default': 'zsts.set_workspace_tool',
                'CTRL': 'bpy.ops.zsts.draw_highlight("INVOKE_DEFAULT")',
                'SHIFT': 'zsts.select_mode'
            }

    @classmethod
    def get_template_display_items(cls, context):
        if context.mode == 'EDIT_MESH':
            t_items = {
                'Default': 'Group Selector',
                'CTRL': 'Display',
                'SHIFT': 'Zen Sets Mode'
            }
            if ZenPolls.is_view3d_space_data(context):
                t_items['ALT'] = 'Display Weights'
            return t_items
        else:
            return {
                'Default': 'Group Selector',
                'CTRL': 'Display',
                'SHIFT': 'Zen Sets Mode'
            }

    @classmethod
    def description(cls, context, properties):
        if context.mode == 'EDIT_MESH':
            t_items = 'Default: ' + ZsLabels.TOOL_DESCRIPTION + \
                    '\nCTRL - ' + ZsLabels.OT_DRAW_HIGHLIGHT_DESCRIPTION + \
                    '\nSHIFT - Zen Sets Mode'
            if ZenPolls.is_view3d_space_data(context):
                t_items = t_items + '\nALT - Display Vertex Group Weights'
            return t_items
        else:
            return 'Default: ' + ZsLabels.TOOL_DESCRIPTION + \
                    '\nCTRL - ' + ZsLabels.OT_DRAW_HIGHLIGHT_DESCRIPTION + \
                    '\nSHIFT - Zen Sets Mode'


class ZsBasicTopCaller(ZsBasicPieCaller):
    bl_label = "Smart Select"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def get_label(cls, context):
        if ZenPolls.is_object_and_collection_mode(context):
            return "Smart Select | Duplicate"
        else:
            return "Smart Select"

    @classmethod
    def get_template_items(cls, context):

        if ZenPolls.is_object_and_collection_mode(context):
            return {
                'Default': 'zsts.smart_select',
                'ALT': 'zsts.smart_isolate',
                'CTRL': 'zsto.duplicate_collection',
                'SHIFT': 'zsto.duplicate_collection_linked',
                'CTRL+SHIFT': 'zsto.duplicate_as_intance'
            }
        else:
            return {
                'Default': 'zsts.smart_select',
                'ALT': 'zsts.smart_isolate'
            }

    def _get_select_only_active_group(self):
        return ZsOperatorAttrs.get_operator_attr('zsts.smart_select', 'select_only_active_group', default=False)

    def _set_select_only_active_group(self, value):
        ZsOperatorAttrs.set_operator_attr('zsts.smart_select', 'select_only_active_group', value)

    select_only_active_group: bpy.props.BoolProperty(
        name=ZsLabels.PROP_SEL_ONLY_ONE_GROUP_NAME,
        description=ZsLabels.PROP_SEL_ONLY_ONE_GROUP_DESC,
        get=_get_select_only_active_group,
        set=_set_select_only_active_group
    )

    def _get_keep_active_group(self):
        return ZsOperatorAttrs.get_operator_attr('zsts.smart_select', 'keep_active_group', default=False)

    def _set_keep_active_group(self, value):
        ZsOperatorAttrs.set_operator_attr('zsts.smart_select', 'keep_active_group', value)

    keep_active_group: bpy.props.BoolProperty(
        name=ZsLabels.PROP_KEEP_ACTIVE_GROUP_NAME,
        description=ZsLabels.PROP_KEEP_ACTIVE_GROUP_DESC,
        get=_get_keep_active_group,
        set=_set_keep_active_group
    )

    last_selected_group: bpy.props.StringProperty(
        name='Last selected',
        description='Last selected Group name',
        get=get_last_selected_smart_group,
        set=set_last_selected_smart_group
    )

    def draw(self, context):
        layout = self.layout

        op_props = get_command_props(self.cmd_enum)

        layout.label(text=f'{op_props.bl_label}')

        _template_items = self.get_template_items(context)

        if self.cmd_enum in {_template_items['Default'], _template_items['ALT']}:
            layout.prop(self, 'select_only_active_group')
            layout.prop(self, 'keep_active_group')
            row = layout.row()
            row_s = row.split(factor=0.3)
            row_s.label(text='Last Selected')
            row_s.prop(self, 'last_selected_group', text='')
