# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

# Copyright 2021, Alex Zhornyak

# blender
import bpy

from .basic_pie import (
    ZsBasicPieCaller,
    ZsBasicTopCaller,
    ZsBasicBottomCaller,
    get_command_props)

from ..smart_rename import get_last_selected_smart_group, set_last_selected_smart_group

from ...labels import ZsLabels
from ...blender_zen_utils import (
    ZenPolls,
    ZsOperatorAttrs,
    draw_last_operator_properties,
    draw_ex_last_operator_properties, get_mimic_operator_layout)
from ...preferences import get_prefs


class ZSTO_OT_PieCallerLeft(ZsBasicPieCaller):
    bl_idname = "zsto.pie_caller_left"
    bl_label = "Deselect | Unhide"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def get_template_items(cls, context):
        return {
            'Default': 'zsts.deselect_group',
            'CTRL': 'zsts.unhide_group',
            'SHIFT': 'zsto.invert_hide_group("INVOKE_DEFAULT")',
            'CTRL+SHIFT': 'zsto.unhide_all'
        }

    def draw(self, context):
        layout = self.layout

        op_props = get_command_props(self.cmd_enum)

        layout.label(text=f'{op_props.bl_label}')

        p_template_items = self.get_template_items(context)

        if self.cmd_enum == p_template_items['CTRL']:
            draw_ex_last_operator_properties(context, 'zsts.unhide_group', layout)
        elif self.cmd_enum == p_template_items['SHIFT']:
            draw_ex_last_operator_properties(context, 'zsto.invert_hide_group', layout)
        elif self.cmd_enum == p_template_items['CTRL+SHIFT']:
            draw_last_operator_properties(context, 'zsto.unhide_all', layout)

    cmd_enum: bpy.props.StringProperty(options={'HIDDEN', 'SKIP_SAVE'})
    is_menu: bpy.props.BoolProperty(default=True, options={'HIDDEN', 'SKIP_SAVE'})


class ZSTO_OT_PieCallerTop(ZsBasicTopCaller):
    bl_idname = "zsto.pie_caller_top"

    cmd_enum: bpy.props.StringProperty(options={'HIDDEN', 'SKIP_SAVE'})
    is_menu: bpy.props.BoolProperty(default=True, options={'HIDDEN', 'SKIP_SAVE'})


class ZSTO_OT_PieCallerTopLeft(ZsBasicPieCaller):
    bl_idname = "zsto.pie_caller_top_left"
    bl_label = "Delete Group | Empty"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def get_template_items(cls, context):
        if ZenPolls.is_collection_mode(context):
            return {
                'Default': 'zsto.del_group_only',
                'CTRL': 'zsto.delete_empty_groups',
                'CTRL+SHIFT': 'zsto.delete_hierarchy',
                'ALT': 'zsto.convert_collection_to_object'
            }
        else:
            return {
                'Default': 'zsto.del_group_only',
                'CTRL': 'zsto.delete_empty_groups',
                'CTRL+SHIFT': 'zsto.delete_hierarchy',
            }

    def draw(self, context: bpy.types.Context):
        layout = self.layout

        op_props = get_command_props(self.cmd_enum)

        layout.label(text=f'{op_props.bl_label}')

        addon_prefs = get_prefs()

        if self.cmd_enum == 'zsto.delete_empty_groups':
            draw_last_operator_properties(context, 'zsto.delete_empty_groups', layout)
        elif self.cmd_enum == 'zsto.convert_collection_to_object':
            addon_prefs.op_options.draw_convert_collection_to_parent(layout)

    cmd_enum: bpy.props.StringProperty(options={'HIDDEN', 'SKIP_SAVE'})
    is_menu: bpy.props.BoolProperty(default=True, options={'HIDDEN', 'SKIP_SAVE'})


class ZSTO_OT_PieCallerTopRight(ZsBasicPieCaller):
    bl_idname = "zsto.pie_caller_top_right"
    bl_label = "New Group | Link"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def get_label(cls, context):
        if ZenPolls.is_collection_mode(context):
            return "New Group | Link"
        else:
            return "New Group"

    @classmethod
    def get_template_items(cls, context):
        is_collection_mode = ZenPolls.is_collection_mode(context)
        if is_collection_mode:
            return {
                'Default': 'zsto.new_group_move',
                'CTRL': 'zsto.new_group_link',
                'ALT': 'zsto.convert_object_to_collection'
            }
        else:
            return {
                'Default': 'zsto.new_group',
            }

    new_group_name: bpy.props.StringProperty(
        name='New Group Name',
        description='New Group Name',
        get=get_last_selected_smart_group,
        set=set_last_selected_smart_group
    )

    def draw(self, context):
        layout = self.layout

        op_props = get_command_props(self.cmd_enum)

        layout.label(text=f'{op_props.bl_label}')

        addon_prefs = get_prefs()

        is_collection_mode = ZenPolls.is_collection_mode(context)

        if is_collection_mode:
            if self.cmd_enum == 'zsto.convert_object_to_collection':
                addon_prefs.op_options.draw_convert_parent_to_collection(layout)
            else:
                draw_last_operator_properties(context, 'zsto.new_group', layout, whitelist_props={'keep_nested'}, factor=0.4)
                get_mimic_operator_layout(self, layout, 'new_group_name', factor=0.4)
        else:
            get_mimic_operator_layout(self, layout, 'new_group_name', factor=0.4)

    cmd_enum: bpy.props.StringProperty(options={'HIDDEN', 'SKIP_SAVE'})
    is_menu: bpy.props.BoolProperty(default=True, options={'HIDDEN', 'SKIP_SAVE'})


class ZSTO_OT_PieCallerRight(ZsBasicPieCaller):
    bl_idname = "zsto.pie_caller_right"
    bl_label = "Select | Hide"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def get_template_items(cls, context):
        if ZenPolls.is_collection_mode(context):
            return {
                'Default': 'zsts.select_group',
                'CTRL': 'zsts.hide_group',
                'SHIFT': 'zsto.exclude_toggle_group',
                'ALT': 'bpy.ops.zsto.colorize_selected("INVOKE_DEFAULT")'
            }
        else:
            return {
                'Default': 'zsts.select_group',
                'CTRL': 'zsts.hide_group',
                'ALT': 'bpy.ops.zsto.colorize_selected("INVOKE_DEFAULT")',
                'CTRL+SHIFT': 'zsts.select_ungroupped'
            }

    # DO NOT CHANGE 'nested' !!!

    def _get_nested(self):
        return ZsOperatorAttrs.get_operator_attr('zsto.exclude_group_by_index', 'nested', default=True)

    def _set_nested(self, value):
        ZsOperatorAttrs.set_operator_attr('zsto.exclude_group_by_index', 'nested', value)

    nested: bpy.props.BoolProperty(name='Nested', get=_get_nested, set=_set_nested)

    def draw(self, context):
        layout = self.layout

        op_props = get_command_props(self.cmd_enum)

        layout.label(text=f'{op_props.bl_label}')

        if self.cmd_enum == 'zsts.select_group':
            draw_ex_last_operator_properties(context, 'zsts.select_group', layout)
        elif self.cmd_enum == 'zsto.exclude_toggle_group':
            layout.prop(self, 'nested')

    cmd_enum: bpy.props.StringProperty(options={'HIDDEN', 'SKIP_SAVE'})
    is_menu: bpy.props.BoolProperty(default=True, options={'HIDDEN', 'SKIP_SAVE'})


class ZSTO_OT_PieCallerBottomLeft(ZsBasicPieCaller):
    bl_idname = "zsto.pie_caller_bottom_left"
    bl_label = "Remove | Hierarchy"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def get_label(cls, context):
        if ZenPolls.is_collection_mode(context):
            return "Remove | Hierarchy"
        else:
            return "Remove"

    @classmethod
    def get_template_items(cls, context):
        if ZenPolls.is_collection_mode(context):
            return {
                'Default': 'zsto.remove_from_group("INVOKE_DEFAULT", keep_nested=False, mode="ACTIVE")',
                'CTRL': 'zsto.remove_from_group("INVOKE_DEFAULT", keep_nested=True, mode="ACTIVE")',
                'SHIFT': 'zsto.remove_from_group("INVOKE_DEFAULT", mode="ALL")'
            }
        else:
            return {
                'Default': 'zsto.remove_from_group("INVOKE_DEFAULT", mode="ACTIVE")',
                'SHIFT': 'zsto.remove_from_group("INVOKE_DEFAULT", mode="ALL")'
            }

    @classmethod
    def get_template_display_items(self, context):
        if ZenPolls.is_collection_mode(context):
            return {
                'Default': 'Remove',
                'CTRL': 'Remove (' + ZsLabels.PROP_KEEP_NESTED_NAME + ')',
                'SHIFT': 'Remove from All Groups'
            }
        else:
            return {
                'Default': 'Remove',
                'SHIFT': 'Remove from All Groups'
            }

    @classmethod
    def description(cls, context, properties):
        if ZenPolls.is_collection_mode(context):
            return 'Default: ' + ZsLabels.OT_REMOVE_SEL_ITEM_OBJ_DESC + \
                   '\nCTRL - (' + ZsLabels.PROP_KEEP_NESTED_NAME + ')' + \
                   '\nSHIFT - ' + ZsLabels.OT_REMOVE_SEL_ITEM_OBJ_ALL_DESC
        else:
            return 'Default: ' + ZsLabels.OT_REMOVE_SEL_ITEM_OBJ_DESC + \
                   '\nSHIFT - ' + ZsLabels.OT_REMOVE_SEL_ITEM_OBJ_ALL_DESC

    def draw(self, context):
        layout = self.layout

        p_label = self.get_template_display_name(context)
        if p_label:
            layout.label(text=p_label)

    cmd_enum: bpy.props.StringProperty(options={'HIDDEN', 'SKIP_SAVE'})
    is_menu: bpy.props.BoolProperty(default=True, options={'HIDDEN', 'SKIP_SAVE'})


class ZSTO_OT_PieCallerBottomRight(ZsBasicPieCaller):
    bl_idname = "zsto.pie_caller_bottom_right"
    bl_label = "Append | Hierarchy"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def get_label(cls, context):
        if ZenPolls.is_collection_mode(context):
            return "Append | Hierarchy"
        else:
            return "Append"

    @classmethod
    def get_template_items(cls, context):
        if ZenPolls.is_collection_mode(context):
            return {
                'Default': 'zsto.append_to_group("INVOKE_DEFAULT", keep_nested=False)',
                'CTRL': 'zsto.append_to_group("INVOKE_DEFAULT", keep_nested=True)',
            }
        else:
            return {
                'Default': 'zsto.append_to_group("INVOKE_DEFAULT")',
            }

    @classmethod
    def get_template_display_items(cls, context):
        if ZenPolls.is_collection_mode(context):
            return {
                'Default': 'Append',
                'CTRL': 'Append (' + ZsLabels.PROP_KEEP_NESTED_NAME + ')'
            }
        else:
            return {
                'Default': 'Append',
            }

    @classmethod
    def description(cls, context, properties):
        if ZenPolls.is_collection_mode(context):
            return 'Default: ' + ZsLabels.OT_APPEND_ITEM_OBJ_DESC + \
                   '\nCTRL - (' + ZsLabels.PROP_KEEP_NESTED_NAME + ')'
        else:
            return 'Default: ' + ZsLabels.OT_APPEND_ITEM_OBJ_DESC

    def draw(self, context):
        layout = self.layout

        p_label = self.get_template_display_name(context)
        if p_label:
            layout.label(text=p_label)

        if ZenPolls.is_collection_mode(context):
            draw_last_operator_properties(context, 'zsto.append_to_group', layout, blacklist_props={'keep_nested'})

    cmd_enum: bpy.props.StringProperty(options={'HIDDEN', 'SKIP_SAVE'})
    is_menu: bpy.props.BoolProperty(default=True, options={'HIDDEN', 'SKIP_SAVE'})


class ZSTO_OT_PieCallerBottom(ZsBasicBottomCaller):
    bl_idname = "zsto.pie_caller_bottom"

    cmd_enum: bpy.props.StringProperty(options={'HIDDEN', 'SKIP_SAVE'})
    is_menu: bpy.props.BoolProperty(default=True, options={'HIDDEN', 'SKIP_SAVE'})
